#include <stdio.h>
#include <string.h>
#include <ctype.h>


#define CURSES_INTERFACE

#ifdef CURSES_INTERFACE

extern "C" {
#include <curses.h>
}

#include "hp_curses.h"
#include "input.h"
#include "hpglobals.h"


static char const *numpadkeys[] = { "-", "7", "8", "9", "+", "4", "5", "6",
				    "*", "1", "2", "3", "/", "0", ".", "e" };


ClickableWin::ClickableWin(WINDOW *parent, char const **strings,
			   int nstrings, int fieldw, int numrows, 
			   int numcols, int off_x, int off_y)
{
  int i;

  offset_x = off_x;
  offset_y = off_y;
  nrows = numrows;
  ncols = numcols;
  stringlist = strings;
  numstrings = nstrings;
  fieldwidth = fieldw;
  fieldsperline = numcols / fieldw;

  win = subwin(parent, numrows, numcols, off_y, off_x);
  werase(win);
  
  for (i = 0; i < numstrings; i++)
    write_entry(i);

  wrefresh(win);  
}


CursesIO::CursesIO(KeyInfo const **keyptrs, int nkeys)
{
  int i;
  int maxkeynamelen;

  mainwin = initscr();
  cbreak();
  noecho();
  keypad(mainwin, TRUE);
  meta(mainwin, TRUE);

  can_use_mouse = 0;
  inputwin = stackwin = progwin = memwin = NULL;
  keyswin = numpadwin = NULL;
  credits = NULL;

#if defined ( CURSES_MOUSE ) && NCURSES_MOUSE_VERSION == 1
  if (mousemask(BUTTON1_CLICKED, NULL) == BUTTON1_CLICKED)
    can_use_mouse = 1;
#endif  /* CURSES_MOUSE && NCURSES_MOUSE_VERSION == 1 */

  maxkeynamelen = -1;

  numkeys = nkeys;
  keys = new KeyInfo [nkeys];
  immedstrs = (char const **) malloc(numkeys * sizeof(char const *));
  progstrs = (char const **) malloc(numkeys * sizeof(char const *));
  CHKMEM(immedstrs); CHKMEM(progstrs);
  buffer = NULL;

  for (i = 0; i < nkeys; i++) {
    keys[i] = *keyptrs[i];
    if ((int) strlen(keys[i].keyname) > maxkeynamelen)
      maxkeynamelen = strlen(keys[i].keyname);

    immedstrs[i] = (keys[i].immediate ? keys[i].keyname : (char *) NULL);
    progstrs[i] = (keys[i].program ? keys[i].keyname : (char *) NULL);
  }

  fieldwidth = maxkeynamelen + KEYCAP_PADDING;
  keysperline = COLS / fieldwidth;

  currentview = IMMED_SCRN;
  set_default_immed_view();

  credits = subwin(mainwin, CREDITS_HEIGHT, CREDITS_WIDTH,
		   numpadbottom + 1, numpadleft - 2);
  redraw_credits();
}



CursesIO::~CursesIO(void)
{
  if (buffer != NULL)
    free(buffer);

  free(immedstrs); free(progstrs);
  delete [] keys;
  (void) mousemask(0, NULL);
  destroy_subwindows();
  delwin(mainwin);
  endwin();
}


void CursesIO::draw_numpad(void)
{
  numpadleft = COLS - NUMPAD_WIDTH - 3;
  numpadtop = (numkeys - 1) / keysperline + 2;
  numpadright = numpadleft + NUMPAD_WIDTH;
  numpadbottom = numpadtop + NUMPAD_HEIGHT;

  numpadwin = new ClickableWin(mainwin, numpadkeys, 16, 3, NUMPAD_HEIGHT,
			       NUMPAD_WIDTH, numpadleft, numpadtop);

}


void CursesIO::squeeze_in_memory(void)
{
  if (stackwinsize > 3) {   // only do it if the screen is large enough
    memwinsize = stackwinsize - 2;

    memleft = stackright + HORIZ_SPACE_STACK_TO_MEMORY;
    memright = COLS - (numpadright - numpadleft) - 1 - HORIZ_SPACE_LEFT_OF_NUMPAD;
    memtop = stacktop;
    membottom = stackbottom - MEMORY_SHORTER_THAN_STACK;
    memwin = subwin(mainwin, memwinsize, memright - memleft, memtop, memleft);
  } else {
    memtop = membottom = stacktop;
    memwinsize = memleft = memright = 0;
    memwin = NULL;
  }
}


void CursesIO::set_default_immed_view(void)
{
  destroy_subwindows();

  inputwin = subwin(mainwin, 1, COLS, LINES - DISTANCE_INPUT_FROM_BOTTOM, 0);

  keyswin = new ClickableWin(mainwin, immedstrs, numkeys, fieldwidth,
			     (numkeys - 1) / keysperline + 1, COLS, 0, 0);

  draw_numpad();

  stackleft = 0;
  stacktop = (numkeys - 1) / keysperline + 1 + GAP_BETWEEN_STACK_AND_KEYPAD;
  stackright = MAXDIGS + 9;
  stackbottom = LINES - DISTANCE_INPUT_FROM_BOTTOM - GAP_BETWEEN_X_AND_INPUT;
  stackwinsize = stackbottom - stacktop;

  squeeze_in_memory();

  stackwin = subwin(mainwin, stackwinsize, stackright - stackleft,
		    stacktop, stackleft);

  redraw_credits();
  currentview = IMMED_SCRN;
}


void CursesIO::set_immed_step_view(void)
{
  destroy_subwindows();

  inputwin = subwin(mainwin, 1, COLS, LINES - DISTANCE_INPUT_FROM_BOTTOM, 0);


  keyswin = new ClickableWin(mainwin, immedstrs, numkeys, fieldwidth,
			     (numkeys - 1) / keysperline + 1, COLS, 0, 0);

  draw_numpad();

  stackleft = 0;
  stacktop = (numkeys - 1) / keysperline + 1 + GAP_BETWEEN_STACK_AND_KEYPAD + 1 + STEP_WINDOW_SIZE;
  stackright = MAXDIGS + 9;
  stackbottom = LINES - DISTANCE_INPUT_FROM_BOTTOM - GAP_BETWEEN_X_AND_INPUT;
  stackwinsize = stackbottom - stacktop;

  squeeze_in_memory();

  stackwin = subwin(mainwin, stackwinsize, stackright - stackleft,
		    stacktop, stackleft);

  progleft = PROGRAM_INDENT;
  progtop = stacktop - 1 - STEP_WINDOW_SIZE;
  progright = COLS - (numpadright - numpadleft) - 1 - HORIZ_SPACE_LEFT_OF_NUMPAD;
  progbottom = stacktop - 1;
  progwinsize = progbottom - progtop;

  progwin = subwin(mainwin, progwinsize, progright - progleft,
		   progtop, progleft);

  mvwaddstr(mainwin, progtop + 1, 0, CURRENTCELLPTR);    

  redraw_credits();
  currentview = IMMED_STEP_SCRN;
}


void CursesIO::set_default_prog_view(void)
{
  destroy_subwindows();

  inputwin = subwin(mainwin, 1, COLS, LINES - DISTANCE_INPUT_FROM_BOTTOM, 0);

  keyswin = new ClickableWin(mainwin, progstrs, numkeys, fieldwidth,
			     (numkeys - 1) / keysperline + 1, COLS, 0, 0);

  draw_numpad();

  progleft = PROGRAM_INDENT;
  progtop = (numkeys - 1) / keysperline + 1 + GAP_BETWEEN_PROG_AND_KEYPAD;
  progright = COLS - (numpadright - numpadleft) - 1 - HORIZ_SPACE_LEFT_OF_NUMPAD;
  progbottom = LINES - DISTANCE_INPUT_FROM_BOTTOM - GAP_BETWEEN_PROG_AND_INPUT;
  progwinsize = progbottom - progtop;

  progwin = subwin(mainwin, progwinsize, progright - progleft, progtop,
		   progleft);

  mvwaddstr(mainwin, progtop + (progwinsize + 1) / 2 - 1, 0, CURRENTCELLPTR);

  redraw_credits();
  currentview = PROG_SCRN;
}


void CursesIO::clear_input_line(void)
{
  wmove(inputwin, 0, 0);
  werase(inputwin);
  wrefresh(inputwin);
}


void CursesIO::write_input_line(char const *text)
{
  mvwaddstr(inputwin, 0, 0, text);
  wclrtobot(inputwin);
  wrefresh(inputwin);
}


void CursesIO::append_input_line(char singlechar)
{
  waddch(inputwin, singlechar);
  wrefresh(inputwin);
}


input_codes CursesIO::get_input_line(char *linebuffer, int buffsize)
{
  int linedone;
  int inanumber;
  int inexponent;
  int press;
  int pos;
  int seendecimal;
  static char recovery[5];  // Enough to hold an 'e+'
  int numericcomment = 0;
  int nocompletion = 0;

  linedone = inanumber = inexponent = pos = seendecimal = 0;
  if (dispmode == HEX)
    nocompletion = 1;

  clear_input_line();

  linebuffer[0] = 0;

  if (recovery[0] != 0) {
    strcpy(linebuffer, recovery);

    if (recovery[1] == 0 &&
	strchr(OPERATORS, recovery[0]) != NULL) {
      pos = 0;
      recovery[0] = 0;
      write_input_line(linebuffer);
      return TEXTINPUT;
    }
	
    pos = strlen(recovery);
    write_input_line(linebuffer);
    recovery[0] = 0;
  }

  do {

    press = get_next_char();

    if (press < 0) {
      if (pos != 0) {
	beep();
	continue;  // Special keys only work at start of line
      }

      switch(-press) {
      case ENDOFINPUT:
	linebuffer[0] = 0;
	return ENDOFINPUT;
	break;
      case PREVLINE:
	if (runmode == ENTER_PROG) {
	  linebuffer[0] = 0;
	  return PREVLINE;
	} else {
	  beep();
	  continue;
	}
      case NEXTLINE:
	if (runmode == ENTER_PROG) {
	  linebuffer[0] = 0;
	  return NEXTLINE;
	} else {
	  beep();
	  continue;
	}
      case DELETELINE:
	if (runmode == IMMED) {
	  strcpy(linebuffer, CLEARXSTR);
	  write_input_line(linebuffer);
	  return TEXTINPUT;
	} else {
	  linebuffer[0] = 0;
	  return DELETELINE;
	}
      case STEPSHORTCUT:
	strcpy(linebuffer, STEPSTR);
	write_input_line(linebuffer);
	return TEXTINPUT;
      }
    }

    if (press == NEWLINE && pos != 0) {
      linebuffer[pos] = 0;
      if (nocompletion &&
	  linebuffer[0] == NUMERICPREFIX) {   // strip out the '{' character
	int shiftamt;
	char *cptr = linebuffer;

	shiftamt = 1 + strspn(linebuffer + 1, WHITESPACE);
	do {
	  *(cptr) = *(cptr + shiftamt);
	} while (*(cptr++) != 0);
      }
      return TEXTINPUT;
    }

// The first character on the line will pretty well define the way
// things are going here.
    if (pos == 0) {

      if (press == REDRAWKEY) return REDRAW;
      
      if (press == NUMERICPREFIX) nocompletion = 1;

      if (strchr(OPERATORS, press) != NULL) {
	linebuffer[0] = 0;
	append_char(linebuffer, press);
	return TEXTINPUT;
      }

      if (press == NEWLINE) {
	strcpy(linebuffer, ENTERSTR);
	write_input_line(linebuffer);
	return TEXTINPUT;

      }

      if (press == BLANK || press == RUBOUT)
	continue;   // Silently ignore leading blanks or delete characters

      if (isdigit(press))
	inanumber = 1;

      if (press == '.') {
	inanumber = seendecimal = 1;
	linebuffer[0] = '0';
	append_input_line('0');
	pos = 1;
      }

    } else {


      if (press == RUBOUT) {   // Delete a character
	--pos;
	if (inexponent &&
	    (linebuffer[pos] == 'e' ||
	     linebuffer[pos] == 'E'))
	  inexponent = 0;
	if (seendecimal &&
	    linebuffer[pos] == '.')
	  seendecimal = 0;
	if (numericcomment &&
	    linebuffer[pos] == COMMENT)
	  numericcomment = 0;
      	linebuffer[pos] = 0;
	write_input_line(linebuffer);
	if (pos == 0) {
	  inanumber = 0;
	  nocompletion = 0;
	}
	continue;
      }

      
      if (inanumber && 
	  !nocompletion) {   // The line started with a valid number
	if (! inexponent &&
	    (press == 'e' ||
	     press == 'E')) {   // Exponent character

	  inexponent = 1;

	} else if (press == '.') {

	  if (seendecimal || inexponent)
	    continue;        // Silently ignore extra decimals, as the HP-67

	  seendecimal = 1;
	  
	} else if ((press == '+' || press == '-') &&
		   (linebuffer[pos-1] == 'e' || linebuffer[pos-1] == 'E')) {
	  ;  // Nothing

	} else if (press == COMMENT) {
	  numericcomment = 1;
	} else if (press == BLANK) {
	  ;  // Nothing
	} else if (!isdigit(press) &&
		   !numericcomment) {   // A command is starting
	  int n = 0;

	  if (linebuffer[pos-1] == 'e' || linebuffer[pos-1] == 'E')
	    recovery[n++] = linebuffer[--pos];

	  if (pos > 1 &&
	      (linebuffer[pos-2] == 'e' || linebuffer[pos-2] == 'E') &&
	      (linebuffer[pos-1] == '+' || linebuffer[pos-1] == '-')) {
	    recovery[n++] = linebuffer[pos-2];
	    recovery[n++] = linebuffer[pos-1];
	    pos -= 2;
	  }

	  recovery[n++] = press;
	  recovery[n++] = 0;
	  linebuffer[pos] = 0;
	  write_input_line(recovery);
	  return TEXTINPUT;
	}

      }

    }

    append_input_line(press);
    linebuffer[pos] = press;
    pos++;
    if (pos == buffsize - 1) {
      linebuffer[pos] = 0;
      clear_input_line();
      return TEXTINPUT;
    }

  } while (1);
}



void CursesIO::compute_positions(void)
{
  keysperline = COLS / fieldwidth;
  numkeyrows = (numkeys - 1) / keysperline + 1;
}



int CursesIO::break_run(void) const
{
  nodelay(mainwin, TRUE);
  if (getch() == ERR) {
    nodelay(mainwin, FALSE);
    return FALSE;
  }

  nodelay(mainwin, FALSE);
  return TRUE;
}



// This routine filters inputs. It converts keypad sequences to
// the corresponding single keycap value, interprets the cursor
// keys, and looks for mouse clicks. It discards control characters
// and converts tabs to spaces.
int CursesIO::get_next_char(void)
{
  int press;
  static int nbuffed = 0;
  int buffsize;
  static char *nextp = NULL;
  char *cp1;
  const char kp_special_3[] = KPSPECIAL3;
  const char kp_special_4[] = KPSPECIAL4;


// Allocate a buffer
  if (buffer == NULL) {
    buffsize = ((fieldwidth + 2 > (int) strlen(RUNSTR) + 6) ? 
		fieldwidth + 2 :
		strlen(RUNSTR) + 6);
    buffer = (char *) malloc(buffsize);
    CHKMEM(buffer);
    nbuffed = 0;
    nextp = NULL;
  }

// If the buffer is not empty, send the next character.
  if (nbuffed != 0) {
    press = *nextp;
    nextp++;
    nbuffed--;
    return press;
  }

// Loop until return

  do {

// Get a key from the keyboard
    press = get_a_key();

// If that key was the sequence which begins a keypad sequence, wait for
// the next key.

// Special case if the same key announces a meta key as a keypad key.
#if KPSPECIAL1 == ESC
    if (press == (KPSPECIAL2 | META)) {  // Has first two characters of keypad key
      press = get_a_key();
    
      if ((cp1 = strchr(kp_special_3, press)) != NULL) {  // it's in the list #3
	press = kp_special_4[cp1 - kp_special_3];
      } else {     // It's not in the list, make it "run O"

	if (!isprint(KPSPECIAL2)) { // Can't make it into a run command
	  beep();
	  continue;   // Try again from the top
	}

	strcpy(buffer, RUNSTR);
	append_char(buffer, ' ');
	append_char(buffer, KPSPECIAL2);
	append_char(buffer, NEWLINE);
	press = buffer[0];
	nextp = &buffer[1];
	nbuffed = strlen(buffer) - 1;
      }     
      return press;
    }

#else  /* KPSPECIAL1 != ESC */
    if (press == KPSPECIAL1) {
      press = get_a_key();
      if (press != KPSPECIAL2) {   /* Not a keypad key */
	if (!isprint(KPSPECIAL1)) {
	  beep();
	  continue;
	} else {
	  if (isprint(press)) {
	    buffer[0] = press;
	    nextp = &buffer[0];
	    nbuffed = 1;
	  } else {
	    beep();
	  }
	  return KPSPECIAL1;
	}
      } else {    // Keypad key? First two characters are right for it
	press = get_a_key();

	if ((cp1 = strchr(kp_special_3, press)) != NULL) {  // it's in the list #3
	  press = kp_special_4[cp1 - kp_special_3];
	  return press;
	} else {     // It's not in the list, must be an error
	  beep();
	  continue;
	}
      }
    }

#endif  /* KPSPECIAL != ESC */

// We can only get here if the system has decided that it didn't
// just see a composite keypad character (one composed of three
// returned numbers in rapid succession)
      

    if (press <= MAX_SIMPLE_CHAR &&
	press & META) {
      press &= ~META;

      if (press == BLANK) return -STEPSHORTCUT;

      if (!isprint(press)) {
	beep();
	continue;
      }
      
      if (runmode == ENTER_PROG)
	strcpy(buffer, LABELSTR " ");
      else
	strcpy(buffer, RUNSTR " ");

      append_char(buffer, press);
      append_char(buffer, NEWLINE);
      press = buffer[0];
      nextp = &buffer[1];
      nbuffed = strlen(buffer) - 1;
      return press;
    }
      

    if (press == KEY_A1) return '7';
    if (press == KEY_A3) return '9';
    if (press == KEY_B2) return '5';
    if (press == KEY_C1) return '1';
    if (press == KEY_C3) return '3';
    if (press == KEY_ENTER || press == NEWLINE) return NEWLINE;

// Check for special editing mode sequences, and if received, return
// the negative value of the corresponding enum.
    if (press == KEY_UP) return -PREVLINE;
    if (press == KEY_DOWN) return -NEXTLINE;
    if (press == QUIT) return -ENDOFINPUT;
    if (runmode == ENTER_PROG &&
	(press == RUBOUT || press == KEY_DC))
	 return -DELETELINE;
    

#if defined ( CURSES_MOUSE ) && NCURSES_MOUSE_VERSION == 1

// Now, check for a mouse click.
    if (press == KEY_MOUSE) {
      MEVENT place;
      char const *textclicked;
      int keynum;

      if (getmouse(&place) == OK) {
	
	if ((textclicked = keyswin->string_clicked(place.x, place.y, &keynum)) != NULL) {
	  strcpy(buffer, textclicked);
	  if (keys[keynum].can_have_arg) {
	    append_char(buffer, BLANK);
	  } else {
	    if (!(textclicked[1] == 0 &&
		  strchr(OPERATORS, textclicked[0]) != NULL))
	      append_char(buffer, NEWLINE); /* not after operators */
	  }

	  nextp = &buffer[1];
	  nbuffed = strlen(buffer) - 1;
	  return buffer[0];
	} else if ((textclicked = numpadwin->string_clicked(place.x, place.y, &keynum)) != NULL) {
	  if (isdigit(*textclicked) || *textclicked == '.') {  // part of number
	    return *textclicked;
	  } else {         // Function key.
	    nbuffed = 0;
	    return *textclicked;
	  }
	} else {
	  beep();
	  wrefresh(inputwin);
	  continue;
	}

      }

    }

#endif  /* CURSES_MOUSE && NCURSES_MOUSE_VERSION == 1 */


// OK, that covers just about everything. Look for a regular key

    if (press == TAB) press = BLANK;

    if (press == KEY_BACKSPACE || press == RUBOUT) return RUBOUT;

    if (press == REDRAWKEY) return REDRAWKEY;

    if (!isprint(press)) {
      beep();
      wrefresh(inputwin);
      continue;
    }

    return press;

  } while (1);
}



void CursesIO::show_stack(VStack &from)
{
  if (currentview != IMMED_SCRN)
    set_default_immed_view();
  draw_stack_elements(from);
  draw_memory_elements();
  refresh();
}


void CursesIO::pretty_draw(WINDOW *win, char *string)
{
  char *eptr, *mantissa;
  int nblanks;

  if (string[0] != '-') {
    waddch(win, BLANK);   // leave blank in sign position
    mantissa = &string[0];
  } else {
    waddch(win, '-');     // put in the minus sign
    mantissa = &string[1];
  }

  if (strstr(mantissa, HEXPREFIX) != NULL ||
      (eptr = strchr(mantissa, 'e')) == NULL) {
    waddstr(win, mantissa);
  } else {     // Has an 'e' in it, break up like the HP-67 display
    *(eptr++) = 0;
    if (*eptr == '+') *eptr = BLANK;
    waddstr(win, mantissa);
    for (nblanks = MAXDIGS + 2 - strlen(mantissa); nblanks > 0; nblanks--)
      waddch(win, BLANK);
    waddstr(win, eptr);
  }
  
}


void CursesIO::draw_stack_elements(VStack &from)
{
  unsigned int i;
  unsigned int stop;
  char ostring[1000];
  int rows, cols;

  stop = from.get_size();
  if (stackwinsize < stop) stop = stackwinsize;

  werase(stackwin);
  for (i = 0; i < stop; i++) {
    from.reveal_value(i).printvalue(dispdigs, dispmode, ostring, 1000,
				    &rows, &cols);
    wmove(stackwin, stackwinsize - i - 1, 0);
    pretty_draw(stackwin, ostring);
  }

  wrefresh(stackwin);
}


void CursesIO::show_stack_and_step(VStack &from, CalcProgMem const *pmem, int pnum)
{
  if (currentview != IMMED_STEP_SCRN)
    set_immed_step_view();

  draw_stack_elements(from);
  draw_memory_elements();
  draw_program_elements(pmem, pnum);
  refresh();
}


void CursesIO::show_prog_space(CalcProgMem const *pmem, int pnum)
{
  if (currentview != PROG_SCRN)
    set_default_prog_view();

  draw_program_elements(pmem, pnum);
  refresh();
}


void CursesIO::draw_memory_elements(void)
{
  unsigned int i;
  Arithdatatype *memval;
  char ostring[1000];
  int rows, cols;
  char const *label;

  if (memwin == NULL) return;

  werase(memwin);
  for (i = 0; i < memwinsize; i++) {
    if ((memval = memories.recent_access(i, &label)) == NULL) break;
    memval->printvalue(dispdigs, dispmode, ostring, 1000, &rows, &cols);
    wmove(memwin, memwinsize - i - 1, 0);
    wprintw(memwin, "%*.*s = ", memright - memleft - MAXDIGS - 13,
	    memright - memleft - MAXDIGS - 13, label);
    pretty_draw(memwin, ostring);
  }
  wrefresh(memwin);
}


void CursesIO::draw_program_elements(CalcProgMem const *pmem, int pnum)
{
  char *buff;
  int bufflen = progright - progleft + 1;
  CalcProgMem const *pptr;
  int yval;
  intstring lnstring;   // line number
  int stepnum = pnum;


  werase(progwin);

  buff = (char *) malloc(bufflen);
  CHKMEM(buff);
 
// First, print the current line. If it's an invalid cell it's because
// the current pointer is at one of the "barrier" program elements
  wmove(progwin, yval = (progwinsize + 1) / 2 - 1, 0);
  sprintf(lnstring, "%03d: ", stepnum);
  waddstr(progwin, lnstring);
  if (pmem->validcell == 0) {
    waddstr(progwin, BARRIERSTR);
  } else {
    pmem->printcontents(buff, bufflen);
    buff[progright - progleft - 6] = 0;  // truncate the line if it's too long
    waddstr(progwin, buff);
  }

// Now, start backing up, examining those cells earlier than the current 
  pptr = pmem->get_prev_mem();
  yval--;
  while (pptr != NULL && yval >= 0) {
    wmove(progwin, yval, 0);
    sprintf(lnstring, "%03d: ", --stepnum);
    waddstr(progwin, lnstring);
    if (pptr->validcell == 0) {
      waddstr(progwin, BARRIERSTR);
    } else {
      pptr->printcontents(buff, bufflen);
      buff[progright - progleft - 6] = 0;  // truncate the line if it's too long
      waddstr(progwin, buff);
    }

    pptr = pptr->get_prev_mem();
    yval--;
  }

// Now, wind forward, examining later cells
  pptr = pmem->get_next_mem();
  yval = (progwinsize + 1) / 2;
  stepnum = pnum;
  while (pptr != NULL && yval < (int) progwinsize) {
    wmove(progwin, yval, 0);
    sprintf(lnstring, "%03d: ", ++stepnum);
    waddstr(progwin, lnstring);
    if (pptr->validcell == 0) {
      waddstr(progwin, BARRIERSTR);
    } else {
      pptr->printcontents(buff, bufflen);
      buff[progright - progleft - 6] = 0;  // truncate the line if it's too long
      waddstr(progwin, buff);
    }

    pptr = pptr->get_next_mem();
    yval++;
  }
  free(buff);
  wrefresh(progwin);
}

#endif  /* CURSES_INTERFACE */
