/* OrderedCollection is an indexed collection in which the
 elements are usually chronologically ordered, i.e. elements
 at the end were added to the collection after the ones at the
 beginning.

 The most obvious use of an OrderedCollection is a stack.  You
 can think of an OrderedCollection as a stack, if you prefer,
 and we have even provided pop and push methods.

 An OrderedCollection is considered empty if its two instance
 variables, firstElement and lastElement, are equal.  */!!

/* inherit(Array, #OrderedCollection, #(firstElement
lastElement ), 2, 1) */!!

now(class(OrderedCollection)) !!

now(OrderedCollection) !!

/* Evaluate the one-argument block over the elements of the
  receiver.  */
Def  do(self, aBlock)
{       ^do(new(Interval, firstElement, lastElement, 1),
           {using(idx)   eval(aBlock, at(self:Object, idx) )
           });
} !!

/* Initializes an OrderedCollection by setting firstElement and
  lastElement equal to zero.  You can empty an existing OrderedCollection
  simply by sending it an init message.  */
Def  init(self)
{       firstElement := lastElement := 0
} !!

/* Return the current number of elements in the collection.  */
Def  size(self)
{       ^lastElement - firstElement;
} !!

/* Return (but not remove) the first
  element in the collection, if any.
  If there isn't a first element, i.e. if
  the receiver is empty, an "Empty
  collection" error is generated.  */
Def  first(self)
{ ?hasElements(self);
  ^at(self:Object, firstElement);
}!!


/* Return (but not remove) the last element in the collection, if any.
  If there isn't a last element, i.e. if the receiver is empty,
  an "Empty collection" error is generated.  */
Def  last(self)
{       ?hasElements(self);
        ^at(self:Object, lastElement - 1);
} !!

/* Grow the OrderedCollection so that it
  can hold more elements.  Works by copying
  elements into larger collection and then
  swapping object pointers with the new
  collection.  */
Def  grow(self | newColl)
{       newColl := new( class(self), limit(self) + 8);
        do(self,
          {using(elem)  add(newColl, elem)
          });
        swap(self, newColl);
} !!

/* Removes and return the last element in the collection, if any.
  If the collection is empty, an "Empty collection" error is generated.
  (The pop method of this class uses this method; removeLast is
  another name for pop.)  */
Def  removeLast(self | val)
  {
  if lastElement > firstElement
  then val := self:Object[lastElement :=
    lastElement - 1];
    put(self:Object, nil, lastElement + 1);
    ^val;
  endif;
  error(self, stackTop(0), #emptyError);
  } !!

/* Removes and return the last element in the collection.  If the
  collection is empty, an "Empty collection" error is generated.  */
Def  pop(self)
{ ^removeLast(self:OrderedCollection)
}!!

/* Add a new last element, anObj, to the collection.  */
Def  push(self, anObj)
{ add(self:OrderedCollection, anObj)
} !!

/* Makes sure that the index is within
 the valid range of the collection.   (Private method)  */
Def checkRange(self, idx)
{       if idx < firstElement or idx > lastElement
        then error(self, stackTop(0), #rangeError);
        endif;
}!!

/* Insert a new element at the specified index in the
 collection.  Reports an error if the index is not
 in the current valid range.  Grow the collection if necessary. */
Def insert(self, elem, idx | index)
  {
  if idx < firstElement or idx > lastElement
  then error(self, stackTop(0), #rangeError);
  endif;

  if (index := lastElement) == limit(self)
  then grow(self);
  endif;
  loop
  while index > idx
  begin      put(self:Object, at(self:Object,
    index - 1), index);
    index := index - 1;
  endLoop;
  put(self:Object, elem, idx);
  lastElement := lastElement + 1;
  }!!

/* Insert any indexed collection into the receiver,
  starting at the specified idx. */
Def insertAll(self, coll, idx | srcIdx)
  {  srcIdx := size(coll);
  loop
  while srcIdx > 0
  begin  insert(self, coll [srcIdx-1], idx);
    srcIdx := srcIdx - 1;
  endLoop;
  }!!

/* Removes the element at the specified idx.  If idx is not
  valid, i.e. if it is less than firstElement or greater than
  or equal to lastElement, then a "Range Error" is generated.
  The removed element doesn't leave a "hole;" elements above 
  the removed element (i.e. with indices greater than idx) are
  moved down.  */
Def remove(self, idx | elem, index, lim)
  {
  if idx < firstElement
  or idx > (lim := lastElement - 1)
  then error(self, stackTop(0), #rangeError);
  endif;
  index := idx;
  loop
  while index < lim:Int
  begin put(self:Object, at(self:Object, index
    + 1), index);
    index := index + 1;
  endLoop;
  lastElement := lastElement - 1;
  put(self:Object, nil, lastElement);
  }!!


/* Removes the first element in the collection, if there is one.
  If there isn't one, then a "Range error" is generated. */
Def  removeFirst(self | val)
{ ^remove(self:OrderedCollection,
    firstElement);
} !!

/* Generate an "Empty collection" error
  if the collection is empty.  If not, then
  it just returns the receiver.  This
  method is used as an error checking 
  mechanism by some of the other methods 
  of this class.  */
Prim ?hasElements(self):OrderedCollection!!

/* Add the specified object to the
  receiver at its end.  The add method
  is synonomous with push if you consider
  an OrderedCollection to be a stack.  In
  fact, this class's push method uses
  this method.  */
Prim add(self, anObject):OrderedCollection!!

/* Returns a new collection with all
  of the receiver's elements, but in
  reverse order.  Differs from
  IndexedCollection:reverse by the
  fact that it returns a new collection
  rather than reversing the receiver
  in place.  */
Prim reverse(self):OrderedCollection!!