/* The File class provides the methods to read and write
 data to and from DOS files. */!!

/* inherit(Object, #File, 
#(handle   /* File handle, an integer value */
fileName   /* File name, as a null-terminated string */
), nil, 2, nil) */ !!

now(class(File)) !!

/* Boolean method which determines whether
  or not a file with the specified
  dosFileName exists.  The mode determines
  which operations are valid for the
  given file: 0=read only, 1=write only,
  and 2=read/write.  For example, if the
  file "test.dat" is a valid, writeable
  file on drive A:, the message
  exists(File, "a:test.dat", 1) would
  return an unopened File,
  File("a:test.dat").  If a DOS file with
  the specified name does not exist, or if
  it does but the mode is invalid for that
  file, then exists returns nil.  */
Def  exists(self, dosFileName, mode | file)
{  file := new(self);
  setName(file, dosFileName);
  if open(file, mode)
  then close(file);
    ^file
  endif;
  ^nil
}!!




now(File);!!

/* Move absolutely (i.e. relative to the
  beginning of the file) to lPos.  For
  example, to move to the beginning of a
  file called Joe, you would say moveTo(Joe,
  0).  Also return the new value of
  the file pointer.

  Note: lPos is converted to a Long
  for you, so you can use any number as an
  argument to moveTo.  Also note that
  moveTo uses lseek, so refer there to see
  what happens when you moveTo past the
  end of a file.  */
 Def moveTo(self, lPos)
{       ^lseek(self,asLong(lPos),0)
}!!

/* Move the specified number of bytes in
  the file relative to the current
  posisition.  move converts lPos to a Long,
  so you can use any number as an argument
  to move.  lPos can be either positive or
  negative, depending on whether you want
  to go forward or backwards in the file.
  Also, move returns the new value of
  the file pointer.

  Note: move uses the lseek method in this
  class, so refer there for notes about
  moving past the end of file.  */
 Def move(self, lPos)
{       ^lseek(self, asLong(lPos), 1)
}!!

/* Return a long integer containing
  length of the file in bytes.  The file
  must be open.  */
 Def length(self)
{       ^lseek(self, 0L, 2)
}!!

/* Copy one unopened file to another.
  Return a result code (see the Actor
  manual, Guide to the Actor Classes,
  the File class, for an explanation of
  the various result codes). */
 Def copyAll(self, aFile | rc)
{       open(self, 0);
  checkError(self);
  open(aFile,1);
  checkError(aFile);
  rc := copy(self, aFile, 10000000L);
  checkError(aFile);
  close(self);
  close(aFile);
  ^rc;
}!!

/* Display an error dialog box with
  str as the caption.  The second
  argument, bp, is displayed at the end
  of the caption. "File Error" is the
  title of the dialog. (The checkError
  method for this class uses this
  method.)  */
Def dosError(self, bp, str | strm)
{ strm := streamOver(
  "");
  printOn(self, strm);
  printOn(str, strm);
  printOn(bp, strm);
  errorBox(
  "File Error", strm.collection);
} !!


/* Print the receiver file onto the specified stream.
  */
Def printOn(self, aStrm)
{ printOn( tuple( class(self), '(', '"',
  removeNulls(fileName), '"', ')'), aStrm);
}!!


/*  Allow files to be used like streams.
  Write the argument string to the file. */
 Def nextPutAll(self, str)
{  write(self, str);
}!!

/*  Return the current position (the
  current value of the file pointer).  */
 Def position(self)
{ ^lseek(self, 0l, 1);
}!!


/* Return true if file is at eof (end
  of file).  */
 Def atEnd(self)
{ ^position(self) >= length(self) - 1;
}!!

/* Return a string copied from the
  specified range in the file.  */
 Def copyFrom(self, lo, hi)
{ moveTo(self, lo);
  checkError(self);
  ^read(self, asInt(hi-lo));
}!!

/* Read and return the next Char in the
  file. */
 Def next(self)
{ ^readChar(self);
}!!

/* Write a Char to the file. */
 Def nextPut(self, aChar)
{ ^writeChar(self, aChar);
}!!

/* Save an image of object memory with
  values for default static and dynamic
  memory allocation. */
 Def install(self, static, dynamic)
{ create(self);
  snap(self, static, dynamic);
  close(self);
}!!

/* Executed after another file operation.
  If an error occurred in the process of
  executing the operation, then checkError
  will display an error dialog displaying
  the file name and the error number.
  After every file operation, you should
  either send a checkError or getError
  message to the File object. Please refer
  to the Actor manual (Guide to the Actor
  Classes, the File Class) for a list of
  the error numbers that checkError can
  display.  See also the getError method in
  this class (checkError essentially just
  puts up a dialog box displaying any
  nonzero value that getError returns).  */
 Prim checkError(self):nil!!

/* Close the DOS file and also frees up
  its handle so another file can use it.  */
Prim close(self):File!!

/* Copy the specified number of bytes
  from the receiver to the destination
  file.  Both files must be opened, and
  the number of bytes must be a Long
  integer.  It returns a result code
  in the form of an Int.  The various
  result codes are summarized in the
  Actor manual, in the Guide to the Actor
  Classes, the File class.  */
Prim copy(self, aFile, numBytes):Int!!

/* Create the DOS file with the
  specified name and return the DOS handle
  to that file.  If a file by the same name
  already exists and it is not marked as
  read-only, then the existing file will be
  deleted.  If it is marked as read-only,
  then an error will be generated. If
  create returns nil instead of an integer
  file handle, then an error occurred.
  Note: create effectively does an open,
  too, so you don't have to open the file
  after you create it. */
 Prim create(self):FileHandle!!

/* Delete the receiver file.  If the
  deletion succeeded, then an integer
  is returned (its value is undefined,
  however).  If it did not succeed, such
  as if the file was read-only or did not
  exist, then delete returns nil.  */
Prim delete(self):nil!!


/* Return true if the current file
  pointer is at the end of a line, i.e.
  if the current character in the file
  being pointed to is a CR character. */
Prim eol(self):Boolean!!

/* Return the last result code for a
  file operation.  If no error occurred,
  then getError returns 0, otherwise it
  returns a number corresponding to the
  DOS error.  For a complete list of the
  possible values getError can return,
  refer to the Guide to the Actor Classes,
  the File class.  After you do any file
  operation, you should call either
  getError or checkError, or else the
  error will go undetected.  */
Prim getError(self):Int!!

/* Move the file pointer the specified
  number of bytes according to the
  specified mode.  If mode is 0, then the
  pointer is moved relative to the
  beginning of the file.  If mode is 1, the
  pointer is moved relative to the current
  position.  If mode is 2, the pointer is
  moved relative to the end of the file.
  lseek returns a long integer representing
  the new value of the file pointer.  Some
  notes: the file must have been open, and
  numBytes must be a Long.  Also, you can
  lseek past the end of the file and
  no error will be generated (i.e.
  getError will return 0).  It's up to
  the programmer to protect against
  lseeking past the end of file--if you
  lseek past the end of file and then
  try to read/write, an error will occur.
  */
 Prim lseek(self, numBytes, mode):Long!!


/* Open the file.  The allowable
  operations on the file are specified
  by the mode argument.  If mode=0, then
  reads only are allowed.  If mode=1, then
  writes only are allowed.  If mode=2, then
  reads and writes are both allowed.

  If the open is successful, then open
  will return the DOS handle for the file.
  If not, then it returns nil.  If an open
  message returns nil, then you can use
  checkError or getError to find out
  exactly what happened.  */
Prim open(self, mode):FileHandle!!

/* Return the specified number of
  bytes read from the file, represented
  as a string.  For instance, if you have
  a File object called Sam, then the
  following message would read 100L bytes
  starting from the current position in
  the file and return the data as a
  String: read(Sam, 100L).

  Note that numBytes must be a Long
  integer.  */
Prim read(self, numBytes):String!!

/* Read a character from the current file
  position and return that character.  */
Prim readChar(self):Char!!

/* Rename a DOS file to newFileName, a
  String.  Does not change the value of the
  File object's instance variable,
  fileName.  The argument newFileName
  must be a string in the ASCIIZ format,
  i.e. terminated with a null byte.  Please
  see the rename method in this class.  */
 Prim rnam(self, newFileName):nil!!


/* Assigns a DOS filename to an Actor File
  object.  A File's instance variable,
  fileName, is set by this method.  A
  File has to have a valid DOS filename
  before you can do any operations on it,
  so you should send a setName message
  to every newly-created File object.
  Convert aFileName, an Actor string,
  into the ASCIIZ format (null-terminated).
  Example: setName(f, "test.dat").  */
Prim setName(self, aFileName):File!!

/* Write the String argument to the receiver file. */
Prim write(self, aStr):File !!

/* Write the specified character to the receiver file.  */
Prim writeChar(self, aChar):File !!

/* Rename the DOS file to newFileName
  and also update the Actor File object's
  instance variable, fileName.  */
 Def reName(self, newFileName)
{ rnam(self, asciiz(newFileName));
  ^setName(self, newFileName);
} !!


/* Save an image of object memory with
  values for default static and dynamic
  memory allocation. */
Prim snap(self, static, dynamic):File!!


/* Save an image of object memory to the
  receiver File.  Default static and dynamic
  memory values are supplied (static=90,
  dynamic=50).  */
Prim snapshot(self):File!!
