package sorcererII;

import java.io.IOException;
import java.util.logging.Level;


public class SorcererDiskDrive
{
  public static final int SECTORS_PER_TRACK = 16;
  public static final int NUMBER_OF_TRACKS = 77;
  public static final int BYTES_PER_SECTOR = 256+14;

  private int _activeCount = 0;
  private int _sectorNumber = 0;
  private int _trackNumber = 0;
  private boolean _newSector = false;
  private SorcererDisk _disk = null;
  private int _sectorIndex = 0;
  private int _writeIndex = 0;
  private final int _unitNumber;
  
  public SorcererDiskDrive(final int unitNumber) {
    _unitNumber = unitNumber;
  }
  
  public char getUnitLetter() {
    return "ABCD".charAt(_unitNumber);
  }
  
  public final void insertDisk( SorcererDisk disk )
  {
    _disk = disk;
  }
  
  public SorcererDisk getDisk() {
    return _disk;
  }
  
  public final boolean diskIn()
  {
    return _disk != null;
  }

  public final boolean dataReady()
  {
    return true;
  }

  public final boolean home()
  {
    return _trackNumber == 0;
  }

  public final void stepForward()
  {
    if( _trackNumber < ( NUMBER_OF_TRACKS - 1 ) )
    {
      ++_trackNumber;
    }
    //System.out.println( "Step forward to " + _trackNumber );
  }

  public final void stepBackward()
  {
    if( _trackNumber > 0 )
    {
      --_trackNumber;
    }
    //System.out.println( "Step backward to " + _trackNumber );
  }

  public final boolean active()
  {
    return _activeCount > 0;
  }

  public final void activate()
  {
    if(_activeCount == 0 && _disk != null) {
      try
      {
        _disk.activate();
      }
      catch(final IOException e )
      {
        Sorcerer.LOGGER.log(Level.SEVERE, "Problem reading disk " + getUnitLetter(), e);
      }      
    }
    _activeCount = 400;
  }

  private final void writeReg0( int b )
  {
    switch( b )
    {
      case 0xA0:
        break;
      case 0x20:
      case 0x21: // Disk b:
        activate();
        break;
      case 0x60:
        stepBackward();
        break;
      case 0x61:
        stepForward();
        break;
    }
  }

  public final void readyWrite()
  {
    _writeIndex = 0;
  }

  private final void writeReg1( int b )
  {
    switch( b )
    {
      case 0xA0:
        break;
      case 0x20:
      case 0x21: // Disk b:
        activate();
        break;
      case 0x60:
        stepBackward();
        break;
      case 0x61:
        stepForward();
        break;
    }
  }

  public final void writeReg2( int b )
  {
    _disk.write( _trackNumber, _sectorNumber, _writeIndex++, (byte)b );
  }

  public final int readReg0()
  {
    int r = _sectorNumber;
    if( _newSector )
    {
      r |= 0x80;

      _newSector = false;
    }

    return r;
  }

  private final int readReg1()
  {
    int r = 0;

    if( active() ) r |= 0x20;
    if( home() ) r |= 0x08;
    if( dataReady() ) r |= 0x80;

    return r;
  }

  public final int readReg2()
  {
    if( _disk != null )
    {
      if( _sectorIndex < BYTES_PER_SECTOR )
      {
        byte data = _disk.read( _trackNumber, _sectorNumber, _sectorIndex++ );

        return (int)data & 0xff;
      }
    }
    return 0;
  }

  public final void write( int address, int b )
  {
    if( active() ) activate();

    //System.out.println( "A: Writing " + Z80.toHexByte( b ) + " to " + Z80.toHexWord( address ) );
    switch( address )
    {
      case 0: writeReg0( b ); break;
      case 1: writeReg1( b ); break;
      case 2: writeReg2( b ); break;
    }
  }

  public final int read( int address )
  {
    if( active() ) activate();

    int r = 0;
    switch( address )
    {
      case 0: r = readReg0(); break;
      case 1: r = readReg1(); break;
      case 2: r = readReg2(); break;
    }
    //System.out.println( "A: Reading " + Z80.toHexByte( r ) + " from " + Z80.toHexWord( address ) );
    return r;
  }

  public final void tick()
  {
    if( active() )
    {
      _sectorNumber++;
      _sectorIndex = 0;

      if( _sectorNumber >= SECTORS_PER_TRACK )
      {
        _sectorNumber = 0;
      }
      _newSector = true;

      _activeCount--;

      //System.out.println( "Sector " + _sectorNumber );

      if( !active() )
      {
        if( _disk != null )
        {
          try
          {
            _disk.deactivate();
          }
          catch( Exception e )
          {
            Sorcerer.LOGGER.log(Level.SEVERE, "Problem writing disk " + getUnitLetter(), e);
          }
        }
      }
    }
  }
}