/* crt0.h - header file to define structures and flags
 * to control the operation of the C startup code.
 *
 * Copyright (C) 1997-1999 by Object Software Inc., All Rights Reserved.
 *
 * The authors hereby grant permission to use, copy, modify, distribute,
 * and license this software and its documentation for any purpose, provided
 * that existing copyright notices are retained in all copies and that this
 * notice is included verbatim in any distributions. No written agreement,
 * license, or royalty fee is required for any of the authorized uses.
 *
 * The C runtime startup code supplied with the Object Software Inc. port of Gnu CC
 * performs several operations to make it easier to adapt the compiler and runtime
 * library to embedded systems.  The exact sequence can be inferred by reviewing the
 * source code in the file crt0.S.  Here's a summary:
 *
 * - The variable 'crt0_flags' (if defined by user code) is loaded into a CPU register.
 * - Peripheral registers are initialized according to a user-supplied table of
 *   init values.
 * - If a runtime stack value is defined (symbol name __stack), it is loaded into the
 *   stack pointer. This value may be different, if necessary, from the stack pointer
 *   stored in the exception vector table (symbol name __boot_stack).
 * - If defined, the function 'hardware_init_hook' is called to perform further system
 *   initialization. This function may be called normally, or using a JMP instruction
 *   with the return address in a CPU register, depending upon the value in crt0_flags.
 * - The bss (uninitialized) data section is cleared to all zeros.  This may be disabled
 *   if necessary by setting a bit in crt0_flags.
 * - Initialized variables (in the .data section) are loaded with their initial values
 *   by copying an image of the data section from ROM to RAM.
 * - an initial stack frame is created.
 * - If a function named 'software_init_hook' is defined, it is called here to permit final
 *   initialization before entering main ().
 * - Finally, main is called, and if it returns, the library function exit () is called
 *   with main's return value as its parameter.
 */

/* Peripheral Initialization
 * *************************
 * Crt0 looks for an array of initialization records which are used to set up
 * critical system resources (like chip selects, watchdog timers, etc) before
 * any other initialization is done.
 *
 * The structure of each array element is defined by the struct sCrt0InitEntryHeader.
 * Each entry defines a list of values to be written to the specified address. A
 * list of values follows the entry header; the number of entries in the list is
 * specified by the least-significant 14 bits of 'count'. The two most significant bits
 * of 'count' specify the size of each list entry:
 * 0:0	one byte (8 bits) per entry
 * 0:1	two bytes (16 bits) per entry
 * 1:x	four bytes (32 bits) per entry
 *
 * If a list of bytes has an odd number of bytes, the next entry in the list
 * must be word-aligned.
 *
 * The last header contains an address of 0, indicating there are no more entries to
 * be processed. This null header is inserted by the linker script; application code
 * should not define an entry with a null address.
 *
 * As a special case, if the count value is 0, then the address member is treated as
 * a function pointer, and there is no list of values following the header. In this case,
 * crt0 calls the function addressed by the pointer.
 *
 * Reminder: count is the number of values in the list, NOT a byte count!
 */

#ifndef	CRT0_H
#define	CRT0_h

#ifdef __cplusplus
extern "C" {
#endif

#include	<sys/sizes.h>
struct sCrt0InitEntryHeader
{
	void *address;
	WORD count;
} __attribute__ ((packed, aligned (2)));

/* values for the size bits in count */
#define	CRT0_INITBYTE	0
#define	CRT0_INITWORD	0x4000
#define	CRT0_INITLONG	0x8000

/* Initialization records must be stored in a linker section called '.crt0ini'. This is
 * handled with the macro CRT0_SECTION.
 */
#define CRT0_SECTION __attribute__ ((section (".crt0ini")))

/* Here's an example of how to define an init table entry:
 * #include "ram.h"
 * static struct sCrt0InitHeader InitRAM CRT0_INITATTRIBUTE = {&RAMBAR, CRT0_INITWORD | 1};
 * static WORD InitRAMBAR [] CRT0_INITATTRIBUTE = {0x200};
 */

/* crt0 also looks for a long word variable named crt0_flags. If such a variable is
 * defined, then the bit pattern in this variable is used by crt0 to control how it
 * initializes the system.
 */
/* 68K only: if the bit CRT0_HWINITHOOK_JMP is clear, then the system initialization function
 * 'hardware_init_hook' (if defined) is called using a normal JSR/RTS sequence. If this
 * bit is set, the function is entered by a JMP instruction, with the return address
 * saved in register D0; the function must return by executing a JMP instruction to the
 * return address stored in register D0.
 */

#define	CRT0_HWINITHOOK_JMP	1

/* If the bit CRT0_NOCLEARBSS is set, then the '.bss' uninitialized data section
 * will NOT be cleared to zero by crt0 (default is to write zeros to all locations in bss).
 */
#define	CRT0_NOCLEARBSS		2

/* crt0 defines an integer variable, '__unhandled_exception', which indicates the cause of
 * entry into the C startup code. If this variable contains 0, then the program was entered
 * as a result of a hardware reset. If the value is positive but non-zero, then it indicates
 * the exception vector which caused entry into the default exception handler. The other
 * possible value is -1, which indicates that an exception caused entry into the startup code,
 * but there is no way to determine which exception it was. This can only occur on the original
 * MC68000 processor or equivalent devices, since they do not put the exception vector number
 * on the stack.
 */

extern LONG __unhandled_exception;

/* Another variable, __unhandled_exception_pc, stores the value of the program counter at the
 * time the exception occurred.
 * Note: For the original MC68000, and equivalent devices (eg MC68EC000, MC68302, MC68306),
 * the value stored here is only valid for vectors 2 (bus error) and 3 (address error),
 * because it is not possible to get the program counter for other exception types.
 */
extern void * __unhandled_exception_pc;

#ifdef __cplusplus
}
#endif

#endif

/* end of crt0.h */

