/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* System-related configuration details for the API Shark platform */
/* Begun by Stig Telfer, Alpha Processor Inc, 8 September 2000 */

/*
 * $log:
 */

#define TRACE_ENABLE
#undef SHARK_P1_CONFIG		/* Define for Diags that boots Linux on P1 */
#undef SHARK_USE_ADMS		/* If for some reason these devices play up */

#include "lib.h"
#include "info.h"

#include "specifics.h"			/* platform-specific definitions */
#include "platform.h"			/* interface for this module */
#include "northbridge.h"		/* Generic NB access */
#include "southbridge.h"		/* Generic SB access */

#include "uilib.h"
#include "pci.h"

#include "osf.h"
#include "mcheck.h"
#include "cpu/ev6.h"
#include "cmos_rtc.h"
#include "beepcodes.h"
#include "hwrpb.h"			/* for systype settings */

#include "i2c.h"
#include "i2c/adm9240.h"
#include "i2c/pcf8574.h"

#include "tabledriver.h"

const unsigned plat_sys_type = ST_DEC_TSUNAMI;
#ifdef SHARK_P1_CONFIG
const unsigned plat_sys_variation = 1 << 10;	/* Emulate DP264 */
#else
const unsigned plat_sys_variation = 5 << 10;	/* Emulate ES40 / Clipper */
#endif

const char *Prompt = "Shark> ";

static const PCISlot_t pcihose0[] = { 
	{ PCI0_ALI_ISA, "ALI ISA bridge" },
	{ PCI0_ALI_IDE, "ALI IDE ctrllr" },
	{ PCI0_ALI_PMU, "ALI Power mgmt" },
	{ PCI0_ALI_USB, "ALI USB" },
	{ PCI0_SYM_SCSI,"Symbios SCSI" },
	{ PCI0_INTC_ETH0,"Intel primary" },
        { PCI0_SLOT1,	"PCI Slot 1" },
        { 0, 0 } };

static const PCISlot_t pcihose1[] = { 
        { PCI1_SLOT2,	"PCI Slot 2" },
	{ PCI1_INTC_ETH1,"Intel secondary" },
        { 0, 0 } };


/* note - ordering here is critical, it must match the bus numbering as 
 * decided by the PCI device tree building algorithm */

const PCIStructure_t PCIStructure[] = {
    { VARIANT_PCI, pcihose0 },			/* PCI hose 0 layout */
    { VARIANT_PCI, pcihose1 },			/* PCI hose 1 layout */
};

const short plat_pci_hose_count = ARRAYLEN( PCIStructure );





DBM_STATUS plat_setup( void )
{
    TRACE( "Running\n" );

    mobo_logf( LOG_DBG "STIG: before setup - CChip DRIR is 0x%016lX (%s)\n", 
	cchip[DRIR].r, cchip[DRIR].r & IRQ_SMI_M ? "SMI set" : "SMI clear" );

    /* Add early information about the system */
    info_submit( SRC_DIAGS, SUBJ_MOTHERBOARD, "Motherboard", "API CS20" );

    /* Set the yellow light to on */
    set_yellow_light( TRUE );

    mobo_logf( LOG_DBG "STIG: after setup - CChip DRIR is 0x%016lX (%s)\n", 
	cchip[DRIR].r, cchip[DRIR].r & IRQ_SMI_M ? "SMI set" : "SMI clear" );

    return STATUS_SUCCESS;
}


DBM_STATUS plat_fixup( void )
{
    unsigned long misc;
    uint8 epldrev, mbcfg, rdata;
    String interp;
    static const TblArray irq_fixups[] = {

	CFGWB( 0, PLAT_ISAID,		0,	PCIHDR_IRQ,	IRQ_INTR ),
	CFGWB( 0, PCI0_INTC_ETH0,	0,	PCIHDR_IRQ,	IRQ_ETH0 ),
	CFGWB( 0, PCI1_INTC_ETH1,	0,	PCIHDR_IRQ,	IRQ_ETH1 ),
	CFGWB( 0, PCI0_SYM_SCSI,	0,	PCIHDR_IRQ,	IRQ_SCSIA ),
	CFGWB( 0, PCI0_SLOT1,		0,	PCIHDR_IRQ,	IRQ_P0_SLOT1 ),
	CFGWB( 0, PCI1_SLOT2,		0, 	PCIHDR_IRQ, 	IRQ_P1_SLOT1 ),

	OP_EOT
    };

    mobo_logf( LOG_DBG "STIG: before fixup - CChip DRIR is 0x%016lX (%s)\n", 
	cchip[DRIR].r, cchip[DRIR].r & IRQ_SMI_M ? "SMI set" : "SMI clear" );

    TRACE( "Running\n" );

    /* Program in the device IRQs */
    TblWrite( irq_fixups );


    /* clear any outstanding NXM interrupts incurred during startup */
    /* This is achieved by writing 1 to Cchip CSR MISC:NXM if set */

    misc = cchip[ MISC ].r;
    if ( misc & misc_m_nxm )
    {
	cchip[ MISC ].r = misc_m_nxm;		/* write that bit for W1C */
	mb();					/* make sure it goes */
    }

    /* Enable relevant interrupts in the Tsunami interrupt mask register */
    /* These are the interrupts that SRM console enables for Linux */
    cchip[ DIM0 ].r = IRQ_INTR | IRQ_P0_ERROR | IRQ_P1_ERROR | IRQ_ERR_NXM;
    cchip[ DIM1 ].r = 0;


    /*
     * Information about Processors and Tsunami : 
     */
    mbcfg = tigrb( TIG_MBCFG );


    /* CPU clock frequency */
    /*   833 MHz = 0x6 */
    switch( ( mbcfg >> 3 ) & 0x7U )
    {
	case 0x6:
	    interp = "833 MHz";
	    break;

	default:
	    interp = "Unknown frequency setting!";
	    break;
    }
    info_submit( SRC_CHIPSET, SUBJ_CPU, "Processor clock frequency", interp );


    /* EV6 bus frequency */
    /*   75 MHz = 0x1 */
    /*   83 MHz = 0x4 */
    switch( mbcfg & 0x7U )
    {
	case 0x4:
	    interp = "83 MHz";
	    break;

	case 0x1:
	    interp = "75 MHz";
	    break;

	default:
	    interp = "Unknown frequency setting!";
	    break;
    }
    info_submit( SRC_CHIPSET, SUBJ_CHIPSET, "Tsunami/EV6 clock frequency",
		interp );


    /* L2 cache size */
    /*   4MB = 0x1 */
    switch( ( mbcfg >> 6 ) & 0x3U )
    {
	case 0x1:
	    interp = "4 MBytes";
	    break;

	default:
	    interp = "Unknown L2 cache size setting!";
	    break;
    }
    info_submit( SRC_CHIPSET, SUBJ_CPU, "L2 cache size", interp );


    /* Information about EPLDs and supporting logic */
    epldrev = tigrb( TIG_EPLDREV );

    /* Power EPLD */
    /*   03 = 64-0022-3A */
    /*   05 = 64-0022-5A */
    rdata = epldrev & 0xFU;
    switch( rdata )
    {
    	case 3:
	    info_submit( SRC_CHIPSET, SUBJ_CHIPSET, "Power EPLD Revision",
		"Rel 3 (API P/N 64-0022-3A)" );
	    break;

	case 5:
	    info_submit( SRC_CHIPSET, SUBJ_CHIPSET, "Power EPLD Revision",
		"Rel 5 (API P/N 64-0022-5A)" );
	    break;

	default:
	    info_submit( SRC_CHIPSET, SUBJ_CHIPSET, "Power EPLD Revision",
		"Unknown Rev %X", rdata );
	    break;
    }

    
    /* Reset EPLD */
    /*   00 = 64-0023-0A */
    rdata = epldrev >> 4;
    if ( rdata == 0 )
    {
	info_submit( SRC_CHIPSET, SUBJ_CHIPSET, "Reset EPLD Revision",
		"Rel 0 (API P/N 64-0023-0A)" );
    }
    else
    {
	info_submit( SRC_CHIPSET, SUBJ_CHIPSET, "Reset EPLD Revision",
		"Unknown Rev %X", rdata );
    }

    /* PCI Arbiter */
    /*   48 = Rev 2.8 (64-0001-0B) */
    rdata = tigrb( TIG_ARBREV );
    if ( rdata == 0x48 )
    {
	info_submit( SRC_CHIPSET, SUBJ_CHIPSET, "PCI Arbiter Revision",
		"Rel 2.8 (API P/N 64-0001-0B)" );
    }
    else
    {
	info_submit( SRC_CHIPSET, SUBJ_CHIPSET, "PCI Arbiter Revision",
		"Unknown Rev %02X", rdata );
    }

    /* TIG PAL */
    /*   8F = Rev 4.15 (64-0001-0B) */
    rdata = tigrb( TIG_PALREV );
    if ( rdata == 0x8F )
    {
	info_submit( SRC_CHIPSET, SUBJ_CHIPSET, "TIG PAL Revision", 
		"Rel 4.15 (API P/N 64-0001-0B)" );
    }
    else
    {
	info_submit( SRC_CHIPSET, SUBJ_CHIPSET, "TIG PAL Revision",
		"Unknown Rev %02X", rdata );
    }



    /* Log information about other integrated chips on the board */
    /* Symbios SCSI rev */
    info_submit( SRC_CHIPSET, SUBJ_CHIPSET, "Symbios SCSI Revision",
		"Rev %02X",
                pcicfgrb( 0, PCI0_SYM_SCSI, 0, PCI_CFG_REG_REVISION_ID ) );

    /* Ethernet 0 rev */
    info_submit( SRC_CHIPSET, SUBJ_CHIPSET, "Primary Ethernet Revision",
		"Rev %02X",
                pcicfgrb( 0, PCI0_INTC_ETH0, 0, PCI_CFG_REG_REVISION_ID ) );

    /* Ethernet 1 rev */
    info_submit( SRC_CHIPSET, SUBJ_CHIPSET, "Secondary Ethernet Revision",
		"Rev %02X",
                pcicfgrb( 0, PCI1_INTC_ETH1, 0, PCI_CFG_REG_REVISION_ID ) );


    /* PCI bus self-test results : FIXME need interpretation */
    info_submit( SRC_CHIPSET, SUBJ_MOTHERBOARD, "PCI bus self-test",
		"Bus 0: TIG says %d", tigrb( TIG_PCI0TST ) & 0x1U );

    info_submit( SRC_CHIPSET, SUBJ_MOTHERBOARD, "PCI bus self-test",
		"Bus 1: TIG says %d", tigrb( TIG_PCI1TST ) & 0x1U );


    /* Log the jumper setting behavioural code */
    info_submit( SRC_JUMPERS, SUBJ_MISC, "Software jumper setting",
		"Code 0x%02X", primary_impure->BEHAV_DATA & 0xFFU );


    mobo_logf( LOG_DBG "STIG: after fixup - CChip DRIR is 0x%016lX (%s)\n", 
	cchip[DRIR].r, cchip[DRIR].r & IRQ_SMI_M ? "SMI set" : "SMI clear" );

    return STATUS_SUCCESS;
}


DBM_STATUS plat_post( void ) 
{
    TRACE( "Running\n" );

    if ( smp_primary_cpu != 0 )
    {   
	BeepCode( beep_k_hw_mismatch );

        bring_up_console();

	mobo_logf(
	    LOG_WARN "POST: CPU 0 is either disabled or malfunctioning!\n"
	    LOG_WARN "POST: Boostrap processor is CPU %d\n", smp_primary_cpu );

        mobo_box( r_lrgapp, "Anomalous Processor Configuration" );
        printf_dbm(
            "Diags has detected that the primary processor in this machine\r"
            "is not CPU 0.  CPU 0 may be malfunctioning, or disabled by\r"
	    "jumper settings.  The system is capable of functioning\r"
            "without CPU 0, but operating systems may not expect this and\r"
            "unpredictable results may occur.\r"
            "Diags suggests that CPU 0 be used if possible.\r"
            "Press any key if you wish to continue anyway." );
 
        mobo_key( 0 );
    }

    return STATUS_SUCCESS;
}


DBM_STATUS plat_reset( int argc, char *argv[] )
{
    uint8 rdata;
    DBM_STATUS sval;

    /* Reset on shark is done via the I2C bus */

    /* We only need to initialise the controller, not configure the devices */
    /* So we don't call the platform I2C init routine here */
    i2cdev_init();
    sval = i2cdev_read( SHARK_I2C_REG0, -1, 1, &rdata );
    if ( sval == STATUS_FAILURE )
    {
	mobo_logf( LOG_WARN "I2C didn't read register correctly!\n" );
	/* attempt to press on anyway... */
    }

    TRACE( "I2C control register is currently 0x%02X\n", rdata );

    rdata &= ~SH_R0_RESET;		/* set the bit low for a pulse */

    TRACE( "Updating to 0x%02X...\n", rdata );
    sval = i2cdev_write( SHARK_I2C_REG0, -1, 1, &rdata );
    if ( sval == STATUS_FAILURE )
    {
	mobo_logf( LOG_WARN "I2C didn't write register correctly!\n" );
	/* attempt to press on anyway... */
    }

    msleep( 500 );			/* debug - let it set in? */

    rdata |= SH_R0_RESET;		/* Set the bit high again */

    TRACE( "Updating to 0x%02X...\n", rdata );
    sval = i2cdev_write( SHARK_I2C_REG0, -1, 1, &rdata );
    if ( sval == STATUS_FAILURE )
    {
	mobo_logf( LOG_WARN "I2C didn't write register correctly!\n" );
	/* attempt to press on anyway... */
    }

    mobo_logf( LOG_CRIT "Reset didn't work at all!\n" );
    return STATUS_FAILURE;			/* didn't reset OK */
}




/*----------------------------------------------------------------------*/
/* System status string, for printing out during memory stress test */
/* we have four lines of room here */

void plat_sysstat( String S )
{
    char p0stat[160], p1stat[160];
    unsigned ipl = swpipl( 7 );
    swpipl( ipl );                      /* find our IPL */

    /* Note: in error cases pchip_err_analyse_clear will print two lines */
    /* In which case we need to exchange \n for \r */
    pchip_err_analyse_clear( p0stat, 0 );
    pchip_err_analyse_clear( p1stat, 1 );
    ntor( p0stat );
    ntor( p1stat );

    sprintf_dbm( S,
	"PCI hose 0: %s\r"
	"PCI hose 1: %s\r"
	"System interrupts are at level %d (0=all enabled, 7=all disabled)",
		p0stat, p1stat, ipl );
}


/*----------------------------------------------------------------------*/
/* setup of all system interrupts - o = 0 (off), 1 (on) */

void plat_intsetup( int o ) 
{
    plat_intclr();                      /* wipe the slate clean beforehand */

    if ( o )                            /* enabling case */
    {
        mobo_logf( LOG_INFO "Enabling ISA interrupts...\n");
        outportb( 0x21, 0x00 );                     /* IRQs 0-7, active low */
        outportb( 0xA1, 0x00 );                     /* IRQs 8-15, active low */

	diags_subsys_stat( SYS_IRQ, DEV_SETUP );
    }
    else
    {
        /* disabling case */
        mobo_logf( LOG_INFO "Disabling ISA interrupts...\n");
        outportb( 0x21, 0xFF );                     /* IRQs 0-7, active low */
        outportb( 0xA1, 0xFF );                     /* IRQs 8-15, active low */
    }
}



/* Clear any pending interrupts (except machine checks) on this machine */


void plat_intclr( void )
{
    unsigned char iis0, iis1;
    int iters;
    int irq;
    int ipl;

    ipl = swpipl( 7 );                  /* noodling this stuff is dangerous */

#define MAX_ACKS        16

    /* until there are no more interrupts in service, send EOI signals */
    for ( iters=0; iters < MAX_ACKS; iters++)          /* 8 IRQ lines per reg */
    {
        irq = inIack();                         /* ack the int, returns IRQ */

        /* read Int in-service regs */
        outportb( 0x20, 0x0B );
        iis0 = inportb( 0x20 );
        outportb( 0xA0, 0x0B );
        iis1 = inportb( 0xA0 );

        if ( iis0 == 0 && iis0 == 0 )   break;

        if ( iis0 )      outportb( 0x20, 0x20 );                /* EOI */
        if ( iis1 )      outportb( 0xA0, 0x20 );                /* EOI */
    }

    swpipl( ipl );                      /* restore previous IPL */

    if ( iters >= MAX_ACKS )
        mobo_logf( LOG_CRIT "INTERRUPT: recurring interrupts - not clearing\n");
}



/* Clear any pending system machine checks (NMIs) */

void plat_nmiclr( void )
{
    int val;

    /* try to de-assert in the southbridge */

    /* Technique 1: */
    /* by writing this back, we hope to clear any nasty error flags set */

    val = pcicfgrw( 0, PCI0_ALI_ISA, 0, 0x6 );
    pcicfgww( 0, PCI0_ALI_ISA, 0, 0x6, val );


    /* Technique 2: */
    /* read port 61. bit 7=1 on NMI.  bit 2 ->1->0 to reset NMI */

    val = inportb( 0x61 );                        /* NMI ctrl/status port */
    val |= 0x0C;
    outportb( 0x61, val );                        /* write 1's to clear NMIs */
    val &= ~0x0C;
    outportb( 0x61, val );                        /* write 0's to re-enable */


    /* now attempt clearing of the NXM error state by writing 1 to MISC:NXM */
    cchip[ MISC ].r = misc_m_nxm;               /* write NXM bit to clear */
    mb();                                       /* make sure it goes */
}




/*----------------------------------------------------------------------*/
/* Non-maskable interrupt (NMI) analysis */
/* NMI on Tsunami systems can come from a Pchip, or the southbridge, 
 * Or the halt button assertion.  Or a system management interrupt.
 */

void plat_mchkinterp( String interp, int scb, LogoutFrame_t *L)
{
    unsigned long tsu_dir0;

    switch (scb) {

    	case SCB_Q_SYS_EVENT:			/* a power management event */

	    /* no system events defined for Shark as yet */

	    break;

	case SCB_Q_SYSMCHK:

	    tsu_dir0 = cchip[ DIR0 ].r;		/* read the interrupt request */

	    if ( tsu_dir0 & IRQ_P0_ERROR )
	    {
		pchip_err_analyse_clear( interp, 0 );
		break;
	    }

	    if ( tsu_dir0 & IRQ_P1_ERROR )
	    {
		pchip_err_analyse_clear( interp, 1 );
		break;
	    }

	    if ( tsu_dir0 & IRQ_ERR_NXM )
	    {
		cchip_nxm_analyse_clear( interp );
		break;
	    }

	    if ( tsu_dir0 & (IRQ_INTR | IRQ_NMI) )
	    {
		sprintf_dbm( interp, "MACHINE CHECK?  ISA interrupt?\n"
				     "No further analysis available in diags "
				     MOBO_VERSION );
		break;
	    }
	    /* If we reached here we drop down into the no-match case */

	default:
	    sprintf_dbm( interp, "System Machine Check (NMI)\n"
				 "No further analysis available in diags "
				 MOBO_VERSION );
	    break;
    }

}


/*----------------------------------------------------------------------*/
/* SMP support */

/* This procedure is not well-documented, I lifted it from the DP264 hardware
 * reference manual and by reading the SROM sources... */

void plat_smpstart( unsigned long palbase )
{
    volatile unsigned long *t0 = (unsigned long *)(DBM_SUPER | 0x100UL);
    volatile unsigned long *t1 = (unsigned long *)(DBM_SUPER | 0x108UL);
    
    /* attempt to start the other CPU - whether it is 0 or 1 */
    *t0 = palbase;
    asm("mb");
    *t1 = palbase;
    asm("mb");

    tigwb( TIG_CPU0START, 0 );
    tigwb( TIG_CPU1START, 0 );
}



/*----------------------------------------------------------------------*/
/* I2C access functions */

/* Bus layout data */
const I2Cbus_t plat_i2c_bus[] = {

#ifdef SHARK_USE_ADMS
      /* Dev */	/* Sensor */	/* Chip */	/* Addr */	/* priv. data */
    { I2C_SMD0,	I2CSENS_TEMP0,	I2CCHIP_ADM9240,	SHARK_I2C_SMD0, 0,
	"Primary heatsink temperature" },
    { I2C_SMD1,	I2CSENS_TEMP0,	I2CCHIP_ADM9240,	SHARK_I2C_SMD1, 0,
	"Secondary heatsink temperature" },
    { I2C_SMD2,	I2CSENS_TEMP0,	I2CCHIP_ADM9240,	SHARK_I2C_SMD2, 0,
	"Motherboard temperature" },
    { I2C_SMD3,	I2CSENS_TEMP0,	I2CCHIP_ADM9240,	SHARK_I2C_SMD3, 0,
	"IO section temperature" },

    { I2C_SMD0,	I2CSENS_FAN0,	I2CCHIP_ADM9240,	SHARK_I2C_SMD0, 0,
	"Front Fan 1 speed" },
    { I2C_SMD0,	I2CSENS_FAN1,	I2CCHIP_ADM9240,	SHARK_I2C_SMD0, 0,
	"Front Fan 2 speed" },
    { I2C_SMD1,	I2CSENS_FAN0,	I2CCHIP_ADM9240,	SHARK_I2C_SMD1, 0,
	"Front Fan 3 speed" },
    { I2C_SMD1,	I2CSENS_FAN1,	I2CCHIP_ADM9240,	SHARK_I2C_SMD1, 0,
	"Front Fan 4 speed" },
    { I2C_SMD2,	I2CSENS_FAN0,	I2CCHIP_ADM9240,	SHARK_I2C_SMD2, 0,
	"Front Fan 5 speed" },
#if 0		/* not connected on current (P2) CS20 boards? */
    { I2C_SMD2,	I2CSENS_FAN1,	I2CCHIP_ADM9240,	SHARK_I2C_SMD2, 0,
	"PSU Fan 3 speed" },
#endif
    { I2C_SMD3,	I2CSENS_FAN0,	I2CCHIP_ADM9240,	SHARK_I2C_SMD3, 0,
	"IO card fan 1 speed" },
    { I2C_SMD3,	I2CSENS_FAN1,	I2CCHIP_ADM9240,	SHARK_I2C_SMD3, 0,
	"IO card fan 2 speed" },

    { I2C_SMD0,	I2CSENS_5V,	I2CCHIP_ADM9240,	SHARK_I2C_SMD0, 0,
	"Primary module 5V supply" },
    { I2C_SMD1,	I2CSENS_5V,	I2CCHIP_ADM9240,	SHARK_I2C_SMD1, 0,
	"Secondary module 5V supply" },
    { I2C_SMD2,	I2CSENS_5V,	I2CCHIP_ADM9240,	SHARK_I2C_SMD2, 0,
	"Motherboard 5V supply" },
    { I2C_SMD3,	I2CSENS_5V,	I2CCHIP_ADM9240,	SHARK_I2C_SMD3, 0,
	"IO card 5V supply" },

    { I2C_SMD0,	I2CSENS_12V,	I2CCHIP_ADM9240,	SHARK_I2C_SMD0, 0,
	"Primary module 12V supply" },
    { I2C_SMD1,	I2CSENS_12V,	I2CCHIP_ADM9240,	SHARK_I2C_SMD1, 0,
	"Secondary module 12V supply" },
    { I2C_SMD2,	I2CSENS_12V,	I2CCHIP_ADM9240,	SHARK_I2C_SMD2, 0,
	"Motherboard 12V supply" },
    { I2C_SMD3,	I2CSENS_12V,	I2CCHIP_ADM9240,	SHARK_I2C_SMD3, 0,
	"IO card 12V supply" },

    { I2C_SMD0,	I2CSENS_2_5V,	I2CCHIP_ADM9240,	SHARK_I2C_SMD0, 0,
	"Primary module 2.5V supply" },
    { I2C_SMD1,	I2CSENS_2_5V,	I2CCHIP_ADM9240,	SHARK_I2C_SMD1, 0,
	"Secondary module 2.5V supply" },
    { I2C_SMD2,	I2CSENS_2_5V,	I2CCHIP_ADM9240,	SHARK_I2C_SMD2, 0,
	"Motherboard 2.5V supply" },

    { I2C_SMD0,	I2CSENS_3_3V,	I2CCHIP_ADM9240,	SHARK_I2C_SMD0, 0,
	"Primary module 3.3V supply" },
    { I2C_SMD1,	I2CSENS_3_3V,	I2CCHIP_ADM9240,	SHARK_I2C_SMD1, 0,
	"Secondary module 3.3V supply" },
    { I2C_SMD2,	I2CSENS_3_3V,	I2CCHIP_ADM9240,	SHARK_I2C_SMD2, 0,
	"Motherboard 3.3V supply" },
    { I2C_SMD3,	I2CSENS_3_3V,	I2CCHIP_ADM9240,	SHARK_I2C_SMD3, 0,
	"IO card 3.3V supply" },



#if 0
    /* FIXME: I need to be able to describe scaling factors for these sensors */
    { I2C_SMD0,	I2CSENS_VCCP1,	I2CCHIP_ADM9240,	SHARK_I2C_SMD0, 0,
	"Primary CPU core voltage" },
    { I2C_SMD1,	I2CSENS_VCCP1,	I2CCHIP_ADM9240,	SHARK_I2C_SMD1, 0,
	"Secondary CPU core voltage" },
    { I2C_SMD0,	I2CSENS_VCCP2,	I2CCHIP_ADM9240,	SHARK_I2C_SMD0, 0,
	"System interface term power" },
    { I2C_SMD1,	I2CSENS_VCCP2,	I2CCHIP_ADM9240,	SHARK_I2C_SMD1, 0,
	"System -12V level" },
    { I2C_SMD2,	I2CSENS_VCCP1,	I2CCHIP_ADM9240,	SHARK_I2C_SMD2, 0,
	"1.5V Power for VDDQ" },
    { I2C_SMD2,	I2CSENS_VCCP2,	I2CCHIP_ADM9240,	SHARK_I2C_SMD2, 0,
	"5V Standby level" },

    /* Chassis intrusion not yet implemented */
    { I2C_SMD0,	I2CSENS_CI,	I2CCHIP_ADM9240,	SHARK_I2C_SMD0, 0,
	"Chassis intrusion detector" },
#endif

#endif		/* SHARK_USE_ADMS */

    { I2C_MOBO_REV,   I2CSENS_EPROM,  I2CCHIP_PCF8582, SHARK_I2C_EPROM, 0,
	"Motherboard asset info" },

    { I2C_REG0,	I2CSENS_REGMUX,	I2CCHIP_PCF8574,	SHARK_I2C_REG0, 0,
	"Motherboard control" },

    { I2C_REG1,	I2CSENS_REGMUX,	I2CCHIP_PCF8574,	SHARK_I2C_REG1, 0,
	"Motherboard revision" },

    { I2C_REG2,	I2CSENS_REGMUX,	I2CCHIP_PCF8574,	SHARK_I2C_MUX, 0,
	"Control multiplexer" },

    { I2C_NODEV, 0, 0, 0, 0, NULL }			/* End marker */
};


/* Platform-specific I2C initialisations */
DBM_STATUS plat_i2cinit( void )
{
    DBM_STATUS sval;
    uint8 rdata;
    static BOOLEAN i2c_init_done = FALSE;


    if ( i2c_init_done == TRUE )
	return STATUS_SUCCESS;

    diags_subsys_stat( SYS_I2C, DEV_PROBING );

    /* Bring up the controller */
    sval = i2cdev_init();
    if ( sval == STATUS_FAILURE )
    {
	mobo_logf( LOG_CRIT "platform I2C initialisation failed\n" );
	diags_subsys_stat( SYS_I2C, DEV_FAILED );
	return sval;
    }


    /* DEBUG: take a look at the value of the SROM/COM selector bit */
    sval = i2cdev_read( SHARK_I2C_REG0, -1, 1, &rdata );
    if ( sval == STATUS_FAILURE )
    {
	mobo_alertf( "I2C access didn't work",
		"I2C didn't read serial control register correctly!\n" );
    }
    else 
    {
	mobo_logf( LOG_DBG "I2C: control reg 0x%02X is 0x%02X, "
			"meaning %s selected\n",
		SHARK_I2C_REG0, rdata, 
		(rdata & SH_R0_SERIAL) == SH_R0_COM ? "SERIAL" : "SROM" );
    }


#ifdef SHARK_USE_ADMS
    /* Bring up the individual devices */
    sval = adm_init( SHARK_I2C_SMD0 );
    if( sval == STATUS_FAILURE )
    {
	mobo_logf( LOG_CRIT "I2C: Primary processor monitor failed init!\n" );
	mobo_alertf("I2C bus error",
		"Primary processor monitor failed initialisation\n");
    } else {
	adm_add_dev( SHARK_I2C_SMD0, I2CSENS_TEMP0, 70*I2C_TEMP_SCALE, 65*I2C_TEMP_SCALE );
	adm_add_dev( SHARK_I2C_SMD0, I2CSENS_FAN0, 7800, 7000 );
	adm_add_dev( SHARK_I2C_SMD0, I2CSENS_FAN1, 7800, 7000 );
	adm_start( SHARK_I2C_SMD0 );
	msleep( 500 );
    }

    sval = adm_init( SHARK_I2C_SMD1 );
    if( sval == STATUS_FAILURE )
    {
	mobo_logf( LOG_CRIT "I2C: Secondary processor monitor failed init!\n" );
	mobo_alertf("I2C bus error",
		"Secondary processor monitor failed initialisation\n");
    } else {
	adm_add_dev( SHARK_I2C_SMD1, I2CSENS_TEMP0, 70*I2C_TEMP_SCALE, 65*I2C_TEMP_SCALE );
	adm_add_dev( SHARK_I2C_SMD1, I2CSENS_FAN0, 7800, 7000 );
	adm_add_dev( SHARK_I2C_SMD1, I2CSENS_FAN1, 7800, 7000 );
	adm_start( SHARK_I2C_SMD1 );
	msleep( 500 );
    }

    sval = adm_init( SHARK_I2C_SMD2 );
    if( sval == STATUS_FAILURE )
    {
	mobo_logf( LOG_CRIT "I2C: Motherboard monitor failed init!\n" );
	mobo_alertf("I2C bus error",
		"Motherboard monitor failed initialisation\n");
    } else {
	adm_add_dev( SHARK_I2C_SMD2, I2CSENS_TEMP0, 60*I2C_TEMP_SCALE, 55*I2C_TEMP_SCALE );
	adm_add_dev( SHARK_I2C_SMD2, I2CSENS_FAN0, 7800, 7000 );
	adm_add_dev( SHARK_I2C_SMD2, I2CSENS_FAN1, 7800, 7000 );
	adm_start( SHARK_I2C_SMD2 );
	msleep( 500 );
    }

    sval = adm_init( SHARK_I2C_SMD3 );
    if( sval == STATUS_FAILURE )
    {
	mobo_logf( LOG_CRIT "I2C: IO card monitor failed init!\n" );
	mobo_alertf("I2C bus error",
		"IO card monitor failed initialisation\n");
    } else {
	adm_add_dev( SHARK_I2C_SMD3, I2CSENS_TEMP0, 60*I2C_TEMP_SCALE, 55*I2C_TEMP_SCALE );
	adm_add_dev( SHARK_I2C_SMD3, I2CSENS_FAN0, 7800, 7000 );
	adm_add_dev( SHARK_I2C_SMD3, I2CSENS_FAN1, 7800, 7000 );
	adm_start( SHARK_I2C_SMD3 );
	msleep( 500 );
    }
#endif 		/* SHARK_USE_ADMS */

    i2c_init_done = TRUE;
    diags_subsys_stat( SYS_I2C, DEV_SETUP );
    return STATUS_SUCCESS;
}



/* Wire up the I2C bus (eg, by setting multiplexers) for this device */
/* For Shark, this is very straightforward, there's nothing to be done.
 * Processors are hard wired down onto the motherboard so all the devices
 * should be always present.
 */

DBM_STATUS plat_i2c_select( const I2Cbus_t *I )
{ 
    /* nothing */
    return STATUS_SUCCESS;
}




/*----------------------------------------------------------------------*/
/* Debugging LED writes */

#define LEDPORT 0x80

void outled( unsigned d)
{
    outportb(LEDPORT, d);
}


/*----------------------------------------------------------------------*/
/* Any Platform-specific Magic widgetry */

/* Select between COM port and SROM UART, which are multiplexed to the 9-pin 
 * connectors on the back of the box
 * We have a problem: on startup, we may be in SROM state (after powerup)
 * or we may be in COM state (after OS halt) so we have to deduce the state by
 * reading the I2C bus.  This isn't immediately available until PCI bus setup
 * has been done.
 * However, if we assume that the SROM is connected then we will get early 
 * output on startup (but nothing on OS halt).
 */


DBM_STATUS shark_serial_select( const int select )
{
    static uint8 state = SH_R0_SROM;		/* FIXME: ASSUMPTION? */
    static uint8 recursion = FALSE;
    static uint8 init_done = FALSE;
    uint8 newstate;
    uint8 rdata;
    DBM_STATUS sval;
    UIstate oldstate;


    /* If a secondary processor, we shouldn't go around switching stuff about */
    if ( !smp_primary() )
	return STATUS_SUCCESS;

    /* If we're recursing (we may be making console output that calls us),
     * silently return here before doing anything else */
    if ( recursion )
	return STATUS_SUCCESS;

    recursion = TRUE;


    if ( select == SHARK_COM )
	newstate = SH_R0_COM;
    else
	newstate = SH_R0_SROM;

    if( newstate == state )
    {
	recursion = FALSE;
	return STATUS_SUCCESS;			/* no action required */
    }

    
    /* Prevent any errors from blocking, which would obstruct mux flipping */
    oldstate = mobo_uistate( UI_BATCH );

    /* Bring the i2c bus online on first call */
    if ( !init_done )
    {
	nb_setup();		/* May need PCI bus configured for ALI part */
	i2cdev_init();		/* not doing platform setup - no need here */
	init_done = TRUE;
    } 

    sval = i2cdev_read( SHARK_I2C_REG0, -1, 1, &rdata );
    if ( sval == STATUS_FAILURE )
    {
#if 0
	mobo_alertf( "I2C access didn't work",
		"I2C didn't read register correctly!\n" );
	recursion = FALSE;
	mobo_uistate( oldstate );
	return sval;
#else
	/* If we failed, we try to guess the value in case a write works */
	rdata = 0xFFU;
#endif
    }

    /* Flip the bit */
    rdata &= ~SH_R0_SERIAL;
    rdata |= newstate;
    sval = i2cdev_write( SHARK_I2C_REG0, -1, 1, &rdata );

    msleep( 50 );		/* DEBUG BODGE DELAY FIXME necessary? */
    state = newstate;
    recursion = FALSE;
    mobo_uistate( oldstate );
    return sval;
}


/* Set (or clear) the yellow activity LED.  Currently this LED is set solid-on
 * for firmware operation, but must be cleared when diags is exited for other
 * firmware or OS.
 */

void set_yellow_light( const BOOLEAN on )
{
    static const TblArray light_on[] = {
	CFGSETB( 0, PLAT_PMUID, 0, PMU_DCGPIO, 1<<6 ),       /* Output GPIO */
	CFGCLRB( 0, PLAT_PMUID, 0, PMU_DOGPIO, 1<<6 ),       /* active low */
	OP_EOT
    };

    static const TblArray light_off[] = {
	CFGSETB( 0, PLAT_PMUID, 0, PMU_DCGPIO, 1<<6 ),       /* Output GPIO */
	CFGSETB( 0, PLAT_PMUID, 0, PMU_DOGPIO, 1<<6 ),       /* active low */
	OP_EOT
    };

    if( on )
	TblWrite( light_on );
    else
	TblWrite( light_off );
}

