/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* 
 * $Id:
 * Author:
 * 	Chris Gearing 14 June 1999 and Stig Telfer, 30 August 2000
 */

#include <stdarg.h>

#undef TRACE_ENABLE

#include "lib.h"
#include "uilib.h"
#include "platform.h"
#include "southbridge.h"

#include "i2c.h"
#include "i2c/adm9240.h"

/*------------------------------------------------------------------------*/
/* Private implementation details */


/* Conversions between ADM9240 temperature format and i2c module format */
#define ADM2TEMP(x)	((x) * I2C_TEMP_SCALE)
#define TEMP2ADM(x)	((x) / I2C_TEMP_SCALE)

#define TEMP_MAX	60		/* max temperature threshold */
#define	TEMP_HYST	55


/* Conversions between ADM9240 voltage format and i2c module format */
#define VOLT2ADM(v,nom) ( ( (v)==0) ? 0xFF : ( (192 * (v)) / (nom) ) )
#define ADM2VOLT(x,nom) ( ((x) * (nom)) / 192 )

/* For voltages we're accepting a threshold of +/- 5% (100%=192) */
#define VOLTAGE_UPPER	203
#define VOLTAGE_LOWER	182


/* Conversion macros between fan timer count and RPM */
#define COUNTVALID(rpm)		( rpm != 0x00 && rpm != 0xFF )
#define RPM2COUNT(rpm,div)	( (22500 * 60) / ( (rpm) * (div) ) )
#define COUNT2RPM(cnt,div)	( (22500 * 60) / ( (cnt) * (div) ) )



#ifdef TRACE_ENABLE
/* Debug: dump the ADM state */

static void adm_dump( const uint8 A )
{
    uint8 cfg, isr1, isr2, imr1, imr2;

    mobo_logf( LOG_DBG "ADM: state of adm9240 at 0x%02X\n", A );

    cfg = 0xFFU;
    i2cdev_read( A, ADM_CFG, 1, &cfg );
    mobo_logf( LOG_DBG "ADM:   config register  0x%02X\n", cfg );

    isr1 = isr2 = 0xFFU;
    i2cdev_read( A, ADM_INT1, 1, &isr1 );
    i2cdev_read( A, ADM_INT2, 1, &isr2 );
    mobo_logf( LOG_DBG "ADM:   Interrupt status 0x%02X 0x%02X\n", isr1, isr2 );

    imr1 = imr2 = 0xFFU;
    i2cdev_read( A, ADM_INTMASK1, 1, &imr1 );
    i2cdev_read( A, ADM_INTMASK2, 1, &imr2 );
    mobo_logf( LOG_DBG "ADM:   Interrupt mask   0x%02X 0x%02X\n", imr1, imr2 );
}
#endif		/* TRACE_ENABLE */


/* A handy widget to lookup data on a voltage sensor */

typedef struct {
    i2c_sensor sens;
    uint8 nom;				/* nominal voltage x10 */
    uint8 reg, llim, hlim;
} adm_volt_sensor_t;

static const adm_volt_sensor_t *sens2data( const i2c_sensor sens )
{
    static const adm_volt_sensor_t volttab[] = {
	{ I2CSENS_2_5V,  25,  ADM_2_5V,  ADM_2_5V_LLIM,  ADM_2_5V_HLIM },
	{ I2CSENS_3_3V,  33,  ADM_3_3V,  ADM_3_3V_LLIM,  ADM_3_3V_HLIM },
	{ I2CSENS_5V,    50,  ADM_5V,    ADM_5V_LLIM,    ADM_5V_HLIM },
	{ I2CSENS_12V,	120,  ADM_12V,   ADM_12V_LLIM,   ADM_12V_HLIM },
	{ I2CSENS_VCCP1, 20,  ADM_VCCP1, ADM_VCCP1_LLIM, ADM_VCCP1_HLIM },
	{ I2CSENS_VCCP2, 20,  ADM_VCCP2, ADM_VCCP2_LLIM, ADM_VCCP2_HLIM }
    };

    int i;

    for ( i=0; i<ARRAYLEN( volttab ); i++ )
    {
	if ( volttab[ i ].sens == sens )
	    return volttab + i;
    }
 
    mobo_logf( LOG_CRIT "ADM: INTERNAL ERROR - non-existent volt sensor???\n" );
    return NULL;
}




/* Read from nreads ADM9240 registers, indexed in src, results in dst */

static DBM_STATUS adm_read( const uint8 A, const uint8 *src, const int nreads,
				uint8 *dst )
{
    int i;
    DBM_STATUS sval;

    for( i=0; i<nreads; i++ )
    {
	sval = i2cdev_read( A, src[i], 1, dst + i );
	if ( sval == STATUS_FAILURE )
	{
	    mobo_logf( LOG_CRIT "I2C: ADM9240 read failed\n" );
	    return sval;
	}
    }
    return STATUS_SUCCESS;
}



/* The input vector is formatted as address0/data0, address1/data1 etc. */

static DBM_STATUS adm_write( const uint8 A, const uint8 *rawdata,
				const int nwrites )
{
    int i;
    const uint8 *val;
    DBM_STATUS sval;

    for( i=0, val=rawdata; i<nwrites; i++, val+=2 )
    {
	sval = i2cdev_write( A, val[0], 1, &val[1] );
	if ( sval == STATUS_FAILURE )
	{
	    mobo_logf( LOG_CRIT "I2C: ADM9240 write failed\n" );
	    return sval;
	}
    }
    return STATUS_SUCCESS;
}



/* Middle-level routines for obtaining measured quantities in the right units */

DBM_STATUS adm_therm_read( const uint8 A, i2c_therm_t *result )
{
    DBM_STATUS sval;

#define THERM_READ_BYTES 3
    static const uint8 therm_read_addrs[ THERM_READ_BYTES ] = { 
	ADM_TEMP,
	ADM_TEMP_HLIM,
	ADM_TEMP_HYST
    };
    uint8 therm_read_data[ THERM_READ_BYTES ]; 

    TRACE( "Running\n" );

    sval = adm_read( A, therm_read_addrs, THERM_READ_BYTES, therm_read_data );
    if ( sval == STATUS_FAILURE )
    {
	mobo_logf( LOG_CRIT "ADM9240: thermal sensor read failure at 0x%02X\n",
		A );
	return sval;
    }

    result->temp = therm_read_data[0] * I2C_TEMP_SCALE;
    result->max = therm_read_data[1] * I2C_TEMP_SCALE;
    result->hyst = therm_read_data[2] * I2C_TEMP_SCALE;

    TRACE( "Got temp = %d, max = %d, hyst = %d\n",
		therm_read_data[0], therm_read_data[1], therm_read_data[2] );

    return STATUS_SUCCESS;
}


DBM_STATUS adm_fan_read( const uint8 A, const i2c_sensor s, i2c_fan_t *fan )
{
    uint8 rdata;
    DBM_STATUS sval;
    uint8 addr, mask, hlim, fan_div=1;

    TRACE( "Running\n" );

    /* find the fan divisor */
    sval = i2cdev_read( A, ADM_FDIV, 1, &rdata );
    if ( sval == STATUS_FAILURE )
	return sval;
    TRACE( "FDIV register is 0x%02X\n", rdata );

    /* Which fan are we talking about? */
    if ( s == I2CSENS_FAN0 )
    {
	addr = ADM_FAN1, hlim = ADM_FAN1_HLIM;
	mask = FDIV_FAN1_MASK;

	rdata = rdata & mask;
	if ( rdata == FDIV_FAN1_8 )		fan_div = 8;
	if ( rdata == FDIV_FAN1_4 )		fan_div = 4;
	if ( rdata == FDIV_FAN1_2 )		fan_div = 2;
	if ( rdata == FDIV_FAN1_1 )		fan_div = 1;

	TRACE( "fan divisor for fan 1 is %d\n", fan_div );

    } else {
	addr = ADM_FAN2, hlim = ADM_FAN2_HLIM;
	mask = FDIV_FAN2_MASK;

	rdata = rdata & mask;
	if ( rdata == FDIV_FAN2_8 )		fan_div = 8;
	if ( rdata == FDIV_FAN2_4 )		fan_div = 4;
	if ( rdata == FDIV_FAN2_2 )		fan_div = 2;
	if ( rdata == FDIV_FAN2_1 )		fan_div = 1;

	TRACE( "fan divisor for fan 2 is %d\n", fan_div );
    }


    /* Find the fan speed threshold */
    sval = i2cdev_read( A, hlim, 1, &rdata );
    if ( sval == STATUS_FAILURE )
	return sval;
    if ( COUNTVALID( rdata ) )
    {
	fan->threshold = COUNT2RPM( rdata, fan_div );
    } else {
	mobo_logf( LOG_CRIT "ADM9240: Fan threshold CSR has bad value 0x%02X\n",
		rdata );
	fan->threshold = 0;
	return STATUS_FAILURE;
    }



    /* Read the fan speed measurement */
    sval = i2cdev_read( A, addr, 1, &rdata );
    if ( sval == STATUS_FAILURE )
	return sval;

    if ( COUNTVALID( rdata ) )
    {
	fan->rpm = COUNT2RPM( rdata, fan_div );
    } else {
	mobo_logf( LOG_WARN
		"ADM9240: Fan monitored at 0x%02X is not working (0x%02X)\n",
		A, rdata );
	fan->rpm = 0;
#if 0
	/* Should this really return an error or is it just a warning? */
	return STATUS_FAILURE;
#endif
    }

    TRACE( "read fan period of %d\n", rdata );
    return STATUS_SUCCESS;
}


DBM_STATUS adm_voltages( const uint8 A, const i2c_sensor s, i2c_volt_t *V )
{
    uint8 nominal = 0;
    uint8 rdata;
    DBM_STATUS sval;
    const adm_volt_sensor_t *sdata;

    TRACE( "Running on addr 0x%02X sensor %d...\n", A, s );

    sdata = sens2data( s );

    if ( sdata == NULL )
    {
	mobo_alertf( "Diags bizarreness",
	    "Asking a volt sensor about something that isn't a voltage" );
	return STATUS_FAILURE;
    }


    /* Did the caller override the default settings for the voltage readings? */
    if ( V->nominal != 0 )
	nominal = V->nominal;
    else
	nominal = sdata->nom;


    /* Read the value, upper limit and lower limit, and calibrate */
    sval = i2cdev_read( A, sdata->reg, 1, &rdata );
    if ( sval == STATUS_FAILURE )
	return sval;
    V->v = ADM2VOLT( rdata, nominal );

    sval = i2cdev_read( A, sdata->hlim, 1, &rdata );
    if ( sval == STATUS_FAILURE )
	return sval;
    V->max = ADM2VOLT( rdata, nominal );

    sval = i2cdev_read( A, sdata->llim, 1, &rdata );
    if ( sval == STATUS_FAILURE )
	return sval;
    V->min = ADM2VOLT( rdata, nominal );

    return STATUS_SUCCESS;
}



/* Enable a device's interrupt mask register bit by setting it low */

static DBM_STATUS enable_device( const uint8 A, const uint8 R, const uint8 M )
{
    uint8 maskval;
    DBM_STATUS sval;

    sval = i2cdev_read( A, R, 1, &maskval );
    if ( sval == STATUS_FAILURE )
	return sval;

    maskval &= ~M;				/* set the bit low to enable */
    sval = i2cdev_write( A, R, 1, &maskval );
    return sval;
}






/*------------------------------------------------------------------------*/
/* Public interface */

#if 1
DBM_STATUS adm_init( uint8 A )
{
    DBM_STATUS sval;
    uint8 rval;
    TRACE( "Running on ADM9240 at 0x%02X\n", A );


    i2cdev_read( A, ADM_CFG, 1, &rval );
    mobo_logf( LOG_DBG "ADM: Config = 0x%02X\n", rval );

    i2cdev_read( A, ADM_INTMASK1, 1, &rval );
    mobo_logf( LOG_DBG "ADM: Int mask 1 = 0x%02X\n", rval );

    i2cdev_read( A, ADM_INT1, 1, &rval );
    mobo_logf( LOG_DBG "ADM: Int stat 1 = 0x%02X\n", rval );

    i2cdev_read( A, ADM_INTMASK2, 1, &rval );
    mobo_logf( LOG_DBG "ADM: Int mask 2 = 0x%02X\n", rval );

    i2cdev_read( A, ADM_INT2, 1, &rval );
    mobo_logf( LOG_DBG "ADM: Int stat 2 = 0x%02X\n", rval );

    mobo_logf( LOG_INFO
	"Setup of ADM9240 system monitor at 0x%02X was successful\n", A );
    return STATUS_SUCCESS;
}


#else
DBM_STATUS adm_init( uint8 A )
{
    DBM_STATUS sval;
    uint8 rval;

    static const uint8 adm_init_vector[] = {

#if 0	/* PARANOIA: Enable this to first reset the chip to factory settings */
	ADM_CFG,	CFG_INIT,
#endif

#if 0	/* Enable this to start all sensor limits from sensible values */
	/* Default value initialisation vector */

	ADM_VCCP1_HLIM,	VOLTAGE_UPPER,
	ADM_VCCP1_LLIM,	VOLTAGE_LOWER,
	ADM_VCCP2_HLIM,	VOLTAGE_UPPER,
	ADM_VCCP2_LLIM,	VOLTAGE_LOWER,
	ADM_2_5V_HLIM,	VOLTAGE_UPPER,
	ADM_2_5V_LLIM,	VOLTAGE_LOWER,
	ADM_3_3V_HLIM,	VOLTAGE_UPPER,
	ADM_3_3V_LLIM,	VOLTAGE_LOWER,
	ADM_5V_HLIM,	VOLTAGE_UPPER,
	ADM_5V_LLIM,	VOLTAGE_LOWER,
	ADM_12V_HLIM,	VOLTAGE_UPPER,
	ADM_12V_LLIM,	VOLTAGE_LOWER,

	ADM_TEMP_HLIM,	TEMP_MAX,		/* may need to be dynamic */
	ADM_TEMP_HYST,	TEMP_HYST,		/* may need to be dynamic */

	ADM_FAN1_HLIM,	0xFF,			/* maximum fan period */
	ADM_FAN2_HLIM,	0xFF,			/* maximum fan period */
	ADM_FDIV,	FDIV_FAN1_8 | FDIV_FAN2_8,
#endif
	/* Disable all sensor interrupts */
	ADM_INTMASK1,   MASK1_2_5V |
			MASK1_VCCP1 |
			MASK1_3_3V |
			MASK1_5V |
			MASK1_TEMP |
			MASK1_FAN1 |
			MASK1_FAN2,

	ADM_INTMASK2,	MASK2_12V | MASK2_VCCP2 | MASK2_CHASSIS
    };
#define INIT_VECTOR_LENGTH	(sizeof(adm_init_vector) / (sizeof(uint8) * 2))


    TRACE( "Running on ADM9240 at 0x%02X\n", A );

#ifdef TRACE_ENABLE
    /* PARANOIA: check that the self address means something */
    rval = 0xFFU;
    i2cdev_read( A, ADM_SADDR, 1, &rval );
    TRACE( "Serial address is 0x%02X\n", rval );
#endif

    /* Program our limits for voltages and temperature */
    sval = adm_write( A, adm_init_vector, INIT_VECTOR_LENGTH );
    if ( sval == STATUS_FAILURE )
    {
	mobo_logf( LOG_CRIT
	    "I2C: ADM9240 at 0x%02X failed on setup of threshold parameters\n",
		A );
	return sval;
    }
    mobo_logf( LOG_INFO
	"Setup of ADM9240 system monitor at 0x%02X was successful\n", A );
    return STATUS_SUCCESS;
}

#endif


DBM_STATUS adm_add_dev( uint8 A, i2c_sensor sens, ... )
{
    va_list ap;
    uint8 temp_max, temp_hyst;
    uint16 nominal_rpm, minimum_rpm;
    uint8 fan_divisor, hlim_val, fdiv_mask, fdiv_val, hlim_reg, irq_mask, rdata;
    int min_voltage, max_voltage;
    uint8 hlim, llim;
    const adm_volt_sensor_t *sdata;
    DBM_STATUS sval;

    va_start(ap, sens);

    TRACE( "looking at addr 0x%02X, sensor %d\n", A, sens );
    switch( sens ) 
    {
	/* For temp sensors, supply a max temp and hysteresis temp */
	case I2CSENS_TEMP0:

	    temp_max = TEMP2ADM( va_arg( ap, int ));
	    sval = i2cdev_write( A, ADM_TEMP_HLIM, 1, &temp_max );
	    if ( sval == STATUS_FAILURE )
		return sval;

	    temp_hyst = TEMP2ADM( va_arg( ap, int ) );
	    sval = i2cdev_write( A, ADM_TEMP_HYST, 1, &temp_hyst );
	    if ( sval == STATUS_FAILURE )
		return sval;

	    enable_device( A, ADM_INTMASK1, MASK1_TEMP );
	    break;


	/* Here we provide the option to override defaults for volt sensors */
	/* Provide a minimum voltage and maximum voltage (both x10) */
	case I2CSENS_VCCP1:
	case I2CSENS_VCCP2:
	case I2CSENS_2_5V:
	case I2CSENS_3_3V:
	case I2CSENS_5V:
	case I2CSENS_12V:
	    sdata = sens2data( sens );
	    min_voltage = va_arg( ap, int );
	    max_voltage = va_arg( ap, int );

	    llim = VOLT2ADM( min_voltage, sdata->nom );
	    hlim = VOLT2ADM( max_voltage, sdata->nom );
	    i2cdev_write( A, sdata->llim, 1, &llim );
	    i2cdev_write( A, sdata->hlim, 1, &hlim );
	    // enable_device?
	    return STATUS_SUCCESS;


	/* For Fan RPM sensors, provide a nominal RPM and minimum RPM */
	case I2CSENS_FAN0:
	case I2CSENS_FAN1:
	    nominal_rpm = va_arg( ap, int );
	    minimum_rpm = va_arg( ap, int );

	    /* Choose an RPM scaling multiplier.  We have four to choose from */
	    fan_divisor = 8, fdiv_val = FDIV_FAN1_8 | FDIV_FAN2_8;
	    if ( nominal_rpm > 1100 )
		fan_divisor = 4, fdiv_val = FDIV_FAN1_4 | FDIV_FAN2_4;
	    if ( nominal_rpm > 2200 )
		fan_divisor = 2, fdiv_val = FDIV_FAN1_2 | FDIV_FAN2_2;
	    if ( nominal_rpm > 4400 ) 
		fan_divisor = 1, fdiv_val = FDIV_FAN1_1 | FDIV_FAN2_1;
	    if ( nominal_rpm > 8800 )
	    {
		TRACE( "Woh! Can't go above 8800 RPM (%d RPM wanted)\n",
			nominal_rpm );
		return STATUS_FAILURE;
	    }


	    if ( sens == I2CSENS_FAN0 )
	    {
		hlim_reg = ADM_FAN1_HLIM;
	 	fdiv_mask = FDIV_FAN1_MASK;
		irq_mask = MASK1_FAN1;
	    } else {
		hlim_reg = ADM_FAN2_HLIM;
		fdiv_mask = FDIV_FAN2_MASK;
		irq_mask = MASK1_FAN2;
	    }


	    /* Program the threshold RPM as a count value */
	    hlim_val = RPM2COUNT(minimum_rpm,fan_divisor);
	    sval = i2cdev_write( A, hlim_reg, 1, &hlim_val );
	    if ( sval == STATUS_FAILURE )
		return sval;

	    /* Program the Fan divisor in */
	    sval = i2cdev_read( A, ADM_FDIV, 1, &rdata );
	    if ( sval == STATUS_FAILURE )
		return sval;
	    rdata &= ~fdiv_mask;
	    fdiv_val &= ~fdiv_mask;
	    rdata |= fdiv_val;
	    sval = i2cdev_write( A, ADM_FDIV, 1, &rdata );
	    if ( sval == STATUS_FAILURE )
		return sval;


	    TRACE( "Fan speed %d, minimum %d, divisor 0x%02X, thresh %d\n",
		nominal_rpm, minimum_rpm, fdiv_val, hlim_val );

	    enable_device( A, ADM_INTMASK1, irq_mask );
	    TRACE( "Device enabled\n" );
	    break;


	default:
	    mobo_alertf( "Internal error",
		"ADM9240 received init request for unknown device %d\r"
		"Temp = %d, fans = %d, %d",
		sens, I2CSENS_TEMP0, I2CSENS_FAN0, I2CSENS_FAN1 );
	    sval = STATUS_FAILURE;
	    break;
    }

    va_end( ap );
    return sval;
}

/* Set the device sampling */
DBM_STATUS adm_start( uint8 A )
{
    uint8 startval = CFG_START;
    TRACE( "Starting ADM9240 sensor at 0x%02X\n", A );
    return i2cdev_write( A, ADM_CFG, 1, &startval );
}

