/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* Alpha Diagnostics Asset Info Diagnostic Functions */
/* Begun by Stig Telfer, API NetWorks, 21 February 2001 */


#undef TRACE_ENABLE

#include "lib.h"
#include "uilib.h"

#include "platform.h"
#include "i2c.h"
#include "i2c/pcf-eprom.h"

#include "asset.h"


/*----------------------------------------------------------------------*/
/* Private stuff */

#define EEPROM_PROMPT	"EEPROM> "

static eeprom_t *eeprom_list = NULL;


static void draw_ui( void )
{
    eeprom_t *E;
    char statbuf[ 64 ];
    int i;

    mobo_box( r_scrn, "Motherboard and Processor Asset Information" );
    printf_dbm( "Asset Information EEPROMs in this system:\r\r" );

    for( E = eeprom_list, i=0; E != NULL; E=E->next, i++ )
    {
	i2c_status( E->I, statbuf );
	printf_dbm( "%2d) %-28s : %s\r", i, E->I->desc, statbuf );
    }

    mobo_goto( p_help );
    printf_dbm( "Please select an EEPROM to examine or modify "
		"(or RETURN to quit):" );

    mobo_goto( p_prompt );
    printf_dbm( EEPROM_PROMPT );
}





static eeprom_t *new_e( const I2Cbus_t *I )
{
    eeprom_t *E = malloc( sizeof( eeprom_t ) );

    if ( E == NULL )
	return NULL;

    E->I = I;
    memset( E->data, 0, EEPROM_MAXLEN );
    E->next = NULL;

    return E;
}


/*----------------------------------------------------------------------*/

#define INPUTBUFLEN 	4

DBM_STATUS asset( int argc, char *argv[] )
{
    int eeprom_count=0;
    eeprom_t *E, *tmp;
    I2Cbus_t const *I; 
    char buf[ INPUTBUFLEN ];
    int rval;
    int i;
    char statbuf[ 64 ];
    DBM_STATUS sval;
    

    /* Screen setup */
    mobo_box( r_scrn, "Motherboard and Processor Asset Information" );
    printf_dbm( "Scanning EEPROMs in this system, please wait...\r\r" );

    /* Scan I2C data structures for EEPROMS.  If none found, report & quit */
    TRACE( "Scanning for EEPROMs\n" );
    for( I=plat_i2c_bus; I->dev != I2C_NODEV; I++ )
    {
	if ( I->sensor != I2CSENS_EPROM )
		continue;

	printf_dbm( "%2d) %-28s : ", eeprom_count, I->desc );

	TRACE( "Adding EEPROM '%s'\n", I->desc );
	E = new_e( I );

	/* Read the EEPROM only if it can be reached. */
	if ( plat_i2c_select( I ) == STATUS_SUCCESS )
	{
	    E = eeprom_scan( E );
	    if ( E == NULL )
	    {
		mobo_alertf( "Error while reading EEPROM",
		    "The EEPROM '%s'\r"
		    "Could not be read successfully", I->desc );
		printf_dbm( "\r" );
		continue;
	    }
	}

	/* Append to the END of the list (ie preserving the order) */
	if ( eeprom_list == NULL )
	{
	    eeprom_list = E;
	}
	else 
	{
	    for( tmp = eeprom_list; tmp->next != NULL; tmp=tmp->next )
		    /* Index along to end of list */;
	    tmp->next = E;
	}

	i2c_status( I, statbuf );
	printf_dbm( "%s\r", statbuf );
	eeprom_count++;
    }


    if ( eeprom_count <= 0 )
    {
	mobo_alertf( "No EEPROMs found",
		"This platform does not support asset information EEPROMS\r"
		"Or no functioning EEPROMS have been found" );
	return STATUS_FAILURE;
    }



    /* We have our EEPROMs copied to memory.  Now go interactive */
    do {
	draw_ui( );
	rval = mobo_input( buf, INPUTBUFLEN );
	if( rval == 0 )			/* Did the user hit return to quit? */
		break;

	/* What number did the user choose? */
	if( sscanf( buf, "%d", &rval ) != 1 )
		continue;

	/* Does it relate to a valid EEPROM? */
	if ( rval < 0 || rval > eeprom_count - 1 )
		continue;
	
	/* Now give the use a full look at this EEPROM */
	for( E = eeprom_list, i=0; i<rval; i++ )
		E = E->next;
	sval = eeprom_examine( E );

	if ( sval != STATUS_SUCCESS );	/* If it didn't work, don't edit */
		continue;

	rval = mobo_alertf( "Modify this EEPROM?",
		"Press 'y' to modify the contents of this EEPROM.\r"
		"Press any other key to return." );
	if ( tolower( rval ) != 'y' )
		continue;


	/* Modify the contents of the EEPROM */
	eeprom_edit( E );

    } while( TRUE ); 


    /* Free off those EEPROM data structures */
    E = eeprom_list;
    while ( E != NULL )
    {
	tmp = E;
	E = E->next;
	free( tmp );
    }
    eeprom_list = NULL;

    return STATUS_SUCCESS;
}



/*------------------------------------------------------------------------*/
/* Asset EEPROM integrity test: can they be written to and read back? */

static void setstate( const String msg )
{
    static Point status, last;
    static Rect statrgn;

    /* Set up our coordinates and regions */
    status.x = r_app.x + 2, status.y = r_app.y + 1;
    statrgn.x = status.x, statrgn.y = status.y, statrgn.w = 72, statrgn.h = 1;
    last = cursor;

    /* Zap the old state */
    mobo_zap( statrgn );

    /* Draw the new one */
    mobo_goto( status );
    printf_dbm( "Status: %s", msg );
}

#define RETRY_COUNT	4

static DBM_STATUS dotest( const i2c_device addr, uint8 wdata )
{
    uint8 rdata, sdata;
    DBM_STATUS sval;
    int retries=0;

    sval = pcfeprom_read( addr, EEPROM_MAXLEN-1, 1, &sdata );
    if ( sval == STATUS_FAILURE )
    {
	mobo_logf( LOG_CRIT
		"EEPROM: unable to read last byte of EEPROM at 0x%02X\n",
		addr );
	return sval;
    }

    pcfeprom_write( addr, EEPROM_MAXLEN-1, 1, &wdata );

    do {

	sval = pcfeprom_read( addr, EEPROM_MAXLEN-1, 1, &rdata );
	if ( sval != STATUS_FAILURE )
		break;

	retries++;
    } while ( retries < RETRY_COUNT );


    /* Restore saved value (at least, attempt to do so) and quit */
    pcfeprom_write( addr, EEPROM_MAXLEN-1, 1, &sdata );
    return sval;
}



DBM_STATUS eeprom_integrity( int argc, char *argv[] )
{
    char buf[64];
    Point list, current;
    int eeprom_count=0;
    DBM_STATUS sval1, sval2, final_sval = STATUS_SUCCESS;
    I2Cbus_t const *I;

    list.x = r_app.x + 2, list.y = r_app.y + 3;

    mobo_box( r_app, "Asset EEPROM integrity test" );
    setstate( "Searching for EEPROMs..." );

    for( I=plat_i2c_bus; I->dev != I2C_NODEV; I++ )
    {
        if ( I->sensor != I2CSENS_EPROM )
                continue;

	sprintf_dbm( buf, "Testing '%s'", I->desc );
	setstate( buf );

	current = list;
	current.y += eeprom_count;
	mobo_goto( current );
        printf_dbm( "%2d) %-30s : ", eeprom_count, I->desc );

	/* Dump some status about the EEPROM */
	sval1 = i2c_status( I, buf );
	if( sval1 == STATUS_FAILURE )
	{
	    printf_dbm( "FAILED on read!" );
	    mobo_alertf( "EEPROM test failure",
		"Asset Information EEPROM '%s' has failed.\r"
		"It could not be read", I->desc );
	    final_sval = STATUS_FAILURE;
	} 
	mobo_logf( "%sASSET: %s: %s\n",
		sval1==STATUS_FAILURE ? LOG_CRIT : LOG_INFO,
		I->desc, buf );

	/* Can we actually read and write this EEPROM? */
	if ( plat_i2c_select( I ) != STATUS_SUCCESS )
		continue;

	/* Now do our testing of the EEPROM - can we write and read back? */
	sval1 = dotest( I->addr, 0xAA );
	sval2 = dotest( I->addr, 0x55 );

	if ( sval1 == STATUS_FAILURE || sval2 == STATUS_FAILURE )
	{
	    printf_dbm( "FAILED on write!" );
	    mobo_logf( LOG_CRIT "ASSET: EEPROM '%s' FAILED the write test\n",
			I->desc );
	    mobo_alertf( "EEPROM test failure",
		"Asset Information EEPROM '%s' has failed.\r"
		"It failed a write/read test.", I->desc );
	    final_sval = STATUS_FAILURE;
	}
	else
	{
	    printf_dbm( "Passed." );
	}

	eeprom_count++;
    }

    if( final_sval == STATUS_FAILURE )
    {
	setstate( "EEPROM test has FAILED" );
	mobo_logf( LOG_CRIT "ASSET: Asset Information EEPROM test FAILED\n" );
	mobo_alertf( "EEPROM test Failure",
		"Errors were found, the test has failed." );
    }
    else
    {
	setstate( "EEPROM test has PASSED" );
	mobo_logf( LOG_INFO "ASSET: Asset Information EEPROM test PASSED\n" );
	mobo_alertf( "EEPROM test Passed",
		"All EEPROM tests passed successfully." );
    }

    return final_sval;
}

