/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* The functions in this access library make use of platform-dependent
 * ROM access mechanisms.  To keep things clean, those functions are in 
 * platform-specific files, eg, platform/nautilus.c
 */

#undef TRACE_ENABLE	 	/* define this to get verbose diagnosis */

#include "lib.h"
#include "uilib.h"
#include "specifics.h"
#include "platform.h"
#include "ledcodes.h"
#include "cserve.h"
#include "northbridge.h"

#include "rom.h"		/* the interface this file instantiates */


/*--------------------------------------------------------------------*/
/* Global data */

/* Firmware object lists, according to storage location */
rom_t *rom_phys_map = NULL, *rom_ram_map = NULL;

/* Have the firmware object scans been done? */
BOOLEAN rom_mapped = FALSE;


/*--------------------------------------------------------------------*/
/* Checksumming widgets */

unsigned short compute_checksuml( unsigned val, unsigned short total )
{
    int i, byte;

    /* checksum is computed on a byte by byte basis */
    for ( i=0; i<4; i++ )
    {
	byte = (val >> (i * 8)) & 0xFF;

	/* rotate */
	if ( total & 1 )	total = (total >> 1) + 0x8000;
	else			total = total >> 1;

	total += byte;
	total &= 0xFFFF;
    }
    return total;
}

unsigned short compute_checksumb( unsigned char val, unsigned short total )
{
    /* rotate */
    if ( total & 1 )	total = (total >> 1) + 0x8000;
    else			total = total >> 1;

    total += val;
    total &= 0xFFFF;

    return total;
}



/*--------------------------------------------------------------------*/
/* Firmware ID, colour and brief description records */

typedef struct {
    FWID F;
    String col;
    String name;
} FWdesc_t;

static const FWdesc_t img[] = {
{ FW_DBM,	FG_LBLUE,	"Debug Monitor" },	/* 0 debug monitor */
{ FW_WNT,	FG_YELLOW,	"ARC/NT BIOS" },	/* 1 (AlphaBIOS) */
{ FW_SRM,	FG_LGREEN,	"SRM Console" },	/* 2 SRM Console */
{ FW_FSB,	FG_WHITE,	"Failsafe boot" },	/* 6 Fail-safe booter */
{ FW_MILO,	FG_LRED,	"Linux Loader" },	/* 7 Linux loader */
{ FW_VXW,	FG_LRED,	"VxWorks RTOS" },	/* 8 VxWorks */
{ FW_DIAG,	FG_GREEN,	"Diagnostics" },	/* 9 Diagnostics */
{ FW_SR,	FG_LMAGENTA,	"SROM uBIOS" },		/* 10 SROM code */
{ FW_LINUX,	FG_BLUE,	"Linux Kernel" },	/* 11 firmware kernel */
{ FW_CBOX,	FG_BROWN,	"CBOX config" },	/* 12 CBOX config */
{ FW_OSFPAL,	FG_MAGENTA,	"OSF PALcode" },	/* 13 PALcode */
{ FW_NVRAM,	FG_CYAN,	"NVRAM data" },		/* 15 NVRAM */
{ FW_NVLOG,	FG_CYAN,	"Diags log" },		/* 16 NVRAM */
{ FW_NVENV,	FG_CYAN,	"Diags env" },		/* 17 NVRAM */
{ FW_INITRD,	FG_LCYAN, 	"Linux InitRD" },	/* 20 InitRD */

{ FW_EMPTY,	FG_BLACK,	"Free space" }		/* -2 Empty ROM */
};


String rom_I2colour( FWID F )
{
    unsigned i;
    for ( i=0; i < ARRAYLEN( img ); i++ )
        if ( img[i].F == F )
                return img[i].col;

    return FG_BLACK;			/* no match - defined to return black */
}

String rom_I2name( FWID F )
{
    unsigned i;
    for ( i=0; i < ARRAYLEN( img ); i++ )
        if ( img[i].F == F )
                return img[i].name;

    return "Unknown ID";		/* no match */
}


/*--------------------------------------------------------------------*/
/* Useful functions for manipulating headers */

/* From header data and location in ROM, return ROM address of body */
unsigned rom_body_base( const romheader_t *H, const unsigned romptr )
{
    if ( ROMH_VERSION( H ) == 3 )		/* Disjoint header and body */
    {
	return H->romh.V2.rom_offset & ~1U;
    }

    /* Body directly follows header */
    return romptr + H->romh.V0.hsize;
}


/* From header data, deduce the size of the ROM image */
unsigned rom_body_size( const romheader_t *H )
{
    if ( ROMH_VERSION( H ) == 0 )
    {
	/* Not in this kind of header - guess from uncompressed memory size */
	return H->romh.V0.size;
    }

    /* Otherwise refer to the V1 info */
    return H->romh.V1.rimage_size;
}


/* From header data (if available) deduce the FW ID of the ROM image */
FWID rom_fwid( const romheader_t *H )
{
    if ( ROMH_VERSION( H ) == 0 )
    {
	/* No way of knowing the FWID of a version 0 header */
	return FW_UNKNOWN;
    }

    /* Otherwise refer to the V1 info */
    return H->romh.V1.fw_id;
}


/* A function to parse a header into one of its multitudinous interpretations */
String rom_parseRevision( const romheader_t *H, String fwidstr )
{
    const cpq_fwid_t *C = &(H->romh.V1.fwoptid.fw_id_S);
    const api_fwid_t *A = &(H->romh.V1.fwoptid.api_id_S);
    char tmp[8];

    /* Cover the possibility that the header type is not matched */
    sprintf_dbm( fwidstr, "Unknown" );

    if ( (H == NULL) || (ROMH_VERSION( H ) == 0) )
    {
	return fwidstr;
    }


    /* Heuristic: If we're using an API part number, we know some things must
     * be true about the fields in the fwoptid structure:
     *
     * 1) The api_category is less than 100
     * 2) The api_rev is between 0 and 9 (inclusive)
     * 3) The api_letter is between 0 and 25 (inclusive)
     * 4) api_major and api_minor are both less than 100
     *
     * If the above are all true, we can guess its an API firmware part.
     * Otherwise, guess again...
     */

    if ( (A->api_category < 100)	&&
	 (A->api_major < 100)		&&
	 (A->api_minor < 100)		&&
	 (A->api_rev < 10)		&& 
	 (A->api_letter < 26) )
    {
	sprintf_dbm( fwidstr, "%d.%d-%d API P/N %02d-%02d%02d-%d%c",
		A->major, A->minor, A->rev, A->api_category, A->api_major,
		A->api_minor, A->api_rev, A->api_letter + 'A' );
    }
    else 
    {
	/* Some firmware IDs use a major-minor revision number sequence.
	 * Others use the 8 bytes as a short string.
	 * Here we sort the wheat from the chaff.
	 */
	switch( H->romh.V1.fw_id )
	{
	    case FW_WNT:		/* ARC / AlphaBIOS */
		tmp[ 0 ] = '\0';
		if( C->revision != 0) {
		    sprintf_dbm(tmp, "-%X", C->revision);
		}
		sprintf_dbm( fwidstr, "%X.%X%s",
			C->major_version, C->minor_version, tmp);
		break;

	    case FW_SRM:		/* SRM Console */
		sprintf_dbm( fwidstr, "%X.%X-%X",
			C->major_version, C->minor_version, C->revision);
		break;

	    case FW_DBM:		/* Compaq Debug Monitor (DBM) */
	    case FW_FSB:		/* FIXME: API FSB instead? */
		sprintf_dbm( fwidstr, "%X.%X",
			C->major_version, C->minor_version );

		break;

	    default:		/* Anything else: treat as string */
		sprintf_dbm( fwidstr, H->romh.V1.fwoptid.id );
		fwidstr[8] = '\0';		/* procrustean */
		break;
	}
    }
    return fwidstr;
}


/* Is a header (memory resident copy) a valid, well-formed ROM header? */
BOOLEAN rom_hdr_valid( const romheader_t *H )
{
    int i;
    unsigned r, csum;
    romheader_t local;		/* local copy sidesteps alignment issues */

    memcpy( &local, H, sizeof( romheader_t ) );

    /* Check for header magic */
    if( local.romh.V0.signature != ROM_H_SIGNATURE ||
	local.romh.V0.csignature != ~ROM_H_SIGNATURE )
    {
	return FALSE;
    }

    /* check the header checksum */
    for ( i=0, csum=0; i < sizeof(romheader_t)/sizeof(unsigned) - 1; i++ )
    {
	r = local.romh_array[i];
	csum = compute_checksuml( r, csum );
    }

    TRACE( "Checksum: 0x%04X (wanted 0x%04X)\n", csum, local.romh.hchecksum );

    return ( csum == local.romh.hchecksum );
}


/* Logs details about the ROM header supplied */
void rom_dump_hdr( const romheader_t *H )
{
    ui version;
    char fwidstr[ 32 ];

    if ( H == NULL )		/* Defensive */
	return;

    version = ROMH_VERSION( H );
    mobo_logf( LOG_INFO "ROM header (Version %d):\n", version );

    switch ( version )
    {
	case 3:
	case 2:
	    mobo_logf( LOG_INFO "  Body/requested ROM address: %d KB\n",
		       H->romh.V2.rom_offset >> 10 );
	    /* Drop through */

	case 1:
	    mobo_logf( LOG_INFO "  Firmware type: %d (%s)\n"
		       LOG_INFO "  Firmware revision: %s\n"
		       LOG_INFO "  Firmware size: %d KB\n",
		       H->romh.V1.fw_id, rom_I2name( H->romh.V1.fw_id ),
		       rom_parseRevision( H, fwidstr),
		       H->romh.V1.rimage_size >> 10 );
	    /* Drop through */

	default:
	case 0:
	    mobo_logf( LOG_INFO "  Memory base address: 0x%08X%08X\n"
		       LOG_INFO "  Checksums: header 0x%04X body 0x%04X\n",
		       H->romh.V0.destination.high,
		       H->romh.V0.destination.low,
		       H->romh.hchecksum, H->romh.V0.checksum );
	    break;
    }
}


/*--------------------------------------------------------------------*/
/* ROM Body manipulation routines */

/* Is an image (memory-resident copy) a valid, well-formed ROM image? */
BOOLEAN rom_body_valid( const romheader_t *h, const rombody_t *b )
{
    int i;
    unsigned csum;
    const unsigned char *body = (unsigned char *)b;

    TRACE( "Running\n" );

    /* check the body checksum */
    for ( i=0, csum=0; i < h->romh.V1.rimage_size; i++ )
    {
	csum = compute_checksumb( body[i], csum );
    }

    TRACE( "Checksum was 0x%04X (wanted 0x%04X)\n", csum, h->romh.V0.checksum );

    return ( csum == h->romh.V0.checksum );
}





/*------------------------------------------------------------------------*/
/* Building and freeing of rom_t lists */


/* Create a new rom_t data structure (allocated on the heap) */
static rom_t *new_rom_t( unsigned hstart, unsigned rstart, unsigned rsize,
			 char fwid, unsigned char flags )
{
    rom_t *R;

    R = malloc( sizeof( rom_t ) );
    if ( R == NULL ) {
	mobo_logf( LOG_CRIT "ROM: couldn't continue - no heap memory!\n" );
	return NULL;
    }
    memset( R, 0, sizeof( rom_t ) );	/* blank all fields in struct */

    R->hstart = hstart;
    R->rstart = rstart;
    R->rsize = rsize;
    R->astart = plat_romlower( rstart );
    R->asize = plat_romupper( rstart + rsize - 1 ) + 1 - R->astart;

    R->fwid = fwid;
    R->defcol = rom_I2colour( fwid );
    R->name = rom_I2name( fwid );
    R->flags = flags;

    R->next = NULL;
    R->hdr = NULL;		/* not necessarily null=0 */

    return R;
}



/* Real, proper header has been found & fully setup: insert it into list */
/* Ordering: Everything goes according to its position in the ROM, in 
 * ascending order.  BootMem images can get there first by starting at 0 */

static rom_t *insert_new_header( rom_t *Hnew, rom_t *src_list )
{
    rom_t *H=src_list, *Htmp;

    /* Is the list empty, or are we before the first entry? */
    if ( src_list == NULL || Hnew->rstart <= src_list->rstart )
    {
	Hnew->next = src_list;
	return Hnew;
    } 

    /* Or find the entry we go after */
    while ( H->next != NULL )
    {
	if ( H->next->rstart >= Hnew->rstart )
		break;				/* insert before next link */

	H = H->next;
    }

    /* At this point either H->next is NULL, or its greater, so we insert */
    Htmp = H->next;
    H->next = Hnew;
    Hnew->next = Htmp;
    return src_list;					/* hurrah! */
}



/* Add rom_t structures describing empty headers to a list */
#define EMPTY_RGN_THRESHOLD (64U << 10)		/* not worth looking at below */

static rom_t *insert_empty_headers( rom_t *src )
{
    rom_t *R;
    int rgnsize, rgnstart;

    TRACE( "starting up...\n" );

    if ( src == NULL )		/* ROM is empty, everything is free */
    {				/* not impossible: user could have erased it */
	R = new_rom_t( 0, 0, plat_romsize, FW_EMPTY, ROM_EMPTYRGN );
	if ( R == NULL )
	{
	    mobo_logf( LOG_CRIT "ROM: alloc failed during ROM processing\n" );
	    return src;
	}

	src = insert_new_header( R, src );
        return R;
    }


    /* Look for space before the first image */

    if ( src->astart >= EMPTY_RGN_THRESHOLD )
    {
	TRACE( "Adding %dKB before first header in ROM\n",
		(src->astart >> 10 ) );
	R = new_rom_t( 0, 0, src->astart, FW_EMPTY, ROM_EMPTYRGN );
	if ( R == NULL )
	{
	    mobo_logf( LOG_CRIT "ROM: alloc failed during ROM processing\n" );
	    return src;
	}

	src = insert_new_header( R, src );
    }


    /* Look for space between the images */

    for( R=src; R->next!=NULL; R=R->next )
    {
	rgnstart = R->astart + R->asize;	/* 1 beyond sector req'mnt */
	rgnsize = R->next->astart - rgnstart;

	/* Paranoia: we must check to see if the ROM images overlap, in
	 * which case we'll get silly numbers.  They might overlap because
	 * A firmware is flashed over the NVRAM area (for example), or 
	 * because there genuinely is badness in the ROM... */

	if ( rgnsize < 0 ) {
	    mobo_logf( LOG_WARN 
		"ROM: %s (at %dKB) and %s (at %dKB) overlap in the ROM!\n",
		R->name, R->astart>>10, R->next->name, R->next->astart>>10 );

	    continue;
	}


	TRACE( "Checking for available space at %dKB\n", (rgnstart>>10));

	/* while we're here, max out the image's available space setting */
	R->asize = R->next->astart - R->astart;

        if ( rgnsize >= EMPTY_RGN_THRESHOLD )
	{
	    TRACE( "Inserting %dKB empty space at %dKB\n",
		     (rgnsize >> 10), (rgnstart >> 10) );

	    R = new_rom_t( 0, rgnstart, rgnsize, FW_EMPTY, ROM_EMPTYRGN );
	    if ( R == NULL )
	    {
		mobo_logf( LOG_CRIT "ROM: alloc failed during ROM processing\n" );
		return src;
	    }

	    src = insert_new_header( R, src );
	}
    }

    /* Now we've looked between the images, what is left at the end? */
    rgnstart = R->astart + R->asize;
    rgnsize = plat_romsize - rgnstart;

    R->asize = plat_romsize - R->astart;	/* fix up last image too */

    if ( rgnsize >= EMPTY_RGN_THRESHOLD )
    {
	TRACE( "Appending %dKB empty space to end of ROM at %dKB\n",
		(rgnsize >> 10), (rgnstart >> 10) );

	R = new_rom_t( 0, rgnstart, rgnsize, FW_EMPTY, ROM_EMPTYRGN );
	if ( R == NULL )
	{
	    mobo_logf( LOG_CRIT "ROM: alloc failed during ROM processing\n" );
	    return src;
	}

	src = insert_new_header( R, src );
    }

    return src;
}





/* Index the physical ROM for images, which regenerates rom_phys_map */
DBM_STATUS rom_phys_scan( void )
{
    unsigned char *rom;
    unsigned i;
    romheader_t *H;
    rom_t *R;
    rombody_t *B;
    unsigned char flags;
    unsigned bodybase;
    unsigned hdrlen, bodylen;
    unsigned nhdrs = 0;

    TRACE( "Running...\n" );

    /* Allocate ROM buffer */
    rom = malloc( plat_romsize );
    if ( rom == NULL )
    {
	mobo_logf( LOG_CRIT "ROM: firmware scan failed: allocation error!\n" );
	return STATUS_FAILURE;
    }

    /* Copy the entire ROM into memory */
    for( i=0; i<plat_romsize; i++ )
    {
	rom[i] = plat_inromb( i );
    }

    /* Index the memory region for headers and check associated bodies */
    /* NB we cannot assume that body immediately follows header */
    for( i=0; i<plat_romsize - sizeof( romheader_t ); i++ )
    {
	H = (romheader_t *)(rom + i);

	if ( rom_hdr_valid( H ) )
	{
	    /* We have found a valid header! */
	    TRACE( "Valid header found at 0x%X\n", i );
	    nhdrs++;
	    flags = ROM_IMGCSUM;
	    bodybase = rom_body_base( H, i );
	    hdrlen = H->romh.V0.hsize;
	    bodylen = rom_body_size( H );
	    B = (rombody_t *)(rom + bodybase);

	    if ( rom_body_valid( H, B ) )
	    {
		flags |= ROM_IMGVALID;
	    }

	    TRACE( "allocating for new container structure\n" );
	    R = new_rom_t( i, bodybase, bodylen, rom_fwid( H ), flags );
	    
	    /* Make heap copies of header and body for linking in */
	    TRACE( "allocating for private copies\n" );
	    R->hdr = malloc( hdrlen );
	    R->body = malloc( bodylen );

	    if ( R->hdr == NULL || R->body == NULL )
	    {
		mobo_logf( LOG_CRIT "ROM: alloc failed on scan for images\n" );
		return STATUS_FAILURE;
	    }

	    TRACE( "Making private copies\n" );
	    memcpy( R->hdr, H, hdrlen );
	    memcpy( R->body, B, bodylen );

	    /* incorporate new header into the linked structure */
	   TRACE( "Adding new header into list\n" );
	   rom_phys_map = insert_new_header( R, rom_phys_map );
	}
    }


    /* Private copies of everything have been made for the linked list,
     * so we can just free off the ROM buffer here */
    TRACE( "releasing shadow-ROM copy\n" );
    free( rom );
    if ( nhdrs == 0 )
    {
	mobo_logf( LOG_WARN "ROM: found no firmware in the flash!\n" );
	return STATUS_WARNING;
    }

    rom_mapped = TRUE;
    return STATUS_SUCCESS;
}


/* Index reserved regions from the platform info */
DBM_STATUS rom_rsvd_scan( void )
{
    rom_t const *R = plat_rom_rsvd;
    rom_t *Rtmp;

    TRACE( "Transferring platform reserved regions...\n" );

    while( R != NULL )
    {
	mobo_logf( LOG_INFO "ROM: allocating for reserved region %s at 0x%X\n",
		R->name, R->rstart );
		
	/* make a duplicate copy whose pointers we can modify */
	Rtmp = malloc( sizeof( rom_t ));
	memcpy( Rtmp, R, sizeof( rom_t ) );
	Rtmp->next = NULL;		/* remove any trace of rsvd chain */
	rom_phys_map = insert_new_header( Rtmp, rom_phys_map );

	R = R->next;
    }
    return STATUS_SUCCESS;
}


/* Index the BootMem memory region for images, which regenerates rom_ram_map */
/* The pointer supplied is to the base of the BootMem region for searching */
DBM_STATUS rom_ram_scan( unsigned char *RAM, const size_t RAM_size )
{
    DBM_STATUS sval = STATUS_SUCCESS;
    unsigned i;
    rom_t *R;
    romheader_t *H;
    rombody_t *B;
    unsigned nhdrs=0;
    unsigned char flags;
    unsigned bodybase, bodylen, hdrlen;

    TRACE( "Running on %dKB at 0x%lX\n", RAM_size >> 10, RAM );

    /* Index the memory region for headers and check associated bodies */
    /* NB we cannot assume that body immediately follows header */
    for( i=0; i<RAM_size - sizeof( romheader_t ); i++ )
    {
	H = (romheader_t *)(RAM + i);

	if ( rom_hdr_valid( H ) )
	{
	    TRACE( "Valid header found\n" );

	    /* We have found a valid header! */
	    rom_dump_hdr( H );
	    nhdrs++;
	    flags = ROM_RAMIMG | ROM_IMGCSUM;
	    bodybase = rom_body_base( H, i );
	    hdrlen = H->romh.V0.hsize;
	    bodylen = rom_body_size( H );
	    B = (rombody_t *)(RAM + bodybase);

	    if ( rom_body_valid( H, B ) )
	    {
		flags |= ROM_IMGVALID;
	    }
	    else
	    {
		sval = STATUS_FAILURE;
	    }

	    R = new_rom_t( i, bodybase, bodylen, rom_fwid( H ), flags );
	    
	    /* Make heap copies of header and body for linking in */
	    R->hdr = malloc( hdrlen );
	    R->body = malloc( bodylen );

	    if ( R->hdr == NULL || R->body == NULL )
	    {
		mobo_logf( LOG_CRIT "ROM: alloc failed on scan for images\n" );
		return STATUS_FAILURE;
	    }

	    memcpy( R->hdr, H, hdrlen );
	    memcpy( R->body, B, bodylen );

	    /* incorporate new header into the linked structure */
	    rom_ram_map = insert_new_header( R, rom_ram_map );
	}
    }

    /* If there were no headers found, this is considered a warning issue */
    if ( nhdrs == 0 )
	return STATUS_WARNING;

    /* Otherwise return value depends on what condition the BootMem is in */
    return sval;
}


/* Clear rom_map, created by the above */
/* We have made it a policy that all structs referenced are heap-allocated */
/* NB it is now up to the caller to update rom_mapped etc. where appropriate */
DBM_STATUS rom_free( rom_t *R )
{
    rom_t *Rtmp;

    while ( R != NULL )
    {
	if ( R->hdr != NULL )		free( R->hdr );
	if ( R->body != NULL )		free( R->body );
	
	/* Skip onto the next list member, freeing current in the process */
	Rtmp = R;
	R = R->next;
	free( Rtmp );
    }

    return STATUS_SUCCESS;
}



/* takes an ID, returns a header struct.  A simple but handy calculation */
rom_t *rom_search( const FWID I, rom_t *R )
{
    while( R != NULL && R->fwid != I )		R = R->next;
    return R;		/* either returns NULL or the right match */
}



/* Given descriptor, load into memory [dest non-NULL to override ROM addr] */
/* Returns NULL if the load failed */

char *rom_load( rom_t *R, char *dst_ovr )
{
    unsigned long addr;
    char *start;

    /* validation - descriptor and corresponding romheader_t must be good */
    if ( R == NULL || R->hdr == NULL ) 		return NULL;

    TRACE( "Running on image of type %d flags 0x%X\n", R->fwid, R->flags );

    /* check for destination address override */
    if ( dst_ovr == NULL )
    {
	addr = R->hdr->romh.V0.destination.high;
	addr = (addr << 32) + R->hdr->romh.V0.destination.low;
	start = (char *)(DBM_SUPER | addr);
    }
    else start = dst_ovr;

    TRACE( "Chose destination of 0x%016lX\n", start );

    /* Perform copy from memory to memory */
    memcpy( start, R->body, R->rsize );
    return start;
}


/* Given descriptor, load and execute the firmware image */
DBM_STATUS rom_exec( rom_t *R )
{
    char *base;
    String name;

    TRACE( "Running on image at 0x%lX\n", R );

    if ( R == NULL )
	return STATUS_FAILURE;

    name = rom_I2name( R->fwid );
    base = rom_load( R, NULL );

    if ( base == NULL )
    {
	/* If we get here, the load failed */
	mobo_logf( LOG_CRIT "ROM: Load of firmware '%s' failed!\n", name );
	return STATUS_FAILURE;
    }

    mobo_logf( LOG_INFO "ROM: Loaded firmware image '%s' at 0x%X\n", 
		name, base );

#ifdef CONFIG_SHARK
    set_yellow_light( FALSE );			/* switch off the light! */
#endif

    imb();
    outled( LED_STARTING_BIOS );               /* trigger bus analyser */
    cServe( (unsigned long)base & ~DBM_SUPER, 0, CSERVE_K_JTOPAL);

    /* If we get the chance to return, it failed... */
    mobo_logf( LOG_CRIT "ROM: Exec of firmware '%s' failed!\n", name );
    return STATUS_FAILURE;
}


/* Diagnostics: Dump a ROM header's details to the log stream */

void rom_dump( const rom_t *R )
{
    BOOLEAN valid;

    if ( (R->flags & ROM_IMGCSUM) == 0 )
    {
	valid = TRUE;
    }
    else
    {
	valid = R->flags & ROM_IMGVALID;
    }

    mobo_logf( LOG_INFO "ROM header for image at offset %dKb:\n"
	       LOG_INFO "  Firmware state: %s\n",
	       R->rstart >> 10, valid ? "valid" : "CORRUPT" );

    if ( R->flags & ROM_PLATRSVD )
	mobo_logf( LOG_INFO "  This is a platform-reserved ROM region\n" );

    if ( R->flags & ROM_RAMIMG )
	mobo_logf( LOG_INFO "  This image is a ROM-in-RAM image\n" );

    if ( R->hdr != NULL )
	rom_dump_hdr( R->hdr );
}



/* High-level indexer routine, which cleans up any existing state, and rebuilds
 * an image of the physical ROM map, along with empty regions
 */
DBM_STATUS rom_index( void )
{
    int sval;

    /* Clear out any previously indexed state */
    rom_free( rom_phys_map );
    rom_phys_map = NULL;

    sval = rom_phys_scan( );
    rom_rsvd_scan( );
    rom_phys_map = insert_empty_headers( rom_phys_map );

    return sval;
}



/*--------------------------------------------------------------------*/
/* A quick check to see if a header is in one of the more likely places 
 * (ie a sector boundary).  Run this first when seeking a particular image
 * and it will save a few seconds load time.
 * Note: this only works properly for V1/V2 headers (ie, the normal case).
 */

rom_t *rom_quickscan( FWID I )
{
    rom_t *R = NULL;
    rombody_t *B;
    unsigned bsize;
    romheader_t *H;
    unsigned romptr;
    unsigned sig;
    unsigned i;

    TRACE( "Running\n" );

    /* Do we have a linked list of firmware headers already in place? */
    if ( rom_phys_map != NULL )
    {
	/* If it exists at all, try here first (although may be incomplete) */
	R = rom_search( I, rom_phys_map );
	if ( R != NULL )		/* get a match? */
	    return R;
    }

    H = malloc( sizeof( romheader_t ) );
    if ( H == NULL )			/* "No can do, my son" */
	return NULL;

    /* Note: in this loop we assume that the ROM is contiguous, which should
     * be a fair assumption but you never know... */
    for( romptr=0; romptr < plat_romsize; romptr=plat_romupper(romptr)+1 )
    {
	/* Look for the tell-tale signature bytes */
	sig = plat_inroml( romptr );
	if ( sig != ROM_H_SIGNATURE )
	    continue;

	/* At this point it may be worthwhile bringing the header into memory */
	TRACE( "Copying header into memory from ROM address 0x%X\n", romptr );
	for ( i=0; i<sizeof( romheader_t ) / sizeof(ui); i++ )
	{
	    H->romh_array[i] = plat_inroml( romptr + i*sizeof(ui) );
	}
	
	/* Did we really have a header or was it coincidence? */
	if ( !rom_hdr_valid( H ) )
	    continue;

	/* Can we handle this kind of header (ie, V1 or V2)? */
	i = ROMH_VERSION( H );
	if ( i!=1 && i!=2 )
	    continue;

	/* Check to see whether it is the right kind of firmware */
	TRACE( "Header is for image of type %d (%s)\n", 
		H->romh.V1.fw_id, rom_I2name( H->romh.V1.fw_id ) );
	if ( H->romh.V1.fw_id != I )
	    continue;

	/* We have a header of the right kind.  Copy the body into memory too */
	bsize = H->romh.V1.rimage_size;
	romptr += H->romh.V0.hsize;		/* Skip past header */
	B = malloc( bsize );
	if ( B == NULL )
	{
	    free( H );
	    return NULL;
	}
	
	for( i=0; i < bsize; i++ )
	{
	    B[i] = plat_inromb( romptr + i );
	}


	/* Now create the necessary holding structure */
	R = new_rom_t( plat_romlower(romptr), romptr, bsize, I,
			ROM_IMGCSUM );
	R->hdr = H;
	R->body = B;
	return R;
    }

    return NULL;
}


/*--------------------------------------------------------------------*/
