/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* HWRPB setup. (Hardware Reset Parameter Block)
 * This routine includes the initialisation of all data structures 
 * referenced by the HWRPB, which is defined in the Console Interface section
 * of the Alpha Architecture Reference Manual.
 */

#undef TRACE_ENABLE 

#include "lib.h"
#include "uilib.h"

#include "specifics.h"
#include "platform.h"

#include "osf.h"
#include "cpu/ev6.h"

#include "smp.h"
#include "hwrpb.h"


/*------------------------------------------------------------------------*/
/* HWRPB data structures and definitions - see Alpha Arch Ref Section III */

#define HWRPB_VALID_ID 0x4857525042000000UL

HWRPB_t *hwrpb = (HWRPB_t *)DIAGS_HWRPB;

/*------------------------------------------------------------------------*/
/* Private data and functions */

static smp_mutex hwrpb_lock = M_FREE;


static unsigned long hwrpb_compute_checksum( void )
{
    unsigned long sum = 0, *l;
    for (l = (unsigned long *)hwrpb; l < (unsigned long *)&hwrpb->chksum; ++l)
            sum += *l;

    return sum;
}

static void hwrpb_update_checksum( void )
{
    hwrpb->chksum = hwrpb_compute_checksum();
}       
 


/*------------------------------------------------------------------------*/
/* Interface implementation */

BOOLEAN hwrpb_valid( void )
{
    unsigned long hwrpb_addr = (unsigned long)hwrpb & ~DBM_SUPER;

    /* is phys_addr valid? */
    if ( hwrpb->phys_addr != hwrpb_addr )
    {
	return FALSE;
    }

    /* is the signature present? */
    if ( hwrpb->id != HWRPB_VALID_ID )
    {
	TRACE( LOG_CRIT "HWRPB signature corrupted, exp 0x%lx got 0x%lx\n",
		HWRPB_VALID_ID, hwrpb->id );
	return FALSE;
    }

    /* is the checksum valid? */
    if ( hwrpb->chksum != hwrpb_compute_checksum() )
    {
        TRACE( LOG_CRIT "HWRPB checksum corrupted, exp 0x%lx got 0x%lx\n",
                hwrpb_compute_checksum(), hwrpb->chksum );
        return FALSE;
    }

    return TRUE;			/* Everything's OK after all */
}


void hwrpb_dump_memcfg( void )
{
    memdesc_t *desc;
    memclust_t *clust;
    int i;

    desc = (memdesc_t *)( (char *)hwrpb + hwrpb->mddt_offset );
    clust = (memclust_t *)desc->cluster;

    mobo_logf( LOG_DBG "HWRPB: memory layout:\n" );

    for ( i=0; i< desc->numclusters; i++, clust++ )
    {
	mobo_logf( LOG_DBG
		"  %d: %d pages (%d Kb) at PFN %d (%d Kb), usage %d\n",
		i, clust->numpages, clust->numpages*PAGE_SIZE/(1<<10),
		clust->start_pfn, clust->start_pfn*PAGE_SIZE/(1<<10),
		clust->usage );
    }
}


void hwrpb_init(void)
{
    TRACE( "Running...\n" );
    smp_acquire( &hwrpb_lock );

    if ( hwrpb_valid() )
    {
	TRACE( "valid HWRPB already present, doing nothing...\n" );
	smp_release( &hwrpb_lock );
	return;
    }
    TRACE( "Creating a new HWRPB at 0x%lX\n", hwrpb );

    memset( (void *)hwrpb, 0, sizeof(HWRPB_t) );

    hwrpb->phys_addr = (unsigned long) hwrpb & 0xFFFFFFFFUL;
    hwrpb->id = HWRPB_VALID_ID;
    hwrpb->size = sizeof(HWRPB_t);
    hwrpb->pagesize = PAGE_SIZE;		/* Derive these values? */
    hwrpb->pa_bits = 44;
    hwrpb->max_asn = 255;

    /* Here we could deceive the kernel that it is being loaded from Milo.
     * Since the rest of this code is roughly copied from milo this is probably
     * a good idea since generic kernels need to know this... */

    strcpy(hwrpb->ssn, "DBLX-" MOBO_VERSION );

    /* these fields are critical: they determine machine vector selection! */
    hwrpb->sys_type = plat_sys_type;
    hwrpb->sys_variation = plat_sys_variation;

    /* Interval frequency, scaled by 4096 */
    hwrpb->intr_freq = HZ * 4096;
    hwrpb->cycle_freq = 1000000000000UL / primary_impure->CYCLE_CNT;
    hwrpb->nr_processors = 0;

    /* perCPU structs follow directly, here we zap them too */
    hwrpb->processor_offset = sizeof(HWRPB_t);	
    hwrpb->processor_size = sizeof(perCPU_t);
    memset( (char *)hwrpb + hwrpb->processor_offset, 0, 
		MAX_CPUS * hwrpb->processor_size );

    /* Configure the memory layout, which has the side-effect of building the
     * HWRPB's memory description structure
     */
    TRACE( "Setting up memory regions...\n" );
    mem_init();

    hwrpb_update_checksum();
    smp_release( &hwrpb_lock );		/* finished modifications */

    TRACE( "Done.\n" );
}



/*---------------------------------------------------------------------*/
/* Register my processor and fill in my processor-specific details */

void hwrpb_add( const int phys_id )
{
    perCPU_t *cpu;
    String cpurev;
    int cpurevno;
    unsigned *palbase;

    /* Build an HWRPB if one is not already present */
    if ( !hwrpb_valid() )
    {
	TRACE("CPU %d - HWRPB invalid, reconstructing!\n", phys_id );
	hwrpb_init();
    }

    cpu = (perCPU_t *)( (char *)hwrpb
		+ hwrpb->processor_offset 
		+ phys_id * hwrpb->processor_size );

    /* Am I already present? */
    if ( cpu->flags != 0UL )
    {
	TRACE( "Cpu %d already registered in HWRPB\n", phys_id );
	return;
    }

    smp_acquire( &hwrpb_lock );

    hwrpb->nr_processors++;		/* We are on-line */

    cpurevno = ev6_cpurevno( impure[phys_id]->I_CTL );
    if ( cpurevno > EV6__EV6_REVBASE )
	    cpu->type = HWRPB_CPU_EV6;
    if ( cpurevno > EV6__EV67_REVBASE )
	    cpu->type = HWRPB_CPU_EV67;

    cpurev = ev6_cpurev( impure[phys_id]->I_CTL );
    strncpy( (char *)cpu->serial_no, cpurev, 15 );

    /* Defined on page 2-21, section III, Alpha Architecture Reference */
    /* This is magic required by Linux, (and BIP set for startup signal) */
    cpu->flags = PERCPU_PALCODE_LOADED | PERCPU_MEM_VALID | PERCPU_PAL_VALID |
		 PERCPU_CPU_PRESENT | PERCPU_CPU_AVAIL;

    palbase = (unsigned *)(DBM_SUPER | impure[phys_id]->PAL_BASE);
    cpu->pal_revision = palbase[2];	/* major, minor encoded here */

    /* [many fields are still zero which may cause problems] */


    hwrpb_update_checksum();
    smp_release( &hwrpb_lock );

    /* Paranoia: Make sure its allocated */
    page_mark_range( ADDR2PFN(cpu, DBM_SUPER), HWRPB_CPU_PAGES, ALLOCATED_PAGE );
}



/*---------------------------------------------------------------------*/
/* Update the memdesc/memclust structures according to the mem_map */

void hwrpb_update_mem( char *mem_map, int map_pages )
{
    int pfn;
    char this_page;
    memclust_t *cluster;
    memdesc_t *memdesc;

    TRACE( "Running...\n" );
    smp_acquire( &hwrpb_lock );

    hwrpb->mddt_offset = hwrpb->processor_offset
		+ MAX_CPUS * hwrpb->processor_size;
    memdesc = (memdesc_t *)( (char *)hwrpb + hwrpb->mddt_offset );

    memset( (void *)memdesc, 0, sizeof( memdesc_t ) );
    cluster = (memclust_t *)memdesc->cluster;


    /*
     * Build clusters describing the clumps of memory that are available.
     */

    memdesc->numclusters = 1;
    cluster->start_pfn = 0;
    cluster->numpages = 1;
    this_page = mem_map[0];
    if ( this_page != ALLOCATED_PAGE && this_page != BOOTMEM_PAGE )
	this_page = FREE_PAGE;
    cluster->usage = this_page;

    for ( pfn=1; pfn < map_pages; pfn++ )
    {
	this_page = mem_map[pfn];
	if ( this_page != ALLOCATED_PAGE && this_page != BOOTMEM_PAGE )
	    this_page = FREE_PAGE;

        if ( this_page == cluster->usage)
	{
            cluster->numpages++;

        } else {

            /* new cluster */
            cluster++;
            cluster->usage = this_page;
            cluster->start_pfn = pfn;
            cluster->numpages = 1;
	    memdesc->numclusters++;
        }
    }


#ifdef TRACE_ENABLE
    hwrpb_dump_memcfg();
#endif

    hwrpb_update_checksum();
    smp_release( &hwrpb_lock );
    TRACE( "Done.\n" );
}


/* Look in the HWRPB memory structures for anything that is passed down from
 * the OS to be preserved by diags. */

void *hwrpb_bootmem_addr = NULL;
size_t hwrpb_bootmem_size = 0;

void hwrpb_update_mem_from_hwrpb( void )
{
    memdesc_t *desc;
    memclust_t *clust;
    int i;

    desc = (memdesc_t *)( (char *)hwrpb + hwrpb->mddt_offset );
    clust = (memclust_t *)desc->cluster;

    for ( i=0; i < desc->numclusters; i++, clust++ )
    {
	switch( clust->usage ) {

	    case BOOTMEM_PAGE:		/* copy this into our mem config */
		page_mark_range( clust->start_pfn,
				 clust->numpages,
				 BOOTMEM_PAGE );

		/* Note: this code would break if there was more than one
		 * bootmem region.  But if there was, it would be very odd... */
		hwrpb_bootmem_addr = PFN2ADDR( clust->start_pfn, DBM_SUPER );
		hwrpb_bootmem_size = clust->numpages * PAGE_SIZE;
		break;

	    default: 			/* otherwise do nothing */
		break;
	}
    }
}



static void percpu_dump( perCPU_t *P )
{
    mobo_logf(	LOG_DBG "perCPU at 0x%lX:\n"
		LOG_DBG "  Flags:     0x%lX\n"
		LOG_DBG "  Serial:    %s\n"
		LOG_DBG "  PAL rev:   0x%lX\n",
		P, P->flags, (char *) &(P->serial_no), P->pal_revision );
}


void hwrpb_dump( void )
{
    memdesc_t *memdesc;
    perCPU_t *percpu;
    int i;

    mobo_logf(	LOG_DBG "HWRPB at 0x%lx (%d bytes)\n"
		LOG_DBG "  ID:        %s\n"
		LOG_DBG "  Phys Addr: 0x%lX\n"
		LOG_DBG "  Nr CPUs:   %ld\n"
		LOG_DBG "  Pagesize:  %ld\n"
		LOG_DBG "  PA bits:   %ld\n"
		LOG_DBG "  Max ASN:   %ld\n"
		LOG_DBG "  SSN:       %s\n"
		LOG_DBG "  Sys type:  %ld\n"
		LOG_DBG "  Sys var:   %ld\n"
		LOG_DBG "  Intr Freq: %ld\n"
		LOG_DBG "  Cycle:     %ld\n",
		hwrpb, sizeof(HWRPB_t), (char *)&(hwrpb->id),
		hwrpb->phys_addr, hwrpb->nr_processors,
		hwrpb->pagesize, hwrpb->pa_bits, hwrpb->max_asn,
		hwrpb->ssn, hwrpb->sys_type, hwrpb->sys_variation,
		hwrpb->intr_freq, hwrpb->cycle_freq );

    percpu = (perCPU_t *)((char *)hwrpb + hwrpb->processor_offset);
    memdesc = (memdesc_t *)((char *)hwrpb + hwrpb->mddt_offset );

    for (i=0; i < MAX_CPUS; i++, percpu++ )
    {
	percpu_dump( percpu );
    }

    hwrpb_dump_memcfg();

    mobo_logf( LOG_INFO "Total memory used: 0x%0X bytes\n",
	sizeof( HWRPB_t ) + MAX_CPUS * sizeof( perCPU_t ) + sizeof( memdesc_t )
	+ memdesc->numclusters * sizeof( memclust_t ) );
}


HWPCB_t *hwrpb_gethwpcb( int cpuid )
{
    perCPU_t *cpu = (perCPU_t *)( (char *)hwrpb + hwrpb->processor_offset
                        + cpuid * hwrpb->processor_size);
    return (HWPCB_t *)cpu->hwpcb;
}


extern void secondary_polling_loop( unsigned long, void **,
                        unsigned long, unsigned long, unsigned long * );

void hwrpb_await_signal( unsigned long palbase, unsigned long vptbr )
{
    unsigned phys_id = smp_phys_id();
    perCPU_t *cpu = (perCPU_t *)((char *)hwrpb 
				+ hwrpb->processor_offset
				+ phys_id * hwrpb->processor_size );
    HWPCB_t *hwpcb = hwrpb_gethwpcb( phys_id );
    unsigned long raw_hwpcb = (unsigned long)hwpcb & ~DBM_SUPER;
    void **restart_ptr = (void **) &hwrpb->CPU_restart;

    TRACE( "Looking for restart address at 0x%x\n", restart_ptr );

    /* go straight to the assembler routine - we shouldn't actually be touching 
     * stack at this point but if we can't help it then best to get it out of 
     * the way sooner rather than later...
     */
    secondary_polling_loop( 	palbase, 
				restart_ptr,
				raw_hwpcb,
				vptbr,
				&cpu->flags );
}




uint64 percpu_getflags( const int cpuid )
{
    perCPU_t *cpu;

    if ( hwrpb_valid() != TRUE )
    {
	mobo_logf( LOG_WARN "HWRPB: attempt to read flags in invalid HWRPB!\n");
	return 0;
    }

    cpu = (perCPU_t *)((char *)hwrpb 
		+ hwrpb->processor_offset
		+ cpuid * hwrpb->processor_size );

    return cpu->flags;
}



uint64 percpu_setflags( const int cpuid, const uint64 setmask )
{
    perCPU_t *cpu;

    if ( hwrpb_valid() != TRUE )
    {
	mobo_logf( LOG_WARN "HWRPB: attempt to set flags in invalid HWRPB!\n");
	return 0;
    }
    cpu = (perCPU_t *)((char *)hwrpb 
		+ hwrpb->processor_offset
		+ cpuid * hwrpb->processor_size );

    cpu->flags |= setmask;
    return cpu->flags;
}



uint64 percpu_clrflags( const int cpuid, const uint64 clrmask )
{
    perCPU_t *cpu;

    if ( hwrpb_valid() != TRUE )
    {
	mobo_logf( LOG_WARN "HWRPB: attempt to set flags in invalid HWRPB!\n");
	return 0;
    }

    cpu = (perCPU_t *)((char *)hwrpb 
		+ hwrpb->processor_offset
		+ cpuid * hwrpb->processor_size );

    cpu->flags &= ~clrmask;
    return cpu->flags;
}

