/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* ALI M1543C Southbridge access routines */
/* (c) 1999 Alpha Processor Inc */
/* Begun 11 Feb 1999 by Stig Telfer, API */

#define TRACE_ENABLE		/* enable this for verbose southbridge debug */
#undef DUP_BYTE_FIXUP

#include "lib.h"
#include "uilib.h"
#include "info.h"

#include "platform.h"
#include "northbridge.h"
#include "southbridge.h"

#include "cmos_rtc.h"
#include "tabledriver.h"


#ifdef USE_PMU
static void ali_acpi_cfg( void );
#endif



/*----------------------------------------------------------------------*/
/* IO Operation tables */

static const TblArray set_up_sb[] = {

    CFGSETB( 0, PLAT_ISAID, 0, ISA_PIC, 0x67 ),		/* some optimisations */
    CFGSETB( 0, PLAT_ISAID, 0, ISA_ISAC1, 0x48 ),	/* 32-bit DMA */
    CFGSETB( 0, PLAT_ISAID, 0, ISA_ISAC2, 0x80 ),	/* port 92 reset */

#ifdef CONFIG_SHARK
    /* Remove PS/2 keyboard setting */
    CFGCLRB( 0, PLAT_ISAID, 0, ISA_IORC, 0x80 ),

    /* Disable USB controller (upper byte of 16-bit reg) */
    CFGCLRB( 0, PLAT_ISAID, 0, ISA_IOPW2+1, 0x40 ),
#endif

    /* No interrupt lines are connected */

    CFGWB( 0, PLAT_ISAID, 0, ISA_PIRT1, 0x00 ),
    CFGWB( 0, PLAT_ISAID, 0, ISA_PIRT2, 0x00 ),
    CFGWB( 0, PLAT_ISAID, 0, ISA_PIRT3, 0x00 ),
    CFGWB( 0, PLAT_ISAID, 0, ISA_PIRT4, 0x00 ),

    /* Disable the USB, AGP, SMBus, PMU & IDE IRQ routings */
    /* NB be aware of reserved bits */

    CFGWB( 0, PLAT_ISAID, 0, ISA_USBIR, 0x80 ),
    CFGCLRB(0,PLAT_ISAID, 0, ISA_AGPIS, 0x73 ),

    /* SMBus - disable the interrupt but set up as edge trigger input */
    CFGCLRB(0,PLAT_ISAID, 0, ISA_SMBIR, 0x0F ),
    CFGSETB( 0, PLAT_ISAID, 0, ISA_SMBIR, 0x10 ),

    CFGSETB( 0, PLAT_ISAID, 0, ISA_SCIIR, 0x90 ),	/* Disable, level->edge */
    CFGCLRB( 0, PLAT_ISAID, 0, ISA_IDENR1, 0xDF ),
    CFGCLRB( 0, PLAT_ISAID, 0, ISA_IDENR2, 0x1F ),

    /* All non-ISA interrupt sources should now be disabled */

    /* initialise the 8259 controllers via PC-IO space */

    IOWB( 0x20, 0x11 ),
    IOWB( 0x21, 0x00 ),                 /* IRQ0->7 are vectors 0->7 */
    IOWB( 0x21, 0x04 ),                 /* slave on IRQ 2 */
    IOWB( 0x21, 0x01 ),                 /* use x86 mode */

    IOWB( 0xA0, 0x11 ),
    IOWB( 0xA1, 0x08 ),                 /* IRQ8->15 are vectors 8->15 */
    IOWB( 0xA1, 0x02 ),                 /* slave on IRQ 2 */
    IOWB( 0xA1, 0x01 ),                 /* use x86 mode */

    /* disable all ISA interrupts */

    IOWB( 0x21, 0xFF ),                 /* ints 0-7 active low */
    IOWB( 0xA1, 0xFF ),			/* ints 8-F active low */

    /* Configure system programmable timer */

    IOWB( 0x43, 0x30 ),         /* timer 0 into mode 0 */
    IOWB( 0x40, 0x00 ),         /* send LSB of count */
    IOWB( 0x40, 0x00 ),         /* send MSB of count */

    OP_EOT
};


#ifdef USE_IDE
static const TblArray set_up_ide[] = {		/* Soohoon's setup code */

    CFGWB( 0, PLAT_ISAID, 0, ISA_IDEIC, 0x4D ),
    CFGWB( 0, PLAT_ISAID, 0, ISA_PIRT1, 0x31 ),
    CFGWB( 0, PLAT_ISAID, 0, ISA_PIRT2, 0xB9 ),
    CFGWB( 0, PLAT_ISAID, 0, ISA_PILET, 0x0F ),
    CFGWB( 0, PLAT_IDEID, 0, IDE_CMD, 0x05 ),

    /* writes to the northbridge's PCI configuration space */

    CFGWB( 0, 0, 0, 0x04, 0x06 ),
    CFGWB( 0, 0, 0, 0x5A, 0x07 ),
    CFGWB( 0, 0, 0, 0x56, 0xCC ),

    CFGWL( 0, PLAT_IDEID, 0, IDE_BA5, 0x0000F000 ),

    OP_EOT
};
#endif



#ifdef USE_PMU
static const TblArray set_up_pmu_acpi[] = {

    /* configure STPCLK break events */
    CFGWB( 0, PLAT_PMUID, 0, PMU_BES1,	0x12 ),
    CFGWB( 0, PLAT_PMUID, 0, PMU_BES1,	0x0 ),

    /* program ISA bridge to enable stopping of various device clocks */
    CFGWW( 0, PLAT_ISAID, 0, ISA_SMCC1,	0xF0E0 ),
    CFGWB( 0, PLAT_ISAID, 0, ISA_SCIIR,	0x81 ),	/* SCI: to IRQ9 */

    /* configure PMU */
    CFGWB( 0, PLAT_PMUID, 0, PMU_ATPC,	0x4F ),
    CFGWW( 0, PLAT_PMUID, 0, PMU_PTS,	0x0A69 ),
    CFGWB( 0, PLAT_PMUID, 0, PMU_PIIGS,	0x06 ),
    CFGWB( 0, PLAT_PMUID, 0, PMU_BES1,	0x12 ),	/* break events */
    CFGWB( 0, PLAT_PMUID, 0, PMU_PCCA,	0x04 ),
    CFGWB( 0, PLAT_PMUID, 0, PMU_DSC,	0x40 ),

    OP_EOT
};

#endif

/*----------------------------------------------------------------------*/
/* GPIO pin cache widgetry.  Every application using the GPIO bits must
 * use this cached copy and keep the cached copy up to date, or strange
 * things may happen...
 */

uint8 ali_gpio_data=0, ali_gpio_dirn=0;
uint8 ali_gpio_cached = FALSE;

void ali_load_gpio_cache( void )
{
    ali_gpio_dirn = pcicfgrb( 0, PLAT_PMUID, 0, PMU_DCGPIO );
    ali_gpio_data = pcicfgrb( 0, PLAT_PMUID, 0, PMU_DOGPIO );
    ali_gpio_cached = TRUE;
}

/*----------------------------------------------------------------------*/
/* Initialisation of ISA Bus and other devices hanging off a typical 
 * southbridge */

void sb_setup( void )
{
    mobo_logf( LOG_DBG "STIG: before sb setup - CChip DRIR is 0x%016lX (%s)\n",
        cchip[DRIR].r, cchip[DRIR].r & IRQ_SMI_M ? "SMI set" : "SMI clear" );

#ifdef USE_IDE
    TRACE( "Running IDE table\n");
    TblWrite( set_up_ide );	/* Enable the M5229 IDE interface */
#endif

    TRACE("Running set_up_sb\n" );
    TblWrite( set_up_sb );


    /* now do some device initialisations */

    diags_subsys_stat( SYS_ISA, DEV_PROBING );

    TRACE("Running AcerSIOInit\n" );
    AcerSIOInit();	/* SuperIO devices (embedded within Southbridge) */
    InitRTC();		/* Real time clock setup */

    diags_subsys_stat( SYS_ISA, DEV_SETUP );

    mobo_logf( LOG_DBG "STIG: after sb setup - CChip DRIR is 0x%016lX (%s)\n",
        cchip[DRIR].r, cchip[DRIR].r & IRQ_SMI_M ? "SMI set" : "SMI clear" );
}



/*----------------------------------------------------------------------*/
/* Fixups required during initialisation process */
void sb_fixup( void )
{
    unsigned char val; 
    int alirev;

    mobo_logf( LOG_INFO "Enabling Southbridge NMI...\n");

    val = inportb( 0x61 );                        /* NMI ctrl/status port */
    val |= 0x0C;
    outportb( 0x61, val );                        /* write 1's to clear NMIs */
    val &= ~0x0C;
    outportb( 0x61, val );                        /* write 0's to re-enable */

    /* Enable PCI bus parity check -> NMI */
    val = pcicfgrb( 0, PLAT_ISAID, 0, ISA_PIPM );
    val |= 1;
    pcicfgwb( 0, PLAT_ISAID, 0, ISA_PIPM, val );


#ifdef USE_PMU
    /* configure the power management unit */
    ali_acpi_cfg();
#endif

    ali_load_gpio_cache();


    /* Register information about this chip */
    alirev = pcicfgrb(0, PLAT_ISAID, 0, ISA_REV );
    info_submit( SRC_CHIPSET, SUBJ_CHIPSET, "ALI Southbridge Rev",
		"Rev %02X", alirev );


    mobo_logf( LOG_DBG "STIG: after sb fixup - CChip DRIR is 0x%016lX (%s)\n",
        cchip[DRIR].r, cchip[DRIR].r & IRQ_SMI_M ? "SMI set" : "SMI clear" );


    ali_smi();		/* STIG: FIXME DEBUG BODGE */
}


/*----------------------------------------------------------------------*/
/* ACPI configuration routine */

#ifdef USE_PMU
static void ali_acpi_cfg( void )
{
    unsigned b;

    /* configuration via ACPI according to the ALI 1543C ref p292 */
    mobo_logf( LOG_INFO "Configuring ACPI...\n" );

    /* ACPI BIOS initialise chipset registers: program ACPI/SMB IO addrs */

#if 0	/* PCI bus configuration has allocated resources */

    b = pcicfgrl( 0, PLAT_PMUID, 0x0, PMU_ABA ) & ~0x1U;
    mobo_logf( LOG_WARN "ACPI: base addr is 0x%X\n", b);

#else	/* choose base addresses ourselves */

    b = 0xDF00U;
    pcicfgwl( 0, PLAT_PMUID, 0x0, PMU_ABA, b );		/* PMU base = 0xDF00 */
    pcicfgwl( 0, PLAT_PMUID, 0x0, PMU_SBA, b + 0x80 ); 	/* SMB base = 0xDF80 */

    pcicfgwb( 0, PLAT_PMUID, 0x0, PMU_CMD, 0x01 );	/* enable IO response */

    b = pcicfgrl( 0, PLAT_PMUID, 0x0, PMU_ABA ) & ~0x1U;
    mobo_logf( LOG_WARN "ACPI: base addr is 0x%X\n", b);
#endif


    mobo_logf( LOG_WARN "writing ACPI config table\n" );
    TblWrite( set_up_pmu_acpi );			/* guts of config op */


    mobo_logf( LOG_WARN "Configuring ACPI in IO space\n" );

    /* enable ACPI clock control function */
    outportb( b + 0x11, 0x02 );

    /* Initialise ACPI features - see ALI doc p193 */

    outportw( b + 0x00, 0xFFFFU );	/* clear PM1_STS */
    outportw( b + 0x02, 0x0000U );	/* disable PM1_EN */

    outportw( b + 0x18, 0xFFFFU );	/* clear GPE0_STS */
    outportw( b + 0x1A, 0x0000U );	/* disable GPE0_EN */

    outportw( b + 0x1C, 0xFFFFU );	/* clear GPE1_STS */
    outportw( b + 0x1E, 0x0000U );	/* disable GPE1_EN */

}

#endif 


/*----------------------------------------------------------------------*/
/* I2C and System Management Bus support */

#define ALI_BUGGY_DELAY		5000		/* breather period in usec */
#define TIMEOUTVAL 1000000			/* in usec (a second!) */


/* Operations relating to the IO-space-mapped SMBus registers */
static uint32 smb_base_reg = 0;

static void smb_write( const int reg, const int val )
{
    //usleep( ALI_BUGGY_DELAY );
    outportb( smb_base_reg + reg, val );
    //inportb( smb_base_reg + reg );			/* DUMMY OP */
}

static int smb_read( const int reg )
{
    //usleep( ALI_BUGGY_DELAY );
    return inportb( smb_base_reg + reg );
}

static DBM_STATUS i2c_wait_ready( void );


static void i2c_error_analyse( void )
{
    int i;
    static const String interp[] = {
	"Host Slave Busy (slave is going to receive a command)",
	"Host Slave Status (SMI raised by slave interface)",
	"SMBus is at idle status",
	"Host Busy (SMBus host controller is completing cmd)",
	"SMI asserted by SMBus controller (after command)",
	"SMI asserted due to the generation of an error on I2C bus",
	"I2C Bus collision occurred (or no acknowledge from target)",
	"Terminated bus transaction in response to TERMINATE command"
    };
    uint8 status = smb_read( SMBUS_SMBSTS );

    mobo_logf( LOG_CRIT "ALI M1543C SMBus error analysis (0x%02X):\n", status );

    for ( i=0; i<8; i++ )
    {
	if ( status & (1<<i) )
	    mobo_logf( LOG_CRIT "  %s\n", interp[ i ] );
    }

#ifdef CONFIG_SHARK
    mobo_logf( LOG_DBG "I2C: CChip DRIR is 0x%016lX\n", cchip[DRIR].r );
#endif

}


/* Given an error condition, do our best to recover from it */
static DBM_STATUS i2c_error_recover( void )
{
    int elapsed;
    int retries;
    int rval;

#ifdef CONFIG_SHARK
    mobo_logf( LOG_DBG "I2C: CChip DRIR is 0x%016lX\n", cchip[DRIR].r );
#endif

    /* See if we can get back to idle without doing anything */
    if( i2c_wait_ready() != STATUS_FAILURE)
	return STATUS_SUCCESS;


    /* Bus timeout command - this is the most we can try? */
    mobo_logf( LOG_WARN "I2C: attempting entire bus timeout/reset\n" );
    smb_write( SMBUS_SMBCMD, SMBUS_SMBCMD_T_OUT_CMD ); 

    /* DEVICE_ERR signals end of T_OUT_CMD */
    elapsed = 0, retries = 0;
    while( (smb_read( SMBUS_SMBSTS ) & SMBUS_SMBSTS_DEVICE_ERR) == 0 )
    {
	smb_write( SMBUS_SMBSTS, 0xFFU );	/* clear faults */
	usleep( ALI_BUGGY_DELAY );
	elapsed += ALI_BUGGY_DELAY;
	if ( elapsed > TIMEOUTVAL )
	{
	    if( retries > 4 )
	    {
		mobo_logf( LOG_CRIT "I2C: giving up on bus timeout!\n" );
		break;
	    }
	    mobo_logf( LOG_WARN "I2C: response timed out, retrying (0x%02X)...\n", smb_read( SMBUS_SMBSTS ) );
	    TRACE( "IRQ ACK reads %X\n", inIack() );
	    smb_write( SMBUS_SMBCMD, SMBUS_SMBCMD_T_OUT_CMD ); 
	    smb_write( SMBUS_START, 0xFFU );

	    elapsed = 0;
	    retries++;
	}
    }
    smb_write( SMBUS_SMBSTS, 0xFFU );	 /* Clear the error */
    TRACE( "T_OUT_CMD completed\n" );

    /* See if we can get back to idle from here */
    if( i2c_wait_ready() != STATUS_FAILURE)
	return STATUS_SUCCESS;


    /* controller reset (a unilateral reset) */
    mobo_logf( LOG_WARN "I2C: attempting controller reset\n" );
    smb_write( SMBUS_SMBCMD, SMBUS_SMBCMD_TERMINATE ); 
    /* TERMINATE signals end of TERMINATE */
    while( (smb_read( SMBUS_SMBSTS ) & SMBUS_SMBSTS_TERMINATE) == 0 )
	/* poll again */;
    smb_write( SMBUS_SMBSTS, 0xFFU );	 /* Clear the error */

    TRACE( "TERMINATE completed\n" );

    /* See if we can get back to idle from here */
    if( i2c_wait_ready() != STATUS_FAILURE)
	return STATUS_SUCCESS;

    mobo_logf( LOG_CRIT "I2C: TERMINATE had no effect\n" );

    /* Bollocks!  Try disabling and re-enabling the controller */
    rval = pcicfgrb( 0, PLAT_PMUID, 0, PMU_SMBHSI );
    rval = rval & ~0x03U;
    pcicfgwb( 0, PLAT_PMUID, 0, PMU_SMBHSI, rval );

    /* Set speed to 37.5K with IDLE detect as BaseClk *128 (NB conservative). */
    rval = pcicfgrb( 0, PLAT_PMUID, 0, PMU_SMBHCBC );
    rval = (rval & 0x07U) | PMU_SMBHCBC_37K | PMU_SMBHCBC_DL128;
    pcicfgwb( 0, PLAT_PMUID, 0, PMU_SMBHCBC, rval );

    /* Re-enable the host interface */
    rval = pcicfgrb( 0, PLAT_PMUID, 0, PMU_SMBHSI );
    rval = (rval & ~0x03U) | PMU_SMBHSI_HOST_EN;
    pcicfgwb( 0, PLAT_PMUID, 0, PMU_SMBHSI, rval );


    mobo_logf( LOG_CRIT "I2C: unrecoverable fault on I2C bus!\n");
    return STATUS_FAILURE;
}



/* Wait until the I2C bus is idle */

static DBM_STATUS i2c_wait_ready( void )
{
    int n;
    int status, lastStatus;


    /* Wait for Bus Idle. */
    for ( n=0, status=0, lastStatus=-1; n < TIMEOUTVAL; n+= ALI_BUGGY_DELAY )
    {
	/* First, clear old status bits */
	smb_write( SMBUS_SMBSTS, 0xFFU );

	/* Read status. */
	status = smb_read( SMBUS_SMBSTS );

	/* Debug */
	if (status != lastStatus)
	{
	    lastStatus = status;
	}

	if ( status & SMBUS_SMBSTS_IDL_STS )
		return STATUS_SUCCESS;

	/* DEBUG: ALI SMBus controller needs a breather? */
	usleep( ALI_BUGGY_DELAY );
    }

    /* Timeout condition */
    mobo_logf( LOG_CRIT "ALI M1543C SMBus is locked up busy.\n");
    return STATUS_FAILURE;
}


/* Wait until a transaction has been completed */

static DBM_STATUS i2c_wait_complete( void ) 
{
    int n;
    uint8 status, lastStatus;

    for ( n=0, status=0, lastStatus=0xFF; n < TIMEOUTVAL; n += ALI_BUGGY_DELAY )
    {
        /* Read status. */
        status = smb_read( SMBUS_SMBSTS );

	/* DEBUG - ALI controller may need short breather before ready? */
	usleep( ALI_BUGGY_DELAY );

#if 0
        /* Debug */
        if (status != lastStatus)
        {
            lastStatus = status;
            TRACE("status=%x\n", status);
        }
#endif

#if 0
	/* Soohoon magic - check for bus idle when it should be completing */
	if ( status & SMBUS_SMBSTS_IDL_STS )
	{
	    mobo_logf( LOG_CRIT "ALI: controller dropped transaction!\n" );
	    i2c_error_analyse( );
	    i2c_error_recover( );
	    return STATUS_FAILURE;
	}
#endif

 	/* Check for presence of any errors */
	if ( status & SMBUS_SMBSTS_ERROR )
	{
	    i2c_error_analyse( );
	    i2c_error_recover( );
	    return STATUS_FAILURE;
	}

        if ( status & SMBUS_SMBSTS_SMI_I_STS )	/* completed successfully */
	{
	    smb_write( SMBUS_SMBSTS, 0xFF );
	    return STATUS_SUCCESS;
	}
    }

    /* Timeout condition */
    mobo_logf( LOG_CRIT "ALI SMBus timed out on completing transaction.\n");
    i2c_error_analyse( );
    i2c_error_recover( );
    return STATUS_FAILURE;
}


/* The ALI M1543C southbridge supports one i2c bus interface.  No knowledge
 * of device target addresses, wiring, multiplexers etc is in here.  Just the
 * reading and writing of a sequence of bytes...
 */

DBM_STATUS i2cdev_init( void )
{
    uint16 val16;
    uint8 rval;

    mobo_logf( LOG_DBG "STIG: before I2C init - CChip DRIR is 0x%016lX (%s)\n",
        cchip[DRIR].r, cchip[DRIR].r & IRQ_SMI_M ? "SMI set" : "SMI clear" );


    TRACE( "Looking at ALI SMB config...\n");

    /* Enable the PMU to respond to IO space transactions */
    /* NOTE: should I do this and what if the PMU doesn't respond to IO? */
    val16 = pcicfgrw( 0, PLAT_PMUID, 0, PMU_CMD );
    if ( (val16 & 1) == 0 )
    {
	mobo_logf( LOG_CRIT "PMU was not enabled for IO transactions!\n" );
	val16 |= 1;
	pcicfgww( 0, PLAT_PMUID, 0, PMU_CMD, val16 );
    }

#if 0
    /* Set speed to 223K with IDLE detect as BaseClk *32 (experimental setup) */
    rval = pcicfgrb( 0, PLAT_PMUID, 0, PMU_SMBHCBC );
    rval = (rval & 0x07U) | PMU_SMBHCBC_223K | PMU_SMBHCBC_DL32;
    pcicfgwb( 0, PLAT_PMUID, 0, PMU_SMBHCBC, rval );

    /* Set speed to 74K with IDLE detect as BaseClk *64 (typical setup). */
    rval = pcicfgrb( 0, PLAT_PMUID, 0, PMU_SMBHCBC );
    rval = (rval & 0x07U) | PMU_SMBHCBC_74K | PMU_SMBHCBC_DL64;
    pcicfgwb( 0, PLAT_PMUID, 0, PMU_SMBHCBC, rval );

#else
    /* Set speed to 37.5K with IDLE detect as BaseClk *128 (NB conservative). */
    rval = pcicfgrb( 0, PLAT_PMUID, 0, PMU_SMBHCBC );
    rval = (rval & 0x07U) | PMU_SMBHCBC_37K | PMU_SMBHCBC_DL128;
    pcicfgwb( 0, PLAT_PMUID, 0, PMU_SMBHCBC, rval );

#endif

    smb_base_reg = pcicfgrl( 0, PLAT_PMUID, 0, PMU_SBA ) & ~0x3U;
    TRACE("smb_base_reg=%x\n", smb_base_reg);
    // smb_write( SMBUS_SMBSTS, 0xFF );

#if 0
    /* Drain anything held in the I2C data registers */
    TRACE( "I2C data registers are 0x%02X, 0x%02X\n",
	smb_read( SMBUS_DATA_A ), smb_read( SMBUS_DATA_B ) );
    TRACE( "I2C data registers are 0x%02X, 0x%02X\n",
	smb_read( SMBUS_DATA_A ), smb_read( SMBUS_DATA_B ) );
#endif

    /* Enable SMB Bus Controller (NB attention to reserved bits) */
    rval = pcicfgrb( 0, PLAT_PMUID, 0, PMU_SMBHSI );
    rval = (rval & ~0x03U) | PMU_SMBHSI_HOST_EN;
    pcicfgwb( 0, PLAT_PMUID, 0, PMU_SMBHSI, rval );


    /* Soohoon magic... */
    pcicfgrb( 0, PLAT_PMUID, 0, PMU_SMBHSC ); 
    pcicfgrb( 0, PLAT_PMUID, 0, PMU_ATPC );

    /* Cancel anything outstanding and reset the host controller */
    smb_write( SMBUS_SMBCMD, SMBUS_SMBCMD_TERMINATE );

#ifdef CONFIG_SHARK
    mobo_logf( LOG_DBG "STIG: after i2c - CChip DRIR is 0x%016lX\n",
		cchip[DRIR].r );
#endif

    return STATUS_SUCCESS;
} 




/* I2C bus read.  The bus numberings used here start at zero.
 * Read a byte from the i2c bus target specified */

/* node = I2C bus address of target
 * cmd = goes into SMBus command byte, often used by targets as a reg index
 * 	(set to -1 if not required)
 * bytes = number of bytes to be read from target
 * data = storage of read data
 */


DBM_STATUS i2cdev_read( const uint8 node, const int cmd,
			const size_t bytes, uint8 *data)
{
    static unsigned last_byte_read = ~0;

    TRACE( "I2C bus read of %d bytes from node %x cmd %02x\n",
		bytes, node, cmd  );

#ifdef CONFIG_SHARK
    ali_smi();
#endif

    /* Paranoia: if the read fails, we don't want spurious data returned */
    memset( data, 0xFF, bytes );

    if ( i2c_wait_ready() != STATUS_SUCCESS )
    {
	i2c_error_analyse();
        if ( i2c_error_recover( ) == STATUS_FAILURE )
		return STATUS_FAILURE;
    }

    /* Set address.  Adding LSB=1 signifies read request */
    smb_write( SMBUS_ADDR, node | 0x01 );

    /* Set command. */
    if (cmd == -1 )
    {
	/* NOTE: here we only support 1-byte transfer requests */

	/* Set Send/Receive byte. */
	smb_write( SMBUS_SMBCMD, SMBUS_COMMAND_SNDRCV );

    } else {

	/* NOTE: here we only support 1-byte or 2-byte transfer requests */

	if (bytes == 1)				/* Set Read/Write byte. */
	    smb_write( SMBUS_SMBCMD, SMBUS_COMMAND_WRRDB );
	else					/* Set Read/Write word. */
	    smb_write( SMBUS_SMBCMD, SMBUS_COMMAND_WORD );

	/* Set command register, interpreted by targets in various ways. */
	smb_write( SMBUS_CMD, cmd );

    }

    /* Start I2C transaction. */
    smb_write( SMBUS_START, 0xFFU );


    /* Wait for command completion. */
    if ( i2c_wait_complete() != STATUS_SUCCESS )
	return STATUS_FAILURE;


    /* Read back the requested byte(s) */
    data[0] = smb_read( SMBUS_DATA_A );
    if ( bytes == 2 ) 
	data[1] = smb_read( SMBUS_DATA_B );	/* high byte in here */

#ifdef DUP_BYTE_FIXUP
    /* ALI problem - a byte can get duplicated and read the next transaction */
    if ( last_byte_read != data[0] )
    {
	last_byte_read = (unsigned)data[0];
    } else {
	mobo_logf( LOG_WARN 
		"I2C: duplicated byte on read from device at 0x%02X\n",
		node );
	
	data[0] = smb_read( SMBUS_DATA_A );
    }
#endif		/* DUP_BYTE_FIXUP */

    /* DEBUG - ALI controller may require a short breather before ready */
    usleep( ALI_BUGGY_DELAY );

    TRACE( "Read data 0x%02X\n", data[0] );

    /* DEBUG BODGE: Cancel anything outstanding and reset the host controller */
    smb_write( SMBUS_SMBCMD, SMBUS_SMBCMD_TERMINATE );

    return STATUS_SUCCESS;
}



/* node = I2C bus address of target
 * cmd = goes into SMBus command byte, often used by targets as a reg index 
 *      (set to -1 if not required)
 * bytes = number of bytes to be written to target
 * data = location of data to write
 */

#define WRITE_RETRIES	4

DBM_STATUS i2cdev_write( const uint8 node, const int cmd,
			 const size_t bytes, const uint8 *data)
{
    int retry_count = 0;

    TRACE( "I2C bus write of %d bytes to node %x cmd %02x\n",
		bytes, node, cmd  );

#ifdef CONFIG_SHARK
    ali_smi();
#endif

    do {
	if ( retry_count > WRITE_RETRIES )
	{
	    mobo_logf( LOG_CRIT
		"I2C: could not complete write transaction to %02X\n", node );
	    return STATUS_FAILURE;
	}

	if ( retry_count > 0 )
		mobo_logf( LOG_WARN "I2C: retrying write to %02X\n", node );


	/* Wait for Bus Idle. */
	if ( i2c_wait_ready() == STATUS_FAILURE )
	{
	    TRACE( "I2C bus was not ready for transaction\n" );
	    i2c_error_analyse();
	    if ( i2c_error_recover() == STATUS_FAILURE )
		return STATUS_FAILURE;
	}


	/* Set address. */
	smb_write( SMBUS_ADDR, node );


	if ( cmd == -1 )		/* No command byte to be specified */
	{
	    /* Set Send/Receive byte. */
	    smb_write( SMBUS_SMBCMD, SMBUS_COMMAND_SNDRCV );

	    /* Queue first byte */
	    smb_write( SMBUS_CMD, data[0] );

	} else {

	    if (bytes == 1)			/* Set Read/Write byte. */
		smb_write( SMBUS_SMBCMD, SMBUS_COMMAND_WRRDB );
	    else				/* Set Read/Write word. */
		smb_write( SMBUS_SMBCMD, SMBUS_COMMAND_WORD );

	    /* Set command register, interpreted by targets in various ways. */
	    smb_write( SMBUS_CMD, cmd );

	    /* Queue first byte(s) */
	    smb_write( SMBUS_DATA_A, data[0] );

	    if ( bytes == 2 )
		smb_write( SMBUS_DATA_B, data[1] );
	}

	/* Start I2C transaction. */
	smb_write( SMBUS_START, 0xFF );


	/* Wait for command completion. */
	retry_count++;

    } while ( i2c_wait_complete() == STATUS_FAILURE );

    /* DEBUG BODGE: Cancel anything outstanding and reset the host controller */
    smb_write( SMBUS_SMBCMD, SMBUS_SMBCMD_TERMINATE );

    return STATUS_SUCCESS;
}


/*----------------------------------------------------------------------*/
/* ALI specific interface routines */

String ali_revstr( String buf )
{
    int alirev = pcicfgrb(0, PLAT_ISAID, 0, ISA_REV );
    sprintf_dbm( buf, "Revision %02X", alirev );
    return buf;
}


/* Poke around the SMI registers and look for interesting stuff */

void ali_smi( void )
{
    uint8 smicntl;
    int status, newstatus;
    int rval, newrval;

    mobo_logf( LOG_DBG "STIG: before SMI test - CChip DRIR is 0x%016lX (%s)\n",
        cchip[DRIR].r, cchip[DRIR].r & IRQ_SMI_M ? "SMI set" : "SMI clear" );

    smicntl = pcicfgrb( 0, PLAT_PMUID, 0, PMU_SMI_CNTL );
    mobo_logf( LOG_DBG "SMI: Control register is 0x%02X\n", smicntl );

    /* Enable SMI reporting if not set */
    smicntl |= SMI_CNTL_SMI_ENA;

    /* Clear ACK if bizarrely set and set no delay for SMI */
    smicntl &= ~(SMI_CNTL_SMIACK | SMI_CNTL_DELAY_M);
    pcicfgwb( 0, PLAT_PMUID, 0, PMU_SMI_CNTL, smicntl );

    /* Attempt an SMI ACK cycle */
    pcicfgwb( 0, PLAT_PMUID, 0, PMU_SMI_CNTL, smicntl | SMI_CNTL_SMIACK );

    status = pcicfgrw( 0, PLAT_PMUID, 0, PMU_SEOG );
    mobo_logf( LOG_DBG "SMI: SEOG is 0x%04X\n", status );

    status = pcicfgrw( 0, PLAT_PMUID, 0, PMU_SSOG );
    mobo_logf( LOG_DBG "SMI: SSOG is 0x%04X\n", status );

    status = pcicfgrw( 0, PLAT_PMUID, 0, PMU_SEWS );
    pcicfgww( 0, PLAT_PMUID, 0, PMU_SEWS, status );
    newstatus = pcicfgrw( 0, PLAT_PMUID, 0, PMU_SEWS );
    mobo_logf( LOG_DBG "SMI: SEWS is 0x%04X -> 0x%04lX\n", status, newstatus );

    status = pcicfgrl( 0, PLAT_PMUID, 0, PMU_SSWS );
    if( status & (1<<19) )
    {
	mobo_logf( LOG_DBG "SMI: SMBus bit not clearing, looking around...\n" );
	rval = smb_read( SMBUS_SMBSTS );
	smb_write( SMBUS_SMBSTS, rval );
	newrval = smb_read( SMBUS_SMBSTS );
	mobo_logf( LOG_DBG "SMI: SMBus status is 0x%02X -> 0x%02X\n", 
		rval, newrval );
    }

    pcicfgwl( 0, PLAT_PMUID, 0, PMU_SSWS, status );
    newstatus = pcicfgrl( 0, PLAT_PMUID, 0, PMU_SSWS );
    mobo_logf( LOG_DBG "SMI: SSWS is 0x%06lX -> 0x%06lX\n", status, newstatus );


    pcicfgwb( 0, PLAT_PMUID, 0, PMU_SMI_CNTL, smicntl );

    
    msleep( 10 );		/* Wait for things to subside... */

    mobo_logf( LOG_DBG "STIG: after SMI test - CChip DRIR is 0x%016lX (%s)\n",
        cchip[DRIR].r, cchip[DRIR].r & IRQ_SMI_M ? "SMI set" : "SMI clear" );

}


