
	title "Debugger Cold Start"
;
; A Small Debugger for the 65802/816
;
; Author:	A. Kossow     (AED)
; Version:	X1.0	(June 1984)
;
; Commands are of the form:
;
; <hex adr><cmd> or <cmd>
;
; Valid operators are:
;
; lf	open next seq wrd or byt loc
; ^	open prev seq wrd or byt loc
; /	reopen wrd loc last opened
; \	reopen byt loc last opened
; <adr>/	open new word loc
; <adr>\	open new byte loc
;
;
; ;b		reset breakpoints
; ;nb		reset breakpoint (0-3)
; <adr>;b	set breakpoint 0
; <adr>;nb	set breakpoint   (0-3)
;
; ;c		reset bpt counts
; ;nc		reset bpt count n
; <val>;c	set bpt 0 count to <val>
; <val>;nc	set bpt n count to <val>
; 
; r		display processor & breakpoint registers
; ;r		reset registers to initial state
;
; <adr>;m	display 16 memory locations starting at <adr>
; m		and advance to <adr> + 16.
;
; <adr>;i	disassemble <adr> and advance to next instr.
; i
;
; <adr>;g	start program execution at <adr>
;
; <val>;a	load accumulator with <val>
; <val>;p	load program counter with <val>
; <val>;s	load stack pointer with <val>
; <val>;x	load x register with <val>
; <val>;y	load y register with <val>
; <val>;z	load direct register with <val>
; <val>;f	load processor status reg with <val>
; <val>;d	load data bank register with <val>
; <val>;k	load program bank register with <val>
;
; '		display current instruction
; s		single step instruction mode
; p		proceed from breakpoint
; h		<help>  dump command summary
;
; Note:		When actually in the debugger,  interrupts
;		are disabled. As a result,  communications
;		with the terminal are not interrupt-driven
;	
	title	" "
;
;
; Debugger memory layout:
;
;		debugger variables
; dpr-> dirpag>	------------------
;		------------------ 4 byte firewall
; sp ->	dbgstk>	------------------
; (dbgr)	debugger stack (64 bytes)
;		------------------ 4 byte firewall
; sp ->	inisp>	------------------
; (user)	test pgm stack
;

dirpg	equ	1		; psect of direct page data
code	equ	0		; psect of code

byte	equ	1		; length of a byte
word	equ	2		; length of a word
long	equ	3		; length of a long adr
quad	equ	4		; length of a quadbyte (32 bits)
				;
trace 	equ	8		; flag bits in bmode
resume	equ	16
 
	title	"Debugger Entry Point"
	psect	code
;
;	debugger cold entry point
;
dbg::
	sei		; disable interrupts
	clc
	xce		; go to native mode

	rep	#0x30	; 16 bit memory / index

	lda	##dbgstk
	tcs		; run from our own stack

	pea	##dirpag
	pld		; run from our own direct page

	bsl	dbgini	; do cold start initialization

;
;	set up debugger direct page
;
	stz	<0	; first location
	ldx	##dirpag
	ldy	##dirpag+1
	lda	##dirlen-2
	mvn	>0,>0	; clear it out

	sep	#0x20		; 8 bit mem		

	phk			; set debugger data bank=pgm bank 
	plb
	phk			; also save it for future reference
	pla
	sta	<dbgpbk
;
;	print startup msg and go
;
	per	vermsg
	bsl	putstr
	brl	rstregs		; jmps to cmd loop after rsting regs

;
;	Enter here from break trapper after
;	capturing context of user program.
;	Check if we knew about this breakpoint
;	and print an error if we didn't
;

dbgbrk:
	rep	#0x30
	ldx	##16
$loop:
	lda	<bkptbl,x
	cmp	<spc		; <spc = break addr ?
	bne	$1		; branch if no
	lda	<bkptbl+2,x	; get pbr for this bkpt
	bpl	$1		; also branch if this bkpt not busy
	and	##0xff
	cmp	<spb
	beq	$found		; known bkpt if yes

$1:	cpx	##0		; more bkpts ?
	beq	$2		; branch if no
	dex			; point x to next bkpt entry
	dex
	dex
	dex
	bra	$loop
;
;	no match, print an error
;

$2:	per	unkbkp
	bsl	putstr		; print error msg
	bra	$debug		; enter debug command loop

;
;	restore the instruction at breakpoint
;

$found:
	txa			; convert bkptbl index
	lsr	a		; to bkpval index - map
	lsr	a		; (0,4,8...) to (0,1,2...)
	tax			; save bkpval index in x

	lda	<spc		; set up current addr, pbr
	sta	<curadr
	lda	<spb
	sta	<curadr+2

	sep	#0x20		; 8 bit mem
	lda	<bkpval,x	; get val to replace
	sta	[<curadr]	; replace break with real instr.

;
;	figure out what mode of break occurred.
;	possibilities and actions are 
;	trace -	single-step , but don't enter command loop
;		unless current instruction is a goto,
;	step  - single-step, but don't enter command loop
;		unless count = 0 (execute count instructions),
;	normal- user-set break, don't enter command loop
;		unless count = 0 (execute current instr count times)
;		if count <> 0, set a break at next executable for ...
;	resume- set the break at the PREVIOUS instruction and
;		resume execution.  This restores the break that was
;		undone to execute the previous instruction.
;		
	lda	<bmode		; get mode of previous break
	bit	#trace		; trace mode ?
	bne	$trace		; branch if trace mode
	bit	#resume		; previous break set by user ?
	bne	$resume		; yes, go restore it
	bra	$normal		; user or ss bkpt
$trace:
	lda	<bkpval,x	; get current instruction
	bsl	tstgoto		; current instr a goto ?
	bcc	$t1		; branch if no
	brl	$debug		; enter command loop if goto
$t1:	brl	tcmd		; not goto, resume execution.

;
;	restore a user bkpt - number in <bmode.
;

$resume:
	rep	#0x20			; 16 bit mem
	lda	<bmode			; get bp number to restore
	stx	<bmode			; clear resume flag and save
					; bkpt number
	and	##3		
	asl	a			; make index to bkptbl
	asl	a
	tax
	lda	<bkptbl,x		; get user break addr
	sta	<nextin
	lda	<bkptbl+2,x
	sta	<nextin+2
	sep	#0x20			; 16 bit mem
	lda	#0			; restore user break
	sta	[<nextin]
	ldx	<bmode
	cpx	##4
	blt	$normal
	brl	pcmd			; resume execution

;
;	dec bp counter and resume if not 0
;

$normal: 
	rep	#0x20		; 16 bit mem
	txa			; make index into bp counters
	asl	a
	tax
	dec	<bkpcnt,x	; dec count for this bkpt
	beq	$debug		; counted down, enter command loop

	txa			; get bpt number
	lsr	a
	cmp	##4		; single step bkpt ?
	bge	$n1		; yes, don't set resume flag
	ora	##resume	; set mode so that next break
	sta	<bmode		; will restore this one (bp num
$n1:				; in least 2 bits of <bmode)
	brl	scmd		; resume in single step fashion
		
	
$debug:
	bsl	putsp
	bsl	dasm		; do the disassembly
	
;
;	print bp counts
;

	pei	<bkpcnt+0
	pei	<bkpcnt+2
	pei	<bkpcnt+4
	pei	<bkpcnt+6
	pei	<bkpcnt+8
	pea	##10		; printf arg byte count
	per	$ctfmt		; bp count format
	bsl	printf		; output counts

	per	crlfsp3
	bsl	putstr	; crlf and 3 spaces
	brl	rcmd1	; print all the registers
			; will jump to the cmd loop when done

$ctfmt: dcs	"   %x %x %x %x %x\0"

	title	"Command Decoder"
;
;	Main command loop
;	Get a chr from the keyboard and do something with it
;
cmdloop:
	rep	#0x30		; 16 bit mem/idx
	stz	<daccum
	stz	<daccum+2	; clear out digit accumulator
	stz	<digcnt		; no digits in the accumulator

	lda	##0xffff
	sta	<bptnum		; no special bp in mind yet
	stz	<bmode		; clear break mode flags
	per	prompt
	bsl	putstr
;
;	enter here if you don't want to change the digit accumulator
;	or output a prompt
cmdl1:
	bsl	getchr
;
;	enter here with a chr ready in the accumulator
cmdl2:
	sep	#0x20		; 8 bit mem
;
;	Scan for a valid command character...
;
	cmp	#'\'	;'\'
	bne	$1
;
;	\ = open currently selected address
;	    accesses will be one byte at a time
;
	lda	#1
	sta	<onebyt
	bra	$2

$1:	cmp	#'/'		; '/'
	bne	$3
;
;	/ = open currently selected address
;	    accesses will be two bytes at a time
;

	stz	<onebyt

$2:	
	lda	<digcnt		; new address entered?
	beq	$21		; nope, use old one

	rep	#0x20		; 16 bit memory
	lda	<daccum		; update open address
	sta	<curadr
	lda	<daccum+2
	sta	<curadr+2
$21:
	bsl	crlf
	brl	opencmd		; open the memory location

$3:
	cmp	#0x0a
	bne	$4
;
;	\n = increment currently open address by 1 or 2
;	     increment stays within a 65k boundary
;
	rep	#0x20
	inc	<curadr		; bump curadr to next byte
	lda	<onebyt		; need to bump again for word ?
	ror	a
	bcs	$31		; branch if no
	inc	<curadr		; now curadr points to next word

$31:
	bsl	crlf
	brl	opencmd

$4:
	cmp	#'^'		; '^'
	bne	$5
;
;	^ = decrement currently open address and examine it
;	    decrement stays within 65k boundary
;
	rep	#0x20
	dec	<curadr
	lda	<onebyt
	ror	a
	bcs	$41
	dec	<curadr
$41
	bsl	crlf
	brl	opencmd

$5:
	cmp	#';'
	bne	$6
;
;	; - commands starting with semi-colon
;	    ;r ;b ;#b ;a ;x ;y ;f ;s ;i
;
	brl	semicmd
$6:
	cmp	#'s'
	bne	$7
;
;	s - single step (once)
;
	brl	scmd

$7:
	cmp	#'t'
	bne	$71
;
;	t - execute until next 'goto'
;
	brl	tcmd
$71:
	cmp	#'p'
	bne	$8
;
;	p - proceed from breakpoint
;
	brl	pcmd

$8:
	cmp	#'h'
	bne	$9
;
;	h - print the help message
;
	per	hlpmsg
	bsl	putstr
	brl	cmdloop

$9:
	cmp	#'r'
	bne	$10

;
;	r - print registers
;
	brl	rcmd

$10:
	cmp	#0xd
	bne	$11
;
;	<cr> - ignored
;
	brl	cmdloop

$11:
	cmp	#0x27
	bne	$12
;
;	' - disassemble current location
;
	bsl	dasm
	brl	cmdloop

$12:
	cmp	#'m'
	bne	$13
;
;	m - memory dump of cur adr
;
	brl	mcmd

$13:
	cmp	#'l'
	bne	$14
;
;	l - load ram from parallel link
;
	bsl	lcmd
	brl	cmdloop

$14:
	cmp	#'i'
	bne	$xx
;
;	i - disassemble and advance
;
	brl	icmd
		
;
;	check for numeric input
;
$xx:
	bsl	ishex
	bcs	badchr		; not a digit, don't know what
				; it is
	bsl	xtol
	sep	#0x20		; 8 bit mem
	brl	cmdl1		; try for more chrs
;
;	echo a '?' for unknown commands
;
badchr:
	lda	#'?'		; echo ? for unknown cmds
	bsl	putchr
	brl	cmdloop		; carry set if not

	title	"Examine a Location"
;
; open a location specifed by open adr
;
opencmd:
	rep	#0x30		; 16 bit index and memory
	pei	<curadr+2	; get value for data bank reg
	pei	<curadr		; get address of data
	lda	[<curadr]	; get the data
	pha			; save it 
	pea	##6		; push arg byte count
	lda	<onebyt
	ror	a		; display data byte or word?
	bcs	$1		; branch if byte
	per	opnwfmt		; push format string pointer
	bra	$2		; for word opens
$1:	
	per	opnbfmt		; push open byte fmt pointer
$2:
	bsl	printf		; print dbr,addr,data
;
;	initialize the digit accumulator for a new value to store
;
	stz	<digcnt		; clear out digit count	
	stz	<daccum
	stz	<daccum+2	; reset digit accumulator
$3:
	bsl	getchr
	sep	#0x20
	cmp	#0xd		; <cr> = store current val
	bne	$4
	bra	$8		; all updates done below
$4:
	cmp	#0xa		; <lf> = store and incr open adr
	bne	$5
	bra	$8		; all updates done below
$5:
	cmp	#'^'		; <^> = store and decr open adr
	bne	$6
	bra	$8		; all updates done below
$6:
	cmp	#0x27		; '  = disassemble this location
	bne	$7
	bsl	dasm		; output opcode,operands
	brl	cmdl1		; go get next char

$7:	bsl	ishex
	bcs	$10		; nope
	bsl	xtol
	bra	$3
;
;	update the currently open location
;	if there are any digits in the digit accumulator
$8:
	pha			; save the character
	lda	<digcnt
	beq	$9		; no digits, just exit

	lda	<onebyt		; update byte or word ?
	bne	$81		; branch if byte
	rep	#0x20		; word - set 16 bit mem
$81	lda	<daccum		; get data byte/word
	sta	[<curadr]	; store it 

$9	sep	#0x20		; restore 8 bit mem
	pla			; restore character
$10:
	brl	cmdl2		; finish decoding in the main loop

	title	"Semi-colon Command Decoder"
;
; process a command with a semi colon
;
semicmd:
	bsl	getchr

	cmp	#'i'		; i-disassemble & advance
	bne	$1
	brl	icmd
$1:
	cmp	#'c'
	bne	$1a
	brl	ccmd
$1a:
	cmp	#'b'		; b-breakpoint
	bne	$2
	brl	bcmd
$2:
	cmp	#'m'		; m-dump mem
	bne	$3
	brl	mcmd
$3:
	cmp	#'g'		; g-go
	bne	$4
	brl	gcmd

$4:
	cmp	#'a'		; a-load acc
	bne	$5
;
	ldx	<daccum
	stx	<sac
	brl	cmdloop

$5:
	cmp	#'s'		; s-load sp
	bne	$6
;
	ldx	<daccum
	stx	<ssp
	brl	cmdloop

$6:
	cmp	#'x'		; x-load ix
	bne	$7
;
	ldx	<daccum
	stx	<six
	brl	cmdloop

$7:
	cmp	#'y'		; y-load iy
	bne	$8
;
	ldx	<daccum
	stx	<siy
	brl	cmdloop

$8:
	cmp	#'z'		; z-load dpr
	bne	$9
;
	ldx	<daccum
	stx	<sdp
	brl	cmdloop

$9:
	cmp	#'f'		; f-load psw
	bne	$10
;
	lda	<daccum
	sta	<sps
	brl	cmdloop

$10:
	cmp	#'d'		; d-load dbr
	bne	$11
;
	lda	<daccum
	sta	<sdb
	brl	cmdloop

$11	cmp	#'k'		; k-load pbr
	bne	$12
;
	lda	<daccum
	sta	<spb
	brl	cmdloop

$12:	
	cmp	#'p'		; p-load pc
	bne	$13
;
	ldx	<daccum
	stx	<spc
	brl	cmdloop

$13:
	cmp	#'r'		; r-reset registers
	bne	sem14
;
;	reset all registers to intial state
;
rstregs:
	rep	#0x20
	lda	##inisp-4
	sta	<ssp
	lda	##inipc
	sta	<curadr		; default current adr is code base
	stz	<curadr+2
	stz	<onebyt
	inc	<onebyt		; default to 1 byte open
	stz	<sscnt		; default to 1 instr s step
	inc	<sscnt
	sta	<spc
	stz	<sdp
	stz	<six
	stz	<siy
	stz	<sac
	stz	<sdb
	stz	<spb
	sep	#0x20
	lda	#inipsw
	sta	<sps		; 8 bit a, 8 bit x, intr off
	brl	cmdloop
;
;	allow one digit from 0 -> 3 for a breakpoint
sem14:
	sec
	sbc	#'0'
	cmp	#5
	bge	$xx	

	sta	<bptnum
	stz	<bptnum+1
	brl	semicmd		; get the rest of the ';' cmd
;
;	unknown cmd, print a ?
$xx:
	lda	#'?'		; unrecognized cmd
	bsl	putchr
	brl	cmdloop		; all done

	title	"GO command"
;
;	;g	start processor execution
;
gcmd:
	sep	#0x20		; 8 bit memory
	bsl	crlf

	lda	<digcnt
	beq	$1		; no new adr specifed, use old one

	ldx	<daccum
	stx	<spc		; update pc
	ldx	<daccum+2
	stx	<spb
$1:	brl	pcmd		; restore user context and 
				; resume execution

	title	"Register Display"
;
;	r	display register list
;
rcmd:
;
;	display header
	per	regmsg
	bsl	putstr
;
;	r cmd alternate entry
;	(doesn't print header)
rcmd1:
	rep	#0x30		; 16 bit memory/ index
;	
;	display	pc		<pc>
	pei	<spb
	pei	<spc
;
;	display accumulator	<ac>
	pei	<sac
;
;	display ix		<ix>
	pei	<sdb
	pei	<six
;
;	display	iy		<iy>
	pei	<sdb
	pei	<siy
;
;	display dir pg reg	<dp>
	pei	<sdp
;
;	display	stk ptr		<sp>
	pei	<ssp
;
;	display flags		<ps>
	pei	<sps

	pea	##20		; # arg bytes to printf
	per	regfmt		; format string pointer
	bsl	printf
;
;	display breakpoints	(0-4)

	ldx	##0
	ldy	##5		; number of bkpts and ss bpt

bploop:
	lda	<bkptbl+2,x	; get high byte bp
	bmi	bpl1		; negative means bp is valid
	per	offmsg
	bsl	putstr		; just print '------ '
	bra	bpl2
bpl1:
	pha
	lda	<bkptbl,x	; get low byte bp
	pha
	pea	##4
	per	addfmt
	bsl	printf
bpl2:
	inx
	inx
	inx
	inx			; point to next bp
	dey
	bne	bploop

	brl	cmdloop

	title	"Set/Reset Breakpoints"
;
;	;c	set/clear breakpoint counts
;
ccmd:
	rep	#0x20
	lda	<digcnt		; value entered ?
	bne	$1		; yes, branch
	lda	##1		; no, set default value
	sta	<daccum
$1:	lda	<bptnum		; bp number entered ?
	bpl	$setcnt		; yes, do that one
;
;	set all counts
;
	lda	<daccum
	sta	<bkpcnt+0
	sta	<bkpcnt+2
	sta	<bkpcnt+4
	sta	<bkpcnt+6
	brl	cmdloop
;
;	set count for specified bkpt
;

$setcnt:
	lda	<bptnum		; get bkpt number
	asl	a		; convert to bkpcnt index
	tax
	lda	<daccum		; get count value
	sta	<bkpcnt,x	; store in bkpt counter
	brl	cmdloop

;
;	;b	set/reset breakpoints
;
bcmd:
	lda	<digcnt
	beq	rstbp		; no digits means reset bpt(s)
;
;	set a breakpoint
;
	lda	<bptnum
	bpl	setbp		; a specific bp was asked for, do that one
;
;
	lda	#0		; default bp is zero
setbp:
	rep	#0x30		; 16 bit idx/ mem
	and	##0x0003	;

	asl	a
	asl	a		; multiply index by 4
	tax			; put adr into x
	lda	<daccum
	sta	<bkptbl,x	; set 2 low bytes
	lda	<daccum+2
	ora	##0xff00	; set bkpt busy flag
	sta	<bkptbl+2,x	; set 2 high bytes

;
;	before we set the break, save the instr at the
;	break addr in bkpval.  the break service routine
;	will restore it for execution by pcmd.
;	also set counter for bp to 1.

	txa			; convert bkptbl index to
	lsr	a		; bkpcnt index
	tax
	lda	##1		; set bp count to 1.
	sta	<bkpcnt,x

	txa			; make index into bkpval
	lsr	a	
	tax

	sep	#0x20

	lda	[<daccum]	; get the instr
	sta	<bkpval,x	; squirrel it away

	lda	#0		; BRK instruction
	sta	[<daccum]	; try to set the breakpoint
	lda	[<daccum]	; see if the breakpoint got set
 	beq	setb1		; branch if probably it did
	
	txa			; get bp number in acc
	bsl	clrbpt		; go undo what we did
	per	bpserr		; couldn't set bp - PROM or NXM.
	bsl	putstr
setb1:
	brl 	cmdloop
;
;	reset a breakpoint
;

rstbp:
	lda	<bptnum
	bpl	$2		; no bkpt num means reset all bps

	lda	#3
$1:	bsr	clrbpt
	dec	a
	bpl	$1	
	brl	cmdloop
$2:
	bsr	clrbpt
	brl	cmdloop

;
;	clear a breakpoint - enter with bkpt 
;	number in acc.  if bp was busy, restores
;	original instr.  in any case resets bp
;	count and addr.
;

clrbpt:
	php
	rep	#0x30
	pha
	phx
	phy

	and	##3		; clear all but bp tbl idx
	tax
	lda	<bkpval,x	; get saved instruction.
	pha			; save it on stack.

	txa			; make index into bkpcnt table.
	asl	a
	tax

	stz	<bkpcnt,x	; set count to default (1)
	inc	<bkpcnt,x

	txa			; make index into bkptbl
	asl	a
	tax

	lda	<bkptbl,x	; get address of break.
	sta	<nextin		; store in <nextin.
	lda	<bkptbl+2,x
	sta	<nextin+2
	bpl	$1		; skip next step if bp wasn't set.
	
	sep	#0x20
	lda	1,s		; get the instruction
	sta	[<nextin]	; put it back
$1:
	ply			; clean stack

	rep	#0x20
	stz	<bkptbl,x	; reset bp addr.
	stz	<bkptbl+2,x

	ply
	plx
	pla
	plp	
	rts

	title	"Memory Dump"
;
;	;m	memory dump
;	dump memory starting at [<daccum]
;

mcmd:	
	lda	<digcnt		; new address entered ?
	beq	$0		; branch if no.

	per	crlfsp3		; output crlf and 3 spaces
	bsl	putstr		; subsequent ;m

	ldx	<daccum		; move new addr to <curadr.
	stx	<curadr
	ldx	<daccum+2
	stx	<curadr+2
;
;	display address of mem to be dumped.
;
$0:
	pei	<curadr+2	; push dump bank.
	pei	<curadr		; push dump address.
	pea	##4		; push arg byte count.
	per	addfmt		; push ptr to format string.
	bsl	printf		; display bank,address.

	sep	#0x30		; set 8 bit M/X
	ldx	#16		; and a count of 16
	ldy	#0
$1:
	bsl	putsp		; output a blank
	lda	[<curadr],y	; get a byte
	bsl	btox		; print in hex
	iny			; point to next byte
	dex			; decrement count
	bne	$1		; loop for all 16

	bsl	putsp		; output 2 spaces
	bsl	putsp
	ldx	#16		; reload count
	ldy	#0
$2:
	lda	[<curadr],y	; get the chr
	and	#0x7f		; mask parity bit
	cmp	#0x20		; check for printable
	bge	$3		; >= space ?
	lda	#'.'		; no, print dot
$3:	cmp	#0x7f		; del char ?
	blt	$4		; no
	lda	#'.'
$4:	bsl	putchr

	iny			; adv ptr
	dex			; dcr count
	bne	$2		; loop back while > 0

	rep	#0x30
	clc
	lda	<curadr		; get the adr
	adc	##16
	sta	<curadr		; move cur adr up by 16
	brl	cmdloop

;
; i command - disassemble instruction at curadr then
;	 advance curadr to the next instruction
;

icmd:
	rep	#0x30
;
;	update <curadr if new address entered
;
	lda	<digcnt
	and	##0xf
	beq	$1
	lda	<daccum
	sta	<curadr
	lda	<daccum+2
	sta	<curadr+2
$1:
	pei	<curadr+2	; output current address
	pei	<curadr
	pea	##4
	per	$lafmt
	bsl	printf
	
	lda	[<curadr]	; get opcode 
	bsl	inslen		; get # bytes in instruction
	pha			; save for updating curadr

;
;	output the instruction bytes
;
	tax
	ldy	##0
$2:	lda	[<curadr],y
	bsl	btox
	iny
	dex
	bne	$2

;
;	output some spaces to line things up
;
$3:	bsl	putsp
	bsl	putsp
	iny
	cpy	##5
	bne	$3

;
;	output disassembly
;

	bsl	dasm

;
;	now point curadr at the next instruction
;

	pla				; recover instr length

	clc				; bump curadr
	adc	<curadr
	sta	<curadr
	brl	cmdloop

$lafmt:	dcs	"  %la \0"

	title	"Single Step"


;
; s - do single stepping
;
; this is painful on a 816 since each instruction
; must be disassembled to determine where to place
; the next break.
;

tcmd:
	rep	#0x30
	lda	##trace		; set break mode to trace
	sta	<bmode
	bra	scmd1
scmd:
	rep	#0x30		; 16 bit M/X
	lda	##trace		; clear break trace mode
	trb	<bmode	
	lda	<sscnt
	sta	<bkpcnt+8
scmd1:
	lda	<spc		; point nextin at current instruction
	sta	<nextin
	lda	<spb
	sta	<nextin+2

	bsl	nxtexe		; point nextin at next instr to run

	lda	[<nextin]	; get the instruction
	and	##0xff		
	bne	$1		; branch if it's not a break
	stz	<bkptbl+18	; clear busy flag
	brl	pcmd		; break already set
$1:
;
; set break at next instruction to execute
;
	lda	<nextin		; save break address in table
	sta	<bkptbl+16
	lda	<nextin+2
	ora	##0xff00	; set break busy flag
	sta	<bkptbl+18

	sep	#0x20		; 8 bit memory/ 16 bit idx
	lda	[<bkptbl+16]	; get the opcode at that adr
	sta	<bkpval+4	; save the opcode value
	lda	#0		; brk instruction
	sta	[<bkptbl+16]	; try to set the break
	lda	[<bkptbl+16]	; see if we did
	beq	$2		; branch if (probably) successful
	per	bpserr		; couldn't set bp, ROM or nxm
	bsl	putstr
$2:
	rep	#0x30
	brl	pcmd		; start it up with an implied 'p'

	title	"Binary Conversions"
;
; wtox - convert 16 bit accumulator to ascii
; convert 16 bit integer to ascii hex
; and output to aux port
;
wtox:
	php
	rep	#0x20	; set 16 bit memory
	pha		; save accum
	xba		; get high byte
	bsr	btox
	lda	1,s	; get low byte
	bsr	btox
	pla		; restore accumulator
	plp
	rts

;
; btox - convert 8 bit accumulator to ascii
;
; convert 8 bit integer to ascii hex
; and output to aux port
;
;
btox:
	php		; save memory mode
	rep	#0x20	; set to 16 bit memory
	pha		; save all 16 bits of accumulator
	ror	a	; get high nibble of low byte
	ror	a
	ror	a
	ror	a
	bsr	ntox
	lda	1,s	; get another copy of the accumulator
	bsr	ntox
	pla		; restore accumulator
	plp		; restore memory modes
	rts
	eject
;
; ntox()
; low four bits of accumulator to hex digit
;
; the contents of the accumulator are trashed
;
ntox:
	php
	rep	#0x30		; 16 bit memory and index
	phy
	and	##0xf		; clear all but low 4 bits
	tay
	per	hextbl		; push adr of hex table
	lda	(1,s),y		; get the ascii chr
	bsl	putchr
	ply			; subtract 2 from stack
	ply			; restore callers y
	plp			; restore memory modes
	rts

	title	"I/O Routines"
;
; putstr(string);
; output null terminated string to aux port
;
; Note:
;  You can't call this routine from a jsl instruction
;  as it expects the return address to be 2 bytes long
;
putstr:
	php		; save incoming context
	rep	#0x30	; 16 bit mem, idx
	phx
	lda	6,s	; get starting adr
	tax		; put it into x
	sep	#0x20	; 16 bit idx, 8 bit mem

	lda	0,x	; get the chr
	bne	$1	; print {null} if null string

	per	nulmsg
	plx		; set to point to {null} msg

$1:	lda	0,x	; fetch the chr, 0 -> done
	beq	$2

	bsl	putchr

	inx		; point to next chr
	bra	$1

$2:
	rep	#0x20	; 16 bit memory
	plx		; restore x

	lda	2,s	; get ret adr
	sta	4,s	; wipe out argument
	lda	0,s	; get psw to restore
	sta	2,s	; store psw at adr below new ret adr 3(sp)
	pla		; move stack up 2 bytes
	plp		; restore memory/idx modes
	rts		
	eject
;
;	crlf
;	output a cr/lf to the aux port
;
crlf:
	php		; save memory mode
	sep	#0x20	; set to 8 bits
	pha		; save accumulator
	lda	#0xd	; output <cr>
	bsl	putchr
	lda	#0xa	; and    <lf>
	bsl	putchr	; print chr
	pla		; restore accumulator
	plp		; restore psw
	rts		; exit
;
;	putsp()
;	output a single space
;
putsp:
	php
	sep	#0x20
	lda	#' '
	bsl	putchr
	plp
	rts

	title	"Hex to Binary Conversions"
;
;	test for a hex digit (0-9 & a-f)
;	carry set if it's not hex
;
ishex:
	cmp	#'0'
	blt	$2		; not hex if < 0
	cmp	#':'
	blt	$1		; is  hex if <= 9
	cmp	#'a'
	blt	$2		; not hex if < a
	cmp	#'g'
	bge	$2		; not hex if > f
$1:
	clc			; it's hex
	rts
$2:
	sec			; it's not hex
	rts
	eject
;
; convert ascii chr to hex digit
; digit stored in 32 bit digit accumulator "daccum"
;
xtol:
	php
	sep	#0x20
;
;	first convert ASCII hex to AED modified hex
;
	cmp	#0x61		; is it a-f?
	blt	$1		; nope
	sec
	sbc	#0x27		; convert a -> :
$1:
	rep	#0x30
;
;	convert modified hex to binary
;
	and	##0x0f
	pha			; save new low nibble
;
;	shift long <daccum 4 times
;
	lda	<daccum
$2:
	asl	a		; bit 15 -> carry, 0 -> bit 0
	rol	<daccum+2	; carry -> bit 16
	asl	a		; bit 15 -> carry, 0 -> bit 0
	rol	<daccum+2	; carry -> bit 16
	asl	a		; bit 15 -> carry, 0 -> bit 0
	rol	<daccum+2	; carry -> bit 16
	asl	a		; bit 15 -> carry, 0 -> bit 0
	rol	<daccum+2	; carry -> bit 16

	ora	1,s		; OR in new low nibble.
	sta	<daccum
	pla			; clean up stack

	sep	#0x20
	inc	<digcnt		; daccum has another valid digit
	plp
	rts
	end





