---------------------------------------------------------------------------
------------------------ JEDEC TO VHDL PRIMITIVES -------------------------
------------------------------- VERSION 1.30c -----------------------------
---------------------------------------------------------------------------
---------------------------------------------------------------------------
---									---
---  Copyright 1987-1990  Data I/O Corporation  All Rights Reserved	---
---									---
---			PROPRIETARY NOTICE				---
---  This document is the property of the Data I/O Corporation, with	---
---  the information herein reserved as proprietary to Data I/O		---
---  Corporation, and is not to be published, reproduced, copied,	---
---  disclosed, or used without the express written consent of a duly	---
---  authorized representative of Data I/O Corporation.			---
---									---
---------------------------------------------------------------------------
--

PACKAGE j2vlib IS

function g_delay (
		  data: VLBIT;
		  tphl, tplh: TIME) 
		RETURN TIME;

procedure g_bufif0 (
	        signal dout :  out vlbit;
	        signal data   :  in  vlbit;
	        signal enable :  in  vlbit;
             	tpLH   :  in  TIME;
	     	tpHL   :  in  TIME);

procedure g_bufif1 (
	        signal dout :  out vlbit;
	        signal data   :  in  vlbit;
	        signal enable :  in  vlbit;
             	tpLH   :  in  TIME;
	     	tpHL   :  in  TIME);

procedure g_notif0 (
	        signal dout_b :  out vlbit;
	        signal data   :  in  vlbit;
	        signal enable :  in  vlbit;
             	tpLH   :  in  TIME;
	     	tpHL   :  in  TIME);

procedure g_notif1 (
	        signal dout_b :  out vlbit;
	        signal data   :  in  vlbit;
	        signal enable :  in  vlbit;
             	tpLH   :  in  TIME;
	     	tpHL   :  in  TIME);

-- D type flip flop
--    positive edge triggered
--    async. clear  preset(active high) 
--
   PROCEDURE g_dff (
      SIGNAL   q        :  INOUT VLBIT;
      SIGNAL   qb       :  INOUT VLBIT;
      CONSTANT data     :  IN    VLBIT;   
      SIGNAL   clk      :  IN    VLBIT;
      SIGNAL   preset   :  IN    VLBIT;   
      SIGNAL   clear    :  IN    VLBIT;   
      SIGNAL   pwrupstate    :  IN    VLBIT;   
      t_clkout	        :  in  	 TIME);
--
-- D type flip flop
--    positive edge triggered
--    async. clear (active high) 
--
   PROCEDURE g_dff_e (
      SIGNAL   q        :  INOUT VLBIT;
      SIGNAL   qb       :  INOUT VLBIT;
      SIGNAL   din      :  IN    VLBIT;   
      SIGNAL   clk      :  IN    VLBIT;
      SIGNAL   preset   :  IN    VLBIT;   
      SIGNAL   clear    :  IN    VLBIT;   
      SIGNAL   pwrupstate    :  IN    VLBIT;   
      t_clkout	        :  in  	 TIME);
--
-- D type flip flop
--    positive edge triggered
--    async. clear and preset (active high)
--
   PROCEDURE g_dffpc (
      SIGNAL   q        :  INOUT VLBIT;
      SIGNAL   qb       :  INOUT VLBIT;
      CONSTANT data     :  IN    VLBIT;   
      SIGNAL   clk      :  IN    VLBIT;
      SIGNAL   preset   :  IN    VLBIT;   
      SIGNAL   clear    :  IN    VLBIT;   
      SIGNAL   pwrupstate    :  IN    VLBIT);   
--
--
   PROCEDURE g_dffa (
      SIGNAL   q        :  INOUT VLBIT;
      SIGNAL   qb       :  INOUT VLBIT;
      CONSTANT data     :  IN    VLBIT;   
      SIGNAL   clk      :  IN    VLBIT;
      SIGNAL   preset   :  IN    VLBIT;   
      SIGNAL   clear    :  IN    VLBIT;   
      SIGNAL   pwrupstate    :  IN    VLBIT;   
      t_clkout	        :  in  	 TIME);
--
-- JK flip flop
--    positive edge triggered
--    async. clear and preset (active high)
--
   PROCEDURE g_jkff (
      SIGNAL   q        :  INOUT VLBIT;
      SIGNAL   qb       :  INOUT VLBIT;
      SIGNAL   j        :  IN    VLBIT;   
      SIGNAL   k        :  IN    VLBIT;   
      SIGNAL   preset   :  IN    VLBIT;   
      SIGNAL   clear    :  IN    VLBIT;   
      SIGNAL   clk      :  IN    VLBIT;
      SIGNAL   pwrupstate    :  IN    VLBIT;   
      t_clkout	        :  in  	 TIME);
--
--
-- T flip flop
--    positive edge triggered
--    async. clear and preset (active high)
--
   PROCEDURE g_tff (
      SIGNAL   q        :  INOUT VLBIT;
      SIGNAL   qb       :  INOUT VLBIT;
      SIGNAL   t        :  IN    VLBIT;   
      SIGNAL   clk      :  IN    VLBIT;
      SIGNAL   preset   :  IN    VLBIT;   
      SIGNAL   clear    :  IN    VLBIT;   
      SIGNAL   pwrupstate    :  IN    VLBIT;   
      t_clkout	        :  in  	 TIME);
--
-- RS flip flop
--    negative edge triggered
--    async. clear and preset (active high)
--
   PROCEDURE g_rsff (
      SIGNAL   q        :  INOUT VLBIT;
      SIGNAL   qb       :  INOUT VLBIT;
      SIGNAL   s        :  IN    VLBIT;   
      SIGNAL   r        :  IN    VLBIT;   
      SIGNAL   clk      :  IN    VLBIT;
      SIGNAL   preset   :  IN    VLBIT;   
      SIGNAL   clear    :  IN    VLBIT;   
      SIGNAL   pwrupstate    :  IN    VLBIT;   
      t_clkout	        :  in  	 TIME);
--
--
   PROCEDURE g_maxff (
      SIGNAL   q        :  INOUT VLBIT;
      SIGNAL   qb       :  INOUT VLBIT;
      CONSTANT data     :  IN    VLBIT;   
      SIGNAL   clk      :  IN    VLBIT;
      SIGNAL   preset   :  IN    VLBIT;   
      SIGNAL   clear    :  IN    VLBIT;   
      SIGNAL   pwrupstate :  IN  VLBIT;   
      t_clkout	        :  IN  	 TIME);
---
--
   PROCEDURE g_lff (
      SIGNAL   q        :  INOUT VLBIT;
      SIGNAL   qb       :  INOUT VLBIT;
      CONSTANT data     :  IN    VLBIT;   
      SIGNAL   gate     :  IN    VLBIT;
      SIGNAL   preset   :  IN    VLBIT;   
      SIGNAL   clear    :  IN    VLBIT;   
      SIGNAL   pwrupstate :  IN  VLBIT;   
      t_clkout	        :  in  	 TIME);
--
--
   PROCEDURE g_lffa (
      SIGNAL   q        :  INOUT VLBIT;
      SIGNAL   qb       :  INOUT VLBIT;
      CONSTANT data     :  IN    VLBIT;   
      SIGNAL   gate     :  IN    VLBIT;
      SIGNAL   preset   :  IN    VLBIT;   
      SIGNAL   clear    :  IN    VLBIT;   
      SIGNAL   pwrupstate :  IN  VLBIT;   
      t_clkout	        :  in  	 TIME);
--
   PROCEDURE g_lffh (
      SIGNAL   q        :  INOUT VLBIT;
      SIGNAL   qb       :  INOUT VLBIT;
      CONSTANT data     :  IN    VLBIT;   
      SIGNAL   gate     :  IN    VLBIT;
      SIGNAL   preset   :  IN    VLBIT;   
      SIGNAL   clear    :  IN    VLBIT;   
      SIGNAL   pwrupstate :  IN  VLBIT;   
      t_clkout	        :  in  	 TIME);
--
--
   PROCEDURE g_raff (
      SIGNAL   q        :  INOUT VLBIT;
      SIGNAL   qb       :  INOUT VLBIT;
      CONSTANT data     :  IN    VLBIT;   
      SIGNAL   clk      :  IN    VLBIT;
      SIGNAL   preset   :  IN    VLBIT;   
      SIGNAL   clear    :  IN    VLBIT;   
      SIGNAL   pwrupstate :  IN  VLBIT;   
      t_clkout	        :  IN  	 TIME);
--
--
function g_smux2 (
	        signal i0   : vlbit;
	        signal i1   : vlbit;
	        signal s0   : vlbit;
             	tpLH   :  TIME;
	     	tpHL   :  TIME)
		RETURN VLBIT;

--
--
procedure g_smux (
	        signal dout :  out vlbit;
	        signal i0   :  in  vlbit;
	        signal i1   :  in  vlbit;
	        signal i2   :  in  vlbit;
	        signal i3   :  in  vlbit;
	        signal s0   :  in  vlbit;
	        signal s1   :  in  vlbit;
             	tpLH   :  in  TIME;
	     	tpHL   :  in  TIME);

--
procedure g_jkdff (
      SIGNAL   q        :  INOUT VLBIT;
      SIGNAL   qb       :  INOUT VLBIT;
      SIGNAL   ji       :  IN    VLBIT;   
      SIGNAL   ki       :  IN    VLBIT;   
      SIGNAL   jf       :  IN    VLBIT;   
      SIGNAL   kf       :  IN    VLBIT;   
      SIGNAL   clk      :  IN    VLBIT;
      SIGNAL   ld       :  IN    VLBIT;
      SIGNAL   sel      :  IN    VLBIT;
      SIGNAL   preset   :  IN    VLBIT;   
      SIGNAL   clear    :  IN    VLBIT;   
      SIGNAL   pwrupstate    :  IN    VLBIT;   
      t_clkout	        :  in  	 TIME);
---
procedure g_vxff (
      SIGNAL   q     	:  INOUT VLBIT;
      SIGNAL   qb       :  INOUT VLBIT;
      SIGNAL   din      :  IN    VLBIT;
      SIGNAL   clk      :  IN    VLBIT;
      SIGNAL   preset   :  IN    VLBIT;
      SIGNAL   clear    :  IN    VLBIT;
      SIGNAL   qsel     :  IN    VLBIT;
      SIGNAL   osel     :  IN    VLBIT;
      SIGNAL   pwrupstate    :  IN    VLBIT;
      t_clkout          :  in    TIME);
---
   PROCEDURE p_chksuho (
      SIGNAL   data     :  IN    VLBIT;
      SIGNAL   clk      :  IN    VLBIT;
      t_setup           :  IN     TIME;
      t_hold            :  IN     TIME;
      t_clk             :  INOUT  TIME;
      t_data            :  INOUT  TIME;
      setname           :  IN     STRING);
---
   PROCEDURE p_chksuho_hl (
      SIGNAL   data     :  IN    VLBIT;
      SIGNAL   clk      :  IN    VLBIT;
      t_setup           :  IN     TIME;
      t_hold            :  IN     TIME;
      t_clk             :  INOUT  TIME;
      t_data            :  INOUT  TIME;
      setname           :  IN     STRING);

---
END j2vlib;

--
PACKAGE BODY j2vlib IS
--
---
procedure g_bufif0 (
	        signal dout   :  out vlbit;
   	     	signal data   :  in  vlbit;
	        signal enable :  in  vlbit;
             	tpLH   :  in  TIME;
	     	tpHL   :  in  TIME)
   is
   begin
       case enable is
         when '1' => dout <= 'Z' AFTER g_delay (data,tpLH,tpHL);
         when '0' => dout <= data AFTER g_delay (data,tpLH,tpHL);
         when others => dout <= 'X' AFTER g_delay (data,tpLH,tpHL);
       end case;

end g_bufif0;
--
procedure g_bufif1 (
	        signal dout   :  out vlbit;
   	     	signal data   :  in  vlbit;
	        signal enable :  in  vlbit;
             	tpLH   :  in  TIME;
	     	tpHL   :  in  TIME)
   is
   begin
       case enable is
         when '0' => dout <= 'Z' AFTER g_delay (data,tpLH,tpHL);
         when '1' => dout <= data AFTER g_delay (data,tpLH,tpHL);
         when others => dout <= 'X' AFTER g_delay (data,tpLH,tpHL);
       end case;

end g_bufif1;
--
procedure g_notif1 (
	        signal dout_b :  out vlbit;
   	     	signal data   :  in  vlbit;
	        signal enable :  in  vlbit;
             	tpLH   :  in  TIME;
	     	tpHL   :  in  TIME)
   is
   begin
       case enable is
         when '0' => dout_b <= 'Z' AFTER g_delay (data,tpLH,tpHL);
         when '1' => dout_b <= not(data) AFTER g_delay (data,tpLH,tpHL);
         when others => dout_b <= 'X';-- AFTER g_delay (data,tpLH,tpHL);
       end case;

end g_notif1;
--
--
procedure g_notif0 (
	        signal dout_b :  out vlbit;
   	     	signal data   :  in  vlbit;
	        signal enable :  in  vlbit;
             	tpLH   :  in  TIME;
	     	tpHL   :  in  TIME)
   is
   begin
       case enable is
         when '1' => dout_b <= 'Z' AFTER g_delay (data,tpLH,tpHL);
         when '0' => dout_b <= not(data) AFTER g_delay (data,tpLH,tpHL);
         when others => dout_b <= 'X';-- AFTER g_delay (data,tpLH,tpHL);
       end case;

end g_notif0;
--
--
-- D type flip flop
--    positive edge triggered
--    async. clear and preset (active high)
--
   PROCEDURE g_dffpc (
      SIGNAL   q        :  INOUT VLBIT;
      SIGNAL   qb        :  INOUT VLBIT;
      CONSTANT data     :  IN    VLBIT;   
      SIGNAL   clk      :  IN    VLBIT;
      SIGNAL   preset   :  IN    VLBIT;   
      SIGNAL   clear    :  IN    VLBIT;   
      SIGNAL   pwrupstate    :  IN    VLBIT)   
   IS 

   BEGIN
      ASSERT not( prising(clk) and pchanging(preset))
         REPORT "On g_dffpc: Clk and Preset both changing - Clk action suppressed";

      ASSERT not( prising(clk) and pchanging(clear))
         REPORT "On g_dffpc: Clk and Clear both changing - Clk action suppressed";

--      ASSERT not( pfalling(clear) and pfalling(preset))
--         REPORT "On g_dffpc: Preset and clear both released at same time";

      IF (prising(pwrupstate)) THEN
	 q <= '0';
      ELSIF clear /= '0' AND preset /= '0' THEN 
         q <= 'X';
      ELSIF preset = '1' AND clear = '0' THEN
         q <= '1';
      ELSIF clear = '1' AND preset = '0' THEN
         q <= '0';
--    ELSIF (pfalling(clear) and pfalling(preset)) THEN
--       q <= 'X';
      ELSIF (bitunknown(preset) and not(q = '1')) THEN
         q <= 'X';
      ELSIF (bitunknown(clear) and not(q = '0')) THEN
         q <= 'X';
      ELSIF (bitunknown(clk) and not(q = data)) THEN
         q <= 'X';
      ELSIF bitz(clk) THEN
         q <= (data and data);
      ELSIF (prising(clk) and not(pchanging(clear) or pchanging(preset))) THEN
         q <= (data and data);
            -- NOTE: The AND in the above lines handles the case where 
            --       data is a Z.  We want Q to be assigned to X.
      END IF;
      qb <= not(q);
END g_dffpc;
--
--
   PROCEDURE g_dffa (
      SIGNAL   q        :  INOUT VLBIT;
      SIGNAL   qb        :  INOUT VLBIT;
      CONSTANT data     :  IN    VLBIT;   
      SIGNAL   clk      :  IN    VLBIT;
      SIGNAL   preset   :  IN    VLBIT;   
      SIGNAL   clear    :  IN    VLBIT;   
      SIGNAL   pwrupstate :  IN  VLBIT;  
      t_clkout	        :  in  	 TIME)
   IS 

   BEGIN
      ASSERT not( prising(clk) and pchanging(preset))
         REPORT "On g_dffpc: Clk and Preset both changing - Clk action suppressed";

      ASSERT not( prising(clk) and pchanging(clear))
         REPORT "On g_dffpc: Clk and Clear both changing - Clk action suppressed";

--      ASSERT not( pfalling(clear) and pfalling(preset))
--         REPORT "On g_dffpc: Preset and clear both released at same time";

      IF (prising(pwrupstate)) THEN
	 q <= '0';
      ELSIF clear /= '0' AND preset /= '0' THEN 
         q <= 'X';
      ELSIF preset = '1' AND clear = '0' THEN
         q <= '1';
      ELSIF clear = '1' AND preset = '0' THEN
         q <= '0';
--    ELSIF (pfalling(clear) and pfalling(preset)) THEN
--       q <= 'X';
      ELSIF (bitunknown(preset) and not(q = '1')) THEN
         q <= 'X';
      ELSIF (bitunknown(clear) and not(q = '0')) THEN
         q <= 'X';
      ELSIF (bitunknown(clk) and not(q = data)) THEN
         q <= 'X';
      ELSIF bitz(clk) THEN
         q <= (data and data) after t_clkout;
      ELSIF (prising(clk) and not(pchanging(clear) or pchanging(preset))) THEN
         q <= (data and data) after t_clkout;
            -- NOTE: The AND in the above lines handles the case where 
            --       data is a Z.  We want Q to be assigned to X.
      END IF;
      qb <= not(q);
END g_dffa;
--
--
-- D type flip flop
--    positive edge triggered
--    sync. preset (active high) 
--    async. clear (active high) 
--
   PROCEDURE g_dff (
      SIGNAL   q        :  INOUT VLBIT;
      SIGNAL   qb       :  INOUT VLBIT;
      CONSTANT data     :  IN    VLBIT;   
      SIGNAL   clk      :  IN    VLBIT;
      SIGNAL   preset   :  IN    VLBIT;   
      SIGNAL   clear    :  IN    VLBIT;   
      SIGNAL   pwrupstate    :  IN    VLBIT;   
      t_clkout	        :  in  	 TIME)
   IS 
	
   BEGIN

      IF (not( now > 0 ns) ) THEN
	q <= pwrupstate; -- Power up state
      END IF;

      ASSERT not( prising(clk) and pchanging(clear))
         REPORT "On g_dff: Clk and Clear both changing - Clk action suppressed";

      ASSERT not( prising(clk) and pchanging(preset))
         REPORT "On g_dff: Clk and preset both changing - Clk action suppressed";

      IF clear = '1' THEN
         q <= '0';
      ELSIF preset = '1' THEN
         q <= '1';
      ELSIF (bitunknown(clear) and not(q = '0')) THEN
         q <= 'X';
      ELSIF (bitunknown(preset) and not(q = '1')) THEN
         q <= 'X';
      ELSIF (bitunknown(clk) and not(q = data)) THEN
         q <= 'X';
      ELSIF (prising(clk) and not(pchanging(clear)) and not(pchanging(preset))) THEN
         q <= (data and data) after t_clkout;
            -- NOTE: The AND in the above lines handles the case where 
            --       data is a Z.  We want Q to be assigned to X.

      END IF;
      qb <= not(q);
      --
   END g_dff;
--
--
-- D type flip flop
--    positive edge triggered
--    sync. preset (active high) 
--    async. clear (active high) 
--
   PROCEDURE g_dff_e (
      SIGNAL   q        :  INOUT VLBIT;
      SIGNAL   qb       :  INOUT VLBIT;
      SIGNAL   din      :  IN    VLBIT;   
      SIGNAL   clk      :  IN    VLBIT;
      SIGNAL   preset   :  IN    VLBIT;   
      SIGNAL   clear    :  IN    VLBIT;   
      SIGNAL   pwrupstate    :  IN    VLBIT;   
      t_clkout	        :  in  	 TIME)
   IS 
	
   variable data   : vlbit;

   BEGIN

      IF (not( now > 0 ns) ) THEN
	q <= pwrupstate; -- Power up state
      END IF;

      ASSERT not( prising(clk) and pchanging(clear))
         REPORT "On g_dff_e: Clk and Clear both changing - Clk action suppressed";

      IF clear = '1' THEN
         q <= '0';
      ELSIF (bitunknown(clear) and not(q = '0')) THEN
         q <= 'X';
      ELSIF (bitunknown(clk) and not(q = data)) THEN
         q <= 'X';
      ELSIF bitz(clk) THEN
         data := din OR preset;
      ELSIF (prising(clk) and not(pchanging(clear))) THEN
         data := din OR preset;
         q <= (data and data) after t_clkout;
            -- NOTE: The AND in the above lines handles the case where 
            --       data is a Z.  We want Q to be assigned to X.

      END IF;
      qb <= not(q);
      --
   END g_dff_e;
--


-- JK flip flop
--    positive edge triggered
--    async. clear and preset (active high)
--
   PROCEDURE g_jkff (
      SIGNAL   q        :  INOUT VLBIT;
      SIGNAL   qb       :  INOUT VLBIT;
      SIGNAL   j        :  IN    VLBIT;   
      SIGNAL   k        :  IN    VLBIT;   
      SIGNAL   preset   :  IN    VLBIT;   
      SIGNAL   clear    :  IN    VLBIT;   
      SIGNAL   clk      :  IN    VLBIT;
      SIGNAL   pwrupstate    :  IN    VLBIT;   
      t_clkout	        :  in  	 TIME)
   IS 

   BEGIN

      IF (not( now > 0 ns) ) THEN
	q <= pwrupstate; -- Power up state
      END IF;
      ASSERT not( prising(clk) and pchanging(preset))
         REPORT "On g_jkff: Clk and Preset both changing - Clk action suppressed";

      ASSERT not( prising(clk) and pchanging(clear))
         REPORT "On g_jkff: Clk and Clear both changing - Clk action suppressed";

--    ASSERT not( pfalling(preset) and pfalling(clear))
--       REPORT "On g_jkff: Preset and Clear both released at same time";

      IF preset /= '0' AND clear /= '0' THEN
         q <= 'X';
      ELSIF preset = '1' THEN
         q <= '1';
      ELSIF clear = '1' THEN
         q <= '0';
      ELSIF bitunknown(clk) THEN
         q <= 'X' AFTER t_clkout;
      ELSIF (bitunknown(preset) and not(q = '1')) THEN
         q <= 'X' AFTER t_clkout;
      ELSIF (bitunknown(clear) and not(q = '0')) THEN
         q <= 'X' AFTER t_clkout;
      ELSIF (prising(clk) and not(pchanging(clear) or pchanging(preset))) THEN
         IF (j = '1' AND k = '1') THEN
            q <= NOT q AFTER t_clkout;
         ELSIF (j = '1' AND k = '0') THEN
            q <= '1' AFTER t_clkout;
         ELSIF (j = '0' AND k = '1') THEN
            q <= '0' AFTER t_clkout;
         ELSIF (bitunknown(j)  OR bitunknown(k)) THEN
            q <= 'X' AFTER t_clkout;
         END IF;
      END IF;
      qb <= not(q);
   END g_jkff;
--
--
-- RS flip flop
--    negative edge triggered (H-to-L)
--    async. clear and preset (active high)
--
   PROCEDURE g_rsff (
      SIGNAL   q        :  INOUT VLBIT;
      SIGNAL   qb       :  INOUT VLBIT;
      SIGNAL   s        :  IN    VLBIT;   
      SIGNAL   r        :  IN    VLBIT;   
      SIGNAL   clk      :  IN    VLBIT;
      SIGNAL   preset   :  IN    VLBIT;   
      SIGNAL   clear    :  IN    VLBIT;   
      SIGNAL   pwrupstate    :  IN    VLBIT;   
      t_clkout	        :  in  	 TIME)
   IS 

   BEGIN

      IF (not( now > 0 ns) ) THEN
	q <= pwrupstate; -- Power up state
      END IF;
      ASSERT not( pfalling(clk) and pchanging(preset))
         REPORT "On g_rsff: Clk and Preset both changing - Clk action suppressed";

      ASSERT not( pfalling(clk) and pchanging(clear))
         REPORT "On g_rsff: Clk and Clear both changing - Clk action suppressed";

    ASSERT not( prising(preset) and prising(clear))
       REPORT "On g_rsff: Preset and Clear both released at same time";

      IF preset /= '0' AND clear /= '0' THEN
         q <= 'X';
      ELSIF preset = '1' THEN
         q <= '1';
      ELSIF clear = '1' THEN
         q <= '0';
      ELSIF bitunknown(clk) THEN
         q <= 'X' AFTER t_clkout;
      ELSIF (bitunknown(preset) and not(q = '1')) THEN
         q <= 'X' AFTER t_clkout;
      ELSIF (bitunknown(clear) and not(q = '0')) THEN
         q <= 'X' AFTER t_clkout;
      ELSIF (prising(clk) and not(pchanging(clear) or pchanging(preset))) THEN
         IF (s = '0' AND r = '0') THEN
            q <= q AFTER t_clkout;
         ELSIF (s = '1' AND r = '0') THEN
            q <= '1' AFTER t_clkout;
         ELSIF (s = '0' AND r = '1') THEN
            q <= '0' AFTER t_clkout;
         ELSIF (s = '1' AND r = '1') THEN
            q <= 'X' AFTER t_clkout;
         ELSIF (bitunknown(s)  OR bitunknown(r)) THEN
            q <= 'X' AFTER t_clkout;
         END IF;
      END IF;
      qb <= not(q);
   END g_rsff;
--
--
-- T flip flop
--    positive edge triggered
--    async. clear and preset (active high)
--
   PROCEDURE g_tff (
      SIGNAL   q        :  INOUT VLBIT;
      SIGNAL   qb       :  INOUT VLBIT;
      SIGNAL   t        :  IN    VLBIT;   
      SIGNAL   clk      :  IN    VLBIT;
      SIGNAL   preset   :  IN    VLBIT;   
      SIGNAL   clear    :  IN    VLBIT;   
      SIGNAL   pwrupstate    :  IN    VLBIT;   
      t_clkout	        :  in  	 TIME)
   IS 

   BEGIN

      IF (not( now > 0 ns) ) THEN
	q <= pwrupstate; -- Power up state
      END IF;

      ASSERT not( prising(clk) and pchanging(preset))
         REPORT "On g_tff: Clk and Preset both changing - Clk action suppressed";

      ASSERT not( prising(clk) and pchanging(clear))
         REPORT "On g_tff: Clk and Clear both changing - Clk action suppressed";

--    ASSERT not( pfalling(preset) and pfalling(clear))
--       REPORT "On g_tff: Preset and Clear both released at same time";

      IF preset /= '0' AND clear /= '0' THEN
         q <= 'X';
      ELSIF preset = '1' THEN
         q <= '1';
      ELSIF clear = '1' THEN
         q <= '0';
      ELSIF bitunknown(clk) AND (now > 0 ns) THEN
         q <= 'X' AFTER t_clkout;
      ELSIF (bitunknown(preset) and not(q = '1')) THEN
         q <= 'X' AFTER t_clkout;
      ELSIF (bitunknown(clear) and not(q = '0')) THEN
         q <= 'X' AFTER t_clkout;
      ELSIF (prising(clk) and not(pchanging(clear) or pchanging(preset))) THEN
         IF (t = '1' ) AND q = '1' THEN
            q <= '0' AFTER t_clkout;
         ELSIF (t = '1' ) AND q = '0' THEN
            q <= '1' AFTER t_clkout;
         ELSIF (t = '0') THEN
            q <= q AFTER t_clkout;
         ELSIF (bitunknown(t) ) THEN
            q <= 'X' AFTER t_clkout;
         END IF;
      END IF;
      qb <= not(q);

   END g_tff;
--
--
--
-- MAXFF type flip flop
--    positive edge triggered D ff.
--    async. clear and preset (active high)
--    If both clear and preset HIGH, it become a latch
--
   PROCEDURE g_maxff (
      SIGNAL   q        :  INOUT VLBIT;
      SIGNAL   qb       :  INOUT VLBIT;
      CONSTANT data     :  IN    VLBIT;   
      SIGNAL   clk      :  IN    VLBIT;
      SIGNAL   preset   :  IN    VLBIT;   
      SIGNAL   clear    :  IN    VLBIT;   
      SIGNAL   pwrupstate :  IN  VLBIT;   
      t_clkout	        :  in  	 TIME)
   IS 

   BEGIN
      IF (not( now > 0 ns) ) THEN
	q <= pwrupstate; -- Power up state
      END IF;

      ASSERT not( prising(clk) and pchanging(preset))
         REPORT "On g_maxff: Clk and Preset both changing - Clk action suppressed";

      ASSERT not( prising(clk) and pchanging(clear))
         REPORT "On g_maxff: Clk and Clear both changing - Clk action suppressed";

--      ASSERT not( pfalling(clear) and pfalling(preset))
--         REPORT "On g_maxff: Preset and clear both released at same time";

      IF clear = '1' AND preset = '1' THEN 
         q <= data after t_clkout;
      ELSIF preset = '1' AND clear = '0' THEN
         q <= '1';
      ELSIF clear = '1' AND preset = '0' THEN
         q <= '0';
--    ELSIF (pfalling(clear) and pfalling(preset)  AND (now > 0 ns)) THEN
--       q <= 'X';
      ELSIF (bitunknown(preset) and not(q = '1') AND (now > 0 ns)) THEN
         q <= 'X';
      ELSIF (bitunknown(clear) and not(q = '0') AND (now > 0 ns)) THEN
         q <= 'X';
      ELSIF (bitunknown(clk) and not(q = data) AND (now > 0 ns)) THEN
         q <= 'X';
      ELSIF bitz(clk) THEN
         q <= (data and data) after t_clkout;
      ELSIF (prising(clk) and not(pchanging(clear) or pchanging(preset))) THEN
         q <= (data and data) after t_clkout;
            -- NOTE: The AND in the above lines handles the case where 
            --       data is a Z.  We want Q to be assigned to X.
      END IF;
      qb <= not(q);
END g_maxff;
--
--
--
--
-- RAFF type flip flop
--    positive edge triggered D ff.
--    async. clear and preset (active high)
--    If both clear and preset HIGH, it become a latch
--
   PROCEDURE g_raff (
      SIGNAL   q        :  INOUT VLBIT;
      SIGNAL   qb       :  INOUT VLBIT;
      CONSTANT data     :  IN    VLBIT;   
      SIGNAL   clk      :  IN    VLBIT;
      SIGNAL   preset   :  IN    VLBIT;   
      SIGNAL   clear    :  IN    VLBIT;   
      SIGNAL   pwrupstate :  IN  VLBIT;   
      t_clkout	        :  in  	 TIME)
   IS 

   BEGIN
      IF (not( now > 0 ns) ) THEN
	q <= pwrupstate; -- Power up state
      END IF;

      ASSERT not( prising(clk) and pchanging(preset))
         REPORT "On g_raff: Clk and Preset both changing - Clk action suppressed";

      ASSERT not( prising(clk) and pchanging(clear))
         REPORT "On g_raff: Clk and Clear both changing - Clk action suppressed";

--      ASSERT not( pfalling(clear) and pfalling(preset))
--         REPORT "On g_raff: Preset and clear both released at same time";

      IF clear = '1' AND preset = '1' THEN 
         q <= data;
      ELSIF preset = '1' AND clear = '0' THEN
         q <= '1';
      ELSIF clear = '1' AND preset = '0' THEN
         q <= '0';
--    ELSIF (pfalling(clear) and pfalling(preset)  AND (now > 0 ns)) THEN
--       q <= 'X';
      ELSIF (bitunknown(preset) and not(q = '1') AND (now > 0 ns)) THEN
         q <= 'X';
      ELSIF (bitunknown(clear) and not(q = '0') AND (now > 0 ns)) THEN
         q <= 'X';
      ELSIF (bitunknown(clk) and not(q = data) AND (now > 0 ns)) THEN
         q <= 'X';
      ELSIF bitz(clk) THEN
         q <= (data and data) after t_clkout;
      ELSIF (prising(clk) and not(pchanging(clear) or pchanging(preset))) THEN
         q <= (data and data) after t_clkout;
            -- NOTE: The AND in the above lines handles the case where 
            --       data is a Z.  We want Q to be assigned to X.
      END IF;
      qb <= not(q);
END g_raff;
--
--
   PROCEDURE g_lff (
      SIGNAL   q        :  INOUT VLBIT;
      SIGNAL   qb       :  INOUT VLBIT;
      CONSTANT data     :  IN    VLBIT;   
      SIGNAL   gate     :  IN    VLBIT;
      SIGNAL   preset   :  IN    VLBIT;   
      SIGNAL   clear    :  IN    VLBIT;   
      SIGNAL   pwrupstate :  IN  VLBIT;   
      t_clkout	        :  in  	 TIME)
   IS 

   BEGIN
      IF (not( now > 0 ns) ) THEN
	q <= pwrupstate; -- Power up state
      END IF;

      IF preset /= '0' and clear /= '0' THEN
         q <= 'X';
      ELSIF preset = '1' THEN
         q <= '1';
      ELSIF clear = '1' THEN
         q <= '0';
--      ELSIF (pfalling(clear) and pfalling(preset) and (gate /= '0')) THEN
--         q <= 'X';
      ELSIF (bitunknown(preset) and not(q = '1')) THEN
         q <= 'X';
      ELSIF (bitunknown(clear) and not(q = '0')) THEN
         q <= 'X';
      ELSIF (bitunknown(gate) and not(q = data)) THEN
         q <= 'X' after t_clkout;
      ELSIF bitz(gate) THEN
         q <= (data and data);
      ELSIF (bitunknown(data) and (gate = '0')) THEN
         q <= 'X' after t_clkout;
      ELSIF (gate = '0' ) THEN
         q <= (data and data) after t_clkout;
            -- NOTE: The AND in the above lines handles the case where 
            --       data is a Z.  We want Q to be assigned to X.
      END IF;
END g_lff;

--
   PROCEDURE g_lffa (
      SIGNAL   q        :  INOUT VLBIT;
      SIGNAL   qb       :  INOUT VLBIT;
      CONSTANT data     :  IN    VLBIT;   
      SIGNAL   gate     :  IN    VLBIT;
      SIGNAL   preset   :  IN    VLBIT;   
      SIGNAL   clear    :  IN    VLBIT;   
      SIGNAL   pwrupstate :  IN  VLBIT;   
      t_clkout	        :  in  	 TIME)
   IS 

   BEGIN
      IF (not( now > 0 ns) ) THEN
	q <= pwrupstate; -- Power up state
      END IF;

      IF preset /= '0' and clear /= '0' THEN
         q <= 'X';
      ELSIF preset = '1' THEN
         q <= '1';
      ELSIF clear = '1' THEN
         q <= '0';
--      ELSIF (pfalling(clear) and pfalling(preset) and (gate /= '0')) THEN
--         q <= 'X';
      ELSIF (bitunknown(preset) and not(q = '1')) THEN
         q <= 'X';
      ELSIF (bitunknown(clear) and not(q = '0')) THEN
         q <= 'X';
      ELSIF (bitunknown(gate) and not(q = data)) THEN
         q <= 'X' after t_clkout;
      ELSIF bitz(gate) THEN
         q <= (data and data) after t_clkout;
      ELSIF (bitunknown(data) and (gate = '0')) THEN
         q <= 'X' after t_clkout;
      ELSIF (gate = '0' ) THEN
         q <= (data and data) after t_clkout;
            -- NOTE: The AND in the above lines handles the case where 
            --       data is a Z.  We want Q to be assigned to X.
      END IF;
END g_lffa;
--
   PROCEDURE g_lffh (
      SIGNAL   q        :  INOUT VLBIT;
      SIGNAL   qb       :  INOUT VLBIT;
      CONSTANT data     :  IN    VLBIT;   
      SIGNAL   gate     :  IN    VLBIT;
      SIGNAL   preset   :  IN    VLBIT;   
      SIGNAL   clear    :  IN    VLBIT;   
      SIGNAL   pwrupstate :  IN  VLBIT;   
      t_clkout	        :  in  	 TIME)
   IS 

   BEGIN
      IF (not( now > 0 ns) ) THEN
	q <= pwrupstate; -- Power up state
      END IF;

      IF preset /= '0' and clear /= '0' THEN
         q <= 'X';
      ELSIF preset = '1' THEN
         q <= '1';
      ELSIF clear = '1' THEN
         q <= '0';
--      ELSIF (pfalling(clear) and pfalling(preset) and (gate /= '0')) THEN
--         q <= 'X';
      ELSIF (bitunknown(preset) and not(q = '1')) THEN
         q <= 'X';
      ELSIF (bitunknown(clear) and not(q = '0')) THEN
         q <= 'X';
      ELSIF (bitunknown(gate) and not(q = data)) THEN
         q <= 'X' after t_clkout;
      ELSIF bitz(gate) THEN
         q <= (data and data) after t_clkout;
      ELSIF (bitunknown(data) and (gate = '1')) THEN
         q <= 'X' after t_clkout;
      ELSIF (gate = '1' ) THEN
         q <= (data and data) after t_clkout;
            -- NOTE: The AND in the above lines handles the case where 
            --       data is a Z.  We want Q to be assigned to X.
      END IF;
END g_lffh;


--
function g_delay (
		data: VLBIT; 
		tphl, tplh: TIME) 
   RETURN TIME IS
      VARIABLE delay: TIME:= 0 ns;
      BEGIN
         if (data = '1') then delay := tphl;
         else delay := tplh;
         end if;
      return delay;
END g_delay;
--
--
function g_smux2 (
                signal i0   : vlbit;
                signal i1   : vlbit;
                signal s0   : vlbit;
                tpLH   : TIME;
                tpHL   : TIME)
    RETURN VLBIT IS
    BEGIN
	IF s0 = '0' THEN
         	RETURN i0;
      	ELSIF s0 = '1' THEN
         	RETURN i1;
      	END IF;
        RETURN 'X';
   END g_smux2;
--
procedure g_smux (
	        signal dout :  out vlbit;
	        signal i0   :  in  vlbit;
	        signal i1   :  in  vlbit;
	        signal i2   :  in  vlbit;
	        signal i3   :  in  vlbit;
	        signal s0   :  in  vlbit;
	        signal s1   :  in  vlbit;
             	tpLH   :  in  TIME;
	     	tpHL   :  in  TIME)
   IS

   BEGIN

      IF s0 = '0' AND s1 = '0'  THEN
         dout <= i0;
      ELSIF s0 = '0' AND s1 = '1'  THEN
         dout <= i1;
      ELSIF s0 = '1' AND s1 = '0'  THEN
         dout <= i1;
      ELSIF s0 = '1' AND s1 = '1'  THEN
         dout <= i1;
      ELSIF (bitunknown(s0) OR bitunknown(s1)) THEN
         dout <= 'X';
      END IF;
   END g_smux;
--
--
-- JKD flip flop
--    positive edge triggered
--    async. clear and preset (active high)
--
   PROCEDURE g_jkdff (
      SIGNAL   q        :  INOUT VLBIT;
      SIGNAL   qb       :  INOUT VLBIT;
      SIGNAL   ji       :  IN    VLBIT;   
      SIGNAL   ki       :  IN    VLBIT;   
      SIGNAL   jf       :  IN    VLBIT;   
      SIGNAL   kf       :  IN    VLBIT;   
      SIGNAL   clk      :  IN    VLBIT;
      SIGNAL   ld       :  IN    VLBIT;
      SIGNAL   sel      :  IN    VLBIT;
      SIGNAL   preset   :  IN    VLBIT;   
      SIGNAL   clear    :  IN    VLBIT;   
      SIGNAL   pwrupstate    :  IN    VLBIT;   
      t_clkout	        :  in  	 TIME)
   IS 

   variable j       : vlbit;
   variable k       : vlbit;
 
   BEGIN

      IF ld = '0' THEN
         j := ji;
      ELSIF ld = '1' THEN
         j := jf;
      ELSIF bitunknown(ld) THEN
         j := 'X';
      END  IF;

      IF ld = '0' AND sel = '0' THEN
         k := ki;
      ELSIF ld = '0' AND sel = '1' THEN
         k := not(j);
      ELSIF ld = '1' AND sel = '0' THEN
         k := kf;
      ELSIF ld = '0' AND sel = '1' THEN
         k := kf;
      ELSIF bitunknown(ld) or bitunknown(sel) THEN
         k := 'X';
      END  IF;


      IF (not( now > 0 ns) ) THEN
	q <= pwrupstate; -- Power up state
      END IF;
      ASSERT not( prising(clk) and pchanging(preset))
         REPORT "On g_jkdff: Clk and Preset both changing - Clk action suppressed";

      ASSERT not( prising(clk) and pchanging(clear))
         REPORT "On f_jkdff: Clk and Clear both changing - Clk action suppressed";

--    ASSERT not( pfalling(preset) and pfalling(clear))
--       REPORT "On f_jkdff: Preset and Clear both released at same time";

      IF preset /= '0' AND clear /= '0' THEN
         q <= 'X';
      ELSIF preset = '1' THEN
         q <= '1';
      ELSIF clear = '1' THEN
         q <= '0';
      ELSIF bitunknown(clk) THEN
         q <= 'X' AFTER t_clkout;
      ELSIF (bitunknown(preset) and not(q = '1')) THEN
         q <= 'X' AFTER t_clkout;
      ELSIF (bitunknown(clear) and not(q = '0')) THEN
         q <= 'X' AFTER t_clkout;
      ELSIF (prising(clk) and not(pchanging(clear) or pchanging(preset))) THEN
         IF (j = '1' AND k = '1') THEN
            q <= NOT q AFTER t_clkout;
         ELSIF (j = '1' AND k = '0') THEN
            q <= '1' AFTER t_clkout;
         ELSIF (j = '0' AND k = '1') THEN
            q <= '0' AFTER t_clkout;
         ELSIF (bitunknown(j)  OR bitunknown(k)) THEN
            q <= 'X' AFTER t_clkout;
         END IF;
      END IF;
      qb <= not(q);
   END g_jkdff;
--
--
procedure g_vxff (
      SIGNAL   q        :  INOUT VLBIT;
      SIGNAL   qb       :  INOUT VLBIT;
      SIGNAL   din      :  IN    VLBIT;
      SIGNAL   clk      :  IN    VLBIT;
      SIGNAL   preset   :  IN    VLBIT;
      SIGNAL   clear    :  IN    VLBIT;
      SIGNAL   qsel     :  IN    VLBIT;
      SIGNAL   osel     :  IN    VLBIT;
      SIGNAL   pwrupstate    :  IN    VLBIT;
      t_clkout          :  in    TIME)
   IS
      variable data     : vlbit;
      variable qout     : vlbit;
   BEGIN
      IF (not( now > 0 ns) ) THEN
        q <= pwrupstate; -- Power up state
      END IF;  

      ASSERT not( prising(clk) and pchanging(clear))
         REPORT "On g_vxff: Clk and Clear both changing - Clk action suppressed";
 
      IF clear = '1' THEN
         q <= '0';
      ELSIF (bitunknown(clear) and not(q = '0')) THEN
         q <= 'X';
      ELSIF (bitunknown(clk) and not(q = data)) THEN
         q <= 'X';
      ELSIF bitz(clk) THEN
         data := din OR preset;
      ELSIF (prising(clk) and not(pchanging(clear))) THEN
         data := din OR preset;
         q <= (data and data) after t_clkout;
            -- NOTE: The AND in the above lines handles the case where
            --       data is a Z.  We want Q to be assigned to X.
               
      END IF;
      IF qsel = '1' and osel = '1' THEN
 	q <= (data and data);
      END IF;
--         qout <= q;
--      ELSIF qsel = '1' and osel = '0' THEN
--         qout := qb;
--      ELSIF bitunknown(qsel) THEN
--         qout := 'X';
--      END IF;
--      IF osel = '0' THEN
--         q <= qout;
--      ELSIF osel = '1' THEN
--         q <= din;
--      ELSIF bitunknown(osel) THEN
--         q <= 'X';
--      END IF;
       qb <= not(q);
   END g_vxff;

   PROCEDURE p_chksuho (
      SIGNAL   data     :  IN    VLBIT;
      SIGNAL   clk      :  IN    VLBIT;
      t_setup           :  IN     TIME;
      t_hold            :  IN     TIME;
      t_clk             :  INOUT  TIME;
      t_data            :  INOUT  TIME;
      setname           :  IN     STRING)
   IS BEGIN
         IF (prising (clk)) AND (now > 0 ns) AND ((t_data + t_setup) > now) THEN
            PUT     ("Error: Setup time violation at time ",now);
            PUTLINE (" for ",setname);
            PUTLINE ("       Data last changed at ", t_data);
            PUT     ("       Setup of ",(now - t_data));
            PUTLINE (" less than required ",t_setup);
         END IF;

         IF pfalling (clk) THEN
            t_clk := now;
         END IF;

         IF pchanging (data) THEN
            t_data := now;
            IF (now > 0 ns) AND ((t_clk + t_hold) > now) THEN
               PUT     ("Error: Hold time violation at time ",now);
               PUTLINE (" for ",setname);
               PUTLINE ("       Clock last changed at ", t_clk);
               PUT     ("       Hold of ",(now - t_clk));
               PUTLINE (" less than required ",t_hold);
            END IF;
         END IF;
   END p_chksuho;
--
-- 
   PROCEDURE p_chksuho_hl (
      SIGNAL   data     :  IN    VLBIT;
      SIGNAL   clk      :  IN    VLBIT;
      t_setup           :  IN     TIME;
      t_hold            :  IN     TIME;
      t_clk             :  INOUT  TIME;
      t_data            :  INOUT  TIME;
      setname           :  IN     STRING)
   IS BEGIN
         IF (pfalling (clk)) AND (now > 0 ns) AND ((t_data + t_setup) > now) THEN
            PUT     ("Error: Setup time violation at time ",now);
            PUTLINE (" for ",setname);
            PUTLINE ("       Data last changed at ", t_data);
            PUT     ("       Setup of ",(now - t_data));
            PUTLINE (" less than required ",t_setup);
         END IF;

         IF prising (clk) THEN
            t_clk := now;
         END IF;

         IF pchanging (data) THEN
            t_data := now;
            IF (now > 0 ns) AND ((t_clk + t_hold) > now) THEN
               PUT     ("Error: Hold time violation at time ",now);
               PUTLINE (" for ",setname);
               PUTLINE ("       Clock last changed at ", t_clk);
               PUT     ("       Hold of ",(now - t_clk));
               PUTLINE (" less than required ",t_hold);
            END IF;
         END IF;
END p_chksuho_hl;

END j2vlib;
