/* Backplane PHY 1394 Project - C level source code
 *
 *  Prepared by J. Norris, Technobox Inc
 *
 *  Revision History:
 *
 *  10/14/95: Initial coding
 *  11/02/95: Added support for audio ISO channel
 *  1/16/96: Modified DVCI.C to create BPLN.C for backplane PHY
 *  2/5/96:  Added changes per TI request:
 *
 *                 1. Support saving 12C01 regs in FLASH memory
 *                 2. Add transmit/receive continuous ("Loop" option)
 *                 3. Option to transfer more than 1 quadlet
 *
 */

/* Define hardware register addresses */

#define   la_14c01        (*(volatile int *)0xF000)     /* Load 14C01 chip address latch             */
#define   la_21lv03       (*(volatile int *)0xF100)     /* Load 21LV03 chip address latch            */
#define   la_11c01        (*(volatile int *)0xF200)     /* Load 11C01 chip address latch             */
#define   la_14c01ctrl    (*(volatile int *)0xF300)     /* Load 14C01 control register               */
#define   la_trig         (*(volatile int *)0xF400)     /* Logic Analyzer Trigger                    */
#define   la_ctrlstat     (*(volatile int *)0xF500)     /* Write control register/Read Status reg    */
#define   la_link         (*(volatile int *)0xF600)     /* Read/Write link controller ident reg      */
#define   la_xfercntr     (*(volatile int *)0xF700)     /* Read/write transfer counter               */
#define   la_dmago        (*(volatile int *)0xF800)     /* Write here kicks off DMA transfer         */
#define   la_addrhi       (*(volatile int *)0xF900)     /* Load address counter hi                   */
#define   la_addrlo       (*(volatile int *)0xFA00)     /* Load address counter low                  */
#define   la_geogrd       (*(volatile int *)0xFB00)     /* Read geographical address from backplane  */
#define   la_dmactrl      (*(volatile int *)0xFC00)     /* DMA control register                      */
#define   la_spareFD      (*(volatile int *)0xFD00)     /* Spare address                             */
#define   la_spareFE      (*(volatile int *)0xFE00)     /* Spare address                             */
#define   la_spareFF      (*(volatile int *)0xFF00)     /* Spare address                             */

#define   la_14c01_addr   (0xF000)
#define   la_21lv03_addr  (0xF100)
#define   la_11c01_addr   (0xF200)

/* Define areas of DATA space for DMA transfer buffers */

#define   dmabuff_addr    (0x4000)

/* Define control/status register bits */

int       ctrlreg_image;               /* Image of control/status register */

#define   ctrl_led0       0x0001       /* LED 0 (D4 on silk screen) */
#define   ctrl_led1       0x0002       /* LED 1 (D5 on silk screen) */
#define   ctrl_led2       0x0004       /* LED 2 (D6 on silk screen) */
#define   ctrl_rts        0x0008       /* Request to Send on RS232  */
#define   ctrl_ra         0x0010       /* Reset to 11C01 port       */
#define   ctrl_rb         0x0020       /* Reset to 21LV03 port      */
#define   ctrl_rc         0x0040       /* Reset to 14C01 port       */

/* Define 14C01 control register */

typedef struct
          {
            unsigned enapri:1, oscsel:1, enexid:1, enexpri:1,
                     id:6, pri:4, pwrdn:1, ptest:1;
          }
          ctrl14c01_bits;           /* Bit definitions for 14C01 control reg */

union     ctrl14c01_u                        /* Link/Version bits  */
          {
            ctrl14c01_bits ctrl14c01;
            unsigned int ctrl14c01_image;
          }
          ctrl14c01_union;

#define   ctrl14c01_enapri  0x0001     /* Enable pri                    */
#define   ctrl14c01_oscsel  0x0002     /* Select oscillator             */
#define   ctrl14c01_enexid  0x0004     /* Enable external ID            */
#define   ctrl14c01_enexpri 0x0008     /* Enable external priority      */
#define   ctrl14c01_id      0x03F0     /* External ID                   */
#define   ctrl14c01_pri     0x3C00     /* External priority             */
#define   ctrl14c01_pwrdn   0x4000     /* Power down mode               */
#define   ctrl14c01_ptest   0x8000     /* P test                        */

#define   ctrl14c01_init    ctrl14c01_oscsel    /* Init value for ctrl reg */

/* Define external assembly language procedures/variables */

extern char getc();                     /* Read RS232 character        */
extern void putc();                     /* Write RS232 character       */
extern int  getcpdg();                  /* Return TRUE if RS232 pend.  */
extern void loader();                   /* Load RAM resident code      */
extern char *getstr();                  /* Get string const from flash */
extern int  getconst16();               /* Get 16-bit const from flash */
extern long getconst32();               /* Get 32-bit const from flash */
extern void suspend();                  /* Programmed delay            */
extern void burnloc16();                /* Burn flash memory location  */
extern void burnloc32(unsigned int i,unsigned long j);                /* Burn 32 bit qty in flash    */
extern void eraseblk();                 /* Erase select FLASH block    */
extern void erase();                    /* Erase entire FLASH memory   */
extern void burnsram();                 /* Burn FLASH memory with SRAM */

extern int bitlen;
extern int bitlen2;                     /* Baud rate constants         */

int oddeven;

int chksum;                             /* For Monitor upload */

int global_pos;                        /* # digits in user-entered hex number */

int debug;                       /* Debug mode on/off   */
int expertmode;                  /* Expert mode on/off  */

/* Define misc controls */

#define    TRUE    (1)                  /* Define TRUE */
#define    FALSE   (0)                  /* Define FALSE */

/* Define 12C01 internal register addresses          */

#define   link_version_adr        0x00                    /* Version/Revision    */
#define   link_nodeadr_adr        0x04                    /* Node address        */
#define   link_control_adr        0x08                    /* Control register    */
#define   link_intr_adr           0x0c                    /* Interrupt pending   */
#define   link_imask_adr          0x10                    /* Interrupt mask      */
#define   link_cyctime_adr        0x14                    /* Cycle timer         */
#define   link_isoport_adr        0x18                    /* Iso xfer port adrs  */
#define   link_diags_adr          0x20                    /* Diagnostics         */
#define   link_phychip_adr        0x24                    /* Phy chip access     */
#define   link_atfstatus_adr      0x30                    /* Async xfer status   */
#define   link_itfstatus_adr      0x34                    /* Iso xfer status     */
#define   link_grfstatus_adr      0x3c                    /* Gen rcvr fifo stat  */

#define   link_atffirst_adr       0x80                    /* ATF First data      */
#define   link_atfcont_adr        0x84                    /* ATF continue        */
#define   link_atffirstup_adr     0x88                    /* ATF First & Update  */
#define   link_atfcontup_adr      0x8C                    /* ATF Continue/Update */
#define   link_itffirst_adr       0x90                    /* ITF First           */
#define   link_itfcont_adr        0x94                    /* ITF Continue        */
#define   link_itffirstup_adr     0x98                    /* ITF First & Update  */
#define   link_itfcontup_adr      0x9c                    /* ITF Continue/update */
#define   link_grfdata_adr        0xc0                    /* GRF data            */

/* Define 12c01 register bit structures */

typedef struct
          {
            unsigned revision:16,
                     version:16;
          }
          link_version_bits;                         /* Version register    */

typedef struct
          {
            unsigned rsvd1:4, atack:4, rsvd2:8,
                     nodenumber:6, busnumber:10;
          }
          link_nodeadr_bits;                         /* Node address        */

typedef struct
          {
            unsigned revaen:1, rsvd3:5, irp2en:1, irp1en:1, rsvd2:1,
                     cyten:1, cysrc:1, cymas:1, rsvd1:2, atrc:2,
                     blkbusdep:4, rstrx:1, rsttx:1,
                     psro:1, pson:1, psbz:1, rxen:1,
                     txen:1, bsyctrl:3, rxsld:1, idval:1;
          }
          link_control_bits;                          /*  Control reg         */

typedef struct
          {
            unsigned iarbfl:1, rsvd5:5, carbfl:1, cylst:1,
                     cypnd:1, cydne:1, cyst:1, cysec:1,
                     rsvd4:3, tcerr:1,
                     hdrer:1, sntrj:1,
                     rsvd3:1, atstk:1, itstk:1, rsvd2:3,
                     cmdrst:1, rxdta:1, txrdy:1, rsvd1:1,
                     phrst:1, phrrx:1, phint:1, intr:1;
          }
          link_intr_bits;                 /* Pending interrupts  */

typedef struct
          {
            unsigned iarbfl:1, rsvd5:5, carbfl:1, cylst:1,
                     cypnd:1, cydne:1, cyst:1, cysec:1,
                     rsvd4:3, tcerr:1,
                     hdrer:1, sntrj:1,
                     rsvd3:1, atstk:1, itstk:1, rsvd2:3,
                     cmdrst:1, rxdta:1, txrdy:1, rsvd1:1,
                     phrst:1, phrrx:1, phint:1, intr:1;
          }
          link_imask_bits;               /* Interrupt mask       */

typedef struct
          {
            unsigned cycoffset:12, cyccount_lo:4,
                     cyccount_hi:9, seccount:7;
          }
          link_cyctime_bits;             /* Cycle timer */

typedef struct
          {
            unsigned rsvd1:16,
                     irport2:8, irport1:8;
          }
          link_isoport_bits;             /* Isoc Port Numbers */

typedef struct
          {
            unsigned rsvd3: 16,
                     rsvd2:10, regrw:1, rsvd1:1,
                     frgp:1, arbgp:1, bsyfl:1, ensp:1;
          }
          link_diags_bits;               /* Diagnostics  */

typedef struct
          {
            unsigned phyrxdata:8, phyrxad:4, rsvd2:4,
                     phyrgdata:8, phyrgad:4,
                     rsvd1:2, wrphy:1, rdphy:1;
          }
          link_phychip_bits;             /* Phy interface bits */

typedef struct
          {
            unsigned size:9, rsvd4:3, cir:1, rsvd3:3,
                     empty:1, aie:1,
                     rsvd2:9, fourav:1, rsvd1:2, aif:1, full:1;
          }
          link_atfstatus_bits;           /* ATF status           */

typedef struct
          {
            unsigned size:9, rsvd4:3, cir:1, rsvd3:3,
                     empty:1, aie:1,
                     rsvd2:9, fourav:1, rsvd1:2, aif:1, full:1;
          }
          link_isostatus_bits;           /* ITF status */

typedef struct
          {
            unsigned size:9, rsvd4:3, cir:1, rsvd3:2, cd:1,
                     empty:1, aie:1,
                     rsvd2:1, fourth:1, rsvd1:10, aif:1, full:1;
          }
          link_grfstatus_bits;           /* GRF status   */

/* Define unions of bit structures and 32-bit registers for 12C01 */

union     link_version_u                        /* Link/Version bits  */
          {
            link_version_bits link_version;
            unsigned long link_version_reg;
          }
          link_version_union;

union     link_nodeadr_u                        /* Node address        */
          {
            link_nodeadr_bits link_nodeadr;
            unsigned long link_nodeadr_reg;
          }
          link_nodeadr_union;

union     link_control_u                        /*  Control reg         */
          {
            link_control_bits link_control;
            unsigned long link_control_reg;
          }
          link_control_union;

union     link_intr_u
          {
            link_intr_bits link_intr;
            unsigned long link_intr_reg;
          }
          link_intr_union;

union     link_imask_u             /* Interrupt mask       */
          {
            link_imask_bits link_imask;
            unsigned long link_imask_reg;
          }
          link_imask_union;

union     link_cyctime_u           /* Cycle timer */
          {
            link_cyctime_bits link_cyctime;
            unsigned long link_cyctime_reg;
          }
          link_cyctime_union;

union     link_isoport_u           /* Isoc Port Numbers */
          {
            link_isoport_bits link_isoport;
            unsigned long link_isoport_reg;
          }
          link_isoport_union;

union     link_diags_u             /* Diagnostics  */
          {
            link_diags_bits link_diags;
            unsigned long link_diags_reg;
          }
          link_diags_union;

union     link_phychip_u           /* Phy interface bits */
          {
            link_phychip_bits link_phychip;
            unsigned long link_phychip_reg;
          }
          link_phychip_union;

union     link_atfstatus_u         /* ATF status           */
          {
            link_atfstatus_bits link_atfstatus;
            unsigned long link_atfstatus_reg;
          }
          link_atfstatus_union;

union     link_isostatus_u         /* ITF status */
          {
            link_isostatus_bits link_isostatus;
            unsigned long link_isostatus_reg;
          }
          link_isostatus_union;

union     link_grfstatus_u         /* GRF status   */
          {
            link_grfstatus_bits link_grfstatus;
            unsigned long link_grfstatus_reg;
          }
          link_grfstatus_union;

/* Define PHY chip register addresses */

#define   phy_id_adr        0x00            /* Physical ID, root, CPS */
#define   phy_gc_adr        0x01            /* shbv, ibr, gap count   */
#define   phy_np_adr        0x02            /* Number of ports, speed */
#define   phy_linestat_adr  0x03            /* Line status, ch, con   */

/* Define PHY chip bit structures */

struct    phy_id_bits
          {
            unsigned physicalid:6, root:1, cps:1;
          };

struct    phy_gc_bits
          {
            unsigned rhb:1, ibr:1, gc:6;
          };

struct    phy_np_bits
          {
            unsigned spd:2, rsvd1:1, np:5;
          };

struct    phy_linestat_bits
          {
            unsigned astat:2, bstat:2, ch:1, con:1, rsvd1:2;
          };

/* Define ASYNC quadlet packet format & Define union with 32-bit array */

typedef   struct
          {
            unsigned
                     priority:4, tcode:4, rt:2, tlabel:6, spd:2, rsvd1:14,
                     destoffhigh:16, nodenumber:6, busnumber:10,
                     destofflow_low:16, destofflow_high:16,
                     data_low:16, data_high:16;
          } async_quad_xmt;

union     async_quad_uxmt
          {
            async_quad_xmt bits;        /* Bits of structure as defined above */
            unsigned long regs[4];      /* 4 32-bit values for register equivalents */
          } async_quadxmt_union;

typedef   struct
          {
            unsigned
                     priority:4, tcode:4, rt:2, tlabel:6, dst_nodenumber:6, dst_busnumber:10,
                     destoffhigh:16, src_nodenumber:6, src_busnumber:10,
                     destofflow_low:16, destofflow_high:16,
                     data_low:16, data_high:16,
                     acksent:4, rsvd2:12, spdrcv:2, rsvd3:14;
          } async_quad_rcv;

union     async_quad_urcv
          {
            async_quad_rcv bits;        /* Received data bits */
            unsigned long regs[5];      /* Received data words */
          } async_quadrcv_union;

typedef   struct                        /* Iso Header   */
          {
            unsigned
              sy:4,
              spd:4,
              channum:8,
              datalength:16;
          } iso_header;

union     iso_header_u                  /* Define bits & reg varient of ISO packet header */
          {
            iso_header bits;            /* Bit mapped version */
            unsigned long reg;          /* 32-bit version     */
          } iso_header_union;

/* Define IEEE 1394 Transaction Labels */

#define   tcode_wrquad     0x00            /* Write request for data quadlet  */
#define   tcode_wrblock    0x01            /* Write request for data block    */
#define   tcode_wrresponse 0x02            /* Write response                  */
#define   tcode_rdquad     0x04            /* Read response for data quadlet  */
#define   tcode_rdblock    0x05            /* Read response for data block    */
#define   tcode_rdquadrsp  0x06            /* Response subtract to rqst quad  */
#define   tcode_rdblockrsp 0x07            /* Response subtract to rqst block */
#define   tcode_cyclestart 0x08            /* Request to start iso cycle      */
#define   tcode_lockrqst   0x09            /* Lock request                    */
#define   tcode_isoblock   0x0a            /* ISO data block                  */
#define   tcode_lockrsp    0x0b            /* Response to lock request        */
#define   tcode_selfid     0x0E            /* Self-id packet                  */

/* Miscellaneous Defines */

#define   carret            0x0d            /* Carriage return char     */
#define   backspace         0x08            /* Backspace character      */

/* Miscellaneous 1394-related defines */

#define   link_fifo_size    508            /* Size of 12C01 fifo in quadlets */
#define   link_grf_size     150             /* # quadlets in GRF */
#define   link_atf_size     150             /* # quadlets in ATF */
#define   link_itf_size     (link_fifo_size - (link_grf_size + link_atf_size))

#define   fsaveadr          0x8000          /* Save address for regs in flash */
#define   foffset           0x1000          /* Offset for load into SRAM      */
#define   magicnumb         0x19283746      /* Magic # indicating regs were saved */

/* Send character to RS232 port.....delay to give time for PC at other end to get data */

void putcx(char c)
{
  putc(c);
  suspend(5000);
}

/* Convert 1-nibble binary to ASCII hex string & display on RS232 */

void nibbleascii(unsigned int i)
{
  i &= 0x0f;
  if (i <= 9)
    putcx(i + '0');
  else
    putcx(i + 'A' - 10);
}

/* Convert 8-bit binary number to 2 character ASCII & send to RS232 */

void byteascii(unsigned int i)
{
  nibbleascii(i >> 4);
  nibbleascii(i);
}

/* Display 16-bit binary number on RS232 as 4-character hex string */

void wordascii(unsigned int i)
{
  byteascii(i >> 8);
  byteascii(i);
}

/* Display 32-bit binary number on RS232 as 8-character hex string */

void longascii(unsigned long x)
{
  wordascii(x >> 16);
  wordascii(x);
}

/* Set/Clear LEDs */

void clearled(int i)
{
  ctrlreg_image = ctrlreg_image | i;
  la_ctrlstat = ctrlreg_image;
}

void setled(int i)
{
  ctrlreg_image = ctrlreg_image & ~i;
  la_ctrlstat = ctrlreg_image;
}

/* Set/reset hardware control register bit */

void setctrl(int i)
{
  ctrlreg_image = ctrlreg_image | i;
  la_ctrlstat = ctrlreg_image;
}

void clearctrl(int i)
{
  ctrlreg_image = ctrlreg_image & ~i;
  la_ctrlstat = ctrlreg_image;
}

/* Send string to RS232 port */

void putstring(const char *s)
{
  int i = 0;

  while (s[i] != 0)
  {
    putcx(s[i++]);
  }
}

/* Write string w/carriage return and linefeed to RS232 */

void writeln(char *s)
{
  putstring(getstr(s));
  putstring(getstr("\n\r"));
}

/* Write string to RS232 - no CR/LF */

void write(char *s)
{
  putstring(getstr(s));
}

/* Force character to upper case */

char upcase(char c)
{
  if ((c >= 'a') && (c <= 'z'))
    return(c - 'a' + 'A');
  else
    return(c);
}

/* Convert ASCII hexadecimal character to binary equivalent */

int asciibin(char c)
{
  if ((c >= '0') && (c <= '9'))
    return(c - '0');
  else
    return(c - 'A' + 10);
}

/* Get binary equivalent of hex number received over RS232 interface */

int getbin()
{
  char c;
  int i,pos;

  i = 0;
  pos = 0;
  do
  {
    c = upcase(getc());
    if ((((c >= 'A') && (c <= 'F')) || ((c >= '0') && (c <= '9')))
        && (pos <= 3))
    { i = ((i << 4) & 0xFFF0) | asciibin(c);
      putcx(c);
      pos += 1;
    } else
    if ((c == backspace) && (pos != 0))
    {
      putcx(backspace);
      putcx(' ');
      putcx(backspace);
      i = (i >> 4) & 0x0FFF;
      pos -= 1;
    }
  } while (c != carret);
  global_pos = pos;
  return(i);
}

/* Get 32-bit binary equivalent of hex number received over RS232 interface */

unsigned long getlongbin()
{
  char c;
  unsigned long i;
  int pos;

  i = 0;
  pos = 0;
  do
  {
    c = upcase(getc());
    if ((((c >= 'A') && (c <= 'F')) || ((c >= '0') && (c <= '9')))
        && (pos <= 7))
    { i = ((i << 4) & 0xFFFFFFF0) | asciibin(c);
      putcx(c);
      pos += 1;
    } else
    if ((c == backspace) && (pos != 0))
    {
      putcx(backspace);
      putcx(' ');
      putcx(backspace);
      i = (i >> 4) & 0x0FFFFFFF;
      pos -= 1;
    }
  } while (c != carret);
  return(i);
}

/* Get next character & echo back to screen */

char getcecho()
{
  char c;
  do
  {
    c = getc();
    putc(c);
    if (c == 0x0D) writeln("");
  } while (c <= ' ');
  return(c);
}

/* Convert Ascii Hex character string to binary equivalent */

int cvtbin(char c)
{
  if ((c >= 'A') && (c <= 'F'))
    return(c - 'A' + 10);
  else
    return(c - '0');
}

/* Get next 2-digit Hex from RS232 ..... Return as integer type */

int getbyte()
{
  int a;
  a = cvtbin(getcecho());
  a = (a << 4) | cvtbin(getcecho());
  chksum += a;
  return(a);
}

int getword()
{
  int a;
  a = getbyte();
  return((a << 8) | getbyte());
}

/* Read data locations */

void readdata()
{
  volatile int *loc;

  write("Enter address in data space to read: ");
  loc = (volatile int *)getbin();
  write(" --> ");
  wordascii(*loc);
  writeln("");
}

/* Write data locations */

void writedata()
{
  volatile int *loc;
  int i;

  write("Enter address in data space to write: ");
  loc = (int *)getbin();
  write(" --> ");
  i = getbin();
  *loc = i;
  writeln("");
}

/* Set up DMA controller for read transfer and start DMA transfer */

void dmaread(int srcadr,
             int destadr,
             int cnt)

{
  int i,j;

  la_xfercntr = cnt;                   /* Load transfer counter */
  la_addrhi = destadr >> 8;
  la_addrlo = destadr;                 /* Load addr cntr hi/lo  */
  j = 0;
  i = (la_link & 0x3);                 /* Get currently selected link # */
  if (i == 1)
    {
      j = *(volatile unsigned int *)(la_11c01_addr + srcadr);
    }                                  /* Set up source address */
    else
  if (i == 2)
    {
      j = *(volatile unsigned int *)(la_21lv03_addr + srcadr);
    }
    else
  if (i == 3)
    {
      j = *(volatile unsigned int *)(la_14c01_addr + srcadr);
    };
  la_dmago = 0;                        /* Start DMA transfer    */
}

/* Set up DMA controller for write transfer and start DMA transfer */

void dmawrite(int srcadr,
              int destadr,
              int cnt)

{
  int i,j;

  la_xfercntr = cnt;                   /* Load transfer counter */
  la_addrhi = srcadr >> 8;
  la_addrlo = srcadr;                  /* Load addr cntr hi/lo  */
  j = 0;
  i = (la_link & 0x3);                 /* Get currently selected link # */
  if (i == 1)
    {
      *(volatile unsigned int *)(la_11c01_addr + destadr) = j;
    }                                  /* Set up source address */
    else
  if (i == 2)
    {
      *(volatile unsigned int *)(la_21lv03_addr + destadr) = j;
    }
    else
  if (i == 3)
    {
      *(volatile unsigned int *)(la_14c01_addr + destadr) = j;
    };
  la_dmago = 0;                        /* Start DMA transfer    */
}

/* Return specified 32-bit register from 12C01 chip */

unsigned long getlink(int adr)
{
  dmaread(adr,dmabuff_addr,1);              /* Perform DMA transfer */
  return(*(unsigned long *)dmabuff_addr);
}

/* Write 32-bit quantity to specified 12C01 chip register */

void putlink(int adr, unsigned long dat)
{
  *(unsigned long *)dmabuff_addr = dat;
  dmawrite(dmabuff_addr,adr,1);
}

/* Read and display specified link controller register */

void readlink()
{
  int adr;
  unsigned long x;

  write("Enter 12C01 register address: ");
  adr = getbin();
  x = getlink(adr);
  write(" --> ");
  longascii(x);
  writeln("");
}

/* Write specified link controller register */

void writelink()
{
  int adr;
  unsigned long x;

  write("Enter 12C01 register address: ");
  adr = getbin();
  write(" --> ");
  x = getlongbin();
  putlink(adr,x);
  writeln("");
}

/* Dump a specified 12C01, prefixed with a string, to RS232 */

void dumplinkreg(const char *s,int i)
{
  write((char *)s);
  longascii(getlink(i));
  if ((oddeven & 1) != 0) writeln("");
    else write("      ");
  oddeven += 1;
}

/* Dump 12C01 link registers to RS232 */

void dumplink()
{
  oddeven = 0;
  dumplinkreg("Version/Revision....",link_version_adr);
  dumplinkreg("Node Address........",link_nodeadr_adr);
  dumplinkreg("Control Register....",link_control_adr);
  dumplinkreg("Interrupt Pending...",link_intr_adr);
  dumplinkreg("Interrupt Mask......",link_imask_adr);
  dumplinkreg("Cycle Timer.........",link_cyctime_adr);
  dumplinkreg("Isoch Port Number...",link_isoport_adr);
  dumplinkreg("Diagnostics.........",link_diags_adr);
  dumplinkreg("Phy Chip Access.....",link_phychip_adr);
  dumplinkreg("ATF Status..........",link_atfstatus_adr);
  dumplinkreg("ITF Status..........",link_itfstatus_adr);
  dumplinkreg("GRF Status..........",link_grfstatus_adr);
  writeln("");
}

/* Display string followed by specified phy register contents */

void dumpphyreg(const char *s,int i)
{
  write((char *)s);
  byteascii(getphy(i));
  if ((oddeven & 1) != 0) writeln("");
    else write("      ");
  oddeven += 1;
}

/* Dump PHY registers to RS232 */

void dumpphy()
{
  oddeven = 0;
  dumpphyreg("Physical ID/R/CPS...",phy_id_adr);
  dumpphyreg("RHB/IBR/Gap Count...",phy_gc_adr);
  dumpphyreg("Speed/No of Ports...",phy_np_adr);
  dumpphyreg("Astat/Bstat/Ch/Con..",phy_linestat_adr);
  writeln("");
}

/* Dump a line to RS232, using oddeven to determine new line */

void writemenu(const char *s)
{
  write((char *)s);
  if ((oddeven & 1) != 0) writeln("");
    else write("      ");
  oddeven += 1;
}

/* Get specified PHY register */

int getphy(int adr)
{
  link_phychip_union.link_phychip_reg = 0;
  link_phychip_union.link_phychip.phyrgad = adr;
  link_phychip_union.link_phychip.rdphy = 1;
  putlink(link_phychip_adr,link_phychip_union.link_phychip_reg);
  do
   link_phychip_union.link_phychip_reg = getlink(link_phychip_adr);
  while (link_phychip_union.link_phychip.rdphy != 0);
   link_phychip_union.link_phychip_reg = getlink(link_phychip_adr);
  return(link_phychip_union.link_phychip.phyrxdata);
}

/* Write specified PHY register with specified data */

putphy(int adr, int i)
{
  link_phychip_union.link_phychip_reg = 0;
  link_phychip_union.link_phychip.phyrgad = adr;
  link_phychip_union.link_phychip.phyrgdata = i;
  link_phychip_union.link_phychip.wrphy = 1;
  putlink(link_phychip_adr,link_phychip_union.link_phychip_reg);
  do
   link_phychip_union.link_phychip_reg = getlink(link_phychip_adr);
  while (link_phychip_union.link_phychip.wrphy != 0);
}

/* Read selected PHY register */

void readphy()
{
  int adr;

  write("Enter PHY register address: ");
  adr = getbin();
  write(" --> ");
  wordascii(getphy(adr));
  writeln("");
}

/* Write selected PHY register */

void writephy()
{
  int adr,i;

  write("Enter PHY register address: ");
  adr = getbin();
  write(" --> ");
  i = getbin();
  putphy(adr,i);
  writeln("");
}

/* Read 12C01 pending interrupts & display a message for each cleared */

void clearintr()
{

  link_intr_union.link_intr_reg = getlink(link_intr_adr);

  if (link_intr_union.link_intr.phint != 0) writeln("Phy chip Interrupt (PhInt)");
  if (link_intr_union.link_intr.phrrx != 0) writeln("Phy chip received data (PhRRx)");
  if (link_intr_union.link_intr.phrst != 0) writeln("Phy chip was reset (PhRst)");
  if (link_intr_union.link_intr.txrdy != 0) writeln("Transmitter ready (TxRdy)");
  if (link_intr_union.link_intr.rxdta != 0) writeln("Receiver has Data (RxDta)");
  if (link_intr_union.link_intr.cmdrst != 0) writeln("Command reset received (CmdRst");
  if (link_intr_union.link_intr.itstk != 0) writeln("IT transmitter is stuck (ITStk)");
  if (link_intr_union.link_intr.atstk != 0) writeln("AT transmitter is stuck (ATStk)");
  if (link_intr_union.link_intr.sntrj != 0) writeln("Busy Acknowledge sent by receiver (SntRj)");
  if (link_intr_union.link_intr.hdrer != 0) writeln("Header Error (HdrEr)");
  if (link_intr_union.link_intr.tcerr != 0) writeln("Transaction Code Error (TCErr)");
  if (link_intr_union.link_intr.cysec != 0) writeln("Cycle Second Incremented (CySec)");
  if (link_intr_union.link_intr.cyst != 0)  writeln("Cycle Started (CySt)");
  if (link_intr_union.link_intr.cydne != 0) writeln("Cycle Done (CyDne)");
  if (link_intr_union.link_intr.cypnd != 0) writeln("Cycle pending (CyPnd)");
  if (link_intr_union.link_intr.cylst != 0) writeln("Cycle Lost (CyLst)");
  if (link_intr_union.link_intr.carbfl != 0) writeln("Cycle Arbitration Failed (CArbFl)");
  if (link_intr_union.link_intr.iarbfl != 0) writeln("Isochronous Arbitration Failed (IArbFl)");
  writeln("");

  putlink(link_intr_adr,link_intr_union.link_intr_reg);   /* Clear pending interrupts */
}

/* Dump contents of 12C01 General Receive FIFO (GRF) until no more data in FIFO */

void dumpgrf()
{
  union {
          link_grfstatus_bits bits;
          unsigned long reg;
        } u;

  writeln("GRF contents: ");
  u.reg = getlink(link_grfstatus_adr);
  while (u.bits.empty == 0)
  {
    write("  ");
    longascii(getlink(link_grfdata_adr));
    writeln("");
    u.reg = getlink(link_grfstatus_adr);
  };
  writeln("");
}

/* Initialize 12C01 with appropriate registers to enable transmit/receive */

void initlink()
{
  volatile unsigned int i;
  unsigned long ir;
  unsigned long timout;
  union {
          link_grfstatus_bits bits;
          unsigned long reg;
        } un;

  writeln("Initializing LLC register");
  link_control_union.link_control_reg = 0;              /* Clear Control reg       */
  link_control_union.link_control.idval = 1;            /* Indicate ID valid       */
  link_control_union.link_control.rxsld = 1;            /* Enable receive self-id  */
  link_control_union.link_control.txen = 1;             /* Enable Xmits            */
  link_control_union.link_control.rxen = 1;             /* Enable Receive          */
  link_control_union.link_control.cyten = 1;            /* Enable cycle timer      */
  link_control_union.link_control.rsttx = 1;            /* Reset transmitter       */
  link_control_union.link_control.rstrx = 1;            /* Reset receiver          */
  link_control_union.link_control.irp1en = 1;           /* Enable ISO receipt      */
  link_control_union.link_control.irp2en = 1;
  link_control_union.link_control.revaen = 1;           /* Enable REV A features   */
  putlink(link_control_adr,link_control_union.link_control_reg); /* Write control reg  */

  putlink(link_grfstatus_adr,link_grf_size);            /* Init GRF size */
  putlink(link_atfstatus_adr,link_atf_size);            /* Init ATF size */
  putlink(link_itfstatus_adr,link_itf_size);            /* Init ITF size */

  ir = la_geogrd & 0x1F;                                /* Get phys id from backplane */
  link_isoport_union.link_isoport_reg =
    ((ir << 24) | ((ir + 1) << 16));                    /* Recognize IR ports */
  putlink(link_isoport_adr,link_isoport_union.link_isoport_reg);

  putlink(link_intr_adr,0xffffffff);                    /* Clear any pending interrupts */

  writeln("Initiating bus reset....result of Self ID follows");
  putphy(phy_gc_adr,0x7f);                              /* Bus reset, no holdoff */
  un.reg = getlink(link_grfstatus_adr);
  timout = 10000;
  while ((un.bits.empty != 0) && (timout != 0))          /* Wait for self-id */
  {
    un.reg = getlink(link_grfstatus_adr);                /* Dump self-id     */
    timout--;
  };
  if (timout != 0) dumpgrf();
    else
  writeln("Error: Timeout waiting for Self-ID response from GRF");

  writeln("Retrieving PHY address to put in LLC control reg");
  link_nodeadr_union.link_nodeadr_reg = 0;
  link_nodeadr_union.link_nodeadr.nodenumber
    = (getphy(phy_id_adr) >> 2);                        /* Get PHY link address */
  link_nodeadr_union.link_nodeadr.busnumber = 0x3FF;    /* Set bus number to 1's */
  putlink(link_nodeadr_adr,link_nodeadr_union.link_nodeadr_reg);

  for (i = 0; i < 10000; i++)
  {
    i = i;
  };
  i = getphy(phy_id_adr);                               /* Get root bit */
  if ((i & 0x02) != 0)
  {
    link_control_union.link_control.cymas = 1;          /* Enable cycle master if root */
    putlink(link_control_adr,link_control_union.link_control_reg); /* Write control reg  */
  };

  writeln("Link initialization complete");

}

/* Send quadlet to Async Transfer FIFO */

void putquad()
{
  int i;

  for (i = 0; i < 4; i++)
  { if (i == 0)
      putlink(link_atffirst_adr,async_quadxmt_union.regs[i]);   /* Send first quadlet */
    else if (i == 3)
      putlink(link_atfcontup_adr,async_quadxmt_union.regs[i]);  /* Send last quadlet */
    else
      putlink(link_atfcont_adr,async_quadxmt_union.regs[i]);    /* Send middle quadlets */
  };
}

/* Transmit ASYNC quadlet using info entered by operator */

void xmitquad()
{
  int i;
  unsigned int quadcnt;
  unsigned int datadr;

  for (i = 0; i < 4; i++)
    async_quadxmt_union.regs[i] = 0;

  async_quadxmt_union.bits.tcode = tcode_wrblock;
  write("Enter destid (will force bus# to 3FF): ");
  async_quadxmt_union.bits.busnumber = 0x3ff;
  async_quadxmt_union.bits.nodenumber = getbin();
  writeln("");
  write("Enter number of quadlets to transfer: ");
  quadcnt = getbin();
  writeln("");
  write("Enter DSP Data Space address of data to transmit: ");
  datadr = getbin();
  writeln("");
  async_quadxmt_union.bits.data_high = (quadcnt * 4);
  putlink(link_atffirst_adr,async_quadxmt_union.regs[0]);   /* Send first quadlet */
  for (i = 1; i < 4; i++)
  {
    putlink(link_atfcont_adr,async_quadxmt_union.regs[i]);   /* Send subsequent quadlets */
  };
  for (i = 0; i < quadcnt; i++)
  {
    if (i == (quadcnt - 1))
    {
      putlink(link_atfcontup_adr,*(volatile unsigned long *)(datadr+(i*2)));  /* Send last quadlet */
    }
    else
    {
      putlink(link_atfcont_adr,*(volatile unsigned long *)(datadr+(i*2)));  /* Send last quadlet */
    };
  };
}

/* Given received quadlet, return total number of quadlets in this packet */

int quadnumb()
{
  int j;

  if (async_quadrcv_union.bits.tcode == tcode_selfid) j = 4;
  else if (async_quadrcv_union.bits.tcode == tcode_rdquad) j = 4;
  else if (async_quadrcv_union.bits.tcode == tcode_wrblock) j = 4;
  else j = 5;

  return(j);
}

/* Receive ASYNC packet from GRF */

void getquad()
{
  int i,j,k;

  async_quadrcv_union.regs[0] = getlink(link_grfdata_adr);    /* Get first quadlet */

  j = quadnumb();

  for (i = 1; i < j; i++)
    async_quadrcv_union.regs[i] = getlink(link_grfdata_adr);  /* Get the GRF data */
}

/* Transmit ISO packet with N quadlets */

void xmitiso()
{
  int i,j,k;

  write("Enter destination ISO channel number: ");
  iso_header_union.reg = 0;
  iso_header_union.bits.channum = getbin();
  writeln("");
  write("Enter data length in quadlets: ");
  iso_header_union.bits.datalength = ((i = getbin()) << 2);
  writeln("");

  putlink(link_itffirst_adr,iso_header_union.reg);
  for (j = 0; j < (i - 1); j++)
    putlink(link_itfcont_adr,j);
  putlink(link_itfcontup_adr,j);

  writeln("ISO packet transmitted");

}

/* Display received quadlet data */

void displayquad()
{
  int i,j;

  j = quadnumb();

  if (debug)
  {
    for (i = 0; i < j; i++)
    {
      write("Received Async Packet(");
      byteascii(i);
      write("): ");
      longascii(async_quadrcv_union.regs[i]);
      writeln("");
    };
    writeln("");
  };
}

/* Toggle debug mode */

void toggledebug()
{
  if (debug)
  {
    debug = FALSE;
    writeln("Debug mode now OFF");
  }
  else
  {
    debug = TRUE;
    writeln("Debug mode now ON");
  };
}

/* Toggle expert mode */

void toggleexpert()
{
  if (expertmode)
  {
    expertmode = FALSE;
    writeln("Expert mode now OFF");
  }
  else
  {
    expertmode = TRUE;
    writeln("Expert mode now ON");
  };
}

/* Suspend execution for LED blink delay */

void delayled()
{
  int i;

  for (i = 0; i < 1000; i++) suspend(1000);
}

/* Perform LED test */

void ledtest()
{
  int i;

  for (i = 0; i < 10; i++)
  {
    setled(ctrl_led0);
    delayled();
    clearled(ctrl_led0);
    delayled();
    setled(ctrl_led1);
    delayled();
    clearled(ctrl_led1);
    delayled();
    setled(ctrl_led2);
    delayled();
    clearled(ctrl_led2);
    delayled();
  }
}

/* Reset all 12C01 ports */

void resetports()
{
  clearctrl(ctrl_ra);
  clearctrl(ctrl_rb);
  clearctrl(ctrl_rc);
  suspend(5000);
  setctrl(ctrl_ra);
  setctrl(ctrl_rb);
  setctrl(ctrl_rc);
  writeln("IEEE 1394 ports have been reset");
  ctrl14c01_union.ctrl14c01_image = ctrl14c01_init;    /* Init 14c01 control register */
  la_14c01ctrl = ctrl14c01_union.ctrl14c01_image;
}

/* Read and display FB+ backplane geographical address */

void displaygeog()
{
  int i;

  i = la_geogrd & 0x1F;
  writeln("");
  write("Futurebus+ Backplane Geographical address: ");
  byteascii(i);
  writeln("");
  writeln("");
}

/* Support routine for modify14c01 - prompt user for value */

int getmod(int curval)
{
  int i;

  write("Current value: ");
  byteascii(curval);
  write(" Enter new value: ");
  i = getbin();
  writeln("");
  if (global_pos == 0) i = curval;
  return(i);
}

/* Modify 14c01 control register inputs */

void modify14c01()
{
  writeln("Modify 14C01 Control inputs (Hit return to keep current value)");
  writeln("");
  write("  ENA_PRI  (14C01 pin 4)   ");
  ctrl14c01_union.ctrl14c01.enapri
    = getmod(ctrl14c01_union.ctrl14c01.enapri);
  write("  OSC_SEL  (14C01 pin 50)  ");
  ctrl14c01_union.ctrl14c01.oscsel
    = getmod(ctrl14c01_union.ctrl14c01.oscsel);
  write("  EN_EXID  (14C01 pin 18)  ");
  ctrl14c01_union.ctrl14c01.enexid
    = getmod(ctrl14c01_union.ctrl14c01.enexid);
  write("  EN_EXPRI (14C01 pin 19)  ");
  ctrl14c01_union.ctrl14c01.enexpri
    = getmod(ctrl14c01_union.ctrl14c01.enexpri);
  write("  ID[5..0] (pins 20 to 25) ");
  ctrl14c01_union.ctrl14c01.id
    = getmod(ctrl14c01_union.ctrl14c01.id);
  write("  PRI[3..0] (pins 27-30)   ");
  ctrl14c01_union.ctrl14c01.pri
    = getmod(ctrl14c01_union.ctrl14c01.pri);
  write("  PWRDN    (14C01 pin 58)  ");
  ctrl14c01_union.ctrl14c01.pwrdn
    = getmod(ctrl14c01_union.ctrl14c01.pwrdn);
  write("  PTEST    (14C01 pin 35)  ");
  ctrl14c01_union.ctrl14c01.ptest
    = getmod(ctrl14c01_union.ctrl14c01.ptest);
  writeln("");
  la_14c01ctrl = ctrl14c01_union.ctrl14c01_image;  /* Write hardware with new values */
}

/* Fill data memory locations */

void filldata()
{
  unsigned int adr;
  unsigned int i;
  unsigned int cntr;
  unsigned int j;
  char c;

  write("Enter starting address for fill: ");
  adr = getbin();
  writeln("");
  write("Enter initial 16-bit data pattern: ");
  i = getbin();
  writeln("");
  write("Enter number of words to fill: ");
  cntr = getbin();
  writeln("");
  write("Fill pattern? (I=increment, D=decrement, P=pseudorandom, C=constant): ");
  c = upcase(getc());
  putc(c);
  writeln("");
  if (c == 'I')
  {
    for (j = 0; j < cntr; j++) *(volatile int *)(adr+j) = i++;
  }
  else if (c == 'D')
  {
    for (j = 0; j < cntr; j++) *(volatile int *)(adr+j) = i--;
  }
  else if (c == 'P')
  {
    for (j = 0; j < cntr; j++)
    {
      *(volatile int *)(adr+j) = i;
      if ((i & 0x8000) == 0)
      {
        i = i << 1;
      }
      else
      {
        i = ((i << 1) ^ 0x0429);
      };
    };
  }
  else if (c == 'C')
  {
    for (j = 0; j < cntr; j++) *(volatile int *)(adr+j) = i;
  };
}

/* Pause for key hit from operator */

void pause()
{
  writeln("");
  write("Press any key to continue");
  while (!getcpdg()) suspend(1);
  writeln("");
  suspend(20000);
  writeln("");
}

/* Dump data memory locations */

void dumpdata()
{
  unsigned int i,j;
  volatile int *adr;

  write("Enter starting address of dump: ");
  *adr = getbin();
  writeln("");
  writeln("");
  for (i = 0; i < 16; i++)
  {
    wordascii((unsigned int)adr);
    write(": ");
    for (j = 0; j < 8; j++)
    {
      write(" ");
      wordascii(*adr);
      adr++;
    }
    writeln("");
  }
  pause();
}

/* Execute SRAM test */

void testsram()
{
  unsigned int i,j;
  volatile unsigned int *adr;

  writeln("Testing SRAM....");
  writeln("  Performing Address test");
  for (adr = (unsigned int *)0x800; (unsigned int)adr < 0xf000; adr++) *adr = (unsigned int)adr;
  for (adr = (unsigned int *)0x800; (unsigned int)adr < 0xf000; adr++)
    if (*adr != (unsigned int)adr)
    {
      write("Address test error at ");
      wordascii((unsigned int)adr);
      write(".  Expecting ");
      wordascii((unsigned int)adr);
      write(" but found ");
      wordascii(*adr);
      writeln("");
    };
  writeln("  Performing 0000 pattern test");
  for (adr = (unsigned int *)0x800; (unsigned int)adr < 0xf000; adr++) *adr = 0;
  for (adr = (unsigned int *)0x800; (unsigned int)adr < 0xf000; adr++)
    if (*adr != 0)
    {
      write("Data test error at ");
      wordascii((unsigned int)adr);
      write(".  Expecting 0000");
      write(" but found ");
      wordascii(*adr);
      writeln("");
    };
  writeln("  Performing FFFF pattern test");
  for (adr = (unsigned int *)0x800; (unsigned int)adr < 0xf000; adr++) *adr = 0xffff;
  for (adr = (unsigned int *)0x800; (unsigned int)adr < 0xf000; adr++)
    if (*adr != 0xffff)
    {
      write("Data test error at ");
      wordascii((unsigned int)adr);
      write(".  Expecting FFFF");
      write(" but found ");
      wordascii(*adr);
      writeln("");
    };
  writeln("  Performing 5555 pattern test");
  for (adr = (unsigned int *)0x800; (unsigned int)adr < 0xf000; adr++) *adr = 0x5555;
  for (adr = (unsigned int *)0x800; (unsigned int)adr < 0xf000; adr++)
    if (*adr != 0x5555)
    {
      write("Data test error at ");
      wordascii((unsigned int)adr);
      write(".  Expecting 5555");
      write(" but found ");
      wordascii(*adr);
      writeln("");
    };
  writeln("  Performing AAAA pattern test");
  for (adr = (unsigned int *)0x800; (unsigned int)adr < 0xf000; adr++) *adr = 0xAAAA;
  for (adr = (unsigned int *)0x800; (unsigned int)adr < 0xf000; adr++)
    if (*adr != 0xAAAA)
    {
      write("Data test error at ");
      wordascii((unsigned int)adr);
      write(".  Expecting AAAA");
      write(" but found ");
      wordascii(*adr);
      writeln("");
    };
  writeln("SRAM test completed");
}

/* Display which link is currently selected */

void display_selected_link()
{
  int i;

  i = (la_link & 0x3);
  if (i == 0) writeln("Link not selected");
    else
  if (i == 1) writeln("11C01 port");
    else
  if (i == 2) writeln("21LV03 port");
    else
  if (i == 3) writeln("14C01 port");
}

/* Test link control register - ie, reg which selects which 12C01 to talk to */

void linktest()
{
  int i;

  writeln("Testing link address control register");
  for (i = 0; i < 16; i++)
  {
    la_link = i;
    if ((la_link & 0x3) != (i & 0x3))
    {
      write("  Error - mismatch.  Value is ");
      byteascii(la_link & 0x3);
      write(" should be ");
      byteascii(i & 0x3);
      writeln("");
    };
  };
  writeln("Link address control register test completed");
}

/* Test dma control register */

void dmactrltest()
{
  int i;

  writeln("Testing DMA control register");
  for (i = 0; i < 16; i++)
  {
    la_dmactrl = i;
    if ((la_dmactrl & 0x3) != (i & 0x3))
    {
      write("  Error - mismatch.  Value is ");
      byteascii(la_dmactrl & 0x3);
      write(" should be ");
      byteascii(i & 0x3);
      writeln("");
    };
  };
  writeln("DMA control register test completed");
}

/* Toggle selected link */

void togglelink()
{
  int i;

  i = la_link;
  i = (i + 1) & 0x3;
  la_link = i;
}

/* Test address register for 11C01 port input */

void testreg()
{
  int i,j,k;

  while (TRUE)
    for (i=0; i<100; i++)
      j = *(volatile unsigned int *)(la_11c01_addr + i);
}

/* Test DMA transfer counter by reading/writing it */

void testxfercntr()
{
  int i;

  writeln("Testing DMA transfer counter");
  for (i = 0; i < 500; i++)
  {
    la_xfercntr = i;
    if ((la_xfercntr & 0xFF) != (i & 0xFF))
    {
      write("  Error - mismatch.  Value is ");
      byteascii(la_xfercntr & 0xFF);
      write(" should be ");
      byteascii(i & 0xFF);
      writeln("");
    };
  };
  writeln("DMA transfer counter test completed");
}

/* Test DMA transfers with selected 12C01 by reading IR Port ID with various
   transfer counts and addresses */

void testdma()
{
  unsigned int i,j,k;
  unsigned int cntr;
  unsigned long adr;

  writeln("Performing DMA transfers with currently selected 12C01 chip");
  for (j = 0x1000; j < 0xefff; j += 2) *(unsigned long *)j = 0xDEADBEEF;
  for (adr = 0x8002; adr != 0x8000; adr = (((adr + adr) & 0x7fff) | 0x8000))
  {
    putlink(link_isoport_adr,adr << 16);
    for (cntr = 1; cntr < 0x100; cntr++)
    {
      dmaread(link_isoport_adr,(int)adr,cntr);
      for (j = adr; j < (adr + (cntr*2)); j += 2)
      {
        if ((*(unsigned long *)j) != (adr << 16))
        {
          write("Data miscompare at address ");
          wordascii((unsigned int)adr);
          write(".  Expected: ");
          longascii(adr << 16);
          write(" Found: ");
          longascii(*(unsigned long *)j);
          writeln("");
        };
        *(unsigned long *)j = 0xDEADBEEF;
      };
      if (*(unsigned long *)(adr - 2) != 0xDEADBEEF)
        {
          write("Boundary error at addr ");
          wordascii(adr-2);
          write(", SA = ");
          wordascii(adr);
          write(". TC = ");
          wordascii(cntr);
          writeln("");
        };
      if (*(unsigned long *)(adr + (cntr*2)) != 0xDEADBEEF)
        {
          write("Boundary error at addr ");
          wordascii(adr + (cntr*2));
          write(", SA = ");
          wordascii(adr);
          write(". TC = ");
          wordascii(cntr);
          writeln("");
        };
    };
  };
  writeln("DMA transfer test completed");
}

/* Save state of currently selected 12C01 port in FLASH memory */

void saveflash()
{
  int i,j;

  i = fsaveadr;
  writeln("Erasing FLASH sector");
  for (i = 0; i <= 0x400; i += 2)
  {
    *(unsigned long *)(i+fsaveadr) = getconst32(i+fsaveadr);
  };
  eraseblk(fsaveadr);
  j = la_link & 0x3;
  j = j << 7;
  for (i = 0; i <= 0x40; i += 4)                /* Save LLC regs */
  {
    *(unsigned long *)(i+j+fsaveadr) = getlink(i);
  };
  for (i = 0; i < 4; i++)                       /* Save PHY regs */
  {
    *(unsigned long *)((i * 4)+j+fsaveadr+0x44) = getphy(i);
  };
  *(unsigned long *)(2+j+fsaveadr) = (volatile unsigned long)magicnumb;
  writeln("Saving registers of currently selected port");
  for (i = 0; i <= 0x400; i += 2)
  {
    burnloc32(i+fsaveadr,*(unsigned long *)(i+fsaveadr));
  };


}

/* Restore registers from FLASH memory */

void restorreg()
{
  int i,j,k;

  j = la_link & 0x3;
  j = j << 7;
  if (magicnumb != (getconst32(2+j+fsaveadr)))
  {
    writeln("Sorry, you must first save the registers before restoring");
  }
  else
  {
    writeln("Restoring registers of currently selected port");
    for (i = 0; i <= 0x40; i += 4)
    {
      putlink(i,getconst32(i+j+fsaveadr));
    };
    for (i = 0; i < 4; i++)
    {
      k = getconst32((i*4)+j+fsaveadr+0x44);
      putphy(i,k);
    };
  }
}

/* Continuous transmit */

void contxmit()
{
  char c;
  char ptype;
  unsigned int datadr,quadcnt,destid,quadcnt2;
  unsigned long packetcnt;
  unsigned int i,j;
  unsigned long k;
  unsigned long quadlet,quaddest,quadsrc,temp;
  unsigned long timecnt;
  int waitp;

  union {
          link_grfstatus_bits bits;
          unsigned long reg;
        } u;

  waitp = TRUE;
  writeln("Transmit packets continuously......");
  write("Enter DSP Data Space address of packet data: ");
  datadr = getbin();
  writeln("");
  write("Enter number of quadlets to transfer per packet: ");
  quadcnt = getbin();
  writeln("");
  write("Enter number of packets to transfer: ");
  packetcnt = getbin();
  writeln("");
  write("Enter ISO or ASYNC packet type (I/A): ");
  ptype = upcase(getc());
  if (ptype != 'I') ptype = 'A';
  putc(ptype);
  writeln("");
  if (ptype == 'I')
  {
    write("Enter target ISO channel number: ");
    destid = getbin();
    writeln("");
  }
  else
  {
    write("Enter ASYNC packet destination Node Number (Bus # is always 3FF): ");
    destid = getbin();
    writeln("");
  };
  write("Want to wait for echo packet? (Y/N): ");
  c = upcase(getc());
  putc(c);
  writeln("");
  if (c == 'Y')
  {
    waitp = TRUE;
  }
  else
  {
    waitp = FALSE;
  };
  for (k = 0; k < packetcnt; k++)
  {
    write(".");
    if (ptype == 'I')
    {                                           /* ISO transfer mode */
      quadlet = (((unsigned long)(quadcnt * 4)) << 16);
      quadlet = (quadlet & (0xffff000f)) | (destid << 8);
      putlink(link_itffirst_adr,quadlet);   /* Send first quadlet */
      for (i = 0; i < quadcnt; i++)
      {
        quadlet = *(volatile unsigned long *)(datadr+(i*2));
        if (i == (quadcnt - 1))
        {
          putlink(link_itfcontup_adr,quadlet);  /* Send last quadlet */
        }
        else
        {
          putlink(link_itfcont_adr,quadlet);  /* Send last quadlet */
        };
      };
    }
    else
    {                                           /* ASYNC transfer mode */
      for (i = 0; i < 4; i++)
        async_quadxmt_union.regs[i] = 0;
      async_quadxmt_union.bits.tcode = tcode_wrblock;
      async_quadxmt_union.bits.busnumber = 0x3ff;
      async_quadxmt_union.bits.nodenumber = destid;
      async_quadxmt_union.bits.data_high = (quadcnt * 4);
      for (i = 0; i < 4; i++)
      {
        if (i == 0)
        {
          if (debug)
          {
            longascii(async_quadxmt_union.regs[i]);
            write(" -> First");
            writeln("");
          };
          putlink(link_atffirst_adr,async_quadxmt_union.regs[i]);   /* Send first quadlet */
        }
        else
        {
          if (debug)
          {
            longascii(async_quadxmt_union.regs[i]);
            write(" -> cont");
            writeln("");
          };
          putlink(link_atfcont_adr,async_quadxmt_union.regs[i]);    /* Send middle quadlets */
        };
      };
      for (i = 0; i < quadcnt; i++)
      {
        if (i == (quadcnt - 1))
        {
          if (debug)
          {
            longascii(*(volatile unsigned long *)(datadr+(i*2)));
            write(" -> Contup");
            writeln("");
          };
          putlink(link_atfcontup_adr,*(volatile unsigned long *)(datadr+(i*2)));  /* Send last quadlet */
        }
        else
        {
          if (debug)
          {
            longascii(*(volatile unsigned long *)(datadr+(i*2)));
            write(" -> Cont");
            writeln("");
          };
          putlink(link_atfcont_adr,*(volatile unsigned long *)(datadr+(i*2)));  /* Send last quadlet */
        };
      };
    };
    if (waitp)
    {
      timecnt = 100000;
      do                                  /* Wait for response packet */
      {
        timecnt--;
        u.reg = getlink(link_grfstatus_adr);
      } while ((u.bits.empty != 0) && (timecnt != 0));
      if ((ptype == 'I') && (timecnt != 0))
      {                                     /* Handle ISO case */
        quadlet = getlink(link_grfdata_adr);
        quadcnt2 = quadlet >> 16;
        quadcnt2 = (quadcnt2 >> 2) & 0x0000ffff;
        for (i = 0; i < quadcnt2; i++)
        {
          quadlet = getlink(link_grfdata_adr);
          if (quadlet != *(volatile unsigned long *)(datadr+(i*2)))
          {
            write("ISO data miscompare.  Expected ");
            longascii(*(volatile unsigned long *)(datadr+(i*2)));
            write(" but received ");
            longascii(quadlet);
            writeln("");
          };
        };
        quadlet = getlink(link_grfdata_adr);    /* Dump last quadlet */
      }
      else
      if (timecnt != 0)
      {                                         /* Handle ASYNC response case */
        quaddest = getlink(link_grfdata_adr);
        quadsrc = getlink(link_grfdata_adr);
        quadlet = getlink(link_grfdata_adr);
        quadlet = getlink(link_grfdata_adr);
        quadcnt2 = quadlet >> 16;
        quadcnt2 = (quadcnt2 >> 2) & 0x0000ffff;
        for (i = 0; i < quadcnt2; i++)
        {
          quadlet = getlink(link_grfdata_adr);
          if (quadlet != *(volatile unsigned long *)(datadr+(i*2)))
          {
            write("ASYNC data miscompare.  Expected ");
            longascii(*(volatile unsigned long *)(datadr+(i*2)));
            write(" but received ");
            longascii(quadlet);
            writeln("");
          };
        };
        quadlet = getlink(link_grfdata_adr);    /* Dump last quadlet */
      };
      if (timecnt == 0)
      {
        writeln("Timeout waiting for response packet");
        k = packetcnt;
      };
    };
  };
}

/* Continuous receive */

void contrcvr()
{
  char ptype;
  unsigned int destid;
  unsigned long quadlet,quaddest,quadsrc,temp;
  unsigned int quadcnt;
  unsigned int i,j,k;

  union {
          link_grfstatus_bits bits;
          unsigned long reg;
        } u;


  writeln("This will echo received packets back to originator.");
  write("Enter ISO or ASYNC packet type (I/A): ");
  ptype = upcase(getc());
  if (ptype != 'I') ptype = 'A';
  putc(ptype);
  writeln("");
  if (ptype == 'I')
  {
    write("Enter target ISO channel number: ");
    destid = getbin();
    writeln("");
  }
  else
  {
    write("Enter ASYNC packet destination Node Number (Bus # is always 3FF): ");
    destid = getbin();
    writeln("");
  };
  writeln("Waiting for packets......Press any key to terminate");
  writeln("");
  while (!getcpdg())
  {
    u.reg = getlink(link_grfstatus_adr);
    if (u.bits.empty == 0)
    {
      write(".");
      if (ptype == 'I')
      {                                     /* Handle ISO case */
        quadlet = getlink(link_grfdata_adr);
        quadcnt = quadlet >> 16;
        quadcnt = (quadcnt >> 2) & 0x0000ffff;
        quadlet = (quadlet & (0xffff000f)) | (destid << 8);
        putlink(link_itffirst_adr,quadlet);   /* Send first quadlet */
        for (i = 0; i < quadcnt; i++)
        {
          quadlet = getlink(link_grfdata_adr);
          if (i == (quadcnt - 1))
          {
            putlink(link_itfcontup_adr,quadlet);  /* Send last quadlet */
          }
          else
          {
            putlink(link_itfcont_adr,quadlet);  /* Send last quadlet */
          };
        };
        quadlet = getlink(link_grfdata_adr);    /* Dump last quadlet */
      }
      else     /* Handle ASYNC case */
      {
        quaddest = getlink(link_grfdata_adr);
        quadsrc = getlink(link_grfdata_adr);
        quaddest = (quaddest & 0x0000ffff);
        quadsrc = (quadsrc & 0x0000ffff) | (0xffc00000 | (destid << 16));
        putlink(link_atffirst_adr,quaddest);    /* Speed, tlabel, etc */
        putlink(link_atfcont_adr,quadsrc);      /* Dest ID, Dest off hi */
        quadlet = getlink(link_grfdata_adr);
        putlink(link_atfcont_adr,quadlet);      /* Dest off low */
        quadlet = getlink(link_grfdata_adr);
        putlink(link_atfcont_adr,quadlet);      /* Data length */
        quadcnt = quadlet >> 16;
        quadcnt = (quadcnt >> 2) & 0x0000ffff;
        for (i = 0; i < quadcnt; i++)
        {
          quadlet = getlink(link_grfdata_adr);
          if (i == (quadcnt - 1))
          {
            putlink(link_atfcontup_adr,quadlet);  /* Send last quadlet */
          }
          else
          {
            putlink(link_atfcont_adr,quadlet);  /* Send last quadlet */
          };
        };
        quadlet = getlink(link_grfdata_adr);    /* Dump last quadlet */
      };
    };
  };
}

/* Upload flash image for new Monitor code */

void uploadmonitor()
{
  char c;
  int record_type;
  int record_length;
  int  record_addr;
  unsigned int i,j;
  char buffer[50];
  int badflg;

  for (i = 0; i < 0x0e000; i += 1)
  {
    *(unsigned int *)(i+foffset) = getconst16(i);
  };

  writeln("Send ASCII Now...");

  record_type = 0xff;

  badflg = FALSE;

  while (record_type != 0x01)
  {
    while (getcecho() != ':') writeln("Missing ':' Character");
    chksum = 0;
    record_length = getbyte();
    record_addr = getword();
    record_type = getbyte();
    for (i = 1; i <= record_length + 1; i++)
      buffer[i] = getbyte();
    if ((chksum & 0x00FF) != 0)
    {
      writeln("");
      write(" -- > Checksum Error");
      badflg = TRUE;
    }
    else
      if (record_type == 0)
        for (i = 1; i <= record_length; i += 2)
        {
          *(unsigned int *)(foffset+record_addr) = ((buffer[i] << 8) | buffer[i+1]);
          record_addr++;
        };
  };
  writeln("");
  if (badflg)
  {
    writeln("Sorry, can not burn FLASH memory because of checksum error");
  }
  else
  {
    writeln("Burning FLASH memory with new monitor....please wait");
    burnsram();
  };
}

/* Calculate and display primitive random number polynomials */

void calcpoly()
{
  unsigned int i,j;
  unsigned long k;

  for (i = 1; i <= 0x8000; i += 2)
  {
    j = 1;
    k = 0;
    do
    {
      if ((j & 0x8000) != 0x8000)
      {
        j = j << 1;
      }
      else
      {
        j = (j << 1) ^ i;
      };
      k++;
    }
    while ((j != 1) && (k != 0x10001));
    if ((k > 0xff00) & (k < 0x10000))
    {
      wordascii(i);
      write(",");
      wordascii(k);
      write(" ");
    };
  };
}

/* Top level menu for RS232 interface */

void menu()
{
  char c;

  if (!expertmode)
  {
    oddeven = 0;
    writeln("RS232 Backplane PHY test board Monitor - V1.2 - 2/19/96");
    writeln("");
    writemenu("  A. Toggle link selection         ");
    writemenu("  B. Reset all 12C01 ports         ");
    writemenu("  M. Modify 14C01 control inputs   ");
    writemenu("  L. Dump 12C01 Registers          ");
    writemenu("  P. Dump PHY chip Registers       ");
    writemenu("  K. Clear out Link Interrupts     ");
    writemenu("  R. Read Data Memory location     ");
    writemenu("  W. Write Data Memory location    ");
    writemenu("  E. Dump Data Memory              ");
    writemenu("  F. Fill Data Memory              ");
    writemenu("  U. Read PHY chip register        ");
    writemenu("  V. Write PHY chip register       ");
    writemenu("  X. Read 12C01 Register           ");
    writemenu("  Y. Write 12C01 Register          ");
    writemenu("  G. Dump 12C01 GRF FIFO data      ");
    writemenu("  H. Initialize 12C01 for xmt/rcv  ");
    writemenu("  T. Transmit ASYNC quadlet        ");
    writemenu("  Q. Transmit ISO w/N quadlets     ");
    writemenu("  C. Display FB+ geographical addr ");
    writemenu("  Z. Toggle 'expert' mode          ");
    writemenu("  D. Save regs to FLASH            ");
    writemenu("  I. Restore regs from FLASH       ");
    writemenu("  J. Continuous Transmit packets   ");
    writemenu("  N. Echo Received packets         ");
    writemenu("  0. Toggle DEBUG mode on/off      ");
    writemenu("  1. Test SRAM                     ");
    writemenu("  2. Link control register test    ");
    writemenu("  3. LED test                      ");
    writemenu("  4. Test DMA control register     ");
    writemenu("  5. Test DMA transfer counter     ");
    writemenu("  6. DMA transfer test             ");
    writemenu("  7. Upload/burn new MONITOR code  ");
    writeln("");
  };
  writeln("");
  write("Currently selected link interface: ");
  display_selected_link();
  writeln("");
  write("Enter selection: ");
  c = upcase(getc());
  putcx(c);
  writeln("");
  writeln("");
  if (c == 'A') togglelink();
    else
  if (c == 'D') saveflash();
    else
  if (c == 'I') restorreg();
    else
  if (c == 'J') contxmit();
    else
  if (c == 'N') contrcvr();
    else
  if (c == 'Z') toggleexpert();
    else
  if (c == 'B') resetports();
    else
  if (c == 'E') dumpdata();
    else
  if (c == 'F') filldata();
    else
  if (c == 'M') modify14c01();
    else
  if (c == 'C') displaygeog();
    else
  if (c == 'R') readdata();
    else
  if (c == 'W') writedata();
    else
  if (c == 'U') readphy();
    else
  if (c == 'V') writephy();
    else
  if (c == 'X') readlink();
    else
  if (c == 'Y') writelink();
    else
  if (c == 'L') dumplink();
    else
  if (c == 'K') clearintr();
    else
  if (c == 'P') dumpphy();
    else
  if (c == 'G') dumpgrf();
    else
  if (c == 'H') initlink();
    else
  if (c == 'T') xmitquad();
    else
  if (c == 'Q') xmitiso();
    else
  if (c == '0') toggledebug();
    else
  if (c == '1') testsram();
    else
  if (c == '2') linktest();
    else
  if (c == '3') ledtest();
    else
  if (c == '4') dmactrltest();
    else
  if (c == '5') testxfercntr();
    else
  if (c == '6') testdma();
    else
  if (c == '7') uploadmonitor();
}

/* Main program follows */

main ()
{
  int i;
  char c;

  loader();       /* Load RAM programs from ROM */

  bitlen = 866; /* Set up 38.4K baud constants for 33.3333 MHz DSP clk */
  bitlen2 = (bitlen >> 1);

  debug = FALSE;
  expertmode = FALSE;

  ctrlreg_image = 0;      /* Clear image of control register */
  la_ctrlstat = 0;        /* Clear hardware control register */

  writeln("Clearing LEDs");
  clearled(ctrl_led0);
  clearled(ctrl_led1);
  clearled(ctrl_led2);

  writeln("Reseting 12C01 ports");
  resetports();

  ctrl14c01_union.ctrl14c01_image = ctrl14c01_init;    /* Init 14c01 control register */
  la_14c01ctrl = ctrl14c01_union.ctrl14c01_image;

  writeln("Selecting default IEEE port");
  togglelink();

  writeln("Initializing 12C01 registers");
  initlink();

  writeln(".............12C01 Register Dump.............");
  dumplink();

  writeln(".............1394 PHY chip Dump..............");
  dumpphy();

  writeln("");
  write("Auto-baud constant: ");
  wordascii(bitlen);
  writeln("");

  for(;;) menu();

}