/*--------------------------------------------------------------------------*/
/*                                                                          */
/*                        TIGA DEMO PROGRAM                                 */
/*                                                                          */
/* Draws a series of frames to demonstrate various TIGA functions           */
/*                                                                          */
/* REQUIREMENTS:-                                                           */
/*    TIGA version 1.1 installed in directory \tiga                         */
/*    Microsoft C 5.0 or better with small model library                    */
/*                                                                          */
/* To rebuild this demo type:-                                              */
/*    make tigademo.mak                                                     */
/*                                                                          */
/*   10/10/88...Original version written..................Graham Short      */
/*    3/01/89...Addition of TILOGO and animation frames...Graham Short      */
/*    6/02/89...Addition of installed font................Graham Short      */
/*                                                                          */
/*--------------------------------------------------------------------------*/

/* Microsoft-c insert file */
#include <stdio.h>
#include <malloc.h>

/* TIGA insert files */
#include <typedefs.h>
#include <tiga.h>
#include <extend.h>

/* Demo insert file */
#include "tigademo.h"
#include "colors.h"

#define  ESC 0x1B

/* Declare demo global variables */
CONFIG      config;
FONTINFO    fontinfo;
PTR         gsp_patterns;  /* address patterns start at on GSP */
PTR         gsp_font;
PATTERN     patn =         /* global pattern structure */
{
    16,                         /* pattern width    */
    16,                         /* pattern height   */
    1,                          /* pattern depth    */
    0l                          /* pointer to pattern data in GSP memory */
};

void  (*funcs[NUM_FRAMES])() = { frame1, frame2, frame3, frame4, frame5,
                                 frame6, frame7, frame8 };
char  *(titles[NUM_FRAMES])  = { "",
                                 "",
                                 "Filled Rectangles",
                                 "Fast Lines",
                                 "Pattern Fills",
                                 "Fast Pixblts",
                                 "Polygon Fills",
                                 "Filled Ellipses" };

/*--------------------------------------------------------------------------*/
/* install_fontfile                                                         */
/* function to install the font of the filename passed to it into GSP       */
/* memory and return the id that it is installed with.                      */
/*--------------------------------------------------------------------------*/
int install_fontfile(filename)
char *filename;
{
   int   buffer;
   FILE  *stream;
   FONT  *host_font;
   int   return_value;

   /* attempt to open font file for reading */
   if ((stream = fopen(filename,"rb")) == NULL)
      return_value = FALSE;
   else
   {
      /* skip the first word of the file */
      if (!fread(&buffer, 2, 1, stream))
         return_value = FALSE;
      else
      {
         /* read the second word of the file */
         if (!fread(&buffer, 2, 1, stream))
            return_value = FALSE;
         else
         {
            host_font = (FONT *) malloc(buffer);
            if (!host_font)
               return_value = FALSE;
            else
            {
               fseek(stream, 0L, SEEK_SET);
               if (!fread(host_font, 1, buffer, stream))
                  return_value = FALSE;
               else
               {
                  gsp_font = (PTR) gsp_malloc(buffer);
                  if (!gsp_font)
                     return_value = FALSE;
                  else
                  {
                     host2gsp (host_font, gsp_font, buffer, 0);
                     return_value = install_font(gsp_font);
                  }
               }
               free(host_font);
            }
         }
      }
      fclose(stream);
   }
   return(return_value);
}

/*--------------------------------------------------------------------------*/
/* initialization                                                           */
/* This function:- initializes communication with TIGA                      */
/*                 installs extended primitives if they need installing     */
/*                 deletes all installed fonts                              */
/*                 installs and selects a font for the demo                 */
/*                 initializes the Demo global variables                    */
/*                 downloads fill patterns from host to gsp                 */
/*--------------------------------------------------------------------------*/
initialization()
{
    short   i, return_code, id;

    /*--------------------------------------------------------------*/
    /* Attempt to open the CD                                       */
    /*--------------------------------------------------------------*/
    if( tiga_set( CD_OPEN ) < 0 )
    {
        printf("Tiga CD is not installed\n");
        exit(1);
    }

    /* initialize TIGA interface */
    if(!set_videomode(TIGA, INIT_GLOBALS | CLR_SCREEN))
       error("FATAL ERROR - unable to initialize TIGA, aborting...\n");

    /* initialize demo globals */
    get_config(&config);

    /* Write initial message to screen. */
    init_palet();
    set_colors(0xFFFFFFFF, 0x00000000);
    clear_screen(0x00000000);
    text_out(0, 0, "Downloading TIGA Primitives...");

    /* if not loaded attempt to load them */
    if ((return_code = install_primitives()) < 0)
    {
       /* possibly cant load them due to full heap, reinitialize */
       if  (!set_videomode(TIGA, INIT))
          error("FATAL ERROR - unable to initialize TIGA, aborting...\n");
       else if ((return_code = install_primitives()) < 0)
          error("FATAL ERROR - unable to install TIGA primitives");
    }

    /* initialize colors */
    init_colors( COLOR_CLASS, PSIZE );

    /* delete all currently installed fonts */
    init_text();
    /* if install_fontfile fails it returns 0 which is the system font id */
    id = install_fontfile("austin25.fnt");
    select_font(id);
    get_fontinfo(id, &fontinfo);

    /* download patterns from host to gsp */
    if (gsp_patterns = gsp_malloc(PMAX * sizeof(PAT_ARY)))
       host2gsp (patterns, gsp_patterns, PMAX * sizeof(PAT_ARY), FALSE);
    else
       error("FATAL ERROR - not enough GSP heap to store fill patterns\n");
}


/*--------------------------------------------------------------------------*/
/* main program                                                             */
/*--------------------------------------------------------------------------*/
main()
{
    short i, ch;

    printf("TIGA Demonstration Program\n");
    printf("Press SPACE bar to pause demo after any frame (and any key to continue)\n");
    printf("Press ESCAPE key to stop demo after any frame\n");
    printf("Press a key to start the demo\n");
    getch();
    initialization();
    i = 0;
    for(EVER)
    {
        clear_frame_buffer(_BLACK);
        /* title non-empty ? */
        if (titles[i][0])
            title(titles[i]);
        /* execute the frame */
        funcs[i]();
        /* restore window to full screen */
        set_clip_rect(HRES,VRES,0,0);
        /* check for a key */
        if (kbhit())
        {
            ch = getch();
            /* pause ? */
            if (ch == ' ')  
                ch = getch();
            /* exit ? */
            if (ch == ESC)
            {
                /* switch back to emulation mode */
                gsp_free(gsp_patterns);
                gsp_free(gsp_font);
                set_videomode(PREVIOUS, NO_INIT);
                tiga_set( CD_CLOSE );
                exit(0);
            }
        }
        if (++i >= NUM_FRAMES)
             i = 0;
    }
}


error(s)
char *s;
{
    set_videomode(VGA, NO_INIT);
    tiga_set( CD_CLOSE );
    printf("%s\n",s);
    exit(0);
}

