/*----------------------------------------------------------------------
 *                                    TIGA
 *          Copyright (C) 1989-1990  Texas Instruments Incorporated.
 *                            All Rights Reserved
 *----------------------------------------------------------------------
 *
 *                    Define structured data types
 *
 *  The TIGA file <typedefs.h> must be included before this file.
 *----------------------------------------------------------------------
 *
 *  Summary:
 *     BIT          single bit in packed array
 *     POINT3D      xyz coordinates specifying point in 3-space
 *     VIEWPORT     parameters specifying 3D viewport
 *     CLIP3D       parameters used in clipping to 3D viewport
 *     ZCHECK       parameters for comparing object radius to distance
 *     UNITVEC      three orthogonal unit vectors, u, v and w
 *     VIEWPOINT    viewpoint orientation unit vectors and position
 *     VIEWINFO     sine and cosine of yaw, pitch and roll angles
 *     NODE         node in 3D database organized as binary tree
 *     DBASE3D      header information for 3D object database
 *----------------------------------------------------------------------
 */
typedef enum {      /* single bit in packed array */
    FIELDWIDTH = 1
} BIT;


typedef struct {    /* parameters specifying 3D viewport             */
    long x, y, z;    /* xyz coordinates in FIX8 format */
} POINT3D;


typedef struct {    /* parameters specifying 3D viewport             */
    short xleft, ytop;   /* top left corner of viewport on screen   */
    short sx, sy;        /* viewport half width and half height     */
    short cx, cy;        /* viewport center offsets from xleft,ytop */
    /* Set rdiag = sqrt(sx*sx+sy*sy) */
    short rdiag;         /* viewport half diagonal length           */
    short d;             /* viewer's assumed distance from screen   */
    short znear;         /* z value at "hither" clipping plane      */
} VIEWPORT;


typedef struct {    /* parameters used in clipping to 3D viewport    */
    VIEWPORT vp;        /* parameters describing 3D viewport         */
    short bufsize;      /* size of each buffer in bytes              */
    uchar *buf[2];      /* pointers to buffers for intermediate vals */
    /* Set kx = (d<<18)/sx and ky = (d<<18)/sy. */
    long kx, ky;        /* clipping volume normalization factors     */
} CLIP3D;


typedef struct {    /* params for comparing object radius to distance */
    /* If z/r > kvis, object is too distant to be visible. */
    short kvis;         /* d/Rvis-1, where Rvis = threshhold radius   */
    /* If z/r > kdot, object appears as a single dot on the screen. */
    short kdot;         /* d/Rdot-1, where Rvis = threshhold radius   */
    /* If z/r > knpc, object is too small to require preclipping. */
    short knpc;         /* d/Rnpc-1, where Rvis = threshhold radius   */
} ZCHECK;


typedef struct {    /* three orthogonal unit vectors, u, v and w */
    long ux, uy, uz;  /* view "right" vector     (fraction = 30 LSBs) */
    long vx, vy, vz;  /* view "down" vector      (fraction = 30 LSBs) */
    long wx, wy, wz;  /* viewplane normal vector (fraction = 30 LSBs) */
} UNITVEC;


typedef struct {    /* viewpoint orientation and position */
    UNITVEC uvw;    /* three orthogonal unit vectors, u, v and w    */
    POINT3D xyz;    /* view reference point    (fraction =  8 LSBs) */
} VIEWPOINT;


typedef struct {    /* sine and cosine of yaw, pitch and roll angles */
    long csyaw, snyaw;  /* cosine, sine of yaw angle   (30-bit fract) */
    long cspch, snpch;  /* cosine, sine of pitch angle (30-bit fract) */
    long csrol, snrol;  /* cosine, sine of roll angle  (30-bit fract) */
} VIEWINFO;


typedef struct {    /* node in 3D database organized as binary tree  */
    unsigned bsrad    :  8;  /* radius of node's bounding sphere    */
    unsigned log2f    :  5;  /* log2 of scaling factor for bsrad    */
    unsigned ntyp     :  3;  /* node type = 7 (leaf), 0 (nonleaf)   */
    unsigned dotcolor : 16;  /* color of distant object seen as dot */
} NODE;


typedef struct {    /* header information for 3D object database       */
    long dbsize;       /* length of database in bytes                  */
    long rootnode;     /* bit offset to root node of binary tree       */
    POINT3D start;     /* starting xyz coordinates in world space      */
    PALET palet[16];   /* palette of 16 colors (see typedefs.h)        */
    PALET bwpatn[16];  /* 16 4x8 patterns for monochrome displays      */
    long bcdata;       /* bit offset to object data for bouncing cube  */
    long bcbbox;       /* bit offset to bounding box for bouncing cube */
} DBASE3D;

