/*----------------------------------------------------------------------*/
/*                                 TIGA                                 */
/*       Copyright (c) 1989-1990 Texas Instruments Incorporated.         */
/*                         All Rights Reserved                           */
/*----------------------------------------------------------------------*/
/*  TMS340 Graphics Library						*/
/*----------------------------------------------------------------------*/
/*                                                                      */
/*  name            install_font() -  install a font                    */
/*                  select_font() - select an installed font            */
/*                  delete_font() - delete an installed font            */
/*                  set_textattr() - set text rendering attributes      */
/*                  get_textattr() - get text rendering attributes      */
/*                                                                      */
/*  description     This module contains routines for installing,       */
/*                  selecting, and deleting fonts, and routines for     */
/*                  setting text rendering attributes.                  */
/*----------------------------------------------------------------------*/
/* 01/20/89...Original Version Written....................J.G.Littleton */
/* 09/22/89...Added upper case controls and negative			*/
/*	      values in control string to set_textattr. 		*/
/*	      Fixed arg type defn. in set/get_textattr.....Graham Short */
/* 01/12/90...Added JERRY's fixes ref 22 Nov 1989......Michael Hanrahan */
/* 03/15/90...Fix install_font for static font table......J.R. Van Aken */
/* 03/21/90   Fill gaps option, sysfont now pointer,etc J.G.Littleton   */
/*----------------------------------------------------------------------*/
#include <gsptypes.h>
#include <gspglobs.h>

#define FONT_MAGIC 0x8040u

static int isdigit(), tolower();

/*----------------------------------------------------------------------*/
/*  install_font() - Install a font and return a font ID if successful, */
/*  zero otherwise.                                                     */
/*----------------------------------------------------------------------*/
int install_font( p )
FONT *p;
{
    int i, id;

    if (p->magic != FONT_MAGIC)
      return (0);

    /*--------------------------------------------------------------*/
    /*	Search font table for an empty slot.			    */
    /*--------------------------------------------------------------*/
    for (i = 0, id = -1; i < envtext.allocated && id < 0; i++)
	if (!*(envtext.font + i))
	    id = i;
        
    /*--------------------------------------------------------------*/
    /*	If no empty slot was found, then return 0		    */
    /*--------------------------------------------------------------*/
    if (id < 0)
	return (0);

    *(envtext.font + id) = p;
    envtext.installed += 1;

    return (id + 1);
}

/*----------------------------------------------------------------------*/
/*  select_font() - Select the system font or a previously installed    */
/*  font for rendering text.                                            */
/*  Return nonzero if successful, zero otherwise.                       */                      
/*----------------------------------------------------------------------*/
select_font( id )
register int id;
{
    register int retval = 0;

    if (id == 0)
    {
        envtext.selected = sysfont;
        retval = 1;
    }
    else
    {
        --id;
        if (id >= 0 && id < envtext.allocated && *(envtext.font + id))
        {
            envtext.selected = *(envtext.font + id);
            retval = 1;
        }
    }
    return (retval);
}

/*----------------------------------------------------------------------*/
/*  delete_font() - Remove a previously installed font from the system. */
/*  Return nonzero if successful, zero otherwise. If the font deleted   */
/*  was selected for rendering text, then select the system font.       */
/*----------------------------------------------------------------------*/
delete_font( id )
int id;
{
    register int retval = 0;
    
    /*------------------------------------------------------------------*/
    /*  If the requested ID is in range, and the font table contains    */
    /*  an entry for the ID, then delete the font.                      */
    /*------------------------------------------------------------------*/
    if ((--id >= 0) && (id < envtext.allocated) && (*(envtext.font+id)))
    {
        /*--------------------------------------------------------------*/
        /*  If we're deleting the selected font, then select sysfont.   */
        /*--------------------------------------------------------------*/
        if (*(envtext.font + id) == envtext.selected)
            envtext.selected = sysfont;
        *(envtext.font + id) = 0;
        retval = 1;
    }
    return ( retval);
}

/*----------------------------------------------------------------------*/
/*  set_textattr() - Set text rendering attributes under the control    */
/*  of a string.                                                        */
/*----------------------------------------------------------------------*/
set_textattr( ctl, count, arg)
unsigned char *ctl;
int           count;
short         *arg;
{
    int assigned, val, sign;

    for (assigned = 0; *ctl; ctl++)
    {
        while ((*ctl != '\0') && (*ctl != '%'))
            ctl++;
        if (*ctl == '\0')
            break;
        
        if (*++ctl == '*')
            ++ctl, val = *arg++;
        else
        {
            if (*ctl == '-')
            {
               ctl++;
               sign = -1;
            }
            else
               sign =  1;
            for (val = 0; isdigit(*ctl) ; )
                val = val * 10 + (*ctl++ - '0');
            val *= sign;
        }
        switch (tolower(*ctl))
        {
            case 'a':       /* Alignment */
                envtext.align = (val > 0); 
                ++assigned;
                break;
            case 'e':       /* Extra intercharacter spacing */
                envtext.charextra = val; 
                ++assigned;
                break;
            case 'f':       /* Fill intercharacter gaps */
                envtext.effects &= ~1; 
                envtext.effects |= (val != 0); 
                ++assigned;
                break;
            case 'r':   /* Reset options */
                envtext.align = 0;
                envtext.effects = 0;
                envtext.charextra = 0;
                ++assigned;
                break;
            default:
                break;
        }
    }
    return (assigned);
}

/*----------------------------------------------------------------------*/
/*  get_textattr() - Get text rendering attributes under the control    */
/*  of a string.                                                        */
/*----------------------------------------------------------------------*/
get_textattr( ctl, count, arg)
unsigned char *ctl;
int           count;
short         *arg;
{
    int assigned;

    for (assigned = 0; *ctl; ctl++)
    {
        while ((*ctl != '\0') && (*ctl != '%'))
            ctl++;
        if (*ctl == '\0')
            break;
        switch (++ctl, tolower(*ctl))
        {
            case 'a':       /* Alignment */
                *arg++ = (envtext.align > 0);
                ++assigned;
                break;
            case 'e':       /* Extra spacing */
                *arg++ = (envtext.charextra);
                ++assigned;
                break;
            case 'f':       /* Fill Intercharacter gaps */
                *arg++ = (envtext.effects & 1);
                ++assigned;
                break;
            default:
                break;
        }
    }
    return (assigned);
}

static isdigit(c)
char c;
{
    return ( (c >= '0' && c <= '9'));
}

static tolower(c)
char c;
{
    if (c >= 'A' && c <= 'Z')
        c += ('a' - 'A');
    return (c);
}

