;*********************************************************************
;
; Filename: SLICER.ASM
;
; Author: Carl Nuzman, Hemanth Sampath
; Date: 10/24/95
; Last revision: 11/21/95
;
;*********************************************************************
;                  ROUTINE 1    : SLICER
;*********************************************************************
; Description:  Round the incoming values to an integer grid.  Then
;   give each point a real and imaginary index from 0 to 3 to use in
;   the squared magnitude look-up table.
;
; Algorithm:
;   1) Zero out the bottom n bits of the incoming point
;   2) Add a one into the nth bit, resulting in an odd lattice point
;      scaled by 2^(n-1) . This forms the ideal constellation point.
;   3) Store the upper 16-n bits.  If this is a negative number, add
;      one and negate the result, thus folding the negative numbers
;      onto the corresponding non-negatives 0,1,2,...
;   4) Clip this result to a maximum of 5
;   5) Take this clipped number as an index into the squared magnitude
;      table
;   6) Store the value from the table in sl_c_ims, needed by carrier
;       routine.
;
;*********************************************************************
;
; Usage: Called by "equalize" routine.  
;
; Inputs: modR and modI ( real and imaginary demodulated filter outputs
;         from the adaptive equaliser)
;
; Outputs: cnR and cnI ( Quantised values of modR and modI -- to the nearest
;          constellation point.) ; sl_c_ims which computes the inverse
;          squared magnitude of the constellation point.
; Variables used: see code below.
;
; Registers used: see code below.
;
;*********************************************************************
;*********************************************************************
; DECLARE GLOBAL VARIABLES
;**********************************************************************
      .global slicer,decode,pack
      .global modR,modI,sl_rind,sl_iind,sl_data,sl_ims
      .global cnR,cnI,sl_bits,sl_c_ims,descram_in,quadrant,pack_word

      .mmregs

;*********************************************************************
;               MAIN ROUTINE
;**********************************************************************
slicer:

	ldpk modR                       
	lacc modR, 6                    
	sach sl_rind                    
	lacc sl_rind, 10
	add #1, 9                       ; Real part of ideal constellation
	sacl cnR                                                 ; point.

	lacc modI, 6                    
	sach sl_iind
	lacc sl_iind, 10
	add #1, 9                      ; Imaginary part of ideal constellation 
	sacl cnI                       ;                        point.

	lacc sl_rind                    ;limit sl_rind to range 0-5
	bcnd rfold, LT
rfcont  sub #5
	bcnd r2big, GT
	add #5
rcont   sacl sl_rind

	lacc sl_iind                    ;limit sl_iind to range 0-5
	bcnd ifold, LT
ifcont  sub #5
	bcnd i2big, GT
	add #5
icont   sacl sl_iind


;-----------------------------------------------------------------------
; table lookup - use the indices to look up the inverse mag squared look
; up table. The corresponding value from hte look up table is needed for the 
;"carrier tracking routine" in file "eq.asm".
;


	lacc sl_iind, 3                 ; 6*iind + rind = index into 2d
	sub sl_iind, 1                  ;                    tables
	add sl_rind
	add #sl_ims                     ; address of the look up table.
	samm AR4
	mar *, AR4
	nop
	lacc *
	sacl sl_c_ims

	ret


;----------------------------------------------------------------
; map negative numbers onto non-negative numbers.
; i.e.  -2, -1 go to 1, 0, respectively


rfold:  add #1
	neg
	b rfcont
ifold:  add #1
	neg
	b ifcont


;---------------------------------------------------------------------
; Hard-limit the values, which are outside the constellation.

r2big:
	lacc #5
	b rcont
r2smll:
	lacc #0
	b rcont
i2big:
	lacc #5
	b icont
i2smll:
	lacc #0
	b icont








;****************************************************************
;                ROUTINE 2:   DECODE
;***************************************************************
;DESCRIPTION: 
;This routine differentially decodes a point in a constellation, based on 
;the previous constellation point.
;Using a point specified in cnR & cnI, determine current quadrant and
;determine output bits from current and previous quadrant.
;store the output in descram_in, to be used by the descrambler.
;***************************************************************

decode:
	ldpk cnR                        ; get quadrant
	lacc cnR
	bcnd xneg, LT
xpos:   lacc cnI
	bcnd xpyn, LT
xpyp:   lacc #0
	b dec_cont
xpyn:   lacc #1
	b dec_cont
xneg:   lacc cnI
	bcnd xnyn, LT
xnyp:   lacc #3
	b dec_cont
xnyn:   lacc #2

dec_cont:
	sacb
	sub quadrant                    ; subtract previous quandrant
	and #00003h                     ; implement modulo 4
	sacl descram_in                 ; store on data page 4
	lacb
	sacl quadrant                   ; store current quadrant for 
					; next time.

;--------------------------------------------------------------------------
; The following section decodes a special word sent by transmitter
; that indicates at what speed the data is being transmitted.


pack:                                   
	ldpk pack_word
	lacc pack_word+1, 16
	or   pack_word
	sfl
	sfl
	adds descram_in
	sach pack_word+1
	sacl pack_word
	ret




