;*********************************************************************
;
; Filename: sc.asm
;
;
; Author: Carl Nuzman, Hemanth Sampath, Hayden Metz.
; Date: 21 December, 1995
; Last revision:21 december, 1995.
;
;*********************************************************************
;
; Description: This routine is called by main receiver program, to track 
; the transmitter's symbol clock frequency. An upper and a lower Nyquist 
; Band pass filter is first implemented. The imaginary part of the cross-
; correlation of the filter output forms the timing error. Filters to 
; eliminate the extranneous cross-correlation terms are also implemented.
; A hard limiter and a random walk filter are added to the timing error 
; If the filtered output exeeds a positive threshold, the AIC phase phase 
; is advanced. When output is less than a certain negative threshold, 
; AIC phase is retarded. The phase is not changed when the output remains
; within the thresholds. Please see interrupt routine( xint) in the 
; main receiver program for details.
;*********************************************************************
;
; Usage: Called by main "receiver routine". The routine stores it's 
;        output in the threshold flag that is read by  the interrrupt routine 
;        in the main receiver program.
;
; Inputs: Inpbuffer ( input buffer), NL (have eta_l(n-1), eta_l(n-2),eta_l(n-3))
;         NH (have eta_h(n-1),eta_l(n-2),eta_l(n-3)),
;         Nl and NH ahve intermediate real variables to form high
;         and low band pass Nyquist filters.
; Outputs: NH,NL , thresh ( sets a flag that is used by interrupt routine
;          to bump the clock phase backwards or forwards, by adjusting
;          the AIC. 
; Variables used:  See individual sections below.
;
; Registers used: See individual sections below. 
;
;****************************************************************************
;               DECLARE GLOBAL VARIABLES:
	
	
	.mmregs

	.global sym_clock
	.global inpbuffer
	.global NL,NH,TEMP0,TEMP1,TEMP2,DELTAH,DELTAL,COUNTL,SC_WAIT
	.global VN,final,dc_0,dc_1,dc_bit,ac_0,ac_1,ac_2,ac_bit
	.global x,L,thresh,SC_FIRST,SC_AR1,SC_AR2

;********************************************************************
;       DESCRIPTION OF POINTERS USED
;
;
; AR2 - points to eta_l(n) and eta_h(n)
;       eta_l and eta_h are intermediate real variables used to form
;       high and low Nyquist filters.
; AR3 - It's a running counter; 2,1,0,2,1,0,2,1,0,......
; AR5 - points to current data in inpbuffer
;********************************************************************

;*******************************************************************
;               DESCRIPTION OF TYPE OF DATA USED
;
; The data in inputbuffer (inpbuffer) has both integer and fractional
; parts. The form of the data is indicated at appropriate places.
; Examples:
;
; Form 1:1:15  means that the given data has 1 sign bit, 1 integer bit
; and 15 fractional bits.
;
; Form 1:5:10  means that the data has 1 sign bit, 5 integer bits
; and 10 fractional bits.
;*********************************************************************


;*********************************************************************
;                       sc_init macro
;*********************************************************************
; This macro initialises pointers to appropriate variables.


sc_init .macro

	lar AR0, #inpbuffer
	lar AR2,#(NH+1)                  

	.endm
;***********************************************************************
	
	
	
	
	.text

;********************************************************************
;                       SYM_CLOCK (Main Routine)

sym_clock:
;--------------------------------------------------------------------                                        
;initialize variables, the first time sym_clock is called. This is done
; by checking the appropriate bit in sc_first. If it is not the first time
; sym_clock is called, then restore values of AR2 from previous sampling 
; instant. Every time sym_clock is called, all pointers to be used in the 
; subsequent sampling instant are saved, to prevent them from getting over-
; written by other routines called by main program.
	
	ldpk SC_FIRST                   
	bit SC_FIRST, 15
	bcnd sc_jaded, NTC
	sc_init
	apl #0, SC_FIRST
	b sc_main

sc_jaded:
					; restore values of AR2.
	ldpk SC_AR1

	lar AR2, SC_AR2

;-------------------------------------------------------------------
sc_main:


;--------------------------------------------------------------------
; Implement Upper Nyquist Band Pass filter. The first step is to compute
; the intermediate real variable every sampling instant.
;   eta_h(n) = AR5 + AH*eta_h(n-1) +BR*eta_h(n-2),
;       where AR5 points to the current data in (inpbuffer) inputbuffer.
;       AH = cos(wu*T)*Nu*2   ; wu =  upper Nyquist frequency = 3000Hz.
;                               Nu = constant
;       BR = Nu**2  (square of Nu).

; Initially, #(NH+1) has eta_h(n-2) 
;            #(NH)  has eta_h(n-1)
	   
	
	mar *, AR5                      ; form = 1:4:11
	zpr
					
	lacc *, 9, AR2                  ; shift left 14 for alignment,
					; right 5 for scaling 
	macd BR, *-                     
	macd AH, *
	apac                            
	bsar 14                         

	sacl *-                          ; #(NH) has eta_h(n)
					 ; #(NH+1) has eta_h (n+1) 
					 ; #(NH+2) has eta_h(n+2)

;----------------------------------------------------------------------
; Implement Lower Nyquist Band Pass filter. The first step is to compute
; the intermediate real variable every sampling instant.
;   eta_l(n) = AR5 + AL*eta_l(n-1) + BR*eta_l(n-2),
;       where AR5 points to the current data in (inpbuffer) inputbuffer.
;       AL = cos(wl*T)*Nu*2   ; wl =  lower Nyquist frequency = 600Hz.
;                               Nu = constant
;       BR = Nu**2  (square of Nu).

; Initially, #(NL+1) has eta_l(n-2) 
;            #(NL)  has eta_l(n-1)


	mar *-, AR5                       ; AR2 points to #(NL+1)
	zap
	lacc *, 9, AR2                  

	macd BR, *-
	macd AL, *
	apac
	bsar 14
	sacl *                            ; #(NL) has eta_l(n)
					  ; #(NL+1) has eta_l(n+1)
					  ; #(NL+2) has eta_l(n+2)
;-------------------------------------------------------------------------
; set AR2 back to (NH+1), to be used the next time sym_clock is called.
	
	mar  *,AR2
	adrk 4                          
;-------------------------------------------------------------------------
; The following sections are called only once a baud,i.e; only once every
; 3 sampling instants. AR3 is a running counter which runs like: 2,1,0
;,2,1,0,2,1,0,2,1,0... and so on. Every time an interrrupt gets called
; AR3 is decremented. Hence, by tapping the value of AR3 , we know where we 
; are in a given baud.
	
	mar *, AR3                      ; return here if AR3 not zero
	banz sc_save, *


;--------------------------------------------------------------------------
; Compute the imaginary part of the cross-correlation of high and low 
; Nyquist bandpass filter outputs. In other words,
; compute v(n) = Imaginary(gu*gl')
;               where gu = High Nqst band pass filter output
;                     gl = low Nqst band pass filter output.
; v(n) is called the timing error signal.


	zap
	ldpk NL
	lt (NL+1)
	mpy (NH+1)
	mpya (NH)
	sach TEMP0, 1                   ;Temp0 = eta_l(n-1)*eta_h(n-1)
	pac
	sach TEMP1, 1                   ;Temp1 = eta_l(n-1)*eta_h(n)
	lt (NL)
	mpy (NH+1)
	pac
	sach TEMP2, 1                   ;Temp2= eta_l(n)*eta_h(n-1)
	zap
	mac DH, TEMP2
	mac NDL, TEMP1
	mac CHCL, TEMP0
	apac                            ; acc = Im(gh*gl')
					; form = 2:0:30
	bsar 8                          ; shift right 15 for alignment,
					; left 7 for scaling

;--------------------------------------------------------------------------
; Derive filters to eliminate the extraneous cross-correlation terms
;    dc filter: h(z) = z - z^-2
;    ac filter: h(z) = z + z^-3
; Also, store v(n) in acc. buffer.
; DC-bit can have the values 0,1 
; AC-bit can have values 0,1,2
					
; Implement DC filter.        
; If DC-bit =0, then make DC-bit = 1;  Answer = Acc - dc_0
; If DC-bit =1, then make DC-bit = 0;  Answer = Acc - dc_1   
	
	sacb                              
	lacc dc_bit                        
	bcnd dc_one, GT
	opl #1h, dc_bit
	sub dc_0
	sacl final
	lacb
	sacl dc_0
	b dc_cont
dc_one
	apl #0h, dc_bit
	sub dc_1
	sacl final
	lacb
	sacl dc_1                          

; Implement A.C filter.  
; if ac_bit =0, make ac_bit =1; answer = acc + ac_0
; if ac_bit =1, make ac_bit = 2; answer = acc + ac_1
; if ac_bit =2, make ac_bit =0; answer = acc + ac_2

dc_cont
	sacb
	lacc ac_bit
	bcnd ac_zero, EQ
	sub #1
	bcnd ac_one, EQ
	apl #0h, ac_bit
	add ac_2
	sacl final
	lacb
	sacl ac_2
	b ac_cont
ac_one
	apl #0h, ac_bit
	opl #2h, ac_bit
	add ac_1
	sacl final
	lacb
	sacl ac_1
	b ac_cont
ac_zero
	apl #0h, ac_bit
	opl #1h, ac_bit

	add ac_0
	sacl final
	lacb
	sacl ac_0
ac_cont

;-----------------------------------------------------------------------
; Put a hard limiter to the v(n) generated.
	sacl VN ; debugging only
	
	bcnd hl_lt, LT
	bcnd hl_gt, GT
	lacc #0000h                           
	b hl_end
hl_gt
	lacc #00010h                          ; acc = Threshold (L)  
	b hl_end
hl_lt
	lacc #0fff0h                          ; acc = -L (Threshold) 
hl_end

;------------------------------------------------------------------
; Implement Second order smoothing filter to the hard-limited value
; v(n). However, we found through experimentation that
; the 2nd order loop converges, only if it is turned on after a considerable
; number of bauds. Hence, we decided to implement just the first order
; loop by using the following branch.

	b newfilt


;skipped 2nd order loop which computes output in double precision.
	

	bit SC_WAIT, 15
	bcnd skip_ab, TC
	b newfilt
	bsar 4
	sacb                            ; store hard limiter output in accb
	nop
	lacc DELTAH, 16
	or DELTAL
	addb
	sacl DELTAL
	sach DELTAH
	bsar 16
	bsar 14                         
	addb                            
	b newfilt

skip_ab
	sacb
	lacc COUNTL
	sub #1
	sacl COUNTL
	bcnd restore, GT                
	apl #0h, SC_WAIT                
	splk #0040h, L                  

restore
	lacb



;--------------------------------------------------------------------
; Now implement a random walk filter on to the hard-limited  value of
; v(n) to reduce symbol clock jitter. Implement the following equation:
; y(n) = x(n) + v(n) where
;                       x(n+1) = x(n) + v(n) - q(n)
; Note  q(n) = L  if y(n) >= L       ; L is the threshold which can be 
;            = -L if y(n) <= -L      ; varied in the main receiver program.
;            = 0 otherwise.          ;
;

newfilt:

	add x                           ; acc has y(n) = x(n) + v(n)
	sacl final                      ; store y(n)
	bcnd  plus,GT                   ; if y(n) > 0, check if y(n) >= L
	bcnd  minus,LT                  ; if y(n) < 0, check if y(n) <= -L
	b getout                        ; y(n) = 0, so -L < y(n) < L

plus    sacb                            ; store the current value in 
					; buffer in case y(n) < L
	sub L
	bcnd zero,LT                    ; y(n) < L, get unaltered y(n)
					; from buffer
					; set a flag to be used by
					; interrupt routine.
	opl #1,thresh                   ; set threshold flag to 1 -->
	b getout                        ; y(n) >= L
					; x(n+1) is in ACC


minus   sacb                            ; store the current value in the
					; buffer in case y(n) > -L
	add L
	bcnd zero,GT                    ; y(n) >= L, get unaltered y(n)
					;   from buffer
					; set flag to be used by the interrupt
					;   routine.
	opl #0ffffh, thresh             ; set threshold flag to -1 --> y(n) <= -L
	b getout                        ; x(n+1) is in ACC

zero    lacb                            ; -L < y(n) < L, retrieve y(n)
					;   from buffer
getout
	sacl x                          ; store x(n+1) --> x(n) for next
					;   iteration

;----------------------------------------------------------------------
; end of sym_clock:  Save all AUX registers to be used in the nest iteration.


sc_save:                                ; save AUX register values

	sar AR2, SC_AR2

sc_end:
	ret

;********************************************************************
; Store constants in program memory: 
;
; Convention: The actual values computed by hand were:
;             AU = -1.714730299  {2 *v* cos[w(high)*(T/3)]}
;
;             AL = 1.714730299  {2 *v* cos[w(low)*(T/3)]}
;
;             BR = -0.9801      - { v^2}

; The following constants are used in intermediate computations, for
; upper and Lower Nyquist Band PAss filters.
;             DH = 0.495        { v*sin[w(high)*(T/3)] }
;             NDL = -0.495        -v*sin[w(low)*(T/3)]
;             CHCL = -.848791498     - v*v* sin[ (w(high)-w(low))*T/3 ]
;*********************************************************************


AH: .word 09242h                        ; form:  1:1:14
BR: .word 0c147h                        ; form= 1:1:14
AL: .word 06dbeh                        ; form = 1:1:14

DH: .word 03f5ch                        ; form: 1:0:15
NDL: .word 0c0a4h                       ; form: 1:0:15
CHCL: .word 0935bh                      ; form: 1:0:15


	.end
