**********************************************************************
*
* Filename:  fft3.asm
*
* Author: Carl Nuzman, Hemanth Sampath
* Date:  12/14/95
* Last revision: 12/26/95
*
;**********************************************************************
;*
;* Description: Take 144-point decimation in frequency fft. This routine
;* gets called by fast_equalise routine (in fast.asm file).
;* This routine breaks up the 144 point FFT into 2 4-point DFT's and
;* 2 3-point DFT's.
;*
;* Algorithm: see individual sections below.
;*
;**********************************************************************
;*
;* Usage: Called by fast_equalise routine.
;* Note: An integer equal to (16-k) must have to be loaded into TREG1
;*       before calling the routine. Here k is an integer from 1,2,3...
;*       The output will be shifted with respect to the input by 4*k
;*       bits.
;*
;*
;* Inputs: 144 real time samples stored in fr(i)
;*         144 imaginary samples stored in fi(i)    ; where i = 0.. 143.
;*
;*         samples are stored in an unscrambled order.
;*
;*
;* Outputs: 144 real frequency samples stored in fr(i)
;*          144 imaginary samples stored in fi(i)
;*
;*       The output is in a scrambled order and hence an appropriate
;*       descrambler routine must be called after using this routine.
;*
;* Note: The format of the output is shifted 4*k bits to the right
;*       compared to the format of the input. For example, if the
;*       inputs are in q14 format, then the output is in q6 format.
;*       If k>=2, then there will be no overflows.
;*
;*
;* Variables used:  see code below.
;*
;* Registers used:  see code below.
;*
;***************************************************************************
; Comments:
; 1. 	This routine scales down the answers so that they are
; 	guaranteed not to overflow. It does this by scaling
; 	down by four bits in each stage of the fft computation.
; 2.    This routine uses 2 loop-up tables, stored in appropriate
;       data pages in the main receiver program.

;	They are: "WR" table and "wtable3b".
;	"wr" table is a 180 word cosine table, which computes the cosine
;	of the angle between 0 and 5*Pi/2.
;	"wtable3b" is a table used by routine in computing the 2nd 3-point
;	FFT. This look up table, provides an index to the "wr" table.
;
;**************************************************************************
;               DECLARE GLOBAL VARIABLES
;**************************************************************************

	.global fr
	.global fi
	.global fr
	.global fi
	.global t0r
	.global t0i
	.global t1r
	.global t1i
	.global t2r
	.global t2i
	.global t3r
	.global t3i
	.global t4r
	.global t4i
	.global aar
	.global aai
	.global bbr
	.global bbi
	.global ccr
	.global cci
	.global ddr
	.global ddi
	.global w1r
	.global w1i
	.global w2r
	.global w2i

	.global count4b
	.global wtable3b
	.global wr
	.global fft
	.global hi
	.global hr


	.mmregs


;**************************************************************************
;               ROUTINE: FFT
;**************************************************************************

fft:


;------------------------------------------------------------------------
;  First  compute a 4-point fft on the 144 real and imaginary inputs: fr(j)
; and fi(j).
;  Code: For i = 1..36  do
;
;        aar=fr(i)
;	 bbr=fr(i+36)
;	 ccr=fr(i+72)
;	 ddr=fr(i+108)

;   	 fr(i) =aar+bbr+ccr+ddr

;        aai=fi(i)
;	 bbi=fi(i+36)
;	 cci=fi(i+72)
;	 ddi=fi(i+108)

;   	 fi(i) =aai+bbi+cci+ddi

;        t1r = aar +bbi - ccr - ddi
; 	 t1i = aai - bbr -cci + ddr
; 	 t2r = aar - bbr + ccr -ddr
; 	 t2i = aai -bbi + cci - ddi
; 	 t3r = aar - bbi - ccr + ddi
; 	 t3i = aai +bbr -cci -ddr

;	 fr(i+36)=t1r * wr(1+i) -wr(1+36+i) *t1i
;        fr(i+72)=t2r * wr(1+2*i)-t2i * wr(1 +36+2*i)
;        fr(i+108)=t3r * wr(1+3*i)-t3i * wr(1 +36+3*i)
;	 fi(i+36)=t1i * wr(1+i) + wr(1+36+i) *t1r
;        fi(i+72)=t2i * wr(1+2*i)+ t2r * wr(1+36+2*i)
;        fi(i+108)=t3i * wr(1+3*i)+ t3r * wr(1+36+3*i)

;  end;

;  Where "wr" is the 180 word cosine table in the main receiver program.


	lacc #36
	samm INDX
	lar AR0, #(fr+108)                ; Initialise pointers.
	lar AR1, #(fi+108)
	lar AR2, #(wr-1)
	lar AR4, #(wr-2)
	lar AR7, #(wr-3)

	mar *, AR0                        ; Repeat for i = 1 ..36
	ldp #0
	splk #35, BRCR
	ldpk t0r
	rptb end_4_a - 1

	sbrk #108
	lacc *0+
	sacl aar                          ; aar = fr(i)
	lacc *0+
	sacl bbr                          ; bbr = fr(i+36)
	lacc *0+
	sacl ccr                          ; ccr = fr(i+72)
	lacc *, AR1
	sacl ddr                          ; ddr = fr(i+108)
	sbrk #108
	lacc *0+
	sacl aai                          ; aai = fi(i)
	lacc *0+
	sacl bbi                          ; bbi = fi(i+36)
	lacc *0+
	sacl cci                          ; cci = fi(i+72)
	lact *
	sach ddi, 2                       ; ddi = fi(i+108)


	addt aai
	addt bbi
	addt cci                          ; acc = ddi + cci + bbi + aai
	sbrk #108
	sach *0+,AR0                      ; store answer in fi(i)
	lact aar
	addt bbr
	addt ccr
	addt ddr
	sbrk #108                         ; acc = ddr + ccr + bbr + aar
	sach *0+, AR2                     ; store answer in fr(i)

	lact aar
	addt bbi
	subt ccr
	subt ddi
	sach t1r
	lact aai                          ; t1r = aar +bbi - ccr - ddi
	subt bbr
	subt cci
	addt ddr
	sach t1i                          ; t1i = aai - bbr -cci + ddr

	lact aar
	subt bbr
	addt ccr
	subt ddr
	sach t2r                          ; t2r = aar - bbr + ccr -ddr
	lact aai
	subt bbi
	addt cci
	subt ddi
	sach t2i
					  ; t2i = aai -bbi + cci - ddi
	lact aar
	subt bbi
	subt ccr
	addt ddi
	sach t3r                          ; t3r = aar - bbi - ccr + ddi
	lact aai
	addt bbr
	subt cci
	subt ddr
	sach t3i                          ; t3i = aai +bbr -cci -ddr

	adrk #1         		  ; AR2 is current pointer to wr(1+i)
	lt  t1r
	mpy *0+                           ; P1 = t1r * wr(1+i)
	ltp *0-, AR0
	mpy t1i                           ; P2 = wr(1+36+i) *t1i
	lts t1i
	sach *0+, 2, AR2                  ; acc = p1 -p2  = fr(i+36)
	mpy *0+                           ; P1 = tli * wr(1+i)
	ltp  *0-, AR1                     ;
	mpy  t1r                          ; p2 = t1r * wr(1+ 36+i)
	apac
	sach *0+, 2, AR4                  ; acc = p1 +p2 = fi(i+36)

	adrk #2        			  ; AR4 is current pointer to wr(1+2*i)
	lt  t2r
	mpy *0+                           ; p1 = t2r * wr(1+2*i)
	ltp *0-, AR0
	mpy t2i                           ; p2 = t2i * wr(1+36+2*i)
	lts t2i
	sach *0+, 2, AR4                  ; acc = p1 -p2 = fr(i+72)
	mpy *0+
	ltp *0-, AR1
	mpy t2r
	apac
	sach *0+, 2, AR7                  ; fi(i+72) = t2i * wr(1+i*2) +
					  ; t2r * wr(1+36+i*2)

	adrk #3        			  ; current pointer is AR7 to wr(1+i*3)
	lt  t3r
	mpy *0+                           ; p1 = t3r * wr(1+3*i)
	ltp *0-, AR0
	mpy t3i                           ; p2 = t3i * wr(1+36+3*i)
	lts t3i
	sach *+, 2, AR7                   ; acc = p1 -p2 = fr(i+108)
	mpy *0+
	ltp  *0-, AR1
	mpy  t3r
	apac
	sach *+, 2, AR0                   ; fi(i+108) = t3i *wr(1+3*i) +
					  ; t3r*wr(1+36+3*i)
end_4_a:


;-------------------------------------------------------------------------
; Second 4-point fft :
;code:
;	for k =0..3 do
;	for i=1..9 do

;        aar=fr(36*k+i)
;	 bbr=fr(36*k+i+9)
;	 ccr=fr(36*k+i+18)
;	 ddr=fr(36*k+i+27)

;   	 fr(36*k+i) =aar+bbr+ccr+ddr

;        aai=fi(36*k+i)
;	 bbi=fi(36*k+i+9)
;	 cci=fi(36*k+i+18)
;	 ddi=fi(36*k+i+27)

;   	 fi(36*k+i) =aai+bbi+cci+ddi

;        t1r = aar +bbi - ccr - ddi
; 	 t1i = aai - bbr -cci + ddr
; 	 t2r = aar - bbr + ccr -ddr
; 	 t2i = aai -bbi + cci - ddi
; 	 t3r = aar - bbi - ccr + ddi
; 	 t3i = aai +bbr -cci -ddr

;	 fr(36*k+i+9)=t1r * wr(36*k+1+i*4) -wr(36*k+1+36+i*4) *t1i
;        fr(36*k+i+18)=t2r * wr(36*k+1+8*i)-t2i * wr(36*k+1+36+8*i)
;        fr(36*k+i+27)=t3r * wr(36*k+1+12*i)-t3i * wr(36*k+1+36+12*i)

;	 fi(36*k+i+9)=t1i * wr(36*k+1+i*4) +wr(36*k+1+36+i*4) *t1r
;        fi(36*k+i+18)=t2i * wr(36*k+1+8*i)+t2r * wr(36*k+1+36+8*i)
;        fi(36*k+i+27)=t3i * wr(36*k+1+12*i)+t3r * wr(36*k+1+36+12*i)

;  end;
;  end;



	lacc #3
	sacl count4b

	lacc #9
	samm INDX
	lar AR0, #(fr+27)
	lar AR1, #(fi+27)
	mar *, AR0

start_4_b:                                 ; Initialise pointers.
	lar AR2, #(wr+32)
	lar AR4, #(wr+28)
	lar AR7, #(wr+24)

	ldp #0
	splk #8, BRCR                      ; repeat for i =1..8.
	ldpk t0r
	rptb end_4_b - 1

	sbrk #27
	lacc *0+
	sacl aar                           ; aar = fr(i)
	lacc *0+
	sacl bbr                           ; bbr = fr(i+9)
	lacc *0+
	sacl ccr                           ;ccr = fr(i+18)
	lacc *, AR1
	sacl ddr
	sbrk #27                           ; ddr = fr(i+27)
	lacc *0+
	sacl aai
	lacc *0+                           ; aai = fi(i)
	sacl bbi
	lacc *0+                           ; bbi = fi(i+9)
	sacl cci
	lact *                             ; cci = fi ( i+18)
	sach ddi, 2
					   ; ddi = fi (i+27)
	addt aai
	addt bbi
	addt cci
	sbrk #27
	sach *0+,AR0                       ; fi(i) = aai+bbi+cci+ddi
	lact aar
	addt bbr
	addt ccr
	addt ddr
	sbrk #27
	sach *0+, AR2                      ; fr(i) = aar +bbr+ccr+ddr

	lact aar
	addt bbi
	subt ccr
	subt ddi
	sach t1r                           ;t1r = aar+bbi-ccr-ddi
	lact aai
	subt bbr
	subt cci
	addt ddr
	sach t1i                           ;t1i = aai-bbr-cci+ddr

	lact aar
	subt bbr
	addt ccr
	subt ddr
	sach t2r                           ;t2r = aar-bbr+ccr-ddr
	lact aai
	subt bbi
	addt cci
	subt ddi
	sach t2i                           ;t2i = aai -bbi +cci -ddi

	lact aar
	subt bbi
	subt ccr
	addt ddi
	sach t3r                           ;t3r = aar -bbi -ccr +ddi
	lact aai
	addt bbr
	subt cci
	subt ddr
	sach t3i                            ;t3i = aai+bbr-cci-ddr

	sbrk #32        		    ;AR2 points to wr+i*4
	lt  t1r
	mpy *                               ;P1 = t1r * wr(1+i*4)
	adrk #36
	ltp *, AR0
	mpy t1i                             ;p2 = wr(1+4*i+36)*t1i
	lts t1i
	sach *0+, 2, AR2                    ;fr(i+9) = p1 -p2
	sbrk #36
	mpy *                               ;p1 = wr(1+i*4)*t1i
	adrk #36
	ltp  *, AR1
	mpy  t1r                            ;p2 = wr(1+i*4+36) * t1r
	apac
	sach *0+, 2, AR4                    ;fi(i+9) = p1+p2

	sbrk #28
	lt  t2r                             ; AR4 points to wr(1+8*i)
	mpy *
	adrk #36                            ; p1 = t2r* wr(1+i*8)
	ltp *, AR0
	mpy t2i                             ;p2 =t2i* wr(1+8*i+36)
	lts t2i
	sach *0+, 2, AR4                    ; fr(i+18)=p1 -p2
	sbrk #36
	mpy *                               ; p1 = wr(1+i*8)*t2i
	adrk #36
	ltp *, AR1
	mpy t2r                              ;p2 = wr(1+i*8 +36)*t2r
	apac
	sach *0+, 2, AR7                     ; fr(i+18) =p1+p2

	sbrk #24                             ; AR7 points to wr(1+i*12)
	lt  t3r
	mpy *                                ; p1 = t3r* wr(1+i*12)
	adrk #36
	ltp *, AR0
	mpy t3i                              ;p2 =t3i* wr(1+12*i+36)
	lts t3i                              ;fr(i+27)=p1 -p2
	sach *+, 2, AR7
	sbrk #36
	mpy *                                ;p1 = wr(1+i*12)*t3i
	adrk #36
	ltp  *, AR1
	mpy  t3r                             ;p2 = wr(1+i*12)*t3r
	apac
	sach *+, 2, AR0                      ;fi(i+27) = p1 + p2

end_4_b:

	adrk #27
	mar *, AR1                           ; Increment AR0 and AR1
	adrk #27                             ; by 27 and continue to do
	mar *, AR0                           ; the above algorithm
	lacc count4b                         ; until count4b <0.
	sub #1
	sacl count4b
	bcnd start_4_b, GEQ




;--------------------------------------------------------------------------
; First 3-point fft:
; Code:
;	For k=0..15 do
; 	For i =1..3 do
	;aar = fr(9*k+i)
	;bbr = fr(9*k+i+3)
	;ccr = fr(9*k+i+6)

	;fr(9*k+i) =aar+bbr+ccr
	;aai = fi(9*k+i)
	;bbi = fi(9*k+i+3)
	;cci = fi(9*k+i+6)

	;fi(9*k+i) =aai+bbi+cci

	;t3r=bbr*w1r -w1i*bbi
	;t3i= bbi*w1r +w1i*bbr
	;t4r= ccr*w2r-w2i*cci
	;t4i= cci*w2r +w2i*ccr

	;fr(9*k+i+3)=aar+t3r +t4r
	;fi(9*k+i+3)=aai+t3i+t4i

	;t3r=bbr*w2r -w2i*bbi
	;t3i= bbi*w2r +w2i*bbr
	;t4r= ccr*w1r-w1i*cci
	;t4i= cci*w1r +w1i*ccr

	;fr(9*k+i+6)=aar+t3r +t4r
	;fi(9*k+i+6)=aai+t3i+t4i

	; end
	; end




	lacc #15
	sacl count4b

	lar AR0, #(wr+48)
	lacc *
	sacl w1r                             ;w1r =wr(49)
	adrk #36
	lacc *
	sacl w1i
	adrk #12                             ;w1i = wr(49+36)
	lacc *
	sacl w2r
	adrk #36                             ;w2r = wr(49+36+12)
	lacc *
	sacl w2i
					     ;w2i = wr(49+36+12+36)

	lacc #3
	samm INDX
	lar AR0, #(fr+6)                     ;Initialise pointers.
	lar AR1, #(fi+6)
	mar *, AR0

start_3_a:
	ldp #0
	splk #2, BRCR                        ;repeat for i =1..3
	ldpk t0r
	rptb end_3_a - 1

	sbrk #6                              ;AR0 points to fr(1+6+i)
	lacc *0+
	sacl aar                             ;aar = fr(i)
	lacc *0+
	sacl bbr
	lacc *, AR1                          ;bbr = fr(i+3)
	sacl ccr
	sbrk #6                              ;ccr = fr(i+6)
	lacc *0+
	sacl aai                             ;aai =fi(i)
	lacc *0+
	sacl bbi                             ;bbi =fi(i+3)
	lact *
	sach cci, 2                          ;cci =fi(i+6)

	addt bbi
	addt aai
	sbrk #6
	sach *0+, AR0                        ; fi(i) = cci+bbi+aai

	lact aar
	addt bbr
	addt ccr
	sbrk #6
	sach *0+                             ; fr(i) =aar+bbr+ccr

	lt  bbr
	mpy w1r
	ltp w1i
	mpy bbi
	lts bbi
	sach t3r, 2                          ;t3r=bbr*w1r -w1i*bbi
	mpy w1r
	ltp  w1i
	mpy  bbr
	apac
	sach t3i, 2                          ;t3i= bbi*w1r +w1i*bbr

	lt  ccr
	mpy w2r
	ltp w2i
	mpy cci
	lts cci
	sach t4r, 2                          ;t4r= ccr*w2r-w2i*cci
	mpy w2r
	ltp  w2i
	mpy  ccr
	apac
	sach t4i, 2                          ;t4i= cci*w2r +w2i*ccr

	lact aar
	addt t3r
	addt t4r                             ;fr(i+3)=aar+t3r +t4r
	sach *0+, AR1

	lact aai
	addt t3i
	addt t4i
	sach *0+, AR0                        ;fi(i+3)=aai+t3i+t4i

	lt  bbr
	mpy w2r
	ltp w2i
	mpy bbi
	lts bbi
	sach t3r, 2                          ;t3r = bbr*w2r - bbi*w2i
	mpy w2r
	ltp  w2i
	mpy  bbr
	apac
	sach t3i, 2                          ;t3i = bbi*w2r + w2i*bbr

	lt  ccr
	mpy w1r
	ltp w1i
	mpy cci
	lts cci
	sach t4r, 2                          ;t4r=ccr*w1r-cci*wli
	mpy w1r
	ltp  w1i
	mpy  ccr
	apac
	sach t4i, 2                          ;t4i=cci*w1r+wli*ccr

	lact aar
	addt t3r
	addt t4r
	sach *+, AR1                         ;fr(i+6)=aar+t3r+t4r

	lact aai
	addt t3i
	addt t4i
	sach *+, AR0                         ;fi(i+6)=aai+t3i+t4i
end_3_a

	adrk #6
	mar *, AR1
	adrk #6
	mar *, AR0
	lacc count4b
	sub #1
	sacl count4b
	bcnd start_3_a, GEQ



;------------------------------------------------------------------------
; Second 3-point fft : this involves implementing 48 3-point fft's.
; Description of pointers:

	;AR2 points to table indexing into wtable3b. wtable3b is an 18 word
	; 	table.
	;AR4 points to the wr table, using whatever index is specified
	;	by wtable3b.
	;    For example: wr(wtable3b+36) , points to the 37-th word in the
	;    "wr" table, which is a 180 word cosine table.
	;AR7 points to the wr table, using whatever index is specified
	;	by wtable3b.



; code:

; for i=1..144 (increment in steps of 3) do

	;aar = fr(i);
	;bbr = fr(i+1)
	;ccr =fr(i+2)
	;aai =fi(i)
	;bbi=fi(i+1)
	;cci =fi(i+2)

	;#(wr(table3b+i) = AR4                     ; point to wr table
	;AR7 = #(wr(table3b+36+i))

	;t3r=bbr*AR4-AR7*bbi
	;t3i=bbi*AR4+bbr*AR7

	;#(wr(table3b+1+i) = AR4                   ; point to wr table
	;AR7 = #(wr(table3b+36+1+i))
	;t4r=ccr*AR4-AR7*cci
	;t4i=ci*AR4+ccr*AR7


	;fi(i)=aai+t3i+t4i
	;fr(i)=aar+t3r+t4r;

	;#(wr(table3b+2+i) = AR4                   ; point to wr table
	;AR7 = #(wr(table3b+36+2+i))

	;t3r=bbr*AR4-AR7*bbi
	;t3i=bbi*AR4+bbr*AR7

	;t4r=ccr*AR4-AR7*cci
	;t4i=ci*AR4+ccr*AR7


	;fi(i+1)=aai+t3i+t4i
	;fr(i+1)=aar+t3r+t4r;

	;#(wr(table3b+3+i) = AR4                   ; point to wr table.
	;AR7 = #(wr(table3b+36+3+i))
	;t3r=bbr*AR4-AR7*bbi
	;t3i=bbi*AR4+bbr*AR7

	;#(wr(table3b+4+i) = AR4
	;AR7 = #(wr(table3b+36+4+i))

	;t4r=ccr*AR4-AR7*cci
	;t4i=ci*AR4+ccr*AR7

	;fi(i+2)=aai+t3i+t4i
	;fr(i+2)=aar+t3r+t4r;

	;end.




	mar *, AR0
	lar AR0, #(fr)                  ;Initialise pointers.
	lar AR1, #(fi)
	lar AR2, #(wtable3b)
	lacc #0
	sacl count4b

	ldp #0
	splk #47, BRCR          	;For i=1..48 implement the following
	ldpk t0r                                                    ; code.
	rptb end_3_b - 1

	lacc count4b            ;reinitialize AR2 every third time through
	sub #1
	bcnd cont_3_b, GEQ
	lar AR2, #(wtable3b)
	lacc #2
cont_3_b:
	sacl count4b

	lacc *+
	sacl aar                         ;aar = fr(i)
	lacc *+
	sacl bbr                         ;bbr = fr(i+1)
	lacc *, AR1
	sacl ccr                         ;ccr =fr(i+2)
	lacc *+
	sacl aai                         ;aai =fi(i)
	lacc *+
	sacl bbi                         ;bbi=fi(i+1)
	lacc *, AR2
	sacl cci                         ;cci= fi(i+2)

	lacc #(wr)
	add *+, AR4                      ;acc = #(wr(table3b) = AR4
	samm AR4
	add #36
	samm AR7                         ;AR7 = #(wr(table+36))

	lt  bbr
	mpy *, AR7
	ltp *, AR4
	mpy bbi
	lts bbi
	sach t3r, 2                      ;t3r=bbr*AR4-AR7*bbi
	mpy *, AR7
	ltp  *, AR2
	mpy  bbr
	apac
	sach t3i, 2                      ;t3i=bbi*AR4+bbr*AR7

	lacc #(wr)
	add *+, AR4
	samm AR4                         ;acc = #wr(table3b+1) =AR4
	add #36
	samm AR7                         ;AR7=#wr(wtable3b+1+36)

	lt  ccr
	mpy *, AR7
	ltp *, AR4
	mpy cci
	lts cci
	sach t4r, 2                       ;t4r=ccr*AR4-cci*AR7
	mpy *, AR7
	ltp  *, AR1
	mpy  ccr
	apac
	sach t4i, 2                       ;t4i=cci*AR4+AR7*ccr

	lact aai
	addt t3i
	addt t4i
	sbrk #2
	sach *+, AR0                      ;fi(i)=aai+t3i+t4i

	lact aar
	addt t3r
	addt t4r
	sbrk #2
	sach *+, AR2                      ;fr(i)=aar+t3r+t4r

	;change w1,w2 here

	lacc #(wr)
	add *+, AR4
	samm AR4                          ;AR4=#(wr(wtable3b+2))
	add #36
	samm AR7                          ;AR7=#(wr(wtable3b+2+36))

	lt  bbr
	mpy *, AR7
	ltp *, AR4
	mpy bbi
	lts bbi
	sach t3r, 2                       ;t3r=bbr*AR4 - bbi*AR7
	mpy *,AR7
	ltp  *, AR2
	mpy  bbr
	apac
	sach t3i, 2                       ;t3i=bbi*AR4 + bbr*AR7


	lacc #(wr)
	add *+, AR4                       ;AR4=#(wr(wtable3b+3))
	samm AR4
	add #36
	samm AR7                          ;AR7=#(wr(wtable3b+3+36))


	lt  ccr
	mpy *, AR7
	ltp *, AR4
	mpy cci
	lts cci
	sach t4r, 2                       ;t4r=ccr*AR4-cci*AR7
	mpy *, AR7
	ltp  *, AR0
	mpy  ccr
	apac
	sach t4i, 2                       ;t4i=cci*AR4+ccr*AR7

	lact aar
	addt t3r
	addt t4r
	sach *+, AR1                      ;fr(i+1)=aar+t3r+t4r

	lact aai
	addt t3i
	addt t4i
	sach *+, AR2                      ;fi(i+1)=aai+t3i+t4i

	lacc #(wr)
	add *+, AR4
	samm AR4                          ;AR4=#(wr(wtable3b+4))
	add #36
	samm AR7                          ;AR7=#(wr(wtable3b+4+36))


	lt  bbr
	mpy *, AR7
	ltp *, AR4
	mpy bbi
	lts bbi
	sach t3r, 2                       ;t3r=bbr*AR4-bbi*AR7
	mpy *, AR7
	ltp  *, AR2
	mpy  bbr
	apac
	sach t3i, 2                       ;t3i=bbi*AR4+bbr*AR7

	lacc #(wr)
	add *+, AR4                       ;AR4=#(wr(wtable3b+4))
	samm AR4
	add #36
	samm AR7                          ;AR7=#(wr(wtable3b+4+36))

	lt  ccr
	mpy *, AR7
	ltp *, AR4
	mpy cci
	lts cci
	sach t4r, 2                       ;t4r=ccr*AR4-cci*AR7
	mpy *, AR7
	ltp  *, AR0
	mpy  ccr
	apac
	sach t4i, 2                       ;t4i=cci*AR4+ccr*AR7

	lact aar
	addt t3r
	addt t4r
	sach *+, AR1                      ;fr(i+2)=aar+t3r+t4r

	lact aai
	addt t3i
	addt t4i
	sach *+, AR0                      ;fi(i+2)=aai+t3i+t4i


end_3_b



	ret

