;**********************************************************************
;
;  Filename: CDETECT.ASM
;
;  Author: Hayden Metz
;  Date: 10/23/95
;  Last revision: 12/19/95
;
;**********************************************************************
;
;  Description: 2 carrier detect routines.  The first is used to detect
;    if a signal is present when the receiver is not in normal operating
;    mode.  The second detects when the signal is gone when the receiver
;    is in normal operating mode.  Both routines are called with the
;    same function call.  Once inside the function a test is made to
;    determine which of the two routines is appropriate.
;
;  Algorithm:
;    Test flag and jump to proper CD routine.
;
;    For the first CD routine:
;	1) call powercalc to compute the exponential average of the
;	   input power
;	2) test to see if power is above the carrier power threshold
;	3) if above threshold check to see for how long
;	4) if above threshold for a certain amount of time then put
;	   receiver into "active mode," else repeat
;
;    For second CD routine:
;	1) call powercalc to compute the exponential average of the
;	   input power
;	2) test to see if power is below the reduced power threshold
;	3) if below, check how long
;	4) if below a certain amount of time, but receiver back into
;	   "inactive mode," else continue receiving as normal
;
;**********************************************************************
;
;  Usage:
;
;  Inputs:
;
;  Outputs:
;
;  Variables used:
;
;  Registers used: accesses AR5
;
;**********************************************************************
;
;  Comments:
;	10/23 -- Created one routine to handle all cases.  Routine at
;		this point is very spotty.
;		HM
;	10/24 -- Divided program into two CD routines as described above.
;		HM
;	10/27 -- Optimized code and prepared routines to be integrated
;		with ALL.ASM
;               HM
;	11/13 -- Made minor changes needed to incorporate the carrier
;		detect routines into ALL.ASM
;               HM
;	12/19 -- Combined pre and post-signal carrier detect routines
;		into one to make the jump-table receiver easier
;		HM
;
;**********************************************************************


	.global cdetect, powercalc
	.global power			; p(n-1)	form 1:15:0
	.global c          		; 1:15:0
	.global one_minus_c		; 1:15:0
	.global high_power		; turn-on power threshold
	.global low_power		; turn-off power threshold
	.global time_over               ; time power > high_thresh
	.global time_under		; time power < low_thresh
	.global high_time		; time above h.p.t. before turn-on
	.global low_time		; time below l.p.t. before turn-off
	.global signal			; set if signal is present
	.global TEMP0

	.mmregs

	.text

;*************************************************************************
;
; Carrier detection will be accomplished using exponential averaging of
;   the received signal power.  This method is outlined on p. 175 of the
;   lab manual.  The following equation is used for the current power:
;
;	p(n) = (1-c) * r(nT0)^2 + c * p(n-1)
;
;   where r(nT0) is the current modulated input, and p(n-1) is the
;   previous average power.
;*************************************************************************

;*************************************************************************
; powercalc is the routine called to calculate the exponential average of
;   the input power for the current sample
;*************************************************************************

powercalc:
	ldpk power              	; Data page 4
	mar *, AR5
					; square current input

	lt *           			; form = 1:15:0
	mpy *
	pac				; form = 1:15:0 * 1:15:0 = 2:30:0
	sach TEMP0, 1
	lt TEMP0                	; put product in TREG0
					; calculate r^2 * (1-c)

	mpy one_minus_c         	; form = 1:0:15
	pac				; form = 1:15:0 * 1:0:15 = 2:15:15

					; calculate c * p(n-1)
	lt power	     		; form = 1:15:0
	mpy c                   	; form = 1:15:0 * 1:0:15 = 2:15:15
	apac                    	; add (1-c)*r^2 + c*p(n-1)

	sach power, 1           	; form = 1:15:0

	ret

;*************************************************************************
; NOTE:  The flag "signal" is set to 0 if there is no carrier and -1 if
;   there is a carrier.
;*************************************************************************

cdetect:
	ldpk signal
	lacc signal             	; check if receiver is currently
	bcnd running, LT        	;   running or off
					;   (signal = -1 --> receiver on)
	call powercalc
	sub high_power, 15     		; ACC = 2:15:15, high_power = 1:15:0
	bcnd over, GT           	; check if power is above high_power
	apl #0, time_over		; zero time_over if not over
	ret                             ;   high_power

over:
	lacl time_over          	; form = 1:15:0
	add #1                  	; increment time_over by 1
	sacl time_over
	sub high_time
	bcnd sig_pres, GEQ  		; turn on receiver if above
	ret                             ;   high_power long enough

sig_pres:
	opl #0ffffh, signal		; signal present, signal = -1
	ret

running:
	call powercalc
	sub low_power, 15      		; ACC = 2:15:15, low_power = 1:15:0
	bcnd under, LT          	; check if power is below low_power
	apl #0, time_under		; zero time_under if not under
	ret                             ;   low_power

under:
	lacl time_under			; form = 1:15:0
	add #1                  	; increment time_under by 1
	sacl time_under
	sub low_time
	bcnd no_signal, GEQ		; turn off if below low_power long
	ret    				;   enough

no_signal:
	apl #0, signal			; signal no longer present,
	ret				;   signal = 0

	.end