//----------------------------------------------------------------
// SCREEN.CPP
// Keith Larson
// TMS320 DSP Applications
// (c) Copyright 1995, 1996, 1997
// Texas Instruments Incorporated
//
// This is unsupported freeware code with no implied warranties or
// liabilities.  See the disclaimer document for details
//----------------------------------------------------------------
#include <string.h>    // Compiler header files
#include <conio.h>
#include <ctype.h>
#include <dos.h>
#include <stdio.h>
#include <stdlib.h>
#include <bios.h>
#include <alloc.h>
#include <math.h>
#include "dsk.h"       // Application level DSK functions and variables
#include "screen.h"    // Debugger/assembler functions and variables
#include "argsplit.h"
#include "exp_anal.h"
#define  GRAPH_EN  1
#if      GRAPH_EN
MSGS graph(ulong *DSPDATA, ulong Addr, NUM_TYPE DATATYPE);
#endif
//-----------------------------------------------
extern int  Invalidate_DASM;
//-----------------------------------------
struct CMD_BUFFER Cmd_Buf[CMDBFRS+1];
char   Cmd_Line[CMDLENG+1];
char   last_cmd[80];
char   CMSG[80];             // current command message - global
char   Dasm_Big   =0;        // Grow window flag
char   Mem_Win_Big=0;        // Grow window flag
int    CPU_mode = NT_INTEGER;
NUM_TYPE Mem_Mode = NT_INTEGER;// Memory display mode
int    Dasm_Mode = DASM3;    // DASM display mode
//int    Old_Dasm_Mode = DASM3;//
int    CmdNum=0;             // Active Command buffer
int    PCY=0;                // Global for Y cursor offset of PC match
int    SOURCE_DBG=0;         // Turn on/off DSK3A source debugging
//long   dT_Test_NOP = 0x4B; // Constant dT to subtract from each step
ulong  o[CTXTSIZE];
WINDOW *Cmd_Win;
WINDOW *Cpu_Win;
WINDOW *Mem_Win;
WINDOW *Dasm_Win;
ulong  old_mem[512];       // Array of old mem window values
ulong  new_mem[512];       // Array of new mem window values
ulong  oma;                // Old mem start address used for scrolling
//------------------------------------------------------
// strend() finds the last alphanumeric (non-ws)
//------------------------------------------------------
char *strend(char *p)
{
  p = p+strlen(p);
  return p;
}
//------------------------------------------------------
// Cmd_Msg() is used to print a message using an ATTR
// color scheme to the command line of the debugger.
//------------------------------------------------------
void Cmd_Msg(char *ptr,int ATTR,int outmsg, int endpos)
{
  int cx, cy, ch, cw,ox,oy, l;
  ox = wherex();
  oy = wherey();
  cx= Cmd_Win->x; cy= Cmd_Win->y; ch= Cmd_Win->h; cw= Cmd_Win->w;
  l = strlen(ptr);
  if(cw==34)
    if(l > 31)
    {
      Cmd_Win = GrowWindow(cx,cy,80,ch,Cmd_Win);
      CMD_Window(); // redraw the command buffer
    }
  if(cw==80)
    if(l < 31)
    {
      Cmd_Win = GrowWindow(cx,cy,34,ch,Cmd_Win);
      CMD_Window(); // redraw the command buffer
      MEM_Window(MEM_Address);
    }
  cx= Cmd_Win->x; cy= Cmd_Win->y; ch= Cmd_Win->h; cw= Cmd_Win->w;

  PutnChar (cx,cy+ch-3,cw-2,' '   ,ATTR,CARFOND);
  if(outmsg)
  PutString(cx,cy+ch-3,cw-2,ptr,      0,    OLD);
  gotoxy(cx+1+strlen(ptr),cy+ch-2);
  if(ATTR==(ERR_ATTR))
    while(!kbhit());
  if(endpos==0) gotoxy(ox,oy);
}
/*------------------------------------------------------------
   StatusLine() prints the Function key help line to the last
   text line. "#" chars turn on and off color highlighting
  ------------------------------------------------------------*/
void StatusLine( char *p1 )
{
  struct text_info ti;
  int x, y,ta=0, th;
  x = wherex();
  y = wherey();
  gettextinfo(&ti);
  th = ti.screenheight;
  gotoxy(1,th);
  textbackground(BLACK);
  printf("                                       "  //Clear status line
         "                                       ");
  gotoxy(1,th);
  for(;;)
  {
    switch(*p1)
    {
      case  0 :             break; // Nothing to printf
      case '#': ta^=1;
                if(ta) textcolor(   YELLOW);
                else   textcolor(LIGHTGRAY);
                break; // toggle text brightness
      default : putch(*p1); break;
    }
    if(*p1==0) break;
    p1++;
  }
  gotoxy(x,y);
}
//----------------------------------------------------------
// alte_key() returns 1 if the alt key is pressed, 0 if not
//
// NOTE: Some keyboards/motherboards always return _Lt_Alt=1
//       Therefor the _Lt_Alt _Rt_Alt should not be used
//----------------------------------------------------------
int alt_key(void)
{
 // if(bioskey(2) & (_Alt | _Lt_Alt | _Rt_Alt)) return 1;
  if(bioskey(2) & _Alt) return 1;
  else                  return 0;
}
int shift_key(void)
{
  if(bioskey(2) & (_Rt_Sh|_Lt_Sh)) return 1;
  else                             return 0;
}
/*
int ctrl_key(void)
{
  if(bioskey(2) & _Ctrl) return 1;
  else                   return 0;
}
*/
//----------------------------------------------------------
//#define BUF_ATTR (BLACK<<4)|BLACK       /* 0*/
//#define BUF_ATTR (BLACK<<4)|BLUE        /* 1*/
//#define BUF_ATTR (BLACK<<4)|GREEN       /* 2*/
//#define BUF_ATTR (BLACK<<4)|CYAN        /* 3*/
//#define BUF_ATTR (BLACK<<4)|RED         /* 4*/
//#define BUF_ATTR (BLACK<<4)|MAGENTA     /* 5*/
//#define BUF_ATTR (BLACK<<4)|BROWN       /* 6*/
//#define BUF_ATTR (BLACK<<4)|LIGHTGRAY   /* 7*/
//#define BUF_ATTR (BLACK<<4)|DARKGRAY    /* 8*/
//#define BUF_ATTR (BLACK<<4)|LIGHTBLUE   /* 9*/
//#define BUF_ATTR (BLACK<<4)|LIGHTGREEN  /*10*/
//#define BUF_ATTR (BLACK<<4)|LIGHTCYAN   /*11*/
#define BUF_ATTR (BLACK<<4)|LIGHTRED      /*12*/
//#define BUF_ATTR (BLACK<<4)|LIGHTMAGENTA/*13*/
#define BUF_ATTR2 (BLACK<<4)|YELLOW       /*14*/
//#define BUF_ATTR (BLACK<<4)|WHITE       /*15*/
//----------------------------------------------------------
// CMD_Window() is called to display the contents of the
// command buffer to the command buffer window.  Depending
// on the current active command buffer, the color is
// changed to a highlight value.
//----------------------------------------------------------
void CMD_Window(void)
{
  int i,l, cx, cy, ch, cw;
  int k;
  static int off  = 0;
  static int cmd1 = 0;
  static int cmd2 = 4;
  if((screenmode == C4350)&&(off==0)&&(cmd1==0)) cmd2 = 7;
 // int t;
  if(screenmode == C4350) k = 8;
  else                    k = 5;
  cx = Cmd_Win->x; cy = Cmd_Win->y;
  ch = Cmd_Win->h; cw = Cmd_Win->w;
  l  = ch-3;
  cy = cy + CMDBFRS - k;
  if(CmdNum <  cmd1) {cmd1 = CmdNum; cmd2 = cmd1 + (k-1); off = cmd1;}
  if(CmdNum >  cmd2) {cmd2 = CmdNum; cmd1 = cmd2 - (k-1); off = cmd1;}

  for(i=0; i< k; i++)   /* Display is hardcoded to 5 lines */
  {
    if(i+off==CmdNum)
    {
      PutString(cx  ,cy+i-CMDBFRS+l,cw-2,             ">",BUF_ATTR2,CARFOND);
      PutnChar (cx+1,cy+i-CMDBFRS+l,cw-3,             ' ',BUF_ATTR2,CARFOND);
      PutString(cx+1,cy+i-CMDBFRS+l,cw-3,Cmd_Buf[i+off].s,BUF_ATTR2,CARFOND);
    }
    else
    {
      PutString(cx  ,cy+i-CMDBFRS+l,cw-2,             " ",BUF_ATTR ,CARFOND);
      PutnChar (cx+1,cy+i-CMDBFRS+l,cw-3,             ' ',BUF_ATTR ,CARFOND);
      PutString(cx+1,cy+i-CMDBFRS+l,cw-3,Cmd_Buf[i+off].s,BUF_ATTR ,CARFOND);
    }
  }
}
/*--------------------------------------------------------------------
   DisasmAddress() takes the current memory contents for a particular
   address and creates the proper disassembly string.  The screen
   position, which is also input to this function is then used to
   actually print the resulting string to the display.  If a
   breakpoint is found, or the current address matches the processors
   program counter a different color is used.
  --------------------------------------------------------------------*/
void DisasmAddress( int x, int y, ulong address, ulong instr )
{
  char buf[80],attr;
  char addr[9], inst[9], label[13];
  *label = 0;
  if ( Valid_Address(address,address) == OUT_MAP )
  { strcpy(buf,"xx Invalid Address xx");
    sprintf(addr,"%06lx",address);
    sprintf(inst,"");
    attr = ERR_ATTR;
  }
  else
  { Disasm( address, instr, &buf[0]);
    sprintf(addr,"%06lx",address);
    sprintf(inst,"%08lx",instr);
    attr=Dasm_Win->tattr;
    get_label(label,address,0); // label @address and 0 leading WS
    for(int i=0; i<Nb_Brk_Pt; i++ )
      if ( Brk_Tab[i].address == address ) attr=ERR_ATTR;
    if ( address == PC_Appli )
    { attr=PC_ATTR;
      PCY = y;
    }
  }
  PutnChar (x   ,y,Dasm_Win->w- 2,  ' ',attr,CARFOND); // Fill with ' '
  switch(Dasm_Mode)
  {
    default:      //  0 2 4 6 8 + 2 4 6 8 + 2 4
    case DASM3:   //  809800 12345678 LABEL    NOP  (default startup)
    PutString(x   ,y,             6, addr,CYAN,    CAR);
    PutString(x+ 7,y,             8, inst,attr,    CAR);
    PutString(x+16,y,             8,label,attr,    CAR);
    PutString(x+25,y,Dasm_Win->w-27,  buf,attr,    CAR);
    break;                    //  ^-- Width - (Start + 2 border chars)
                              //
                  //  0 2 4 6 8 + 2 4 6 8
    case DASM2:   //  12345678 LABEL    NOP
    PutString(x   ,y,             8, inst,attr,    CAR);
    PutString(x+9 ,y,             8,label,attr,    CAR);
    PutString(x+18,y,Dasm_Win->w-20,  buf,attr,    CAR);
    break;
                  //  0 2 4 6 8 + 2 4 6
    case DASM1:   //  809800 LABEL    NOP
    PutString(x   ,y,             6, addr,CYAN,    CAR);
    PutString(x+ 7,y,             8,label,attr,    CAR);
    PutString(x+16,y,Dasm_Win->w-18,  buf,attr,    CAR);
    break;
                  //  0 2 4 6 8 + 2
    case DASM0:   //  LABEL    NOP
    PutString(x   ,y,             8,label,attr,    CAR);
    PutString(x+ 9,y,Dasm_Win->w-11,  buf,attr,    CAR);
    break;
  }
                            //  ^-- Width - Start + 2 border chars
}
//-------------------------------------------------------------------
// DisasmAddress2() is a variant of DisasmAddress() which is called
// when DSK assembly source debugging is enabled.
//-------------------------------------------------------------------
void DisasmAddress2( int x, int y, ulong address, char *buf, int realcode)
{
  char attr,*p;
  attr=Dasm_Win->tattr;
  if   ((p=strstr(buf,"\n"))!=NULL) *p=  0;
  if   ((p=strstr(buf,"\r"))!=NULL) *p=  0;
  //-----------------------------------
  // Strip out the tabs
  //-----------------------------------
  tabstrip(buf+22,8); // Strip at end of prepended hex fields +22
  //-----------------------------------
  p = buf;
  if(*p=='0') // A Line which is either directive or loadable code
  { p+=4;
    strncpy(p+7,p+18,79);
  }
  if(realcode)
  {
    for(int i=0; i<Nb_Brk_Pt; i++ )
      if ( Brk_Tab[i].address == address ) attr=ERR_ATTR;
    if ( address == PC_Appli )
    {
      attr=PC_ATTR;
      PCY = y;
    }
  }
  PutnChar (x   ,y,Dasm_Win->w- 2,32,attr,CARFOND);
  PutString(x   ,y,Dasm_Win->w- 2, p,attr,    CAR);
}
//----------------------------------------------------------
// DASM_Window() begins disassembling code at address
// calling the appropriate screen draw functions.
//----------------------------------------------------------
MSGS DispCOFF(void);

MSGS DASM_Window( ulong address )
{
  FILE *lfile;
  ulong DASM_data[50];    // 50 lines maximum
  fpos_t last_found; //first_found;
  char strg[200],*eptr;
  int i, xx, yy,SDBG=0;
  int DASM_y = 0;
//int PCY; // New
  ulong *ptr, length;
  ulong STRB0, STRB1, LTMP;
  MSGS err;
  length = Dasm_Win->h-2;
  DASM_Address = address;
  ptr =DASM_data;
  //--------------------------------------------------------------
  // DSK Assembler file tracking is done by matching the loadable
  // address to the current program counter (PC) of the DSP.  If
  // no match is found, the disassembler is used
  //--------------------------------------------------------------
  i=0;
  DASM_y = 0;
  PCY = 0;
  if(SOURCE_DBG)
  {
    if(Dasm_Mode == DASM4)
    {
      err= DispCOFF();
      if(err==NO_ERR) return err;

      if(    ( DASM_Address               >=CTXT[PC] )
          || ((DASM_Address+Dasm_Win->h-2)<=CTXT[PC]))
      {
        address = CTXT[PC]-3;DASM_Address=address;
      }
    }
    else
    {
      SDBG       = 0;
      last_found = 0;
      //
      // If the new PC is outside of the last used
      // source debug boundaries, change the starting
      // point of the window to best fit the new PC
      //
      if((lfile = fopen(DASM_file,"rt")) != NULL)
      {
        for(xx=0;xx<32767;xx++)         // Find where PC matches
        {
          fgetpos(lfile,&last_found);
          if(fgets(strg,199, lfile) == NULL)  break;
          if(*strg=='0')  // Loadables start with '0x'
            if(strtol(strg,&eptr,0)==address)
            {
              SDBG=1;
              if(*(eptr+1)=='0') break; // value at address must be numeric
            }
        }
        // Go back to address matchup for display
        if(SDBG)
        {
          fsetpos(lfile,&last_found);
          for (/* i=0*/; i<Dasm_Win->h-2; i++ )
          { xx = Dasm_Win->x;
            yy = Dasm_Win->y+i;
            if(fgets(strg,199, lfile) == NULL) break; // No more file to read
            if(*(strg+11)=='0')  // If loadable increment address
            {
              if(strtol(strg,&eptr,0)==address)
                DisasmAddress2(xx, yy, address,strg,1);
              else
                DisasmAddress2(xx, yy, address,strg,0);
              address++;
              DASM_y++;
            }
            else
              DisasmAddress2(xx, yy, address,strg,0);
            DASM_Ad_End++;
          }
          fclose(lfile);
          DASM_Ad_End = address-1;
          return NO_ERR;
        }
      }
    }
  }
  fclose(lfile);
  //  if(Valid_Address(address,address+length)==OUT_MAP) return OUT_MAP;
  //
  // If the target being debugged is a C32, the EMIF needs to be
  // configured for 32 bit transfers.  The only catch is that the
  // HPI address cannot be changed since this would effectively
  // corrupt the kernels byte transfer counter
  //
  if(TARGET == C32_DSK)
  {
    // IOSTRB is always 32 bit data access from a 32 bit wide bus;
    if((err = getmem(0x808064L,1,&STRB0 )) != NO_ERR) return err;
    if((err = getmem(0x808068L,1,&STRB1 )) != NO_ERR) return err;
    // Set to STRB0 & STRB1 for 32 bit data (bus width=same)
    LTMP=STRB0|0x30000L; if((err=putmem(0x808064L,1,&LTMP))!=NO_ERR)
                            return err;
    LTMP=STRB1|0x30000L; if((err=putmem(0x808068L,1,&LTMP))!=NO_ERR)
                            return err;
  }
  if((err = getmem(address,length, DASM_data)) != NO_ERR) return err;
//  if((err = getmem(DASM_Address,length, DASM_data)) != NO_ERR) return err;
  for ( i=0; i<Dasm_Win->h-2; i++ )
  {
    xx = Dasm_Win->x;
    yy = Dasm_Win->y+i;
    DisasmAddress(xx, yy, address++,*ptr++);
    DASM_y++;
  }
  //
  // Restore the C32 buses to their original data configuration
  //
  if(TARGET==C32_DSK)
  {
    if((err = putmem(0x808064L,1,&STRB0)) != NO_ERR) return err;
    if((err = putmem(0x808068L,1,&STRB1)) != NO_ERR) return err;
  }
  DASM_Ad_End = address-1;
  return err;
}
//----------------------------------------------------------
// Gets the memory from the DSK starting at address and
// displays it to the screen, highlighting any values that
// change realative to the 'old' memory buffer.
//----------------------------------------------------------
double QMPY = 1.0;

void sprintf2(char *c, long val)
{
  int x;
  *c++=' ';
  *c++=' ';
  *c++=' ';
//*c++=' ';
  for(x=0;x<32;x++)
  {
    if(val < 0) *c++ = '1';
    else        *c++ = '0';
    val = val << 1;
  }
  *c++='b';
  *c=0;
}

MSGS MEM_Window( ulong address )
{
  #define Tatt Mem_Win->tattr
  int Xpos, Ypos, i, j, WX, WY, WXY, ii, n, offs, hilite, Xsz;
  char buf[64];
  MSGS err;
  ulong addr;   // 50*4 max entries fill mem box

  if(Mem_Mode == NT_QFORM)
    QMPY = pow(2.0,-QVAL);

  switch(Mem_Mode)
  {
    case NT_BASE2:   Xsz = 38; break;
//  case NT_BASE8:   Xsz = 36; break;
    case NT_SLONG:
    case NT_ULONG:
    case NT_QFORM:
    case NT_FLOAT:   Xsz = 18; break;
    default:
    case NT_INTEGER: Xsz =  9; break;
  }

  WX = (Mem_Win->w-8)/Xsz;// Horz cells
  WY = (Mem_Win->h  )-2;  // Vert cells
  WXY = WX * WY;          // Total cells
  // get memory to be displayed  <- NOTE: No checking so far
  if((err = getmem(address,WXY, new_mem)) != NO_ERR) return err;
  addr = address;
  Ypos = Mem_Win->y;
  //
  // If extent of window has changed, realign data in old_mem
  if(oma != addr)
  { if(oma > addr)   // shift old_mem up
    { offs = (int)(oma - addr);
      for(n=0;n<WXY;n++)
      { if(n < offs) old_mem[n] = new_mem[n]-1; // Lead highlighted
        else         old_mem[n] = new_mem[n];
      }
    }
    else
    { offs = (int)(addr - oma);
      for(n=0;n<WXY;n++)
      { if(n >= (WXY-offs)) old_mem[n] = new_mem[n]-1; //Tail highlighted
        else                old_mem[n] = new_mem[n];
      }
    }

  }
  ii = 0;
  for ( j=0; j<WY ; j++ )
  {
    Xpos = Mem_Win->x;
    sprintf(buf,"%06lx",addr);
    PutString(Xpos,Ypos+j,8,buf,CYAN,CAR);
    Xpos += 8; // make room for address field

    for ( i=0; i< WX; i++)
    {
      if( Valid_Address(addr, addr) == NO_ERR )
      {
        if(new_mem[ii]^old_mem[ii]) hilite = WIN_ATTR | 0x8;
        else                    hilite = WIN_ATTR;
        switch(Mem_Mode)
        {
         default     :
         case NT_INTEGER:sprintf(buf," %08lx"   ,new_mem[ii]);      break;
         case NT_SLONG:sprintf (buf," %17ld"   ,new_mem[ii]);      break;
         case NT_ULONG:sprintf (buf," %17lu"   ,new_mem[ii]);      break;
         case NT_FLOAT:sprintf (buf," %+16.10e",TMS_IEEE(new_mem[ii]));break;
         case NT_QFORM:sprintf (buf," %+16.10e",QMPY*(long)new_mem[ii]);break;
         case NT_BASE2:sprintf2(buf,new_mem[ii]);        break;
        }
      }
      else
      {
        hilite = ERR_ATTR;
        if(Mem_Mode==NT_BASE2)
        sprintf(buf,"   xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx ");
        else              sprintf(buf," xxxxxxxxxxxxxxxx ");
      }
      PutString(Xpos,Ypos+j,Xsz,buf,hilite,CARFOND);
      Xpos+=Xsz;
    //if(Mem_Mode==FLOAT) i++;
      addr++;
      ii++;
    }
  }
  MEM_Address = address;
  MEM_Ad_End  = addr   ;
  oma= address;
  for ( i=0; i<WXY; i++ ) old_mem[i] = new_mem[i];  // Keep copy of memory
  return err;
}
/*
//----------------------------------------------------------------------
// By testing the timing of a known opcode in the kernel (INT0 branch)
// the correct difference for any memory space can be calculated.
//----------------------------------------------------------------------
MSGS SSTEP_NOP(void)
{
  MSGS err;
  int temp;
  ulong temp1,temp2,temp3;
  static int SSTEP_NOP_DONE = 0;     // Do only one time
  if(SSTEP_NOP_DONE) return NO_ERR;  //
  SSTEP_NOP_DONE = 1;                //
  if(TARGET==C30_EVM)
  {
    dT_Test_NOP=0x100; return NO_ERR;
  }
  if((err=getmem(DEBUG_CTXT+   PC,1,&temp1))!=NO_ERR) return err;
  if((err=getmem(DEBUG_CTXT+T1DIF,1,&temp2))!=NO_ERR) return err;
  temp3 = STEPLOC; // Safe place to singlestep (no reg modification)
  if((err=putmem(DEBUG_CTXT+   PC,1,&temp3))!=NO_ERR) return err;
  temp = swap_en;
  swap_en = 1;
  if((err=SSTEP_CPU())!=NO_ERR) {swap_en = temp; return err;}
  swap_en = temp;

  if((err=getmem(DEBUG_CTXT+T1DIF,1,&(ulong)dT_Test_NOP))!=NO_ERR)
    return err;
  dT_Test_NOP -= 4;  // if code stepped is a 4 cycle instruction
//dT_Test_NOP -= 1;  // if code stepped is a 1 cycle instruction

  if((err=putmem(DEBUG_CTXT+   PC,1,&temp1))!=NO_ERR) return err;
  if((err=putmem(DEBUG_CTXT+T1DIF,1,&temp2))!=NO_ERR) return err;

  ref_mod("PC",temp1);
  return NO_ERR;
}
*/
//----------------------------------------------------------------------
// Print_Reg prints the 32 bit hexadecimal registers to the display
//----------------------------------------------------------------------
/* organized in order of array storage */
static char *Win_Regsx[] =
{
"F0" ,"F1" ,"F2" ,"F3" ,"F4","F5","F6","F7",
"R0" ,"R1" ,"R2" ,"R3" ,"R4","R5","R6","R7",
"AR0","AR1","AR2","AR3","AR4","AR5","AR6","AR7",
"DP" ,"IR0","IR1","BK" ,"SP" ,"ST" ,"IE" ,"IF" ,
"IOF","RS" ,"RE" ,"RC" ,"PC" ,"FREERUN","_dT" ,"CTXT"
};

void Print_Rx(int Rx,int col, int row)
{
  #define PZ 8  /* 12 for C4x */
  char attr;
  char buf[20];
  int FREG;
  attr=Cpu_Win->tattr & 0xF7;
  switch(Rx)
  {
    case R0F: case R1F:
    case R2F: case R3F:
    case R4F: case R5F:
    case R6F: case R7F: FREG = 1; break;
    default :           FREG = 0; break;
  }
  if(o[Rx  ] !=CTXT[Rx   ])  attr=Cpu_Win->tattr | 0x08;
  if(FREG)
  { if(o[Rx+PZ]!=CTXT[Rx+PZ])  attr=Cpu_Win->tattr | 0x08;}
  PutString(Cpu_Win->x+col,Cpu_Win->y+row,3,Win_Regsx[Rx],CYAN,CAR);
  if(FREG)
  { switch(CPU_mode)
    { case 2: sprintf(buf,"%+10.4e",TMS_IEEE(CTXT[Rx])); break;
      default:
      case 0:
      case 1: sprintf(buf," %02lx%08lx",(CTXT[Rx]>>24) & 0xFFL,CTXT[Rx+PZ]);
              break;
    }
    PutString(Cpu_Win->x+col+2,Cpu_Win->y+row,11,buf,attr,CAR);
  }
  else
  { sprintf(buf,"%08lx",CTXT[Rx]);
    PutString(Cpu_Win->x+col+5,Cpu_Win->y+row,11,buf,attr,CAR);
  }
}
//-----------------------------------------------------------------
// CPU_Window() displays the CPU registers in either
//   - 32 bit hexadecimal
//   - Mixed 32/40 bit hexadecimal
//   - Mixed 32 hexadecimal/floating point
//-----------------------------------------------------------------
MSGS CPU_Window( void )
{
  int i, col, row;
  ulong temp;
  MSGS err;
  if ((err = getmem(CTXT_PTR,CTXTSIZE,&CTXT[0])) != NO_ERR) return err;
  //----------------------------------------------------------
  // This patch is to ensure that the float regsiters loaded
  // from the DSK have the correct overlapping bits for the
  // 40 bit extended registers.  No reload is required
  //----------------------------------------------------------
  CTXT[R0F] = (CTXT[R0F]&0xFF000000L) | (CTXT[R0]>>8);
  CTXT[R1F] = (CTXT[R1F]&0xFF000000L) | (CTXT[R1]>>8);
  CTXT[R2F] = (CTXT[R2F]&0xFF000000L) | (CTXT[R2]>>8);
  CTXT[R3F] = (CTXT[R3F]&0xFF000000L) | (CTXT[R3]>>8);
  CTXT[R4F] = (CTXT[R4F]&0xFF000000L) | (CTXT[R4]>>8);
  CTXT[R5F] = (CTXT[R5F]&0xFF000000L) | (CTXT[R5]>>8);
  CTXT[R6F] = (CTXT[R6F]&0xFF000000L) | (CTXT[R6]>>8);
  CTXT[R7F] = (CTXT[R7F]&0xFF000000L) | (CTXT[R7]>>8);
  //-----------------------------------------------------------
  CURRENT_DP = CTXT[DP];
  if(Dasm_Big) return NO_ERR;
  //  _dT : TIMER DIFFERENCE CALCULATION
  //  ----------------------------------
  //  - _dT is calculated from the difference of two TIM1 readings.
  //    To work TPRD1 must be set to a large value.
  //
  //  - If timer rolls through zero, the value is adjusted by
  //    the period value to get a positive value.  If the value is
  //    still negative the timer period was too short for the code
  //    being benchmarked
  //
  //  - The value subtracted from the timer difference is calculated
  //    from singlestepping a test Branch in the kernel (the vector
  //    branch table).  This should allow kernel and SP relocation
  //
//if((err=SSTEP_NOP())!=NO_ERR)  return err;
  CTXT[T1DIF] = CTXT[T1DIF] - dT_Test_NOP;
  if((long)CTXT[T1DIF] < 0)
  {
    getmem(0x808038L,1,&temp);  // Get timer period directly
    CTXT[T1DIF] += (2*temp);    // This is address OK for all C3x devices
  }
  //
  // Dont display if DASM is fully covered (wide mode)
  if(screenmode==C80)
  {
  row = 0;
  col = 0;
  Print_Rx(PC ,col,row++);
  Print_Rx(R0F,col,row++);
  Print_Rx(R2F,col,row++);
  Print_Rx(R4F,col,row++);
  Print_Rx(R6F,col,row++);
  Print_Rx(AR0,col,row++);
  Print_Rx(AR2,col,row++);
  Print_Rx(AR4,col,row++);
  Print_Rx(AR6,col,row++);
  Print_Rx(IR0,col,row++);
  Print_Rx(ST ,col,row++);
  Print_Rx(RS ,col,row++);
  Print_Rx(DP ,col,row++);
  Print_Rx(IE ,col,row++);
  Print_Rx(IOF,col,row++);
  row = 0;
  col = 14;
  Print_Rx(SP  ,col,row++);
  Print_Rx(R1F ,col,row++);
  Print_Rx(R3F ,col,row++);
  Print_Rx(R5F ,col,row++);
  Print_Rx(R7F ,col,row++);
  Print_Rx(AR1 ,col,row++);
  Print_Rx(AR3 ,col,row++);
  Print_Rx(AR5 ,col,row++);
  Print_Rx(AR7 ,col,row++);
  Print_Rx(IR1 ,col,row++);
  Print_Rx(RC  ,col,row++);
  Print_Rx(RE  ,col,row++);
  Print_Rx(BK  ,col,row++);
  Print_Rx(IF  ,col,row++);
  Print_Rx(T1DIF,col,row++);
  }
  else
  {
  row = 0;
  col = 0;
  Print_Rx(PC ,col,row++);
  Print_Rx(SP ,col,row++);
  Print_Rx(R0F,col,row++);
  Print_Rx(R1F,col,row++);
  Print_Rx(R2F,col,row++);
  Print_Rx(R3F,col,row++);
  Print_Rx(R4F,col,row++);
  Print_Rx(R5F,col,row++);
  Print_Rx(R6F,col,row++);
  Print_Rx(R7F,col,row++);
  Print_Rx(AR0,col,row++);
  Print_Rx(AR1,col,row++);
  Print_Rx(AR2,col,row++);
  Print_Rx(AR3,col,row++);
  Print_Rx(AR4,col,row++);
  Print_Rx(AR5,col,row++);
  Print_Rx(AR6,col,row++);
  Print_Rx(AR7,col,row++);
  Print_Rx(IR0,col,row++);
  Print_Rx(IR1,col,row++);
  Print_Rx(ST ,col,row++);
  Print_Rx(RC ,col,row++);
  Print_Rx(RS ,col,row++);
  Print_Rx(RE ,col,row++);
  Print_Rx(BK ,col,row++);
  Print_Rx(IOF,col,row++);
  Print_Rx(DP ,col,row++);
  Print_Rx(IE ,col,row++);
  Print_Rx(IF ,col,row++);
  Print_Rx(T1DIF,col,row++);
  }

  //
  // copy the present register values to the 'old' buffer
  for ( i = R0F; i<=T1DIF; i++ ) o[i] = CTXT[i];
  return err;
}

//------------------------------------------------------
// This function performs a sanity check to see if the
// communications link is working. If not, parts of the
// screen would go blank with possibly no warning.
//------------------------------------------------------
MSGS Sanity_Check(void)
{
  MSGS err;
  ulong temp;
//if((err=getmem(0x0,1,&temp))!=NO_ERR)
//  return err;
//if(temp!=0x45) return COM_ERR;
  temp = CTXT_PTR;
  if((err=GET_DEBUG_CTXT())!=NO_ERR)
  { CTXT_PTR = temp;
    return err;
  }
  if(CTXT_PTR != temp)
  { CTXT_PTR = temp;
    return COM_ERR;
  }
  return NO_ERR;
}
//--------------------------------------------------------
// Init_Screen() is called to completely refresh each
// window along with the displayed data. The active command
// window is set the CMD_Window
//--------------------------------------------------------
MSGS Init_Screen(void)
{
  MSGS err;
  if((err=Sanity_Check())!=NO_ERR) return err;
  clrscr();
  DisableWindow(Dasm_Win);  // Draw the window frames
  DisableWindow( Cpu_Win);
  DisableWindow( Mem_Win);
  EnableWindow ( Cmd_Win);
  if((err= CPU_Window(            ))!=NO_ERR) return err;// next fill data
  if((err=DASM_Window(DASM_Address))!=NO_ERR) return err;
  if((err= MEM_Window( MEM_Address))!=NO_ERR) return err;
  CMD_Window();
  return NO_ERR;
}
//--------------------------------------------------------
// Update_Screen() is called to refresh the displayed data.
// Unlike Init_Screen the active command window is not set
// back to the CMD_Window
//--------------------------------------------------------
MSGS Update_Screen(ulong addr)// If PC_Cursor is not within
{                                // DASM bounds use it for DASM_Window
  MSGS err;

  //---------------------------------------------------------
  // Perform a sanity check to make sure the communications
  // link is working, instead of having the screen go blank
  //---------------------------------------------------------
  if((err=Sanity_Check())!=NO_ERR) return err;
  if((err = CPU_Window(           ))!=NO_ERR) return err;
  if((err = MEM_Window(MEM_Address))!=NO_ERR) return err;
  PC_Appli = CTXT[PC];
  if((err = DASM_Window(addr))!=NO_ERR) return err;
  return NO_ERR;
}
//-------------------------------------------------------
// DASM_Edit() is called from the command window.  When
// called the disassembly window becomes the active window
// highlighting the border.  Further keystrokes are
// processed here until a quit (escape) occurs, returning
// to the command window.  The parameter int *old passed
// into this function is used as an alternate messaging
// path for the memory, cpu and dasm window to control
// which window should have the next current context when
// the alt key is pressed.
//-------------------------------------------------------
extern int HLL_1;
static MSGS DASM_Edit( int *old )
{
  MSGS err = NO_ERR;
  long CURSOR_ADDR;
  char p[20], *eptr, *p1, *p2;
  int c, dx, dy,x,yc=0;
  *old = 0;
  _setcursortype(_NORMALCURSOR);
  EnableWindow(Dasm_Win);
  dx = Dasm_Win->x + 1; // x text Position
  dy = Dasm_Win->h - 3; // Height in visible address'
  if(kbhit())bioskey(0); // Clear keyboard
  PCY=0;
  c= 0;
  if((err = Update_Screen(DASM_Address))!=NO_ERR) return err;
  yc = PCY-1;
  do
  {
     if(Dasm_Mode==DASM4)
     StatusLine("#F1#Help #F3#DASM3                 #F5#Run "
                "#F6#DispBP #F7#ClrAll #F8#Step #F9#Grow #F10#FStep");
     else
     StatusLine("#F1#Help #F2#BPtgl #F3#BPclr #F4#RunTo #F5#Run "
                "#F6#DispBP #F7#ClrAll #F8#Step #F9#Grow #F10#FStep");
     _setcursortype(_NORMALCURSOR);
     if((c==_F8)||(c==_F10))
     yc = PCY-1;
     gotoxy(dx,yc+2);
     while((c=bioskey(1)&0xff00)==0);// While waiting for a key, allow
     if(getch()==0) getch();      // Get ASCII, clear kb, remove dbl keyhit
     _setcursortype(_NOCURSOR);
     err = NO_ERR;
     /*----------------------------------------------------
       Depending on DASM display mode (source or dasm)
       determine an address for the cursor
     -----------------------------------------------------*/
     if(SOURCE_DBG) /* If source debug */
     {
       // By using the address from the display the correct address
       // is assured when using assembly source debug
       gettext(wherex(),wherey(),wherex()+6,wherey(),&p);
       p1 = p; p2 = p;
       for(x=0;x<6;x++) { *p1=*p2; p1+=1; p2+=2; }
       *p1=0;
       if(strtol(p,&eptr,16)) CURSOR_ADDR = strtol(p,&eptr,16);
     }
     else       /* Else, solve direct, 1 address per line */
     {
       CURSOR_ADDR = DASM_Address + yc;
     }
     /*---------------------------------------------------*/
     switch ( c )
     {
       case _Hm      : yc  =  0; break;
       case _End     : yc  = dy; break;
       case _Dn      : yc +=  1; break;
       case _Up      : yc -=  1; break;
       case _Pdn     : yc += dy; break;
       case _Pup     : yc -= dy; break;
       //--------------------------------------
       case _F1      : err = Dasm_Help();
                       err = Ext_Help(err);
                       break; // Help
       case _F2      : if(Dasm_Mode==DASM4) break;
                       if(BP_SET(CURSOR_ADDR)) err=CB_Cmd(CURSOR_ADDR);
                       else                    err=SB_Cmd(CURSOR_ADDR);
                       break;
       case _F3      : if(Dasm_Mode==DASM4) {Dasm_Mode=DASM3; break;}
                       err = CB_Cmd(CURSOR_ADDR); break; // Clr BP
       case _F4      : Invalidate_DASM = 1;
                       if(Dasm_Mode==DASM4) {Dasm_Mode=DASM3; break;}
                       if(BP_SET(CURSOR_ADDR))
                       { if((err=RUN_Cmd()) != NO_ERR) break;}
                       else
                       {
                         if((err=SB_Cmd(CURSOR_ADDR)) != NO_ERR) break;
                         if((err=RUN_Cmd())           != NO_ERR) break;
                         if((err=CB_Cmd(CURSOR_ADDR)) != NO_ERR) break;
                       }
		       CURSOR_ADDR = PC_Appli;
                       break;
       case _F5:       err = RUN_Cmd();
                       break;
       case _F6      : err = DB_Cmd() ; break;         // Disp BP
       case _F7      : err = CB_Cmd(0) ; break;        // Clr All BP
       case _F8      : err = SStep_Cmd(1);
                       if(PCY) gotoxy(dx,PCY+1);
                       break;
       case _F10     : err = FSTEP_Cmd(1);
                       if(PCY) gotoxy(dx,PCY+1);
                       break;
       case _SF2     : DSK3D_Stats(); break;
       case _F9      : Dasm_Big ^=1;
                       if(Dasm_Big & 1)
                       { Dasm_Win=GrowWindow(1,1,80,Dasm_Win->h,Dasm_Win);
                       }
                       else
                       {
                         if(screenmode==C80)
                         Dasm_Win=GrowWindow(1,1,51,Dasm_Win->h,Dasm_Win);
                         else
                         Dasm_Win=GrowWindow(1,1,64,Dasm_Win->h,Dasm_Win);
                         if((err=CPU_Window())!=NO_ERR) break;
                       }
                       if((err=DASM_Window(DASM_Address))!=NO_ERR) break;
                       err = MEM_Window(MEM_Address);
                       break;
       //--------------------------------------
       case _M       : if(alt_key()) *old = _M; c= _ESC; break;
       case _C       : if(alt_key()) *old = _C; c= _ESC; break;
       case _Enter   :
       case _ESC     :
       default       : c = _ESC; break;
     }
     if((c==_F10)||(c==_F5))
     {
       PC_Appli = CTXT[PC];
       if(PC_Appli<DASM_Address) DASM_Address -= (DASM_Address - PC_Appli);
       if(PC_Appli>DASM_Ad_End ) DASM_Address -= (DASM_Ad_End  - PC_Appli);
     }
     if(((c!=_F8) && (err==NO_ERR)) || (c==_F4))
     {
       if(yc<  0) {DASM_Address+=yc     ; HLL_1+=yc     ; yc= 0;}
       if(yc>=dy) {DASM_Address-=(dy-yc); HLL_1-=(dy-yc); yc=dy;}
       err = Update_Screen(DASM_Address);
     }
     Cmd_Msg("",CMD_ATTR,1,1);
  } while((c != _ESC) && (err==NO_ERR));
  DisableWindow(Dasm_Win);
  DisableWindow(Mem_Win);
  return err;
}
//-------------------------------------------------------
// return 1 (true) if extended bioskey() is alpha or
// numeric, zero if not
//-------------------------------------------------------
int alphanumkey(int key)
{
  switch(key)
  {
    case _0:
    case _1:
    case _2:
    case _3:
    case _4:
    case _5:
    case _6:
    case _7:
    case _8:
    case _9:
    case _A:
    case _B:
    case _C:
    case _D:
    case _E:
    case _F: return(1);
    default: return(0);
  }
}
//-------------------------------------------------------
// Same as above except the + - . characters are now valid
//-------------------------------------------------------
int extalphanumkey(int key, char c)
{
  switch(key)
  {
    case _Spc: return 1;
    case _pls: if(c!='+') return 0; else return 1;
    case _Del: if(c!='+') return 0; else return 1;
    case _mns: if(c!='-') return 0; else return 1;
    case _prd: if(c!='.') return 0; else return 1;
    case _Hyp: if(c!='-') return 0; else return 1;
    case _Eql: if(c!='+') return 0; else return 1;
    case _0:
    case _1:
    case _2:
    case _3:
    case _4:
    case _5:
    case _6:
    case _7:
    case _8:
    case _9:
    case _A:
    case _B:
    case _C:
    case _D:
    case _E:
    case _F: return(1);
    default: return(0);
  }
}

//-------------------------------------------------------
// MEM_Edit() is called from the command window.  When
// called the memory window becomes the active window
// highlighting the border.  Further keystrokes are
// processed here until a quit (escape) occurs, returning
// to the command window.  The parameter int *old passed
// into this function is used as an alternate messaging
// path for the memory, cpu and dasm window to control
// which window should have the next current context when
// the alt key is pressed.
//-------------------------------------------------------


MSGS MEM_Edit( int *old )
{
  static ulong PC_Cursor= DASMBGN; //0x809800L; // static for last used cell retention
  static WINDOW tmp={0,0,0,25,"",0,"",0,""};
  static int XC=0;
  static NUM_TYPE OldMM = NT_SLONG;
  static ulong OldMA = DASMBGN; // 0x809800L;
  struct text_info ti;
  int th;
  int i;
  MSGS err=NO_ERR;
  int Win_Update = 1;
  int c, x, g, Xsz, cx, cy;
  int XCELL=0;
  char p[80];
  char *p1,*p2, *eptr, attr;
  float f=0;
  double db=0;
  ulong CURSOR_ADDR;
  ulong CURSOR_VALU;
  long a,b,d;
  *old= 0;
  _setcursortype(_NOCURSOR);
  EnableWindow(Mem_Win);
  /* Correct display problems if mode changes */
  if((Mem_Mode!=OldMM)||(MEM_Address!=OldMA))
  {
    PC_Cursor = MEM_Address; XC=0;
  }
  OldMM = Mem_Mode;
  OldMA = MEM_Address;
  /* Draw the window to the screen */
  MEM_Window(MEM_Address);
  if(kbhit())bioskey(0); // Clear keyboard

  do
  {
#if GRAPH_EN
     StatusLine("#F1#Help #F2#Addr expr #F3#Data expr #F9#Grow    #G#raph");
#else
     StatusLine("#F1#Help #F2#Addr expr #F3#Data expr #F9#Grow");
#endif
     switch(Mem_Mode)
     {
       case NT_BASE2  : Xsz = 34; break;
       case NT_SLONG :
       case NT_ULONG  :
       case NT_QFORM  :
       case NT_FLOAT  : Xsz = 18; break;
       default:
       case NT_INTEGER: Xsz =  9; break;
     }

     a = Mem_Win->h-2;               // Height in cells
     d = ((Mem_Win->w-8)/Xsz) * a;   // Total cells
     b = 2*d - MEM_Address*a;
     MEM_Ad_End = MEM_Address + d;
     _setcursortype(_NORMALCURSOR);

     gotoxy(Mem_Win->x+1,Mem_Win->y+(int)((a*PC_Cursor+b)/d)-1);
     gettext(wherex(),wherey(),wherex()+6,wherey(),&p);
     p1 = p; p2 = p;
     for(x=0;x<6;x++) { *p1=*p2; p1+=1; p2+=2; }
     *p1=0;
     CURSOR_ADDR = strtoul(p,&eptr,16);
     CURSOR_ADDR += XC;
     // highlight active memory cell

     cx = Mem_Win->x+10+XC*Xsz;
     if(Mem_Mode==NT_BASE2) cx+=2;
     cy = Mem_Win->y+(int)((a*PC_Cursor+b)/d)-1;
     gotoxy(cx,cy);
//   gotoxy(Mem_Win->x+10+XC*Xsz      ,Mem_Win->y+(int)((a*PC_Cursor+b)/d)-1);
//   gotoxy(Mem_Win->x+10+XC*Xsz+XCELL,Mem_Win->y+(int)((a*PC_Cursor+b)/d)-1);
     //gettextinfo(ti);

     switch(Mem_Mode)
     {
       case NT_BASE2: gettext(wherex(),wherey(),wherex()+32,wherey(),&p);
                   attr = p[1]; // Get old attr
                   textattr((BLUE<<4)|attr);  // Add blue background
                   p1=p; p2=p;
                   for(x=0;x<32;x++)
                   { *p1=*p2; p1+=1; p2+=2; }
                   *p1=0;
                   break;
       case NT_FLOAT:
       case NT_SLONG:
       case NT_QFORM:
       case NT_ULONG: gettext(wherex(),wherey(),wherex()+16,wherey(),&p);
                   attr = p[1]; // Get old attr
                   textattr((BLUE<<4)|attr);  // Add blue background
                   p1=p; p2=p;
                   for(x=0;x<17;x++)          // 1 extra cell
                   { *p1=*p2; p1+=1; p2+=2; }
                   *p1=0;
                   break;
       default:    gettext(wherex(),wherey(),wherex()+8,wherey(),&p);
                   attr = p[1]; // Get old attr
                   textattr((BLUE<<4)|attr);  // Add blue background
                   p1=p; p2=p;
                   for(x=0;x<8;x++)
                   { *p1=*p2; p1+=1; p2+=2; }
                   *p1=0;
                   break;
     }

     cputs(p);
     textattr(attr);
     gotoxy(cx+XCELL,cy);
     //
     // Get keystroke
     //
     while((c=bioskey(1)&0xff00)==0)// While waiting for a key, allow
       Release_TSlice();
     if((g=toupper(getch()))==0)
       g=toupper(getch());  // Get ASCII, clear kb, remove dbl keyhit
     _setcursortype(_NOCURSOR);
     err = NO_ERR;
     Win_Update = 1;
     switch( c )
     {
       case _Pup : PC_Cursor = 2 *MEM_Address-MEM_Ad_End; break;
       case _Pdn : PC_Cursor = MEM_Ad_End;
                   MEM_Address=MEM_Ad_End; break;

       case _Lt  : //XC--; XCELL=0;break;
                   XCELL--;
                   Win_Update=0;
                   if(XCELL<0) {XCELL=0;XC--;Win_Update=1;}
                   break;
       case _Rt  : //XC++; XCELL=0; break;
                   XCELL++;
                   Win_Update=0;
                   switch(Mem_Mode)
                   {
                     case NT_BASE2:   if(XCELL>36)
                                   {XCELL=0;XC++;Win_Update=1;}
                                   break;
                     case NT_QFORM:
                     case NT_SLONG:
                     case NT_ULONG:
                     case NT_FLOAT:   if(XCELL>16)
                                   {XCELL=0;XC++;Win_Update=1;}
                                   break;
                     default:
                     case NT_INTEGER: if(XCELL>7)
                                   {XCELL=0;XC++;Win_Update=1;}
                                   break;
                   }
                   break;
       case _TAB : XCELL=0; XC++;
                   switch(Mem_Mode)
                   {
                     case   NT_BASE2: {if(XC>0) XC=0;} break;
                     case   NT_QFORM:
                     case   NT_SLONG:
                     case   NT_ULONG:
                     case   NT_FLOAT: {if(XC>1) XC=0;} break;
                     default:
                     case NT_INTEGER: {if(XC>3) XC=0;} break;
                   }
                   break;

       case _Hm  : PC_Cursor = MEM_Address; break;
       case _End : PC_Cursor = MEM_Ad_End-(Mem_Win->w-8)/Xsz; break;

       case _Up  : PC_Cursor-=(Mem_Win->w-8)/Xsz; break;
       case _Dn  : PC_Cursor+=(Mem_Win->w-8)/Xsz;
                   switch(Mem_Mode)
                   {
                     case NT_BASE2:   if(PC_Cursor >= MEM_Ad_End)
                                     MEM_Address += 1;
                                   break;
                     case NT_QFORM:
                     case NT_SLONG:
                     case NT_ULONG:
                     case NT_FLOAT:   if(PC_Cursor >= MEM_Ad_End)
                                     MEM_Address += 2;
                                   break;
                     default:
                     case NT_INTEGER: if(PC_Cursor >= MEM_Ad_End)
                                     MEM_Address += 4;
                                   break;
                   }
                   break;
       case _F1  : err = Mem_Help();
                   err = Ext_Help(err);
                   break;
       case _F2  : //Win_Update=0;
                   EditBox(&MEM_Address,NT_AUTO); // Open edit box for address value
                   PC_Cursor = MEM_Address;
                   MEM_Ad_End = MEM_Address + d;
                   break;
       case _F3  :
                   err = getmem(CURSOR_ADDR,1,&CURSOR_VALU);
                   if(err!=NO_ERR) break;
                   switch(Mem_Mode)
                   {
                     case NT_FLOAT: i = EditBox(&CURSOR_VALU,NT_FLOAT);
                            CURSOR_VALU = IEEE_TMS(*((float *)&CURSOR_VALU));
                            break;
                     case NT_QFORM: i = EditBox(&CURSOR_VALU,NT_FLOAT);
                            f = *((float *)&CURSOR_VALU);
                            f *= QMPY;
                            CURSOR_VALU = IEEE_TMS(f);
                            break;
                     default: i = EditBox(&CURSOR_VALU,NT_INTEGER);
                            break;
                   }
                   if(i!=NT_NOREF) // Could be NO_LINE (escape)
                     err = putmem(CURSOR_ADDR,1,&CURSOR_VALU);
       break;

       //--------------------------------------------------------------
       case _Enter: // Call memory modifier for MEM_c_address
       _setcursortype(_NORMALCURSOR);
//     gotoxy(Mem_Win->x+10+XC*Xsz,Mem_Win->y+(int)((a*PC_Cursor+b)/d)-1);
       gotoxy(cx,cy);
       switch(Mem_Mode)
       {
         case NT_BASE2:   gettext(wherex(),wherey(),wherex()+32,wherey(),&p);
                       break;
         case NT_QFORM:
         case NT_SLONG:
         case NT_ULONG:
         case NT_FLOAT:   gettext(wherex(),wherey(),wherex()+16,wherey(),&p);
                       break;
         default:
         case NT_INTEGER: gettext(wherex(),wherey(),wherex()+ 8,wherey(),&p);
                       break;
       }
       p1 = p; p2 = p;

       switch(Mem_Mode)
       {
         case NT_BASE2:   for(x=0;x<32;x++) { *p1=*p2; p1+=1; p2+=2; }
                       break;
         case NT_QFORM:
         case NT_SLONG:
         case NT_ULONG:
         case NT_FLOAT:   for(x=0;x<17;x++) { *p1=*p2; p1+=1; p2+=2; }
                       break;
         default:
         case NT_INTEGER: for(x=0;x< 8;x++) { *p1=*p2; p1+=1; p2+=2; }
                       break;
       }
       *p1=0;               // f = *( (float *) &n );
       //----------- Cvt cell to long & put into memory -----------
       switch(Mem_Mode)
       {
         case NT_BASE2:  CURSOR_VALU = strtoul(p,&eptr,2); break;
         case NT_QFORM:  // Use float and QVAL to get long int
         case NT_FLOAT:  db = strtod (p,&eptr);
                      if(db > MAX_FLOAT) db = MAX_FLOAT;
                      if(db < MIN_FLOAT) db = MIN_FLOAT;
                      if(Mem_Mode==NT_FLOAT) CURSOR_VALU = IEEE_TMS(db);
                      else                CURSOR_VALU = db / QMPY;
                      break;
         case NT_SLONG:
         case NT_ULONG:
                     CURSOR_VALU = strtoul(p,&eptr,10);
                     break;
         case NT_INTEGER:
         default:     CURSOR_VALU = strtoul(p,&eptr,16);break;
       }
       err = putmem(CURSOR_ADDR,1,&CURSOR_VALU);
       break;
       //-----------------------------------------------------------
       //
       case _ESC : c = _ESC; break;

       case _F9  : Mem_Win_Big ^= 1;
                   if(Mem_Win_Big & 1)
                   {
                     if ( tmp.h < Mem_Win->h ) break;
                      tmp = *Mem_Win;
                 // Mem_Win=GrowWindow(Mem_Win->x,1,Mem_Win->w,24,Mem_Win);
                  gettextinfo(&ti);
                  th = ti.screenheight;
                  Mem_Win=GrowWindow(Mem_Win->x,1,Mem_Win->w,th-1,Mem_Win);
		      PC_Cursor = MEM_Address;
                   }
		   else
                   {
                     if ( tmp.h >= Mem_Win->h ) break;
    	             Mem_Win=GrowWindow(tmp.x,tmp.y,tmp.w,tmp.h,Mem_Win);
		     PC_Cursor = MEM_Address;
		     MEM_Window(MEM_Address);
                   }
		   break;

#if GRAPH_EN
       case _G   : err = graph (new_mem,MEM_Address,Mem_Mode);
                   textmode(screenmode);
                   Init_Screen();
                   if(err!=NO_ERR) return err;
                   break;
#endif

       case _D   : if(alt_key()) {*old=_D; c = _ESC; break;}
       case _C   : if(alt_key()) {*old=_C; c = _ESC; break;}

       default   : // Allow text type over of memory cell. On ENTER
                   // the 'text' in the cell is converted to a long
                   // then directly stored back to the same location
                   Win_Update=0;
                   if(isalpha(g))
                   {
                     if(g>'F') break;
                     if(g<'A') break;
                   }
                   else
                   {
                     if((c==_Del)||(c==_Hyp)||(c==_Eql)||(c==_Spc)||
                        (c==_prd)||(c==_mns)||(c==_pls))
                     {
                       if(Mem_Mode==NT_INTEGER) break;
                     }
                     else if(!isdigit(g)) break;
                   }

                   if(    (Mem_Mode== NT_SLONG )||(Mem_Mode==NT_ULONG)
                        ||(Mem_Mode== NT_FLOAT)||(Mem_Mode==NT_QFORM) )
                     {if(extalphanumkey(c,g)==0) break;}
                   else
                     {if(   alphanumkey(c  )==0) break;}



                   p[0]=g;
                   p[1]=0;
                   cputs(p);
                   XCELL++;
                   switch(Mem_Mode)
                   {
                     case NT_BASE2:   {if(XCELL>32) XCELL=32;} break;
                     case NT_QFORM:
                     case NT_SLONG:
                     case NT_ULONG:
                     case NT_FLOAT:   {if(XCELL>16) XCELL=16;} break;
                     default:
                     case NT_INTEGER: {if(XCELL>7 ) XCELL= 7;} break;
                   }
                   break;
     }
     if(XC<0) XC=0;
     switch(Mem_Mode)
     {
       case NT_BASE2:   {if(XC>0) XC=0;} break;
       case NT_QFORM:
       case NT_SLONG:
       case NT_ULONG:
       case NT_FLOAT:   {if(XC>1) XC=1;} break;
       default   :
       case NT_INTEGER: {if(XC>3) XC=3;} break;
     }
     if(PC_Cursor < MEM_Address) MEM_Address = PC_Cursor;
     if(PC_Cursor > MEM_Ad_End ) MEM_Address += (PC_Cursor-MEM_Ad_End);
     if(Win_Update)
     {
       if((c!=_Up)&&(c!=_Dn)&&(c!=_Pup)&&(c!=_Pdn)&&(c!=_Rt)&&(c!=_Lt))
         DASM_Window(DASM_Address); // No BigMemBox flicker for cursor move!
       if(Mem_Win_Big==0)
       {
      // DASM_Window(DASM_Address);
         CPU_Window();
       }
       else
       {
         EnableWindow(Mem_Win);  // Draw border
         ClrWin(Mem_Win);        // clear all text
       }
       err = MEM_Window( MEM_Address);
     }
  } while ( (c!=_ESC)  && (err == NO_ERR));
  if(err==HOST_PORT)
  {
    MEM_Address = OldMA;
    PC_Cursor   = OldMA;
    MEM_Window(MEM_Address);
  }
  Mem_Win_Big = 0;
  if( tmp.h < Mem_Win->h )
  {
    Mem_Win=GrowWindow(tmp.x,tmp.y,tmp.w,tmp.h,Mem_Win);
    PC_Cursor = MEM_Address;
    MEM_Window(MEM_Address);
  }
  DisableWindow(Mem_Win);
  CPU_Window();  // Update CPU window, covered up by big mem win?
  return err;
}
//-------------------------------------------------------
// CPU_Edit() is called from the command window.  When
// called the memory window becomes the active window
// highlighting the border.  Further keystrokes are
// processed here until a quit (escape) occurs, returning
// to the command window.  The parameter int *old passed
// into this function is used as an alternate messaging
// path for the memory, cpu and dasm window to control
// which window should have the next current context when
// the alt key is pressed.
//-------------------------------------------------------
MSGS CPU_Edit( int *old )
{
  #define CPUCW  14 /* CPU window column width            */
  #define CPUH    6 /* CPU window hex data column start   */
  #define CPUF    4 /* CPU window float data column start */
  MSGS err=NO_ERR;
//struct text_info ti;
  int YC2;
  int YC1;
  int c, x, g;
  int cs;
  int Xz, Yz;
  static int XC=0;
  int XCELL=-7; // -7 is used to preset difference equation to place cell
  int CPUx;
  int CPUy;
  static int YC=0; // use static for retention of last used cell
#define XC1 0
  int XC2=0;
  char p[40],*p1,*p2;
  char px[30];
  char attr,regstrg[40];
  *old= 0;
  _setcursortype(_NORMALCURSOR);
  Mem_Win_Big=0;
  int Win_update=0;
  EnableWindow(Cpu_Win);
  if(kbhit()) bioskey(0); // Clear keyboard
  do
  {
     StatusLine("#F1#Help");
     CPUx= Cpu_Win->x;
     CPUy= Cpu_Win->y;
     //
     // Get label and data of cell
     //
     gotoxy(CPUx+1+XC*CPUCW,CPUy+1+YC);
     gettext(wherex(),wherey(),wherex()+13,wherey(),&regstrg);
     p1 = regstrg;
     p2 = regstrg;
     cs = 1;
     for(x=0;x<13;x++)
     {
       *p1=*p2;
       if((*p1==' ')||(*p1=='+')||(*p1=='-')) break; // break on WS or num
       cs++;
       p1+=1;
       p2+=2;
     }
     *p1=0;    // Terminate label string
     // determine end of label, and start of data
     for(   ;x<13;x++)
     {
       if(*p2!=' ') break; // Start of data found
       cs++;
       p1+=1;
       p2+=2;
     }
   //  XC1 = cs     ;           // Leftmost for cell

     XCELL = XCELL - (XC2-CPUCW+cs+1);
     if(XCELL < XC1) XCELL=XC1;

     XC2 = CPUCW-cs-1;           // RightMost for cell
     Xz = CPUx+cs+XC*CPUCW;
     Yz = CPUy+1+YC;
     gotoxy(Xz,Yz);
     attr = p2[1];               // Get old attr
     textattr((BLUE<<4)|attr);   // Add blue background
     p1 = p;
     for(   ;x<13;x++)           // Copy remaining text to p
     {
       *p1=*p2;
       p1+=1;
       p2+=2;
     }
     *p1=0;
     cputs(p);
     textattr(attr);
     gotoxy(Xz+XCELL,Yz);
     //---------------------------------------
     //
     // get keystroke
     //
     while((c=bioskey(1)&0xff00)==0)// While waiting for a key, allow
       Release_TSlice();
     if((g=toupper(getch()))==0)
       g=toupper(getch());  // Get ASCII, clear kb, remove dbl keyhit
     //
     //
//   _setcursortype(_NOCURSOR);
     err = NO_ERR;
     Win_update=1;
     switch( c )
     {
       case _Lt  : XCELL--;
                   Win_update=0;
                   if(XCELL<XC1) // Move to left
                   {
                     if(XC==1){XC=0; XCELL=XC2;}
                     else            XCELL=XC1;
                     Win_update=1;
                   }
                   break;
       case _Rt  : XCELL++;
                   Win_update=0;
                   if(XCELL>XC2)
                   {
                     if(XC==0) {XC=1; XCELL=XC1;}
                     else             XCELL=XC2;
                     Win_update=1;
                   }
                   break;
       case _TAB : XCELL=XC1; XC++; if(XC>1) XC=0; break;
       case _Pup :
       case _Hm  : YC=0 ; break;
       case _Pdn :
       case _End : if(screenmode==C80) YC=14;
                   else                YC=28; break;
       case _Up  : YC-- ; if(YC< 0) YC= 0; break;
       case _Dn  : YC++ ;
                   if(screenmode==C80) { if(YC>14) YC=14;}
                   else                { if(YC>28) YC=28;}
                   break;
       case _F1  : err = CPU_Help();
                   err = Ext_Help(err);
                   break;
       case _Enter:// Call memory modifier for MEM_c_address
                   // scan the input 'update' string for WS chars
                   // if one is found, terminate the string.  If not
                   // the expression analyzer will concatenate the
                   // fields resulting in a bad value
                   p1 = p;
                   while(*p1++) if(*p1==' '){*p1=0; break;};
                   //
                   textbackground(BLACK);
                   if(screenmode==C80) {YC1 = 0; YC2 =  5;}
                   else                {YC1 = 1; YC2 = 10;}
                   if((YC >YC1)&&(YC <YC2))
                   {
                     if(CPU_mode==NT_INTEGER) // 40 bit hex
                     {
                       //------- Update upper 32 hex (exponent)
                       strcpy(px,"*_"        );
                       strcpy(px+2,regstrg  );
                       strcat(px  ,"=0x"    );
                       strcat(px  ,        p);
                       px[15]=0;
                       err = ExeCmdStrg(px,0);
                       if(err!=NO_ERR) break;
                       //------- Update upper 32 hex (mantissa)
                       strcpy(px,"*_R"      );
                       strcpy(px+3,regstrg+1);
                       strcat(px,"=0x"      );
                       strcat(px,        p+2);
                       err = ExeCmdStrg(px,0);
                     }
                     else
                     {
                       //------- Update as float
                       strcpy(px,regstrg  );
                       strcat(px,"="      );
                       strcat(px,p        );
                       err = ExeCmdStrg(px,0);
                     }
                   }
                   else
                   {
                     strcpy(px,regstrg);
                     strcat(px,"=0x");
                     strcat(px,p);
                     err = ExeCmdStrg(px,0);
                   }
                   Win_update = 0;            // Screen is already updated
                   EnableWindow (Cpu_Win);    // Change borders back
                   DisableWindow(Cmd_Win);
                   break;
       //
       case _ESC : c = _ESC; break;

       case _D   : if(alt_key()) {*old=_D; c = _ESC; break;}
       case _M   : if(alt_key()) {*old=_M; c = _ESC; break;}
//     case _C   : if(alt_key()) {*old=_C; c = _ESC; break;}
       default   : // Allow text type over of cell. ENTER converts text
                   // in the cell directly to a value
                   Win_update = 0; // Errent key hits dont change screen

                   if(isalpha(g)) // Comment to allow expressions in Fx=...
                   {
                     if(g>'F') break;
                     if(g<'A') break;
                   }
                   else
                   {
                     if(   (YC > 0) && (YC < 5) && (g!='+')
                        && (g!='-') && (g!='.') && (g!=' ')
                       )
                       if(!isdigit(g)) break;
                   }
                   if(   (YC > 0) && (YC < 5) && (g!='+')
                      && (g!='-') && (g!='.') && (g!=' ')
                     )
                     if(alphanumkey(c)==0) break;

                   p[0]=g;  p[1]=0;
                   cputs(p);
                   XCELL++;
                   if(XCELL>XC2) XCELL=XC2;
                   break;
     }

     if(screenmode!=C80) XC  = 0;
     else                XC &= 1;
     if(Win_update)
     {
       ClrWin(Cpu_Win);
       CPU_Window();
       DASM_Window(DASM_Address);
       MEM_Window(MEM_Address);
       EnableWindow (Cpu_Win);    // Change borders back
       DisableWindow(Cmd_Win);
     }
  } while ( (c!=_ESC)  && (err == NO_ERR));
  DisableWindow(Cpu_Win);
  return err;
}
//---------------------------------------------------------------------
// DSK3D_Stats() is a psuedo help window for debugging DSK3D or for
// quickly analyzing the users current runtime setup.  The information
// displayed in this window can be used to determine the current
// run status of several parameters.
//---------------------------------------------------------------------

void DSK3D_Stats(void)
{
  char MB[600]; // Big enough for 40x15 char display
  char *M;
  uint stack;
  uint sseg;
  asm  mov stack,sp  // Copy SP to C variable
  asm  mov sseg,ss   // Copy SP to C variable
  M = MB;
                    M+=sprintf(M,"DSK3 Version    = %2.2f\r\n",DSK3_rev);
                    M+=sprintf(M,"File name       = %s\r\n",DASM_file);
                    M+=sprintf(M,"COFF version    = %d\r\n",CoffVer);
                    M+=sprintf(M,"Port            = 0x%3X\r\n",port);
if((int)WSHIFT==-4) M+=sprintf(M,"Port mode       = NIBBLE (8/4)\r\n");
else                M+=sprintf(M,"Port mode       = BI-DIR (8/8)\r\n");
                    M+=sprintf(M,"DIR ctrl bits   = 0x%04x\r\n",_DIR);
                    M+=sprintf(M,"IO Delays       = %d\r\n",timeout);
if(Windows_Detected)M+=sprintf(M,"Windows detect  = YES\r\n");
else                M+=sprintf(M,"Windows detect  = NO\r\n");
if(Enhanced_Enable) M+=sprintf(M,"PG6 Ext codes   = ON\r\n");
else                M+=sprintf(M,"PG6 Ext codes   = OFF\r\n");

                    M+=sprintf(M,"BP's set        = %d\r\n",Nb_Brk_Pt);
                    M+=sprintf(M,"LastRef         = %d\r\n",last_ref);
                    M+=sprintf(M,"String Table Use= %u of %u bytes\r\n",
                                  (int)(symbol_ptr-symbols),
                                   sym_str_tbl_sz);
                    M+=sprintf(M,"String Tbl base = %04X:%04X\r\n",
                                  FP_SEG(symbols),FP_OFF(symbols));
                    M+=sprintf(M,"String Tbl Ptr  = %04X:%04X\r\n",
                                  FP_SEG(symbol_ptr),FP_OFF(symbol_ptr));
                    M+=sprintf(M,"x86 stack       = %04x:%04x\r\n",sseg,stack);
                    M+=sprintf(M,"Coreleft        = %ld\r\n",coreleft());
                    M+=sprintf(M,"FarCoreleft     = %ld\r\n",farcoreleft());

  Help_Menu(" DSK3D ENVIRONMENT ",MB,10,60,3,20);
}
//----------------------------------------------------------------
// Edit_Cmd() is responsible for command entry as well as calling
// subfunction windows (memory, cpu and dasm).  The main loop in
// this function gets and processes keystrokes into 'F key' commands
// or string commands at the command prompt.
//----------------------------------------------------------------
char Edit_Banner[]="#F1#Help #F2#REG40 #F3#FLOAT #F4#Srce #F5#Run "
                   "#F6#DispBP #F7#ClrAll #F8#SStep #F9#Grow #F10#FStep";
MSGS Edit_Cmd()
{
  int  Insert = 0;  // Insert mode flag
  int  bk, g, y, KeyShiftStat;
  int  cx, cy, ch;
  int  dh;
  MSGS err=NO_ERR;
  char *buf, *ptr, *BUF, *p;
  char temp[80];
  cx= Cmd_Win->x; cy= Cmd_Win->y; ch= Cmd_Win->h;
  dh=Dasm_Win->h;
  EnableWindow(Cmd_Win);
  if(CmdNum<        0)CmdNum=        0;
  if(CmdNum>CMDBFRS  )CmdNum=CMDBFRS  ;
  BUF = Cmd_Buf[CmdNum].s;
  buf=strend(BUF);
  if(*BUF)strcpy (CMSG,BUF);    //sprintf(CMSG,"%s",BUF);
  else    sprintf(CMSG,"CMD>");
  Cmd_Msg(CMSG,CMD_ATTR,1,1);
  StatusLine(Edit_Banner);
  do
  {
    if(Insert) _setcursortype(_NORMALCURSOR);
    else       _setcursortype(_SOLIDCURSOR);
    g = 0, bk=0;
    while(bioskey(1)==0)
    {
      // Trap and clear system key strokes here to keep
      // unwanted sytem messages and other traps from happening
      if(bioskey(2) & _Ctrl)
      {
        bioskey(0);
      }
      Release_TSlice(); //  Allow MTask while waiting
    }
    //
    // NOTE: Some keyboards/motherboards always return _Lt_Alt as
    //       being set.  The _Lt/_Rt_Alt can therefor not be used
    //
    KeyShiftStat = bioskey(2);
 //   if(bioskey(2)&_Alt)
    if(KeyShiftStat & _Alt)
    {
      bk = bioskey(1) & 0xff00;
      DisableWindow(Cmd_Win);
      do
      { switch(bk)
        { case _D: err = DASM_Edit(&bk); break;
          case _M: err = MEM_Edit (&bk); break;
          case _C: err = CPU_Edit (&bk); break;
          default: getch(); break;
        }
      }while(((bk==_C)||(bk==_D)||(bk==_M)) && (err == NO_ERR));
      bk = 0;  // Fall out of switch(bk)
    }
    else  // No modifiers -> key is normal ascii
    {      //mod
      bk = bioskey(1) & 0xFF00;
      //
      // If the num lock key is active, translate those keystrokes
      //
      if(KeyShiftStat & _Nmlck)
      {

        switch(bk)
        {
           case _pls: g = '+'; bk = 0xFF00; break;
           case _mns: g = '-'; bk = 0xFF00; break;
           case _Del: g = '.'; bk = 0xFF00; break;
           case _Ins: g = '0'; bk = 0xFF00; break;
           case _End: g = '1'; bk = 0xFF00; break;
           case _Dn : g = '2'; bk = 0xFF00; break;
           case _Pdn: g = '3'; bk = 0xFF00; break;
           case _Lt : g = '4'; bk = 0xFF00; break;
           case _Cnt: g = '5'; bk = 0xFF00; break;
           case _Rt : g = '6'; bk = 0xFF00; break;
           case _Hm : g = '7'; bk = 0xFF00; break;
           case _Up : g = '8'; bk = 0xFF00; break;
           case _Pup: g = '9'; bk = 0xFF00; break;
           default: g=getch(); break;
        }

        //g = getch();
      }
      else
        g = getch();
    }
    EnableWindow(Cmd_Win);
    StatusLine(Edit_Banner);
    if(kbhit()) g=getch();           // This removes double keyhit
    _setcursortype(_NOCURSOR);

    switch ( bk )
    {
      case _Ins   : Insert^=1;
                    break;
      case _SF1   : err = About_Help();
                    err = Ext_Help(err);
                  //  bk = 0;
                    break;
      case _SF2   : DSK3D_Stats();
                  //  bk = 0;
                    break;
      case _Del   : if(*buf==0) break;
                    ptr = buf+1;
                    p = buf;
                    while(*ptr) *buf++=*ptr++;
                    *buf=0;
                    buf = p;
                    break;
      case _Bs    : g=' ';
                    strcpy(buf-1,buf);
                    buf--;
                    break;             // Erase while backing up

      case _Lt    : buf--;   break;
      case _Rt    : if(*buf==0)
                    {
                      *buf++=' ';
                      bk = 0;
                      g = ' ';
                      *buf=0;
                    }
                    else
                      buf++;
                    break;
      case _Hm    : buf = BUF; break;
      case _End   :
                    if(bioskey(2) & (_Lt_Sh | _Rt_Sh)) *buf=0;
                    else                               buf= strend(BUF);
                    break;

      case _ESC   : buf=BUF; *buf = 0; CMD_Window(); break;
      case _Dn    : CmdNum++; break;
      case _Up    : CmdNum--; break;
      case _Pup   : //bk=_Enter;
                    CmdNum=0;       break;
      case _Pdn   : //bk=_Enter;
                    CmdNum=CMDBFRS; break;

      case _TAB   : strcpy(BUF, last_cmd);
                    buf=strend(BUF);
      case _Enter : bk=_Enter;
                    err= NO_ERR;
                    break;  // Re-execute last command
      case _F1    : err = Edit_Help();
                    err = Ext_Help(err);
                    break;
      case _F2    : CPU_mode = NT_INTEGER; CPU_Window(); break;
      case _F3    : CPU_mode = NT_FLOAT  ; CPU_Window(); break;
      case _F4    : Invalidate_DASM = 1;
                    SOURCE_DBG^=1;
                    if(strstr(DASM_file,".OUT"))
                    {
                      if(SOURCE_DBG) Dasm_Mode=DASM4;
                      else           Dasm_Mode=DASM3;
                    }
                    else Dasm_Mode=DASM3;
                    err = DASM_Window(DASM_Address);
                    break;
      case _F5    :
                    if(RUN_Cmd()!=NO_ERR) err=RUN_ERR;
                    //
                    // If a key was pressed to break the
                    // RUN condition clear it
                    g=0;
                    PC_Appli=CTXT[PC];
                    break;

      case _F6    : err = DB_Cmd(); break;
      case _F7    : err = CB_Cmd(0);
                    err =DASM_Window(DASM_Address);
                    break;
      case _F8    : err=SStep_Cmd(1); g=1; break;
      case _F10   : err=FSTEP_Cmd(1); PC_Appli=CTXT[PC];
                    g=0;
                    break;
      case _SF8   : swap_en = 1;
                    err=SStep_Cmd(1);
                    swap_en = 0;
                    g=1;
                    break;
      case _SF10  : swap_en = 1;
                    err=FSTEP_Cmd(1); PC_Appli=CTXT[PC];
                    swap_en = 0;
                    g=0;
                    break;

      case _F9    : Dasm_Big^=1;
                    if(Dasm_Big) Dasm_Win=GrowWindow(1,1,80,dh,Dasm_Win);
                    else
                    {
                      if(screenmode==C80)
                      Dasm_Win=GrowWindow(1,1,51,dh,Dasm_Win);
                      else
                      Dasm_Win=GrowWindow(1,1,64,dh,Dasm_Win);
                    }
                    if((err = DASM_Window(DASM_Address)) != NO_ERR) break;
                    if(!Dasm_Big)  err = CPU_Window();
                    break;
      case  0     : break;
      case  0xFF00:
      default     : if(Insert)
                    { if(*buf)    // If not at end, copy and insert
                      { strcpy(temp,buf);
                        *buf++=g;
                        strcpy(buf,temp);
                      }
                      else        // If at end, append new char
                      { *buf++=g;
                        *buf=0;
                      }
                    }
                    else
                    {
                      if(*buf) *buf++=g;          // Overtype
                      else    {*buf++=g; *buf=0; }// Append
                    }
                    break;
    }
  // All Screen updates (except SSTEP) moved to main loop
    switch(bk)
    {
      case   0:
      case _F8:
      case _Enter: break;
      case _Up:
      case _Dn:
      case _Pup:
      case _Pdn:// 5/2/96 nxt ln mod: cmd buf slct limit to displayed cmds
                //                    org alwd selct of non-buffer cmd
             // if(CmdNum>=CMDBFRS-1) CmdNum=CMDBFRS-1;
                if(CmdNum>=CMDBFRS  ) CmdNum=CMDBFRS  ; // original
                if(CmdNum<         0) CmdNum=        0;
                sprintf(CMSG,"CMD>" ,CmdNum); gotoxy(cx+5,cy+ch-2);
                Cmd_Msg(CMSG,CMD_ATTR,1,1);
                CMD_Window();
                BUF = Cmd_Buf[CmdNum].s;
                buf = strend(BUF);
                break;

      default : CMD_Window();
                if(g!=0) break;  // Dont update screen for chars
                y = (int)(CTXT[PC]-DASM_Address);
                if(y<  0) { DASM_Address  = CTXT[PC]; y=   0; }
                y = (int)(CTXT[PC] - DASM_Ad_End);
                if(y>=0) DASM_Address += y;
                if(err!=NO_ERR) break;
                err = Update_Screen(DASM_Address);
                break;
    }
    if(buf<BUF) buf=BUF;
    if(buf>BUF+CMDLENG-2) {buf=BUF+CMDLENG-2; *buf=0;}
    //
    // Command Window
    //
    Cmd_Msg(BUF,CMD_ATTR,1,0);
    gotoxy((int)(cx+1+(buf-BUF)),(int)(cy+ch-2));
  }while ((bk != _Enter) && (err==NO_ERR));

  strcpy(last_cmd,BUF);
  strcpy(Cmd_Line,BUF);

  // Rotate command stack by copying.  If current
  // buffer is the top
  if(bk==_Enter)
  {
    if(CmdNum==CMDBFRS)
    {
      for(int x=0;x<CMDBFRS;x++)
        strcpy(Cmd_Buf[x].s,Cmd_Buf[x+1].s);
      strcpy(Cmd_Buf[x].s,"");
    }
    else
      CmdNum++;
  }
  return err;
}

#define  sym_hlp_attr ((LIGHTGRAY<<4) | WHITE)
//---------------------------------------------------------------
// SYM_Help displays the symbol table within a window similar
// to Help_Menu() except that each string lines is created as
// it is needed.  The advantage is that a large string table is
// not needed to hold the string table information.
//-------------------------------------------------
enum OutDisplay{WSCREEN=0,WFILE};
MSGS SYM_Help(void)
{
  FILE *file;
  char *banner=" DEFINED SYMBOLS ";
//char symstrg[80];
  char abuf[80];     /* attribute type string */
  char bbuf[80];
  WINDOW *tmp;
  int refnum, x, bk, y;
  int Con;
  int Begin, End;
  int By_Valu = 0;
  ulong ltemp;
  bk = 0;
//#define attr ((LIGHTGRAY<<4) | WHITE)
  tmp = DSKCreateWindow(10,3,60,20,banner,FRAME2,0x70,sym_hlp_attr);
  OpenWindow(tmp);
  //
  if(last_ref > post_boot_sym) refnum = post_boot_sym;
  else                         refnum = last_ref-18;

  if(refnum > last_ref - 18) refnum = last_ref - 18;
  if(refnum < 0) refnum = 0;
  Con = WSCREEN;
  //
  for(;;)
  {
    if(Con==WSCREEN) ClrWin(tmp);
    y=3;
    Begin = refnum;
    End   = refnum+18;
    if(Con==WFILE)
    {
       Begin = 0;
       End = last_ref;
       file = fopen("DSK3SYMB.LST","wt");
       if(file==NULL) return OPEN_ERR;
    }
    /*-----------------------------------------------------------*/
    if(Con==WFILE) fprintf(file,"DSK3 symbol listing\n");
    if(Begin <= 0) Begin = 0;
#define S_SPTR  SYM[x].sptr
#define S_VALUE SYM[x].value
#define HP_ADDR 0xF00000L
    for(x=Begin;x<End;x++)
    {
      switch(SYM[x].numt)
      {
        default:
        case NT_INTEGER:
                switch(SYM[x].symt)
                {
                  case SYM_GLBL:sprintf(bbuf," fix    Label(G)" );break;
                  case SYM_LLBL:sprintf(bbuf," fix    Label(L)" );break;
                  case SYM_VAR:sprintf (bbuf," fix    Variable");break;
                  case SYM_DEF:sprintf (bbuf," fix    Define"  );break;
                }
   if(By_Valu && ((SYM[x].symt==SYM_GLBL)|(SYM[x].symt==SYM_LLBL)))
                {
                  if((S_VALUE & HP_ADDR)==HP_ADDR)
                    sprintf(abuf, "%-20s >bad address<  ",S_SPTR);
                  else
                  {
                    sprintf(abuf,"*0x%08lx",S_VALUE);
                    expressionz(abuf,&ltemp,NT_INTEGER);
                    if(By_Valu==1)
                      sprintf(abuf, "%-20s\"%08ld\"    ",S_SPTR,ltemp);
                    else
                    sprintf(abuf, "%-20s\"0x%08lx\"    ",S_SPTR,ltemp);
                  }
                }
                else
                sprintf(abuf, "%-20s 0x%08lx     ",S_SPTR,S_VALUE);

                break;
        case NT_FLOAT  :
                switch(SYM[x].symt)
                {
                  case SYM_GLBL:
                  case SYM_LLBL:
                  switch(SYM[x].symt)
                  {
                  case SYM_GLBL:sprintf(bbuf,"<float> Label(G)" );break;
                  case SYM_LLBL:sprintf(bbuf,"<float> Label(L)" );break;
                  }
                  if(By_Valu)
                  {
//                  sprintf(abuf,"*0x%08lx",S_VALUE);
                    if((S_VALUE & HP_ADDR)==HP_ADDR)
                      sprintf(abuf, "%-20s >bad address<  ",S_SPTR);
                    else
                    {
                      sprintf(abuf,"*%s",S_SPTR);
                      expressionz(abuf,&ltemp,NT_INTEGER);
                      if(By_Valu==1)
                      sprintf(abuf, "%-20s[%+8e] ",S_SPTR,TMS_IEEE(ltemp));
                      else
                      sprintf(abuf, "%-20s[0x%08lx]    ",S_SPTR,ltemp);
                    }
                  }
                  else
                    sprintf(abuf, "%-20s 0x%08lx     ",S_SPTR,S_VALUE);
                  break;
                  case SYM_VAR:sprintf(bbuf,"<float> Variable");
                  sprintf(abuf,"%-20s %+8e  ",S_SPTR, *(float *)&S_VALUE);
                  break;
                  case SYM_DEF:sprintf(bbuf,"<float> Define"  );
                  sprintf(abuf,"%-20s %+8e  ",S_SPTR, *(float *)&S_VALUE);
                  break;
                }
                break;
      }
      strcat(abuf,bbuf);
      /*------- Output action, write to screen or file ----------*/
      if(Con==WFILE) fprintf(file,"%s\n",abuf);
      else           PutString(12,y++,60,abuf,0x70,CAR);
    }
    if(Con==WFILE)
      fclose(file);
    /*-----------------------------------------------------------*/
    PutString(
    10,21,60, " Move Up/Dn/Pup/Pdn [S]ave2file [Spc]contents at Label",
    sym_hlp_attr,CAR);
    // Get a key from the user
    while(!kbhit());
    bk = bioskey(0) & 0xFF00;
    Con = WSCREEN;
    switch(bk)
    {
      case _Spc: By_Valu+=1  ;
                 if(By_Valu > 2) By_Valu = 0;
                 break;
      case _Up : refnum -=  1; break;
      case _Dn : refnum +=  1; break;
      case _Pup: refnum -= 18; break;
      case _Pdn: refnum += 18; break;
      case _S  : Con = WFILE; break;
      default  :
      case _Q  : CloseWindow(tmp);
                 EraseWindow(tmp);
                 return NO_ERR;
    }
    if(refnum <        0) refnum =        0;
    if(refnum > last_ref-18) refnum = last_ref-18;
  }
}
