//---------------------------------------------------
// COMMANDS.CPP
// Keith Larson
// TMS320 DSP Applications
// (c) Copyright 1995, 1996, 1997
// Texas Instruments Incorporated
//
// This is unsupported freeware code with no implied warranties or
// liabilities.  See the disclaimer document for details
//---------------------------------------------------
#include <stdio.h>   // Compiler header files
#include <io.h>
#include <bios.h>    // Needed for one call to bioskey() keytrap
#include <stdlib.h>
#include <conio.h>
#include <ctype.h>
#include <math.h>

#include "dsk.h"     // Application DSK functions and variables
#include "screen.h"  // Debugger and assembler functions and variables
#include "exp_anal.h"

BRKPOINT Brk_Tab[MAX_BRK_PT+1];   // Addr & instructions xref table
int      Nb_Brk_Pt=          0;   // Number of breakpoints set
ulong    BP_TRAP  =0x74000008L;   // ETRAP_8
ulong    MEM_Address ;
ulong    MEM_Ad_End  ;
ulong    DASM_Address;
ulong    DASM_Ad_End ;
//---------------------------------------------------------------------
// TMS320C31 Memory Map:
//    Non-sequential ordering is used for faster search in typical code
//---------------------------------------------------------------------
MEMORY_MAP Memory[] =
{
     { 0x00809800L, 0x00000400L, M_READ_WRITE}, //
     { 0x00809C00L, 0x00000400L, M_READ_WRITE}, //
     { 0x0080A000L, 0x00EFFFFFL, M_READ_WRITE}, // External
     { 0x00000000L, 0x00001000L, M_READ      }, // ROM
     { 0x00F00000L, 0x00100000L, M_INVALID   }, // Host port IF
     //                 808000
     { 0x00001000L, 0x007FF000L, M_READ_WRITE}, // External MEM
     { 0x00800000L, 0x00008000L, M_INVALID   }, // reserved 32K
     // Peripherals 6K
     { 0x00808000L, 0x00000001L, M_READ_WRITE}, // DMA0 Ctrl
     { 0x00808001L, 0x00000003L, M_INVALID   }, //
     { 0x00808004L, 0x00000001L, M_READ_WRITE}, //     Srce
     { 0x00808005L, 0x00000001L, M_INVALID   }, //
     { 0x00808006L, 0x00000001L, M_READ_WRITE}, //     Dest
     { 0x00808007L, 0x00000001L, M_INVALID   }, //
     { 0x00808008L, 0x00000001L, M_READ_WRITE}, //     Cntr
     { 0x00808009L, 0x00000007L, M_INVALID   }, // reserved

     { 0x00808010L, 0x00000001L, M_READ_WRITE}, // DMA1 Ctrl (C32 only)
     { 0x00808011L, 0x00000003L, M_INVALID   }, //
     { 0x00808014L, 0x00000001L, M_READ_WRITE}, //     Srce
     { 0x00808015L, 0x00000001L, M_INVALID   }, //
     { 0x00808016L, 0x00000001L, M_READ_WRITE}, //     Dest
     { 0x00808017L, 0x00000001L, M_INVALID   }, //
     { 0x00808018L, 0x00000001L, M_READ_WRITE}, //     Cntr
     { 0x00808019L, 0x00000007L, M_INVALID   }, // reserved

     { 0x00808020L, 0x00000001L, M_READ_WRITE}, // TIM0 Glbl
     { 0x00808021L, 0x00000003L, M_INVALID   }, //
     { 0x00808024L, 0x00000001L, M_READ_WRITE}, //      Cntr
     { 0x00808025L, 0x00000003L, M_INVALID   }, //
     { 0x00808028L, 0x00000001L, M_READ_WRITE}, //      Prd
     { 0x00808029L, 0x00000007L, M_INVALID   }, //

     { 0x00808030L, 0x00000001L, M_READ_WRITE}, // TIM1 Glbl
     { 0x00808031L, 0x00000003L, M_INVALID   }, //
     { 0x00808034L, 0x00000001L, M_READ_WRITE}, //      Cntr
     { 0x00808035L, 0x00000003L, M_INVALID   }, //
     { 0x00808038L, 0x00000001L, M_READ_WRITE}, //      Prd
     { 0x00808039L, 0x00000007L, M_INVALID   }, //

     { 0x00808040L, 0x00000001L, M_READ_WRITE}, //SERP0 Glbl
     { 0x00808041L, 0x00000001L, M_INVALID   }, //
     { 0x00808042L, 0x00000001L, M_READ_WRITE}, //      Xctl
     { 0x00808043L, 0x00000001L, M_READ_WRITE}, //      Rctl
     { 0x00808044L, 0x00000001L, M_READ_WRITE}, //      Tctl
     { 0x00808045L, 0x00000001L, M_READ_WRITE}, //      Tcnt
     { 0x00808046L, 0x00000001L, M_READ_WRITE}, //      Tprd
     { 0x00808047L, 0x00000001L, M_INVALID   }, //
     { 0x00808048L, 0x00000001L, M_READ_WRITE}, //      XMIT
     { 0x00808049L, 0x00000003L, M_INVALID   }, //
     { 0x0080804CL, 0x00000001L, M_READ_WRITE}, //      RECV
     { 0x0080804DL, 0x00000003L, M_INVALID   }, // C30

//   { 0x0080804DL, 0x00000013L, M_INVALID   }, // C31
     { 0x00808050L, 0x00000001L, M_READ_WRITE}, //SERP1 Glbl  (C30 only)
     { 0x00808051L, 0x00000001L, M_INVALID   }, //
     { 0x00808052L, 0x00000001L, M_READ_WRITE}, //      Xctl
     { 0x00808053L, 0x00000001L, M_READ_WRITE}, //      Rctl
     { 0x00808054L, 0x00000001L, M_READ_WRITE}, //      Tctl
     { 0x00808055L, 0x00000001L, M_READ_WRITE}, //      Tcnt
     { 0x00808056L, 0x00000001L, M_READ_WRITE}, //      Tprd
     { 0x00808057L, 0x00000001L, M_INVALID   }, //
     { 0x00808058L, 0x00000001L, M_READ_WRITE}, //      XMIT
     { 0x00808059L, 0x00000003L, M_INVALID   }, //
     { 0x0080805CL, 0x00000001L, M_READ_WRITE}, //      RECV
     { 0x0080805DL, 0x00000003L, M_INVALID   }, //

     { 0x00808060L, 0x00000001L, M_READ_WRITE}, //  XBUS control
     { 0x00808061L, 0x00000003L, M_INVALID   }, //    Exists on C31!!
//   { 0x00808060L, 0x00000004L, M_INVALID   }, //  XBUS control (C30 only)

     { 0x00808064L, 0x00000001L, M_READ_WRITE}, //  PBUS control
     { 0x00808065L, 0x0000000BL, M_INVALID   }, //

     { 0x00808070L, 0x00001790L, M_INVALID   }, //      reserved

     { 0x01000000L, 0xFE000000L, M_INVALID   }, // N/A


     // always terminated by the following values
     { 0x00000000L, 0x00000000L, M_INVALID    }
};

//---------------------------------------------------------------------
// Valid_Address() Checks if an address lies within the defined
// Memory Map array.  A NO_ERR is returned if the memory is valid
// and an M_INVALID is returned if it is not.
//---------------------------------------------------------------------
#define inside(x,a,b) ((a<=x) && (x<b))
//
MSGS Valid_Address( ulong a1, ulong a2)
{
 int i = 0;
 ulong addr, length;

 for(i=0;;i++)
 {
   addr   = Memory[i].address;
   length = Memory[i].length;
   if(length == 0) break;      // exit if end of table
// if(inside(start, addr, addr + length))
   if((a1 >= addr) && (a2 < (addr+length)))
   {
     if(Memory[i].type == M_INVALID)
       return OUT_MAP;
     return NO_ERR;
   }
 }
 return NO_ERR;
}
//----------------------------------------------------
// Checks if a floating point value is out of range
//----------------------------------------------------
int Is_TMS_Float( double v )
{
 if(((v<=MOST_POSITIVE) && (v>=LEAST_POSITIVE))
      || ((v>=MOST_NEGATIVE) && (v<=LEAST_NEGATIVE)) ) return 0;
 if ( v== 0.0 ) return 0;
 return 1;
}
//----------------------------------------------------
//   Resets debugger session
//----------------------------------------------------
MSGS RESET_Cmd(void)
{
  DSK_reset();
  Init_Communication(10000);
  Init_Screen();
  return NO_ERR;
}
//----------------------------------------------------
// Sets a breakpoint at an address in the breakpoint table
// The breakpoint itself placed in DSK memory just before
// executing a run command.  Otherwise no BP's are set.
//----------------------------------------------------
MSGS SB_Cmd( ulong address )
{
  int i;
  ulong P[3];
  MSGS err;
  if(Nb_Brk_Pt >= MAX_BRK_PT) return TOO_MANY_BRKPT;
  for(i=0; i<Nb_Brk_Pt; i++)
  {
    if(Brk_Tab[i].address == address )
      return NO_ERR;
  }
  if((err = Valid_Address(address,address)) != NO_ERR) return err;
  /*----------------------------------------------------*/
  /* Check to see if previous three opcodes are delayed */
  /*----------------------------------------------------*/
  if((err=getmem(address-3,3,P))!=NO_ERR) return err;
  for(i=0;i<3;i++)
  {
    if((P[i] & 0xFF000000L) == 0x61000000L) return DELAYED_ERR; // BRD
    if((P[i] & 0xFDE00000L) == 0x68200000L) return DELAYED_ERR; //BcondD
    if((P[i] & 0xFC200000L) == 0x6C200000L) return DELAYED_ERR; //DBcondD
  }
  /*-----------------------------------*/
  /*    Put breakpoint in BP table     */
  /*-----------------------------------*/
  Brk_Tab[Nb_Brk_Pt].address = address;
  if((err=getmem(address,1,&Brk_Tab[Nb_Brk_Pt].instr))!=NO_ERR) return err;
  Nb_Brk_Pt++;
  /*                                      */
  /*   Removed ver 1.16                   */
  /* Cant remember why extra code is here */
  /* Remove after beta test...            */
  /*                                      */
//CPU_Window(/*0*/);
//if((PC_Appli<DASM_Address) || (PC_Appli>=DASM_Ad_End))
//MEM_Window(MEM_Address);
  return NO_ERR;
}
//------------------------------------------------
//           Clears breakpoints
// CB <address>  clears a breakpoint
// CB            clears all breakpoints
//------------------------------------------------
MSGS CB_Cmd(ulong address)
{
  int i,j;
  MSGS err;
  if(address == 0L)       /* Clear all breakpoints if no address is given */
  {
    if((err = clr_all_BP())!=NO_ERR) return err;
    Nb_Brk_Pt = 0;
    address = DASM_Address;
  }
  else
  {
    for(i=0; i<Nb_Brk_Pt; i++)
    {
      if(Brk_Tab[i].address == address)
      {
      if((err=
        putmem(Brk_Tab[i].address,1, &Brk_Tab[i].instr))!=NO_ERR)
        return err;
      break;
      }
    }
    for(j=i ; j<Nb_Brk_Pt; j++)
      Brk_Tab[j] = Brk_Tab[j+1];

    if(i < Nb_Brk_Pt)
      Nb_Brk_Pt--;
    else
      return NO_BRKPT;
  }
  return NO_ERR;
}
//-------------------------------
//       Displays breakpoints
//-------------------------------
MSGS DB_Cmd(void)
{
  int i=0;
  WINDOW *tmp;
  char buf[20];
  if ( Nb_Brk_Pt == 0 ) return NO_ERR;
  tmp = DSKCreateWindow
        (30,22-Nb_Brk_Pt,16,Nb_Brk_Pt+2," BREAKPOINTS ",
         FRAME2,WIN_ATTR, WIN_ATTR );
  OpenWindow(tmp);
  EnableWindow(tmp);
  for ( i=0; i<Nb_Brk_Pt; i++)
  { sprintf(buf,"  %02d : %06lx",i+1,Brk_Tab[i].address);
    PutString(tmp->x,tmp->y+i,tmp->w-2,buf,tmp->tattr,CARFOND);
  }
  while(!kbhit());
  getch();
  DisableWindow(tmp);
  CloseWindow(tmp);
  EraseWindow(tmp);
  return NO_ERR;
}
//----------------------------------
//   Open an edit box to modify val
//----------------------------------
//MSGS scans(char *p, int l);

NUM_TYPE EditBox(ulong *val, NUM_TYPE nt)
{
  WINDOW *tmp;
  MSGS err;
//  NUM_TYPE nt;
  char buf[40];
  int cx, cy;
//float f;
  _setcursortype(_NORMALCURSOR);
  cx = wherex();
  cy = wherey();
  tmp = DSKCreateWindow(30,21,40,3," Enter an Expression ",
                    FRAME2,WIN_ATTR, WIN_ATTR );
  OpenWindow(tmp);
  EnableWindow(tmp);
  //
  if(nt == NT_FLOAT) sprintf(buf,"%e ",TMS_IEEE(*val));
  else               sprintf(buf,"0x%08lx ",*val);
  gotoxy(tmp->x+2,tmp->y +1); // Print string
  printf(buf);
  gotoxy(tmp->x+2,tmp->y +1); // Go back to start of string
  /*----------------------------------------------*/
  /* Get string expression and convert to a value */
  /*----------------------------------------------*/
  if((err = scans(buf,36))==NO_ERR)              // Call input string editor
     nt = expressionz(buf,val,nt);  // Evaluate string
  /*----------------------------------------------*/
  DisableWindow(tmp);
  CloseWindow(tmp);
  EraseWindow(tmp);
  gotoxy(cx,cy);
  if(err!=NO_ERR) return NT_NOREF;
  return nt;
}

// ------------------------------------------------
//     Modify or Fill Memory
//      MM < address > [ < length > <value> ]
//-------------------------------------------------
MSGS MM_Cmd( ulong address, ulong length, ulong value)
{
 ulong i;
 char nvalue[11],*flag;
 MSGS err;
 WINDOW *tmp;
 /* memory block modifying */
 if(length)
   err = Valid_Address(address,address+length-1L);
 else
   err = Valid_Address(address,address);
 if ( err != NO_ERR ) return err;

 if ( length )
  {
    for ( i=0; i<length; i++,address++)
	if (putmem(address,1,&value) != NO_ERR) return COM_ERR;
    address -= length;
  }
 else {
	tmp=DSKCreateWindow(24,23,28,3," MEMORY MODIFY ", FRAME2,
		     WIN_ATTR, WIN_ATTR);
	OpenWindow(tmp);
	if (getmem(address,1,&value) != NO_ERR) return COM_ERR;
	sprintf(nvalue," %06lx",address);
	PutString(tmp->x,tmp->y,7,nvalue,CYAN,CAR);  /* y+1 */
	sprintf(nvalue," %08lx ",value);
	PutString(tmp->x+7,tmp->y,9,nvalue,tmp->tattr,CARFOND); /* y+1 */
	textattr(tmp->tattr^0x8);
	_setcursortype(_NORMALCURSOR);
	nvalue[0]=9; gotoxy(tmp->x+18, tmp->y+1); cgets(nvalue);
	_setcursortype(_NOCURSOR);
	CloseWindow(tmp);
        EraseWindow(tmp);
	i = strtoul(nvalue+2,&flag,16);
	if ( !*flag && *(nvalue+2) ) value = i;
	if (putmem(address,1,&value) != NO_ERR) return COM_ERR;
	length=1L;
      }
 return NO_ERR;
}
//---------------------------------------------------------
// SSTEPS until address N is reached
//---------------------------------------------------------
MSGS XG_Cmd(ulong N)
{
  MSGS err;
  for(;;)
  {
    err =  SStep_Cmd(1);
    Cmd_Msg("",CMD_ATTR,0,1);
    if(CTXT[PC] == N) break;
    if(err == USER_STOP) {err = NO_ERR; break;}
  }
  return err;
}
//--------------------------------------------------------
// Set a breakpoint at an address.  The actual breakpoint
// is not loaded to the DSK by this function.  Only the
// breakpoint table is searched.
//--------------------------------------------------------
char BP_SET(long addr)
{
  int i;
  for(i = 0; i < Nb_Brk_Pt; i++)        // search breakpoint table
  { if(Brk_Tab[i].address== addr)
    { if(Brk_Tab[i].instr != BP_TRAP) return 1; }
  }
  return 0;
}
//--------------------------------------------------------
// set_all_BP() Sets all breakpoints found in the
// breakpoint table by swapping a TRAP with each location.
// with the original opcode.  NOTE: since the singlestep
// operation does not require TRAP emulation, this function
// is only called when the processor is to be run and then
// possible halted by a breakpoint (or host interrupt).
//--------------------------------------------------------
MSGS set_all_BP(void)
{
  int i;
  MSGS err;
  for(i = 0; i < Nb_Brk_Pt; i++)               // place all breakpoints
  {
    if((err= getmem(Brk_Tab[i].address,1,&Brk_Tab[i].instr)) !=NO_ERR)
      return err;
    if((err= putmem(Brk_Tab[i].address,1,&BP_TRAP))!=NO_ERR) return err;
  }
  return NO_ERR;
}
//--------------------------------------------------------
// clr_all_BP() writes back all of the original pre-run
// opcodes from the breakpoint table.
//--------------------------------------------------------
MSGS clr_all_BP(void)
{
  int i;
  MSGS err;
  for(i = 0; i < Nb_Brk_Pt; i++)     // remove all breakpoints
  if((err=putmem(Brk_Tab[i].address,1, &Brk_Tab[i].instr))!=NO_ERR)
     return err;
  return NO_ERR;
}
//-----------------------------------------------------------
//  RUN commands
//  RUN_Cmd(void) sets all breakpoints before running
//  RUNF_Cmd(void) does not set breakpoints before running
//   Both functions call RUNCmd, passing a flag to that function
//-----------------------------------------------------------
MSGS RUNCmd  (int BPRUN);
MSGS RUN_Cmd (void) { return RUNCmd(1); } // Run with breakpoints
MSGS RUNF_Cmd(void) { return RUNCmd(0); } // Run without breakpoints
MSGS RUNCmd(int BPRUN)
{
  int bk;
  MSGS err, err2=NO_ERR;
  ulong temp;
  ulong Ctxt_RS, Ctxt_RE, Ctxt_ST,Ctxt_RC;

  char BP_FOUND;

  // All BP's should be clear at this time since DSP is not in
  // the run mode.  should not need this line
//  if((err = clr_all_BP()) != NO_ERR) {enable_Mtask(); return(err);}

  if((err = SSTEP_CPU())  != NO_ERR) {enable_Mtask(); return(err);}
  enable_Mtask();
  //
  // During the process of setting the breakpoints the CPU is still
  // accepting and processing interrupts.  Since it is not advisable
  // to hit a BP trap at this time, a BP set within a ISR could cause
  // trouble.  To solve this the interrupts are temporarily masked
  // during the BP set procedure.
  //
  // At the time of the BP set procedure the CPU is in the spin0 loop
  // accepting new commands.  Simply turning them off here might also
  // fix the problem.
  //
  if(BPRUN)
  { if((err = set_all_BP()) != NO_ERR)
    { enable_Mtask(); CMD_Window();
      return(err);
    }
  }
  Cmd_Msg(
  "CPU Running: Press any key to stop, END to exit DSK3D while running",
  CMD_ATTR,1,1);
  /*------------------------------------------------------------------
    Disabling Mtask before allows RUN_CPU() to execute and recover
    before another application has a chance to re-stop the processor
    ------------------------------------------------------------------*/
  disable_Mtask();
  if((err = RUN_CPU()) != NO_ERR)
  {
    enable_Mtask();
    CMD_Window();
    return(err);
  }
  /*--------------------------------------------------------------------
     The DSP now runs until a breakpoint is found or the user hits a key.
     If the DSP hits a BP, it will transmit a word to the host as an echo
     reply that a BP was found.
    --------------------------------------------------------------------*/
  for(;;)
  {
    /*-------------------------------------------------------------------
      If a BP is encountered the HP_ACK line will go active low MTask
      should be off to prevent another app from stoping a stopped
      processor.
      ------------------------------------------------------------------*/
    if(HPI_ACK())      // If DSP has hit a BP
    { //disable_Mtask();
      if(HPI_ACK())     // Make sure not a glitch
      { if(HPI_ACK())
        { if((err = recv_long(&temp))!= NO_ERR)
          { enable_Mtask();
            CMD_Window();
            return err;
          }
          BP_FOUND = 1;  // Only TRAP BP's send back an acknowledge
          break;
        }
      }
      enable_Mtask();
    }
    enable_Mtask(); //**
    if(kbhit())                        // If user is impatient and exits
    {                                  // before breakpoint is found,
      bk = bioskey(1) & 0xFF00;        // Halt CPU, but do not --PC
      if(bk==_End)
      {
        fcloseall();
        clrscr();
        _setcursortype(_NORMALCURSOR);
        printf("DSK3D exited normaly with application running\n");
        printf(Exit_Hlp);
        exit(0);
      }
      if((err = HALT_CPU())!= NO_ERR)
      { if(err==BP_FOUND) break;
        enable_Mtask();
        Cmd_Msg("",CMD_ATTR,1,1);
        //
        // If a key was pressed to exit the
        // RUN command, it needs to be cleared
        if(bioskey(1)) bioskey(0);
        CMD_Window();
        return err;
      }
      BP_FOUND = 0;
      break;
    }
    disable_Mtask(); //**
  }
  //
  // If a key was pressed to exit the
  // RUN command, it needs to be cleared
//if(bioskey(1)) bioskey(0);  // Clear in higher level function
  Cmd_Msg("",CMD_ATTR,1,1);
  enable_Mtask();

  // Clear all BP's, get Reg Values
  if((err = clr_all_BP()) != NO_ERR)
  {enable_Mtask(); CMD_Window();return err;}

  if((err=getmem(CTXT_PTR,CTXTSIZE,&CTXT[0]))!=NO_ERR)
  {enable_Mtask(); CMD_Window();return err;}             // get PC
  //------------------------------------------------
  // if SP is inside kernel space return an error
//if((CTXT[SP] < KERNEL_END)&&(CTXT[SP] > KERNEL_START))
//  return SP_CONFLICT;
  //
  // If halted by a BP (TRAP), the TRAP pushed TRAP+1
  if(BP_FOUND)
  {
    //-------------------------------------------------------------
    // Ver 1.12 fix. If a BP TRAP (0x78000008) is set at the end
    // of an active block, the result is that the PC will be at the
    // blocks RS.  By then analyzing the RPT_MODE bit, and if a BP
    // is set at the end of the block, the correct state can be
    // recovered.
    Ctxt_RS = CTXT[RS];
    Ctxt_RE = CTXT[RE];
    Ctxt_ST = CTXT[ST];
    Ctxt_RC = CTXT[RC];
    if(CTXT[PC] == CTXT[RS])
    {
      if((BP_SET(Ctxt_RE  )==1) && (BP_SET(Ctxt_RS-1)==1))
      {
        err2 = RPTB_ERR;
      }
      if(BP_SET(Ctxt_RE  )==1)
//    if(CTXT[ST] & 0x100)
      {
        CTXT[PC] = CTXT[RE];
        Ctxt_RC++;
        Ctxt_ST|=0x100;
        putmem(CTXT_PTR+RC,1,&Ctxt_RC);
        putmem(CTXT_PTR+ST,1,&Ctxt_ST);
      }
      else                 CTXT[PC] --;          // Normal BP process
    }
    else CTXT[PC]--; // End ver 1.12 fix
    //--------------------------------------------------------------
 // CTXT[PC]--; // Original code follows

    if((err = putmem(CTXT_PTR+PC,1,&CTXT[PC]))!= NO_ERR)
    {enable_Mtask();   CMD_Window();return err;}
  }
  PC_Appli = CTXT[PC];
  CMD_Window();
  if(err2!=NO_ERR)
  {
    Cmd_Msg(Error_Strg(err2),ERR_ATTR,1,1);
    while(!kbhit());
  }
  return NO_ERR;
}
//----------------------------------------------------------------
// Single Step instruction by forcing INT into the pipe on context save
// where the INT vector has been pointed back to the context save routine.
//----------------------------------------------------------------
MSGS SStep_Cmd(long N)
{
  long n;
  char MSG[20];
  MSGS err=NO_ERR;
  n = N;
  if((err = clr_all_BP())!= NO_ERR)
    return err;
  while(n--)
  { Release_TSlice();
    err = SSTEP_CPU();
    if(err == Recover1)
    {
      Cmd_Msg(Error_Strg(err),ERR_ATTR,1,1);
      err = NO_ERR;
      n = 0;
    }
    if(err != NO_ERR)
      return err;

    if((err = CPU_Window())!=NO_ERR) return err;
    PC_Appli = CTXT[PC];
    if(PC_Appli<DASM_Address) DASM_Address -= (DASM_Address - PC_Appli);
    if(PC_Appli>DASM_Ad_End ) DASM_Address -= (DASM_Ad_End  - PC_Appli);
    if((err = MEM_Window(MEM_Address))!=NO_ERR)   return err;
    if((err = DASM_Window(DASM_Address))!=NO_ERR) return err;
    // If a breakpoint is set at the current PC,
    // return to caller with correct message
    //if(bioskey(1)) return USER_STOP;
    if(kbhit()) return USER_STOP;
    if(n)
    {
      if(BP_SET(PC_Appli)) return NO_ERR;
      sprintf(MSG,"%ld  %ld",N,n);
      Cmd_Msg(MSG,CMD_ATTR,1,1);
    }
  }
  return err;
}
// ---------------------------------------------------------
// FSTEP_Cmd() 'runs' through function calls when they are
// encountered by setting a BP immediately after the call
// ---------------------------------------------------------
MSGS FSTEP_Cmd(long N)
{
  MSGS err;
  char MSG[20];
  ulong ptr;
  ulong opc;
  long X;
  if(kbhit()) getch();
//for(X=N;X>0;X--)
  X = N;
  while(X--)
  {
//  Release_TSlice();    // Do not Release_Tslice() here
    if((err = getmem(PC_Appli,1,&opc))!=NO_ERR) return err;
    if((err = clr_all_BP())!= NO_ERR) return err;

    if(((opc & 0xFF000000L)==0x62000000L) ||  // Call or Call conditional
       ((opc & 0xFDE00000L)==0x70000000L))    // cannot FSTEP @ EO Block
    {
//    Release_TSlice();  // Do not Release_Tslice() here
  //set_all_BP();  // Removed in version 1.23  not sure why this was here
      //-------------------------------------
      if(BP_SET(PC_Appli+1))           // Use existing BP if already set
      { if((err = RUN_Cmd())!=NO_ERR)  // Traps are written to DSK here
        { enable_Mtask();
          return err;
        }
      }
      else                            // Add temp BP just after CALL/CALLc
      { ptr = PC_Appli + 1;           // RUN_Cmd mods PC_Appli, need temp
        SB_Cmd(ptr);                  // Get instr & check -3 delayed codes
        if((err=RUN_Cmd())!=NO_ERR)
        { enable_Mtask();
          CB_Cmd(ptr);
          return err;
        }
        CB_Cmd(ptr);
      }
      enable_Mtask();  //
      // Because a singlestep is executed before a RUN from a call
      // the timer delta will be off by the CALL count -1 (4-1=3 cycles).
      // NOTE: This value is preadjusted in the context
      CTXT[T1DIF] += 3;
      putmem(CTXT_PTR+T1DIF,1,&CTXT[T1DIF]);
      if((err = CPU_Window())!=NO_ERR) return err;
      PC_Appli = CTXT[PC];
      if(PC_Appli<DASM_Address) DASM_Address -= (DASM_Address - PC_Appli);
      if(PC_Appli>DASM_Ad_End ) DASM_Address -= (DASM_Ad_End  - PC_Appli);
      if((err = MEM_Window(MEM_Address))!=NO_ERR)   return err;
      if((err = DASM_Window(DASM_Address))!=NO_ERR) return err;
      // If a breakpoint is set at the current PC,
      // return to caller with correct message
      if(kbhit()) return USER_STOP;
    }
    else
    {
      err = SStep_Cmd(1);        // Step one opcode
      if(err==USER_STOP) break;
    }
    if(X)
    {
      if(BP_SET(PC_Appli))
      {
        return NO_ERR;
      }
      sprintf(MSG,"%ld  %ld",N,X);
      Cmd_Msg(MSG,CMD_ATTR,1,1);
    }
  }
  sprintf(MSG,"CMD>");
  Cmd_Msg(MSG,CMD_ATTR,1,1);
  if(N)
    for(;;)
    {
      Release_TSlice();
      if(kbhit()) break;
    }
  return NO_ERR;
}

