//------------------------------------------------------------
// REGVIEW.CPP
// Keith Larson
// TMS320 DSP Applications
// (c) Copyright 1995, 1996
// Texas Instruments Incorporated
//
// This is unsupported freeware code with no implied warranties or
// liabilities.  See the disclaimer document for details
//------------------------------------------------------------
//  REGVIEW.CPP is a DOS text based program which is a modifed
//  version of MEMVIEW.CPP.  REGVIEW continually downloads the CPU
//  context array and displays the general purpose registers in
//  32/40 bit hex or floating point.  REGVIEW can be run concurrently
//  with the DSK3D debugger.
//
//  The project file should include the following
//
//    REGVIEW.CPP   This file
//    DRIVER.CPP    Low level printer port drivers
//    TARGET.CPP    DSK Command level
//    OBJECT.CPP    Application setup routines
//    DSK_COFF.CPP  DSK and COFF file loaders and other utils
//    ERRORMSG.CPP  Messages used for most function returns
//    SYMBOLS.CPP   Symbol tables (needed to link DSK_COFF)
//    TEXTWIN.CPP   DOS level text window functions
//    HARDWARE.CPP  Command line help message
//                  (Built from HARDWARE.HLP source)
//
//  Borland C++ version 3.1 setup (use defaults if not shown)
//
//    Compiler
//     Memory Model      - Large        (objects reused in DSK3D link)
//     Processor         - 80286        (old PC support)
//     Floating Point    - Emulation
//     Code Gen          - DOS Standard (works in Win DOS prompt box)
//     Calling convention- C
//     Optimization      - Speed        (code reduction is not significant)
//
//    Linker
//     Output              - DOS Standard EXE   (Not a windows app!)
//     Libraries           - Graphics
//     Object Windows Lib  - None
//     Container Class     - None
//     Standard Runtime Lib- Static
//
//  NOTE: Other than using '//' comments and other simple C++ features
//        this code follows ANSI C.  The C++ compiler is used primarily
//        for convenience as well as its performance and advanced error
//        checking and warnings.
//
//        An 80286 output is used for DSK users who are using old PC's
//        for automated 'smart' data collection boxes.
//
//  Windows 3.x/95
//        This is a DOS text window executable which uses the timelsice
//        managment interrupt hooks to make the application more efficient
//        at swapping in and out as a Windows task.  This application can
//        coexist with the debugger and other MEMVIEW applications.
//
//        A PIF file can be created to automate setup and execution.
//--------------------------------------------------------------

#include <bios.h>
#include <conio.h>
#include <dos.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include "dsk.h"
#include "keydef.h"
#include "tmsfloat.h"

enum {Fixed,Float}MemModes;
ulong memaddr = 0x809800;
int   speed   = 3;
int   Mode    = Fixed;
char  disp[8192];   // char display buffer

void check_key(void)
{
  int key;
  ulong tmp, eaddress;
  key = bioskey(0) & 0xFF00;
  switch(key)
  {
    case  _Q  : exit(0);                // Quit
    case  _F  : switch(Mode)
                {
                  case Fixed: Mode = Float; break;
                  default:
                  case Float: Mode = Fixed; break;
                }
                clrscr();
                break;

    case  _Pup: memaddr-=24; break;  // Page window up
    case  _Pdn: memaddr+=24; break;  // Page window down
    case  _Up : memaddr-= 4; break;  // Scroll up
    case  _Dn : memaddr+= 4; break;  // Scroll down

    case  _A  : cprintf("\r\n\nNew address ");
                scanf("%lx",&memaddr);
                clrscr();
                break;

    case  _pls: speed+=1; break;        // Slow down screen update rate
    case  _mns: speed-=1; break;        // speed up
    default   : break;
  }
  if(speed <  0) speed =  0;
  if(speed > 10) speed = 10;
}

MSGS UPDATE_CPU(void)
{
// If a XHALT command is issued while the DSK is halted, the current
// context (debugger mode) will be saved instead of the runtime
// context.  If a XRUN were then issued the resulting restore operation
// would corrupt the registers.
//
// A method of detecting a halt condition is provided in the C3X.ASM
// communications and debug kernel.  This method involves polling the
// FREERUN memory location at the end of the register context save structure.
//
//   FREERUN==0 -> DSK is currently stopped... DO NOT HALT_CPU
//   FREERUN!=0 -> DSK is currently running    HALT_CPU/RUN_CPU updates regs
//
//
  int t;
  ulong CMD;
  MSGS err;
  getmem(CTXT_PTR+FREERUN,1,&CMD);
  if(CMD==0) return NO_ERR;      // Registers have already been updated
  //else                         // Update register context before return

  CMD = XHALT;                      // Write command
  disable_Mtask();
  if((err=xmit_long(CMD))!=NO_ERR) { enable_Mtask(); return RECV_ERR; }
  for(t=500;t>0;t--)     // Allow time for context save in slow DSK
  { if(HPI_ACK()) break;
    delay(1);
  }
  if(t==0) { enable_Mtask(); return RECV_ERR; }
  err = recv_long(&CMD);

  CMD = XRUNF;                // Write command
  err = xmit_long(CMD);
  delay(10);  // Allow time for context restore
  enable_Mtask();
  return err;
}
#define memsize 24
//------------------------------------------------------
// regprintf() keeps the code inside main easier to read
//------------------------------------------------------
int regprintf(char *c,int R, int Fndx,int Mndx)
{
  int  i;
  int  t;
  long f;
  long m;
  f = CTXT[Fndx];
  m = CTXT[Mndx];
  t = f>>24;
  i=sprintf(c,"R%d %+8e   %02x %08lx    %+4d %+12ld\r\n",
               R,TMS_IEEE(f),t&0xFF,m      ,t  ,m);
  return i;
}
//------------------------------------------------------
// main()
//------------------------------------------------------
void main()
{
  clock_t start, end;
  ulong *lp, temp;
  ulong MEM[memsize];     // Array to hold contents of stack
  char *c;
  int x, y;
  MSGS err;
  //
  // Check for command line arguments
  //
  Scan_Command_line("REGVIEW.EXE");
  //
  // Initialize the communications link to the DSK
  //
  for(;;)
  {
    if(Init_Communication(10000) == NO_ERR) break;
    if(kbhit()) exit(0);
  }
  //
  // If windows is being run, is set the appropriate flags
  // to use Windows timeslice management interrupts.
  //
  Detect_Windows();
  clrscr();
  for(;;)
  {
    // Get the DSK registers and put into local context storage array
    UPDATE_CPU();  // HALT/RUN update of register context
    if((err=getmem(CTXT_PTR,CTXTSIZE,CTXT)) != NO_ERR)
    { printf("%s\ngetmem() has crashed!\n",Error_Strg(err));
      exit(0);
    }
    if(kbhit()) check_key();
    c = disp;
    c+=sprintf(c,
    "C31DSK REGVIEW   (Q)uit +/- screen rate: %d \r\n"
    "--------------------------------------------\r\n",speed);
    c+=regprintf(c,0,R0F,R0);
    c+=regprintf(c,1,R1F,R1);
    c+=regprintf(c,2,R2F,R2);
    c+=regprintf(c,3,R3F,R3);
    c+=regprintf(c,4,R4F,R4);
    c+=regprintf(c,5,R5F,R5);
    c+=regprintf(c,6,R6F,R6);
    c+=regprintf(c,7,R7F,R7);
    //
    c+=sprintf(c,"\r\n");
    c+=sprintf(c,"PC  %08lx   SP  %08lx\r\n",CTXT[ PC],CTXT[ SP]);
    c+=sprintf(c,"AR0 %08lx   AR1 %08lx\r\n",CTXT[AR0],CTXT[AR1]);
    c+=sprintf(c,"AR2 %08lx   AR3 %08lx\r\n",CTXT[AR2],CTXT[AR3]);
    c+=sprintf(c,"AR4 %08lx   AR5 %08lx\r\n",CTXT[AR4],CTXT[AR5]);
    c+=sprintf(c,"AR5 %08lx   AR7 %08lx\r\n",CTXT[AR6],CTXT[AR7]);
    c+=sprintf(c,"\r\n");

    getmem(memaddr,memsize,MEM); // NOTE: SP -> TOP of stack...
    temp=memaddr;
    lp = MEM;
    //
    // Display in either Fixed (Hex) or Floating point
    //
    switch(Mode)
    {
      case Fixed: for(x=0;x<memsize;x+=4)
                  { c+=sprintf(c,"%06lx> %08lx %08lx %08lx %08lx\r\n",
                                temp+x,lp[0],lp[1],lp[2],lp[3]);
                    lp +=4;
                  }
                  break;
      default:
      case Float: for(x=0;x<memsize;x+=4)
                  { c+=sprintf(c,"%06lx> %+8e %+8e %+8e %+8e\r\n",
                                temp+x,
                                TMS_IEEE(lp[0]),
                                TMS_IEEE(lp[1]),
                                TMS_IEEE(lp[2]),
                                TMS_IEEE(lp[3])
                                );
                    lp +=4;
                  }
                  break;
    }
    _setcursortype(_NOCURSOR);
    gotoxy(1,1);
    cprintf(disp);
    _setcursortype(_NORMALCURSOR);
    //
    // Release_TSlice() was made conditional on speed to make sure that
    // when speed==0 the timeslice was not given up too often.  Typicaly
    // Windows task swaps at a 20mS interval and if the application loop
    // is faster, it is permanently loosing the timeslice
    //
    //Release_TSlice();
    if(speed)
    { Release_TSlice();
      //delay(20);
      start = clock();
      for(;;)
      { end = clock();
        if((end - start) > speed) break;
      }
    }
  }
}

