/*
   HEX2ROM.EXE HEX format to popular EPROM format conversion utility

   Formats supported by HEX2ROM include

      INTEL, MOTOROLA, TI-TAG,ASCII and TEKTRONIX

      Boot width is determined in hex file
      Buswidth is always 32
      ROM width determins splitting
*/
#include <bios.h>
#include <dos.h>
#include <conio.h>
#include <stdio.h>
#include <stdlib.h>
#include "dsk.h"
#include "keydef.h"
#include "dsk_coff.h"
#define CLR_1 LIGHTGRAY  /* Normal  text color */
#define CLR_2    WHITE   /* Hilight text color */

ulong Address     = 0;
int Boot_Width    =-1;
int  SWAP         = 0;
int  BYTE         = 0;
char file_to_load[80]; // Name of file to be loaded
int  ROM          = 1;
int  ROMS         = 1;
int  WIDTH        = 8;
FILE *hex_infile;
FILE *rom_outfile;

typedef enum   HEX2ROMTASK{SAMPLE, WRITE,NOCVRT};
typedef enum FORMAT{INTEL,MOT_S1,MOT_S2,MOT_S3,TI,TEKTRONIX,ASCII};
FORMAT form=INTEL;
//-----------------------------------------
char nib2hex(char c)
{
  c&=0xF;
  switch(c)
  {
    default:
    case  0x0: return '0';
    case  0x1: return '1';
    case  0x2: return '2';
    case  0x3: return '3';
    case  0x4: return '4';
    case  0x5: return '5';
    case  0x6: return '6';
    case  0x7: return '7';
    case  0x8: return '8';
    case  0x9: return '9';
    case  0xA: return 'A';
    case  0xB: return 'B';
    case  0xC: return 'C';
    case  0xD: return 'D';
    case  0xE: return 'E';
    case  0xF: return 'F';
  }
}
int hex2int(char c)
{
//c&=0xF;
  switch(c)
  {
    default:
    case  '0': return 0x0;
    case  '1': return 0x1;
    case  '2': return 0x2;
    case  '3': return 0x3;
    case  '4': return 0x4;
    case  '5': return 0x5;
    case  '6': return 0x6;
    case  '7': return 0x7;
    case  '8': return 0x8;
    case  '9': return 0x9;
    case  'A': return 0xA;
    case  'B': return 0xB;
    case  'C': return 0xC;
    case  'D': return 0xD;
    case  'E': return 0xE;
    case  'F': return 0xF;
  }
}

void output(HEX2ROMTASK task, char *ostrg)
{
  switch(task)
  {
    case NOCVRT: break;
    case SAMPLE: cputs(ostrg); break;
    case  WRITE: fputs(ostrg,rom_outfile); break;
                 //cprintf(ostrg); break;
  }
}
//-------------------------------------------------------------------------
// bytejuggler swaps around the individual bytes to account for SWAP
// ordering, rom width and the number of roms
//-------------------------------------------------------------------------
// Width  Swap| roms=1       roms=2            roms=4
//            | r1           r1   r2         r1 r2 r3 r4
// -----------------------------------------------------
// 8 bit   0  | 0x11223344   1133 2244       11 22 33 44
// 8 bit   1  | 0x44332211   3311 4422       44 33 22 11
//            |
// 16 bit  0  | 0x11223344   1122 3344       N/A
// 16 bit  1  | 0x33441122   3344 1122
//            |
// 32 bit  0  | 0x11223344   N/A             N/A
// 32 bit  1  | 0x11223344
//-------------------------------------------------------------------------
ulong bytejuggler(ulong l)
{
  ulong t;
  if(WIDTH==32) return l;
  //-----------------------------------------------
  if(WIDTH==16)
  { if((ROMS==1)||(ROMS==2)) /* Always true */
    { if(SWAP)
      { t =  ((l&0xFFFF0000L)>>16)      // 1122 3344
            |((l&0x0000FFFFL)<<16);     // 3344 1122
        l = t;
      }
    }
  }
  //-----------------------------------------------
  if(WIDTH==8)
  { if(ROMS==1)
    { if(SWAP)
      { t =  ((l&0xFF000000L)>>24)      //  11223344
            |((l&0x00FF0000L)>> 8)      //  44332211
            |((l&0x0000FF00L)<< 8)
            |((l&0x000000FFL)<<24);
        l = t;
      }
    }
    if(ROMS==2)
    { if(SWAP)
      { t = ((l&0xFF000000L)>> 8)      //  11223344
           |((l&0x00FF0000L)>>16)      //  33114422
           |((l&0x0000FF00L)<<16)
           |((l&0x000000FFL)<< 8);
        l = t;
      }
      else
      { t = ((l&0xFF000000L)>> 0)      //  11223344
           |((l&0x00FF0000L)>> 8)      //  11332244
           |((l&0x0000FF00L)<< 8)
           |((l&0x000000FFL)<< 0);
        l = t;
      }
    }
    if(ROMS==4)
    { if(SWAP)
      { t =  ((l&0xFF000000L)>>24)      //  11223344
            |((l&0x00FF0000L)>> 8)      //  44332211
            |((l&0x0000FF00L)<< 8)      //
            |((l&0x000000FFL)<<24);     //
        l = t;
      }
    }
  }
  return l;
}
//-----------------------------------------------------------------------
// extract_bytes() takes a long number l and writes its byte values to
// either the screen or a file ouput.  The ordering of the bytes that
// are eventualy displayed or written to a file are swapped using the
// byte_juggler() function.
//
// NOTE: The ASCII file format is not compatible with this function
// and is handled as a special case.
//
// Width  Swap| roms=1       roms=2            roms=4
//            | r1           r1   r2         r1 r2 r3 r4
// -----------------------------------------------------
// 8 bit   0  | 0x11223344   1133 2244       11 22 33 44
// 8 bit   1  | 0x44332211   3311 4422       44 33 22 11
//            |
// 16 bit  0  | 0x11223344   1122 3344       N/A
// 16 bit  1  | 0x33441122   3344 1122
//            |
// 32 bit  0  | 0x11223344   N/A             N/A
// 32 bit  1  | 0x11223344
//------------------------------------------------------------------------
char *extract_bytes
(ulong l,char *c,int rom,int roms,int width,ulong *chks, int *BCNT,char *t)
{
  char *s;
  s = c;
  l = bytejuggler(l);  // Swap bytes
  if(roms==1)        // build 8,16 or 32 bit single rom
  {
    if((form==TI)            ) *s++='B';
    s+=sprintf(s,"%02lX%s",(l>>24)&0xFF,t); *chks += ((l>>24) & 0xFF);
    if((form==TI)&&(width==8)) *s++='B';
    s+=sprintf(s,"%02lX%s",(l>>16)&0xFF,t); *chks += ((l>>16) & 0xFF);
    if((form==TI)&&((width==8)||(width==16))) *s++='B';
    s+=sprintf(s,"%02lX%s",(l>> 8)&0xFF,t); *chks += ((l>> 8) & 0xFF);
    if((form==TI)&&(width==8)) *s++='B';
    s+=sprintf(s,"%02lX%s",(l>> 0)&0xFF,t); *chks += ((l>> 0) & 0xFF);
  }
  if(roms==2) // build 8 or 16 bit double roms
  {
    if((form==TI)            ) *s++='B';
    if(rom==1)
    {
      s+=sprintf(s,"%02lX%s",(l>>24)&0xFF,t); *chks += ((l>>24) & 0xFF);
      if((form==TI)&&(width==8)) *s++='B';
      s+=sprintf(s,"%02lX%s",(l>>16)&0xFF,t); *chks += ((l>>16) & 0xFF);
    }
    if(rom==2)
    {
      s+=sprintf(s,"%02lX%s",(l>> 8)&0xFF,t); *chks += ((l>> 8) & 0xFF);
      if((form==TI)&&(width==8)) *s++='B';
      s+=sprintf(s,"%02lX%s",(l>> 0)&0xFF,t); *chks += ((l>> 0) & 0xFF);
    }
    *BCNT-=2; // Wrote two bytes
  }
  if(roms==4)  // build four 8 bit roms
  {
    if((form==TI)            ) *s++='B';
    if(rom==1) {s+=sprintf(s,"%02lX%s",(l>>24)&0xFF,t);*chks+=((l>>24)&0xFF);}
    if(rom==2) {s+=sprintf(s,"%02lX%s",(l>>16)&0xFF,t);*chks+=((l>>16)&0xFF);}
    if(rom==3) {s+=sprintf(s,"%02lX%s",(l>> 8)&0xFF,t);*chks+=((l>> 8)&0xFF);}
    if(rom==4) {s+=sprintf(s,"%02lX%s",(l>> 0)&0xFF,t);*chks+=((l>> 0)&0xFF);}
    *BCNT-=3; // Wrote one byte
  }
  return s;
}
MSGS formatter(char *filename,HEX2ROMTASK task,int y,int rom,int roms,int width,FORMAT form)
{
  #define slen  120
  ulong chks;
  MSGS err = NO_ERR;
//ulong t;
  ulong l;
  long curpos;
  int RTYP;
  int line = 0;
  int romx;
  ulong ADDR;
  char strg[slen];
  char ostrg[slen];
  char *p;
  char *s;
  char *n;
  int BCNT;
  int TBCNT;
  int MAXBCNT;

  if(rom  ==-1) return NO_ERR; /* Temp to keep away warning msgs */
  if(width==-1) return NO_ERR; /* Temp to keep away warning msgs */
  //----------------------------------------------------------
  // Append .HEX to file... always *.HEX no others are allowed!
  //----------------------------------------------------------
  if((p=strstr(filename,"."))!=NULL) strcpy(p,".HEX");
  else  strcat(filename,".HEX");
  //----------------------------------------------------------
  // Open hex output file to be used for fast loads
  //----------------------------------------------------------
  if((hex_infile = fopen(filename,"rb"))==NULL)
  {
    fcloseall();
    return FATAL_OPEN;
  }
  //-------------------------------------------------------------------
  // From a bootable imamage a file can be simply loaded to an EEPROM
  // if the EEPROM is properly configured to accept data writes
  // The proper address and timing is all that is required
  //-------------------------------------------------------------------
  fgets(strg,slen,hex_infile);
  if((p=strstr(strg,"\r"))!=NULL) *p=0;
  if((p=strstr(strg,"\n"))!=NULL) *p=0;
  if(strstr(strg,"FILE2HEX")==NULL)   // C32 not supported yet
  { fcloseall();
    return OPEN_ERR;  // A better error return MSG would be nice
  }
  gotoxy(1,y+1);
//ADDR =  (int) (Address & 0xFFFF);
  ADDR =  Address;

  curpos = ftell(hex_infile);
  n=fgets(strg,slen,hex_infile);
  if(n!=strg) {fcloseall(); return OPEN_ERR;}

  Boot_Width = (int)strtoul(strg,&p,0);
  if(task==NOCVRT) {fcloseall(); return NO_ERR;}
  //
  // This is the bootwidth as reported by the HEX file
  //
  fseek(hex_infile, curpos, SEEK_SET);
  //
  // Open output files with names based on which ROM is being
  // converted and which ROM format is being used
  //

  if(SWAP) romx = rom;        // HEX30 changes the rom names during a swap
  else     romx = roms-rom+1; //

  if(task==WRITE)
  {
    switch(form)
    {
      case    INTEL : sprintf(ostrg,"ROM_%d_%d.INT",romx,roms); break;
      case    MOT_S1: sprintf(ostrg,"ROM_%d_%d.MT1",romx,roms); break;
      case    MOT_S2: sprintf(ostrg,"ROM_%d_%d.MT2",romx,roms); break;
      case    MOT_S3: sprintf(ostrg,"ROM_%d_%d.MT3",romx,roms); break;
      case        TI: sprintf(ostrg,"ROM_%d_%d.TAG",romx,roms); break;
      case TEKTRONIX: sprintf(ostrg,"ROM_%d_%d.TEK",romx,roms); break;
      case     ASCII: sprintf(ostrg,"ROM_%d_%d.ASC",romx,roms); break;
    }
    if((rom_outfile = fopen(ostrg,"wb"))==NULL)
    {
      fcloseall();
      return FATAL_OPEN;
    }
  }

  //
  // The following commented out section will create and use output
  // filenames similar to those used by the HEX30 ROM utility.
  // However, as they are non-descriptive of how they were built,
  // and conflict with regression matching tests with HEX30, they
  // are not used.
  //
 /*
  if(task==WRITE)
  {
    strcpy(strg,file_to_load);
    p = strstr(strg,".");
    switch(form)
    {
      case    INTEL : sprintf(p,".I%d",romx-1); break;
      case    MOT_S1: sprintf(p,".M%d",romx-1); break;
      case    MOT_S2: sprintf(p,".M%d",romx-1); break;
      case    MOT_S3: sprintf(p,".M%d",romx-1); break;
      case        TI: sprintf(p,".T%d",romx-1); break;
      case TEKTRONIX: sprintf(p,".T%d",romx-1); break;
      case     ASCII: sprintf(p,".A%d",romx-1); break;
    }
    if((rom_outfile = fopen(strg,"wb"))==NULL)
    {
      fcloseall();
      return FATAL_OPEN;
    }
  }
  */
  //
  // Begin rom output formatting
  //
  s = ostrg;
  if(task==SAMPLE) MAXBCNT=32;
  else             MAXBCNT=32;

  switch(form)
  {
    /**************************************************************/
    /* INTEL FORMATTED OUTPUT                                     */
    /**************************************************************/
    case INTEL:
    if(ADDR >> 16)
    {
      s= ostrg;
      s+=sprintf(s,":"                    ); // start char
      s+=sprintf(s,"02"                   ); // byte count
      s+=sprintf(s,"0000"                 ); // dummy ROM address for 16 lsbs
      s+=sprintf(s,"04"                   ); // extended record type
      s+=sprintf(s,"%04lX",    (ADDR>>16)&0xFFFF); // 16 MSB's of ROM address
      chks = 2 + 4 + ((ADDR>>16)&0xFF) + ((ADDR>>24)&0xFF); // Checksum
      chks *= -1;
      s+=sprintf(s,"%02X\r\n",chks&0xFF); // Checksum
      output(task,ostrg);
    }
    for(line=0;line<10000;line++)
    {
      RTYP = 0x00; /* Data type */
      chks  = ((ADDR>>0)&0xFF);
      chks += ((ADDR>>8)&0xFF);
      chks += ((RTYP>>0)&0xFF);
      s = ostrg;
      s += sprintf(s,":"        );   // start char
      s += sprintf(s,"00"       );   // byte count  (this gets changed)
      s += sprintf(s,"%04lX",ADDR&0xFFFF);   // ROM offset address (16 lsbs)
      s += sprintf(s,"%02X",RTYP);   // record type
      //
      // Convert up to 16 bytes of hex line values (one 32 bit value each)
      //
      for(BCNT=0;BCNT<MAXBCNT;BCNT+=4)    // max of 4 32 bit words per line
      {
        n=fgets(strg,slen,hex_infile);
        if(n!=strg) break;
        l = strtoul(strg,&p,0);
        //
        // Override the width shown in the converted HEX file to match user
        //
        if((line==0)&&(BCNT==0)) l = ROMS*WIDTH;
        s=extract_bytes(l,s,rom,roms,width,&chks,&BCNT,"");
      }
      //
      // Update the address according to rom words actualy written
      //
 if(BYTE) ADDR += (float)BCNT*(8.0/(float)8.0); // Count address in bytes
 else     ADDR += (float)BCNT*(8.0/(float)width);
      //
      // Patch the output string and send to output
      //
      s = ostrg+1;
      *s++ = nib2hex(BCNT >>4);            // redo byte count
      *s++ = nib2hex(BCNT    );
      chks += ((BCNT>>0)&0xFF);
      chks *= -1;                          // Take 2's compliment
      sprintf(strg,"%02lX\r\n",chks&0xFF); // concatenate byte checksum
      strcat(ostrg,strg);
      output(task,ostrg);
      if((task==SAMPLE)&&(line > 4)) break;
      if(n==NULL) break;
    }
    sprintf(ostrg,":00000001FF\r\n"); // End of record
    output(task,ostrg);
    break;
    /**************************************************************/
    /* TI TAG FORMATTED OUTPUT                                    */
    /**************************************************************/
    case TI       :
    s = ostrg;
    s += sprintf(s,"K000COFFTOTI\r\n");  // Program identifier
    output(task,ostrg);
    for(line=0;line<10000;line++)
    {
      chks  = 0;
      s = ostrg;
      s += sprintf(s,"9"         );  // address follows
      s += sprintf(s,"%04lX",ADDR&0xFFFF);  // address
      //
      // Convert up to 16 bytes of hex line values (one 32 bit value each)
      //
      MAXBCNT=16;
      for(BCNT=0;BCNT<MAXBCNT;BCNT+=4)    // max of 4 32 bit words per line
      {
        n=fgets(strg,slen,hex_infile);
        if(n!=strg) break;
        l = strtoul(strg,&p,0);
        //
        // Override the width shown in the converted HEX file to match user
        //
        if((line==0)&&(BCNT==0)) l = ROMS*WIDTH;
        s=extract_bytes(l,s,rom,roms,width,&chks,&BCNT,"");
      }
      //
      // Update the address according to rom words actualy written
      //
  if(BYTE) ADDR += (float)BCNT*(8.0/(float)8.0); // Count ADDR in bytes
  else     ADDR += (float)BCNT*(8.0/(float)width);

      strcat(ostrg,"7");
      chks = 0;
      s = ostrg;
      while(*s)           // 2's compliment sum of ASCII values
        chks+= *s++;
      chks *= -1;
      sprintf(strg,"%04lXF\r\n",chks&0xFFFF); // 16 bit checksum
      strcat(ostrg,strg);
      output(task,ostrg);
      if((task==SAMPLE)&&(line > 4)) break;
      if(n==NULL) break;
    }
    sprintf(ostrg,":\r\n"); // End of record
    output(task,ostrg);
    break;
    /**************************************************************/
    /* TEKTRONIX FORMATTED OUTPUT                                 */
    /**************************************************************/
    case TEKTRONIX:
    s= ostrg;
    for(line=0;line<10000;line++)
    {
      s= ostrg;
      *s++='%';                        // start char
      s+=sprintf(s,"00"         );   // block length (this gets changed)
      s+=sprintf(s,"6"          );   // block type   (data)
      s+=sprintf(s,"00"         );   // checksum     (this gets changed)
      s+=sprintf(s,"8"          );   // 8 char hex address field
      s+=sprintf(s,"%08lX", ADDR);// address
      //
      // Convert up to 16 bytes of hex line values (one 32 bit value each)
      //
      chks = 0;
      for(BCNT=0;BCNT<MAXBCNT;BCNT+=4)    // max of 4 32 bit words per line
      {
        n=fgets(strg,slen,hex_infile);
        if(n!=strg) break;
        l = strtoul(strg,&p,0);
        //
        // Override the width shown in the converted HEX file to match user
        //
        if((line==0)&&(BCNT==0)) l = ROMS*WIDTH;
        s=extract_bytes(l,s,rom,roms,width,&chks,&BCNT,"");
      }
      //
      // Update the address according to rom words actualy written
      //

if(BYTE) ADDR += (float)BCNT*(8.0/(float)8.0);  // Count ADDR in bytes
else     ADDR += (float)BCNT*(8.0/(float)width);
      //
      // Patch the output string and send to output
      //
      s = ostrg+1;
      BCNT = strlen(ostrg)-1;
      *s++ = nib2hex(BCNT >>4);        // redo byte count
      *s++ = nib2hex(BCNT    );
      //
      chks = 0;
      s = ostrg+1;
      while(*s)
        chks+=hex2int(*s++);
      s = ostrg+4;
      *s++ = nib2hex(chks >>4);        // redo checksum
      *s++ = nib2hex(chks    );
      //
      //
      strcat(ostrg,"\r\n");
      output(task,ostrg);
      if((task==SAMPLE)&&(line > 4)) break;
      if(n==NULL) break;
    }
    ostrg[0]='%';
    sprintf(ostrg+1,"0E81E800000000\r\n"); // End of record
    output(task,ostrg);
    break;
    /**************************************************************/
    /* ASCII FORMATTED OUTPUT                                     */
    /**************************************************************/
    case ASCII:
    s = ostrg;
    *s++ = 2;                             // Program identifier
    s += sprintf(s," $A%04lX,\r\n",ADDR&0xFFFF);  // address
    output(task,ostrg);
    for(line=0;line<10000;line++)
    {
      // Convert up to 16 bytes of hex line values (one 32 bit value each)
      s = ostrg;
      *s=0;
      MAXBCNT=16;
      for(BCNT=0;BCNT<MAXBCNT;BCNT+=4)    // max of 4 32 bit words per line
      {
        n=fgets(strg,slen,hex_infile);
        if(n!=strg) break;
        l = strtoul(strg,&p,0);
        //
        // Override the width shown in the converted HEX file to match user
        //
        if((line==0)&&(BCNT==0)) l = ROMS*WIDTH;
        l = bytejuggler(l);
        //
        // Need to add support for non multiple ROM 32 bit
        //
     // s+=sprintf(s,"%02lX ",(l>>24)&0xFF);
     // s+=sprintf(s,"%02lX ",(l>>16)&0xFF);
     // s+=sprintf(s,"%02lX ",(l>> 8)&0xFF);
     // s+=sprintf(s,"%02lX ",(l>> 0)&0xFF);
        s=extract_bytes(l,s,rom,roms,width,&chks,&BCNT," ");
      }
      if(ostrg[0]==0) break;
      s+=sprintf(s,"\r\n");
      output(task,ostrg);
      if((task==SAMPLE)&&(line > 4)) break;
      if(n==NULL) break;
    }
    s = ostrg;
    *s++ = 3;
    *s++ = 0;
    output(task,ostrg);
    break;
    /**************************************************************/
    /* MOTOROLA FORMATTED OUTPUT                                  */
    /**************************************************************/
    case MOT_S1:
    case MOT_S2:
    case MOT_S3:
    s= ostrg;
    s+=sprintf(s,"S0"                     ); // Header record
    s+=sprintf(s,"00"                     ); // byte count
    s+=sprintf(s,"%04lX",      ADDR&0xFFFF); // address
    s+=sprintf(s,"000000"                 ); // Header field
    s+=sprintf(s,"%02lX\r\n",6+(ADDR&0xFF)); // byte checksum
    output(task,ostrg);
    for(line=0;line<10000;line++)
    {
      s= ostrg;
      switch(form)
      {
        case MOT_S3:
        s+=sprintf(s,"S3"                    ); // 32 bit (S3) record
        s+=sprintf(s,"00"                    ); // byte count (gets changed)
        s+=sprintf(s,"%08lX",  ADDR          ); // S3 32 bit address
        break;
        case MOT_S2:
        s+=sprintf(s,"S2"                    ); // 24 bit (S3) record
        s+=sprintf(s,"00"                    ); // byte count (gets changed)
        s+=sprintf(s,"%06lX",  ADDR&0xFFFFFFL); // S3 32 bit address
        break;
        case MOT_S1:
        s+=sprintf(s,"S1"                    ); // 16 bit (S3) record
        s+=sprintf(s,"00"                    ); // byte count (gets changed)
        s+=sprintf(s,"%04lX",  ADDR&0xFFFF   ); // S3 32 bit address
        break;
      }
      //
      // Convert up to 16 bytes of hex line values (one 32 bit value each)
      //
      chks=0;
      for(BCNT=0;BCNT<MAXBCNT;BCNT+=4)    // max of 4 32 bit words per line
      {
        n=fgets(strg,slen,hex_infile);
        if(n!=strg) break;
        l = strtoul(strg,&p,0);
        //
        // Override the width shown in the converted HEX file to match user
        //
        if((line==0)&&(BCNT==0)) l = ROMS*WIDTH;
        s=extract_bytes(l,s,rom,roms,width,&chks,&BCNT,"");
      }
      //
      // Patch the output string and send to output
      //
//    TBCNT = BCNT;
      TBCNT = 1+((strlen(ostrg)-4)/2);
      s = ostrg+2;
      *s++ = nib2hex(TBCNT >>4);        // redo byte count
      *s++ = nib2hex(TBCNT    );
      //
      chks += TBCNT;
      chks += ( ADDR     &0xFF);
      chks += ((ADDR>> 8)&0xFF);
//    chks += ((ADDR>>16)&0xFF);
//    chks += ((ADDR>>24)&0xFF);

 if(BYTE) ADDR += (float)BCNT*(8.0/(float)8.0);  // Address count is in bytes
 else     ADDR += (float)BCNT*(8.0/(float)width);
      //
      // Update the address according to rom words actualy written
      //
      chks ^=0xFF;
      sprintf(strg,"%02X\r\n",chks&0xFF);
      strcat(ostrg,strg);
      output(task,ostrg);
      if((task==SAMPLE)&&(line > 4)) break;
      if(n==NULL) break;
    }
    switch(form)
    {
      case MOT_S3: sprintf(ostrg,"S70500000000FA\r\n");break;// Term 32 bit (S3)
      case MOT_S2: sprintf(ostrg,"S804000000FD\r\n");  break;// Term 24 bit (S2)
      case MOT_S1: sprintf(ostrg,"S9030000FC\r\n");    break;// Term 16 bit (S1)
    }
    output(task,ostrg);
    break;
    /**************************************************************/
  }
  fcloseall();
  return err;
}
/*----------------------------------------------------------*/
/* Trapped key actions are defined here                     */
/*----------------------------------------------------------*/
void check_key(void)
{
  FILE *file;
  char file_to_write[40];
  char strg[80];
  int key;
  int rom;
//int romx;
//int   x;
  MSGS err;
  ulong tmp;
  char *p;
  key = bioskey(0) & 0xFF00;
  switch(key)
  {
    case  _Q  : exit(0);                  // Quit
    case  _S  : SWAP^=1; break;
    case  _B  : BYTE^=1; break;
    case  _F  : clrscr();
                cprintf("New filename : ");
                scanf("%s",file_to_load);
                strupr(file_to_load);
                //-----------------------------------
                // Append .HEX to file... *.HEX only!
                //-----------------------------------
                if((p=strstr(file_to_load,"."))!=NULL)
                {
                  if(strstr(file_to_load,".HEX")==NULL)
                  {
                    printf("Converting %s to HEX format\r\n",file_to_load);
                    err = Load_File(file_to_load,FILE2HEX);
                    if(err==NO_ERR) printf("Conversion complete\r\n");
                    else
                    {
   printf("\r\n>>> Conversion error\r\n>>> %s\r\nHit any key",Error_Strg(err));
                      getch();
                    }
                  }
                  strcpy(p,".HEX");
                }
                else  strcat(file_to_load,".HEX");
                clrscr();
                if((file = fopen(file_to_load,"rb"))==NULL)
                {
                  printf("%s does not exist",file_to_load);
                  while(!kbhit());
                }
                fclose(file);
                // Dummy load file to set HEX file boot width
                formatter(file_to_load,NOCVRT,1,ROM,ROMS,WIDTH,form);
                clrscr();
                break;
    case  _W  : printf("HEX->EEPROM Write in progress, please wait");
                for(rom=1;rom<ROMS+1;rom++)
                {
                err = formatter(file_to_load,WRITE,0,rom,ROMS,WIDTH,form);
                if(err!=NO_ERR) break;
                }
                if(err!=NO_ERR)
                {
                textcolor(CLR_2);  // Hilight the error
                cprintf("  --> ");
                textcolor(CLR_1);
                getch();
                }
                clrscr();
                break;       // Program the EEPROM (Download the HEX file)
    case  _R  : cprintf("New address 0x%08lx",Address);
                gotoxy(wherex()-8,wherey());
                scanf("%lx",&tmp);
                Address = tmp;
            //  DumpAddr = Address;
                clrscr();
                break;
    case  _1  : ROM  +=1 ;
                if(ROM>ROMS) ROM  =1;
                break;
    case  _2  : ROMS *=2;
                if(ROMS > 32/WIDTH) ROMS =1;
                if(ROM  > ROMS) ROM  =1;
              //if(ROMS < ROM ) ROM  =1;
                break;
    case  _3  : WIDTH*=2 ;
                if(WIDTH>32) WIDTH=8;
                if(ROMS > 32/WIDTH) ROMS =1;
                if(ROM  > ROMS) ROM  =1;
                break;
    case  _G  : // Generate a HEX30 command file
                //
                strcpy(file_to_write,"HEX30.CMD");
                if((file = fopen(file_to_write,"wb"))==NULL)
                {
                  printf("Cannot open output file: ",file_to_write);
                  while(!kbhit());
                  break;
                }

                //
                // Write commands line by line to file
                //
          fprintf(file,"/************************************************/\r\n"
                  "/* HEX2ROM Auto generated HEX30 command file    */\r\n"
                  "/************************************************/\r\n");
                strcpy(strg,file_to_load);
                if((p=strstr(strg,"."))!=NULL)  strcpy(p   ,".OUT");
                else                            strcat(strg,".OUT");
                fprintf(file,"%s\r\n",strg);
                //
                // If needed, rename the output file here
                //
                /*
                p=strstr(strg,".");
                switch(form)
                {
                  case INTEL    : strcpy(p,".i"); break;
                  case MOT_S1   :
                  case MOT_S2   :
                  case MOT_S3   : strcpy(p,".m"); break;
                  case TEKTRONIX: strcpy(p,".k"); break;
                  case TI       : strcpy(p,".t"); break;
                  case ASCII    : strcpy(p,".a"); break;
                }
                fprintf(file,"-o %s\r\n",strg);
                */
                fprintf(file,"-memwidth  %d\r\n",ROMS*WIDTH); //ROMS*WIDTH);
                fprintf(file,"-romwidth  %d\r\n",     WIDTH);
                fprintf(file,"-datawidth %d\r\n",32);
                fprintf(file,"-boot\r\n");
                fprintf(file,"-bootorg   0x%08lx\r\n",(long)Address);
                fprintf(file,"-e         0x%08lX\r\n",(long)ENTRY);
                if(TARGET==C32_DSK)
                {
                  fprintf(file,"-iostrb    0x%08lx\r\n",(long)IOSTRB);
                  fprintf(file,"-strb0     0x%08lx\r\n",(long)STRB0);
                  fprintf(file,"-strb1     0x%08lx\r\n",(long)STRB1);
                }
                else
                {
                  fprintf(file,"-cg        0x000010F8\r\n");
                }
                switch(form)
                {
                  case INTEL    : fprintf(file,"-i \r\n",file_to_load); break;
                  case MOT_S1   : fprintf(file,"-m1\r\n",file_to_load); break;
                  case MOT_S2   : fprintf(file,"-m2\r\n",file_to_load); break;
                  case MOT_S3   : fprintf(file,"-m3\r\n",file_to_load); break;
                  case TEKTRONIX: fprintf(file,"-x \r\n",file_to_load); break;
                  case TI       : fprintf(file,"-t \r\n",file_to_load); break;
                  case ASCII    : fprintf(file,"-a \r\n",file_to_load); break;
                }
                if(SWAP) fprintf(file,"-order LS\r\n");
                else     fprintf(file,"-order MS\r\n");
                if(BYTE) fprintf(file,"-byte\r\n");

                fcloseall();
                //
                // Spawn a command shell to call HEX30 and make the files
                //
                /*
                int x;
                if(kbhit()) getch();
                clrscr();
                x = system("HEX30 HEX30.CMD");
                if(x==-1)
                {
                  printf("system call error\r\n");
                  perror("HEX30 HEX30.CMD");
                  while(!kbhit());
                }
                while(!kbhit());
                */
                break;
    case  _4  : switch(form)
                {
                  case INTEL    : form = MOT_S1   ; break;
                  case MOT_S1   : form = MOT_S2   ; break;
                  case MOT_S2   : form = MOT_S3   ; break;
                  case MOT_S3   : form = TEKTRONIX; break;
                  case TEKTRONIX: form = TI       ; break;
                  case TI       : form = ASCII    ; break;
                  case ASCII    : form = INTEL    ; break;
                }

    default   : break;
  }
}
//----------------------------------------
void main(void)
{
  int y;
  int i;
  int romx;
  Scan_Command_line("HEX2ROM.EXE");
  for(;;)
  {
    Address = 0x0L;
    strcpy(file_to_load,"NOFILE.HEX");
    textbackground(BLACK);
    //
    for(;;)
    {
      clrscr();
      _setcursortype(_NOCURSOR);     // Avoid cursor flash
      textcolor(CLR_1);
      cprintf(
"DSK,OUT & HEX ROM FILE CONVERTER      All input files are converted to *.HEX\r\n"
"----------------------------------------------------------------------------\r\n");
      if(SWAP) romx = ROM;          // HEX30 swaps rom names
      else     romx = ROMS-ROM+1;
         cprintf("(Q)uit\r\n");
         cprintf("(F)ilename            = %s      ENTRY=0x%08lx\r\n",file_to_load,ENTRY);
         cprintf("(R)OM address start   = %08lx\r\n",Address);

if(BYTE) cprintf("(B)yte counting       = ENABLE\r\n");
else     cprintf("(B)yte counting       = DISABLE\r\n");

if(SWAP) cprintf("(S)wap bytes          = LS FIRST\r\n");
else     cprintf("(S)wap bytes          = MS FIRST\r\n");
         cprintf("(1)ROM                = %d (of %d ROMS)  \r\n",romx,ROMS);
         cprintf("(2)ROMS               = %d (installed in system)\r\n",ROMS);
         cprintf("(3)ROM Width          = %d (bits each ROM)  \r\n",WIDTH);
         cprintf("(4)Format             = ");
      switch(form)
      {
        case     INTEL: cprintf("INTEL      \r\n"); break;
        case        TI: cprintf("TI-TAG     \r\n"); break;
        case TEKTRONIX: cprintf("TEKTRONIX  \r\n"); break;
        case MOT_S1   : cprintf("MOTOROLA S1\r\n"); break;
        case MOT_S2   : cprintf("MOTOROLA S2\r\n"); break;
        case MOT_S3   : cprintf("MOTOROLA S3\r\n"); break;
        case     ASCII: cprintf("ASCII      \r\n"); break;
      }
      cprintf("Hex file boot width   = %d bits \r\n",Boot_Width);
      cprintf("Boot width            = %d bits \r\n",ROMS*WIDTH);
      cprintf("(W)rite EPROM file\r\n");
      cprintf("(G)enerate HEX30.CMD\r\n");
//    if(Boot_Width == ROMS*WIDTH) cprintf("                   \r\n");
//    else                         cprintf("HEXWIDTH!=BOOTWIDTH\r\n");

cprintf(
"_________________________Sample_Output_ROM_%d_of_%d___________________\r\n",
romx,ROMS);
      //
      // Show sample of output if file exists
      // Clear lower half of screen first
      y = wherey();
      for(i=0;i<8;i++)
      {
        cprintf("                                             "
                "                                  \r\n");
      }
      cprintf("                                             "
              "                                  ");
      gotoxy(1,y);
      formatter(file_to_load, SAMPLE,y-1,ROM,ROMS,WIDTH,form);//Show sample
      _setcursortype(_NORMALCURSOR);                          //Show cursor
      check_key();
    }
  }
}

