//------------------------------------------------------------
// FFT_256B.CPP
// Keith Larson
// TMS320 DSP Applications
// (c) Copyright 1995, 1996
// Texas Instruments Incorporated
//
// This is unsupported freeware code with no implied warranties or
// liabilities.  See the disclaimer document for details
//------------------------------------------------------------
//  FFT_256B.CPP is a DOS graphics based program which is a modifed
//  version of DSK_OSC.CPP.  In this case the data array is converted
//  to the frequency domain before it is sent to the PC for display
//  using the oscilliscope interface.  An FFT specific display is used
//  in the FFT applications FFT_256.EXE, FFT_512.EXE and FFT_1024.EXE
//
//  +--------------------------------------------------------------+
//  | PLEASE NOTE: THIS APPLICATION USES EGA GRAPHICS              |
//  |                                                              |
//  | YOU MUST EITHER CREATE AND LINK IN EGAVGA.OBJ AND ALSO USE   |
//  | REGISTER THE DRIVER IN YOUR CODE WITH registerbgidriver();   |
//  | -OR-                                                         |
//  | EGAVGA.BGI MUST BE IN THE CURRENT DIRECTORY                  |
//  |                                                              |
//  | -ALSO-                                                       |
//  | THE LINKER OPTIONS NEED TO BE SET TO LINK IN GRAPHICS.LIB    |
//  +--------------------------------------------------------------+
//
//  The project file should include the following
//
//    FFT_256B.CPP  This file
//    DRIVER.CPP    Low level printer port drivers
//    TARGET.CPP    DSK Command level
//    OBJECT.CPP    Application setup routines
//    DSK_COFF.CPP  DSK and COFF file loaders and other utils
//    ERRORMSG.CPP  Messages used for most function returns
//    SYMBOLS.CPP   Symbol tables (needed to link DSK_COFF)
//    TEXTWIN.CPP   DOS level text window functions
//  * EGAVGA.OBJ    EGA and VGA graphics driver
//
//  Borland C++ version 3.1 setup (use defaults if not shown)
//
//    Compiler
//     Memory Model      - Large        (objects reused in DSK3D link)
//     Processor         - 80286        (old PC support)
//     Floating Point    - Emulation
//     Code Gen          - DOS Standard (works in Win DOS prompt box)
//     Calling convention- C
//     Optimization      - Speed        (code reduction is not significant)
//
//    Linker
//     Output              - DOS Standard EXE   (Not a windows app!)
//     Libraries           - Graphics
//     Object Windows Lib  - None
//     Container Class     - None
//     Standard Runtime Lib- Static
//
//  NOTE: Other than using '//' comments and other simple C++ features
//        this code follows ANSI C.  The C++ compiler is used primarily
//        for convenience as well as its performance and advanced error
//        checking and warnings.
//
//        An 80286 output is used for DSK users who are using old PC's
//        for automated 'smart' data collection boxes.
//
//  Windows 3.x/95
//        This is a DOS graphics executable which uses the timelsice
//        managment interrupt hooks to make the application more efficient
//        at swapping in and out as a Windows task.
//
//        However since this application is a DOS graphics executable
//        it must be run full screen.  Compatiblity with other background
//        applications was NOT tested.
//
//        A PIF file can be created to automate setup and execution.
//--------------------------------------------------------------
#include <math.h>
#include <float.h>
#include <dos.h>
#include <stdio.h>
#include <conio.h>
#include <stdlib.h>
#include <string.h>
#include <graphics.h>
#include <bios.h>

#include "dsk.h"
#include "keydef.h"
#include "errormsg.h"
#include "DSK_COFF.h"
#include "C3XMMRS.H"

ulong buf_0[512];
ulong buf_1[512];   // Data buffers
ulong MSG;
ulong ZERO=0;

char DSK_APP[]="FFT256B.DSK";
char DSK_EXE[]="FFT_256B.EXE";

typedef enum messages
{
  STOP =1,
  START=2
}message;

void init_graphics(void);
int  check_key(void);
void setup_menu(void);
void disp_scope(int mode);
void draw_vectors(ulong *ptr1);
void Synch_light(char ON);
int  Q15MPY(int x,int y);

#define MSG_BOX   0x809C00L
#define TRG_BOX   0x809C01L
#define A_BOX     0x809C02L
#define B_BOX     0x809C03L
#define C_BOX     0x809C04L
#define EDGESEL   0x809C05L
#define SAMPLES   0x809C06L
#define DATABLOCK 0x809900L

#define Samples 256  // Get 512 samples from DSK
#define THx 40e-9 // DSP cycle time H1/H3 clock rate

int   TA =  10;     // DAC divisors
int   TB =  14;     //
int   RA =  10;     // ADC divisors
int   RB =  14;     //
int   TAP=   1;     // TA' and RA' are not used in this application
int   RAP=   1;     //
int C_REG=0x03;     // AIC control register bits
ulong   Edge=0;

// The graph X-Y dimensions and accelerator
// constants are defined here
#define  dY     300    // Y dimension of display in pixels
int     dY2 =  dY/2;   // Y midpoint
float Ygrid = dY/10;   // Step size of grid
#define  dX     520
#define  dXd    512
int     dX2 =  dX/2;
float Xgrid = dX/10;
//
// view port definitions          (X1 Y1     X2  Y2,clip)
#define text_vwport()  setviewport( 0, 0,    80, dY,   1)
#define graph_vwport() setviewport(81, 0, 81+dX, dY,   1)
//
// The ADC conversion factor depends on the AIC internal reference
// and internal gain.  The units are V/bit
//
float Vref    = 3.25;
float Vscale =    0;

// Volts/division selection table... 0.000 indicates table end point
float gaintbl[]={0.000, 0.002, 0.005, 0.010, 0.020, 0.050, 0.100,
                        0.200, 0.500, 1.000, 2.00, 0.000 };

float *v_per_div = &gaintbl[7]; // pointer to V/div gain setting

int   q15gain;                  // Q15 Y scaling for faster display
float uS_div=100, uS_offs=0.0;
float Fsr=1000.0, Fsx=1000.0;
ulong T0_prd_val =  1;
int   px_delay=  0;
long  n_bias = 0;    // Y pixel offset for shifting display
float V_offs = -1.3; // calculated from V_offs, gain display size etc...

typedef enum
{
  T_AUTO,      // Trgr is set to x% of PP
  T_DC,        // Trgr is set to x% of full scale
  T_NORM,      // DC trigger
  T_WRAP
}Trg_modes;
Trg_modes Trg_mode = T_AUTO;

long TLVL = 50;
long Nmin =  0;
long Nmax =  0;
long TLVL_V;

long inline clip(long x, long min, long max)
{
  if(x<min) return min;
  if(x>max) return max;
  return x;
}
#define A_REG  ((TA <<9)+(RA <<2)+0) // A divisors.. set SCF rate
#define AP_REG ((TAP<<9)+(RAP<<2)+1) // TA RA prime registers (not used)
#define B_REG  ((TB <<9)+(RB <<2)+2) // B divisors

void calc_trgr(void)
{
  switch(Trg_mode)
  {
    case T_DC   : TLVL_V= -32768.0 + 65536.0*TLVL/100.0 ; break;//% fl scale
    case T_AUTO : TLVL_V= Nmin+ (Nmax - Nmin)*TLVL/100.0; break;//% pp
    case T_NORM : TLVL_V= Nmin+ (Nmax - Nmin)*TLVL/100.0; break;
    default     : break;
  }
}
void main(void)
{
  MSGS err;
  int New_Params = 1;
  int reset_flag = 0;
//
  ulong  *tmp2, *tmpx, aic, *tmp1;
  clrscr();
  Scan_Command_line(DSK_EXE);
  Detect_Windows();
  //
  // An outer loop is used in case of momentary power
  // loss or data corruption.  If a failure ocurs the
  // inner loop will exit and the DSK will be reinitialized
  for(;;)
  {
    // Download the communications kernel
    for(;;)
    { if(Init_Communication(10000) == NO_ERR) break;
      if(kbhit()) exit(0);
    }
    // Load applications code
    // If running, Halt the DSK before attempting to load a file
    HALT_CPU();
    if((err=Load_File(DSK_APP,LOAD))!=NO_ERR)
    { printf("%s %s\n",DSK_APP,Error_Strg(err));
      exit(0);
    }
    RUN_CPU();
    // DSK is now initialized and able to communicate
    clrscr();
    init_graphics();
    tmp1 = buf_0;
    tmp2 = buf_1;
    calc_trgr();
    setup_menu();        // White draw new data vectors
    disp_scope(1);       // White draw text overlay within scope display
    calc_trgr();
    draw_vectors(tmp1);  // xor draw the data vectors
    //  The inner loop is where the application
    //  normaly interacts with the host
    for(;;)
    { // The DSK is at the top of the program waiting for the DSK to
      // respond to a host request.  Since the sampling period, or
      // AIC reprogramming can take a long time to complete an infinite
      // synch is created by bypassing the getmem timeout.
      reset_flag = 0;
      HPI_STRB(0);            // Wait for DSK to stop with full buffer
      for(;;)
      { if(kbhit())
        { draw_vectors(tmp1);   // xor undraw data vectors on top of text
          disp_scope(0);        // black undraw text within scope display
          reset_flag=check_key();//check keys, terminate here if requested
          disp_scope(1);        //white draw text within scope display
          draw_vectors(tmp1);   //xor draw data vectors on top of text
          New_Params = 1;
        }
        if(HPI_ACK())break;  // Note: break last to ensure keytrap!
        delay(10);
      }
      if(reset_flag) break;
      for(;;)
      {
        err=getmem(MSG_BOX,1,&MSG);
        if(err!=NO_ERR) break;
        if(MSG== STOP) break;
        if(MSG!=START) break;
      }
      if(MSG!= STOP) break;
      if(err!=NO_ERR) break;
      //
      // If not triggerred on last go around, set the trigger...
      //
      //{
      if(putmem(TRG_BOX,1,&(ulong)TLVL_V)!=NO_ERR) break;
      //}
      // If a key was pressed, update the AIC setup
      if(New_Params)
      {
        if(putmem(T0_prd  ,        1,&T0_prd_val)!=NO_ERR) break;
        if(putmem(T0_count,        1,      &ZERO)!=NO_ERR) break;
        if(putmem(T1_prd  ,        1,&T0_prd_val)!=NO_ERR) break;
        if(putmem(T1_count,        1,      &ZERO)!=NO_ERR) break;

        if(putmem(EDGESEL,1,&         Edge)!=NO_ERR) break;
        if(TB>TA)
        {
          aic = A_REG; if(putmem(A_BOX,1,&aic)!=NO_ERR) break;
          aic = B_REG; if(putmem(B_BOX,1,&aic)!=NO_ERR) break;
        }
        else
        {
          aic = B_REG; if(putmem(A_BOX,1,&aic)!=NO_ERR) break;
          aic = A_REG; if(putmem(B_BOX,1,&aic)!=NO_ERR) break;
        }
        aic = C_REG; if(putmem(C_BOX,1,&aic)!=NO_ERR) break;
      }
      New_Params = 0;
      if(getmem(DATABLOCK,Samples/2,       tmp2)!=NO_ERR) break;
      MSG = START; if(putmem(MSG_BOX,1,&MSG)!=NO_ERR) break;
      draw_vectors(tmp1);  // xor undraw the old data vectors
      draw_vectors(tmp2);  // xor draw the new data vectors
      tmpx = tmp1;         // A single buffer can be used if the undraw
      tmp1 = tmp2;         // function is put above getmem(..,samples,..)
      tmp2 = tmpx;         // To prevent flickering, double swapping is used
    }
    closegraph();
    printf("%s: %s\n",DSK_APP, Error_Strg(err));
    printf("Communications are being reinitialized");
    delay(1000);
    DSK_reset();
  }
}

void disp_scope(int mode)
{
  char buf[80];
  if(mode==0) setcolor(BLACK    );
  else        setcolor(LIGHTGRAY);
  sprintf(buf,"%6.1f uS/div  ",uS_div    ); outtextxy( 15,  5,buf);
  sprintf(buf,"%6.1f uS delay",uS_offs   ); outtextxy( 15, 15,buf);
  sprintf(buf,"%1.3f V/div   ",*v_per_div); outtextxy(275,  5,buf);
  sprintf(buf,"%+1.3f Voffs  ",V_offs    ); outtextxy(275, 15,buf);
  sprintf(buf,"%7.1fHz %5.1fuS Fsx",Fsx,1e6/Fsx); outtextxy( 10,260,buf);
  sprintf(buf,"%7.1fHz %5.1fuS Fsr",Fsr,1e6/Fsr); outtextxy( 10,290,buf);
  switch(Trg_mode)
  {
    case T_AUTO    : sprintf(buf,"%2d\% AUTO    ",TLVL ); break;
    case T_DC      : sprintf(buf,"%2d\% DC      ",TLVL ); break;
    case T_NORM    : sprintf(buf,"%2d\% NORM    ",TLVL ); break;
  }
  outtextxy(275,285,buf);
}

float AIC_gain = 0.25;

void binsprintf(char *s,int val)
{
  char *p;
  unsigned int t;
  p = s;
  t = 0x80;  // scan 8 bits
  for(;t>0;t=t>>1)
  {
    if(val & t) *p++ = '1';
    else        *p++ = '0';
  }
  *p=0;
}

void setup_menu(void)
{
  char buf[80];
  int Yt=0;
  // Voltage scaling constants
  switch(C_REG & 0xC0)
  {
    case 0x00:
    case 0xC0: AIC_gain = 1.00; break;
    case 0x40: AIC_gain = 0.50; break;
    case 0x80: AIC_gain = 0.25; break;
  }
  // V_offs is a measure of the amount of DC bias (%full scale)
  // added to the signal of interest.  The value n_bias is calculated
  // from V_offs such that the
  //
  n_bias = V_offs * Ygrid/ *v_per_div;
  n_bias = clip(n_bias,-32768L,32767L);
  //
  Vscale  = AIC_gain * Vref / 2.0;
  q15gain = Ygrid * Vscale / *v_per_div;    // result is dimensionless
  // Time base scaling constants

  Fsx     = 1/(2*TA*TB*(2*THx * T0_prd_val));
  if(C_REG & 0x20) Fsr = Fsx;
  else             Fsr = 1/(2*RA*RB*(2*THx * T0_prd_val));

  uS_div  = Xgrid * .5 * 1e6/Fsr; // 1/(2 pix/sample)*(samples/sec)
  uS_offs = uS_div * px_delay/(.5 * Xgrid);
  text_vwport();
  clearviewport();
  setcolor(LIGHTCYAN);            // Bright blue
  sprintf(buf,"Q-uit   "  ); outtextxy(0,Yt+=10,buf);
  sprintf(buf,"R-eset  "  ); outtextxy(0,Yt+=10,buf);
  sprintf(buf,"T-rigger"  ); outtextxy(0,Yt+=10,buf);
  if(Edge)
  sprintf(buf,"E-dge F"  );
  else
  sprintf(buf,"E-dge R"  );  outtextxy(0,Yt+=10,buf);
  sprintf(buf,"In/Del TIM"); outtextxy(0,Yt+=10,buf);
  sprintf(buf,"Hm/End V/d"); outtextxy(0,Yt+=10,buf);
  sprintf(buf,"Pu/dwn Trg"); outtextxy(0,Yt+=10,buf); Yt+=10;

  sprintf(buf,"L/R move"  ); outtextxy(0,Yt+=10,buf);
  sprintf(buf,"U/D move"  ); outtextxy(0,Yt+=10,buf); Yt+=10;

  Yt = 120;
  sprintf(buf,"Tprd=%02ld",  T0_prd_val); outtextxy(0,Yt+=10,buf);
  Yt+=10;
  sprintf(buf,"1-6 ctlbit"); outtextxy(0,Yt+=10,buf);
  binsprintf(buf,    C_REG); outtextxy(0,Yt+=10,buf);
  Yt+=10;
  sprintf(buf,"F1-2 TA"   ); outtextxy(0,Yt+=10,buf);
  sprintf(buf,"F3-4 TB"   ); outtextxy(0,Yt+=10,buf);
  sprintf(buf,"F5-6 RA"   ); outtextxy(0,Yt+=10,buf);
  sprintf(buf,"F7-8 RB"   ); outtextxy(0,Yt+=10,buf);

  Yt = 220;
  if(C_REG & 0x20)
  sprintf(buf,"SYNCH T=R" );
  else
  sprintf(buf,"ASYNCH "   ); outtextxy(0,Yt+=10,buf);
  Yt+=10;
  sprintf(buf,"TA=%02d",TA); outtextxy(0,Yt+=10,buf);
  sprintf(buf,"TB=%02d",TB); outtextxy(0,Yt+=10,buf);
  Yt+=10;
  if(C_REG & 0x20)
  {
  setcolor(RED);
  sprintf(buf,"RA=%02d",RA); outtextxy(0,Yt+=10,buf);
  sprintf(buf,"RB=%02d",RB); outtextxy(0,Yt+=10,buf);
  }
  else
  {
  sprintf(buf,"RA=%02d",RA); outtextxy(0,Yt+=10,buf);
  sprintf(buf,"RB=%02d",RB); outtextxy(0,Yt+=10,buf);
  }
  graph_vwport();
}
int check_key(void)
{ int key;
  float v_accel;
  static int old_key= 0;  // If key is hit repeatedly, speed up
  static int accel=1;
  key = bioskey(0) & 0xFF00;
  if(old_key==key) accel = accel + 1;
  else             accel = accel / 4;
  accel = clip(accel,1,200); old_key = key;

  v_accel = accel * *v_per_div/Ygrid;
  switch(key)
  {
    case _E  : Edge ^= 1;
    case _Pup: TLVL+=accel; break;
    case _Pdn: TLVL-=accel; break;
    case _Hm : v_per_div--; if(*v_per_div==0.0)v_per_div++; break;// -V/Div
    case _End: v_per_div++; if(*v_per_div==0.0)v_per_div--; break;// +V/div
    case _Ins: T0_prd_val -=accel; break;                         // ms/Div
    case _Del: T0_prd_val +=accel; break;                         // ms/div

    case _T  :switch(Trg_mode)
              {
                default:
                case T_DC  : Trg_mode = T_NORM; break;
                case T_AUTO: Trg_mode = T_DC  ; break;
                case T_NORM: Trg_mode = T_AUTO; break;
              }
              break;

    case _R  :return 1;
    case _Q  :closegraph();
              asm sti
              exit(0); // Quit
    case _Up : V_offs += v_accel ; break; // Mv up
    case _Dn : V_offs -= v_accel ; break; // Mv dn
    case _Rt : px_delay+=8; break; //
    case _Lt : px_delay-=8; break; //
    case _F1 : TA++; break;
    case _F2 : TA--; break;
    case _F3 : TB++; break;
    case _F4 : TB--; break;
    case _F5 : RA++; break;
    case _F6 : RA--; break;
    case _F7 : RB++; break;
    case _F8 : RB--; break;
    case _1  : C_REG = C_REG ^ 0x80; break; // bit 7
    case _2  : C_REG = C_REG ^ 0x40; break; // bit 6
    case _3  : C_REG = C_REG ^ 0x20; break; // bit 5
    case _4  : C_REG = C_REG ^ 0x10; break; // bit 4
    case _5  : C_REG = C_REG ^ 0x08; break; // bit 3
    case _6  : C_REG = C_REG ^ 0x04; break; // bit 2
    default : return 0;
  }
  if(V_offs >  5.0) V_offs =  5.0;
  if(V_offs < -5.0) V_offs = -5.0;
  TA         = clip(TA        ,     3,         31);
  TB         = clip(TB        ,    12,         63);
  RA         = clip(RA        ,     3,         31);
  RB         = clip(RB        ,    12,         63);

  n_bias     = clip(n_bias    ,-32768L,      32767L);
  T0_prd_val = clip(T0_prd_val,     1,         64);
  px_delay   = clip(px_delay  ,     0,Samples-128);
  TLVL       = clip(TLVL      ,     0,        100);
  setup_menu();
  calc_trgr();
  return 0;
}

void init_graphics(void)
{ int gdriver = EGA, gmode = EGAHI, errorcode, Y, X;
//
// register EGAVGA_driver which is the name of the driver in EGAVGA.OBJ
// EGAVGA.OBJ is created from EGAVGA.BGI using the BGIOBJ.EXE utility
// and is linked by either the link list or project file
//
// NOTE: This section of code can be ommitted if EGAVGA.BGI is
//       located in the applications startup directory
//
errorcode = registerbgidriver(EGAVGA_driver);
if (errorcode < 0)  // report any registration errors
{
   printf("Graphics error: %s\n", grapherrormsg(errorcode));
   printf("Press any key to halt:");
   getch();
   exit(1); // terminate with an error code
}

  initgraph(&gdriver, &gmode, "");        // if possible open EGA mode
  errorcode = graphresult();
  if (errorcode != grOk)
  { printf("Graphics error: %s\n", grapherrormsg(errorcode));
    printf("Press any key to halt:");
    getch();
    exit(1);
  }
  graph_vwport();
  setcolor(GREEN);                                 // reticle is green

  for(Y=0;Y<=   dY;Y+=   Ygrid) line(    0,   Y,   dX,   Y); // Y reticle
  for(Y=0;Y<=10*dY;Y+= 2*Ygrid) line(dX2-2,Y/10,dX2+2,Y/10); // Y Hash

  for(X=0;X<=   dX;X+=   Xgrid) line(X   ,     0,   X,    dY);  // X Reticle
  for(X=0;X<=10*dX;X+= 2*Xgrid) line(X/10,dY/2-2,X/10,dY/2+2);  // X Hash
  setcolor(WHITE);
}
//----------------------------------------------------
// ANSI C does not have fast Q15 multiply.  This
// function uses inline assembly to speed things up
//----------------------------------------------------
int Q15MPY(int x,int y)
{
  asm  mov   ax,word ptr x
  asm  imul  word ptr y
  asm  shl   dx,1
  asm  add   ax,ax
  asm  adc   dx,0
  asm  shl   dx,1
  asm  add   ax,ax
  asm  adc   dx,0
  asm  shl   dx,1
  asm  add   ax,ax
  asm  adc   dx,0

  asm  mov   word ptr x,dx
  return x;
}

inline void draw_marker(int color, int y)
{
  setcolor(color);
  line(0,y-1, 8,y-1);
  line(0,y  ,10,y  );
  line(0,y+1, 8,y+1);
}

void draw_vectors(ulong *ptr1)
{
  long x, y1, yold1;
  int *iptr, temp, mn,mx,tr,gr;
  setwritemode(1);  // XOR mode
  // Draw display vectors
  (void *)iptr = ptr1;
  iptr += px_delay;
  y1 = Q15MPY(*iptr,q15gain) + n_bias;
  yold1 =  dY2 - y1;
  setcolor(WHITE);

  Nmin =  32767L;
  Nmax = -32768L;

  for(x=0;x<dXd;x+=2)
  {
    temp = *iptr++ & 0xfffc;
    if(temp < Nmin) Nmin = temp;
    if(temp > Nmax) Nmax = temp;
    y1 = dY2 - (Q15MPY(temp,q15gain) + n_bias);
    line(x,yold1,x+1,y1);
    yold1 = y1;
  }
  // Draw the markers
  //
  calc_trgr();
  tr = dY2- Q15MPY(TLVL_V,q15gain)-n_bias; tr = clip(tr,3,dY-3);
  mn = dY2- Q15MPY(Nmin  ,q15gain)-n_bias; mn = clip(mn,3,dY-3);
  mx = dY2- Q15MPY(Nmax  ,q15gain)-n_bias; mx = clip(mx,3,dY-3);
  gr = dY2                        -n_bias; gr = clip(gr,3,dY-3);

  draw_marker(      RED, mn); // Draw Nmin/Nmax markers
  draw_marker(      RED, mx); // Draw Nmin/Nmax markers
  draw_marker(   YELLOW, tr); // Draw trigger reference
  draw_marker(LIGHTCYAN, gr); // Draw ground reference

  setwritemode(0);  // Normal video mode
  if(*ptr1 & 1) Synch_light(1);
  else          Synch_light(0);
}

char LAST_ON = 3;
void Synch_light(char ON)
{
  char color;
  if(ON == LAST_ON) return; // Do not redraw if already drawn - speedup!
  LAST_ON = ON;
  color = getcolor();
  if(ON) { setfillstyle(SOLID_FILL,LIGHTRED); setcolor(LIGHTRED); }
  else   { setfillstyle(SOLID_FILL,   BLACK); setcolor(   BLACK); }
  pieslice(10,10,0,360,4); setcolor(RED);
  circle(10,10,4);         setcolor(color);
}
