;---------------------------------------------------------------
; DSKDTMF.ASM
; Keith Larson
; TMS320 DSP Applications
; (C) Copyright 1995,1996
; Texas Instruments Incorporated
;
; This is unsupported freeware with no implied warranties or
; liabilities.  See the disclaimer document for details
;----------------------------------------------------------------
; DSKDTMF.ASM is a precision signal generator application which uses
; a table  of complex vectors to create multiple precision sine wave
; oscillations.  The name DSKDTMF is derived from the host control
; program which is designed to generate DTMF tone pairs when the
; numeric keys 0-9 are pressed.  Otherwise this application is a
; precision sine wave signal generator.
;
; NOTE: DSKSG.ASM uses a table lookup scheme with interpolation in both
;       time and level.  DSKSG.ASM and can be used for complex waveforms
;
; Sine Generation
; ---------------
; DSKDTMF.ASM generates each precision sine wave by rotating (a complex
; multiplication) a complex vector pair with a magnitude of 1.0 at
; 2*PI*Freq radians per second.
;
; The vector rotation is produced by a second unity magnitude vector
; pair which defines the rotation rate.  The precision of frequency
; control is therefor determined by the precision of the vector
; accumulation rate which uses 32 bit floating point arithmetic.
; The value of the complex rotation vector rate pair is therefor
; determined by the sampling rate of the DAC and the desired frequency.
;
; Fout = Fdac * Radians/sample
;
; RATE->REAL = cos(Radians/sample)
; RATE->IMAG = sin(Radians/sample)
;
; To prevent the magnitude from decaying or growing over time, the
; magnitude of the vector is controlled by calculating the difference
; of the desired magnitude (1.0) and the actual magnitude (R^2+I^2).
; The difference is then used as a feedback top control the magnitude
; achieving a very high precision.
;
; The output amplitude of each sine wave is then controlled by a scale
; factor which is multiplied into either the real or imaginary component.
;
; NOTE: Both a REAL _and_ IMAGINARY component is available for each
;       sine wave component.  This may be useful for some applications!
;---------------------------------------------------------------
TA       .set   11                 ; AIC Default startup matches host side app
TB       .set   18                 ;
RA       .set   11                 ;
RB       .set   18                 ;
         .include "C3XMMRS.ASM"
;==================================================
          .start  "CODE",0x809C00  ; Start of RAM1
          .sect   "CODE"           ;
;==================================================
MSG_BOX   .word  0                 ; 0x809C00
Secdy_1   .word  (TA<<9)+(RA<<2)+0 ; 0x809C01
Secdy_2   .word  (TB<<9)+(RB<<2)+2 ; 0x809C02
Secdy_3   .word  00000011b         ; 0x809C03  +/- 1.5 V
LOAD      .word  1                 ; 0x809C04 set to 1 to reload AIC

          .word  0                 ; 0x809C05  NOISE ON/OFF
WNOISELVL .float 1.0               ; 0x809C06
WNOISESCL .word  -18<<24           ; 0x809C07 <- NOT MODIFIED BY HOST
OFF_LVL   .float 0.0               ; 0x809C08
MUTE      .float 1.0               ; 0x809C09

;    Vnbl      .set    0
;----------------------------------
; Harmonic array element offsets
;----------------------------------
Venbl     .set    0                ;
Rp        .set    1                ;
Ip        .set    2                ;
Rsig      .set    3                ;
Isig      .set    4                ;
Ampli     .set    5                ;
Freq      .set    6                ;
;-----------------------------------
H_size    .set    7                ; Size of array
;----------------------------------
pi        .set    3.14159265
Rpi       .set    2.0*pi/360.0

sfd
          .loop   16               ; Start table at 0x809C0A
          .float  1.0              ; On Off flag
          .float  cos(Rpi*40.0)    ; R/I phase accumulation rate
          .float  sin(Rpi*40.0)    ;
          .float  1.0              ; Current R/I data pair, sqrt(R^2+I^2)==1
          .float  0.0              ;
          .float  0.0              ; Amplitude multiplier
          .float  0.0              ; Frequency (not used in app)
          .endloop                 ;
          .float  0.0              ; Terminate last term

MaxS      .float  16000.0
SFFTdata  .word   sfd
FLAGS     .word   0               ;
;
S0_gctrl_val  .word  0x0E970300   ; Runtime value XINT/RINT enabled
S0_xctrl_val  .word  0x00000111   ;
S0_rctrl_val  .word  0x00000111   ;
;-------------------------------
main      ldi   0x34,IE           ;
          idle                    ; Wait for next ADC sample
          ;- - - - - - - - - - -
          ldi   0x04,IE
          ldi   @LOAD,R2          ; If signalled by host, reinitialize AIC
          bz    main              ;
          call  AIC_INIT          ; Restart with new AIC setup
          ldi   0,R2              ; reset the LOAD flag
          sti   R2,@LOAD          ;
          b     main              ; close loop
;=============================================================================

FX        .float 0
ADC       push  ST               ; On interrupt, set a software flag to
          push  R0               ; let the CPU know that the RINT has been
          ldi   @S0_rdata,R0     ; received
          ldi   @FLAGS,R0        ;
          or    0x20,R0          ;
          sti   R0,@FLAGS        ;
          pop   R0               ;
          pop   ST               ;
          reti                   ;
          ;- - - - - - - - - - -
DAC       push  ST               ;
          push  R0               ;
          pushf R0               ;
          push  R1               ;
          pushf R1               ;
          push  R2               ;
          pushf R2               ;
          push  R3               ;
          pushf R3               ;
          push  AR1              ;
;=======================================================================
CalcSFFT  ldf   0,R7              ;
          ldi   @SFFTdata,AR1     ; Base pointer to vector rotate table
          ;- - - - - - - - - - - -
NextPair  ldf   *+AR1(Venbl),R0   ; get last used signal R/I pair
          bz    Done
          ldf   *+AR1(Rsig),R0    ; get last used signal R/I pair
          ldf   *+AR1(Isig),R1    ;
CalcERR   mpyf3 R1,R1,R3          ; If magnitude != 1.0
          mpyf3 R0,R0,R2          ; Apply inverse error to R/I
          addf  R3,R2             ; R^2+I^2 = 1.0 + err
     ;----------------------------------------------------------------
     ; Several methods of error feedback can be chosen where an
     ; approximation of f = 1/sqrt(mag^2) is used to rescale the
     ; sin/cos quadrature magnitude of 1.0+/-err to 1.0.  All methods
     ; are based on the following assumptions.
     ;
     ;  1) C3x Floats are nearly equivelent log base 2
     ;  2) Square roots of log values only need a right shift
     ;     which is a multiplication by 1/2.
     ;  3) 1/x inversion in the log domain is a simple multiply by -1
     ;  4) Floating point inversion for numbers close to 1.0 can be
     ;     approximated using  1/x ~= 2-x
     ;
     ; Method 1: This method use the log2() duality of floating point
     ;           registers on the C3x directly.  It will provide a
     ;           close approximation of 1/sqrt(x) for any input
     ;           value x and is not restricted to values close to 1.0.
     ;
     ; Method 2: Consider the number in the register to be in log2()
     ;           format.  If the value is in the range 0.75-1.999 the
     ;           exponent will contain -1 or 0 (11111111b or 00000000b).
     ;           Therefor, no exponent adjustment is needed for the
     ;           square root operation. Inversion is approximated by 2-x.
     ;
     ; Method 3: This method uses the approximation 1/sqrt(x) ~= (3-x)/2
     ;
     ; Method 4: No approximation is used.  Only the inherent stability
     ;           of the floating point format is used.  There is no
     ;           guarentee that this will work
     ;------------------------------------------------------------------
PICK      .set  3                 ; Pick method of approximation
FastSqrt  ;- - - - - - - - - - - -
          .if   PICK=1            ; log2 conversion (works for any value x)
          lsh   1,R2              ; Concatenate exp|mantissa
          pushf R2                ; convert to 8.24 log2 format
          pop   R2                ;
          ash   -1,R2             ; log2(sqrt(x)) = log2(x)/2
          not   R2,R2             ; log2(1/x)     = -log2(x)
          push  R2                ; reverse log2->float
          popf  R2                ;
          lsh   -1,R2             ; clr sign, mantissa in correct position
          .endif                  ;
          ;-----------------------
          .if   PICK=2            ; EXP0-7   S.MANT
          lsh   1,R2    ;Extnd MSB; 00000000 0000000010    2^ 0 * 1.00001
          ash   -2,R2   ;         ; 11111111 0111111100    2^-1 * 1.99999
          lsh   1,R2    ;clr sign ;          1111111110
          lsh   -1,R2   ;         ;          0111111110
          subrf 2.0,R2            ; correction = 1.0 - err
          .endif                  ;
          ;- - - - - - - - - - - -
          .if   PICK=3            ;
          mpyf  0.5,R2            ; 1/sqrt(x) ~= (3-x)/2
          subrf 1.5,R2            ; (this is the fastest!)
          .endif                  ;
          ;- - - - - - - - - - - -
          .if   PICK=4            ;
          ldf   1.0,R2            ; No error correction
          .endif                  ;
          ;-------------------------
          ; Apply correction factor
          ;-------------------------
          mpyf  R2,R0             ; New REAL
          mpyf  R2,R1             ; New IMAG
          ;- - - - - - - - - - - -
CalcRI    mpyf3 *+AR1(Rp),R0,R2   ; R2 = Rp*Rsig
          mpyf  *+AR1(Ip),R0      ; R0 = Ip*Rsig
          mpyf3 *+AR1(Rp),R1,R3   ; R3 = Rp*Isig
          mpyf  *+AR1(Ip),R1      ; R1 = Ip*Isig
          addf  R0,R3             ; R3 = Rp*Isig + Ip*Rsig (New IMAG)
          subf  R1,R2             ; R2 = Rp*Rsig - Ip*Isig (New REAL)
STF_RI    stf   R3,*+AR1(Isig)    ;
          stf   R2,*+AR1(Rsig)    ;
          ;=======================
          mpyf  *+AR1(Ampli),R2   ; if flag is zero exit
          bud   NextPair          ;
          nop   *AR1++(H_size)    ;
          addf  R2,R7             ; Sum either real or imaginary...
          nop                     ;
          ;=======================
Done      mpyf  @MaxS,R7          ; Scale +/-1.0 up to 14 bit integer DAC
          addf  @OFF_LVL ,R7      ; Add DC Offset
          ;-----------------------
          ldf   @WNOISELVL,R0     ; If white noise level != 0.0
ADD_NOISE callnz  RANDX           ; Get next 32 bit random
          float R0,R0             ; convert to float
          mpyf  @WNOISESCL,R0     ; scale to +/-1.0
          mpyf  @WNOISELVL,R0     ;
          addf  R0,R7             ;
          mpyf  @MUTE,R7
          ;-----------------------
NO_NOISE  mpyf  @SATURATE,R7      ; Using the ALU saturation mode saves 2
          mpyf  @UNSATUR8,R7      ; cycles compared to magnitude comparison
          fix   R7,R7             ; Convert to fixed point
          andn  3,R7              ; Clear MSBs to prevent AIC reprogramming
          sti   R7,@S0_xdata      ; output the DAC value
          ;- - - - - - - - - - - -
          pop   AR1               ; Restore register context before return
          popf  R3                ;
          pop   R3                ;
          popf  R2                ;
          pop   R2                ;
          popf  R1                ;
          pop   R1                ;
          popf  R0                ;
          pop   R0                ;
          pop   ST                ;
          reti                    ; return from interrupt
;----------------------------------------
; Fast 32 bit random number generator
;----------------------------------------
RANDX:  ldi     @SEED,R0          ; Call here for last SEED
RAND:   mpyi    @A,R0             ; Calculate RAND(R0)
        addi    @C,R0             ;
        sti     R0,@SEED          ; Result is returned in R0
        rets                      ;
        ;-------------------------
A       .word   0107465h          ; Constants needed for RAND
C       .word   0234567h          ;
SEED    .word   0                 ;
;- - - - - - - - - - - - - - - - -
                                ; By using the saturation feature of the
                                ; TMS320C3x ALU during multiplies, several
                                ; cycles can be shaved off of a compare/load
                                ; clipping operation
                                ;
SATURATE  .word   113<<24       ; Since max AIC magnitude is 2^15, multiply
UNSATUR8  .word   -113<<24      ; by 2^113 which results in 2^128.  The
                                ; multiplier constant is created by directly
                                ; placing the value of 113 into the exponent
                                ; exponent field of a 32 bit hex number.
;--------------------------------
GIE      .set 0x2000

myidle   ; idle
         ; rets

          andn  GIE,ST          ; Use XINT polling during setup to
          tstb  0x10,IF         ; avoid contaminating the DXR during
          bz    $-1             ; programming
          andn  0x10,IF         ;
          ;idle                 ;
          rets                  ;

prog_AIC  push  R1              ;
          push  IE              ;
          ldi   0x10,IE         ;
          andn  0x30,IF         ;
          ldi   @S0_xdata,R1    ; Use original DXR data during 2 ndy
          sti   R1,@S0_xdata    ;
        call myidle
          ldi   @S0_xdata,R1    ; Use original DXR data during 2 ndy
          or    3,R1            ; Request 2 ndy XMIT
          sti   R1,@S0_xdata    ;
        call myidle             ;
          sti   R0,@S0_xdata    ; Send register value
        call myidle             ;
     ;    andn  3,R1            ;
          ldi   0,R1
          sti   R1,@S0_xdata    ; Leave with original safe value in DXR
          pop   IE              ;
          pop   R1              ;
          rets                  ;
;======================================================;
; This section of code is called by the initialization ;
; code as well as by the main program loop.  It is     ;
; therfor assembled into the regular program RAM       ;
;======================================================;
AIC_INIT  push  R0              ;
          LDI   0x10,IE         ; Enable XINT interrupt
          andn  0x34,IF         ;

AIC_reset LDI   2,IOF           ; XF0=0 resets AIC
          ldi   0,R0            ;
          sti   R0,@S0_xdata    ;
          LDI   6,IOF           ; XF0=1 runs AIC
          ;---------------------
AICprog1  ldi   0xfffc  ,R0     ; Program the AIC to be real slow
          call  prog_AIC        ;
          ldi   0xfffc|2,R0     ;
          ;- - - - - - - - - - -
AICprog2  ldi   @Secdy_1,R0     ; Re-load the AIC registers
          call  prog_AIC        ;
          call  prog_AIC        ;
          ldi   @Secdy_2,R0     ; NOTE: Bump up the Fs to final rate last
          call  prog_AIC        ;       (smallest divisor should be last)
          ldi   @Secdy_3,R0     ;
          call  prog_AIC        ;
          ;- - - - - - - - - - -
          ldi   0,R0            ; Put a safe 0 in DXR
          sti   R0,@S0_xdata    ;
          ldi   @S0_rdata,R0    ; Clear receive underrun
          pop   R0              ;
          rets                  ;
;****************************************************************;
; NOTE: START STUBS                                              ;
;                                                                ;
; If a section of code is used only for initialization it can be ;
; overwritten by other data after completion.  In other words,   ;
; the stack or runtime data can overwrite sections of code that  ;
; will never be executed again.                                  ;
;****************************************************************;
          .entry   ST_STUB
ST_STUB   ldp   T0_ctrl         ; Use kernel data page and stack
          ldi   @stack,SP       ;
          ldi   0,R0            ; Halt TIM0 & TIM1
          sti   R0,@T0_ctrl     ;
          sti   R0,@T1_ctrl     ;
          sti   R0,@T0_count    ; Set counts to 0
          sti   R0,@T1_count    ;
          ldi   1,R0            ; Set periods to 1
          sti   R0,@T0_prd      ;
          sti   R0,@T1_prd      ;
          ldi   0x2C1,R0        ; Restart both timers
          sti   R0,@T0_ctrl     ;
          sti   R0,@T1_ctrl     ;
          ;---------------------
          ldi   @S0_xctrl_val,R0;
          sti   R0,@S0_xctrl    ; transmit control
          ldi   @S0_rctrl_val,R0;
          sti   R0,@S0_rctrl    ; receive control
          ldi   0,R0            ;
          sti   R0,@S0_xdata    ; DXR data value
          ldi   @S0_gctrl_val,R0; Setup serial port
          sti   R0,@S0_gctrl    ; global control
          ;---------------------
          andn  0x30,IF         ;
          ldi   0x30,IE         ; Service both RINT/XINT
          call  AIC_INIT        ; Initialize the AIC
          ldi   @S0_rdata,R0    ;
          b     main            ;
stack    .word  $+1
;****************************************************;
; Install the XINT/RINT ISR handler directly into    ;
; the vector RAM location it will be used in         ;
;****************************************************;
          .start   "SP0VECTS",0x809FC5
          .sect    "SP0VECTS"
          B     DAC           ; XINT0
          B     ADC           ; RINT0
