/*=======================================================================
  SFFT_FM.C
  Keith Larson
  TMS320 DSP Applications
  (C) Copyright 1996,1997,1998
  Texas Instruments Incorporated

  This is unsupported freeware with no implied warranties or
  liabilities.  See the C3x DSK disclaimer document for details

  Please read the file SFFT.TXT for an explanation on how the
  Sliding FFT works

 ========================================================================*/
#include <math.h>
#include "C3MMR.H"
/*
  NOTE: The AM/FM detectors do not seem to work if BIN_END >= SFFTSIZE/2
        This is likely a result of some trigonometric identity

        Also, an SFFT filter beginning with BIN0 will *NOT* work since
        a -F value is needed to properly compute the filter.  In addition,
        the Hilbert transform of the DC bin is meaningless (90' phase
        shift of DC?)
*/
#define  Q         2
#define  SFFTSIZE  (Q*16)        /* Sample Window length (FFT size)      */
#define  BIN_START (Q* 2)        /* Start computing SFFT at this bin     */
#define  BIN_END   (Q* 5)        /* End computing SFFT at this bin       */
#define  ANGLE     90.0          /* Filter reconstruction angle (degrees)*/
/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/
#define  TIM0_prd  2             /* AIC reference clock is TIM0          */
#define  TA        8             /* DAC setup                            */
#define  TB        50            /*                                      */
#define  RA        8             /* ADC setup                            */
#define  RB        50            /*                                      */
/*========================================================================
  PARAMETERS BELOW THIS LINE ARE COMPUTED FROM THE INFORMATION ABOVE.
          THERE IS NO NEED TO MODIFY ANYTHING BELOW THIS POINT
  =======================================================================*/
#define  BIN_LEN   (BIN_END-BIN_START)/* Filter length in bins           */
#define  pi      3.14159265           /* Useful in making apple pie      */
#define  wn      2.0*pi/SFFTSIZE      /* angle = F * 2*pi/Fs             */
#define  K1      0.99995              /* K1 is slightly less than 1.0    */
#define  A_REG   ((TA<<9)+(RA<<2)+0)  /* Packed AIC register values      */
#define  B_REG   ((TB<<9)+(RB<<2)+2)  /*                                 */
#define  C_REG   0x3                  /*                                 */
/*efine  S0gctrl 00E970300h */        /* Sport, noninverted clkx/clkr    */
#define  S0gctrl 0x0E973300           /* Sport,    inverted clkx/clkr    */
#define  S0xctrl 0x00000111           /*                                 */
#define  S0rctrl 0x00000111           /*                                 */
/*========================================================================
  If the input parameters won't work, generate a descriptive error for
  the user letting them know what to look for and maybe fix
 =========================================================================*/
#if (BIN_LEN < 1)
  APP MESSAGE: Calculated BIN_LEN must be >1
#endif
/*===================================================================
  The SFFT twiddles, data, and input buffer arrays are allocated to
  be placed into RAM0 to avoid bus conflicts with program fetching
 ====================================================================*/
typedef struct CMPLX
{
  float R;
  float I;
}cmplx;

typedef struct
{
  float Real ;
  float Imag ;
  float WReal ;
  float WImag ;
  float Vol ;
}SigParm;
/*-----------------------------------------*/
void   ST_STUB (void);
void   AIC_INIT(void);
void   prog_AIC(int  prgword);
float  SFFT_A  (float   diff);
/*-----------------------------------------*/
SigParm CmplxGen ;
cmplx  TW  [BIN_LEN+2];
cmplx  BIN [BIN_LEN+2];
float  BUF   [SFFTSIZE];
float  REAL_VEC;
float  IMAG_VEC;
float  K2     = .95000; /* K1^N */
float  Scale  = 2.0/SFFTSIZE;
float  SFFT_RL;
float  SFFT_IM;
float  Fwav =  440;
float  FS   = 1000;
float  Vol  =    1;
int    Tbase = (int)&TW;
int    Bbase = (int)&BIN;
int    SFFTBINS=BIN_LEN+2;          /* Bins to calculate               */
int    MIDBIN  =(BIN_START+BIN_END)/2;
/*===================================================================
  The ADC data is read and buffered here
 ====================================================================*/
float inline Input(void)
{
  int   x;
  float f;
  x = *S0_rdata;
  x = x >> 16;
  f = x;
  return f/32768;
}
/*===================================================================
  The output section is written for both Spectrum analyzer output
  as well as REAL/IMAG filter sum outputs
 ====================================================================*/
void inline Output(float f)
{
  long x;
  f*=32768;
/* f *=512; */ /* Use this to look at weak signals */
  if(f>  32767) f =  32767;
  if(f< -32768) f = -32768;
  x = f;
  x &= 0xFFFC;
  *S0_xdata = x;
}
/*===================================================================*/
float inline diff(float f)
{
  float f2;
  static int bufptr = 0;
  f2 = f - (K2 * BUF[bufptr]);
  BUF[bufptr] = f;
  bufptr++;
  if(bufptr >= SFFTSIZE) bufptr = 0;
  return f2;
}
/*===================================================================
  Initialize runtime constants and arrays
  ===================================================================*/
void init_arrays(void)
{
  int x;
  float n;
  n = BIN_START;
  for(x=0;x<SFFTBINS;x++)
  {
    TW[x].R = K1*cos(n*wn);   /* R/I phase or twiddle coefficients */
    TW[x].I = K1*sin(n*wn);
    BIN[x].R = 0.0;
    BIN[x].I = 0.0;
    n+=1.0;
  }
  for(x=0;x<SFFTSIZE;x++)
  {
    BUF[x] = 0;
  }
  REAL_VEC = -cos(pi*ANGLE/180.0); /* filtered REAL scale factor         */
  IMAG_VEC = -sin(pi*ANGLE/180.0); /* filtered IMAG scale factor         */
  K2 = pow(K1,SFFTSIZE);
}
/*************************************************/
volatile int xdata    = 0;
volatile int free_run = 0;
volatile int RRDY     = 0;

#define R_  (Sig->Real )
#define I_  (Sig->Imag )
#define WR_ (Sig->WReal)
#define WI_ (Sig->WImag)
#define V_  (Sig->Vol  )
/********************************************************/
float fback(float);
asm("        .text                                              ");
asm("_fback                    ; Value is in F0                 ");
asm("__qsqrt lsh   1,R0        ; Fast sqrt of numbers ~1.000    ");
asm("        ash   -2,R0       ; are calcualted by shifting     ");
asm("        lsh   1,R0        ; the mantissa. The EXP field    ");
asm("        lsh   -1,R0       ; is unchanged                   ");
asm("__qinv                    ;                                ");
asm("        stf   R0,@_qtmp   ; Fast inversion is done by      ");
asm("        ldi   @_qtmp,R0   ; inverting all bits except      ");
asm("        xor   @_qmsk,R0   ; the sign bit.                  ");
asm("        sti   R0,@_qtmp   ;                                ");
asm("        ldf   @_qtmp,R0   ;                                ");
asm("        rets              ;                                ");
asm("_qtmp   .float  0         ;                                ");
asm("_qmsk   .word  0FF7FFFFFh ;                                ");


float siggen(SigParm *Sig)
{
  float f;
  f  = R_ * WR_  -  I_ * WI_;
  I_ = I_ * WR_  +  R_ * WI_;
  R_ = f;
  f = fback((R_*R_) + (I_*I_));
  R_ *= f;
  I_ *= f;
  f  = R_;
  f *= V_;
  return f;
}

/*===================================================================
  Initialize runtime constants and arrays
  ===================================================================*/
void init_sig2(SigParm *Sig, float Fsig, float Fs)
{
  float w;
  w   = pi * Fsig / Fs;
  WR_ = cos(w);
  WI_ = sin(w);
}

void init_sig(SigParm *Sig, float Fsig, float Fs)
{
  init_sig2(Sig,Fsig,Fs);
  R_  = 1.0;
  I_  = 0.0;
  V_  = Vol;
}

/*===================================================================*/
#define G1 32
#define G2 0.9999/(G1+1)

#define CGR CmplxGen.Real
#define CGI CmplxGen.Imag

#define GCOS CmplxGen.Real
#define GSIN CmplxGen.Imag
#define SCOS SFFT_RL
#define SSIN SFFT_IM

#define SR SFFT_RL
#define SI SFFT_IM
typedef enum
{
  AM1,      /* 4 quadrant peak detector        */
  AM2,      /* AM using SFFT R/I magnitude     */
  AM3,      /* Product detector                */
  SSBU,     /* Single sideband upper           */
  SSBL,     /*                 lower           */
  SSSBU,    /* Supressed single sideband upper */
  SSSBL,    /*                           lower */
  FMSLOPE,  /* FM - Slope detect               */
  FMPLL,    /* FM - Phase lock loop            */
  FMZERO,   /* FM - Zero crossing              */
  FZERO,    /* AM?                             */
  FDBL      /* Frequency doubler               */
}DETECTOR;

void main(void)
{
   int i;
   DETECTOR detector = FMPLL;
   float f;
   float R1=1.0,I1=1.0;
   float R2=1.0,I2=1.0;
   float fsig;
   float lastR=0,lastI=0;
   float AMDETECT=0;
   float mag=1.0;
   float agc=1.0;
   float fin;
   init_arrays();
   FS = 25E6/(TIM0_prd * 2.0 * TA * TB * 4.0);
   Fwav = 2*FS*MIDBIN/SFFTSIZE;
   init_sig(&CmplxGen,Fwav,FS);

   ST_STUB();
   asm("  ldi   0E4h,IE     ");  /* Enable XINT/RINT/INT2               */
   asm("  idle              ");  /* Wait for Receive Interrupt          */
   f = (float)*S0_rdata;         /* The first interrupt occurs shortly  */
   *S0_xdata = 0;                /* after AIC init is complete, which   */
   for(;;)
   {
     asm("  idle    ");          /* Wait for Receive Interrupt          */
     fin = Input ( );
     f = diff  (fin);
     f = SFFT_A(f);
     SR  /= SFFTSIZE;
     SI  /= SFFTSIZE;
     mag = sqrt((SR*SR)+(SI*SI));
     fsig=siggen(&CmplxGen);
     /* Select AGC if required (FM demod) */
     switch(detector)
     {
       case FMPLL  :
       case FMZERO :
       case FMSLOPE: /* The inverse taken here, and the preceding sqrt()
                        can be substantialy sped up with the knowledge
                        that the magnitude should not be changing rapidly
                        from one sample to the next.  This in turn can
                        lead to a modified 1/sqrt() function which will be
                        much faster and provide full precision.
                     */
                /*   agc = G2*((G1*agc) + (1.0/mag)); break; */
       default     : agc = 1; break;
     }
     SR *=agc;
     SI *=agc;
     /* Detect the signal */
     switch(detector)
     {
       default:
       case SSSBU  :
       case SSBU   : f=(SR*CGR)+(SI*CGI); break; /* SSB upshift   */
       case SSSBL  :
       case SSBL   : f=(SR*CGR)-(SI*CGI); break; /* SSB dnshift   */
       case AM1    : /* Four quadrant peak detector  */
                     SR = fabs(SR); if(SR>AMDETECT) AMDETECT=SR;
                     SI = fabs(SI); if(SI>AMDETECT) AMDETECT=SI;
                     AMDETECT *= .99;
                     f = AMDETECT;
                     break;
       case AM2    : f = mag; break;
       case AM3    : /* AM detection using a product detector requires   */
                     /* a PLL to lock in the phases of the LO and signal */
                 /*  mag = SR*CGR; */
                     AMDETECT = G2*(mag + (G1 * AMDETECT));
                     f = AMDETECT;
                     break;
       case FZERO  : f=(SR* SR)+(SI* SI); break; /* S^2+C^2=w1-w2 */
       case FDBL   : f=(SR* SR)-(SI* SI); break; /* S^2-C^2=w1+w2 */

       case FMPLL  : /* Use the R/I (R/Hilbert) output of the SFFT filter
                        to find the sin/cos of the difference of the two
                        frequencies.
                     */
                     if(1)
                     {
                       R2 = (SSIN*GSIN)+(SCOS*GCOS);  /*  sin(w1-w2)  */
                       I2 = (SSIN*GCOS)-(SCOS*GSIN);  /*  cos(w1-w2)  */
                     }
                     else
                     {
                       R2 = SSIN;    /* Direct use of the SFFT output */
                       I2 = SCOS;
                     }
                     /*
                        The following lines can likely be improved.  Known
                        things include...

                        mag(R1 + jI1) = 1.0
                        mag(R2 + jI2) = 1.0

                        (R1*R2)+(I1*I2) can be zero,
                           but, denominator/numerator = finite

                        The phase angle between (R1+jI1) and (R2+jI2) is
                        phase accumulation rate of the instantaneous
                        frequency w1-w2.
                     */
                     /*
                       f = ((R1*I2)-(R2*I1))/((R1*R2)+(I1*I2));
                     */
                     f = (R1*R2)+(I1*I2);
                     if(f==0.0) f = lastR;
                     else       f = ((R1*I2)-(R2*I1))/f;
                     lastR=f;
                     /* Schmooze filter low pass of detector */
                     /*
                     AMDETECT = .24999*((3*AMDETECT)+f);
                     f = AMDETECT;
                     */
                     f*=.5;
                     R1 = R2;
                     I1 = I2;
                     break;

       case FMSLOPE: lastR = 0.499*(SR + lastR); f=fabs(lastR); lastR=SR;
                     if(f>AMDETECT) AMDETECT=f;
                     lastI = 0.499*(SI + lastI); f=fabs(lastI); lastI=SI;
                     if(f>AMDETECT) AMDETECT=f;
                     AMDETECT *= .99;
                     f = AMDETECT;
                     break;
       case FMZERO : break;
     }
     Output(f);
   }
}
/*===================================================================
  The startup stub is used during initialization only and can be
  overwritten by the stack or data after initialization is complete.
  Note: A DSK or RTOS communications kernel may also use the stack.
  In this case be sure to not put the stack here during debug.
 ====================================================================*/
void ST_STUB(void)
{
   *T0_ctrl = 0;         /* Halt TIM0           */
   *T0_count= 0;         /* Set counts to 0     */
   *T0_prd  = TIM0_prd;  /* Set period          */
   *T0_ctrl = 0x2C1;     /* Restart both timers */
   /* - - - - - - - - - - - - - - - - - - - - - */
   *S0_xctrl = S0xctrl;  /* transmit control    */
   *S0_rctrl = S0rctrl;  /* receive  control    */
   *S0_xdata =       0;  /* DXR data value      */
   *S0_gctrl = S0gctrl;  /* global control      */
    AIC_INIT();
}
/*================================================
  This function initializes the AIC
 ================================================*/
void AIC_INIT(void)
{
  asm("  andn  034h,IF  ");
  asm("  ldi   004h,IE  ");  /* Enable only INT2 */
  *S0_xdata = 0;
  asm(" rpts  0040h     ");
  asm(" ldi   2,IOF     ");  /* XF0=0 resets AIC */
  asm(" ldi   6,IOF     ");  /* XF0=1 runs AIC   */
  asm(" rpts  040h      ");
  asm(" nop             ");
  asm("  andn  034h,IF  ");
  asm("  ldi   014h,IE  ");  /* Enable only XINT interrupt */
  /*- - - - - - - - - - - -*/
  prog_AIC(C_REG   );      /* program control register */
  prog_AIC(0xFFFC  );      /* Program the AIC to be real slow */
  prog_AIC(0xFFFC|2);      /* Program the AIC to be real slow */
  prog_AIC(B_REG   );      /* Bump up the Fs to final rate    */
  prog_AIC(A_REG   );      /* smaller divisors sent first     */
  asm("  or  080h,ST");    /* Use the overflow mode for fast saturate */
}

/*===================================================================
  prog_AIC is used to transmit new timing configurations to the AIC.
  If you single step this routine, the AIC timing will be corrupted
  causing AIC programming to fail.
  STEP OVER THIS ROUTINE USING THE F10 FUNCTION STEP
 ====================================================================*/
void prog_AIC(int xmit2)
{
  int x;
  *S0_xdata =     0; asm(" idle "); /* Pre transmit a safe value       */
  *S0_xdata =     3; asm(" idle "); /* Request 2ndy xmit               */
  *S0_xdata = xmit2; asm(" idle "); /* Send register porgram value     */
  *S0_xdata =     0; asm(" idle "); /* Leave with a safe value         */
  x = *S0_rdata;                    /* Fix rcvr underrun by dummy read */
}
/*===================================================================
  Install the XINT/RINT ISR branch vectors
 ====================================================================*/
  asm(" .sect  \"SP0VECTS\""); /* secondary branch table */
  asm(" reti              ");  /* XINT0                  */
  asm(" reti              ");  /* RINT0                  */
  asm(" .text             ");
/*===================================================================*/
