/*=======================================================================
  SFFT_C.C
  Keith Larson
  TMS320 DSP Applications
  (C) Copyright 1996,1997,1998
  Texas Instruments Incorporated

  This is unsupported freeware with no implied warranties or
  liabilities.  See the C3x DSK disclaimer document for details

  Please read the file SFFT.TXT for an explanation on how the
  Sliding FFT works
 ========================================================================*/
#include <math.h>
#include "C3MMR.H"
#define  SFFTSIZE  128           /* Sample Window length (FFT size)      */
#define  BIN_START 8             /* Start computing SFFT at this bin     */
#define  BIN_END   10            /* End computing SFFT at this bin       */
#define  ANGLE     90.0          /* Filter reconstruction angle (degrees)*/
/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/
#define  TIM0_prd  2             /* AIC reference clock is TIM0          */
#define  TA        10            /* DAC setup                            */
#define  TB        15            /*                                      */
#define  RA        10            /* ADC setup                            */
#define  RB        15            /*                                      */
/*========================================================================
  PARAMETERS BELOW THIS LINE ARE COMPUTED FROM THE INFORMATION ABOVE.
          THERE IS NO NEED TO MODIFY ANYTHING BELOW THIS POINT
  =======================================================================*/
#define  BIN_LEN   (BIN_END-BIN_START)/* Filter length in bins           */
int      SFFTBINS = BIN_LEN+2;        /* Bins to calculate               */
#define  N         SFFTSIZE           /* N used as shorthand for SFFTSIZE*/
#define  pi        3.14159265         /* Useful in making apple pie      */
#define  w         2.0*pi/N           /* angle = F * 2*pi/Fs             */
#define  K1        0.99995            /* K1 is slightly less than 1.0    */
#define  A_REG   ((TA<<9)+(RA<<2)+0) /* Packed AIC register values       */
#define  B_REG   ((TB<<9)+(RB<<2)+2) /*                                  */
#define  C_REG   0x3                 /*                                  */
/*efine  S0gctrl 00E970300h */       /* Sport, noninverted clkx/clkr     */
#define  S0gctrl 0x0E973300          /* Sport,    inverted clkx/clkr     */
#define  S0xctrl 0x00000111          /*                                  */
#define  S0rctrl 0x00000111          /*                                  */
#define  bigval  000010000h          /* Used in overflow mode saturation */
void   ST_STUB (void);
float  Input   (void);
void   AIC_INIT(void);
void   prog_AIC(int  prgword);
float  SFFT    (float   diff);
void   Output  (float outval);
float  SFFT_A  (float   diff);
float  SFFT_C  (float   diff);
float  diff    (float      f);
/*========================================================================
  If the input parameters won't work, generate a descriptive error for
  the user letting them know what to look for and maybe fix
 =========================================================================*/
#if (BIN_LEN < 1)
  APP MESSAGE: Calculated BIN_LEN must be >1
#endif
/*===================================================================
  The SFFT twiddles, data, and input buffer arrays are allocated to
  be placed into RAM0 to avoid bus conflicts with program fetching
 ====================================================================*/
typedef struct CMPLX
{
  float R;
  float I;
}cmplx;

cmplx TW  [BIN_LEN+2];
cmplx BIN [BIN_LEN+2];
float BUF   [SFFTSIZE];
float REAL_VEC = 1.0;
float IMAG_VEC = 0.0;
float K2     = .99000; /* K1^N */
float Scale  = 4.0/SFFTSIZE;
int   Tbase = (int)&TW;
int   Bbase = (int)&BIN;
float SFFT_RL;
float SFFT_IM;
/*===================================================================
  Initialize runtime constants and arrays
  ===================================================================*/
void init_arrays(void)
{
  int x;
  float n;
  n = BIN_START;
  for(x=0;x<SFFTBINS;x++)
  {
    TW[x].R = K1*cos(n*w);   /* R/I phase or twiddle coefficients */
    TW[x].I = K1*sin(n*w);
    BIN[x].R = 0.0;
    BIN[x].I = 0.0;
    n+=1.0;
  }
  for(x=0;x<SFFTSIZE;x++)
  {
    BUF[x] = 0;
  }
  K2 = pow(K1,SFFTSIZE);
  REAL_VEC = cos(ANGLE);
  IMAG_VEC = sin(ANGLE);
}

/*===================================================================
  The main loop consists of waiting for a new ADC sample.
  When an receive interrupt occurs, the new data is loaded into the
  data delay line buffer, followed by the SFFT and output routines.
  Four dummy writes to the external bus have been added in the main
  loop to allow real time benchmarking of the three functions using
  and oscilloscope to monitor the address bus LSB's
 ====================================================================*/
void main(void)
{
   float   f;
   float mag=1.0;
   float agc=1.0;
   float fin;
   init_arrays();
   ST_STUB();
   asm("  ldi   0E4h,IE     ");  /* Enable XINT/RINT/INT2               */
   asm("  idle              ");  /* Wait for Receive Interrupt          */
   f = (float)*S0_rdata;         /* The first interrupt occurs shortly  */
   *S0_xdata = 0;                /* after AIC init is complete, which   */
                                 /* will not leave enough time for SFFT */
   for(;;)
   {
     asm("  idle    ");          /* Wait for Receive Interrupt          */
     asm("  sti R0,@080A000h");
     fin = Input ( );              /* ADC sample                        */
     f = diff  (fin);
     f = SFFT_A(f);
     f = SFFT_RL * Scale;
     asm("  sti R0,@080A0FFh");
     Output(f);                  /* Output result                       */
   }
}

/*===================================================================
  The ADC data is read and buffered here
 ====================================================================*/
float Input(void)
{
  int   x;
  float f;
  x = *S0_rdata;
  x = x >> 16;
  f = x;
  f/=32000; /* Scale to +/- 1 */
  return f;
}
/*===================================================================
  The output section is written for both Spectrum analyzer output
  as well as REAL/IMAG filter sum outputs
 ====================================================================*/
void Output(float f)
{
  long x;
  if(f>  1.0) f =  1.0;
  if(f< -1.0) f = -1.0;
  x = f * 32000;
  x &= 0xFFFC;
  *S0_xdata = x;
}

float diff(float f)
{
  float f2;
  static int bufptr = 0;
  f2 = f - (K2 * BUF[bufptr]);
  BUF[bufptr] = f;
  bufptr++;
  if(bufptr >= SFFTSIZE) bufptr = 0;
  return f2;
}
/*===================================================================
  The startup stub is used during initialization only and can be
  overwritten by the stack or data after initialization is complete.
  Note: A DSK or RTOS communications kernel may also use the stack.
  In this case be sure to not put the stack here during debug.
 ====================================================================*/
void ST_STUB(void)
{
   *T0_ctrl = 0;         /* Halt TIM0           */
   *T0_count= 0;         /* Set counts to 0     */
   *T0_prd  = TIM0_prd;  /* Set period          */
   *T0_ctrl = 0x2C1;     /* Restart both timers */
   /* - - - - - - - - - - - - - - - - - - - - - */
   *S0_xctrl = S0xctrl;  /* transmit control    */
   *S0_rctrl = S0rctrl;  /* receive  control    */
   *S0_xdata =       0;  /* DXR data value      */
   *S0_gctrl = S0gctrl;  /* global control      */
    AIC_INIT();
}
/*================================================
  This function initializes the AIC
 ================================================*/
void AIC_INIT(void)
{
  asm("  andn  034h,IF  ");
  asm("  ldi   004h,IE  ");  /* Enable only INT2 */
  *S0_xdata = 0;
  asm(" rpts  0040h     ");
  asm(" ldi   2,IOF     ");  /* XF0=0 resets AIC */
  asm(" ldi   6,IOF     ");  /* XF0=1 runs AIC   */
  asm(" rpts  040h      ");
  asm(" nop             ");
  asm("  andn  034h,IF  ");
  asm("  ldi   014h,IE  ");  /* Enable only XINT interrupt */
  /*- - - - - - - - - - - -*/
  prog_AIC(C_REG   );      /* program control register */
  prog_AIC(0xFFFC  );      /* Program the AIC to be real slow */
  prog_AIC(0xFFFC|2);      /* Program the AIC to be real slow */
  prog_AIC(B_REG   );      /* Bump up the Fs to final rate    */
  prog_AIC(A_REG   );      /* smaller divisors sent first     */
  asm("  or  080h,ST");    /* Use the overflow mode for fast saturate */
}

/*===================================================================
  prog_AIC is used to transmit new timing configurations to the AIC.
  If you single step this routine, the AIC timing will be corrupted
  causing AIC programming to fail.
  STEP OVER THIS ROUTINE USING THE F10 FUNCTION STEP
 ====================================================================*/
void prog_AIC(int xmit2)
{
  int x;
  *S0_xdata =     0; asm(" idle "); /* Pre transmit a safe value       */
  *S0_xdata =     3; asm(" idle "); /* Request 2ndy xmit               */
  *S0_xdata = xmit2; asm(" idle "); /* Send register porgram value     */
  *S0_xdata =     0; asm(" idle "); /* Leave with a safe value         */
  x = *S0_rdata;                    /* Fix rcvr underrun by dummy read */
}
/*===================================================================
  Install the XINT/RINT ISR branch vectors
 ====================================================================*/
  asm(" .sect  \"BRTBL\" "); /* secondary branch table */
  asm(" reti             ");  /* XINT0                  */
  asm(" reti             ");  /* RINT0                  */
  asm(" .text            ");
/*===================================================================
  Recreate the C31 vector table in the bootrom if using an EVM
 ====================================================================*/
  asm(" .sect  \"VECTS\"  "); /* secondary branch table  */
  asm(" .word  0000045h   ");  /* Reset                  */
  asm(" .word  0809FC1h   ");  /* INT0                   */
  asm(" .word  0809FC2h   ");  /* INT1                   */
  asm(" .word  0809FC3h   ");  /* INT2  (Debug Host int) */
  asm(" .word  0809FC4h   ");  /* INT3                   */
  asm(" .word  0809FC5h   ");  /* XINT0                  */
  asm(" .word  0809FC6h   ");  /* RINT0                  */
  asm(" .word  0809FC7h   ");  /* XINT1 (Debug SSTEP)    */
  asm(" .word  0809FC8h   ");  /* RINT1                  */
  asm(" .word  0809FC9h   ");  /* TINT0                  */
  asm(" .word  0809FCAh   ");  /* TINT1                  */
  asm(" .word  0809FCBh   ");  /* DMA                    */
  asm(" .text             ");
/*===================================================================*/

/*===================================================================
  The forward and reverse SFFT are calculated within this one loop
  containing both the REAL and IMAG filter summations.  If only one
  sum is required, the other can be removed
 ====================================================================*/
float SFFT_C(float diff)
{
  float ftemp;
  float REAL_sum=0;
  float IMAG_sum=0;
  int x;
  for(x=0;x<SFFTBINS;x++)
  {
    ftemp    = (BIN[x].R*TW[x].R) - (BIN[x].I*TW[x].I + diff);
    REAL_sum = ftemp    - REAL_sum;
    BIN[x].I = (BIN[x].I*TW[x].R) + (BIN[x].R*TW[x].I);
    IMAG_sum = BIN[x].I - IMAG_sum;
    BIN[x].R = ftemp;
  }
  REAL_sum -= (0.5 * BIN[SFFTBINS-1].R);
  IMAG_sum -= (0.5 * BIN[SFFTBINS-1].I);
  if((SFFTBINS&1)==0)
  {
    REAL_sum = -REAL_sum;
    IMAG_sum = -IMAG_sum;
  }
  REAL_sum -= (0.5 * BIN[         0].R);
  IMAG_sum -= (0.5 * BIN[         0].I);
  SFFT_RL = REAL_sum;
  SFFT_IM = REAL_sum;
  return((REAL_sum * REAL_VEC) + (IMAG_sum * IMAG_VEC));
}



