//----------------------------------------------------------------------
// DSKBODE.CPP    (Uses Sliding FFT technology)
// Keith Larson
// TMS320 DSP Applications
// (c) Copyright 1995, 1996
// Texas Instruments Incorporated
//
// This is unsupported freeware code with no implied warranties or
// liabilities.  See the disclaimer document for details
//----------------------------------------------------------------------
#include <dos.h>
#include <stdio.h>
#include <conio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <bios.h>
#include <graphics.h>
#include "DSK.H"
#include "DSK_COFF.H"
#include "C3XMMRS.H"
#include "keydef.h"  
#include "TMSFLOAT.H"
#include "math.h"
#include "symbols.h"
//===================================================================
//Constants for functions and setup
//===================================================================
#define zSFFTSIZE 32                    //
#define BIN_LEN   (BIN_END-BIN_START)   // Filter length in bins
#define pi        3.14159265            // Useful in making apple pie
#define w         2.0*pi/(float)SFFTSIZE// angle = F * 2*pi/Fs
#define K1        0.99995               // K1 is slightly less than 1.0
#define THx       40e-9                 // DSP cycle time H1/H3 clock rate
#define Xd        512                   // X pix width
#define xd        52                    // X pix/division
#define Yd        260                   // Y pix height
#define yd        20                    // Y pix/division
//===================================================================
ulong SFFTSIZE   = zSFFTSIZE;           // Sample Window length (FFT size)
ulong SFFTsz1    = zSFFTSIZE;           // Allocated 'max' size on target
ulong BIN_START  =         4;           // Start computing SFFT at this bin
ulong BIN_END    =         5;           // End computing SFFT at this bin
ulong SFFTBINS   =         3;//BIN_LEN+2// filter needs 3 bins to calculate
//----------------------------------------
ulong T_LOC      = 0x809C00L;           // xxx_LOC variables are loaded
ulong A_LOC      = 0x809C00L;           // from the DSK applications symbol
ulong B_LOC      = 0x809C00L;           // table.  This allows the host
ulong C_LOC      = 0x809C00L;           // application to know where these
ulong TRG_LOC    = 0x809C00L;           // variables are located within the
ulong BIN_LOC    = 0x809C00L;           // DSP memory space.
ulong TWD_LOC    = 0x809C00L;           //
ulong SIZE_LOC   = 0x809C00L;           //
//----------------------------------------
ulong T0_prdv    =         2;           // Initial timing parameters for AIC
ulong C_REG      =    0x0003;           // AIC control register bits
int   TA         =         3;           // DAC divisors
int   TB         =        48;           //
int   RA         =         4;           // ADC divisors
int   RB         =        36;           //
//-------------------------------------------------
float freq       =    1000.0;           //
float Fsr        =    1000.0;           //
float Fsx        =    1000.0;           //
float maxft      =       0.0;           // Used for displaying max error
int   graphmode  =         1;           // Which type of bode display
char  DSK_APP    []="DSK_BODE.OUT";     //
char  DSK_EXE_APP[]="DSKBODE.EXE";      //
//-------------------------------------------------
typedef struct CMPLX
{
  ulong R;
  ulong I;
}cmplx;
//---------------------------------------------------
cmplx TW  [3];   // This application is essentialy a narrowband filter
cmplx BIN [3];   // do only 3 bins are required
//---------------------------------------------------
void  binsprintf       (char *s,int val);
int   check_key        (void);
MSGS  update_DSK_params(void);
void  graph            (void);
float BINMAG           (void);
void  init_arrays      (void);
float bode             (int x);
//---------------------------------------------------
void init_graphics(void)
{
  int gdriver = EGA, gmode = EGAHI, errorcode;
  //
  // register EGAVGA_driver which is the name of the driver in EGAVGA.OBJ
  // EGAVGA.OBJ is created from EGAVGA.BGI using the BGIOBJ.EXE utility
  // and is linked by either the link list or project file
  //
  // NOTE: This section of code can be ommitted if EGAVGA.BGI is
  //       located in the applications startup directory
  //
  errorcode = registerbgidriver(EGAVGA_driver);
  if (errorcode < 0)  // report any registration errors
  {
    printf("Graphics error: %s\n", grapherrormsg(errorcode));
    printf("Press any key to halt:");
    getch();
    exit(1); // terminate with an error code
  }
  initgraph(&gdriver, &gmode, "");        // if possible open EGA mode
  errorcode = graphresult();
  if (errorcode != grOk)
  { printf("Graphics error: %s\n", grapherrormsg(errorcode));
    printf("Press any key to halt:");
    getch();
    exit(1);
  }
  clearviewport();
}
//---------------------------------------------------------
int findbin(float fsig)
{
  float ft;
  float Hz_Bin;
  int l;
  int bin;
  // Find Hz/Bin, then determine if this will coincide to <1% error
  Fsr = 1/(2*RA*RB*(2*THx * T0_prdv));
  Hz_Bin = Fsr/(float)SFFTSIZE;
  bin = floor(SFFTSIZE * fsig / Fsr);
  SFFTSIZE = SFFTsz1 * 0.9;
  for(l=0;l<200;l++)
  {
    Fsr = 1/(2*RA*RB*(2*THx * T0_prdv));
    Hz_Bin = Fsr/(float)SFFTSIZE;
    bin = 0.5 + (SFFTSIZE * fsig/Fsr); // Include rounding
    ft = bin * Hz_Bin;
    ft = ft/fsig;
    //
    // Keeping the bin number relatively high, >10, will give
    // better isolation or rejection ratio of neighboring bins
    //
    // If within 5% of the desired frequency, adjust the
    // SFFT length for a better fit
    //
    // If the nominal length is 512 pts, the frequency
    // error can be adjusted to apx 1/512 of 5% or 0.01%
    //
    if(   (ft<1.1)&&(ft>0.9) && (bin>10)    )
    {
      // Adjust SFFT length to match signal
      // if
      //      factual > fsig, the best fit bin is higher than fsig
      //                      the bin freq needs to be lowered
      //
      if(ft==1.0) return bin;
      SFFTSIZE = 0.5 + SFFTSIZE * ft;   // Adjust SFFT size with rounding
      //
      // This next iteration should not actual improve the
      // accuracy.  It is here for a test...
      //
      Hz_Bin = Fsr/(float)SFFTSIZE;
      bin = 0.5+SFFTSIZE * fsig/Fsr; // Include rounding
      maxft = fabs(((bin * Hz_Bin)/fsig) - 1.0);
      return bin;
    }
    T0_prdv+=1;
  }
  return bin;
}

int inline xwarp(float in)
{
  return(1 + 50*log(in));
}
//---------------------------------------------------------
void graph(void)
{
  int dB;
  int Yt;
  int Y;
  int x;
  int d;
  float f;
  float f1;
  float f2;
  int xc;
  long OldT0;
  char MSG[32];
  init_graphics();
  setcolor(WHITE);
  moveto(0,0);
  //------------------------------------------
  // Plot vertical lines and Freq markers
  //------------------------------------------
  setcolor(LIGHTGRAY);
  if(graphmode==0)
  {
    for(d=0;d<11;d++)
    {
      x = d * (float)SFFTSIZE/20.0;
      f2 = Fsr * x / SFFTSIZE;
      if((f2>=  0)&&(f2<1E3)) sprintf(MSG,"%0.1f" ,f2     );
      if((f2>=1E3)&&(f2<1E6)) sprintf(MSG,"%0.1fk",f2*1E-3);
      if((f2>=1E6)&&(f2<1E9)) sprintf(MSG,"%0.1fM",f2*1E-6);
      xc = Xd*d/10.0;
      line(xc-1,Yd,xc-1,Yd+4);
      line(xc  ,Yd,xc  ,Yd+4);
      line(xc+1,Yd,xc+1,Yd+4);
      outtextxy(xc,Yd+10,MSG);
      setcolor(GREEN);
      line(xc,0,xc,Yd);
      setcolor(LIGHTGRAY);
    }
    xc = Xd;
  }
  else  // Log frequency display
  {
    f1=1;
    f2=1;
    setcolor(LIGHTGRAY);
    for(f=1;f<100e3;)
    { if(f1==1)
      { if((f2>=  0)&&(f2<1E3)) sprintf(MSG,"%0.0fhz" , f1*f2);
        if((f2>=1E3)&&(f2<1E6)) sprintf(MSG,"%0.0fkhz",(f1*f2)*1E-3);
        if((f2>=1E6)&&(f2<1E9)) sprintf(MSG,"%0.0fMhz",(f1*f2)*1E-6);
        xc = xwarp(f1*f2);
        line(xc-1,Yd,xc-1,Yd+4);
        line(xc  ,Yd,xc  ,Yd+4);
        line(xc+1,Yd,xc+1,Yd+4);
        outtextxy(xc,Yd+10,MSG);
      }
      f = f1 * f2;
      xc = xwarp(f1*f2);
      if(xc<0) xc=0; if(xc>Xd) xc=Xd;
      setcolor(GREEN);
      line(xc,0,xc,Yd);
      setcolor(LIGHTGRAY);
      f1++;
      if(f1>=10)
      {
        f1=1;
        f2*=10;
      }
      f = f1*f2;
      if(f> Fsr) break;
    }
  }
  //------------------------------------------
  // Plot horizontal lines and dB markers
  //------------------------------------------
  setcolor(GREEN);
  Yt = yd-11;
  dB = 20;
  for(Y=0;Y<=Yd;Y+=yd)
  {
    line(0,Y,xc,  Y);  // draw reticle
    if(Y<Yd)
    {
      sprintf(MSG,"%+02d",dB);
      outtextxy(2,Yt,MSG);
    }
    dB -= 10;
    Yt += yd;
  }
  //------------------------------------------
  // Now graph the response
  //------------------------------------------
  setcolor(WHITE);
  moveto(0,Yd);
  // Graph response
  //
  OldT0 = T0_prdv;
  if(graphmode==2)
  {
    for(f=10;f<22e3;f*=1.1)
    {
      // Adjust the sample rate, SFFT size and center bin to yield
      // a low error compared to the desired frequency.
      //
      T0_prdv = OldT0;
      x = findbin(f);    // This bin twiddle is used for siggen
      if(bode(x)!=1) break;
    }
  }
  else
  {
    for(x=0;x<SFFTSIZE/2;x++)
    {
      // Adjust the sample rate, SFFT size and center bin to yield
      // a low error compared to the desired frequency.
      //
      if(bode(x)!=1) break;
    }
  }
  getch();
  closegraph();
}
//-----------------------------------------------
float bode(int x)
{
  float dBscale;
  float MAG;
  float Hz_Bin;
  long OldT0;
  int msdelay;
  int yc=0,xc;
  dBscale = yd / 20; //yd/20*log10(10^.5);
  OldT0 = T0_prdv;
  T0_prdv = OldT0;
  BIN_END = x;     // log  // This bin twiddle is used for siggen
  BIN_START = BIN_END-1;   // lower guardband bin
  Fsr = 1/(2*RA*RB*(2*THx * T0_prdv));
  Hz_Bin = Fsr/(float)SFFTSIZE;
  init_arrays();
  HALT_CPU();
  if(putmem(TWD_LOC     ,6,(ulong *)TW)!=NO_ERR) return 0;
  if(putmem(T_LOC      ,1,   &T0_prdv)!=NO_ERR) return 0;
  if(putmem(TRG_LOC    ,1,     &ENTRY)!=NO_ERR) return 0;
  if(putmem(SIZE_LOC,1,  &SFFTSIZE)!=NO_ERR) return 0;
  //
  // Start the signal generator/filter and allow the applications filter
  // to settle down.  Note that this delay requires an additional settling
  // time if the external filter being measured is of an exceptionaly high
  // order, or has a very long group delay.
  //
  RUN_CPU();
  //
  // 'msdelay' is the time it takes for the AIC and SFFT buffer to settle
  // into a steady state condition after restart. 'ext_delay is then added
  // to compensate for external filter delays.
  //
  // For very low frequencies, the time to fill the buffer can be quite
  // long.  The delay function is broken into 10mS chunks to allow the
  // user a way to break the for(;;) loop.
  //
  #define ext_delay 100  // SFFT 1024pt/20Khz ~= 50 ms
  msdelay = 1.25*1000.0 * ((float)(SFFTSIZE+20)/Fsr); // Buffer fill + AIC
  msdelay += ext_delay;
  for(;msdelay>0;msdelay-=10)
  {
    delay(10);
    if(kbhit()) break;
  }
  delay(10);
  if(kbhit()) return 0;
  //
  // The data in the SFFT bins should now be at a steady
  // state condition.  Halt the CPU and get the data.
  //
  HALT_CPU();
  if(getmem(BIN_LOC ,6,(ulong *)BIN)!=NO_ERR) return 0;
  RUN_CPU();
  //
  // Display the magnitude of the measured response
  //
  MAG = BINMAG();
  if(MAG<1E-6) MAG=1E-6;
  yc = 360-dBscale*20*log(MAG);

  if(graphmode==0)
  {
//  if(x>0)
      xc = 2*Xd*(float)x/(float)SFFTSIZE;// Change for log
  }
  else
  {
    if(x>0)
      xc = xwarp(BIN_END * Hz_Bin);
    else
      xc = 0;
  }

  if(yc<0) yc=0; if(yc>Yd) yc=Yd;
  if(xc<0) xc=0; if(xc>Xd) xc=Xd;
  lineto(xc,yc);
  putpixel(xc  ,50+10000*maxft,YELLOW);
  putpixel(xc  ,25+(int)T0_prdv,CYAN);
  moveto(xc,yc);
  return 1;
}
//--------------------------------------------------
MSGS  putmemf(ulong DEST,float FVAL)
{
  ulong temp;
  temp = IEEE_TMS(FVAL);
  return putmem(DEST,1,&temp);
}
//---------------------------------------------
void main(void)
{
  int reset_flag = 0;
  MSGS err;
  clrscr();
  Scan_Command_line(DSK_EXE_APP);
  clrscr();
  //------------------------------------------------
  // The outer loop initializes the DSK on entry
  // or if the application fails
  //------------------------------------------------
  for(;;)
  {
    for(;;)
    { if(Init_Communication(10000) == NO_ERR) break;
      if(kbhit()) exit(0);
    }
    HALT_CPU(); // Halt previously running apps code
    if((err=Load_File(DSK_APP,LOAD))!=NO_ERR)
    { printf("%s %s\n",DSK_APP,Error_Strg(err));
      exit(0);
    }
    //-----------------------------------------------
    // Get the location of the coefficients from the
    // symbol table which has been loaded into the host
    //-----------------------------------------------
    // If COFF version 5.0
    /*
    ref_value("_T_LOC"  , &(long)T_LOC);
    ref_value("_A_LOC"  , &(long)A_LOC);
    ref_value("_B_LOC"  , &(long)B_LOC);
    ref_value("_C_LOC"  , &(long)C_LOC);
    ref_value("_TRG_LOC", &(long)TRG_LOC);
    ref_value("_BIN"    , &(long)BIN_LOC);
    ref_value("_TW"     , &(long)TWD_LOC);
    ref_value("_SZ_LOC" , &(long)SIZE_LOC); // Max SFFT allocation
    getmem(SIZE_LOC,1,&SFFTSIZE);
    SFFTsz1 = SFFTSIZE;
    SFFTSIZE *= 0.9;
    ref_value("_SFFTSIZE",&(long)SIZE_LOC);  // Runtime SFFT size
    putmem(SIZE_LOC,1,&SFFTSIZE);
    getmem(SIZE_LOC,1,&SFFTSIZE);
    */
    // If COFF = 4.70
    ref_value("_T_LOC"  , &(long)T_LOC);
    ref_value("_A_LOC"  , &(long)A_LOC);
    ref_value("_B_LOC"  , &(long)B_LOC);
    ref_value("_C_LOC"  , &(long)C_LOC);
    ref_value("_TRG_LOC", &(long)TRG_LOC);
    ref_value("_BIN"    , &(long)BIN_LOC);
    ref_value("_TW"     , &(long)TWD_LOC);
    ref_value("_SZ_LOC" , &(long)SIZE_LOC); // Max SFFT allocation
    getmem(SIZE_LOC,1,&SFFTSIZE);
    SFFTsz1 = SFFTSIZE;
    SFFTSIZE *= 0.9;
    ref_value("_SFFTSIZE",&(long)SIZE_LOC);  // Runtime SFFT size
    putmem(SIZE_LOC,1,&SFFTSIZE);
    getmem(SIZE_LOC,1,&SFFTSIZE);


    //-----------------------------------------------
    RUN_CPU();  // Start the application running
    clrscr();
    check_key();
    update_DSK_params();
    //***********************************************************//
    // The inner loop is repeated until a keyboard hit exits the //
    // application or the application reports an error and needs //
    // to be reinitialized.                                      //
    //***********************************************************//
    for(;;)
    { for(;;)
      { //
        // Wait for a key to be pressed, and then update the generator
        // analyzer if directed to do so.
        //
        if(kbhit())
        {
          reset_flag = check_key();
          break;
        }
        delay(1);
      }
      if(reset_flag || (err!=NO_ERR)) break;
    }
    printf("Communications are being reinitialized");
    DSK_reset();
  }
}
//---------------------------------------------
int inline clipint(int x, int min, int max)
{
  if(x<min) return min;
  if(x>max) return max;
  return x;
}
//---------------------------------------------
long inline cliplong(long x, long min, long max)
{
  if(x<min) return min;
  if(x>max) return max;
  return x;
}
//---------------------------------------------
void get_float(float *f, char *s)
{
  float ftemp;
  char strng[40];
  printf("%s=%7.8f ",s,*f);
  scanf(" %20s",&strng);
  ftemp = atof(strng);
  *f = ftemp;
}
//---------------------------------------------
float BINMAG(void)
{
  float RSUM;
  float ISUM;
  float MAG;
  RSUM  =-TMS_IEEE(BIN[0].R);
  RSUM += TMS_IEEE(BIN[1].R);
  RSUM += TMS_IEEE(BIN[1].R);
  RSUM -= TMS_IEEE(BIN[2].R);

  ISUM  =-TMS_IEEE(BIN[0].I);
  ISUM += TMS_IEEE(BIN[1].I);
  ISUM += TMS_IEEE(BIN[1].I);
  ISUM -= TMS_IEEE(BIN[2].I);
  MAG = RSUM*RSUM + ISUM*ISUM;
//  MAG+=1;
  if(MAG!=0)
    MAG = sqrt(MAG);
  return MAG;
}
//---------------------------------------------
int check_key(void)
{
  char strng[64];
  int key=0, y;
  //
  // If the last keystroke caused a DTMF signal to be generated
  // wait for the key to be released before moving on.  Note
  // that the keyboards typematic rate is a potential problem here
  // and that the minimum DTMF period is 100mS
  //
  if(kbhit())
  {
// if(bioskey(2) & 3)  shift_key = 1;
    key = bioskey(0) & 0xFF00;
  }

  if(key)
  {
    switch(key)
    {
      //============== Quit (with run optional), Reset ======//
      case _R  : return 1;  // Return reset (break) flag
      case _Q  : clrscr();
                 _setcursortype(_NORMALCURSOR);
                 DSK_reset();
                 exit(0);
                 break;
      //============== Timer period setup ===================//
      case _F9 : T0_prdv -=1; break;
      case _F10: T0_prdv +=1; break;
      //============== Modify DAC paramerters ===============//
      case _F1 : TA++; break;
      case _F2 : TA--; break;
      case _F3 : TB++; break;
      case _F4 : TB--; break;
      //============== Modify ADC paramerters ===============//
      case _SF1: RA++; break;
      case _SF2: RA--; break;
      case _SF3: RB++; break;
      case _SF4: RB--; break;
      //=========== Graph the response =======================//
      case _G:  graph(); break;
      case _M:  graphmode+=1; if(graphmode>2) graphmode=0; break;
      //=========== Amplitude level adjustment ===============//
/*    case _F7 : H.Ampli = (H.Ampli * 1.1) + .0001; break;
      case _F8 : H.Ampli = (H.Ampli / 1.1) - .0001; break;
      case _A  : get_float(&H.Ampli,"A"); break;*/
      //============== Frequency adjustment ==================//
      case _F5 : BIN_START++; break;
      case _F6 : BIN_START--; break;
      case _F  : get_float(&freq,"F");
                 T0_prdv = 1;
                 BIN_END = findbin(freq);
                 BIN_START = BIN_END -1;
                 break;
      //============== AIC control register bit set ==========//
      case _1: C_REG = C_REG ^ 0x80; break; /* Gain 1         */
      case _2: C_REG = C_REG ^ 0x40; break; /* Gain 2         */
      case _3: C_REG = C_REG ^ 0x20; break; /* Fadc=Fdac      */
      case _4: C_REG = C_REG ^ 0x10; break; /* AUX enable     */
      case _5: C_REG = C_REG ^ 0x08; break; /* Loop back      */
      case _6: C_REG = C_REG ^ 0x04; break; /* Band pass      */
      //============== Default - do nothing  =================//
      case _Spc: break;
      default : return 0;
    }
    TA      = clipint (TA     ,  3, 31); // Clip the AIC timer registers
    TB      = clipint (TB     , 12, 63); // to values that typicly work
    RA      = clipint (RA     ,  3, 31);
    RB      = clipint (RB     , 12, 63);
    T0_prdv = cliplong(T0_prdv,  1,200);
  }
  clrscr();
  //
  // Update Fsx and Fsr frequency
  //
  Fsx = 1/(2*TA*TB*(2*THx * T0_prdv));
  if(C_REG & 0x20) Fsr = Fsx;
  else             Fsr = 1/(2*RA*RB*(2*THx * T0_prdv));
  freq = Fsr * ((float)(BIN_END)/(float)SFFTSIZE);
  if(freq > 1e9)   freq  = 1e9; if(freq    < -1e9)   freq  = -1e9;

  BIN_END   = BIN_START+1;
  if(update_DSK_params()!=NO_ERR) return 1; // reload parameters
  //
  binsprintf(strng,(int)C_REG);
  for(y=2;y<10;y++)
  {
    gotoxy(72,y);
    printf("");
  }
  gotoxy(1,1);
  printf(
"ͻ\n"
"  F1:F2  TA  = %02d  (xmit=dac)       Fdac=%7.2f\n"
"  F3:F4  TB  = %02d                   Fadc=%7.2f\n"
" SF1:SF2 RA  = %02d  (recv=adc)       [1-6] set AIC ctrl bits\n"
" SF3:SF4 RB  = %02d                   \n"
"  F5:F6  Bin = %04ld                 \n"
"  F9:F10 TIM0= %02ld  (timer 0 )       %sb AIC ctrl word\n"
"͹\n"
" (Q)uit (R)eset (M)ute         [F5:F6](F)requency (center bin)       \n"
"ͼ\n",
  TA,Fsx,TB,Fsr,RA,RB,BIN_END,T0_prdv,strng);

  textcolor(WHITE);
  switch(graphmode)
  {
    default:
    case 0:cprintf("[G]raph [M]ode = Lin Fscale, Lin Fsweep\n"); break;
    case 1:cprintf("[G]raph [M]ode = Log Fscale, Lin Fsweep\n"); break;
    case 2:cprintf("[G]raph [M]ode = Log Fscale, Log Fsweep"
                  " -> Fs is adjusted for best fit \n");
           break;
  }
  printf("\r");
  textcolor(LIGHTGRAY);

  printf("Fsr        = %1.3f/%ld  khz/pts\n",Fsr/1000.0,SFFTSIZE);
  printf("Hz/Bin     = %1.3f\n"  ,(float)(Fsr/SFFTSIZE));
  printf("Center bin = %ld\n"    ,BIN_END);
  printf("Fsig       = %1.6f khz\n\n",(float)freq/1000.0);

  printf("Fsx SCF rate : %8.3f khz\n"  ,Fsx*TB/1000.0);
  printf("Fsx          : %8.3f khz\n"  ,Fsx   /1000.0);
  printf("Fsx cutoff   : %8.3f khz\n\n",3.700 * Fsx * TB/288000.0);

  printf("Fsr SCF rate : %8.3f khz\n" ,Fsr*RB/1000.0);
  printf("Fsr          : %8.3f khz\n" ,Fsr/1000.0);
  printf("Fsr cutoff 1 : %8.3f khz\n" ,0.300 * Fsr * RB/288000.0);
  printf("Fsr cutoff 2 : %8.3f khz\n"   ,3.500 * Fsr * RB/288000.0);
  return 0;
}
//---------------------------------------------------------
void init_arrays(void)
{
  int x;
  float n;
  n = BIN_START;
  for(x=0;x<SFFTBINS;x++)
  {
    TW[x].R = IEEE_TMS(K1*cos(n*w)); // R/I phase or twiddle coefficients
    TW[x].I = IEEE_TMS(K1*sin(n*w));
    n+=1.0;
  }
}


MSGS update_DSK_params(void)
{
  ulong A_REG;
  ulong B_REG;
  MSGS err = NO_ERR;
  //
  // Load the new SFFT and signal generator coefficients.  This
  // includes the R/I SFFT bins and Twiddles
  // K2 is updated within the DSP
  // A non-zero LOAD_FLAG signals DSK to restart the
  // application with the new runtime parameters
  //
  init_arrays();  // Calculate twiddles
  A_REG = ((TA<<9)+(RA<<2)+0); // A divisors.. set SCF rate
  B_REG = ((TB<<9)+(RB<<2)+2); // B divisors
  HALT_CPU();
  //
  if(TB<TA)
  { if((err=putmem(A_LOC,1,&A_REG))!=NO_ERR) return err;
    if((err=putmem(B_LOC,1,&B_REG))!=NO_ERR) return err;
  }
  else
  { if((err=putmem(A_LOC,1,&B_REG))!=NO_ERR) return err;
    if((err=putmem(B_LOC,1,&A_REG))!=NO_ERR) return err;
  }
  if((err=putmem(C_LOC,1,&C_REG))!=NO_ERR) return err;
  if((err=putmem(TWD_LOC,6, (ulong *)TW   ))!=NO_ERR) return err;
  if((err=getmem(BIN_LOC,6, (ulong *)BIN  ))!=NO_ERR) return err;
  if((err=putmem(T_LOC      ,1, &T0_prdv))!=NO_ERR) return err;
  if((err=putmem(TRG_LOC,1, &ENTRY))!=NO_ERR) return err;
  RUN_CPU();
  return err;
}

void binsprintf(char *s,int val)
{
  char *p;
  unsigned int t;
  p = s;
  t = 0x80;  // scan 8 bits
  for(;t>0;t=t>>1)
  {
    if(val & t) *p++ = '1';
    else        *p++ = '0';
  }
  *p=0;
}
