// This little program is functionally equivalent to the CHOICE command supplied
// with later versions of DOS, NT, and WIN95.  It is useful with older versions
// of DOS where the user does not have the choice of using CHOICE.
//
// Robert A. Eastman
// 25 Feb, 97

// *** HEADER FILES ***
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <conio.h>
#include <bios.h>
#include <time.h>


// *** MACROS ***
#ifndef BOOL
#define BOOL int
#endif
#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif
#define BAD_CHOICE 255

// Help text
#define HELP \
"\n\
Waits for the user to pick one of a set of choices.\n\
\n\
PICK [/C[:]<set>] [/N] [/S] [/T[:]c,nn] [<text>]\n\
/C[:]<set>  Allowable keys, such as 'abcdef'. Default is YN. If key pressed\n\
             is not part of <set>, then PICK waits for another key.\n\
/N          Do not display choices and ? at end of prompt string.\n\
/S          Treat <set> keys as case-sensitive. If key pressed does not match\n\
             case of character in <set>, then PICK waits for another key.\n\
/T[:]c,nn   Default to choice c after nn seconds. If c is not part of <set>,\n\
             then PICK returns after timeout period with ERRORLEVEL set to 255.\n\
<text>      Additional prompt text. Must be last.\n\
ERRORLEVEL is set to offset of key user presses in <set>, beginning with 1.\n\
\n\
Example:  PICK /C:abcdef  /TB,30 Pick a key\n\
PICK will display:  Pick a key [A,B,C,D,E,F]?\n\
If user selects 'd' then PICK will return with ERRORLEVEL set to 4.\n\n\
If user does nothing, 'B' will be selected after 30 seconds and PICK will\n\
return with ERRORLEVEL set to 2."


// *** FUNCTION PROTOTYPES ***
int  main( int argc, char** argv );
BOOL ParseCmdLine( int argc, char** argv );
void EmptyKybdBuffer( void );
char GetKey( void );
char WaitForKey( void );
void StartClock( void );
BOOL CheckClock( void );
void IssuePrompt( void );
int  GetChoice( void );
int  GetCharPosn( char* s, char c );


// *** PRIVATE VARIABLES ***
static clock_t s_startTime;
// The following variables are set by cmd line options
static int     s_iInterval     = 0;       // Timeout interval, seconds
static BOOL    s_bTimed        = FALSE;   // Default to choice after timeout
static BOOL    s_bPrintChoices = TRUE;    // Print choices
static BOOL    s_bIgnoreCase   = TRUE;    // Ignore case in choices
static char    s_sPrompt[256];            // Additional prompt text
static char*   s_sChoices      = "\0";    // Set of choices
static char    s_cDefaultChar  = '\0';    // Choice for timeout


// *** EXECUTABLE CODE ***
//---------------------------------------------------------------------------
int main( int argc, char** argv )
{
  if( !ParseCmdLine( argc, argv ) )
    return BAD_CHOICE;
  return GetChoice();
}

//---------------------------------------------------------------------------
BOOL ParseCmdLine( int argc, char** argv )
{
  char* p;

  s_sPrompt[0] = '\0';  // Assume no user prompt text
  s_sChoices = "YN";    // Default choices (yes/no)

  if( argc >= 2 )
  {
    // Get each argument string and parse it
    for( int i = 1; i < argc; i++ )
    {
      p = argv[i];
      if( '/' == *p )  // Options are preceeded by a slash, arguments not
                       // beginning with a slash are considered the beginning
                       // of user prompt text
      {
        p++;  // Advance past slash
        switch( toupper( *p ) )
        {
          case 'C':  // Set of choices follows
            p++;     // Advance past 'C'
            if( ':' == *p )
              p++;   // Advance past colon
            s_sChoices = p;
            break;

          case 'N':  // Don't display choices and ? at end of prompt
            s_bPrintChoices = FALSE;
            break;

          case 'S':  // Treat choice as case-sensitive
            s_bIgnoreCase = FALSE;
            break;

          case 'T':  // Default choice to <c> after nn seconds
            s_bTimed = TRUE;
            p++;     // Advance past 'T'
            if( ':' == *p )
              p++;   // Advance past colon
            s_cDefaultChar = *p;
            ++++p;   // Advance past character and comma
            s_iInterval = atoi( p );
            break;

          case 'F':  // Just for fun
            printf( "\n\nThe TMS320C3x DSK is Way Cool!\n\n" );
            break;

          case '?':  // Help requested
          default:
            printf( HELP );
            return FALSE;
        } // End switch
      } // End if( first char is slash )
      else
      {
        // Additional prompt text.  Remainder of cmd line is treated as text
        for( int j = i; j < argc; j++ )
        {
          strcat( s_sPrompt, argv[j] );
          strcat( s_sPrompt, " " );
        }
        break;
      } // End else( first char not slash )
    } // End for( all cmd line arguments )
  } // End if( at least two arguments )

  return TRUE;
}

//---------------------------------------------------------------------------
void EmptyKybdBuffer( void )
{
  while( kbhit() )
    getchar();
  return;
}

//---------------------------------------------------------------------------
char GetKey( void )
{
  int c = bioskey( 0 ) & 0x00FF;
  if( s_bIgnoreCase )
    return toupper( c );
  else
    return c;
}

//---------------------------------------------------------------------------
char WaitForKey( void )
{
  while( !kbhit() );
    return GetKey();
}

//---------------------------------------------------------------------------
void StartClock( void )
{
  s_startTime = clock();
  return;
}

//---------------------------------------------------------------------------
BOOL CheckClock( void )
{
  return( ( ( clock() - s_startTime ) / CLK_TCK ) > s_iInterval );
}

//---------------------------------------------------------------------------
void IssuePrompt( void )
{
  char s[ 256 ];
  char* src, dest;

  if( s_bIgnoreCase )
    strupr( s_sChoices );
  s[0] = '\0';
  strcpy( s, s_sPrompt );
  if( s_bPrintChoices )
  {
    // Assemble choices string, separated by commas - ie., [A,B,C]
    strcat( s, "[" );
    {
      char* dest = s + strlen( s );
      char* src = s_sChoices;
      while( *dest++ = *src++ )
        *dest++ = ',';
      *(----dest) = '\0';  // Overwrite last comma & terminate string
    }
    strcat( s, "]?" );
  }
  printf( "%s", s );

  return;
}

//---------------------------------------------------------------------------
int GetChoice( void )
{
  int n;
  char c;

  if( s_bIgnoreCase )
    s_cDefaultChar = toupper( s_cDefaultChar );
  IssuePrompt();
  EmptyKybdBuffer();
#if 0
  if( s_bTimed )
  {
    StartClock();
    while( !kbhit() )
      if( CheckClock() )
      {
        n = GetCharPosn( s_sChoices, s_cDefaultChar );
        printf( "%c\n", s_cDefaultChar );
        break;
      }
  }
  else
  {
    while( BAD_CHOICE == ( n = GetCharPosn( s_sChoices, c = WaitForKey() ) ) );
    printf( "%c\n", c );
  }
#endif
// TEST - new
  if( s_bTimed )
  {
    StartClock();
    while( !CheckClock() )
    {
      if( kbhit() )
      {
        if( BAD_CHOICE != ( n = GetCharPosn( s_sChoices, c = GetKey() ) ) )
        {
          printf( "%c\n", c );
          return n;
        }
      }
    }
    // Timeout
    n = GetCharPosn( s_sChoices, s_cDefaultChar );
    printf( "%c\n", s_cDefaultChar );
    return n;
  }
  else
  {
    while( BAD_CHOICE == ( n = GetCharPosn( s_sChoices, c = WaitForKey() ) ) );
    printf( "%c\n", c );
    return n;
  }
// TEST - end new
}

//---------------------------------------------------------------------------
int GetCharPosn( char* s, char c )
{
  if( s_bIgnoreCase )
  {
    c = toupper( c );
    strupr( s );
  }

  for( int i = 0; i < strlen( s ); i++ )
    if( c == s[ i ] )
      return i + 1;
  return BAD_CHOICE;
}
