//-----------------------------------------
// DASMC40.CPP
// Keith Larson
// TMS320 DSP Applications
// (c) Copyright 1995, 1996, 1997
// Texas Instruments Incorporated
//
// This is unsupported freeware code with no implied warranties or
// liabilities.  See the disclaimer document for details
//----------------------------------------------------------
// This file contains the diassembler function forms for the
// TMS320C3x.  Also included are the functions needed to
// form the register, indirect and direct addressing mode
// substrings
//----------------------------------------------------------
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "opcodes.h"
#include "tmsfloat.h"
#include "symbols.h"
#include "dsk.h"
ulong CURRENT_DP = 0x00000080; // Current processor data page
char  C3Xmode = 1;
ulong PC_Address; // used for PC-Relative addressing mode
//---------------------------------------------------------------------
//      syntax :   <instruction> <src>,<dst>
//     example :   <buf> = "ABSF" <src> = "R1"  <dst> = "R0"
//                     <buf> = "ABSF R1,R0"
//---------------------------------------------------------------------
static char *Concat2( char *buf, char *src, char *dst )
{
  char b[80];
  strcpy(b,buf);
  if ( *dst ) sprintf( buf, "%s %s,%s", b, src, dst );
  else sprintf( buf, "%s %s",b,src);
  return buf;
}
//---------------------------------------------------------------------
//   syntax :  <instruction> <src1>,<src2>,<dst>
//  example :  <buf> = "ASH3"  <src1> = "R0" <src2> = "R1" <dst> = "R7"
//             <buf> = "ASH3 R0,R1,R7"
//---------------------------------------------------------------------
static char *Concat3( char *buf, char *src1, char *src2, char *dst )
{
  char b[80];
  strcpy(b,buf);
  if ( *dst ) sprintf( buf, "%s %s,%s,%s", b, src1, src2, dst );
  else sprintf( buf, "%s %s,%s", b, src1, src2);
  return buf;
}
//---------------------------------------------------------------------
//  Extract 2 strings from src, separated by sep
//
//  example:  <src> = "MPYI3 || STI"  and <sep> = "||"
//            <dst1> = "MPYI3 "  <dst2> = "|| STI"
//---------------------------------------------------------------------
static void Extract2( char *src, char *dst1, char *dst2, char *sep )
{
  char *p;
  strcpy(dst1,src);
  p = strstr(dst1,sep);
  *(p-1)=0;
  sprintf( dst2, " %s",p);
}
/************************************************************************
              Decoding for General Addressing Modes
    ABSF,  ADDF , CMPF, FLOAT, LDE , LDF  , LDFI, LDM , MPYF, NEGF,
    POPF,  PUSHF, RCPF, RND  , SUBF, SUBRF, NORM, LDFc, FIX , TOIEEE
***************************************************************************/
int DForm0( int n, ulong instr, char *buffer )
{
  uint f_src_src2_disp;    //  8 bits,  0- 7
  uint f_src_src2_ARn ;    //  3 bits,  8-10
  uint f_src_src2_modn;    //  5 bits, 11-15
  uint f_src_src1     ;    // 16 bits,  0-15
  uint f_dst          ;    //  5 bits, 16-20  destination
  uint f_g            ;    //  2 bits, 21-22  t switch
//uint f_code         ;    //  9 bits, 23-31  opcode
  char src[80], dst[80]; //, tmp[80];
  f_src_src2_disp = (uint)((instr      ) & 0x000FF);
  f_src_src2_ARn  = (uint)((instr >>  8) & 0x00007);
  f_src_src2_modn = (uint)((instr >> 11) & 0x0001F);
  f_src_src1      = (uint)((instr      ) & 0x0FFFF);
  f_dst           = (uint)((instr >> 16) & 0x0001F);
  f_g             = (uint)((instr >> 21) & 0x00003);
//f_code          = (uint)((instr >> 23) & 0x001FF);

  if (f_dst >= MAXREG) return 1;
  strcpy(buffer,Instr[n].instr);
  strcpy(dst,C30Reg[f_dst].s);
  switch ( f_g )
  { // src = register
    case 0: if(f_src_src1 >= MAXREG) return 1;

         // if((f_src_src1 > 0x07) && (f_src_src1 < 0x1C)) return 1;//C40

            if(Instr[n].en != __FLOAT)  // Any reg for src is valid
              if(f_src_src1 > 0x07) return 1;//C30

            strcpy(src,C30Reg[f_src_src1].s);
            break;

    case 1: *src='@';
            if(get_label(src+1,(CURRENT_DP << 16) | f_src_src1,0)) break;
            sprintf( src, "@0%04xH",f_src_src1);
            break;

    case 2: if( f_src_src2_modn > MAX_MOD ) return 1;
            if( f_src_src2_modn < 8 )
               sprintf( src, Mod[f_src_src2_modn], f_src_src2_ARn, f_src_src2_disp);
            else
            {
              if(f_src_src2_disp) return 1;
              sprintf( src, Mod[f_src_src2_modn], f_src_src2_ARn);
            }
            break;

    case 3: //if(f_code == 0x12)  // LDM
            if(Instr[n].en == __LDM)
            {
              if(f_src_src1 & 0xF000) return 1;
            }
            if(Instr[n].en == __FLOAT)
              sprintf( src, "%4d",(int)f_src_src1);
            else
              sprintf( src, "%10e",Short_Float(f_src_src1));
            break;
    default:return 1;
  }
//if((f_dst > 0x07) && (f_dst < 0x1C)) return 1;  // C40
  if(Instr[n].en!=__FIX)
    if(f_dst > 0x07)                      return 1;  // C30
  /***********************************/
  /* Special case exception handling */
  /***********************************/
  switch(Instr[n].en)
  {
    case __LDFI : if ((f_g!=1) && (f_g!=2) ) return 1;
                  break;
    case __POPF :
    case __PUSHF: strcpy(src,dst);
                  *dst = 0;
    default     : break;
  }
  Concat2( buffer, src, dst );
  return 0;
}
/* ----------------------------------------------------------------------
              Decoding for General Addressing Modes
  Instructions handled :

  ABSI, ADDC , ADDI, AND, ANDN, ASH , CMPI,
  LDP , LDI  , LDII, LSH, MPYI, NEGB,
  NEGI, NOP  , NOT , POP, PUSH, OR  , ROL , ROLC,
  ROR , RORC , RPTS, STF, STFI, STIK, STI , STII, SUBB, SUBC, SUBI,
  SUBRB,SUBRI, TSTB, XOR, IACK,
  LDIc , LDA , FRIEEE
   ---------------------------------------------------------------------- */

int DForm1( int n, ulong instr, char *buffer )
{
  char src[80], dst[80], tmp[80];
  uint f_src_src2_disp;    //  8 bits,  0- 7
  uint f_src_src2_ARn ;    //  3 bits,  8-10
  uint f_src_src2_modn;    //  5 bits, 11-15
  uint f_src_src1     ;    // 16 bits,  0-15
  uint f_dst          ;    //  5 bits, 16-20  destination
  uint f_g            ;    //  2 bits, 21-22  t switch
//uint f_code         ;    //  9 bits, 23-31  opcode

  f_src_src2_disp = (int)((instr      ) & 0x000FF);
  f_src_src2_ARn  = (int)((instr >>  8) & 0x00007);
  f_src_src2_modn = (int)((instr >> 11) & 0x0001F);
  f_src_src1      = (int)((instr      ) & 0x0FFFF);
  f_dst           = (int)((instr >> 16) & 0x0001F);
  f_g             = (int)((instr >> 21) & 0x00003);
//f_code          = (int)((instr >> 23) & 0x001FF);


  if(C3Xmode)                      // All registers valid for C40
    if (f_dst >= MAXREG) return 1;
  /*
  switch(instr >> 28)                // Test for LDFcnd / LDIcnd
  {
    case 0x4:
    case 0x5: if((instr >> 16) != 0x5070)  break;  // Exclude LDI x,DP
              else strcpy(buffer,Instr[n].instr); break;
    default : strcpy(buffer,Instr[n].instr); break;
  }
  */
  strcpy(buffer,Instr[n].instr);
  strcpy(dst,C30Reg[f_dst].s);
  switch ( f_g )
  {
   // Register
   case 0 :
   if(C3Xmode) {if(Instr[n].en == __STIK) return 1;} // STIK exception!
   else        {if(Instr[n].en == __STIK) break;   }

            if(f_src_src1 >= MAXREG) return 1;
            strcpy(src,C30Reg[f_src_src1].s);
            break;
   // Direct
   case 1 :
            *src='@';
            if(get_label(src+1,(CURRENT_DP << 16) | f_src_src1,0)) break;
            sprintf( src, "@0%04xH",f_src_src1);
            break;
   // Indirect
   case 2 : if( f_src_src2_modn>MAX_MOD ) return 1;
            if( f_src_src2_modn < 8 )
               sprintf( src, Mod[f_src_src2_modn], f_src_src2_ARn, f_src_src2_disp);
            else
            {
              if(f_src_src2_disp) return 1;
              sprintf( src, Mod[f_src_src2_modn], f_src_src2_ARn);
            }
            break;
   // Immediate
   case 3 :
            switch(Instr[n].en)
            {

              case __ABSI:  // Integer opcodes that display decimal (not hex)
              case __ADDC:
              case __ASH :
              case __LSH :
              case __NEGB:
              case __SUBRB:
              case __ROR  :
              case __RORC :
              case __SUBB:
              case __SUBC:
              case __CMPI:
              case __ADDI:
              case __MPYI:
              case __SUBRI:
              case __SUBI:
              case __IACK:
              case __NEGI:
              case __LDI :
              case __STIK:
              case __LDII:  sprintf( src, "%d",(int) f_src_src1);
                            break;
                            // Is RPTS value sign extended?
              case __RPTS : sprintf( src, "%u",(int) f_src_src1);
                            break;
            //case __FRIEEE:
            //               sprintf( src, "%8g",Short_Float(f_src_src1));
            //               break;
        //    AND, ANDN, LDP
        //    NOP, NOT, OR, POP, PUSH, ROL, ROLC, STF
        //    TSTB, XOR     // opcodes that bit field displays are better
              default:       sprintf( src, "0%04xh",(int) f_src_src1);
                             break;
            }
            break;
    default:return 1;
  }
  /* -------- exceptions relative to destination and flag operands ---------*/
  if((instr >> 16) == 0x5070)  // LDP
#ifdef DASMTEST
    sprintf(src,"@0%04x0000h",(uint)(instr & 0xFFFF));
#else
    sprintf(src,"0%04xh",(uint)(instr & 0xFFFF));
#endif
  switch(Instr[n].en) // ( f_code )
  {
    case __LDII:
    case __SIGI:if((f_g == 0) || (f_g == 3) ) return 1;
                break;

    case __NOP: *dst = 0;
                if ((f_g ==1) || (f_g == 3) ) return 1;
                if ( f_g == 0 ) *src=0;
                break;

    case __LDA: if(C3Xmode) return 1;         // Enable only for C4x
                if(f_dst <  0x08) return 1;  // dst = R0-R7 not allowed
                if(f_dst <  0x13) break;     // AR0-AR7,DP,IR0,IR1 valid
                if(f_dst == 0x14) break;
                return 1;

    case __FRIEEE:
                if(C3Xmode) return 1;         // Enable only for C4x
                if((f_dst > 0x7) && (f_dst < 0x1C))  return 1;
                if((f_g == 0) || (f_g == 3) ) return 1;
                break;
//  case __POP:
//  case __PUSH: strcpy(src,dst);
//               *dst = 0;
//               break;
    case __ROL :
    case __ROLC:
    case __ROR :
    case __RORC:if (f_g != 3) return 1;
    case __POP:
    case __PUSH:strcpy(src,dst);
                *dst = 0;
                break;
    case __RPTS:
                if ( f_dst != 0x1B ) return 1;
                *dst = 0;
                break;
    case __NORM:
             // if((f_dst > 0x07) && (f_dst < 0x1C)) return 1; //C40
                if(f_dst > 0x07)                      return 1; //C30
                break;
    case __IACK:
                if (( (f_g != 1) && (f_g != 2)) ) return 1;
                *dst=0;
                break;
//  case __LDEP: if(f_src_src1 & 0xFFFE) return 1; // Only ITVP, TVTP allowed
//  /* LDEP */  if(f_src_src1) strcpy(src,"TVTP"); strcpy(src,"IVTP");
//              break;

//  case __LDEP: if(f_dst & 0x1E) return 1; // Only ITVP, TVTP allowed
//  /* LDPE */  if(f_dst ) strcpy(dst,"TVTP"); strcpy(dst,"IVTP");
//              break;
    case __STFI:
    case __STF:
    case __TOIEEE:
             // if((f_dst > 0x07) && (f_dst < 0x1C)) return 1; //C40
                if(f_dst > 0x07)                      return 1; //C30
    case __STI:
    case __STIK:
    case __STII:
                if(Instr[n].en == __STIK)
                {
                  switch (f_g)  // STIK exceptions only
                  {
                    case 0 : *src='@';
                       if(get_label(src+1,(CURRENT_DP<<16) | f_src_src1,0))
                             break;
                             sprintf( src, "@0%04xH",f_src_src1);
                             break;
                    case 3 : if( f_src_src2_modn>MAX_MOD ) return 1;
                             if( f_src_src2_modn < 8 )
                               sprintf(src,
                                       Mod[f_src_src2_modn],
                                       f_src_src2_ARn,
                                       f_src_src2_disp);
                             else
                             {
                               if(f_src_src2_disp) return 1;
                               sprintf(src,
                               Mod[f_src_src2_modn], f_src_src2_ARn);
                             }
                             break;
                    default: break;
                  }
                }
                strcpy(tmp,src); strcpy(src,dst), strcpy(dst,tmp);
                if ((f_g == 0) || (f_g == 3))
                {
                   if(Instr[n].en==__STIK)
                //  if(f_code == 0x2A)  /* STIK - C40 ONLY */
                     sprintf(src,"%d",(((int)f_dst) << 11) >> 11);
                   else
                     return 1; // if C30
                }
                break;
    default   : break;
  }
  Concat2( buffer, src, dst );
  //
  // Special case for LDP when using dasm_assm
  //
  if(dasm_assm_on)
    if((instr & 0xFFFFFF00L) == 0x08700000L)
      sprintf(buffer,"LDP 0%06lxh,DP",(instr & 0xFF)<<16);

  return 0;
}
/* ----------------------------------------------------------------------
              Decoding for Parallel Addressing Modes
     MPYF3 || ADDF3,  MPYF3 || SUBF3,  MPYI3 || ADDI3,  MPYI3 || SUBI3
   ---------------------------------------------------------------------- */
int DForm3( int n, ulong instr, char *buffer )
{
  char srcA[80], srcB[80], srcC[80], srcD[80], tmp[80];
  uint f_ARn          ;    //  3 bits,  0- 2
  uint f_modn         ;    //  5 bits,  3- 7
  uint f_ARm          ;    //  3 bits,  8-10
  uint f_modm         ;    //  5 bits, 11-15
  uint f_src2         ;    //  3 bits, 16-18
  uint f_src1         ;    //  3 bits, 19-21
  uint f_d2           ;    //  1 bits, 22
  uint f_d1           ;    //  1 bits, 23     destination
  uint f_p            ;    //  2 bits, 24-25  t switch
//uint f_code         ;    //  6 bits, 26-31  opcode

  f_ARn           = (int)((instr      ) & 0x00007);
  f_modn          = (int)((instr >>  3) & 0x0001F);
  f_ARm           = (int)((instr >>  8) & 0x00007);
  f_modm          = (int)((instr >> 11) & 0x0001F);
  f_src2          = (int)((instr >> 16) & 0x00007);
  f_src1          = (int)((instr >> 19) & 0x00007);
  f_d2            = (int)((instr >> 22) & 0x00001);
  f_d1            = (int)((instr >> 23) & 0x00001);
  f_p             = (int)((instr >> 24) & 0x00003);
//f_code          = (int)((instr >> 26) & 0x0003F);

  strcat(buffer,Instr[n].instr);
//------------------------------------------------
#if __XON1
  if((instr & 0xE0E0L)==0xE0E0) return 1;
#endif
  if ( (f_modm > MAX_MOD) || (f_modn > MAX_MOD) )
  {
    if(Enhanced_Enable == 0)   return 1;
    if(((f_modm & 0x1C)!=0x1C) && (f_modm > MAX_MOD)) return 1;
    if(((f_modn & 0x1C)!=0x1C) && (f_modn > MAX_MOD)) return 1;
  }
  //------------------------------------------------
  if(Enhanced_Enable)
  {
    if ( f_modm < 8 ) sprintf( srcA, Mod[f_modm], f_ARm, 1);
    else
    {
      if((f_modm & 0x1C) != 0x1C)
        sprintf( srcA, Mod[f_modm], f_ARm);
      else // Enhanced mode...
      {
        int reg = (int)((instr>>8) & 0x1F);
        if(reg>MAXREG-1) return 1;         // All regs are legal
        strcpy( srcA, C30Reg[reg].s);
      }
    }
    //----------------------------------------------------------
    if ( f_modn < 8 ) sprintf( srcB, Mod[f_modn], f_ARn, 1);
    else
    {
      if((f_modn & 0x1C) != 0x1C)
        sprintf( srcB, Mod[f_modn], f_ARn);
      else // Enhanced mode
      {
        int reg = (int)(instr & 0x1F);
        if(reg>MAXREG-1) return 1;  // All regs are legal
        strcpy( srcB, C30Reg[reg].s);
      }
    }
  }
  else  //---- Old version of DASM ----------------
  {
    if (f_modn>7) sprintf(srcB,Mod[f_modn],f_ARn  );
    else           sprintf(srcB,Mod[f_modn],f_ARn,1);
    if (f_modm>7) sprintf(srcA,Mod[f_modm],f_ARm  );
    else           sprintf(srcA,Mod[f_modm],f_ARm,1);
  }
  //----------------------------------------------------------
  strcpy(srcD, C30Reg[f_src1].s);
  strcpy(srcC, C30Reg[f_src2].s);
  switch ( f_p )
  {
     case 0 : strcpy(tmp,srcB); strcpy(srcB,srcA); strcpy(srcA,tmp);
              break;
     case 1 : /*
              strcpy(tmp,srcB); strcpy(srcB,srcC); strcpy(srcC,tmp);
              strcpy(tmp,srcC); strcpy(srcC,srcD); strcpy(srcD,tmp);
              strcpy(tmp,srcB); strcpy(srcB,srcC); strcpy(srcC,tmp);
              */
              strcpy(tmp,srcB); strcpy(srcB,srcD); strcpy(srcD,tmp);
              break;
     case 2 : strcpy(tmp,srcD); strcpy(srcD,srcA); strcpy(srcA,tmp);
              strcpy(tmp,srcC); strcpy(srcC,srcB); strcpy(srcB,tmp);
              strcpy(tmp,srcA); strcpy(srcA,srcB); strcpy(srcB,tmp);
              break;
     case 3 : strcpy(tmp,srcC); strcpy(srcC,srcB); strcpy(srcB,tmp);
              strcpy(tmp,srcD); strcpy(srcD,srcB); strcpy(srcB,tmp);
              break;
    default : return 1;
  }
  Extract2(buffer, buffer, tmp, "||");
  Concat3( buffer, srcA, srcB, C30Reg[f_d1].s );
  Concat3( tmp, srcC, srcD, C30Reg[f_d2+2].s );
  strcat(buffer,tmp);
  return 0;
}

/* ----------------------------------------------------------------------
                 Decoding for Parallel Addressing Modes

   ABSF || STF,  ABSF || STI, ADDF3 || STF, AND3 || STI, ASH3 || STI,
   FIX || STI, FLOAT || STF, LDF || STF, LDI || STI, LSH3 || STI,
   MPYF3 || STF, MPYI3 || STI, NEGF || STF, NEGI || STI, NOT || STI,
   OR3 || STI, SUBF3 || STF, SUBI3 || STI, XOR3 || STI, ADDI3 || STI,

                    STF || STF, STI || STI,
                    LDF || LDF, LDI || LDI
   ---------------------------------------------------------------------- */
int DForm4( int n, ulong instr, char *buffer )
{
  char src1[80], src2[80], src3[80], dst1[80], dst2[80], tmp[80];
  uint f_ARn          ;    //  3 bits,  0- 2
  uint f_modn         ;    //  5 bits,  3- 7
  uint f_ARm          ;    //  3 bits,  8-10
  uint f_modm         ;    //  5 bits, 11-15
  uint f_C            ;    //  3 bits, 16-18
  uint f_B            ;    //  3 bits, 19-21
  uint f_A            ;    //  3 bits, 22-24
//uint f_code         ;    //  7 bits, 25-31  opcode
  f_ARn           = (int)((instr      ) & 0x00007);
  f_modn          = (int)((instr >>  3) & 0x0001F);
  f_ARm           = (int)((instr >>  8) & 0x00007);
  f_modm          = (int)((instr >> 11) & 0x0001F);
  f_C             = (int)((instr >> 16) & 0x00007);
  f_B             = (int)((instr >> 19) & 0x00007);
  f_A             = (int)((instr >> 22) & 0x00007);
//f_code          = (int)((instr >> 25) & 0x0007F);

//if ( f_code > 0x78 ) // ???
//  return 1;
//if ( (f_modn > MAX_MOD) || (f_modm > MAX_MOD) ) return 1;
//------------------------------------------------
//#if __XON1
  if((instr & 0xE0E0L)==0xE0E0) return 1;
//#endif
  if ( (f_modm > MAX_MOD) || (f_modn > MAX_MOD) )
  {
    if(Enhanced_Enable == 0)   return 1;
    if(((f_modm & 0x1C)!=0x1C) && (f_modm > MAX_MOD)) return 1;
    if(((f_modn & 0x1C)!=0x1C) && (f_modn > MAX_MOD)) return 1;
  }
  //------------------------------------------------
  if(Enhanced_Enable)
  {
//  #if __XON1
    if((instr & 0xE000L) == 0xE000L) return 1;
//  #endif
    if ( f_modm < 8 ) sprintf( dst2, Mod[f_modm], f_ARm, 1);
    else
    {
      if((f_modm & 0x1C) != 0x1C)
        sprintf( dst2, Mod[f_modm], f_ARm);
      else // Enhanced mode...
      {
        int reg = (int)((instr>>8) & 0x1F);
        if(reg>MAXREG-1) return 1;  // All regs are legal
        strcpy( dst2, C30Reg[reg].s);
      }
    }
    //----------------------------------------------------------
    if ( f_modn < 8 ) sprintf( dst1, Mod[f_modn], f_ARn, 1);
    else
    {
      if((f_modn & 0x1C) != 0x1C)
        sprintf( dst1, Mod[f_modn], f_ARn);
      else // Enhanced mode
      {
        int reg = (int)(instr & 0x1F);
        if(reg>MAXREG-1) return 1;  // All regs are legal
        strcpy( dst1, C30Reg[reg].s);
      }
    }
  }
  else  //---- Old version of DASM ----------------
  {
    if (f_modn>7) sprintf(dst1,Mod[f_modn],f_ARn  );
    else           sprintf(dst1,Mod[f_modn],f_ARn,1);
    if (f_modm>7) sprintf(dst2,Mod[f_modm],f_ARm  );
    else           sprintf(dst2,Mod[f_modm],f_ARm,1);
  }
  //------------------------------------------------

  strcpy(buffer,Instr[n].instr);
  *src1 = '#';
  switch ( Instr[n].en) // switch( f_code )
  {
    case __STF_STF:
    case __STI_STI: //if ( f_B ) return 1;
                    *src1=0;
                    strcpy(src2,C30Reg[f_A].s);
                    strcpy(src3,C30Reg[f_C].s);
                    break;
    case __LDF_LDF:
    case __LDI_LDI: //if ( f_C ) return 1;
                    *src1 = 0;
                    strcpy(src2,dst1);
                    strcpy(dst1,C30Reg[f_A].s);
                    strcpy(src3,dst2);
                    strcpy(dst2,C30Reg[f_B].s);
                    break;
    case __ABSF_STF:
    case __ABSI_STI:
    case __FIX_STI :
    case __FLOAT_STF:
    case __LDF_STF  :
    case __LDI_STI  :
    case __NEGF_STF :
    case __NEGI_STI :
    case __NOT_STI  :
//  case __TOIEEE_STF:
//  case __FRIEEE_STF:
                       if ( f_B ) return 1;
//  case 0x6F : /* MPYF3 || STF */
                       *src1 = 0;
    default :          strcpy(src2,dst1);
                       strcpy(dst1,C30Reg[f_A].s);
                       strcpy(src3,C30Reg[f_C].s);
                       if (*src1) strcpy(src1,C30Reg[f_B].s);
                       break;
  }
  Extract2(buffer, buffer, tmp, "||");
  if (*src1) Concat3(buffer, src1, src2, dst1);
  else Concat2(buffer, src2, dst1);
  Concat2(tmp, src3, dst2);
  strcat(buffer,tmp);
  return 0;
}

/* ----------------------------------------------------------------------
           Type 1 Integer Three-Operand Addressing Modes
              ADDC3, ADDI3, AND3, ANDN3, ASH3, CMPI3,
            LSH3, MPYI3, OR3, SUBB3, SUBI3, TSTB3, XOR3
   ---------------------------------------------------------------------- */
int DForm5( int n, ulong instr, char *buffer )
{
  char src1[20], src2[20], dst[20];
  uint f_src2;    // 8 bits,  0- 7  source 2
  uint f_src1;    // 8 bits,  8-15  source 1
  uint f_dst ;    // 5 bits, 16-20  destination
  uint f_t   ;    // 2 bits, 21-22  t switch
//uint f_code;    // 9 bits, 23-31  opcode
  f_src2 = (int)((instr      ) & 0x00FF);
  f_src1 = (int)((instr >>  8) & 0x00FF);
  f_dst  = (int)((instr >> 16) & 0x001F);
  f_t    = (int)((instr >> 21) & 0x0003);
//f_code = (int)((instr >> 23) & 0x01FF);
/*
  f_src2 = (int)((instr & 0x000000FFL)      );
  f_src1 = (int)((instr & 0x0000FF00L) >>  8);
  f_dst  = (int)((instr & 0x001F0000L) >> 16);
  f_t    = (int)((instr & 0x00600000L) >> 21);
  f_code = (int)((instr & 0xFF800000L) >> 23);
  */
  strcat(buffer,Instr[n].instr);

  if(f_dst >= MAXREG)
    return 1;

  strcpy(dst,C30Reg[f_dst].s);
  switch( f_t)
  {
    case 0:  // Reg , Reg
             if((f_src1 >= MAXREG) || (f_src2 >= MAXREG)) return 1;
             strcpy(src1,C30Reg[f_src1].s);
             strcpy(src2,C30Reg[f_src2].s);
             break;

    case 1:  // Ind, Reg
             if((f_src1 >> 3) < 8)
               sprintf(src1, Mod[f_src1 >> 3], f_src1 & 0x7, 1 );
             else
             { if((f_src1 >> 3) > MAX_MOD) return 1;
               sprintf(src1, Mod[f_src1 >> 3], f_src1 & 0x7); }
             if(f_src2 >= MAXREG) return 1;
             strcpy(src2,C30Reg[f_src2].s);
             break;

    case 2:  // Reg, Ind
             if(f_src1 >= MAXREG) return 1;
             strcpy(src1,C30Reg[f_src1].s);
             if((f_src2 >> 3) < 8)
               sprintf(src2, Mod[f_src2 >> 3], f_src2 & 0x7, 1 );
             else
             { if((f_src2 >> 3) > MAX_MOD) return 1;
               sprintf(src2, Mod[f_src2 >> 3], f_src2 & 0x7); }
             break;

    case 3:  // Ind, Ind
             if((f_src1 >> 3) > 25) return 1;
             if((f_src2 >> 3) > 25) return 1;
             if((f_src1 >> 3) < 8)
               sprintf(src1, Mod[f_src1 >> 3], f_src1 & 0x7, 1 );
             else
             { if((f_src1 >> 3) > MAX_MOD) return 1;
               sprintf(src1, Mod[f_src1 >> 3], f_src1 & 0x7); }
             if((f_src2 >> 3) < 8)
                sprintf(src2, Mod[f_src2 >> 3], f_src2 & 0x7, 1 );
             else
             { if((f_src2 >> 3) > MAX_MOD) return 1;
               sprintf(src2, Mod[f_src2 >> 3], f_src2 & 0x7); }
             break;
    default: break;
  }
  if((Instr[n].en == __CMPI3)||(Instr[n].en == __TSTB3)) *dst = 0;
//if((f_code == 0x47) || (f_code == 0x4F)) *dst = 0; // CMPI3 TSTB3
  Concat3(buffer, src2, src1, dst);
  return 0;
}
/* ----------------------------------------------------------------------
       Type 1 Float, Decoding Three-Operand Addressing Modes
                    ADDF3, CMPF3, MPYF3, SUBF3
   ---------------------------------------------------------------------- */
int DForm5b( int n, ulong instr, char *buffer )
{
  char src1[20], src2[20], dst[20];

  uint f_src2;    // 8 bits,  0- 7  source 2
  uint f_src1;    // 8 bits,  8-15  source 1
  uint f_dst ;    // 5 bits, 16-20  destination
  uint f_t   ;    // 2 bits, 21-22  t switch
//uint f_code;    // 9 bits, 23-31  opcode

  f_src2 = (int)((instr      ) & 0x00FF);
  f_src1 = (int)((instr >>  8) & 0x00FF);
  f_dst  = (int)((instr >> 16) & 0x001F);
  f_t    = (int)((instr >> 21) & 0x0003);
//f_code = (int)((instr >> 23) & 0x01FF);

  strcat(buffer,Instr[n].instr);
  strcpy(dst,C30Reg[f_dst].s);

//if((f_dst > 0x7) && (f_dst < 0x1C)) return 1; // C40 float regs
  if(f_dst > 0x7)                      return 1; // C30

  switch( f_t)
  {
    case 0:  // Reg , Reg
             if((f_src1 >= MAXREG) || (f_src2 >= MAXREG)) return 1;
             if((f_src1 > 0x7) && (f_src1 < 0x1C)) return 1; // R0-R11 regs only
             if((f_src2 > 0x7) && (f_src2 < 0x1C)) return 1; // R0-R11 regs only
             strcpy(src1,C30Reg[f_src1].s);
             strcpy(src2,C30Reg[f_src2].s);
             break;
    case 1:  // Ind, Reg
             if((f_src1 >> 3) < 8)
               sprintf(src1, Mod[f_src1 >> 3], f_src1 & 0x7, 1 );
             else
             { if((f_src1 >> 3) > MAX_MOD) return 1;
               sprintf(src1, Mod[f_src1 >> 3], f_src1 & 0x7); }
             if(f_src2 >= MAXREG) return 1;
             if((f_src2 > 0x7) && (f_src2 < 0x1C)) return 1; // R0-R11 regs only
             strcpy(src2,C30Reg[f_src2].s);
             break;

    case 2:  // Reg, Ind
             if(f_src1 >= MAXREG) return 1;
             if((f_src1 > 0x7) && (f_src1 < 0x1C)) return 1; // R0-R11 regs only
             strcpy(src1,C30Reg[f_src1].s);
             if((f_src2 >> 3) < 8)
               sprintf(src2, Mod[f_src2 >> 3], f_src2 & 0x7, 1 );
             else
             { if((f_src2 >> 3) > MAX_MOD) return 1;
               sprintf(src2, Mod[f_src2 >> 3], f_src2 & 0x7); }
             break;

    case 3:  // Ind, Ind
             if((f_src1 >> 3) < 8)
               sprintf(src1, Mod[f_src1 >> 3], f_src1 & 0x7, 1 );
             else
             { if((f_src1 >> 3) > MAX_MOD) return 1;
               sprintf(src1, Mod[f_src1 >> 3], f_src1 & 0x7); }

             if((f_src2 >> 3) < 8)
               sprintf(src2, Mod[f_src2 >> 3], f_src2 & 0x7, 1 );
             else
             { if((f_src2 >> 3) > MAX_MOD) return 1;
               sprintf(src2, Mod[f_src2 >> 3], f_src2 & 0x7); }
             break;
    default: break;
  }
  if(Instr[n].en == __CMPF3) *dst = 0;  // CMPF3
//if(f_code == 0x46) *dst = 0;  // CMPF3
  Concat3(buffer, src2, src1, dst);
  return 0;
}

/* ----------------------------------------------------------------------
               Conditional-Branch Addressing Modes
         DB<cond>,  DB<cond>D, B<cond>, B<cond>D, B_AT, CALL<cond>
   ---------------------------------------------------------------------- */
int DForm7( int n, ulong instr, char *buffer )
{
  char src[80], dst[80];
  long d_addr, r_addr;

  uint f_src   ;    // 16 bits,  0-15  source 2
  uint f_cond  ;    //  5 bits, 16-20  source 1
  uint f_delay ;    //  1 bits, 21-21  destination
  uint f_ARn   ;    //  3 bits, 22-24  t switch
  uint f_branch;    //  1 bits, 25-25  t switch
//uint f_code  ;    //  6 bits, 26-31  opcode

  f_src   = (int)((instr      ) & 0x0FFFF);
  f_cond  = (int)((instr >> 16) & 0x0001F);
  f_delay = (int)((instr >> 21) & 0x00001);
  f_ARn   = (int)((instr >> 22) & 0x00007);
  f_branch= (int)((instr >> 25) & 0x00001);
//f_code  = (int)((instr >> 26) & 0x0003F);

  strcat(buffer,Instr[n].instr);
  *src=0;
  *dst=0;
// __DBc, __DBDc,
// __Bc , __BDc , __LDIc, __LDFc,

  if( (f_cond > MAX_COND) || (f_cond == 0xB)) return 1;
  switch(Instr[n].en)
  {
    case __Bc   :
    case __BDc  :
    case __DBc  :
    case __DBDc :
    case __RETIc:
    case __CALLc:
    case __RETSc: break;
    default :     strcat(buffer,Cond_tab[f_cond]); break;
  }
  /*
  if( f_delay )
  {
    strcat(buffer,"D");
//  PC_Address += 2L;
  }
  */
  if(( f_branch ) != 0)
  {
    /* Add symbol search here */
     if(f_delay) r_addr = 3;
     else r_addr = 1;
  // r_addr = 1;
     r_addr =  r_addr + (long)(int)f_src;
     d_addr = PC_Address + r_addr;

     get_label(dst,d_addr, 1); // Add 1 leading WS to symbol before print
     if(*dst==0) sprintf(dst," $%+ld",r_addr); // numeric value if !found
  }
  if(( f_branch ) == 0)
  {
    switch(Instr[n].en)
    {
      case __RETSc:
      case __RETIc: break;
      default:      if( f_src >= MAXREG) return 1;
                    strcpy(dst,"  ");
                    strcat(dst,C30Reg[f_src].s);
                    break;
    }
  }
  switch ( Instr[n].en )
  {
    case __RETIc :
    case __BDc   :
    case __RETSc : break;
    case __DBDc  :
    case __DBc   : strcpy(src,C30Reg[f_ARn+8].s); break; // DB<cond>
    // For CALLc and _Bc if the ARx fields are set, DASM output should
    // be a .word.  __CALLc should also not have the delay bit set
    case __CALLc : if( f_delay)return 1;                 // CALL<cond>
    case __Bc    : if( f_ARn ) return 1;          break; // B<cond>
    default      : return 1;
  }
  if(*src ) Concat2(buffer,src,dst);
  else strcat(buffer,dst);
  return 0;
}

/* ----------------------------------------------------------------------
        Particular Conditional-Branch Addressing Mode ( form 2 )
                           TRAP<cond> <n>
   ---------------------------------------------------------------------- */
int DForm9( int n, ulong instr, char *buffer )
{
  uint f_n     ;    //  5 bits,  0- 4  source 2
//uint f_z     ;    // 11 bits,  5-15  source 1
  uint f_cond  ;    //  5 bits, 16-20  destination
//uint
//uint f_code  ;    // 11 bits, 21-31  opcode
  f_n     = (int)((instr      ) & 0x0001F);
//f_z     = (int)((instr >>  5) & 0x007FF);
  f_cond  = (int)((instr >> 16) & 0x0001F);
//f_code  = (int)((instr >> 21) & 0x007FF);

  char tmp[80];
  strcat(buffer,Instr[n].instr);
  if(f_cond > MAX_COND) return 1;

  switch(Instr[n].en)
  {
    case __TRAP4x: if(f_cond == 0)  // If TRAP - Add U
                   sprintf(tmp, "%s %d",Cond_tab[f_cond],(int)instr&0x1FF);
                   else
                   sprintf(tmp, " %d",(int)instr&0x1FF);
                   break;
    case __TRAP  : if(f_cond == 0)  // If TRAP - Add U
                   sprintf(tmp, "%s %d",Cond_tab[f_cond],(int)f_n);
                   else
                   sprintf(tmp, " %d",(int)f_n);
                   break;
  }
  strcat(buffer,tmp);
  return 0;
  /*
  typedef struct
  {
    uint n    :  9; // trap #
    uint z    :  7; // opcode field all 0's
    uint cond :  5;
    uint code : 11; // TRAPcond
  } FORM9;
  FORM9 *f;
  char tmp[15];
  f = (FORM9 *) &instr;
  strcat(buffer,Instr[n].instr);
  if(f->cond > MAX_COND) return 1;
  sprintf(tmp, "%s %d",Cond_tab[f->cond],f->n);
  strcat(buffer,tmp);
  return 0;
  */
}

/* ----------------------------------------------------------------------
                        Particular Instructions
                           SWI, IDLE, SIGI
   ---------------------------------------------------------------------- */
int DFormA( int n, ulong instr, char *buffer )
{
  if(instr==0) return 0;  // Dummy return to prevent C warnings
  strcat(buffer,Instr[n].instr);
  return 0;
}
/* ----------------------------------------------------------------------
              Decoding Long Immediate Addressing Modes
                  RPTB,RPTBD, BR, BRD, CALL, LAJ
   ---------------------------------------------------------------------- */
int DFormB( int n, ulong instr, char *buffer )
{
  /*
  typedef struct
  {
    uint srcl: 16;
    uint srch:  8;
    uint code:  8;  // always 0x64
  } FORMB;
  FORMB *f;
  */
  long d_addr; //, r_addr;
  char tmp[80];
//f = (FORMB *) &instr;
  strcat( buffer, Instr[n].instr);
//  r_addr = 1;

  // C4x operands use PC relative call and jumps
  // for the C3x, each address of this form is absolute
  /*
  switch( f->code)
  {
    case 0x63:            // LAJ   24 bit +4
    case 0x61:            // BRD   24 bit +3
    case 0x65:            // RPTBD 24 bit +3
    case 0x60:            // BR    24 bit
    case 0x62:            // CALL  24 bit
    case 0x64:            // RPTB  24 bit

    if(f->code == 0x63)                        r_addr = 3;
    if((f->code == 0x61) || (f->code == 0x65)) r_addr = 3;

                r_addr = r_addr + (((long)instr << 8) >> 8) ;
                d_addr = PC_Address + r_addr;
                for(int j=0; j<MAX_SYMBOLS; j++)
                {
                  if(*DASM_Symbols[j].name == 0)
                  {
                 // sprintf(tmp," 0%06lxH", d_addr);
                    sprintf(tmp," $%+ld",   r_addr);
#ifdef DASMTEST
                    if(r_addr < 0) return 1;         // TEST ONLY
#endif
                    break;
                  }
                  if(DASM_Symbols[j].address == d_addr)
                  {
                    sprintf(tmp," %.12s",DASM_Symbols[j].name);
                    break;
                  }
                }
                break;
    case 0x79: // RPTB  register mode - Both normal and delayed
                sprintf(tmp," %s",C30Reg[f->srcl]);
                break;
    default  : return 1;
  }
  */
  // C3x version of these codes - all are 24 bit absolute addressed

//  if(C3Xmode)
  {
    d_addr = instr & 0xFFFFFFL;
    get_label(tmp,d_addr, 1); // Add 1 leading WS to symbol before print
    if(*tmp==0) sprintf(tmp," 0%lxh",d_addr); // numeric value if !found
  }
  strcat(buffer,tmp);
  return 0;
}
/*------------------------------------------------------------------
  Function which returns the string to be displayed in <buffer>
  --------------------------------------------------------------- */
/*  Set aside a N bit hash table for fast lookup of codes
    The upper HASH_TABLE_SIZE bits (in this case 9) are used after
    shifting FLSH bits to the right.

    If N=9 (512 entries), the shift value is 32-9=23

*/
#define HASH_TABLE_SIZE           512 /* 2^9 table size             */
#define HASH_TABLE_SHIFT           23 /* 32-9                       */
#define HASH_TABLE_MASK   0xFF800000L /* Mask upper 9 bits of instr */
#define HASH_TABLE_MASK2  0x007FFFFFL /* Mask upper 9 bits of instr */
#define FULL_TABLE_LOOKUP          -1 /* Flag for no match found    */
static int FastLook[HASH_TABLE_SIZE];

long Hash_hit=0;
long Hash_miss=0;
int Lookup(ulong c)
{
  int i = FastLook[(int)(c >> HASH_TABLE_SHIFT)];
  if(i!=FULL_TABLE_LOOKUP) {Hash_hit++; return i;}
  Hash_miss++;
  //-------------------------------------------
//if((c & Instr[i].mask) != Instr[i].pattern)
//{
    for(i=0;;i++)
    {
      if(Instr[i].pattern == (c & Instr[i].mask)) break;
  /*  if(Instr[i].en == __word) break; */ /* Never executes */
    }
//}
  return i;
}

int isC4code(int x)
{
  switch(Instr[x].en)
  {
    case __TRAP4x:           /* C4x specific codes   */
    case __STIK  :
    case __LDA   :
    case __FRIEEE:
    case __TOIEEE: return 1;
    default      : return 0; /* Bypass for C3x codes */
  }

}

/*---------------------------------------------------*/
/* Disassembles instr and address into a string      */
/*---------------------------------------------------*/
int DLLEXTEND_EX Disasm(ulong address, ulong instr, char *buffer)
{
  int x;
  *buffer=0;
  x = Lookup(instr);
  /*---------------------------------------------------*/
  /* Include this next section (old method) to detect  */
  /* any differences in the returned opcodes           */
  /*---------------------------------------------------*/
  if(Instr[x].mask == 0x0L)
  { sprintf( buffer, ".word 0%08lxh", instr );
    return(1);
  }
  /*------------------------------------------------------*/
  /* In C3xmode return C4x opcodes as .word values        */
  /* This section needs to hash tabled or to use a global */
  /* global enable flag for speed during simulation       */
#ifndef UTSIM3X
  if(C3Xmode)
  {
    if(isC4code(x))
    { sprintf( buffer, ".word 0%08lxh", instr );
      return 1;
    }
  }
#endif
  /*------------------------------------------------*/
  PC_Address = address;
  if ( Instr[x].dform( x, instr, buffer) )
  {
    sprintf( buffer, ".word 0%08lxh", instr );
    return(1);
  }
  return 0;
}
/*----------------------------------------------------------
   Calling this function builds the fast lookup table

   - If the upper 8 bits of the code are unique, the offset into the
     table is returned for that code.  In this case the number of found
     matches is 2.  Once for the found code, and once for the .word
     that terminates the end of the list

   - If the upper 8 bits of the code match more than one opcode, the
     entire list must be searched.  In this case return FULL_TABLE_LOOKUP.
     In this case the number of found matches is >2

   - If the upper 8 bits of the code match no known codes, then it is
     a .word and only 1 match will be found (IE the .word itself)
  ---------------------------------------------------*/
void DLLEXTEND_EX BuildFastLook(void)
{
  int i;
  static int FL_Built = 0;
  if(FL_Built) return;
  FL_Built = 1;
//int h0=0, h1=0, h2=0, h3=0;
  for(i=0;i<HASH_TABLE_SIZE;i++)
  {
    int x;
    int x2=FULL_TABLE_LOOKUP;
    int found = 0;
    for (x=0;;x++)
    {
      long l = (long)i << HASH_TABLE_SHIFT;
      if((l & Instr[x].mask) == (Instr[x].pattern & HASH_TABLE_MASK))
      {
         // If there are bits in the lower field that define the
         // opcode, then a match has NOT been found
         if((Instr[x].mask & HASH_TABLE_MASK2)==0) // Check LSB's
         found++;         // found match
         if(Instr[x].en == __word) break;
         x2 = x;          // remember last match that was not a .word
      }
    }
    // found=0   Should never occur since .word is a match
    // found=1   if only .word found as match
    // found=2   if 1 match found + .word
    // found=3++ if more than 1 match + .word
    //
    // The only case that is significant is if found==2
    // In all other cases, FULL_TABLE_LOOKUP is returned
    if(found==2) FastLook[i] = x2;
    else         FastLook[i] = FULL_TABLE_LOOKUP;
    /*
    switch(found)
    {
      case  0: h0++; break;
      case  1: h1++; break;
      case  2: h2++; break;
      default: h3++; break;
    }
    */
  }
//printf("\r\n%d %d %d %d\r\n",h0,h1,h2,h3);
}

