//------------------------------------------------------------
// TEXTWIN.CPP
// Keith Larson
// TMS320 DSP Applications
// (c) Copyright 1995, 1996
// Texas Instruments Incorporated
//
// This is unsupported freeware code with no implied warranties or
// liabilities.  See the disclaimer document for details
//------------------------------------------------------------
#include <process.h>
#include <bios.h>      /* Compiler header files     */
#include <alloc.h>
#include <string.h>
#include <conio.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <dos.h>
//---------------------------------------------------------------
#include "typedefs.h"
#include "textwin.h"   /* DSK supplied header files */
#include "errormsg.h"
#include "keydef.h"
#include "dsk.h"
//---------------------------------------------------------------
SCREEN   *PtrScreen;
#define  Screen (*PtrScreen)
int    screenmode =   C80;  // default debugger screen text mode
int oldscreenmode =   C80;  // startup text mode
/*---------------------------------------------------------------
   The included file is a C formatted help screen message.  To save
   time, a conversion utility TXT2SRC.EXE was written to convert a
   text file to a formatted C string.  This makes editing much more
   convenient. The Borland program transfer utility is also used,
   creating auto-dependancies for easy project management
 -----------------------------------------------------------------*/
char huge  Hard_Hlp[]=    // These help messages are common to all apps
  #include "HARDWARE.CPP"
char Exit_Hlp[] =
"\n"
"   +---------------------------------------+\r\n"
"   | TMS320C3x DSK                         |\r\n"
"   | Texas Instruments Incorporated        |\r\n"
"   |                                       |\r\n"
"   | The Leading Provider of DSP Solutions |\r\n"
"   +---------------------------------------+\r\n\r\n";
/*-----------------------------------------------*/
/* Standard switches copied to command line help */
/*-----------------------------------------------*/
char huge St_Sw[]=
"  ? or HELP       This help screen\r\n"
"  Reset, R        Reset (cold boot) DSK\r\n"
"  LPTx, LPT=x     Use LPTx (LPT1 is default)\r\n"
"  PORT=0x378      Use port at IO address 0x???\r\n"
"  BW=8, Byte      8 bit bi-directional mode\r\n"
"  BW=4, Nibble    4 bit standard port\r\n"
"  TEST            Search LPT1,2,3\r\n"
"  AUTO            Detect 8/4 bit mode\r\n"
"  T=??            Add extra I/O bus cycles for noisy cables\r\n"
"  WIN=1,0         Use Windows TimeSlice management\r\n"
"  MTASK=0,NOMTASK Do not protect comm link against multitask\r\n"
"  C32             Select C32 DSK target mode (C31 is default)\r\n"
"  WS=n            Waitstates in bus control register\r\n"
"  IOSTRB=0xnnn    C32 IOSTRB boot value\r\n"
"  STRB0=0xnnn     C32 STRB0 boot value\r\n"
"  STRB1=0xnnn     C32 STRB1 boot value\r\n"
"  PBUSCON=0xnnn   C31 p_buscon, or C32 STRB0\r\n"
"  P_BUSCON=0xnnn  C31 p_buscon, or C32 STRB0\r\n"
"  K=C3X.DSK       Use a different kernel\r\n"
"\r\n"
"  Note: No spaces are allowed before or after '='\r\n";

/*-------------------------------------------------------------------
  The functions that follow can be used in DOS text based programs
  They include the creation and display of help screens.
  -----------------------------------------------------------------
   Edit_Manual(,,,) is used to fill in the WS area of the standard
   help menu with the information supplied by the application
  ------------------------------------------------------------------*/
void Edit_Help_Msg(char *app_name, char *sw1, char *desc)
{
  char huge *p1;
  static char used=0;  // Use a static variable to enable this
  if(used) return;     // function one and only one time.
  used = 1;
  p1 = Hard_Hlp;
  if((*sw1==0)&&(*desc==0))
  {
    p1 += sprintf(Hard_Hlp,
     "Use: %s [switches]...     DSK3 Ver:%2.2f\r\n\r\n%s",
                                app_name,DSK3_rev,St_Sw);
  }
  else
  {
    p1+=sprintf(Hard_Hlp,
    "Use: %s %s[switches]    DSK3 Ver:%2.2f\r\n\r\n%s%s",
                               app_name, sw1,DSK3_rev,desc,St_Sw);
  }
  *p1 = ' ';                    // remove sprintf terminator
}
/*--------------------------------------------------------*/
/* View_Manual() opens a text help window for display     */
/*--------------------------------------------------------*/
MSGS View_Manual(char *app_name)
{
  MSGS err;
  struct text_info ti;
  gettextinfo(&ti);                                     // What mode now
  err = Disp_Menu(app_name,Hard_Hlp ,1,80,1,25);        // Open & display
  window(ti.winleft,ti.wintop,ti.winright,ti.winbottom);// Full screen window
  clrscr();
  _setcursortype(_NORMALCURSOR);
  return err;
}
//--------------------------------------------------------------------
//putcharxy() is used to directly write a character to the displays
//text memory after adding in the currently defined text window size
//--------------------------------------------------------------------
void putcharxy(int x,int y,char car)
{
  char far *p;
  struct text_info ti;
  int i;
  gettextinfo(&ti);
  x+=ti.winleft;
  y+=ti.wintop;
  i = 2*(ti.screenwidth*(y-2)+(x-2));
  p = (char far *) MK_FP(0xB800,i);
  *p = car;
}
/*----------------------------------------------------------------------
   Disp_Menu opens a text window, creates a frame, including a banner
   message and help line, and then displays the text string pointed to
   by *msg.  The string can be scrolled up and down as well as saved
   to a file with the name 'banner'.
----------------------------------------------------------------------*/
MSGS Disp_Menu(char *banner,char huge *msg,
               int Xstart,int Xwidth, int Ystart, int Ywidth)
{
  struct text_info ti;
  FILE *out;
  int line;
  int x;
  int bk = 0;
  char buf[16];
  char huge *p1;
  char huge *p2;
  char old;
  int last;
  int my;
//  static int HW_nest = 0;

  gettextinfo(&ti);
  my = Ywidth -2;
  line = 0;
  window(Xstart,Ystart,Xstart+Xwidth-1,Ystart+Ywidth-1);
  clrscr();
  /* draw the frame */
  gotoxy(1,     1); for(x=1;x<Xwidth;x++) putch(FRAME2[2]); // Top Margin
  gotoxy(1,Ywidth); for(x=1;x<Xwidth;x++) putch(FRAME2[2]); // Bot Margin
  for(x=1;x<Ywidth;x++)
  { gotoxy(1,x); putch(FRAME2[5]);} // Left Margin
  for(x=1;x<Ywidth;x++)
  { gotoxy(Xwidth,x); putch(FRAME2[5]);} // Right Margin

  window(Xstart,Ystart,Xstart+Xwidth,Ystart+Ywidth-1);

  putcharxy(1     ,     1,FRAME2[0]);  // Up_Lt
  putcharxy(Xwidth,     1,FRAME2[1]);  // Up_Rt
  putcharxy(1     ,Ywidth,FRAME2[3]);  // Lo_Lt
  putcharxy(Xwidth,Ywidth,FRAME2[4]);  // Lo_Rt

  /* Write banner and Footer messages */
  gotoxy((Xwidth - strlen(banner))/2,1);
  cputs(banner);
  gotoxy(4,Ywidth);
  cputs(" Move Up/Dn/Pup/Pdn  H-Xtra help  S-save help to file ");
  window(Xstart+2,Ystart+1,Xstart+Xwidth-2,Ystart+Ywidth-2);
  _setcursortype(_NOCURSOR);
  for(;;)
  {
    /* Align HELP Message start counting line '\n' chars */
    p1 = msg;
    for(x=0;x<line;x++)  // Window
    {
      while((*p1!='\n')&&(*p1!=0)) p1++;
      if(*p1==0) break;
      p1++;
    }
    /* Find end at position last, or EOL, whichever is first */
    p2 = p1;
    for(last=line;last<line+my;last++)
    {
      while((*p2!='\r')&&(*p2!=0)) p2++;
      if(*p2==0) break;
      p2++;
    }
    old = *p2;             // Keep last terminating character
    *p2 = 0;               // Force string termination here
    clrscr();              // Clear text
 // cprintf(p1);           // Fast
    cputs((char far *)p1); // Even faster
    *p2 = old;        // put back the 'old' exit character
    // Get a key from the user
    while(!kbhit());
    bk = bioskey(1) & 0xFF00;
    switch(bk)
    { case _Up : line -=  1; break;
      case _Dn : line +=  1; break;
      case _Pup: line -= my; break;
      case _Pdn: line += my; break;
      case _S  :
                 p1 = banner;
                 while(*p1 == ' ') p1++;
                 strncpy(buf,(char far *)p1,12);
                 p1 = buf;
                 while(*p1 != ' ') p1++;
                 *p1 = 0;
                 strcat(buf,".HLP");

                 if((out=fopen(buf,"wb"))== NULL)
                 { window(ti.winleft,ti.wintop,ti.winright,ti.winbottom);
                   return OPEN_ERR;
                 }
                 fprintf(out,"%s",msg);
                 fcloseall();
                 break;
      case _H  : bioskey(0);
                 return EXT_HELP; // The calling func displays hardware help
      default  : bioskey(0);
      case _Q  : //ungetch(' ');
                 window(ti.winleft,ti.wintop,ti.winright,ti.winbottom);
                 _setcursortype(_NORMALCURSOR);
                 return NO_ERR;
    }
    bioskey(0);
    if(line <    0) line =    0;
    if(line > last) line = last;
  }
}

//---------------------------------------------------------------
// Help_Menu() opens a window within the debugger shell and
// then displays the message string allowing the user to move
// up and down or copy the entire message to a file.  The
// 'banner' string is displayed at the top of the open window.
//---------------------------------------------------------------
#define hlp_attr ((LIGHTGRAY<<4) | WHITE)
MSGS Help_Menu(char *banner, char huge *msg,
               int Xstart,int Xwidth, int Ystart, int Ywidth)
{
  struct text_info ti;
  MSGS err;
  WINDOW *tmp;
  gettextinfo(&ti);
  /* DSKCreateWindow allocates memory to preserve the overwritten text
     in a malloc array.  This array is accessed by the *tmp pointer
     to a WINDOW structure which contains window information        */
  tmp = DSKCreateWindow(
        Xstart,Ystart,Xwidth,Ywidth,banner,FRAME2,0x70,hlp_attr);
  OpenWindow(tmp);
  /* Define a text region for the window */
  window(Xstart+2,Ystart+1,Xstart+Xwidth-2,Ystart+Ywidth-2);
  /*------------------------------------------------*/
  err = Disp_Menu(banner,msg,Xstart,Xwidth,Ystart,Ywidth);
  CloseWindow(tmp);             // Close the top window, restore previous
  EraseWindow(tmp);             // Erase existance of top window
  window(ti.winleft,ti.wintop,ti.winright,ti.winbottom);
  return err;
}
/*-------------------------------------------------------------
    Opens the hardware level help display, usually following a
    window specific help using the _F1 key
  -----------------------------------------------------------*/
MSGS Ext_Help(MSGS err)
{
  if(err == EXT_HELP)
  {
    if(screenmode==C4350)
    err = Help_Menu("DSK3D",Hard_Hlp,1,80,1,39);
    else
    err = Help_Menu("DSK3D",Hard_Hlp,1,80,1,25);
    err = NO_ERR;
  }
  return err;
}

/*----------------------------------------------
   Initialize video to text mode EGA/VGA
  --------------------------------------------*/
void InitVideo(void)
{
  textmode(screenmode);
  PtrScreen = (SCREEN *) MK_FP(0xB800,0x0000); // Pointer to text memory
  /*Screen = (SCREEN *) MK_FP(0xB800,0x0000);   Pointer to text memory*/
}
/*-------------------------------------------------------
   put a n characters starting at the location x,y
   using various attributes for color and highlighting
  -----------------------------------------------------*/
void PutnChar(int x, int y, int n, char car, char Attr,int mode)
{
 register int i;
 if ( car == '#' ) car = 32;
 for ( i=x; i<x+n; i++ )
  {
   switch ( mode )
    {
     case CAR     : Screen[y][i].attr &= 0xF0;
		    Screen[y][i].attr |= Attr & 0x0F; break;
     case FOND    : Screen[y][i].attr &= 0x0F;
		    Screen[y][i].attr |= Attr & 0xF0; break;
     case OLDCAR  :
     case CARFOND : Screen[y][i].attr = Attr; break;
     case OLD     :
     default      : break;
    }
   if ( mode != OLDCAR )  Screen[y][i].car = car;
  }
}
/*-------------------------------------------------------
   put a string of up to n characters at the location x,y
   using various attributes for color and highlighting
  -----------------------------------------------------*/
void PutString(int x, int y, int n, char *string, char Attr, int mode)
{
  int i;
  char car;
  if((string==NULL)||(*string==0)) return;
  i = strlen(string);
  if ( n > i ) n = i;
  for(;n>0;n--)
  {
    car = *string++;
    if ( car == '#' ) car = 32;
    switch ( mode )
    {
      case CAR     : Screen[y][x].attr &= 0xF0;
      	             Screen[y][x].attr |= Attr & 0x0F; break;
      case FOND    : Screen[y][x].attr &= 0x0F;
    		     Screen[y][x].attr |= Attr & 0xF0; break;
      case OLDCAR  :
      case CARFOND : Screen[y][x].attr = Attr; break;
      case OLD     :
      default      : break;
    }
    if ( mode != OLDCAR )  Screen[y][x].car = car;
    x++;
  }
}

//---------------------------------------------------------------
// Clear the window associated with the WINDOW structure pointer
//---------------------------------------------------------------
void ClrWin(WINDOW *win)
{
  register j;
  int x,y,w,h;
  char Attr;
  x=win->x  ;
  y=win->y  ;
  w=win->w-2;
  h=win->h-2;
  Attr=win->tattr;
  for ( j=0; j<h; j++)
     PutnChar(x,y+j,w,' ',Attr,CARFOND);
}
//---------------------------------------------------------------
// Write the window title text string to the position x,y
//---------------------------------------------------------------
void WinTitle( char *p1, int x, int y )
{
  int ta=0,tattr;
  char buf[8];
//textbackground(BLACK);
  x++;
  y++;
  gotoxy(x,y);
  gettext(x,y,x+1,y,buf);
  tattr = buf[1];
  tattr &= 0xff;
  //textbackground(tattr);
  textattr(tattr);
  for(;;)
  {
    switch(*p1)
    {
      case  0 :             break; // Nothing to printf
      case '#': ta^=1;
                if(ta) textcolor(   YELLOW);
                else   textcolor(LIGHTGRAY);
                break; // toggle text brightness
      default : putch(*p1); break;
    }
    if(*p1==0) break;
    p1++;
  }
}
//-------------------------------------------------------------
// Create the text line draw character frame around the
// window defined by the WINDOW structure
//-------------------------------------------------------------
void WinFrame(WINDOW *win);
void WinFrame(WINDOW *win)
{
  register int i;
  int wx, wy, wh, ww;
  wx = win->x; wy = win->y;
  wh = win->h; ww = win->w;

  PutnChar(wx-1     ,wy-1,   1,*(win->frame  ),win->fattr,CARFOND);
  PutnChar(wx-1+1   ,wy-1,ww-2,*(win->frame+2),win->fattr,CARFOND);
  PutnChar(wx-1+ww-1,wy-1,   1,*(win->frame+1),win->fattr,CARFOND);
  for(i=0;i<wh-2;i++)
  {
  PutnChar(wx-1     ,wy-1+1+i,1,*(win->frame+5),win->fattr,CARFOND);
  PutnChar(wx-1+ww-1,wy-1+1+i,1,*(win->frame+5),win->fattr,CARFOND);
  }
  PutnChar(wx-1     ,wy-1+wh-1,   1,*(win->frame+3),win->fattr,CARFOND);
  PutnChar(wx-1+1   ,wy-1+wh-1,ww-2,*(win->frame+2),win->fattr,CARFOND);
  PutnChar(wx-1+ww-1,wy-1+wh-1,   1,*(win->frame+4),win->fattr,CARFOND);

  i = (ww-strlen(win->title))/2;
  if( i<0 )
  PutString(wx-1  ,wy-1,              wh-2,win->title,win->fattr,CARFOND);
  else
//PutString(wx-1+i,wy-1,strlen(win->title),win->title,win->fattr,CARFOND);
  WinTitle(win->title, wx-1+i, wy-1);
}

//----------------------------------------------------------
// Dealocate a window instance from memory
//----------------------------------------------------------

int EraseWindow( WINDOW *win )
{
  free(win->title);
  free(win->frame);
  free(win);
  return 0;
}

//----------------------------------------------------------
// Create a window of type WINDOW and return a pointer to
// the structure after filling in its attributes.
//----------------------------------------------------------
WINDOW *DSKCreateWindow( int x, int y, int w, int h, char *title, char *frame,
		  int tattr, int fattr)
{
 WINDOW *win;

 win = (WINDOW *) malloc(sizeof(WINDOW));   // 80*50=4000bytes max

 if(win == NULL)
 {
   clrscr();
   printf("CreateWindow malloc failed");
   exit(0);
 }

 if ( win != NULL )
  {
   win->x = x;
   win->y = y;
   win->h = h;
   win->w = w;
   win->tattr = tattr;
   win->fattr = fattr;
   win->title = (char *) malloc(strlen(title)+1);

   if(win->title == NULL)
   {
     clrscr();
     printf("CreateWindow win->title malloc failed");
     exit(0);
   }

   if ( win->title != NULL )
    {
     strcpy(win->title,title);
     win->frame = (char *) malloc(strlen(frame)+1);

     if(win->frame == NULL)
     {
       clrscr();
       printf("CreateWindow win->frame malloc failed");
       exit(0);
     }

     if ( win->frame != NULL )
      {
       strcpy(win->frame,frame);
       return win;
      }
     free(win->title);
    }
   free(win);
  }
 return NULL;
}

//------------------------------------------------------------
// Allocate the memory required to hold the contents of a
// window and then create the window, drawing in a blank box
//------------------------------------------------------------
void OpenWindow( WINDOW *win )
{
  win->buf = (char *) malloc( 2 * win->h * win->w );
  if(win->buf == NULL)
  {
    clrscr();
    printf("OpenWindow malloc failed");
    exit(0);
  }

  gettext(win->x,win->y,win->x+win->w-1,win->y+win->h-1,win->buf);
  WinFrame(win);
  ClrWin(win);
}
//-------------------------------------------------------
// Blank out the window text, then deallocate its buffer
//-------------------------------------------------------
void CloseWindow( WINDOW *win )
{
 puttext(win->x,win->y,win->x+win->w-1,win->y+win->h-1,win->buf);
 free(win->buf);
 win->buf = NULL;
}
//--------------------------------------------------------
// Change the border of a window to highlight double lines
//--------------------------------------------------------
void EnableWindow( WINDOW *win )
{
 strcpy(win->frame,FRAME2);
 win->fattr |= 0x8;
 WinFrame(win);
}
//--------------------------------------------------------
// Change the border of a window to regular single lines
//--------------------------------------------------------
void DisableWindow( WINDOW *win )
{
 strcpy(win->frame,FRAME1);
 win->fattr &= 0xF7;
 WinFrame(win);
}
//---------------------------------------------------------
// Deallocate the memory for a particular window and then
// reallocate memory for the new size
//---------------------------------------------------------
WINDOW *GrowWindow(int x, int y, int w, int h, WINDOW *win)
{
  WINDOW *tmp;
  CloseWindow(win);
  tmp = DSKCreateWindow
        ( x, y, w, h,win->title, win->frame,win->tattr, win->fattr);
  free(win->title);
  free(win->frame);
  free(win);
  OpenWindow(tmp);
  ClrWin(tmp);
  return tmp;
}

