//---------------------------------------------------------
// DRIVER.CPP
// Keith Larson
// TMS320 DSP Applications
// (c) Copyright 1995, 1996, 1997
// Texas Instruments Incorporated
//
// This is unsupported freeware code with no implied warranties or
// liabilities.  See the disclaimer document for details
//---------------------------------------------------------
// This file contains the code for controlling the DSK
// printer port interface.  At this level you will find
// the code which sends and receives individual words,
// resets the device and control over the port direction
//---------------------------------------------------------
#include "dsk.h"
#include "keydef.h"
#if __DSKWINAPP
#include <windows.h>
#else
#include <bios.h>
#endif
#include <io.h>
#include <dos.h>
#include <stdio.h>
#include <conio.h>  
#include <stdlib.h>
//
// Context save area CTXT[] is accessable to all levels of code hierarchy
//
//ulong * DLLEXTEND_EX CTXT;
ulong              CTXT[CTXTSIZE];
ulong              CTXT_PTR;
int                _DIR        =    0x00;
int                timeout     =       0;
int                bk          =       1;
uint               oldport     =       0;
long               WSHIFT      =      -8; // -4
ulong              WSCOUNT     =       3; //  7
#if __WIN32__
WORD              port        =   0x378;
WORD              status      =   0x379;
WORD              ctrl        =   0x37a;  //LPT1 default
WORD              EVMbase     =   0x240;  // EVM default
#else
uint              port        =   0x378;
uint              status      =   0x379;
uint              ctrl        =   0x37a;  //LPT1 default
uint              EVMbase     =   0x240;  // EVM default
#endif

char              LO_PWR      =       0;
long              MAX_WAIT    = 100000L;
int               FAST_FLAG   =       0;
int               cpu_running =       1;
//-------------------------------------------------------------
// Win32 applications cannot directly access data, arrays or
// strings which are contained in a DLL.  A function is used
// as an abstraction layer to do this
//-------------------------------------------------------------
long  DLLEXTEND_EX  Get_MAX_WAIT    (void)       {return MAX_WAIT;}
void  DLLEXTEND_EX  Set_MAX_WAIT    (long x)     {MAX_WAIT=x;}
int   DLLEXTEND_EX  Get_No_MTask    (void)       {return No_MTask;}
void  DLLEXTEND_EX  Set_No_MTask    (int x)      {No_MTask=x;} 
void  DLLEXTEND_EX  Set_PC_Appli    (ulong x)    { PC_Appli = x; }
ulong DLLEXTEND_EX  Get_PC_Appli    (void)       { return PC_Appli; }
void  DLLEXTEND_EX  set_cpu_running (int x)      { cpu_running = x; }
int   DLLEXTEND_EX  get_cpu_running (void)       { return cpu_running; }
void  DLLEXTEND_EX  Set_BW_force    (int x)      { BW_force = x; }
void  DLLEXTEND_EX  Set_Windows_Detected(int x)  { Windows_Detected = x;}
ulong DLLEXTEND_EX  SPDFLT          (void)       { return SP_DFLT; }
ulong DLLEXTEND_EX  CTXTPTR         (void)       { return CTXT_PTR; }
void  DLLEXTEND_EX  WriteCTXT(int offs,ulong val){CTXT[offs]=val;}
ulong DLLEXTEND_EX  ReadCTXT        (int offs)   { return CTXT[offs];}

MSGS DLLEXTEND_EX Load_CTXT(void)
{ return(getmem(CTXTPTR(),CTXTSIZE,&CTXT[0]));}

// -----------------------------------------------------------------
// C31 DSK interface and parallel printer port signal definitions
//
// Bit definitions at printer control port (write)
//
//    B7      B6     B5    B4     B3        B2         B1        B0
// +--------------------+--------------+----------+---------+---------+
// |  DIR |   x  |  DIR |  INT |/SLCTIN|   INIT   |/AUTOFEED| /STROBE |
// +--------------------+--------------+----------+---------+---------+
//                                      RESET                 HPSTB
//
// Bit definitions at printer status port (read)
//    B7         B6         B5          B4        B3     B2    B1   B0
// +----------+----------+-----------+----------+---------------------+
// | /BUSY    |  ACK     |  PAPER    |  SELECT  |ERROR| ACK |  x |  x |
// +----------+----------+-----------+----------+---------------------+
//  <------------------ D0 - D3 ---------------> HPACK
//
//
//************************************************************************
//          BOOTLOADER/HOST PORT INTERFACE RESET TIMING DIAGRAM
//
//
// Reset/INIT   -----+        +----------------------------+       +---
//                   +--------+    INT2                    +-------+
// HPSTB/STROBE -------------------+        +-----------------------
//                   :        :    +--------+
// HPACK/ERROR  ----------------------+     :  +---+        -----------
//              XXXXX/        :    :  +--------+   +-------/
//                   :        :    :  :     :  :   :       :
//                   :        :    :  :     :  :   :       :
//                   A        B    C  D     E  F   G       H
// A) RESET=0, HPSTB=1 clears any previous /HPACK
// B) RESET=1  Release causes bootloader to start
// C) First HPSTB starts bootloader by driving INT2 low
// D) HPACK goes low as DSP accesses byte zero
// E) HPSTB=1 rising causes READY to pulse
// F) HPACK goes high as DSP completes byte read
// G) Bootloader reads byte one and is again interlocked
// H) HPI is again reset.
// I) Return to main bootloader code, which sends valid data to DSP

MSGS SystemCmd(char *strg)
{
  #if    __DSKWINAPP
  if(*strg==0) return INIT_ERR;   // Dummy function
  MessageBox(NULL,
            "The C30 EVM is not supported for Windows\n"
            "Run EVMRESET and EVMLOAD C3XEVM.OUT seperately",
            "EVM INIT ERR",MB_OK);
  #else
  if(system(strg)!=0)
  {
    perror("EVM Init error: ");
    return INIT_ERR;
  }
  #endif
  return NO_ERR;
}

MSGS   DLLEXTEND_EX InitEVM(void)
{
  MSGS err;
  char strg[80];
  #if    __DSKWINAPP
  MessageBox(
	  NULL,
	  "EVM initialize files\nC3XEVM.DSK, EVMRESET.EXE, EVMLOAD.EXE, RESVCT.002\n",
	  "EVM Init Info",MB_OK);

  #else
  clrscr();
  printf("Required files for EVM to initialize\r\n"
         "C3XEVM.DSK, EVMRESET.EXE, EVMLOAD.EXE, RESVCT.002\r\n");
  #endif
  if(access("C3XEVM.DSK"  ,0)) return INIT_ERR;
  if(access("EVMRESET.EXE",0)) return INIT_ERR;
  if(access("EVMLOAD.EXE" ,0)) return INIT_ERR;
  sprintf(strg,"EVMRESET -p%d",(int)EVMbase);
    if((err=SystemCmd(strg))!=NO_ERR) return err;
  sprintf(strg,"EVMLOAD C3XEVM.OUT -p%d",EVMbase);
    if((err=SystemCmd(strg))!=NO_ERR) return err;
  return NO_ERR;
}

MSGS DLLEXTEND_EX DSK_reset(void)     // 25*80 = 4000 chars per screen
{
  char MSG2[160];
  if(TARGET==C30_EVM)
  {
    return InitEVM();
  }
  if(port!=oldport) bk = 1; // If the port number changes or a
  oldport=port;             // key is hit update the display.
  for(;;)
  {
    outctrl(RESET_LO);
    mydelay(20);  // 1/60 s catches 1/2 cycle of bad pwr
    if((instat & 0x8)==0)
    {
      sprintf(MSG2,
      ">>>> HPACK (ERROR pin) did not go high during reset\r\n");
    }
    else
    { outctrl(RESET_HI); mydelay(20); // 1/60 s catches 1/2 cycle of bad pwr
      if(instat & 0x8)
      {
        outctrl(PPSTRB_LO); mydelay(1);
        if((instat & 0x8) == 0)
        {
          sprintf(EXTMSG," RESET/HPSTB/HPACK TEST - PASS\r\n"
          " >>>> LOADING AND VERIFYING KERNEL: PLEASE WAIT <<<<\r\n");
       // Leave with DSK waiting to receive 1st byte, and since
       // send_byte() is re-entrant, a second reset is not required
          return EXT_MSG_OK;
        }
        else
        sprintf(MSG2,
">>> HPACK did not respond (low) to HPSTB low.  Check cable and power\r\n");
      }
      else // an error has occurred
      sprintf(MSG2,
">>> HPACK toggled low without HPSTB pulse.  Noisy cable or bad power\r\n");
    }
    if(bk)  // print if first pass or keystroke (new port select)
    {
      sprintf(EXTMSG,
"  TESTING TMS320C3x DSK RESET AT PORT 0x%03x (LPT%d)\r\n"
"  %s\r\n"
"SELECT 1) LPT1 0x378\r\n"
"       2) LPT2 0x278\r\n"
"       3) LPT3 0x3BC\r\n"
"       H) Help\r\n"
"\r\n"
"CHECK-TARGET POWER LED CYCLES R-Y-G IF THE KERNEL LOADS\r\n"
"      LPTx PORT SELECT\r\n"
"      I/O CONNECTIONS AND CABLES\r\n"
"      LAPTOP POWER DOWN SOFTWARE\r\n"
"      AUTOEXEC.BAT, CONFIG.SYS AND BIOS\r\n"
"      DAUGHTER CARDS\r\n"
"\r\n"
"The DSK software hard codes the names LPT1, LPT2 and LPT3 to IO\r\n"
"addresses 378h, 278h and 3BCh respectively.  The OS on the other hand\r\n"
"may map these names differently.  Try the other ports.\r\n"
      ,port,ports(port),MSG2);
      return EXT_MSG_ERR;
    }
  }
}

//-----------------------------------------------------------
// HPI_STRB() drives the strobe line high or low.  Use this
// function with HPI_ACK() to create a user defined interlock
//-----------------------------------------------------------
void DLLEXTEND_EX HPI_STRB(int val)
{
  if(TARGET==C30_EVM)
  {
              // Read a value
    return;
  }
  if(val==0) outctrl((char)(PPSTRB_LO|_DIR));
  else       outctrl((char)(PPSTRB_HI|_DIR));
}
#if __WIN32__ | _WIN32
#else
#define WORD  int   // undefine 'WORD' for DOS application build
#endif

char DLLEXTEND_EX HPI_ACK(void)
{
  #define CMD_OK 0
  int i;
  if(TARGET==C30_EVM)
  {
    outword((WORD)(EVMbase+ 0x14),0x6044);  // Update status
    i = inword((WORD)(EVMbase+0x400));      // Get status flags
    if((i & 0x0006)!=0)
    {
      outword((WORD)(EVMbase+0x400),0x0006);  // Clear status flags
      if(inportb((WORD)(EVMbase+0x808))==CMD_OK) return 1;
    }
    return 0;
  }
  if((instat & 0x8) == 0) return 1;
                          return 0;
}
//----------------------------------------------------------------
// Interlock(( can be used in front of most fuuctions when the
// application might not respond for some time.  Note the use of
// a TIMEOUT for windows applications since keystrokes are not
// available.
//----------------------------------------------------------------
MSGS DLLEXTEND_EX Interlock(void) //_inp
{
  int i;
  if(TARGET==C30_EVM)
  {
	return NO_ERR;
  }
  for(i=0;;i++)
  {
    outctrl(PPSTRB_LO);
    if((instat&0x8)==0) return NO_ERR; // exit on good handshake
    #if    __DSKWINAPP
    if(i>16000)
    {
      return TIMEOUT;
    }
    #else
    if(kbhit())  break;
    #endif
  }
  #if    __DSKWINAPP
  #else
  return KEY_HIT;
  #endif
}
//-----------------------------------------------------------
// recv_long() receives a 32 bit value from the DSK interface
// placing the value in the address *x
//-----------------------------------------------------------
//#define MAX_WAIT 100000L  //  3-10-97 for wav file buffer
MSGS DLLEXTEND_EX recv_long(ulong *x)  //  512/10Khz = 51mS buffer
{
  ulong ul=0, n;
  long t;
  char shift;
  t=MAX_WAIT;
  if(TARGET==C30_EVM)
  {
    *x = 0;
    for(int b=0;b<2;b++)
    {
      for(;t>=0;t--)
      {
        if(inword((WORD)(EVMbase+0x400))!=0) break;
      }
      *x = *x >> 16;
      *x |= ((long)inword((WORD)(EVMbase+0x808)) << 16);
      outword((WORD)(EVMbase+0x14),6);
    }
    return NO_ERR;
  }
  //------------------------------------------
  // Need for quirky ports that shut down
  // or do not switch direction fast enough
  if(LO_PWR)
    outctrl((char)(PPSTRB_HI | _DIR));
  //------------------------------------------
  for(shift=0;shift<32;shift-=(char)WSHIFT)
  {
    // Handshake for 1st xfer and slow down for long cables
    for(;t>0;t--)
    {
      if((instat & 0x8) == 0) break;
      if(t<=1)
      {
        enable_Mtask();
        return(RECV_ERR);
      }
    }
    t = timeout;
    // Get inport bits and strobe for next xfer
    //
    outctrl((char)(PPSTRB_LO | _DIR));
    if(WSHIFT == -8)  n = inbyte & 0xff;
    else              n = innible & 0xf;
    ul = ul | (n << shift);    
    outctrl((char)(PPSTRB_HI | _DIR));
  }
  /*--------------------------------------------*/
  /* Driving bus high source's power, not sinks */
  /*--------------------------------------------*/
  if(LO_PWR)
  {
    outbyte(0xff);
    outctrl(PPSTRB_HI);
  }
  *x = ul;
  return NO_ERR;
}
//*******************************************************
// xmit_long() sends the 32 bit value 'snd' to the
// printer port using the DSK interface protocol
//*******************************************************
MSGS DLLEXTEND_EX xmit_long(ulong snd)
{
  long t;
  int b;
  t = MAX_WAIT;  /* MAX_WAIT * 2 uS timeout */
  if(TARGET==C30_EVM)
  {
    outword((WORD)(EVMbase+ 0x14),0x6); //? Add for stability
   // outword(EVMbase+ 0x14,0x6);
    for(b=0;b<2;b++)
    {
      outword((WORD)(EVMbase+0x808),(WORD) snd);
      for(;t>=0;t--)
      {
        if(inword((WORD)(EVMbase+0x400))!=0) break;
      }
      outword((WORD)(EVMbase+ 0x14),0x6);
      snd = snd >> 16;
    }
  //  outword(EVMbase+ 0x14,0x6); //? Add for stability
    return NO_ERR;
  }
  //
  //
  for(b=0;b<4;b++)
  {
    outbyte((char)snd);   // 1.23
    outctrl(PPSTRB_LO);
    for(;t>0;t--)                // Loop will not execute for t<=0
    {
      if((instat&0x8)==0) break; // exit on good handshake
      if(t<=1)
      { outctrl((char)(PPSTRB_HI|_DIR));
        enable_Mtask();
        return(XMIT_ERR);
      }
    }
    t = timeout;
    if(b==3) outctrl((char)(PPSTRB_HI | _DIR));
    else     outctrl((char)PPSTRB_HI);
    snd = snd>>8;
  }
  if(LO_PWR)
  {
    outbyte(0xff);
    outctrl((char)PPSTRB_HI);
  }
  return NO_ERR;
}
//*******************************************************
// xmit_byte() sends a single 8 bit value 'snd' to the
// printer port using the DSK interface protocol
//*******************************************************
MSGS DLLEXTEND_EX xmit_byte(char snd)
{
  int t;
  if(TARGET==C30_EVM)
  {
    return XMIT_ERR;
  }
  outbyte(snd);
  outctrl(PPSTRB_LO);  // Signal data XMIT
  for(t=50; t>0; t--)  // wait no more than 50 ms for handshake
  {
    if((instat & 0x8) == 0) break;
    mydelay(1);
  }
  if(t==0)
  {
    outctrl((char)(PPSTRB_HI|_DIR));
    enable_Mtask();
    return(XMIT_ERR);
  }
  if(LO_PWR)
  {
    outctrl(PPSTRB_HI);       // Finish transfer
    outbyte(0xff);            // Drive bus high
  }
  else
    outctrl((char)(PPSTRB_HI|_DIR));  // Complete byte transfer
  return NO_ERR;
}
/*----------------------------------------------------------------------
   get_buswidth() is a special purpose function that is used to
   determine the printer port mode which a potentialy active kernel
   is using.  This function is what enables a users application to
   reliably start and interact with other applications when no command
   line arguments (causing a reset for example) are given.
  ----------------------------------------------------------------------*/
MSGS DLLEXTEND_EX get_buswidth(void)
{
  MSGS err;
  uchar n;
  int t,i;
  long shift;
  ulong CMD[4];
  /*
   This routine assumes that all printer ports can be used in nibble mode
   and that the kernels bus width value can be reconstructed from 4 bit
   reads.  By strobing only half of the reads, the mode can be reconstructed
   from nibble readback.  Note that the DSK hardware enables both the nibble
   and byte bus drivers at the same time.

   - Four (4) strobes are applied in nibble mode and a value is
     reconstructed from nibble readback

   - If byte width is indicated, change the host settings and return

   - If nibble mode is indicated, send four more strobes and verify
     nibble mode before returning.

   - If neither nibble or byte mode is indicated, reset the kernel
     or pick another port

  */
//#define SIZELOC 0x809FFEL
  if(TARGET==C30_EVM)
  {
    return XMIT_ERR;
  }

  _DIR = 0;           // Use 'standard' nibble mode for partial readback
  WSCOUNT = 7;        //
  WSHIFT  =-4;        //
  CMD[0] = XREAD;     // Read command
  CMD[1] = 1;         // return data packet length is 1
  CMD[2] = SIZELOC;   // address of WSCOUNT
  CMD[3] = 1;         // srce indx
  disable_Mtask();
  /*---------------------------------------------
    Send the command to initiate a readback
    even though the readback mode is unknown
  ----------------------------------------------*/
  for(i=0;i<4;i++)
  if((err = xmit_long(CMD[i]))!=NO_ERR)
  {
    enable_Mtask();
    return err;
  }
  /*---------------------------------------------
    Do a partial receive (16 bits) using nibble mode
  ----------------------------------------------*/
  WSCOUNT = 0;
  for(shift=0;shift<16;shift-=WSHIFT)
  {
    for(t=200;t>0;t--)
    { if((instat & 0x8) == 0) break;
      if(t<=1)
      {
        enable_Mtask();
        WSCOUNT= 7;
        WSHIFT =-4;
        return RECV_ERR;
      }
    }
    outctrl((char)(PPSTRB_LO | _DIR));
    if(WSHIFT == -8)  n = (uchar)(inbyte & 0xff); // Never executed
    else              n = (uchar)(innible & 0xf);
    WSCOUNT = WSCOUNT | (n << (int)shift);
    outctrl((char)(PPSTRB_HI | _DIR));
  }
  /*------------------------------------------------
  Determine the mode 0x0003=Byte, 0x0007=Nibble
  -------------------------------------------------*/
  switch((uint)WSCOUNT)
  {
    case 3: WSHIFT = -8;
            _DIR = 0xA0;
            break;            // good kernel, byte mode
    case 7: WSHIFT = -4;      // good kernel, finish nibble read
            _DIR=0;
            for(shift=16;shift<32;shift-=WSHIFT)
            {
              for(t=200;t>0;t--)
              { if((instat & 0x8) == 0) break;
                if(t<=1)
                {
                  enable_Mtask();
                  WSCOUNT=7;
                  return RECV_ERR;
                }
              }
              outctrl((char)(PPSTRB_LO | _DIR));
              n = (uchar)(innible & 0xf);
              WSCOUNT = WSCOUNT | (ulong)(n << (int)shift);
              outctrl((char)(PPSTRB_HI | _DIR));
            }
            if(WSCOUNT==7) break;  // Re-verify
            WSCOUNT = 7;
            WSHIFT = -4;
            enable_Mtask();
            return COM_ERR;
   default: _DIR = 0;     /* bad or incomplete kernel         */
            WSCOUNT =  7; /* return with nibble mode settings */
            WSHIFT  = -4;
            enable_Mtask();
            return COM_ERR;
  }
  /*--------------------------------------------------------
  Now double check the kernels bus mode by reading the back
  the shift count using a standard getmem readback
  ---------------------------------------------------------*/
  if((err=getmem(SIZELOC+1,1,&(ulong &)WSHIFT ))==NO_ERR)
  {
    if((WSCOUNT == 7) && (WSHIFT == -4))
    {
      _DIR = 0;
      enable_Mtask();
      return NO_ERR;
    }
    if((WSCOUNT == 3) && (WSHIFT == -8))
    {
      _DIR = 0xA0;
      enable_Mtask();
      return NO_ERR;
    }
    err = COM_ERR;
  }
  _DIR = 0;
  WSCOUNT =  7;
  WSHIFT  = -4;  // MUST return with a valid mode
  enable_Mtask();
  return err;
}
/*---------------------------------------------------------------
   set_buswidth() forces the kernel buswidth to operate in a
   particular mode.  set_buswidth() should only be called if the
   kernel is operational and not in use by other applications
  ---------------------------------------------------------------*/
MSGS DLLEXTEND_EX set_buswidth(void)
{
  MSGS err;
  if(TARGET==C30_EVM)
  {
    return XMIT_ERR;
  }
  switch(BW_force)
  {
    case 0: WSCOUNT=3; WSHIFT=-8; /* Force byte mode first */
            _DIR = 0xA0;
            if((err=putmem(SIZELOC  ,1,&WSCOUNT))!=NO_ERR) return err;
            if((err=putmem(SIZELOC+1,1,&(ulong &)WSHIFT ))!=NO_ERR) return err;
            if((err=get_buswidth())==NO_ERR) return NO_ERR;
            /*                                      */
            /* If fail, fall through to nibble mode */
            /*                                      */
    case 4: WSCOUNT=7; WSHIFT=-4;        // Nibble... uni-dir-port
            _DIR = 0;
            if((err=putmem(SIZELOC  ,1,&WSCOUNT))!=NO_ERR) return err;
            if((err=putmem(SIZELOC+1,1,&(ulong &)WSHIFT ))!=NO_ERR) return err;
            if((err=get_buswidth())==NO_ERR) return NO_ERR;
            return err;
    case 8: WSCOUNT=3; WSHIFT=-8;        // Byte... bi-dir-port
            _DIR = 0xA0;
            if((err=putmem(SIZELOC  ,1,&WSCOUNT))!=NO_ERR) return err;
            if((err=putmem(SIZELOC+1,1,&(ulong &)WSHIFT ))!=NO_ERR) return err;
            if((err=get_buswidth())==NO_ERR) return NO_ERR;
            return err;
    case -1: return NO_ERR; // Do not modify buswidth
  }
  return COM_ERR;
}

void DLLEXTEND_EX Pulse_Init(void)
{
  if(TARGET==C30_EVM)
  {
    return;
  }
  outctrl(RESET_LO); mydelay(1); // this port by pulsing the INIT line
  outctrl(RESET_HI); mydelay(1);
}
//
// This file recreates the IO port functions which are not included
// in Borland C++ builds for Win32.  MS Dev Studio has equivlent functions
// using different names that are built into the compiler
//
#if __WIN32__
//#include <dos.h>
//#include <windows.h>
// 0x66 is the 0x86 CPU operand-size override prefix. This prefix complements
// the D bit in the segment descriptor, making the CPU assume it is running
// in a 16-bit (same as DOS) segment.  It must be modified to work in a
// 32-bit segment since Borland asm cannot generate OUT  DX, EAX must force in
void outportb(WORD port, unsigned char value)
{
/*
   _DX  = port;
   _EAX = (long)value;
   __emit__(0x66, 0xEF);
  */
//_asm push dx
//_asm push ax
  _asm mov  dx,port
  _asm mov  al,value
  _asm out  dx,al
//_asm pop  ax
//_asm pop  dx

}
void outport(WORD port, WORD value)
{
// _DX  = port;
// _EAX = (long)value;
// __emit__(0x66, 0xEF);
  _asm push dx
  _asm push ax
  _asm mov  dx,port
  _asm mov  ax,value
  _asm out  dx,ax
  _asm pop  ax
  _asm pop  dx
}

unsigned char inportb(WORD port)
{
  unsigned char temp;
//_DX = port;
//__emit__(0x66, 0xED);
// return(_EAX);
//temp = (SHORT)_AX;
//return temp;
  _asm push ax
  _asm push dx
  _asm mov  dx,port
  _asm in   al,dx
  _asm mov  temp,al
  _asm pop  dx
  _asm pop  ax
  return(temp);

}
WORD inport(WORD port)
{
  WORD temp;
//_DX = port;
//__emit__(0x66, 0xED);
// return(_EAX);
//temp = (SHORT)_AX;
//return temp;
  _asm push ax
  _asm push dx
  _asm mov  dx,port
  _asm in   ax,dx
  _asm mov  temp,ax
  _asm pop  dx
  _asm pop  ax
  return(temp);

}

#endif



