/*
 *  ======== dpi.c ========
 *  New Pipe Driver
 *
 *  The pipe device is a software device used to stream data between
 *  SPOX tasks.  The old DPI driver accomplished this by creating a
 *  shared pair of queues distinct from the original queues created by
 *  SIO.  This allowed efficient I/O plus the ability to have multiple
 *  readers and writers. This method has some limitations, the most
 *  significant being that stacking drivers won't stack on top of the
 *  old DPI (idling is the other).
 *
 *  The new DPI is a pipe implementation which can have stacking drivers
 *  on top. Instead of criss-crossing device pointers to the shared
 *  queues, the orignal SIO queues are left intact and the input/output
 *  function handles the transferring between the device queues.  This
 *  method has the limitation of only one reader and one writer allowed.
 *  It does correctly handle device idling, with one exception: if the
 *  input side of the pipe is closed before the output side, the output
 *  side will throw away remaining output frames instead of waiting for
 *  them to drain if it's been created with attrs.flush TRUE.
 *
 *  The new DPI also fixes the simultaneous SIO_select problem per MR 2172
 *
 *  As usual, SEM semaphores are used to keep track of the number of
 *  available buffers on each of the fromdevice queues.
 *
 *! Revision History
 *! ================
 *! 08-Sep-1995 rt
 *!	Fixed bug in DPI_issue not setting frame size to 0 for output.
 *! 06-Jul-1995 rt
 *!	Added ISSUERECLAIM support, plus other slight mods for new DEV.
 *! 14-Mar-1995 rt
 *!	Created from old dpi.c.
 */

#include <std.h>

#include <dev.h>
#include <mem.h>
#include <que.h>
#include <sem.h>
#include <sys.h>
#include <tsk.h>

#include <dpi.h>

#ifdef _2100_
int atoi(const char *s);
#else
#include <stdlib.h>	/* for atoi() */
#endif

/*
 *  ======== SPipeObj ========
 * *ONE* SPipeObj is allocated for each pipe (e.g. "/pipe0").  readers
 * and writers adjust their semaphore pointers (in DPI_open()) to use
 * the common pair in SPipeObj.
 */
typedef struct SPipeObj {
    QUE_Elem	    link;	/* MUST be first element of this structure */
    SEM_Handle	    dataSem;
    SEM_Handle	    freeSem;
    SEM_Handle	    readySem[2];
    DEV_Handle      device[2];
    Int		    id;
} SPipeObj;

/*
 * One PipeObj is allocated for each open device.
 */
typedef struct PipeObj {
    SEM_Handle	    toSem;
    SEM_Handle	    fromSem;
    SPipeObj	    *sPipe;
} PipeObj;

#define Static static

/*
 *  Driver function table.
 */
Static Int	DPI_close(DEV_Handle dev);
Static Int	DPI_open(DEV_Handle dev, String name);
Static Int	DPI_idle(DEV_Handle dev, Bool flush);
Static Int	DPI_ioFunc(DEV_Handle dev);
Static Bool	DPI_ready(DEV_Handle dev, SEM_Handle sem);
Static Int	DPI_issue(DEV_Handle dev);
Static Int	DPI_reclaim(DEV_Handle dev, Uns timeout);

DEV_Fxns DPI_FXNS = {
    DPI_close,		/* close */
    DEV_CTRL,		/* ctrl */
    DPI_idle,		/* idle */
    DPI_ioFunc,		/* input */
    DPI_open,		/* open */
    DPI_ioFunc,		/* output */
    DPI_ready,		/* ready */
    DPI_issue,		/* issue */
    DPI_reclaim,	/* reclaim */
};

Static SEM_Handle mutex;	/* for mutual exclusion in open and close */
Static QUE_Handle sPipeList;	/* list of all shared pipe objects */

Static SPipeObj *mkSPipe(DEV_Handle dev);


/*
 *  ======== DPI_close ========
 */
Static Int DPI_close(DEV_Handle dev)
{
    PipeObj	    *pipe = (PipeObj *)dev->object;
    SPipeObj	    *sPipe = pipe->sPipe;

    MEM_free(0, pipe, sizeof (PipeObj));
    
    SEM_pend(mutex, SYS_FOREVER);

    sPipe->device[dev->mode] = NULL;
    sPipe->readySem[dev->mode] = NULL;
	     
    if (sPipe->device[DEV_INPUT] == NULL &&
      sPipe->device[DEV_OUTPUT] == NULL) {
	/* delete all shared pipe sub-objects */
	SEM_delete(sPipe->dataSem);
	SEM_delete(sPipe->freeSem);

	/* remove sPipe obj from sPipeList */
	QUE_remove(&sPipe->link);
	
	/* delete sPipe object itself */
	MEM_free(0, sPipe, sizeof (SPipeObj));
    }

    SEM_post(mutex);

    return SYS_OK;
}

/*
 *  ======== DPI_idle ========
 */
Static Int DPI_idle(DEV_Handle dev, Bool flush)
{
    PipeObj *pipe = (PipeObj *)dev->object;
    SPipeObj *sPipe = pipe->sPipe;
    DEV_Frame *frame;
    Int i;

    if (dev->mode == DEV_INPUT && dev->model == DEV_STANDARD) {
	while ((frame = QUE_get(dev->fromdevice)) !=
	  (DEV_Frame *)dev->fromdevice) {
	    QUE_put(dev->todevice, frame);
	}

	SEM_reset(sPipe->dataSem, 0);
    }
    else {	/* DEV_OUTPUT || DEV_ISSUERECLAIM */
	if (flush) {
	    /*
	     * Throw away all remaining output frames.
	     */
	    while ((frame = QUE_get(dev->todevice)) !=
	      (DEV_Frame *)dev->todevice) {
		QUE_put(dev->fromdevice, frame);
	    }
	}
	else {
	    /*
	     * Pending 'nbufs' times will ensure that all frames have
	     * recycled back to the fromdevice queue.
	     */
	    for (i = 0; i < dev->nbufs; i++) {
		SEM_pend(sPipe->freeSem, SYS_FOREVER);
	    }
	}

	SEM_reset(sPipe->freeSem, dev->nbufs);
    }

    return SYS_OK;
}
    
/*
 *  ======== DPI_init ========
 */
Void DPI_init(Void)
{
    mutex = SEM_create(1, NULL);
    sPipeList = QUE_create(NULL);

    if (mutex == NULL || sPipeList == NULL) {
	SYS_abort("DPI");
    }
}
    
/*
 *  ======== DPI_ioFunc ========
 *  this function is used for both input and output.
 */
Static Int DPI_ioFunc(DEV_Handle dev)
{
    PipeObj	    *pipe = (PipeObj *)dev->object;
    SPipeObj	    *sPipe = pipe->sPipe;
    DEV_Handle	    otherdev = sPipe->device[dev->mode ^ 0x1];
    SEM_Handle	    otherReady = sPipe->readySem[dev->mode ^ 0x1];

    /*
     * We know there's an available frame on dev's todevice.
     * Move it to the other guy's fromdevice only if we can grab
     * a frame from the other guy's todevice.
     */
    TSK_disable();
    if (otherdev != NULL &&
        !QUE_empty(otherdev->todevice) && !QUE_empty(dev->todevice)) {

	QUE_put(dev->fromdevice, QUE_get(otherdev->todevice));
	QUE_put(otherdev->fromdevice, QUE_get(dev->todevice));

	TSK_enable();

	SEM_post(pipe->toSem);

	if (otherReady != NULL) {
	    SEM_post(otherReady);
	}
    }
    else {
	TSK_enable();

	SEM_pend(pipe->fromSem, SYS_FOREVER);
    }

    return SYS_OK;
}

/*
 *  ======== DPI_issue ========
 */
Static Int DPI_issue(DEV_Handle dev)
{
    PipeObj	    *pipe = (PipeObj *)dev->object;
    SPipeObj	    *sPipe = pipe->sPipe;
    DEV_Handle	    otherdev = sPipe->device[dev->mode ^ 0x1];
    SEM_Handle	    otherReady = sPipe->readySem[dev->mode ^ 0x1];
    DEV_Frame       *otherframe;
    DEV_Frame       *frame;

    /*
     * We know there's an available frame on dev's todevice.
     * Move it to the other guy's fromdevice only if we can grab
     * a frame from the other guy's todevice.
     */
    if (otherdev != NULL &&
      (otherframe = QUE_get(otherdev->todevice)) !=
      (DEV_Frame *)otherdev->todevice) {

	frame = QUE_get(dev->todevice);

/*
 * #define COPYBUFS to cause buffers to be copied through the pipe
 * instead of being exchanged.  Doing so retains the semantics of
 * the ISSUERECLAIM model, but is slow.  If COPYBUFS is *not* defined,
 * then one side reclaims buffers issued by the other side, thereby
 * not strictly retaining buffer ordering.
 */
#ifdef COPYBUFS
	if (dev->mode == DEV_INPUT) {
	    dstQ = dev->todevice;
	    srcQ = otherdev->todevice;
	}
	else {
	    dstQ = otherdev->todevice;
	    srcQ = dev->todevice;
	}
	memcpy(dstQ, srcQ, otherframe->size);

	QUE_put(dev->fromdevice, frame);
	QUE_put(otherdev->fromdevice, otherframe);
	/*
	 * frames reclaimed from an output device must have size 0.
	 */
	if (dev->mode != DEV_INPUT) {
	    frame->size = 0;
	}
#else
	QUE_put(dev->fromdevice, otherframe);
	QUE_put(otherdev->fromdevice, frame);
	/*
	 * frames reclaimed from an output device must have size 0.
	 */
	if (dev->mode != DEV_INPUT) {
	    otherframe->size = 0;
	}
	else {
	    frame->size = 0;
	}
#endif
    }

    SEM_post(pipe->toSem);

    if (otherReady != NULL) {
	SEM_post(otherReady);
    }

    return SYS_OK;
}

/*
 *  ======== DPI_open ========
 */
Static Int DPI_open(DEV_Handle dev, String name)
{
    PipeObj	    *pipe;
    SPipeObj	    *sPipe, *tmpPipe;

    /* decode and validate devid */
    if (dev->devid < 0) {
	dev->devid = atoi(name);
    }

    SEM_pend(mutex, SYS_FOREVER);

    /* search pipe list for previously opened pipe with same id */
    sPipe = MEM_ILLEGAL;
    if (!QUE_empty(sPipeList)) {
	tmpPipe = (SPipeObj *)QUE_head(sPipeList);
	do {
	    if (tmpPipe->id == dev->devid) {
		sPipe = tmpPipe;
		break;
	    }
	    tmpPipe = (SPipeObj *)QUE_next((&tmpPipe->link));
	} while (tmpPipe != (SPipeObj *)sPipeList);
    }

    if (sPipe == MEM_ILLEGAL) {
	/*
	 * Allocate and initialize SPipeObj on first open.
	 */
        sPipe = mkSPipe(dev);
	if (sPipe == MEM_ILLEGAL) {
	    return SYS_EALLOC;
	}
	QUE_put(sPipeList, &sPipe->link);
    }
    else {	/* sPipe found on list */
	if (sPipe->device[dev->mode] != NULL) {
	    /*
	     * Only one input and one output allowed
	     */
	    return SYS_EBUSY;
	}
    }

    SEM_post(mutex);

    pipe = MEM_alloc(0, sizeof (PipeObj), 0);
    if (pipe == MEM_ILLEGAL) {
	/*
	 * We may want to undo work done by mkSPipe() if first open.
	 */
	return SYS_EALLOC;
    }

    /*
     * Criss-cross SEM handles so both sides are referencing
     * the same physical objects.
     */
    if (dev->mode == DEV_INPUT) {
	pipe->fromSem = sPipe->dataSem;
	pipe->toSem = sPipe->freeSem;
    }
    else {
	pipe->toSem = sPipe->dataSem;
	pipe->fromSem = sPipe->freeSem;

	/*
	 * DEV_ISSUERECLAIM model doesn't prime the pump.
	 */
	if (dev->model == DEV_STANDARD) {
	    SEM_reset(sPipe->freeSem, dev->nbufs);
	}
    }

    /*
     * Point things around.
     */
    sPipe->device[dev->mode] = dev;
    pipe->sPipe = sPipe;
    dev->object = (Ptr)pipe;

    return SYS_OK;
}

/*
 *  ======== DPI_ready ========
 *  called by SIO_select(), returns TRUE if device is ready.
 */
Static Bool DPI_ready(DEV_Handle dev, SEM_Handle sem)
{
    PipeObj	    *pipe = (PipeObj *)dev->object;

    pipe->sPipe->readySem[dev->mode] = sem;
    
    return !(QUE_empty(dev->fromdevice));
}

/*
 *  ======== DPI_reclaim ========
 */
Static Int DPI_reclaim(DEV_Handle dev, Uns timeout)
{
    PipeObj	    *pipe = (PipeObj *)dev->object;

    if (SEM_pend(pipe->fromSem, timeout)) {
	return SYS_OK;
    }
    else {
	return SYS_ETIMEOUT;
    }
}

/*
 *  ======== mkSPipe ========
 */
Static SPipeObj *mkSPipe(DEV_Handle dev)
{
    SPipeObj *sPipe = MEM_alloc(0, sizeof (SPipeObj), 0);
    
    if (sPipe != MEM_ILLEGAL) {
	sPipe->dataSem = SEM_create(0, NULL);
	if (sPipe->dataSem == NULL) {
	    goto e2;
	}
	sPipe->freeSem = SEM_create(0, NULL);
	if (sPipe->freeSem == NULL) {
	    goto e1;
	}
	sPipe->readySem[DEV_INPUT] = NULL;
	sPipe->readySem[DEV_OUTPUT] = NULL;
	sPipe->device[DEV_INPUT] = NULL;
	sPipe->device[DEV_OUTPUT] = NULL;
	sPipe->id = dev->devid;

	return sPipe;
    }
    else {
	return MEM_ILLEGAL;
    }
e1:
    SEM_delete(sPipe->dataSem);
e2:
    /* delete sPipe object itself */
    MEM_free(0, sPipe, sizeof (SPipeObj));

    return MEM_ILLEGAL;
}


