;/***************************************************************************\
;* checkmem.s                                                                *
;* Version 1.00                                                              *
;* Non-disruptive memory checker and sizer.                                  *
;* Copyright (C) 1992 Advanced RISC Machines Limited. All rights reserved.   *
;* Written by Dave Flynn.                                                    *
;* Project started on 2nd January 1992.                                      *
;* Last change was 2nd January 1992.                                         *
;\***************************************************************************/

; ---------------------------------------------------------------
; Find size of main RAM and return in r0. r1-r7 corrupted.

FindRAMSize
                MOV     r0, #0          ; start at base of physical RAM
                MOV     r1, #4          ; 4 byte grain is fine enough

        ; Drop into checksize


; Non-invasive memory sizer
;  parameters:  r0 is base pointer
;               r1 is test increment (multiple of 4 please)
;  return:      r0 is size (bytes)
;  registers    r1-r7 corrupted.
;
; Memory is left intact after the size test.
;  register usage:
;                  +------+
;   base(r6)  ---->|(base)| (save old -> r4)
;       inc in r7  |      |
;                  |      |
;   probe(r5) ---->|      | (save in r1)
;                  +------+
;   wrapped image  |(base)|
; basic algorithm:
;      save copy of base location
;      write pattern to base
; loop save copy of word from memory
;      write a pattern, read back and exit if write failed
;      write inverted pattern, read back and exit as above
;      read base and see if probe has wrapped
;      restore probe location
;      advance probe and loop if base still okay

checksize
                MOV     r7, r1          ; keep step size in r7
                MOV     r6, r0          ; keep base address in r6
                MOV     r5, r6          ; r5 is probe address
                LDR     r4, [r6, #0]    ; preserve original base contents
                MOV     r2, #&55        ; build adjacent bit tests
                ORR     r2, r2, #&AA00  ; make AA55 pattern
                ORR     r2, r2, r2, LSL #16 ; final AA55AA55 (helps check endian-ness)
                MVN     r3, r2          ; use inverse copy as alternate pattern

        ; Move up memory in steps, probing each location with two different
        ; patterns to see if memory is present, and checking the base of
        ; memory too (to detect aliasing of wrapped memory).

                ; save AA55AA55 in base which should be there until wrapped
                STR     r2, [r6, #0]    ; store first pattern at base
                LDR     r0, [r6, #0]    ; and read back
                CMP     r0, r2          ; check okay
                BNE     checksizefail   ; give up if not

        ; byte testing valuable as ARM broadcasts 4 bytes
        ; and any faulty write enables will corrupt other bytes

checkbyte0
        ; check byte writes work independently (endian-independent)

                LDRB    r1, [r6, #0]    ; byte at a time
                MOV     r0, #&A5        ; using local pattern
                STRB    r0, [r6, #0]    ; test byte 0
                LDRB    r0, [r6, #0]    ; read back
                CMP     r0, #&A5        ; test
                BNE     checksizefail   ; fail if different
                STRB    r1, [r6, #0]    ; restore byte
                LDR     r0, [r6, #0]    ; test word still intact
                CMP     r0, r2          ; still pattern one
                BNE     checksizefail   ; fail if different

        ; in line for zero stack usage, etc.

checkbyte1
                LDRB    r1, [r6, #1]    ; preserve byte 1
                MOV     r0, #&A5        ; using local pattern
                STRB    r0, [r6, #1]    ; test byte 1
                LDRB    r0, [r6, #1]    ; read back
                CMP     r0, #&A5        ; test
                BNE     checksizefail   ; fail if different
                STRB    r1, [r6, #1]    ; restore byte
                LDR     r0, [r6, #0]    ; test word still intact
                CMP     r0, r2          ; still pattern one
                BNE     checksizefail   ; fail if different
checkbyte2
                LDRB    r1, [r6, #2]    ; preserve byte 2
                MOV     r0, #&A5        ; using local pattern
                STRB    r0, [r6, #2]    ; test byte 2
                LDRB    r0, [r6, #2]    ; read back
                CMP     r0, #&A5        ; test
                BNE     checksizefail   ; fail if different
                STRB    r1, [r6, #2]    ; restore byte
                LDR     r0, [r6, #0]    ; test word still intact
                CMP     r0, r2          ; still pattern one
                BNE     checksizefail   ; fail if different
checkbyte3
                LDRB    r1, [r6, #3]    ; preserve byte 3
                MOV     r0, #&A5        ; using local pattern
                STRB    r0, [r6, #3]    ; test byte 3
                LDRB    r0, [r6, #3]    ; read back
                CMP     r0, #&A5        ; test
                BNE     checksizefail   ; fail if different
                STRB    r1, [r6, #3]    ; restore byte
                LDR     r0, [r6, #0]    ; test word still intact
                CMP     r0, r2          ; still pattern one
                BNE     checksizefail   ; fail if different

                ADD     r5, r6, r7      ; r5 is probe address (base+step)
checksizeloop
                LDR     r1, [r5, #0]    ; preserve probed value in r1
                STR     r2, [r5, #0]    ; store first pattern
                LDR     r0, [r5, #0]    ; read it back
                CMP     r0, r2          ; is it correct?
                BNE     checksizefail   ; no working memory here
                STR     r3, [r5, #0]    ; store second pattern
                LDR     r0, [r5, #0]    ; read it back
                CMP     r0, r3          ; is it correct?
                BNE     checksizefail   ; not working
                ; if reads and writes okay did last write wrap to base?
                LDR     r0, [r6, #0]
                STR     r1, [r5, #0]    ; restore probed value
                CMP     r0, r2          ; base still reads as first pattern?
                ADDEQ   r5, r5, r7      ; if okay then increment probe
                BEQ     checksizeloop   ; and loop, else fall through to fail

        ; Address of first failing location in r2, original contents in r6
        ; Original contents of base in r7

checksizefail
                STR     r1, [r5, #0]    ; restore original contents of probe
                STR     r4, [r6, #0]    ; restore original contents of base
                SUB     r0, r5, r6      ; valid size is (probe - base)
                MOV     pc, lr

                END

