/***************************************************************************\
* serdrive.c                                                                *
* Version 1.00                                                              *
* PC Serial Line Driver (Zortech & Watcom C)                                *
* Copyright (C) 1991 Advanced RISC Machines Limited. All rights reserved.   *
* Written by Dave Jaggar.                                                   *
\***************************************************************************/

#include <dos.h>
#include <bios.h>
#include "serdrive.h"

void ConfigRDP(int port, unsigned Speed) ;

/* I have hacked the code in here to not use _bios_serialcom in the open
routine (so baud rates of 19k2 and 38k4 would be selectable). Unfortunatly
the PC could not keep up with the PIE at 38k4. I thought about using direct
dos calls, but the bios code is ghastly and slow, so I call outp and inp to
hopefully produce good code. This required knowledge of the serial line, i.e.
never wait for DTR, CTS, DSR or RTS when reading or writing. Anyway it goes.
  Dave Jaggar 25/2/1992 */

#define INIT 0
#define WRITE 1
#define READ 2
#define STATUS 3

#define RXD port
#define TXD port
#define INTENAB port + 1
#define INTID port + 2
#define DATA port + 3
#define MODEM port + 4
#define LSTATUS port + 5
#define MSTATUS port + 6
#define BRLOW port
#define BRHIGH port + 1

#define DTR 1 /* bits in the modem control and status registers */
#define RTS 2
#define DSR 16
#define CTS 32

#define RXDREADY 0x1
#define TXDREADY 0x20

#define INITDATA 3 + 0 + 0
    /* length        stop bits          parity
       2 = 7           0 = 1           0 = none
       3 = 8           4 = 2           8 = odd
                                      24 = even */

static int logging;

void RDPLogging(int val)
{
  logging = val;
}

/***********************************************************************\
*                       Convert a word to four bytes.                   *
\***********************************************************************/

void WordToBytes(unsigned long word, unsigned char bytes[])
{bytes[0] = (char)(word & 0xff) ;
 bytes[1] = (char)(word >> 8 & 0xff) ;
 bytes[2] = (char)(word >> 16 & 0xff) ;
 bytes[3] = (char)(word >> 24 & 0xff) ;
}

/***********************************************************************\
*                       Convert four bytes to a word.                   *
\***********************************************************************/

void BytesToWord(unsigned char bytes[], unsigned long *word)
{*word = ((unsigned long)bytes[0] | (unsigned long)bytes[1] << 8 |
          (unsigned long)bytes[2] << 16 | (unsigned long)bytes[3] << 24) ;
}

/***********************************************************************\
*                         Open the serial line                          *
\***********************************************************************/

int OpenRDP(const Dbg_ConfigBlock *config)
{int port ;

 if (config->serialport == 2)
    port = 0x2f8 ;
 else if (config->serialport > 2)
    port = config->serialport ;
 else
    port = 0x3f8 ;

#ifdef RDI_VERBOSE
 if (logging & 1)
    pisd_DebugPrint("Using port %d\n", port);
#endif

 ConfigRDP(0,0) ;
 outp(DATA, INITDATA) ;
 outp(INTENAB, 0) ;

 while (inp(LSTATUS) & RXDREADY)
    (void)inp(RXD) ;

 return(port);
}

/***********************************************************************\
*       Set the serial line speed, and set up its initial state.        *
\***********************************************************************/

void ConfigRDP(int port, unsigned speed)
{ /* speed = 0 or 1 for 9.6K, 2 for 19.2K, 3 for 38.4K.
     Unfortunately, our PC (hw? sw?) can only manage 9.6K.
   */
 int br, dlab ;

 switch (speed) {
    default :
    case 0 :
    case 1 :
       br = 12 ;
       break ;
    case 2 :
       br = 6 ;
       break ;
    case 3 :
       br = 3 ;
       break ;
    }
 dlab = inp(DATA) ;
 outp(DATA, dlab | 0x80) ;
 outp(BRLOW, br & 0xff) ;
 outp(BRHIGH, (br >> 8) & 0xff) ;
 outp(DATA, dlab) ;
}

/***********************************************************************\
*            Read a block of len bytes from the serial line.            *
\***********************************************************************/

void ReadRDP(int port, unsigned char block[], unsigned len)
{int i ;

 for (i = 0 ; i < len ; i++) {
    while (!(inp(LSTATUS) & RXDREADY)) ;
    block[i] = inp(RXD) ;
    }

#ifdef RDI_VERBOSE
 if (logging & 2)
    for (i = 0 ; i < len ; i++)
       pisd_DebugPrint("<%02x ", block[i]);
#endif
 }

/***********************************************************************\
*            Write a block of len bytes to the serial line.             *
\***********************************************************************/

void WriteRDP(int port, unsigned char block[], unsigned len)
{int i ;

#ifdef HOURGLASS
 HOURGLASS();
#endif

#ifdef RDI_VERBOSE
/* write log output first, so that overrun is less likely */
 if (logging & 2)
    for (i = 0 ; i < len ; i++)
       pisd_DebugPrint(">%02x ", block[i]);
#endif

 for (i = 0 ; i < len ; i++) {
    while (!(inp(LSTATUS) & TXDREADY)) ;
    outp(TXD, block[i]) ;
    }
 }

/***********************************************************************\
*                       Close the serial line.                          *
\***********************************************************************/

void CloseRDP(int port)
{

}
