/*
  Copy a file to a RCA MicroDOS disk image. File size is limited to slightly
  less than 64K (127*512)

  Copyright 2016 David W. Schultz
  david.schultz@earthlink.net
  http://home.earthlink.net/~david.schultz

  Licensed under Creative Commons Attribution-NonCommercial-ShareAlike 3.0 
  license:

  http://creativecommons.org/licenses/by-nc-sa/3.0/us/


  If you need a blank file system, you can create one with:

  sudo dd if=/dev/zero of=blank.img bs=1k count=315

  If you must have something in the disk ID block, run SYSGEN.


  The top level format of a RCA MicroDOS diskette (page 90 of manual) is:

  1 2 3 4 5 6 7 8 9   -- Sector
T I D D D D D D D D
R C O O O O O O O O
A O O O O O O O O O
C O O O O O O O O U
K U U U U U U U U U
S

I - Disk Identification block
D - Directory
C - Cluster Allocation Table
O - Operating System
U - Unused

  For the rest of the details, see the manual, MPM-241.

  This program does not support non-contiguous files. It searches from the 
  end of the cluster allocation vector until it finds an allocated cluster.
  It then writes the file after that cluster.

  28 Nov. 2016 - Added file locking to help prevent conflicts with 1802sim.
  29 Nov. 2016 - Fixed an off by one error in the allocation table
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <stdint.h>
#include <endian.h>
#include <string.h>
#include <ctype.h>
#include <libgen.h>

#define LAST (78*8-2)
#define DISKSIZE (512*9*70)

/*
  MicroDOS directory entry 
 */
struct DIRENT {
  unsigned char filename[6];
  unsigned char extension[3];
  char fill;
  uint16_t ssn;           // note: big-endian 
  unsigned char attribute;
  char fill2[3];
};

int btruncate = 0;        // flag to enable truncation of binary files
unsigned char buffer[512*9*70];         // buffer entire disk image

/*
  MicroDOS uses a hashing function to decide which directory sector to
  use/search for an entry. If the directory entry isn't where it is
  expected to be, it will not be found.

  The key bit of trickiness here is that MicroDOS uses an add with carry
  while adding the characters in the filename together.

  The lower 3 bits of that result are used to index into the eight
  directory sectors.
 */
int hash(unsigned char *dp)
{
  unsigned int i, h, carry;

  carry = 0;
  for(i = 0, h = 0; i < 9; i++)
    {
      h += carry + *dp++;

      if(h&0x100)
	{
	  carry = 1;
	  h &= 0xff;
	}
      else
	carry = 0;
    }
  h &= 0x07;

  return h+1;
}

/*
  Search the directory for an empty slot. Return pointer to that entry
  or NULL if error. First scan sector for a matching file name.
 */

struct DIRENT *find_dir_ent(int fd, struct DIRENT *dp)
{
  int off, i;

  off = 512*hash(dp->filename);

  for(i = 0; i < 32; i++)
    {
      if(strncmp((char*)&buffer[off+i*sizeof(struct DIRENT)], (char*)dp->filename, 9) == 0)
	{
	  fprintf(stderr, "Duplicate file name\n");
	  exit(1);
	}
    }
  
  for(i = 0; i < 32; i++)
    {
      if(!isprint(buffer[off+i*sizeof(struct DIRENT)]))
	return (struct DIRENT *)&buffer[off + i * sizeof(struct DIRENT)];
    }
  return NULL;
}

/*
  After running SYSGEN on a blank file system and not copying any files,
  the allocation vector looks like:

001200 e0 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
001210 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
001220 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
001230 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
001240 00 00 00 00 00 00 00 00 00 00 00 00 00 01 00 00


 */
/*
  Test a bit in the allocation vector given the cluster number.
 */

int allocated(int cluster)
{
  return buffer[512*9+cluster/8] & (0x80 >> (cluster%8));
}

int find_first_cluster(void)
{
  int i;

  for(i = LAST; i > 0; i--)
    if(allocated(i))
      break;

  return i+1;
}

/*
  Allocate space for the file. Returns the first cluster number on
  success, zero otherwise.
 */

int allocate(int size)
{
  int i, first;

  first = find_first_cluster();

  if(size > (LAST-first)*512)
    return 0;

  for(i = first; i < LAST; i++)
    {
      buffer[512*9+i/8] |= 0x80 >> (i % 8);
      size -= 512;
      if(size <= 0)
	break;
    }

  return first;
}

/*
  Copy the filename into the directory entry. 

  Changes filename to upper case and limits to 6.3 format.
 */
void fixup_filename(char *p, struct DIRENT *d)
{
  int i;

  p = basename(p);                // remove extraneous path info
  
  fprintf(stderr, "file=%s\n",p);
  
  for(i = 0; i < 6; i++)          // fill with spaces to start
    d->filename[i] = ' ';
  for(i = 0; i < 3; i++)
    d->extension[i] = ' ';
  
  for(i = 0; i < 6; i++)          // copy filename
    {
      if(*p == '.' || *p == 0)
	break;
      d->filename[i] = toupper(*p++);
    }
  while(*p != '.' && *p)          // shorten filename to six characters
    p++;
  
  if(*p++ == '.')
    {
      for(i = 0; i < 3; i++)
	{
	  if(*p == 0)
	    break;
	  d->extension[i] = toupper(*p++);
	}
    }

  d->fill = 0x15;
  d->fill2[0] = 0;
  d->fill2[1] = d->fill2[2] = 0;
  d->attribute = 0x2;             // ASCII
}

/*
  Setup the pointer block.
 */
void pointer_block(int first, int size)
{
  int i, N;
  unsigned char *p;

  p = &buffer[(first+7)*512];
  for(i = 0; i < 512; i++)           // start with a cleared out block
    *p++ = 0;

  N = size/512;                     // number of sectors 

  if(size%512 != 0)                 // adjust for partial sector
    N++;

  p = &buffer[(first+7)*512];

  i = N;
  while(i > 0)
    {
      if(i > 127)
	{
	  *p++ = 127;
	  *p++ = (first+7) >> 8;
	  *p++ = (first+7)&0xff;
	  i -= 127;
	  first += 127;
	}
      else
	{
	  *p++ = i;
	  *p++ = (first+8) >> 8;
	  *p++ = (first+8)&0xff;
	  break;
	}
    }
  *p++ = 0x80 | (N >> 8);
  *p++ = N & 0xff;
}
  
int main(int argc, char **argv)
{
  struct DIRENT d, *dir;
  int ifd, ofd;
  int size, first;
  char c;
  
  while((c = getopt(argc,argv, "t")) != -1)
    {
      switch(c)
	{
	case 't':
	  btruncate = 1;
	  break;
	case '?':
	  fprintf(stderr, "Unknown option character %c\n", optopt);
	  break;
	default:
	  break;
	}
    }
  if(argc == optind)
    {
      fprintf(stderr, "Usage: microdoswrite -t filename filesystem\n");
      exit(1);
    }

  /*
    Read entire disk image into memory.
   */
  if((ofd = open(argv[optind+1], O_RDWR)) == -1)
    {
      fprintf(stderr, "Failed to open <%s>\n", argv[optind+1]);
      exit(1);
    }
  /*
    Check to see if the file is locked, presumably by 1802sim.
   */
  if(lockf(ofd, F_TLOCK, 0) == -1)
    {
      fprintf(stderr, "File <%s> is locked.\nContinue? y/N:", argv[optind+1]);
      scanf("%c", &c);
      if(toupper(c) == 'Y')
	{
	  fprintf(stderr, "Performing sync() first.\n");
	  sync();
	}
      else
	exit(1);
    }
  if(read(ofd, buffer, DISKSIZE) != DISKSIZE)
    {
      fprintf(stderr, "file system size != 315K\n");
      exit(1);
    }

  fixup_filename(argv[optind], &d);

  if((dir = find_dir_ent(ofd, &d)) == 0)
    {
      fprintf(stderr, "No directory space.\n");
      exit(1);
    }

  if((ifd = open(argv[optind], O_RDONLY)) == -1)
    {
      fprintf(stderr, "Failed to open <%s>\n", argv[optind]);
      exit(1);
    }


  size = lseek(ifd, 0, SEEK_END);

  if((first = allocate(size+512)) == 0)  // include pointer block space
    {
      fprintf(stderr, "Insufficient space for file\n");
      exit(1);
    }

  
  pointer_block(first, size);
  
  d.ssn = htobe16(first+7);                  // point to pointer block
  *dir = d;

  printf("first=%d size=%d\n",first, size);


  lseek(ifd, 0, SEEK_SET);
  read(ifd, &buffer[(first+7+1)*512], size);
  close(ifd);
  
  lseek(ofd, 0, SEEK_SET);                   // write image back to file
  write(ofd, buffer, DISKSIZE);

  close(ofd);
}

  
    
