/*
  Read the files from a RCA MicroDOS disk image.

  Copyright 2016 David W. Schultz
  david.schultz@earthlink.net
  http://home.earthlink.net/~david.schultz

  Licensed under Creative Commons Attribution-NonCommercial-ShareAlike 3.0 
  license:

  http://creativecommons.org/licenses/by-nc-sa/3.0/us/


  I used setfdprm (in the fdutils package) and dd (Linux) to read
  RCA MicroDOS diskettes:

  sudo ./setfdprm -p /dev/fd0 SS DD sect=9 cyl=80 ssize=512
  sudo dd if=/dev/fd0 of=image bs=512 count=630

  If dd fails to read the entire disk then I give ddrescue a shot.

  The top level format of a RCA MicroDOS diskette (page 90 of manual) is:

  1 2 3 4 5 6 7 8 9   -- Sector
T I D D D D D D D D
R C O O O O O O O O
A O O O O O O O O O
C O O O O O O O O U
K U U U U U U U U U
S

I - Disk Identification block
D - Directory
C - Cluster Allocation Table
O - Operating System
U - Unused

  For the rest of the details, see the manual, MPM-241.


 */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <stdint.h>
#include <endian.h>
#include <string.h>
#include <ctype.h>

/*
  MicroDOS directory entry 
 */
struct DIRENT {
  char filename[6];
  char extension[3];
  char fill;
  uint16_t ssn;           // note: big-endian 
  unsigned char attribute;
  char fill2[3];
};

int btruncate = 0;        // flag to enable truncation of binary files

/*
  Read the disk ID sector and print the information there. Which is a disk
  label and a date.

  Expects the open file descriptor for the MicroDOS file system.
 */
void read_id(int fd)
{
  char buf[512];
  char date[9];
  char ID[45];
  
  if(lseek(fd, 0, SEEK_SET) != 0)
    {
      fprintf(stderr, "read_id lseek to start of file failed\n");
      exit(1);
    }
  if(read(fd, buf, 512) != 512)
    {
      fprintf(stderr, "read_id: data read failed\n");
      exit(1);
    }
  strncpy(date, &buf[12], 8);
  date[8] = 0;
  strncpy(ID, &buf[20], 44);
  ID[44] = 0;
  printf("Disk ID = %s\n", ID);
  printf("Date = %s\n", date);
}

struct DIRENT directory[8*512/16];

/*
  Read the MicroDOS director sectors into a buffer.

  Expects the open file descriptor for the MicroDOS file system.
 */
int read_dir(int fd)
{
  int i;
  if(lseek(fd, 512, SEEK_SET) != 512)
    {
      fprintf(stderr, "read_dir lseek failed\n");
      exit(1);
    }
  i = read(fd, directory, 512*8);
  if(i != 512*8)
    {
      fprintf(stderr, "read_dir read only %d bytes\n", i);
      exit(1);
    }

  return 0;
}
/*
  Scan the MicroDOS disk directory (previously read into the buffer)
  and return a pointer to the next non-NULL entry. Or a NULL pointer
  if no more are available.

  On the MicroDOS system disk the entries either looked good or were
  NULL filled. When I tried this on the BASIC1 disk, I found some entries
  that began with 0xff. I must assume that this is a deletion mark. Those
  files were "xATA.IN" and "xOUT."

  So now rather than a simple check for a NULL, I check for a printable
  character.
 */
struct DIRENT *next_dir(void)
{
  struct DIRENT *rp;
  static struct DIRENT *p = directory;

  while((int)(p-directory) < 256)
    {
      rp = p;
      if(isprint(p++->filename[0]))
	return rp;
    }
  return NULL;
}

/*
  Parse the file name in the directory entry into a form more suitable
  for use by Linux to create a file.

  The MicroDOS directory entry consists of a six character file name
  followed by a 3 character extension. These are concatenated into the 
  usual filename.ext form and spaces (all trailing, so far) are removed.

  Returns a pointer to (static) local character string. Either use it
  right away or forget about it since it will be changed the next time this
  function is called.
 */

char *dir_filename(struct DIRENT *p)
{
  int i, j;
  static char filename[20];

  for(i = 0; i < 6; i++)            // process the file name part
    {
      if(p->filename[i] == ' ')
	break;
      filename[i] = p->filename[i];
    }
  filename[i++] = '.';
  for(j = 0; j < 3; j++)            // then the extension
    {
      if(p->extension[j] == ' ')
	break;
      filename[i++] = p->extension[j];
    }
  filename[i] = 0;

  return filename;
}

/*
  Read the given file and write it to a file of the same name on the
  host file system. Also prints the load and entry addresses in case
  it happens to be a binary file.
 */

void readfile(int fd, struct DIRENT *p)
{
  int ofd;
  int length, start, entry;
  uint8_t pb[512];
  uint8_t buf[512*127];
  uint8_t *pp;
  char *fname;

  fname = dir_filename(p);
  
  /*
    MicroDOS uses a pointer block that holds data describing the location
    of 1 or more blocks of data that make up the file. The blocks do not
    need to be contiguous.

    Note that the pointer block is the first block in the file according
    to this data so it has to be carefully dropped from the output file.
   */
#ifdef DEBUG
  fprintf(stderr, "Reading pointer block at 0x%X\n", 512*be16toh(p->ssn));
#endif
  lseek(fd, 512*be16toh(p->ssn), SEEK_SET);
  if(read(fd, pb, 512) != 512)
    {
      fprintf(stderr, "Read of pointer block failed on file %s\n", fname);
      exit(1);
    }
  pp = pb;
  ofd =open(fname, O_CREAT|O_WRONLY|O_TRUNC, DEFFILEMODE);  
  while(1)
    {
      /*
	Check the termination bit to see if we are done. There is some
	extra information here but we don't need it.
       */
      if(*pp & 0x80)           
	break;
      // Extract the sector address and number of sectors in this block
      length = (*pp & 0x7f) + 1;
      start = (*(pp+1) << 8) | *(pp+2);
      // move to the appropriate location and read the data
      if(lseek(fd, 512*start, SEEK_SET) != 512*start)
	{
	  fprintf(stderr, "File read of <%s> seek to file start failed.",
		  fname);
	  close(ofd);
	  return;
	}
#ifdef DEBUG
      fprintf(stderr, "Reading %d blocks starting at %d\n", length, start);
#endif
      if(read(fd, buf, length*512) != length*512)
	{
	  fprintf(stderr, "Failed to read <%s> data.\n", fname);
	  close(ofd);
	  return;
	}
      // The first block needs to have the pointer block removed from
      // the beginning.
      if(pp == pb)
	write(ofd, buf+512, (length-1)*512);
      else
	write(ofd, buf, length*512);
      pp += 3;
    }
  /*
    Binary files have data at the end of the Pointer Block indicating
    how much of the final sector is actually used. Check the attribute
    byte and if it is flagged as a binary file, and the user has invoked
    the "-t" option, adjust the file length.
   */
  if((p->attribute & 7) == 1)
    {
      length = lseek(ofd, 0, SEEK_END);
      length -= 512 - (*(pb+500) << 8 | *(pb+501));
      if(btruncate)
	ftruncate(ofd, length);
    }
  close(ofd);
  /* Print some additional information about the file from the pointer block
     Load address and start. Useful only for binary files.
   */
  start = *(pb+504) << 8 | *(pb+505);
  entry = *(pb+506) << 8 | *(pb+507);
  printf("%10s load=0x%04x entry=0x%04x\n", fname, start, entry);
}

int main(int argc, char **argv)
{
  struct DIRENT *p;
  int fd;
  char c;
  
  while((c = getopt(argc,argv, "t")) != -1)
    {
      switch(c)
	{
	case 't':
	  btruncate = 1;
	  break;
	case '?':
	  fprintf(stderr, "Unknown option character %c\n", optopt);
	  break;
	default:
	  break;
	}
    }
  if(argc == optind)
    {
      fprintf(stderr, "Usage: microdosread -t filename\n");
      exit(1);
    }

  if((fd = open(argv[optind], O_RDONLY)) == -1)
    {
      fprintf(stderr, "Failed to open <%s>\n", argv[1]);
      exit(1);
    }

  read_id(fd);   // Read disk ID block
  read_dir(fd);  // Read disk directory

  while((p = next_dir()))  // Scan disk directory
    readfile(fd, p);  // read a file and write to local file system

  return 0;
}
