/************************************************************************
*                                                                       *
*                             ispCODE3.C                                *
*   Lattice Semiconductor Corp. Copyright 1995,1996.                    *
*   ispCODE for all Lattice Semiconductor's 5 volts ispLSI devices.     *
*                                                                       *
*   3.00  Howard Tang   04/12/95 1st V3.0 Release.                      *
*   3.01  Howard Tang   05/11/95 Add dynamic ues support for operation  *
*                                "PU".                                  *
*   3.02  Howard Tang   10/09/95 Port in the isp_setpin() from ispCODE  *
*                                V1.0 for easier hardware modifications.*
*   3.03  Howard Tang   01/30/96 Port in the isp_SDO() from ispCODE V1.0*
*                                for easier hardware modifications.     *
*   3.04  Howard Tang   02/14/96 Add verify_only parameter to ispstream_*
*                                pump to illustrate how to void the     *
*                                programming instructions in the        *
*                                ispSTREAM.                             *
************************************************************************/

#include <stdio.h>
#include <dos.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <ctype.h>
#include "lattice.h"

/*Global variables */
static int      inputport,       /*port address for the input port*/
                outputport,      /*port address for the output port*/
                port;            /*LPT number */
static int      isp_pins=NUL;    /*3.02 holds the value of parallel port*/
                                 /*     intialized to drive all pins to LOW*/
/*prototypes*/
void            find_port(void);
int             power_ok(void);
int             port_ok(void);
void            pulse_width(unsigned int milliseconds);
void            execute(void);
void            move_to_id_state(void);
int             program_ues(unsigned char *ues, unsigned char *ues_mask,
                                        unsigned char *buff, int data_length);
int             ispstream_pump(char *filename, int verify_only, int *end);
void            error_handler(int rcode, char *message);

/***************************************************************************
   Function: isp_setpin(byte pins, byte value)

   Purpose:
   To apply the specified value to the pins indicated. This routine will
   likely be modified for specific systems. As an example, this code
   is for the PC, as described below.

   This routine uses the IBM-PC standard Parallel port, along with the
   schematic shown in Lattice documentation, to apply the signals to the
   programming loop. 

   PC Parallel port pin    Signal name
   --------------------    -----------
         2                   out_SDI
         3                   out_SCLK
         4                   out_MODE
         5                   out_ISP
         6                   out_RESET
         7                   DO5
         8                   out_SENSE_CABLE_OUT    
         9                   DO7
         10                  in_SDO
         12                  in_CABLE_SENSE_IN
         15                  in_VCC_OK 
         20                  GND

   Parameters:
         - pins, which is actually a set of bit flags (defined in lattice.h) 
           that correspond to the bits of the data port. Each of the I/O port 
           bits that drives an isp programming pin is assigned a flag 
           (through a #define) corresponding to the signal it drives. To 
           change the value of more than one pin at once, the flags are added 
           together, much like file access flags are.

           The bit flags are only set if the pin is to be changed. Bits that 
           do not have their flags set do not have their levels changed. The 
           state of the port is always manintained in the static global 
           variable isp_pins, so that each pin can be addressed individually 
           without disturbing the others.

         - value, which is either HIGH (0x01 ) or LOW (0x00 ). Only these two
           values are valid. Any non-zero number sets the pin(s) high.

   Returns: nothing.


**************************************************************************/
void isp_setpin(unsigned char pins, unsigned char value)
{
  /* isp_pins is a Global value that keeps track of the current state
     of the pins  */

  if( value ) /* set flagged pins HIGH */
      isp_pins = pins | isp_pins;
  else        /* set flagged pins LOW */
      isp_pins = ~pins & isp_pins;

  /* value is put on Parallel port pins */
  outp(outputport, isp_pins);

} /* isp_setpin() */


/***************************************************************************
   Function: isp_SDO()

   Purpose:
   To get the value of the SDO pin from the input port.

   This routine is specific to the PC parallel port setup, but can easily
   be changed to address each user's hardware.

   Parameters: none

   Returns: The value of SDO, as a byte, with a value of either 0 or 1.

   Notes:
   - This routine uses the I/O port addresses contained in the global
     variable isp_active, declared in the isplsi.c file. If that
     variable is not set, these routines will not work.


****************************************************************************/
unsigned char isp_SDO(void)
{
 /* MUST return either 0x00 or 0x01 */
  return( (unsigned char) ((inp(inputport) & in_SDO ) ? HIGH : LOW) );

} /* isp_SDO() */

/*************************************************************
*                                                            *
*                         FIND_PORT                          *
*                                                            *
*  This procedure tries to locate the isp cable on parallel  *
*  ports 1,2, or 3, and sets inputport, and outputport       *
*  appropriately.  It returns upon successfully finding the  *
*  first port which has the ispDOWNLOAD cable attached, or   *
*  upon failing to find the cable attached to any port.      *
*                                                            *
*************************************************************/

void            find_port()
{
   int rcode;
   port = 0;

   do {                                          /* find cable on port */
      ++port;

      if (port == 1) 
         {
          outputport = outport1;
          inputport = inport1;
         }
      else if (port == 2) 
              {
               outputport = outport2;
               inputport = inport2;
              }
      else {
            outputport = outport3;
            inputport = inport3;
           }
   rcode = port_ok();
   } while ((rcode != 1) && (port < 3));
}


/*************************************************************
*                                                            *
*                          POWER OK                          *
*                                                            *
*  This procedure determines if the ispDOWNLOAD cable has    *
*  power.                                                    *
*                                                            *
*************************************************************/

int             power_ok()
{
   int             rcode,
                   oldrcode;

   outp(outputport, NUL);
   oldrcode = (in_VCC_OK & inp(inputport));
   pulse_width(5);
   rcode = (in_VCC_OK & inp(inputport));
   if ((oldrcode != rcode) || (rcode == 0) || (oldrcode == 0))
      rcode = POWER_OFF;
   return (rcode);
}


/*************************************************************
*                                                            *
*                          PORT OK                           *
*                                                            *
*  This procedure determines if the ispDOWNLOAD cable is     *
*  connected to specified parallel port.                     *
*                                                            *
*************************************************************/

int             port_ok()
{
   int             d1,
                   d2;
   int             rcode;
   outp(outputport, isp_pins + out_SENSE_CABLE_OUT);
   d1 = (in_CABLE_SENSE_IN & inp(inputport));
   outp(outputport, isp_pins);
   d2 = (in_CABLE_SENSE_IN & inp(inputport));


   if ((d2 != d1) || (NUL > 0))
      rcode = 1;
   else
      rcode = CABLE_NOT_FOUND;
   return (rcode);
}

/*************************************************************
*                                                            *
*                         PULSE_WIDTH                        *
*                                                            *
*************************************************************/


void            pulse_width(unsigned int milliseconds)
{
   static unsigned int ticks_per_ms = 0;
   static int      mode2 = false;
   static long     pre_ticks = 0L;
   unsigned int    start,
                   end;
   long            ticks,
                   cur_ticks;
   static int      count = 0;


   cur_ticks = 0L;
   outp(0x43, 0x00);
   start = inp(0x40);
   start += inp(0x40) << 8;
   if (ticks_per_ms == 0) {
      ticks_per_ms = (0xFFFF / 55) * 2;
      if (start & 0x0001) {
         ticks_per_ms = 0xFFFF / 55;
         mode2 = true;
      }
   }
   ticks = (long) ticks_per_ms *milliseconds;
   while (ticks > 0L) {
      if (pre_ticks == 0L)
         pre_ticks = cur_ticks;
      else if (count < 5) {                      /* calibration count *//* keep
                                                  * the most conservative ticks
                                                  * per loop as calibrated
                                                  * value */
         if (pre_ticks > cur_ticks)
            pre_ticks = cur_ticks;
         count++;
      }
      outp(0x43, 0x00);
      end = inp(0x40);
      end += inp(0x40) << 8;
      if ((!mode2) && (end & 0x0001)) {
         ticks_per_ms = 0xFFFF / 55;
         ticks -= (long) ticks_per_ms *milliseconds;
         mode2 = true;
         if (ticks <= 0L)
            break;                               /* done already */
      }
      cur_ticks = (long) (start - end);          /* number of ticks elapsed */
      if (cur_ticks == 0L) {                     /* this can happen only when
                                                  * in Window DOS */
         cur_ticks = pre_ticks;
         ticks -= cur_ticks;
      } else if (cur_ticks > 0L)
         ticks -= cur_ticks;
      else {
         cur_ticks = (long) (0xFFFF - end + start);
         ticks -= cur_ticks;
      }
      /* safe guard against stolen ticks in Window DOS */
      if ((pre_ticks > 0L) && ((cur_ticks - pre_ticks) > (ticks_per_ms / 4))) {
         ticks += (cur_ticks - pre_ticks);
         cur_ticks = pre_ticks;
      }
      start = end;
   }
}



/*************************************************************
*                                                            *
*                       SHIFT/EXECUTE                        *
*                                                            *
*************************************************************/
void            execute()
{
   isp_setpin(out_MODE, HIGH);
   isp_setpin(out_SDI,  HIGH);
   isp_setpin(out_SCLK, HIGH);
   isp_setpin(out_SCLK, LOW);
   isp_setpin(out_SDI,  LOW);
   isp_setpin(out_MODE, LOW);
}


/*************************************************************
*                                                            *
*                      MOVE TO ID STATE                      *
*                                                            *
*************************************************************/

void            move_to_id_state()
{
   isp_setpin(out_MODE, HIGH);                   /* MOVE DEVICE TO ID STATE */
   pulse_width(5);                              /* allow 5 mS delay for 1st
                                                  * clock */
   isp_setpin(out_SCLK, HIGH);
   isp_setpin(out_SCLK, LOW);
   isp_setpin(out_MODE, LOW);
}



/*************************************************************
*                                                            *
*                     PROGRAM_UES                            *
*                                                            *
*  ues-      contain the ues string read from ispLSI devices *
*            under operation "PU" in the .dld file with fill *
*            bits if any removed.                            *
*            user can modify the content of ues.             *
*  ues_mask- bit 1 represent a valid ues bit. user shall not *
*            alter its content.                              *
*  buff-     contain the ues string with fill bits if needed.*
*            user shall not alter its content                *
*  This function is to program the UES at *ues in place of   *
*  the UES embedded in the ispSTREAM file or JEDEC file.     *
*  Example: The .dld file is as follows:                     *
*           1032  PU 1032.jed                                *
*           3256  PV 3256.jed                                *
*           1032  PU 1032_1.jed
*                                                            *
*           The exisiting UES in the ispLSI 1032 devices are *
*           read and store in *ues. The 1032.jed files may or*
*           may not have UES, however it will be ignored.    *
*           Users may alter the content in *ues prior to     *
*           re-program the UES into the ispLSI 1032 devices. *
*           Please note that the string length of *ues is the*
*           maximum length of the UES. Excess UES will be    *
*           truncated. The ues string length is the sum of   *
*           the UES length of devices with operation "PU".   *
*           So the example above has ues string length of    *
*           160+160=320 bits or 40 ASCII characters long.    *
*************************************************************/
int            program_ues(unsigned char *ues, unsigned char *ues_mask,
                                         unsigned char *buff, int data_length)
{
   int             i,
                   pos,
                   col,
                   in_pos,
                   in_col,
                   ues_length;
   unsigned char   curch,
                   xch;

   pos = in_pos = 0;
   col = in_col = 8;
   ues_length = 0;
   curch = xch = LOW;
   /*This is the location to insert your own UES operation codes*/
   /*example: */
   for (i = 0; i < data_length; i++)
       if (( ues_mask[i/8] << i%8) & 0x80) ues_length++;
   printf("\nThe UES string in HEX:\n");
   for (i = 0; i < ues_length/8; i++)
       printf("%02x",ues[i]);
   printf("\nThe UES string in ASCII:\n");
   for (i = 0; i < ues_length/8; i++)
       {if (isprint(ues[i])) printf("%c",ues[i]);
        else printf(".");
       }
   printf("\n");
               /*3.04*/
               /*******************************************************/
               /*If you use "PU" function and want to use the UES in  */
               /*the JEDEC files, insert the code "return (1); " here */
               /*to bypass the rest of the codes of program_ues.      */
               /*******************************************************/
   for (i = 0; i < data_length; i++) {
      --col;
      if (col < 0) {
         col = 7;
         pos++;
      }
      if ((ues_mask[pos] >> col) & 0x01) {
         --in_col;
         if (in_col < 0) {
            in_col = 7;
            in_pos++;
         }
         if ((ues[in_pos] >> in_col) & 0x01) {
            curch = HIGH;
            buff[pos] |= 0x01 << col;
         } else {
            curch = LOW;
            buff[pos] &= ~(0x01 << col);
         }
      } else {
         buff[pos] |= 0x01 << col;
         curch = HIGH;
      }
      if (xch != curch) {
         isp_setpin(out_SDI,curch);
         xch = curch;
      }
      isp_setpin(out_SCLK, HIGH);          /* shift the ues bits in */
      isp_setpin(out_SCLK,  LOW);
   }
   execute();
 return (OK);
}


/*************************************************************
*                                                            *
*                     ISPSTREAM_PUMP                         *
*                                                            *
*************************************************************/

int   ispstream_pump(char *filename, int verify_only, int *end)
{
   unsigned int    n,
                   error,
                   length,
                   index,
                   width;
   int             rcode,
                   i,
                   j,
                   out_pos,
                   out_col,
                   in_col;
   unsigned char  *data,
                  *buf,
                  *buff,
                   curch,
                   sdo,
                   cur_bit,
                   uch;
   char            do_shift_instruction,
                   condition,
                   shiftstate;
   char            do_program,
                   do_erase,
                   do_shift_in_data,
                   do_shift_out_data,
                   do_store_data;
   char            do_verify_delay,
                   do_ues,
                   done_shift_in_data,
                   xch,
                   ych;
   unsigned int    bit_1_count;
   register int    row;
   int             last_row,
                   erase_pulse,
                   program_pulse,
                   devices_in_chain;
   unsigned int    maxi_data,
                   data_length,
                   ues_length;
   int             last_be_row;
   FILE           *fp;
   unsigned long   checksum;
   unsigned char   var[2];
   unsigned char  *ues = NULL,
                  *ues_mask = NULL;

   do_ues = false;                             /*Not "PU" operation by default*/
   checksum = 0;                               /*Keep track of ispSTREAM checksum*/
   if ((fp = fopen(filename, "rb")) == NULL)   /*Look for ispSTREAM file */ 

      return FILE_NOT_FOUND;
   /*Start reading the header of the ispSTREAM file*/
   if ((xch = fgetc(fp)) != 0x0F)             /*Byte 0 of the header of the ispSTREAM*/
      return FILE_NOT_JEDEC;                  /*file is the file type*/
   checksum += xch;
   devices_in_chain = fgetc(fp);              /*Byte 1 of ispSTREAM is number of*/
   checksum += devices_in_chain;              /*devices in the given daisychain.  */
   var[1] = fgetc(fp);                        /*Byte 2 and 3 is the width of the */
   var[0] = fgetc(fp);                        /*of the erase pulse in miniseconds.*/
   checksum += var[1];                        /*The erase instruction is ignored*/
   checksum += var[0];                        /*if erase pulse width is 0.*/
   erase_pulse = var[1];
   erase_pulse = erase_pulse * 0x100 + var[0];
   program_pulse = fgetc(fp);                 /*Byte 4 contains the width of the*/
   checksum += program_pulse;                 /*programming pulse in miniseconds.*/
                                              /*The program instruction is ignored*/
                                              /*if program pulse width is 0.*/
   /*3.04*/
   /************************************************************/
   /* If you want to perform verification "V" only with an     */
   /* ispSTREAM file built with programming and verification   */
   /* "PV", or verify UES then program and verify "PU", you can*/
   /* call ispstream_pump by setting verify_only to true to    */
   /* set:                                                     */
   /*         program_pulse = 0;                               */
   /*         erase_pulse = 0;                                 */
   /* It will override all the bulk_erase and programming      */
   /* instructions embedded in the ispSTREAM file.             */
   /************************************************************/
   if (verify_only) {program_pulse = 0;
                     erase_pulse = 0;}
   
   var[1] = fgetc(fp);                        /*Byte 5 and 6 contains the number of*/
   var[0] = fgetc(fp);                        /*data rows in the ispSTREAM file.*/
   checksum += var[1];                        /*It is to flag where is the end of*/
   checksum += var[0];                        /*the ispSTREAM file.*/
   last_row = var[1];
   last_row = last_row * 0x100 + var[0];
   var[1] = fgetc(fp);                        /*Byte 7 and 8 is the longest data row*/
   var[0] = fgetc(fp);                        /*in the ispSTREAM file in bits. */
   checksum += var[1];                        /*It is for allocating enough memory*/
   checksum += var[0];                        /*to process any data row in the */
   maxi_data = var[1];                        /*ispSTREAM file.*/
   maxi_data = maxi_data * 0x100 + var[0];   
   var[1] = fgetc(fp);                        /*Byte 9 and 10 is the end of the */
   var[0] = fgetc(fp);                        /*post Bulk Erase Verify data row */
   checksum += var[1];                        /*in the ispSTREAM file.*/
   checksum += var[0];                        /*It is for skipping post Bulk Erase*/
   last_be_row = var[1];                      /*verify if erase operation is ignored.*/
   last_be_row = last_be_row * 0x100 + var[0];
   /*End of the header of the ispSTREAM file.*/   
   
   xch = NUL;                                 /*out_SDI = 0 by default.*/
   rcode = OK;                                /*Success by default.*/
   
   /*Allocate memory to store row data of the ispSTREAM file.*/
   if ((buf = (unsigned char *) (malloc(maxi_data / 8 + 1))) == NULL) {
      free(buf);
      return ABORT;
   }

   /*Allocate alternative memory to store row data of the ispSTREAM file.*/
   if ((buff = (unsigned char *) (malloc(maxi_data / 8 + 1))) == NULL) {
      free(buf);
      free(buff);
      return ABORT;
   }
   /*Allocate memory to store the UES mask in the ispSTREAM file assuming 
     the "PU" operation is used.*/
   if ((ues_mask = (unsigned char *) (malloc(maxi_data / 8 + 1))) == NULL) {
      free(buf);
      free(buff);
      return ABORT;
   }
   /*Allocate memory to store the UES data read from the devices assuming 
     the "PU" operation is used.*/
   if ((ues = (unsigned char *) (malloc(maxi_data / 8 + 2))) == NULL) {
      free(buf);
      free(buff);
      free(ues_mask);
      return ABORT;
   }
   move_to_id_state();                           /* 3.04 Set devices to idle state.*/
   execute();                                    /* 3.04 set devices to shift state.*/
   move_to_id_state();                           /* Set all devices to idle state */
   shiftstate = true;                            /* IDs are ready to shift out*/
   done_shift_in_data = false;                   /* Flag to turn on simultaneous*/
                                                 /* in/out data shifting.*/
   error = 0;                                    /* Verify failure.*/
   length = 0;                                   /* Number of bits of data to shift out*/

/*Start reading the body of the ispSTREAM file.*/
   curch = fgetc(fp);                            /* Byte 0 of the body of the ispSTREAM file.*/
   checksum += curch;
   in_col = 8;                                   /* The first bit to shift out from byte 0.*/
   /*Start processing all the data rows in the ispSTREAM file.*/
   for (row = 0; row <= last_row; row++) {
      *end = row;                                /* The row number where process ended.*/
      condition = 0;                             /* 13 possible cases in the ispSTREAM file.*/
      do_program = false;                        /* Flag to execute the program instruction.*/
      do_erase = false;                          /* Flag to execute the erase instruction.*/
      do_shift_instruction = false;              /* Flag for the current data row is instructions.*/
      do_shift_in_data = false;                  /* Flag for the current data row is JEDEC data.*/
      do_shift_out_data = false;                 /* Flag to shift out and verify data from devices.*/ 
      do_store_data = false;                     /* Flag to skip the current data row.*/
      do_verify_delay = false;                   /* Flag to execute the verify instructions.*/

      /* The 1st 16 bits of data in the ispSTREAM file is the unsigned integer for data length
         of the current data row of the ispSTREAM file.*/
      for (i = 1; i >= 0; i--) {                 /* Read 16 bits of data from the ispSTREAM to*/
         for (j = 7; j >= 0; j--) {              /* form the unsigned integer value which is */
            --in_col;                            /* the size of the current data row in bits.*/
            if (in_col < 0) {
               in_col = 7;                       /* Exhausted all 8 bits from the previous byte, */
               curch = fgetc(fp);                /* fetch the next byte then continue.           */
               checksum += curch;
            }
            if ((curch >> in_col) & 0x01)
               var[i] |= 0x01 << j;
            else
               var[i] &= ~(0x01 << j);
         }
      }
      data_length = var[1];                      
      data_length = data_length * 0x100 + var[0];
      /*Data_length is the size of the current data row.*/  

      width = data_length;  /*The number of bits to be fetched from the ispSTREAM file.*/
      printf("Operating On Row %04d", row);
      uch = 13;
      printf("%c", uch);
      data = buf;                                /* Default storage space for trashing data. */
      if (row < 9) {                             /* Verify IDs and UES */
         if (data_length == 0)                   /* If the size of the current data row is 0, */
            condition = 9;                       /* then jump to the next row.*/
         else
            condition = row;
         switch (condition) {
            case 0:                              /* Data row 0 of the ispSTREAM file contains*/
                                                 /* 8 bits ID of all the devices in the chain.*/
               data = buff;                      /* Storage space for the 8-bit IDs.*/
               do_store_data = true;             /* Set the flag to store 8-bit IDs in ispSTREAM.*/
               do_shift_out_data = true;         /* Set the flag to shift 8-bit IDs out to */
                                                 /* compare with the stored 8-bit IDs data.*/
               shiftstate = false;               /* Skip the execute command.*/
               break;
            case 1:                              /* Data row 1 contains VERIFY_UES instructions.*/
            case 5:                              /* Data row 5 contains VERIFY_UES instructions */
                                                 /* for "PU" operation.*/
               do_shift_instruction = true;      /* Set the flag to shift the instructions into.*/
                                                 /* the instruction shift register of the devices.*/
               do_verify_delay = true;           /* Set the flag to wait the verify timing.*/           
               break;
            case 2:                              /* Data row 2 contains the data shift instructions.*/
            case 6:                              /* Data row 6 contains the data shift instructions*/
                                                 /* for "PU" operation. */
               do_shift_instruction = true;      /* Set the flag to shift the instructions in.*/
               break;
            case 3:                              /* Data row 3 contains the PES mask.*/
               data = buff;                      /* Use buff to store the PES mask.*/
               do_store_data = true;             /* Set the flag to store PES mask in the*/
               break;                            /* ispSTREAM file.*/
            case 4:                              /* Data row 4 contains the PES data.*/
               do_store_data = true;             /* Set the flag to store PES data in buf.*/
               do_shift_out_data = true;         /* Set the flag to shift PES out from devices*/
                                                 /* and compare with stored PES data.*/
               break;
            case 7:                              /* Data row 7 is not zero means "PU" operation.*/
               do_ues = true;                    /* Operation "PU" is detected. */
               data = ues_mask;                  /* Switch to ues mask storage */
                                                 /* Clear the UES string and UES mask.*/
               for (i = 0; i < (int) (maxi_data / 8 + 1); i++)
                  ues_mask[i] = ues[i] = 0x00;   
               do_shift_out_data = true;         /* Shift the UES data out from devices.*/
               do_store_data = true;             /* Save the UES data read from devices.*/
               i = 0;                            /* First byte of the UES string */
               j = 8;                            /* First bit of the UES string */
               break;
            case 8:
               /*******************************************************/
               /* The UES data read from the devices with the "PU"    */
               /* function is ready here. You can insert here the     */
               /* codes as follows to display the UES string in HEX   */
               /* and in ASCII format.                                */
               /*   ues_length = 0;                                   */
               /*   for (i = 0; i < maxi_data; i++)                   */
               /*   if (( ues_mask[i/8] << i%8) & 0x80) ues_length++; */
               /*   printf("\nThe UES string in HEX:\n");             */
               /*   for (i = 0; i < ues_length/8; i++)                */
               /*   printf("%02x",ues[i]);                            */
               /*   printf("\nThe UES string in ASCII:\n");           */
               /*   for (i = 0; i < ues_length/8; i++)                */
               /*       {if (isprint(ues[i])) printf("%c",ues[i]);    */
               /*        else printf(".");                            */
               /*       }                                             */
               /*   printf("\n");                                     */
               /*******************************************************/

               /*******************************************************/
               /* If want to quit after reading UES from devices      */
               /* insert the statement "goto finish;" here.           */
               /*                                                     */
               /*******************************************************/
               move_to_id_state();                            /*Reset the devices.*/
               execute();                                     /*Set devices to shift state.*/
               shiftstate = true;
               if (erase_pulse > 0)  {                        /*Erase devices or not?.*/
                  do_erase = true;                            /*Execute the erase instructions.*/
                  do_shift_instruction = true;                /*The current row data is instructions.*/
               } else
                  do_store_data = true;                       /*Trash and void the instructions.*/ 
               break;
            case 9:                                           /*The current row data is blank.*/
            default:
               break;
         }
      } else {                                   /*Program and Verify JEDEC patterns and UES.*/
         if (data_length == 0)                   /*Current row data is blank or not action row.*/
            condition = 13;                    
         else if (row == last_row)
            condition = 12;                      /*The last row is always for security fuse. */

         /*If the ispSTREAM file was built with "PV" operation and you want to force it to
           perform "V" operation only by setting erase pulse width to 0, then
           all the relevant instructions to perform the post Bulk Erase verify
           must be trashed and voided.*/
         /*If the ispSTREAM file was built with "V" operation then last_be_row would be set
           to 9 to skip this section of the code.*/

         else if ((erase_pulse == 0) && (row >= 9) && (row < last_be_row)) {
            condition = (row - 9) % 12;          
            if ((condition == 3) || (condition == 8)) {
            } else
               do_store_data = true;            /*Trash and void all post Bulk Erase verify*/ 
            condition = 13;                     /*instructions in the ispSTREAM file.    */
         } 
         
        /*Branch to here if no "PV" operation override or ready to start the process
          exactly per the ispSTREAM file.*/          
         else
            condition = (row - 9) % 12;         /*From row 9 till the last_row, the ispSTREAM
                                                  file is constructed in many loops of 12 
                                                  cases per loop.*/
                                                  
         switch (condition) {                
            case 0:                             /* Current row of ispSTREAM has address shift 
                                                   instructions.*/
            case 2:                             /* Current row has data shift instructions.*/ 
            case 7:                             /* Current row has data shift instructions.*/
               do_shift_instruction = true;     /* Set the flag to shift in and execute the*/
               break;                           /* data shift instructions.*/
            case 1:                             /* Current row has the address data.*/
               do_shift_in_data = true;         /* Set the flag to shift in the address data*/ 
               break;                           /* into the address registers of the devices.*/
            case 3:                             /* Current row has the High Order JEDEC data.*/
            case 8:                             /* Current row has the Low Order JEDEC data.*/
               if (row > last_be_row)           /* In post Bulk Erase verify?*/
                  do_shift_in_data = true;      /* No! Set the flag to shift data into the 
                                                   data shift registers for program and verify
                                                   or just verify.*/
               else                             /* Yes! Build the all '1's data stream for 
                                                   comparison. All '1' data is not packed into
                                                   the ispSTREAM for minimum ispSTREAM file size.*/
                  for (i = 0; i < (int) (maxi_data / 8 + 1); i++)
                     buff[i] = 0xFF;             
               if (done_shift_in_data)          /* If the verify instruction has been executed,*/
                  do_shift_out_data = true;     /* then shift the data out from devices for*/
                                                /* verification.*/
               done_shift_in_data = true;       /* Set the flag to show that data registers of
                                                   devices already have data shifted in. So 
                                                   verify can be done when encounter the next
                                                   data shift instructions.*/
               bit_1_count = 0;                 /* Clear the '1's counter. If the entire JEDEC
                                                   data row is '1's then the program instruction 
                                                   for this row will be voided.*/
               length = width;                  /* Remember the number of JEDEC data bits 
                                                   shifted into the data shift registers of 
                                                   devices. The same number of bits must be
                                                   shifted out for verification.*/ 
               data = buff;                     /* Store the JEDEC data in the ispSTREAM file
                                                   to the designated memory locations. It will
                                                   be used to compare with the data to be 
                                                   shifted out from devices during verify.*/
               /*Special code for "PU" operation support.*/       
               if ((do_ues) && (row == last_row - 4))    /*Is current row for UES?*/
                  {/*Yes! Program into devices the UES data user provided.*/
                   if (program_ues(ues, ues_mask, buff, data_length)==OK) /*3.04*/
                      {shiftstate = true;
                       do_shift_in_data = false;         /*Don't shift UES in ispSTREAM
                                                           into devices.*/
                       do_store_data = true;             /*Trash the UES data in ispSTREAM.*/
                       data = buf;
                      }                                   
                  }                                
               break;
            case 4:                              /* Current row of ispSTREAM has program High 
                                                    Order Data instructions.*/
            case 9:                              /* Current row of ispSTREAM has program Low 
                                                    Order Data instructions.*/

               do_shift_instruction = true;      /* Set the flag to shift instructions into devices.*/
               if (((program_pulse == 0) || ((bit_1_count == length) /*Void the program instructions?*/
                         && (bit_1_count > 0))) || (length == 0)) {
                  do_store_data = true;          /* Yes! Trash the instructions in ispSTREAM.*/
                  do_shift_instruction = false;  /* Don't shift the instructions into devices.*/
                  break;
               } else if (program_pulse > 0)     /* Yes! But is the "PV" operation overrided?*/
                  do_program = true;             /* Go ahead to carry out the programming
                                                    instructions in the ispSTREAM file.*/
               break;
            case 5:                              /* Current row is the instructions to verify
                                                    the High Order Data.*/
            case 10:                             /* Current row is the instructions to verify
                                                    the Low Order Data.*/
               do_shift_instruction = true;      /* Set the flag to shift the verify instructions
                                                    into the devices.*/
               do_verify_delay = true;           /* Set the flag to wait for the verify delay 
                                                    after the verify instructions are shifted
                                                    into the devices and executed.*/ 
               break;
            case 6:                              /* Current row is the data shift instructions
                                                    for High Order Data.*/
            case 11:                             /* Current row is the data shift instructions
                                                    for Low Order Data.*/
               done_shift_in_data = false;       /* The data shift instructions are for shifting
                                                    data out from devices only.*/
               width = length;                   /* The number of data bits to be shifted out
                                                    from devices is the same as what has 
                                                    already been shifted into the devices 
                                                    previously at case 3 or 8. */
               do_shift_instruction = true;      /* Set the flag to shift the instructions to
                                                    devices.*/
               do_shift_out_data = true;         /* Set the flag to shift data out from devices.
                                                    after the data shift instructions were
                                                    shifted into the devices and executed.*/
               break;
            case 12:                             /* Current row is to support the security
                                                    feature of ispLSI devices.*/
               do_shift_instruction = true;      /* Set the flag to shift the program security
                                                    instructions into devices.*/ 
               if (program_pulse > 0)            /* If the "PV" operation is to be carried out,
                                                    then execute the instruction and wait for
                                                    the programming pulse width.*/
                  do_program = true;   
               break;
            case 13:                             /* Current row has no data and so no action 
                                                    is necessary. */
               break;
            default:                             
               break;
         }
      }


/* Start operating with the devices base on the flags set above.*/     

      /*OPTIONS: Read from the ispSTREAM file bit by bit the embedded instructions
                 and shift them into the Instruction Registers of the devices.
                 Read from the ispSTREAM file bit by bit the embedded instructions
                 and put them in the trash to void the instructions.
                 Read from the ispSTREAM file bit by bit the UES data and trash
                 them.
      */
      if ((do_shift_instruction) || (do_store_data)) { /*Shift in the instructions or trash them?*/
         out_pos = 0;                                  /*Byte pointer of the memory to store data.*/ 
         out_col = 8;                                  /*Bit pointer of the memory to store data.*/
         for (index = 0; index < data_length; index++) { 
            --in_col;                                  /*Bit pointer of the ispSTREAM file.*/
            --out_col;
            if (out_col < 0) {                         /*Complete storage of a Byte?*/  
               out_pos++;                              /*Yes! Update the bype pointer to point to 
                                                         the next empty byte for storage.*/
               out_col = 7;                            /*Reset the bit pointer to the most significant
                                                         bit of the empty byte.*/
            }
            if (in_col < 0) {                          /*Complete a Byte from the ispSTREAM?*/
               curch = fgetc(fp);                      /*Yes! Fetch a new Byte from the ispSTREAM.*/
               in_col = 7;                             /*Reset the bit pointer to the most significant
                                                         bit of the new Byte.*/
               checksum += curch;
            }
            if ((curch >> in_col) & 0x01)              /*The current bit is a 1.*/
               xch = HIGH;                             /*Shift a bit of 1 into the instruction 
                                                         registers of devices.*/
            else                                       /*The current bit is a 0.*/
               xch = LOW;                              /*Shift a bit of 0 into the instruction 
                                                         registers of devices.*/
            if (do_store_data) {                       /*Store the current bit.*/
                if (xch)  data[out_pos] |= 0x01 << out_col;
                else data[out_pos] &= ~(0x01 << out_col);
               }
             else {                                    /*Shift the current bit of the instructions 
                                                         to devices.*/
                 isp_setpin(out_SDI, xch);
                 isp_setpin(out_SCLK, HIGH);                   
                 isp_setpin(out_SCLK, LOW);
                }
         }
      }    /* End of shifting instructions or store data.*/
      
      if ((do_shift_instruction) && (shiftstate)) { /* Was instructions shifted into devices?*/
         execute();                                 /* Yes! Set devices to execute state to
                                                       execute the instructions.*/
         shiftstate = false;                        /* Devices in execute state.*/
      }

      /* If the data shift instructions were shifted into devices and executed
         above, then perform data shift in, shift out or simultaneous shift in and out.*/
      
      /*OPTIONS: Read and save from the ispSTREAM file bit by bit the packed ARRAY
                 and UES programming data and shift them into the Data Registers of the 
                 devices.
                 Shift the data in the Data Registers of the devices out and compare
                 them with the data either was saved in the previously stored data
                 operation.
                 Shift the UES data out from devices and save them in the dedicated
                 storage space ues and present it to users.
      */
      if ((do_shift_in_data) || (do_shift_out_data)) {
         ych = 0;
         out_pos = 0;                               /* Byte pointer points to the 1st byte
                                                       of the previously stored data.*/
         out_col = 8;                               /* Bit pointer points to the most significant
                                                       bit of the 1st byte of the stored data.*/
         uch = buff[out_pos];                       /* Fetch the first byte of the stored data.*/
         for (index = 0; index < width; index++) {
            --out_col;
            if (out_col < 0) {                      /* Complete a Byte of stored data?*/
               out_pos++;                           /* Yes! Move the Byte pointer to the next byte.*/
               out_col = 7;                         /* Reset the bit pointer to the most significant
                                                       bit of the next byte.*/
               uch = buff[out_pos];                 /* Fetch the next byte of stored data.*/
            }
            if (do_shift_in_data) {                 /* Shift data bit by bit into devices.*/
               --in_col;
               if (in_col < 0) {                    /* Complete a Byte of ispSTREAM?*/ 
                  curch = fgetc(fp);                /* Yes! Fetch a new Byte.*/
                  in_col = 7;                       /* Reset the bit pointer to the most significant
                                                       bit.*/
                  checksum += curch;
               }
               if ((curch >> in_col) & 0x01) {            /* The current bit a 1 ?*/
                  data[out_pos] |= 0x01 << out_col;       /* Store a bit of 1 in the memory. */
                  xch = HIGH;                             /* Shift a 1 into the Data Registers
                                                             of devices.*/
                  bit_1_count++;                          /* Keep track of the number of 1s 
                                                             shifted into Data Registers of 
                                                             devices. If all the data are
                                                             1s for the current row, the 
                                                             programming instructions
                                                             will be trashed and voided to
                                                             minimize time.*/ 
               } else {                                   /* The current bit is a 0.*/
                  data[out_pos] &= ~(0x01 << out_col);    /* Store a bit of 0 in the memory.*/   
                  xch = LOW;                              /* Shift a 0 into the Data Registers
                                                             of devices.*/
               }
            }
            if (do_shift_out_data) {                      /* Shift data out bit by bit from devices.*/
               if ((uch >> out_col) & 0x01)     
                  cur_bit = HIGH;                         /* A bit of 1 is expected from devices.*/
               else
                  cur_bit = LOW;                          /* A bit of 0 is expected from devices.*/
               sdo = isp_SDO();                           /* Read a bit of data from the SDO pin.*/
               /* Read and Store UES  */
               if ((do_ues) && (row == 7)) {                     /* "PU" operation?*/
                  if ((ues_mask[out_pos] >> out_col) & 0x01) {   /* Yes! Scan the UES mask.*/
                     --j;                         
                     if (j < 0) {                                
                        j = 7;
                        i++;
                     }                          
                     if (sdo)                                    /* Store the UES bit by bit*/
                        ues[i] |= 0x01 << j;                     /* to the designated memory.*/
                  }
                  cur_bit = sdo;                                 /* Read and store only, so
                                                                    comparision is voided.*/
               }
               /*Check PES of devices.*/
               if ((row == 4) && ((buf[out_pos] >> out_col) & 0x01)) 
                  if (error == 0)
                     ych = 1;
               if ((row == 4) && (!((buf[out_pos] >> out_col) & 0x01))) {
                  if (ych) {
                     error++;
                     rcode = NON_ISP_PART;
                     break;
                  } else
                     error = 0;                 
               }
               
               /* Bit by bit data comparsion between expected data and the read data
                  is done here.*/
               /* sdo is the bit of data read from devices.
                  cur_bit is the bit of expected data.
               */
                else if (sdo != cur_bit) {     /* Data read does not match with 
                                                  data expected.*/
                  ych = 0;
                  if (row == 0 )                 /* If fail at ID rows then flag fail
                                                  ID check.*/  
                     rcode = UNKNOWN_CHIP;
                  else
                     rcode = VALIDATION_ERROR;   /* Program and verify fail.*/ 
                  error++;
               }
            }                                    
            /* The rising edge of the clock shifts a bit of data into devices.
               The falling edge of the clock shifts out a bit of data from 
               devices.*/
            isp_setpin(out_SDI, xch);             /* Drive the SDI pin.*/
            isp_setpin(out_SCLK, HIGH);           /* Rising edge of clock.*/
            isp_setpin(out_SCLK, LOW);            /* Falling edge of clock.*/
         }
      }  /*End of shifting in and/or out of data from Data Shift Registers
           of devices.*/

      /*Start Erase or Program operations if the corresponding flags are set.*/
      /*OPTIONS: Start the erase timing by pulsing the clock, wait for the 
                 erase delay then terminate erase timing with the proper 
                 extremely conservative setup time and set the devices to
                 shift state.
                 Start the program timing by pulsing the clock, wait for
                 the program delay then terminate timing and set the devices
                 back to the shift state ready for next instructions.
                 Start the verify timing by pulsing the clock, wait for
                 the 30uS verify delay time then terminate the verify 
                 instruction and set the devices to shift state.
      */
      if ((do_erase) || (do_program)) {          
         if (do_erase) {
            isp_setpin(out_SCLK, HIGH);
            isp_setpin(out_SCLK, LOW);
            pulse_width(erase_pulse);
            isp_setpin(out_MODE, HIGH);
            isp_setpin(out_SDI,  HIGH);
            pulse_width(1);                      /* 1ms delay from mode to
                                                  * clock */
            isp_setpin(out_SCLK, HIGH);
            isp_setpin(out_SCLK, LOW);
            isp_setpin(out_SDI,  LOW);
            isp_setpin(out_MODE, LOW);

            /* to reset the security cell of 22v10 */
            isp_setpin(out_MODE, HIGH);    /* MOVE DEVICE TO ID STATE */
            isp_setpin(out_SCLK, HIGH);
            isp_setpin(out_SCLK, LOW);
            isp_setpin(out_MODE, LOW);
            execute();                           /* back into isp mode */
            shiftstate = true;                   /* chain back in shift state */
         } else if (do_program) {
            isp_setpin(out_SCLK, HIGH);
            isp_setpin(out_SCLK, LOW);
            pulse_width(program_pulse);
            execute();
            shiftstate = true;                   /* chain back in shift state */
         }
      } else if ((!shiftstate) && ((do_shift_in_data) || (do_shift_out_data)
                                   || (do_verify_delay))) {
         if (do_verify_delay) {
            isp_setpin(out_SCLK, HIGH);
            for (i = 0; i < 30; i++)             /* simulate a 30uS min delay */
                 isp_setpin(out_SCLK, LOW);
         }
         execute();
         shiftstate = true;                      /* chain in shift state */
      }                                          
      if (error > 0) {                           /* Terminate operation if 
                                                    failure occured.*/
                     break;
                     }
   } /*End of for (row = 0; row <= last_row; row++) */

   /*Fetch from the ispSTREAM file the 16 bit ispSTREAM file checksum and
     compare it with the calculated checksum while reading the ispSTREAM
     file to ensure that there was no data lost.*/
   in_col = 0;
   for (i = 1; i >= 0; i--) {
      for (j = 7; j >= 0; j--) {
         --in_col;
         if (in_col < 0) {
            in_col = 7;
            curch = fgetc(fp);
         }
         if ((curch >> in_col) & 0x01)
            var[i] |= 0x01 << j;
         else
            var[i] &= ~(0x01 << j);
      }
   }
   n = (unsigned int) checksum & 0xFFFF;
   data_length = var[1];
   data_length = data_length * 0x100 + var[0];
   if ((error == 0) && (n != data_length)) {
      error++;
      rcode = FILE_ERROR;
   }  /* End of ispSTREAM file checksum comparison.*/
finish:
   move_to_id_state();
   if (buf != NULL)
      free(buf);
   if (buff != NULL)
      free(buff);
   if (ues != NULL)
      free(ues);
   if (ues_mask != NULL)
      free(ues_mask);
   if (error > 0) {
      return (rcode);
   } else 
   return (OK);

}


/*************************************************************
*                                                            *
*                         error_handler                      *
*                                                            *
*  rcode                  - error code                       *
*                                                            *
*                                                            *
*  This procedure return the address of the message string   *
*  for the corresponding error code.                         *
*                                                            *
*************************************************************/

void error_handler(int rcode, char *message)
{

/*added by ht for error handling*/
char *error_message[] = {{"PASS"},{""},{""},{"PC Hardware Problem"},{"Chip and JEDEC File Doesn't Match"},
                   {"Can't Find the Download Cable"},{"The VCC on Board Is Not On"},
                   {"No Device Found/Open Download Cable/Open Daisy Chain"},
                   {"Can't Find the File"},{"The File Is Not Complete"},{"Not Enough PC Memory"},
                   {"Verification Fail"},{"The Device is Secured"},{"Non isp Part Found"},
                   {"The File Contain Operation Not Supported in This Mode"},
                   {"The Chip Count in the Chain and the File Do Not Match"},
                   {"Unknown Device Found"},{"Wrong File Type"},{"File Error"},
                   {"The File Has Invalid UES Characters"}};

 strcpy(message, error_message[-rcode]);
} 



/*************************************************************
*                                                            *
*                            MAIN                            *
*                                                            *
*************************************************************/

int             main(int argc, char *argv[])
{
   int             rcode = OK,
                   end = 0;
   FILE           *fpr;
   char           *ptr,
                   filename[300],
                   str[300];


   printf("\n           Lattice Semiconductor Corp.\n");
   printf("\n        ispCODE V3.04 Copyright 1995,1996\n");
   printf("\n              ispSTREAM Driver\n\n");

   if ((argc < 2) || (argc > 2)) {
      printf("\nUsage: turbo [drive:][path]isp_filename\n");
      printf("Example: turbo my_file.isp\n");
      printf("\n");
      printf("isp_filename         The ispSTREAM File Created From A Valid \n");
      printf("                     Daisy Chain Configuration File\n");
      exit(1);
   }
   if ((fpr = fopen(argv[1], "r")) == NULL) {
      printf("\n%s File not found!\n", argv[1]);
      exit(1);
   }
   fclose(fpr);
   strcpy(str, argv[1]);
   ptr = strchr(str, '.');
   if (ptr)
      *ptr = '\0';
   sprintf(filename, "%s.isp", str);
   strcpy(str, argv[1]);
   if (stricmp(str, filename)) {
      printf("\nFile %s Is Not An ispSTREAM File\n", str);
      exit(1);
   }
   printf("\nStart Program and Verify.......\n");

   /*3.02 set the initial condition of the parallel port*/
   /*     All the port_pins defined on lattice.h will be controlled by
          ispstream_pump(). The rest can be initialized here either to
          HIGH or LOW and won't be altered by isptream_pump*/
            
   isp_pins = NUL;          /*3.02 intialize to drive all port pins LOW*/
   isp_setpin(out_ISP, LOW);/*3.02 drive ispEN pin low to set the devices 
                                   to programming mode*/
   find_port();             /*look for the parallel port with download cable*/
   rcode = port_ok();       /*verify if the parallel port found is valid*/
   if (rcode > 0) {
      rcode = power_ok();   /*check if the VCC on Board is on*/
      if (rcode > 0)        /*start programming and verification*/
         rcode = ispstream_pump(filename, 0, &end);  /*3.04 add the verify_only parameter
                                                            0 = carry out programming per
                                                                ispSTREAM.
                                                            1 = ignore any programming 
                                                                instructions in the
                                                                ispSTREAM.*/
   }
   if (rcode != OK) {
      error_handler(rcode, str);
      printf("\nFailed At Row %d in Program and Verify Due To %s\n", end, str);
      printf("\n");
      printf("+-------+\n");
      printf("| FAIL! |\n");
      printf("+-------+\n");
      return (-rcode);
   } else {
      printf("\n");
      printf("+=======+\n");
      printf("| PASS! |\n");
      printf("+=======+\n");
      /* set ISP and RESET high, so customer can see device working */
      isp_setpin(out_ISP+out_RESET, HIGH);
   }

   return(0);
}
