/*===========================================================================
  PALETTE.C

  Functions to set, get, and initialize the palette.

  Copyright (c) 1993-1995 ATI Technologies Inc. All rights reserved
  =========================================================================*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <conio.h>
#include <dos.h>

#include "atim64.h"
#include "sample.h"

/* --------------------------------------------------------------------------
  SET_PALETTE - set a palette entry.

  This function sets a specific palette entry. Each component has a range
  of 0 to 255. The index has a range of 0 to 255.
-------------------------------------------------------------------------- */
void set_palette (int index, PALETTE entry)
{
    /* set DAC write index */
    iow8 (ioDAC_REGS, index);

    /* set red component */
    iow8 (ioDAC_REGS+1, entry.red);

    /* set green component */
    iow8 (ioDAC_REGS+1, entry.green);

    /* set blue component */
    iow8 (ioDAC_REGS+1, entry.blue);
}

/* --------------------------------------------------------------------------
  GET_PALETTE - get a palette entry.

  This function gets a specific palette entry. Each component has a range
  of 0 to 255. The index has a range of 0 to 255.
-------------------------------------------------------------------------- */
PALETTE get_palette (int index)
{
    PALETTE entry;

    /* set DAC read index */
    iow8 (ioDAC_REGS+3, index);

    /* get red component */
    entry.red = ior8 (ioDAC_REGS+1);

    /* get green component */
    entry.green = ior8 (ioDAC_REGS+1);

    /* get blue component */
    entry.blue = ior8 (ioDAC_REGS+1);

    return (entry);
}

/* --------------------------------------------------------------------------
  save_palette - save the palette entries.

  This function reads and saves the 256 palette entries into an the given
  palette buffer. The application should insure that there is enough space
  for 256 entries. Note that this routine should only be called for 4 or
  8 bpp modes.
-------------------------------------------------------------------------- */
void save_palette (PALETTE *palettebuffer)
{
    int index;

    /* make sure DAC palette control is selected */
    iow8 (ioDAC_CNTL, ior8 (ioDAC_CNTL) & 0xFC);

    /* make sure DAC mask is enabled */
    iow8 (ioDAC_REGS + DAC_MASK, 0xFF);

    for (index = 0; index < 256; index++)
    {
        palettebuffer[index] = get_palette (index);
    }
}

/* --------------------------------------------------------------------------
  restore_palette - restore the palette entries.

  This function restores the 256 palette entries from the given palette
  buffer previously filled by PGL_savepalette(). Note that this routine
  should only be called for 4 or 8 bpp modes.
-------------------------------------------------------------------------- */
void restore_palette (PALETTE *palettebuffer)
{
    int index;

    /* make sure DAC palette control is selected */
    iow8 (ioDAC_CNTL, ior8(ioDAC_CNTL) & 0xFC);

    /* make sure DAC mask is enabled */
    iow8 (ioDAC_REGS + DAC_MASK, 0xFF);

    for (index = 0; index < 256; index++)
    {
        set_palette (index, palettebuffer[index]);
    }
}

/* --------------------------------------------------------------------------
  INIT_PALETTE - Set the palette to default values.

  This function initializes the palette table by setting the entries to a
  set of default values. The first 16 entries are set to EGA/VGA colors. Each
  EGA/VGA color is replicated 16 times to fill the 256 palette entries.
-------------------------------------------------------------------------- */
void init_palette (void)
{
    int i, j, index;
    PALETTE entry[16] =
    {
        {0, 0, 0},          // black
        {0, 0, 168},        // blue
        {0, 168, 0},        // green
        {0, 168, 168},      // cyan
        {168, 0, 0},        // red
        {168, 0, 168},      // magenta
        {168, 168, 0},      // brown
        {168, 168, 168},    // light gray
        {84, 84, 84},       // gray
        {0, 0, 255},        // light blue
        {0, 255, 0},        // light green
        {0, 255, 255},      // light cyan
        {255, 0, 0},        // light red
        {255, 0, 255},      // light magenta
        {255, 255, 0},      // yellow
        {255, 255, 255}     // white
    };

    // set first 16 entries
    for (index = 0; index < 16; index++)
    {
        set_palette (index, entry[index]);
    }

    // set other entries by replicating the first 16 entries
    index = 16;
    for (i = 1; i < 16; i++)
    {
        for (j = 0; j < 16; j++)
        {
            set_palette (index, entry[i]);
            index++;
        }
    }
}

/* --------------------------------------------------------------------------
  init_palettized - Set the palette to default values for hi color modes.

  This function initializes the palette for hi color modes (15, 16, 24, 32
  bpp) so that they will behave as the direct color map hi color modes. This
  function is specific for cards with ATI68860 RAMDACs and its variants where
  the mode was set with the palettized flag set.
-------------------------------------------------------------------------- */
void init_palettized (void)
{
    int index;
    PALETTE entry;

    /* make sure DAC palette control is selected */
    iow8 (ioDAC_CNTL, ior8 (ioDAC_CNTL) & 0xFC);

    /* make sure DAC mask is enabled */
    iow8 (ioDAC_REGS + DAC_MASK, 0xFF);

    for (index = 0; index < 256; index++)
    {
        entry.red = index;
        entry.green = index;
        entry.blue = index;
        set_palette (index, entry);
    }
}

