/*==========================================================================
  DRAW.C

  Sample code draw functions such as rectangles, blits, and lines.

  Copyright (c) 1993-1995 ATI Technologies Inc. All rights reserved
  =========================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <dos.h>

#include "atim64.h"
#include "sample.h"

/* --------------------------------------------------------------------------
  DRAW_RECTANGLE - draw a filled rectangle

  A filled rectangle is drawn at (x, y) of size (width x height) using the
  current engine settings. For 24 bpp modes, the engine is actually in 8
  bpp mode and the CRTC is in 24 bpp. For this reason, all horizontal
  parameters must be multiplied by 3. Also, the 24 bpp alignment must
  be determine for the engine to draw pixels in the correct color. Note that
  for 24 bpp modes, the input parameters for this routine are in 24 bpp.
  The rectangle source is determined by the current setting of the DP_SRC
  register.
-------------------------------------------------------------------------- */
void draw_rectangle (int x, int y, int width, int height)
{
    unsigned long temp, rotation;

    wait_for_idle ();       // wait for idle before reading GUI registers

    // save used registers
    temp = regr (DST_CNTL);

    if (modeinfo.bpp == 24)
    {
        // Set 24 bpp alignment while maintaining direction bits. The
        // argument to this macro must be a 24bpp value.
        rotation = GET24BPPROTATION (x);

        // Adjust horizontal parameters. Note that this adjustment is
        // performed AFTER calling the GET24BPPROTATION macro, as this
        // macro expects unadjusted 24 bpp values.
        x = x * 3;
        width = width * 3;

        regw (DST_CNTL, (temp & 0xDF) |
                        DST_24_ROTATION_ENABLE |
                        (rotation << 8));
    }

    // perform rectangle fill
    regw (DST_X, (unsigned long) x);
    regw (DST_Y, (unsigned long) y);
    regw (DST_HEIGHT, (unsigned long) height);
    regw (DST_WIDTH, (unsigned long) width);

    // restore
    regw (DST_CNTL, temp);
}

/* --------------------------------------------------------------------------
  CLEAR_SCREEN - clear a region of video memory

  A BLACK filled rectangle is drawn at (x, y) of size (width x height).
  The current engine mix setting will not affect the operation.
-------------------------------------------------------------------------- */
void clear_screen (int x, int y, int width, int height)
{
    unsigned long temp;

    wait_for_idle ();       // wait for idle before reading GUI registers

    // save used registers
    temp = regr (DP_MIX);

    // perform clear screen
    regw (DP_MIX, FRGD_MIX_ZERO | BKGD_MIX_ZERO);
    draw_rectangle (x, y, width, height);

    // wait for clear screen operation to finish before exiting
    wait_for_idle ();

    // restore DP_MIX register
    regw (DP_MIX, temp);
}

/* --------------------------------------------------------------------------
  DRAW_LINE - draw a line from (x1, y1) to (x2, y2)

  The drawing of the last pixel in the line is determined by the current
  setting of the DST_CNTL register (LAST_PEL bit).
-------------------------------------------------------------------------- */
void draw_line (int x1, int y1, int x2, int y2)
{
    int dx, dy;
    int small, large;
    int x_dir, y_dir, y_major;
    unsigned long err, inc, dec, temp;

    /* call specific routine if mode is in 24 bpp */
    if (modeinfo.bpp == 24)
    {
        draw_line24 (x1, y1, x2, y2);
        return;
    }

    /* determine x & y deltas and x & y direction bits */
    if (x1 < x2)
    {
        dx = x2 - x1;
        x_dir = 1;
    }
    else
    {
        dx = x1 - x2;
        x_dir = 0;
    }

    if (y1 < y2)
    {
        dy = y2 - y1;
        y_dir = 2;
    }
    else
    {
        dy = y1 - y2;
        y_dir = 0;
    }

    /* determine x & y min and max values; also determine y major bit */
    if (dx < dy)
    {
        small = dx;
        large = dy;
        y_major = 4;
    }
    else
    {
        small = dy;
        large = dx;
        y_major = 0;
    }

    /* calculate Bresenham parameters and draw line */
    err = (unsigned long) ((2 * small) - large);
    inc = (unsigned long) (2 * small);
    dec = 0x3FFFF - ((unsigned long) (2 * (large - small)));

    wait_for_idle ();       // wait for idle before reading GUI registers

    // save used registers
    temp = regr (DST_CNTL);

    // draw Bresenham line
    regw (DST_X, (unsigned long) x1);
    regw (DST_Y, (unsigned long) y1);

    // allow setting of last pel bit and polygon outline bit for line drawing

    regw (DST_CNTL, (temp & 0x60) | (unsigned long) (y_major | y_dir | x_dir));
    regw (DST_BRES_ERR, err);
    regw (DST_BRES_INC, inc);
    regw (DST_BRES_DEC, dec);
    regw (DST_BRES_LNTH, (unsigned long) (large + 1));

    // restore
    regw (DST_CNTL, temp);
}


/* --------------------------------------------------------------------------
  DRAW_LINE24 - draw a bressenham line from (x1, y1) to (x2, y2).

  Since the engine does not directly support 24 bpp modes, it is set
  to 8 bpp while the CRTC is set to 24 bpp display mode (RGB). For
  rectangle drawing, all X coordinates and widths must be converted
  to 8 bpp sizes. This is done by taking the 24 bpp value and
  multipling it by 3.
-------------------------------------------------------------------------- */
void draw_line24 (int x1, int y1, int x2, int y2)
{
    int x, y, xend, yend, dx, dy;
    int d, incr1, incr2, incr3;
    unsigned long rotation, temp1, temp2;

    // save register
    wait_for_idle ();
    temp1 = regr (DST_CNTL);
    temp2 = 0xA3;

    // ------------------ Bresenham line routine ----------------------

    dx = abs (x2 - x1);
    dy = abs (y2 - y1);

    // check slope
    if (dy <= dx) // slope <= 1
    {
        if (x1 > x2)
        {
            x = x2;
            y = y2;
            xend = x1;
            dy = y1 - y2;
        }
        else
        {
            x = x1;
            y = y1;
            xend = x2;
            dy = y2 - y1;
        }

        d = (2 * dy) - dx;
        incr1 = 2 * dy;
        incr2 = 2 * (dy - dx);
        incr3 = 2 * (dy + dx);

        regw (DST_HEIGHT, 1);
        regw (DST_Y, y);

        do
        {
            wait_for_fifo (4);
            rotation = GET24BPPROTATION (x);
            regw (DST_CNTL, temp2 | (rotation << 8));
            regw (DST_X, x * 3);
            regw (DST_WIDTH, 3);

            x++;

            if (d >= 0)
            {
                if (dy <= 0)
                {
                    d = d + incr1;
                }
                else
                {
                    y++;
                    regw (DST_Y, y);
                    d = d + incr2;
                }
            }
            else
            {
                if (dy >= 0)
                {
                    d = d + incr1;
                }
                else
                {
                    y--;
                    regw (DST_Y, y);
                    d = d + incr3;
                }
            }
        } while (x <= xend);
    }
    else          // slope > 1
    {
        if (y1 > y2)
        {
            y = y2;
            x = x2;
            yend = y1;
            dx = x1 - x2;
        }
        else
        {
            y = y1;
            x = x1;
            yend = y2;
            dx = x2 - x1;
        }

        d = (2 * dx) - dy;
        incr1 = 2 * dx;
        incr2 = 2 * (dx - dy);
        incr3 = 2 * (dx + dy);

        regw (DST_HEIGHT, 1);

        do
        {
            wait_for_fifo (3);
            rotation = GET24BPPROTATION (x);
            regw (DST_CNTL, temp2 | (rotation << 8));
            regw (DST_Y_X, ((unsigned long) (x * 3) << 16) | y);
            regw (DST_WIDTH, 3);

            y++;

            if (d >= 0)
            {
                if (dx <= 0)
                {
                    d = d + incr1;
                }
                else
                {
                    x++;
                    d = d + incr2;
                }
            }
            else
            {
                if (dx >= 0)
                {
                    d = d + incr1;
                }
                else
                {
                    x--;
                    d = d + incr3;
                }
            }
        } while (y <= yend);
    }

    // restore register
    wait_for_fifo (1);
    regw (DST_CNTL, temp1);
}

/* --------------------------------------------------------------------------
  BLIT - perform a screen to screen blit

  Copy the contents of screen memory at (x1, y1) of size (width x height) to
  (x2, y2) using the current engine settings. This is known as an unbounded
  Y source trajectory blit. For 24 bpp modes, the engine is in 8 bpp and
  the CRTC is in 24 bpp. For this reason, all horizontal parameters must be
  mulitplied by 3. The blit source is determined by the current setting
  of the DP_SRC register.
-------------------------------------------------------------------------- */
void blit (int x1, int y1, int x2, int y2, int width, int height)
{
    unsigned long temp, rotation;

    wait_for_idle ();       // wait for idle before reading GUI registers

    // save used registers
    temp = regr (DST_CNTL);

    if (modeinfo.bpp == 24)
    {
        // get 24 bpp alignment rotation for destination
        rotation = GET24BPPROTATION (x2);

        // adjust horizontal parameters
        x1 = x1 * 3;
        x2 = x2 * 3;
        width = width * 3;

        // set 24 bpp alignment while maintaining direction bits
        regw (DST_CNTL, (temp & 0xDF) |
                        DST_24_ROTATION_ENABLE |
                        (rotation << 8));
    }

    // perform a blit
    regw (SRC_X, x1);
    regw (SRC_Y, y1);
    regw (SRC_HEIGHT1, height);
    regw (SRC_WIDTH1, width);

    regw (DST_X, x2);
    regw (DST_Y, y2);
    regw (DST_HEIGHT, height);
    regw (DST_WIDTH, width);

    // restore
    regw (DST_CNTL, temp);
}

