; ===========================================================================
; ISR8.ASM
;
; Interrupt service routine for system timer.
;
; Compiling:
;   masm /Ml /D<memory model> romcalls.asm;
;       <memory model> = mem_S for SMALL model,
;                        mem_M for MEDIUM model,
;                        mem_L for LARGE model
;
; Copyright (c) 1994-1995 ATI Technologies Inc. All rights reserved
; ===========================================================================

include ..\util\atim64.inc

IFDEF mem_S
PARM        equ     4   ; passed parameters start at bp+4 for small model
ELSE
PARM        equ     6   ; passed parameters start at bp+6 for other models
ENDIF

IFDEF mem_S
.MODEL  SMALL, C
ELSEIFDEF mem_M
.MODEL  MEDIUM, C
ELSE
.MODEL  LARGE, C
ENDIF

.DATA

.CODE
.286

OldIntVector    dw  ?
                dw  ?
SwapFlag        db  1
CrtcPitchOffset dw  ?
                dw  ?

IFDEF mem_S
extrn       ior8:NEAR
extrn       ior16:NEAR
extrn       ior32:NEAR
extrn       iow8:NEAR
extrn       iow16:NEAR
extrn       iow32:NEAR
ELSE
extrn       ior8:FAR
extrn       ior16:FAR
extrn       ior32:FAR
extrn       iow8:FAR
extrn       iow16:FAR
extrn       iow32:FAR
ENDIF

; Macro for 'call' model handling
Mcall       macro   routine
IFDEF mem_S
            call    NEAR PTR routine
ELSE
            call    FAR PTR routine
ENDIF
            endm

; macros to handle IO operations
Min         macro   inp_routine
            push    dx
            Mcall   inp_routine
            add     sp, 2
            endm

Mout        macro   outp_routine
            push    ax
            push    dx
            Mcall   outp_routine
            add     sp, 4
            endm

; ---------------------------------------------------------------------------
; Int8ISR - chain onto IRQ 8.
;
; Timer tick interrupt service routine
;
; ---------------------------------------------------------------------------
            public  Int8ISR

Int8ISR     proc    far

            cli

            push    ax
            push    dx


            ; Prepare to switch CRTC buffer
            mov     dx, ioCRTC_OFF_PITCH

            ; check SwapFlag
            cmp     SwapFlag, 0
            jne     next_frame

            ; SwapFlag = 0: display frame2, update frame 1
            mov     ax, CrtcPitchOffset
            Mout    iow16
            inc     dx
            inc     dx
            mov     ax, CrtcPitchOffset+2
            Mout    iow16

            ; change swapflag to 1
            mov     ax, 1
            jmp     update

next_frame:
            ; SwapFlag = 1: display frame1, update frame 2
            ; maintain pitch setting
            mov     ax, 0
            Mout    iow16

            inc     dx
            inc     dx
            mov     ax, CrtcPitchOffset+2
            and     ax, 0FFC0h
            Mout    iow16

            ; change swapflag to 0
            mov     ax, 0

update:
            mov     SwapFlag, al

skip:
            pop     dx
            pop     ax

            pushf
            call     DWORD PTR OldIntVector
            sti

            iret

Int8ISR     endp

; ---------------------------------------------------------------------------
; GETSWAPFLAG
;
; Retrieve the swap flag controlled by the ISR.
;
; Inputs : none
;
; Outputs: WORD swapflag
;          0 = display frame 2, update frame 1
;          1 = display frame 1, update frame 2
; ---------------------------------------------------------------------------
            public  getswapflag

IFDEF mem_S
getswapflag proc    near
ELSE
getswapflag proc    far
ENDIF
            ; get swapflag
            mov     al, SwapFlag
            xor     ah, ah

            ret

getswapflag endp

; ---------------------------------------------------------------------------
; INITTIMERISR
;
; Chain in timer interrupt routine.
;
; Inputs : none
;
; Outputs: none
; ---------------------------------------------------------------------------
            public  inittimerisr

IFDEF mem_S
inittimerisr proc    near
ELSE
inittimerisr proc    far
ENDIF
            ; save used registers
            push    bx
            push    dx
            push    ds
            push    es

            ; initialize variables for ISR
            mov     al, 0
            mov     SwapFlag, al

            ; get old IRQ 8 vector address (in ES:BX)
            mov     ax, 3508h
            int     21h

            ; save vector address
            mov     ax, es
            mov     OldIntVector, bx
            mov     OldIntVector+2, ax

            ; get address of interrupt service routine (in DS:DX)
            mov     dx, offset cs:Int8ISR
            mov     ax, cs
            mov     ds, ax

            ; set new vector address
            mov     ax, 2508h
            int     21h

            ; restore registers
            pop     es
            pop     ds
            pop     dx
            pop     bx

            ret

inittimerisr endp

; ---------------------------------------------------------------------------
; CANCELTIMERISR
;
; Disable timer interrupt routine.
;
; Inputs : none
;
; Outputs: none
; ---------------------------------------------------------------------------
            public  canceltimerisr

IFDEF mem_S
canceltimerisr proc    near
ELSE
canceltimerisr proc    far
ENDIF
            ; save used registers
            push    dx
            push    ds

            ; restore old IRQ 8 vector address (in DS:DX)
            mov     dx, OldIntVector
            mov     ax, OldIntVector+2
            mov     ds, ax
            mov     ax, 2508h
            int     21h

            ; restore registers
            pop     ds
            pop     dx

            ret

canceltimerisr endp

; ---------------------------------------------------------------------------
; INTERRUPTS_OFF
;
; Disable system interrupts.
;
; Inputs : none
;
; Outputs: none
; ---------------------------------------------------------------------------
            public  interrupts_off

IFDEF mem_S
interrupts_off proc near
ELSE
interrupts_off proc far
ENDIF
            ; disable system interrupts
            cli

            ret

interrupts_off endp

; ---------------------------------------------------------------------------
; INTERRUPTS_ON
;
; Enable system interrupts.
;
; Inputs : none
;
; Outputs: none
; ---------------------------------------------------------------------------
            public  interrupts_on

IFDEF mem_S
interrupts_on proc near
ELSE
interrupts_on proc far
ENDIF
            ; enable system interrupts
            sti

            ret

interrupts_on endp

; ---------------------------------------------------------------------------
; SETCRTCOFFSET
;
; Pass the value used to switch to the second page including the pitch
; setting for the CRTC_OFF_PITCH register.
;
; Inputs : DWORD second page offset to be used in CRTC_OFF_PITCH register
;
; Outputs: none
; ---------------------------------------------------------------------------
            public  setcrtcoffset

IFDEF mem_S
setcrtcoffset proc    near
ELSE
setcrtcoffset proc    far
ENDIF
            ; create frame pointer
            push    bp
            mov     bp, sp

            ; get value and save in local variable
            mov     ax, WORD PTR [bp+PARM]      ; get lower 16 bits of value
            mov     CrtcPitchOffset, ax
            mov     ax, WORD PTR [bp+PARM+2]    ; get upper 16 bits of value
            mov     CrtcPitchOffset+2, ax

            ; remove frame pointer
            mov     sp, bp
            pop     bp


            ret

setcrtcoffset endp

            end

