; ===========================================================================
; ROMSEG.ASM
;
; Interface routines from determining the ROM segment of a Mach64 based card.
;
; Compiling:
;   masm /Ml /D<memory model> getio.asm;
;       <memory model> = mem_S for SMALL model,
;                        mem_M for MEDIUM model,
;                        mem_L for LARGE model
;
; Copyright (c) 1994 ATI Technologies Inc. All rights reserved
; ===========================================================================

include ..\inc\atim64.inc

IFDEF mem_S
PARM        equ     4   ; passed parameters start at bp+4 for small model
ELSE
PARM        equ     6   ; passed parameters start at bp+6 for other models
ENDIF

IFDEF mem_S
.MODEL  SMALL, C
ELSEIFDEF mem_M
.MODEL  MEDIUM, C
ELSE
.MODEL  LARGE, C
ENDIF

.DATA

ati_sig     db      '761295520', 0
m64_sig1    db      'MACH64', 0
m64_sig2    db      'GXCX', 0           ; older ROMs
rom_seg     dw      0c000h

.CODE
.286

; Macro for 'call' model handling
Mcall       macro   routine
IFDEF mem_S
            call    NEAR PTR routine
ELSE
            call    FAR PTR routine
ENDIF
	        endm

; ---------------------------------------------------------------------------
; PGL_ISATIMACH64ROM
;
; Check if ROM belongs to an ATI MACH64 based product. The check is done
; using the following method:
;
; - check if ROM ID is present (AA55h) at location 0 of the ROM
; - check if the ATI signature string is present within the first 1K bytes of
;   the ROM ('761295520')
; - check if the MACH64 signature string is present within the first 1K bytes
;   of the ROM ('MACH64' or 'GXCX')
;
; The ROM segment is a given input parameter and is fetched from the stack.
; It's value should be one of the following: C000h, C800h, C900h, CA00h,
; CB00h, CC00h, CD00h, CE00h, or CF00h. This method will allow detection of
; ATI MACH64 ROMs where the VGA is enabled or disabled, and where more that
; one MACH64 based product is installed.
;
; Inputs : ROM segment (C000h, C800h, C900h, CA00h, CB00h, ..., or CF00h)
;
; Outputs: Returns 1 in ax if signature was found, 0 if not
; ---------------------------------------------------------------------------
            public  pgl_isatimach64rom

IFDEF mem_S
pgl_isatimach64rom proc near
ELSE
pgl_isatimach64rom proc far
ENDIF
            ; create frame pointer
            push    bp
            mov     bp, sp

            ; save registers used
            push    cx
            push    di
            push    si
            push    es

            ; get ROM segment from input parameter
            mov     ax, WORD PTR [bp+PARM]
            mov     es, ax

            ; check for ROM ID code (AA55h) at location 0 of the ROM
            mov     di, 0
            mov     ax, es:[di]
            cmp     ax, 0aa55h
            jne     iamr_notfound

            ; check for ATI signature string in first 1K bytes of ROM
            ; DS:SI = source compare string pointer
            ; ES:DI = test pointer
            cld                         ; insure forward direction
            mov     ax, 0
iamr1:
            mov     cx, 9               ; length of compare string
            mov     di, ax
            mov     si, offset ati_sig
            repe    cmpsb
            jcxz    @F                  ; branch if match found
            inc     ax
            cmp     ax, 1024
            jne     iamr1
            jmp     iamr_notfound
@@:
            ; check for first OR second MACH64 signature string in first 1K
            ; bytes of the ROM

            ; check for first MACH64 signature string
            ; DS:SI = source compare string pointer
            ; ES:DI = test pointer
            mov     ax, 0
iamr2:
            mov     cx, 6               ; length of compare string
            mov     di, ax
            mov     si, offset m64_sig1
            repe    cmpsb
            jcxz    iamr_found          ; branch if match found
            inc     ax
            cmp     ax, 1024
            jne     iamr2

            ; if first string could not be found, check for second one
            ; DS:SI = source compare string pointer
            ; ES:DI = test pointer
            mov     ax, 0
iamr3:
            mov     cx, 4               ; length of compare string
            mov     di, ax
            mov     si, offset m64_sig2
            repe    cmpsb
            jcxz    iamr_found          ; branch if match found
            inc     ax
            cmp     ax, 1024
            jne     iamr3
            jmp     iamr_notfound

iamr_found:
            ; found a valid ATI MACH64 ROM - assign pass code
            mov     ax, 1
            jmp     iamr_ret

iamr_notfound:
            mov     ax, 0               ; assign failure code

iamr_ret:
            ; restore saved registers
            pop     es
            pop     si
            pop     di
            pop     cx

            ; remove frame pointer
            mov     sp, bp
            pop     bp

            ret

pgl_isatimach64rom endp

; ---------------------------------------------------------------------------
; PGL_ISATISIGROM
;
; Check if ROM has ATI signature. This only indicates that the ROM belongs
; to an ATI product. The check is done using the following method:
;
; - check if ROM ID is present (AA55h) at location 0 of the ROM
; - check if the ATI signature string is present within the first 1K bytes of
;   the ROM ('761295520')
;
; Inputs : ROM segment (C000h, C800h, C900h, CA00h, CB00h, ..., or CF00h)
;
; Outputs: Returns 1 in ax if signature was found, 0 if not
; ---------------------------------------------------------------------------
            public  pgl_isatisigrom

IFDEF mem_S
pgl_isatisigrom proc near
ELSE
pgl_isatisigrom proc far
ENDIF
            ; create frame pointer
            push    bp
            mov     bp, sp

            ; save registers used
            push    cx
            push    di
            push    si
            push    es

            ; get ROM segment from input parameter
            mov     ax, WORD PTR [bp+PARM]
            mov     es, ax

            ; check for ROM ID code (AA55h) at location 0 of the ROM
            mov     di, 0
            mov     ax, es:[di]
            cmp     ax, 0aa55h
            jne     iasr_notfound

            ; check for ATI signature string in first 1K bytes of ROM
            ; DS:SI = source compare string pointer
            ; ES:DI = test pointer
            cld                         ; insure forward direction
            mov     ax, 0
iasr1:
            mov     cx, 9               ; length of compare string
            mov     di, ax
            mov     si, offset ati_sig
            repe    cmpsb
            jcxz    @F                  ; branch if match found
            inc     ax
            cmp     ax, 1024
            jne     iasr1
            jmp     iasr_notfound
@@:
            ; found a valid ATI signature - assign pass code
            mov     ax, 1
            jmp     iasr_ret

iasr_notfound:
            mov     ax, 0               ; assign failure code

iasr_ret:
            ; restore saved registers
            pop     es
            pop     si
            pop     di
            pop     cx

            ; remove frame pointer
            mov     sp, bp
            pop     bp

            ret

pgl_isatisigrom endp

; ---------------------------------------------------------------------------
; PGL_SETROMBASE
;
; Sets the 'rom_seg' data variable. This value will be used when GET_ROM_BASE
; is called. The input parameter is fetched from the stack
;
; Inputs : WORD rom_segment
;
; Outputs: none
; ---------------------------------------------------------------------------
            public pgl_setrombase

IFDEF mem_S
pgl_setrombase proc near
ELSE
pgl_setrombase proc far
ENDIF
            ; create frame pointer
            push    bp
            mov     bp, sp

            ; get rom segment and assign it to the local 'rom_seg' variable
            push    ax
            mov     ax, WORD PTR [bp+PARM]
            mov     rom_seg, ax
            pop     ax

            ; remove frame pointer
            mov     sp, bp
            pop     bp

            ret

pgl_setrombase endp

; ---------------------------------------------------------------------------
; PGL_GETROMBASE
;
; Retrieve base segment of MACH64 ROM. Used by functions in ROMCALL?.ASM
; files
;
; Inputs : none
;
; Outputs: Returns base rom segment in ax
; ---------------------------------------------------------------------------
            public pgl_getrombase

IFDEF mem_S
pgl_getrombase proc near
ELSE
pgl_getrombase proc far
ENDIF
            ; fetch ROM segment and return in ax
            mov     ax, rom_seg
            ret

pgl_getrombase endp

            end

