/*==========================================================================
* PALETTE.C - Copyright (c) 1993 ATI Technologies Inc. All rights reserved *
*                                                                          *
* PGL functions to set, get, and initialize the palette.                   *
* ======================================================================== */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <conio.h>
#include <memory.h>
#include <dos.h>

#include "..\inc\atim64.h"
#include "..\inc\pgl.h"
#include "..\inc\pglglob.h"

/* --------------------------------------------------------------------------
  PGL_setpalette - set a palette entry.
  
  This function sets a specific palette entry. Each component has a range
  of 0 to 255. The index has a range of 0 to 255. Note that this routine
  should only be called for 4 or 8 bpp modes.
-------------------------------------------------------------------------- */
void PGL_setpalette(int index, PGL_palette entry)
{
    /* set DAC write index */
    iow8(ioDAC_REGS + DAC_W_INDEX, index);

    /* set red component */
    iow8(ioDAC_REGS + DAC_DATA, entry.red);

    /* set green component */
    iow8(ioDAC_REGS + DAC_DATA, entry.green);

    /* set blue component */
    iow8(ioDAC_REGS + DAC_DATA, entry.blue);
}

/* --------------------------------------------------------------------------
  PGL_getpalette - get a palette entry.
  
  This function gets a specific palette entry. Each component has a range
  of 0 to 255. The index has a range of 0 to 255. Note that this routine
  should only be called for 4 or 8 bpp modes.
-------------------------------------------------------------------------- */
PGL_palette PGL_getpalette(int index)
{
    PGL_palette entry;

    /* set DAC read index */
    iow8(ioDAC_REGS + DAC_R_INDEX, index);

    /* get red component */
    entry.red = ior8(ioDAC_REGS + DAC_DATA);

    /* get green component */
    entry.green = ior8(ioDAC_REGS + DAC_DATA);

    /* get blue component */
    entry.blue = ior8(ioDAC_REGS + DAC_DATA);

    return (entry);
}

/* --------------------------------------------------------------------------
  PGL_savepalette - save the palette entries.
  
  This function reads and saves the 256 palette entries into an the given
  palette buffer. The application should insure that there is enough space
  for 256 entries. Note that this routine should only be called for 4 or
  8 bpp modes.
-------------------------------------------------------------------------- */
void PGL_savepalette(PGL_palette *paletteBuffer)
{
    int index;

    /* make sure DAC palette control is selected */
    iow8(ioDAC_CNTL, ior8(ioDAC_CNTL) & 0xfc);

    /* make sure DAC mask is enabled */
    iow8(ioDAC_REGS + DAC_MASK, 0xff);

    for (index = 0; index < 256; index++)
    {
        paletteBuffer[index] = PGL_getpalette(index);
    }
}

/* --------------------------------------------------------------------------
  PGL_restorepalette - restore the palette entries.
  
  This function restores the 256 palette entries from the given palette
  buffer previously filled by PGL_savepalette(). Note that this routine
  should only be called for 4 or 8 bpp modes.
-------------------------------------------------------------------------- */
void PGL_restorepalette(PGL_palette *paletteBuffer)
{
    int index;

    /* make sure DAC palette control is selected */
    iow8(ioDAC_CNTL, ior8(ioDAC_CNTL) & 0xfc);

    /* make sure DAC mask is enabled */
    iow8(ioDAC_REGS + DAC_MASK, 0xff);

    for (index = 0; index < 256; index++)
    {
        PGL_setpalette(index, paletteBuffer[index]);
    }
}

/* --------------------------------------------------------------------------
  PGL_initpalette - Set the palette to default values.
  
  This function initializes the palette table by setting the entries to a
  set of default values. The first 16 entries are set to EGA/VGA colors. Each
  EGA/VGA color is replicated 16 times to fill the 256 palette entries. Note
  that this routine should only be called for 4 or 8 bpp modes.
-------------------------------------------------------------------------- */
void PGL_initpalette(void)
{
    int i, j, index;

    /* make sure DAC palette control is selected */
    iow8(ioDAC_CNTL, ior8(ioDAC_CNTL) & 0xfc);

    /* make sure DAC mask is enabled */
    iow8(ioDAC_REGS + DAC_MASK, 0xff);

    // set first 16 entries
    for (index = 0; index < 16; index++)
    {
        PGL_setpalette(index, DefaultPalette[index]);
    }

    // set other entries by replicating the first 16 entries
    index = 16;
    for (i = 1; i < 16; i++)
    {
        for (j = 0; j < 16; j++)
        {
            PGL_setpalette(index, DefaultPalette[i]);
            index++;
        }
    }
}

/* --------------------------------------------------------------------------
  PGL_initpalettized - Set the palette to default values for hi color modes.
  
  This function initializes the palette for hi color modes (15, 16, 24, 32
  bpp) so that they will behave as the direct color map hi color modes. This
  function is specific for cards with ATI68860 RAMDACs and its variants where
  the mode was set with the palettized flag set.
-------------------------------------------------------------------------- */
void PGL_initpalettized(void)
{
    int index;
    PGL_palette entry;

    /* make sure DAC palette control is selected */
    iow8(ioDAC_CNTL, ior8(ioDAC_CNTL) & 0xfc);

    /* make sure DAC mask is enabled */
    iow8(ioDAC_REGS + DAC_MASK, 0xff);

    for (index = 0; index < 256; index++)
    {
        entry.red = index;
        entry.green = index;
        entry.blue = index;
        PGL_setpalette(index, entry);
    }
}

