; ===========================================================================
; MEMREG.ASM
;
; Functions to access the 32 bit memory mapped and I/O registers on the
; Mach 64.
;
; Compiling:
;   masm /Ml /D<memory model> ioreg.asm;
;       <memory model> = mem_S for SMALL model,
;                        mem_M for MEDIUM model,
;                        mem_L for LARGE model
;
; Copyright (c) 1994 ATI Technologies Inc. All rights reserved
; ===========================================================================

include ..\inc\atim64.inc

IFDEF mem_S
PARM        equ     4   ; passed parameters start at bp+4 for small model
ELSE
PARM        equ     6   ; passed parameters start at bp+6 for other models
ENDIF

IFDEF mem_S
.MODEL  SMALL, C
ELSEIFDEF mem_M
.MODEL  MEDIUM, C
ELSE
.MODEL  LARGE, C
ENDIF

.DATA

VGAAppFlag      DW      1       ; assume VGA aperture
LinAppBaseLo    DW      0       ; assume zero
LinAppBaseHi    DW      0
MemRegBaseLo    DW      0fc00h  ; assume 4 Meg Linear aperture size
MemRegBaseHi    DW      003fh
RegBufferLo     DW      0
RegBufferHi     DW      0

.CODE
.386

; Macro for 'call' model handling
Mcall       macro   routine
IFDEF mem_S
            call    NEAR PTR routine
ELSE
            call    FAR PTR routine
ENDIF
	        endm

IFDEF mem_S
extrn pgl_movemem:NEAR
ELSE
extrn pgl_movemem:FAR
ENDIF

; ---------------------------------------------------------------------------
; PGL_INITMEMREG
;
; Initialize memory map register functions
;
; Inputs : WORD VGA aperture status flag: 0 - off, 1 - on (stack)
;          DWORD Linear aperture base address (stack)
;          DWORD Register memory offset (stack)
;
; Outputs: none
;
; This function MUST be called before any of the memory mapped register
; routines can be used.
; ---------------------------------------------------------------------------
            public  pgl_initmemreg

IFDEF mem_S
pgl_initmemreg proc near
ELSE
pgl_initmemreg proc far
ENDIF
            ; create frame pointer
            push    bp
            mov     bp, sp

            ; save used registers
            push    ax

            ; store VGA aperture status flag
            mov     ax, WORD PTR [bp+PARM]
            and     ax, 1
            mov     VGAAppFlag, ax

            ; store linear aperture base address
            mov     ax, WORD PTR [bp+PARM+2]
            mov     LinAppBaseLo, ax
            mov     ax, WORD PTR [bp+PARM+4]
            mov     LinAppBaseHi, ax

            ; store linear aperture base register offset
            mov     ax, WORD PTR [bp+PARM+6]
            mov     MemRegBaseLo, ax
            mov     ax, WORD PTR [bp+PARM+8]
            mov     MemRegBaseHi, ax

            ; restore used registers
            pop     ax

            ; remove frame pointer
            mov     sp, bp
            pop     bp

            ret

pgl_initmemreg endp

; ---------------------------------------------------------------------------
; REGW - write to a memory mapped register through an available aperture.
;
; This function will provide write access to the memory mapped registers.
; Each register is 32 bits wide. The appropriate method is selected based
; which aperture is enabled. Preference is given to the small VGA aperture
; since this method will work on all bus types. It is assumed that one of
; the two apertures is enabled.
;
;
; VGA aperture method:
;
;   This method will provide write access on all bus types. It is assumed
;   that the VGA aperture is enabled. The base address of the memory mapped
;   registers is B000:FC00h. Each register occupies 4 bytes. This is also
;   the fastest method for real mode applications.
;
;
; Linear aperture method:
;
;   This method will provide write access on all bus types except ISA. ISA
;   will work if the linear aperture can be enabled without mapping to
;   existing extended memory. It is assumed that the linear memory aperture
;   is enabled. The base address of the memory mapped registers is (base
;   aperture address + 3FFC00h) for a 4M aperture size and (base aperture
;   address + 7FFC00h) for an 8M aperture size. Each register occupies 4
;   bytes. This method will impose a heavy performance hit since the memory
;   mapped registers exist in extended memory above real mode space.
;
; Inputs : WORD register index (stack)
;          DWORD 32 bit data (stack)
;
; Outputs: return code in AX: 0 = no error, 1 = error
; ---------------------------------------------------------------------------
            public  regw

IFDEF mem_S
regw        proc    near
ELSE
regw        proc    far
ENDIF
            ; create frame pointer
            push    bp
            mov     bp, sp

            ; check on access method: 0 = linear, 1 = VGA
            cmp     VGAAppFlag, 0
            je      @F

            ; --- Use VGA aperture to access memory mapped register ---
            ;
            ; Note that GUI registers on BEDROCK chips must be written in
            ; 32 bit size only. Other Mach64 chips permit 8/16/32 bit sizes.
            ;

            ; save used registers
            push    si
            push    es
            push    eax

            ; write to memory mapped register
            mov     ax, 0b000h                  ; base address = b000:fc00h
            mov     es, ax
            mov     si, 0fc00h
            add     si, WORD PTR [bp+PARM]      ; get register index
            mov     eax, DWORD PTR [bp+PARM+2]  ; get full 32 bits of data
            mov     es:[si], eax                ; write it

            ; restore used registers
            pop     eax
            pop     es
            pop     si

            ; setup return code in AX - always no error with VGA aperture
            mov     ax, 0

            ; leave
            jmp     regw_ret

@@:
            ; --- Use Linear aperture to access memory mapped register ---

            ; save used registers
            push    dx

            ; PGL_MOVEMEM() call setup

            ; load data into transfer buffer
            mov     ax, WORD PTR [bp+PARM+2]    ; get low 16 bits of data
            mov     RegBufferLo, ax
            mov     ax, WORD PTR [bp+PARM+4]    ; get high 16 bits of data
            mov     RegBufferHi, ax

            ; input parameter: direction flag = WRITE
            mov     ax, 1
            push    ax

            ; input parameter: words to transfer
            mov     ax, 2
            push    ax

            ; input parameter: aperture address
            mov     ax, LinAppBaseLo
            mov     dx, LinAppBaseHi
            adc     ax, MemRegBaseLo
            add     dx, MemRegBaseHi
            add     ax, WORD PTR [bp+PARM]      ; get register index
            push    dx                          ; push high 16 bits
            push    ax                          ; push low 16 bits

            ; input parameter: far ptr to memory buffer
            mov     ax, offset ds:RegBufferLo
            push    ds                          ; push segment
            push    ax                          ; push offset

            ; call PGL_MOVEMEM to write to memory mapped register
            Mcall   pgl_movemem

            ; setup return code in DX
            mov     dx, ax
            cmp     dx, 0
            je      @F
            mov     dx, 1
@@:
            ; clean up stack after call
            pop     ax
            pop     ax
            pop     ax
            pop     ax
            pop     ax
            pop     ax

            ; place return code in AX
            mov     ax, dx

            ; register used registers
            pop     dx
regw_ret:
            ; remove frame pointer
            mov     sp, bp
            pop     bp

            ret

regw        endp

; ---------------------------------------------------------------------------
; REGR - read from a memory mapped register through an available aperture.
;
; This function will provide read access to the memory mapped registers.
; Each register is 32 bits wide. The appropriate method is selected based
; which aperture is enabled. Preference is given to the small VGA aperture
; since this method will work on all bus types. It is assumed that one of
; the two apertures is enabled.
;
;
; VGA aperture method:
;
;   This method will provide read access on all bus types. It is assumed
;   that the VGA aperture is enabled. The base address of the memory mapped
;   registers is B000:FC00h. Each register occupies 4 bytes. This is also
;   the fastest method for real mode applications.
;
;
; Linear aperture method:
;
;   This method will provide read access on all bus types except ISA. ISA
;   will work if the linear aperture can be enabled without mapping to
;   existing extended memory. It is assumed that the linear memory aperture
;   is enabled. The base address of the memory mapped registers is (base
;   aperture address + 3FFC00h) for a 4M aperture size and (base aperture
;   address + 7FFC00h) for an 8M aperture size. Each register occupies 4
;   bytes.
;
; Inputs : WORD register index (stack)
;
; Outputs: AX = low 16 bits of data
;          DX = high 16 bits of data
; ---------------------------------------------------------------------------
            public  regr

IFDEF mem_S
regr        proc    near
ELSE
regr        proc    far
ENDIF
            ; create frame pointer
            push    bp
            mov     bp, sp

            ; check on access method: 0 = linear, 1 = VGA
            cmp     VGAAppFlag, 0
            je      @F

            ; --- Use VGA aperture to access memory mapped register ---

            ; save used registers
            push    si
            push    es

            ; read from memory mapped register
            mov     ax, 0b000h                  ; base address = b000:fc00h
            mov     es, ax
            mov     si, 0fc00h
            add     si, WORD PTR [bp+PARM]      ; get register index
            mov     ax, es:[si]                 ; read it
            inc     si                          ; next word
            inc     si
            mov     dx, es:[si]                 ; read it

            ; restore used registers
            pop     es
            pop     si

            ; leave
            jmp     regr_ret

@@:
            ; --- Use Linear aperture to access memory mapped register ---

            ; PGL_MOVEMEM() call setup

            ; input parameter: direction flag = READ
            mov     ax, 0
            push    ax

            ; input parameter: words to transfer
            mov     ax, 2
            push    ax

            ; input parameter: aperture address
            mov     ax, LinAppBaseLo
            mov     dx, LinAppBaseHi
            adc     ax, MemRegBaseLo
            add     dx, MemRegBaseHi
            add     ax, WORD PTR [bp+PARM]      ; get register index
            push    dx                          ; push high 16 bits
            push    ax                          ; push low 16 bits

            ; input parameter: far ptr to memory buffer
            mov     ax, offset ds:RegBufferLo
            push    ds                          ; push segment
            push    ax                          ; push offset

            ; call PGL_MOVEMEM to write to memory mapped register
            Mcall   pgl_movemem

            ; clean up stack after call
            pop     ax
            pop     ax
            pop     ax
            pop     ax
            pop     ax
            pop     ax

            ; setup return value
            mov     ax, RegBufferLo
            mov     dx, RegBufferHi

regr_ret:
            ; remove frame pointer
            mov     sp, bp
            pop     bp

            ret

regr        endp

            end

