/*==========================================================================
* INITENG.C - Copyright (c) 1994 ATI Technologies Inc. All rights reserved *
*                                                                          *
* PGL function to initialize the engine to a known context.                *
* ======================================================================== */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <conio.h>
#include <memory.h>
#include <dos.h>

#include "..\inc\atim64.h"
#include "..\inc\pgl.h"
#include "..\inc\pglglob.h"

/* --------------------------------------------------------------------------
  PGL_initengine - initialize the engine to a known context.
  
  This function initializes the engine to a known state. The clipping
  rectangle is set to the modal size, foreground mixe is set to overpaint,
  background mix is set to leave-alone, color compare control is set to false,
  and the write mask is set to affect all planes. The default colors are
  WHITE for the foreground and BLACK for background. This function also
  initiatizes the pattern data structures which are used by various PGL
  drawing functions. This function is called by PGL_initmode().
-------------------------------------------------------------------------- */
void PGL_initengine(void)
{
    int i;
    unsigned long temp;

    // reset engine and clear any FIFO errors
    PGL_resetengine();

    // set overscan color to BLACK
    regw(OVR_CLR, 0);

    // set CRTC pixel order for MSB to LSB - affects 4 bpp modes only
    regw(CRTC_GEN_CNTL, regr(CRTC_GEN_CNTL) & 0xfffff7ff);

    // insure engine idle and empty fifo before engine context setup
    PGL_waitforidle();

    // set engine color depth and chain mask
    switch(PGL_modecfg.bpp)
    {
        case 4:
            regw(DP_PIX_WIDTH, HOST_4BPP | SRC_4BPP | DST_4BPP);
            regw(DP_CHAIN_MASK, 0x8888);
            break;
        case 8:
        case 24:
            regw(DP_PIX_WIDTH, HOST_8BPP | SRC_8BPP | DST_8BPP);
            regw(DP_CHAIN_MASK, 0x8080);
            break;
        case 16:
            if (PGL_modecfg.depth == 555)
            {
                regw(DP_PIX_WIDTH, HOST_15BPP | SRC_15BPP | DST_15BPP);
                regw(DP_CHAIN_MASK, 0x4210);
            }
            else  // 565 weight
            {
                regw(DP_PIX_WIDTH, HOST_16BPP | SRC_16BPP | DST_16BPP);
                regw(DP_CHAIN_MASK, 0x8410);
            }
            break;
        case 32:
            regw(DP_PIX_WIDTH, HOST_32BPP | SRC_32BPP | DST_32BPP);
            regw(DP_CHAIN_MASK, 0x8080);
            break;
    }

    // default context mask
    regw(CONTEXT_MASK, 0xFFFFFFFF);

    // determine engine pitch
    temp = (unsigned long)(PGL_modecfg.pitch / 8);
    if (PGL_modecfg.bpp == 24)
    {
        temp = temp * 3;
    }

    // setup engine destination registers
    regw(DST_OFF_PITCH, ((temp << 22) & 0xffc00000) | PGL_modecfg.memory_offset);
    regw(DST_Y_X, 0);
    regw(DST_HEIGHT, 1);
    regw(DST_BRES_ERR, 0);
    regw(DST_BRES_INC, 0);
    regw(DST_BRES_DEC, 0);
    if (PGL_modecfg.bpp == 24)
    {
        // enable 24 bpp rotation, initial rotation value is set to zero
        regw(DST_CNTL, DST_24_ROTATION_ENABLE | DST_LAST_PEL |
                       DST_Y_TOP_TO_BOTTOM | DST_X_LEFT_TO_RIGHT);
    }
    else
    {
        // disable 24 bpp rotation
        regw(DST_CNTL, DST_LAST_PEL |
                       DST_Y_TOP_TO_BOTTOM | DST_X_LEFT_TO_RIGHT);
    }

    // setup engine source registers
    regw(SRC_OFF_PITCH, ((temp << 22) & 0xffc00000) | PGL_modecfg.memory_offset);
    regw(SRC_Y_X, 0);
    regw(SRC_HEIGHT1_WIDTH1, 1);
    regw(SRC_Y_X_START, 0);
    regw(SRC_HEIGHT2_WIDTH2, 1);
    regw(SRC_CNTL, SRC_LINE_X_LEFT_TO_RIGHT);

    PGL_waitforemptyfifo();

    // default HOST control
    regw(HOST_CNTL, 0);

    // default pattern control
    regw(PAT_REG0, 0);
    regw(PAT_REG1, 0);
    regw(PAT_CNTL, 0);

    // setup engine scissors (clip region) using 'yres' and 'pitch'
    if (PGL_modecfg.bpp == 24)
    {
        // if 24 bpp mode, increment width by 1
        PGL_setclipsize(0, 0, PGL_modecfg.pitch, PGL_modecfg.yres-1);
    }
    else
    {
        PGL_setclipsize(0, 0, PGL_modecfg.pitch-1, PGL_modecfg.yres-1);
    }

    // set foreground color to WHITE
    PGL_setfgcolor(0xffffffff);

    // set background color to BLACK
    PGL_setbgcolor(0);

    // default write mask
    PGL_setwritemask(0xffffffff);

    // set foreground mix to SOURCE (overpaint)
    PGL_setfgmix(S_MIX);

    // set background mix to DESTINATION (leave alone)
    PGL_setbgmix(D_MIX);

    // wait for more fifo entries
    PGL_waitforemptyfifo();

    // set engine source path to the foreground color register
    regw(DP_SRC, FRGD_SRC_FRGD_CLR);

    // default color compare control
    regw(CLR_CMP_CLR, 0);
    regw(CLR_CMP_MASK, 0xffffffff);
    regw(CLR_CMP_CNTL, COMPARE_DESTINATION | COMPARE_FALSE);

    // set the default line pattern to 'solid' line
    PGL_setlinetype(SOLID_LINE);

    // set the default fill pattern to 'solid'
    PGL_setfilltype(SOLID_FILL);
    PGL_attr.fillpatternwidth = 16;
    PGL_attr.fillpatternheight = 16;
    for (i = 0; i < 96; i++)
    {
        PGL_attr.fillpattern[i] = 0xffffffff;
    }

    // setup default line, rectangle, and polygon pattern source coordinates
    PGL_attr.srcline_y = PGL_modecfg.yres;
    PGL_attr.srcrect_y = PGL_modecfg.yres;
    PGL_attr.srcpoly_y = PGL_modecfg.yres;

    // set CRT memory offset without changing CRT pitch
    regw(CRTC_OFF_PITCH, (regr(CRTC_OFF_PITCH) & 0xffc00000) | PGL_modecfg.memory_offset);

    // insure engine is idle and fifo is empty before returning
    PGL_waitforidle();
}

