/*==========================================================================
* INITAPP.C - Copyright (c) 1994 ATI Technologies Inc. All rights reserved *
*                                                                          *
* PGL functions to initialize the apertures (VGA or Linear) for memory     *
* mapped register routines REGW() and REGR().                              *
* ======================================================================== */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <conio.h>
#include <memory.h>
#include <dos.h>
#include <malloc.h>

#include "..\inc\atim64.h"
#include "..\inc\pgl.h"
#include "..\inc\pglglob.h"

/* --------------------------------------------------------------------------
  PGL_initaperture - configure Mach64 apertures according to memory mapped
                     register access method
  
  This function turns on the appropriate aperture according to the memory
  mapped register access method and initializes the PGL internal structures
  relating to memory mapped register access. This function MUST must be
  called before calling the REGW() or REGR() functions if you only intend
  to access memory mapped registers without using any engine draw functions.

  Input parameters:

    Register Access:

        USE_VGA_APERTURE    - use VGA apeture method for register access
        USE_LINEAR_APERTURE - use LINEAR aperture method for register access
        AUTO_APERTURE       - let PGL determine the best method

  Return codes:

    NO_ERROR           - aperture settings completed successfully
    NO_APERTURE        - no aperture exists to read & write memory mapped
                         registers or aperture address is zero
    QUERY_FAILED       - hardware query call to Mach 64 ROM failed
    FAILED_REG_TEST    - failed memory mapped register test
-------------------------------------------------------------------------- */
int PGL_initaperture(int register_access)
{
    int query_size;
    int query_error;
    int retval;
    unsigned int save_gen_cntl;
    unsigned int save_vga_reg;
    unsigned long save_gui;
    unsigned char *ptr;

    // fill query structure by calling Mach 64 ROM - needed for VGA status
    query_error = QUERY_FAILED;
    PGL_blankscreen();
    query_size = PGL_getquerysize(HEADER_ONLY);
    if (query_size != 0)
    {
        // allocate memory buffer for query structure
        ptr = (unsigned char *) malloc (query_size);
        if (ptr != NULL)
        {
            // call the BIOS to fill memory
            if (PGL_queryhardware(HEADER_ONLY, (void far *)ptr) == NO_ERROR)
            {
                // fill global query structure from memory buffer
                memcpy(&PGL_querydata, ptr, sizeof(PGL_queryheader));

                // set no error condition
                query_error = NO_ERROR;
            }
            free(ptr);
        }
    }
    PGL_unblankscreen();
    if (query_error == QUERY_FAILED)
    {
        return (QUERY_FAILED);
    }

    // set special DAC handling type
    switch(PGL_querydata.dac_type)
    {
        case DAC_TVP3026_1:
        case DAC_TVP3026_2:
            PGL_modecfg.dac_attribute = DAC_SP_TVP3026;
            break;

        case DAC_ATI68860_1:
        case DAC_ATI68860_2:
            PGL_modecfg.dac_attribute = DAC_SP_ATI68860;
            break;

        case DAC_IBM514:
            PGL_modecfg.dac_attribute = DAC_SP_IBM514;
            break;

        default:
            PGL_modecfg.dac_attribute = DAC_SP_STANDARD;
            break;
    }

    // ---- determine status of VGA and linear apertures ----

    // check for valid register access code
    if (register_access == USE_VGA_APERTURE)
    {
        if (PGL_querydata.vga_type != VGA_ENABLE)
        {
            return (NO_APERTURE);
        }
    }

    // set aperture status flags according to register access code
    if (PGL_modecfg.chip_attribute == CHIP_SP_GXE_FLOAT)
    {
        //
        // GX rev E with floating I/O:
        // - ioCONFIG_CNTL can't be accessed
        // - linear aperture is enabled by default (8M size/16M boundary)
        // - VGA aperture is not supported
        // - aperture address must be fetched from the ROM query header (this
        //   not as desireable as reading the register because the address
        //   can't changed as easily)
        //

        // this configuration does not support the VGA aperture
        if (register_access == USE_VGA_APERTURE)
        {
            return (NO_APERTURE);
        }

        // get linear aperture address from ROM query header (in Mbytes)
        PGL_modecfg.aperture_address = (unsigned long)PGL_querydata.aperture_addr << 20;

        // setup linear aperture status
        PGL_modecfg.linear_memreg_offset = 0x7ffc00;
        PGL_modecfg.linear_aperture_size = APERTURE_8M_ENABLE;
        PGL_modecfg.linear_aperture_status = LINEAR_APERTURE_ENABLED;

        // update memory mapped register dependent variables
        pgl_updateappvars();
    }
    else
    {
        // standard chip handling
        switch (register_access)
        {
            case USE_VGA_APERTURE:
                PGL_disablelinearaperture();
                PGL_enablevgaaperture();
                break;

            case USE_LINEAR_APERTURE:
                PGL_disablevgaaperture();
                pgl_autolinearaperture();
                break;

            case AUTO_APERTURE:
                pgl_setupaperture();
                break;
        }
        PGL_updateaperturestatus();
    }
    PGL_modecfg.register_access_method = register_access;

    // ---- Return with error if no aperture is available ----

    // return error if no aperture is enabled
    if ((PGL_modecfg.linear_aperture_status == LINEAR_APERTURE_DISABLED) &&
        (PGL_modecfg.vga_aperture_status == VGA_APERTURE_DISABLED))
    {
        return (NO_APERTURE);
    }

    // disable linear aperture if aperture address is zero
    if ((PGL_modecfg.linear_aperture_status == LINEAR_APERTURE_ENABLED) &&
        (PGL_modecfg.aperture_address == 0))
    {
        PGL_disablelinearaperture();

        // if VGA aperture is not enabled, exit with error
        if (PGL_modecfg.vga_aperture_status == VGA_APERTURE_DISABLED)
        {
            return (NO_APERTURE);
        }
    }

    // ---- GUI memory mapped register integrity check ----

    // assume failure
    retval = FAILED_REG_TEST;

    // save GEN_TEST_CNTL register and enable engine to access GUI registers

    // Mach64 CT, DT variants have engine enabled at all times
    save_gen_cntl = ior16(ioGEN_TEST_CNTL);
    iow16(ioGEN_TEST_CNTL, save_gen_cntl | GUI_ENGINE_ENABLE);

    // If the VGA aperture is being used, the VGA controller must be set
    // to 128K size to gain access to the GUI register area for this test
    if (PGL_modecfg.vga_aperture_status == VGA_APERTURE_ENABLED)
    {
        // set controller to 128K size to access GUI regs
        outp(0x3ce, 6);
        save_vga_reg = inp(0x3cf);
        outpw(0x3ce, 0x106);
    }

    // save PAT_REG0 GUI register contents
    save_gui = regr(PAT_REG0);

    // do test - check first write for errors
    if (regw(PAT_REG0, 0x55555555) == NO_ERROR)
    {
        if (regr(PAT_REG0) == 0x55555555)
        {
            regw(PAT_REG0, 0xaaaaaaaa);
            if (regr(PAT_REG0) == 0xaaaaaaaa)
            {
                retval = NO_ERROR;
            }
        }

        // restore PAT_REG0 GUI register contents
        regw(PAT_REG0, save_gui);
    }

    // restore VGA controller size if changed
    if (PGL_modecfg.vga_aperture_status == VGA_APERTURE_ENABLED)
    {
        // restore VGA controller size
        outpw(0x3ce, (save_vga_reg << 8) | 6);
    }

    // restore GEN_TEST_CNTL contents
    iow16(ioGEN_TEST_CNTL, save_gen_cntl);

    // close apertures if failure occurred
    if (retval != NO_ERROR)
    {
        PGL_closeaperture();
    }

    return (retval);
}

/* --------------------------------------------------------------------------
  PGL_closeaperture - turns off both VGA and LINEAR Mach64 apertures.

  This function is normally called before exiting from the application if
  PGL_initaperture() was called previously.
-------------------------------------------------------------------------- */
void PGL_closeaperture(void)
{
    // disable apertures
    PGL_disablevgaaperture();
    PGL_disablelinearaperture();
}

/* --------------------------------------------------------------------------
  pgl_autolinearaperture - sets linear aperture according to query data from
                           ROM
  
  This function enables the linear aperture if the ROM reports 4 or 8M
  aperture size. If not, it is disabled. The query structure MUST be
  filled before calling this function.
-------------------------------------------------------------------------- */
void pgl_autolinearaperture(void)
{
    // Enable linear aperture according to ROM aperture configuration
    switch(PGL_querydata.aperture_cfg & 3)
    {
        case 1:
            PGL_enablelinearaperture(APERTURE_4M_ENABLE);
            break;

        case 2:
            PGL_enablelinearaperture(APERTURE_8M_ENABLE);
            break;

        case 0:
        default:
            PGL_disablelinearaperture();
            break;
    }
}

/* --------------------------------------------------------------------------
  pgl_setupaperture - sets the register access method by hardware detection
  
  This function selects the aperture according to the installed hardware
  for best performance and functionality. Note that the global query
  structure MUST be filled by calling PGL_queryhardware() before calling
  this function.
-------------------------------------------------------------------------- */
void pgl_setupaperture(void)
{
    // Enable linear aperture according to ROM query
    pgl_autolinearaperture();

    // If the VGA controller is enabled, enable the VGA aperture
    if (PGL_querydata.vga_type == VGA_ENABLE) // VGA enable check
    {
        // enable VGA aperture
        PGL_enablevgaaperture();
    }
    else
    {
        // disable VGA aperture requires disabling
        PGL_disablevgaaperture();
    }
}

/* --------------------------------------------------------------------------
  PGL_updateaperturestatus - update aperture status for PGL register access
  
  This function updates the aperture status fields in the PGL global
  structure according to the aperture bit values in the CONFIG_CNTL
  register. This will determine the method used for memory mapped registers
  by the PGL.
-------------------------------------------------------------------------- */
void PGL_updateaperturestatus(void)
{
    unsigned long config;

    /* ---- determine status of VGA and linear apertures ---- */
    config = ior32(ioCONFIG_CNTL);

    // determine addresses and offsets for linear aperture
    if ((PGL_modecfg.chip_type == CHIP_CT_ID) ||
        (PGL_modecfg.chip_type == CHIP_DT_ID))
    {
        // address is in 16Meg increments
        PGL_modecfg.aperture_address = (config & 0x3fc0) << 18;

        // insure that this bit is not set for Mach64 CT & DT variants
        config = config & (~APERTURE_4M_ENABLE);
    }
    else
    {
        // address is in 4Meg increments
        PGL_modecfg.aperture_address = (config & 0x3ff0) << 18;
    }

    // assign linear aperture size and memory mapped register offset
    if ((config & APERTURE_4M_ENABLE) == APERTURE_4M_ENABLE)
    {
        // 4M size
        PGL_modecfg.linear_memreg_offset = 0x3ffc00;
        PGL_modecfg.linear_aperture_size = APERTURE_4M_ENABLE;
    }
    else
    {
        // 8M size
        PGL_modecfg.linear_memreg_offset = 0x7ffc00;
        PGL_modecfg.linear_aperture_size = APERTURE_8M_ENABLE;
    }

    // determine linear aperture status and address
    if ((config & 3) != 0)
    {
        PGL_modecfg.linear_aperture_status = LINEAR_APERTURE_ENABLED;
    }
    else
    {
        PGL_modecfg.linear_aperture_status = LINEAR_APERTURE_DISABLED;
    }

    // assign memory mapped register offset for vga aperture
    PGL_modecfg.vga_memreg_offset = 0xb000fc00;    // segment:offset

    // determine vga aperture status
    if ((config & VGA_APERTURE_ENABLE) == VGA_APERTURE_ENABLE)
    {
        PGL_modecfg.vga_aperture_status = VGA_APERTURE_ENABLED;
    }
    else
    {
        PGL_modecfg.vga_aperture_status = VGA_APERTURE_DISABLED;
    }

    // update memory mapped register dependent variables
    pgl_updateappvars();
}

