#define	PROGRAM_NAME	"fdcch_ci.c"
#define	VERSION	1.01

// rdcch_ci.c
//
//   RDCCH C/I PERFORMANCE TEST PROCEDURE
//
//   Developed to quantify the C/I performance of the Forward DCCH
//   and the effects of various Access Parameters on that performance.
//
//		Usage:
//			fd n c
//
//		Where:
//			n = file number (see file naming convention below)
//			c = forward channel number (see below)
//
//		Output File Naming convention
//			Files are named according to date, number (supplied above),
//			The file is stored in the DATA directory of the current
//			 working directory (DATA\) with name of:
//				mmddnn.log   <test results log file>
//			where:
//				mmdd is the captured month, date
//				nn is the file number specified as command line argument
//
//		NOTE: Will not overwrite an existing file
//
//		File Format (Output)
//			All fields are seperated by tabs and units of measure are
//				km/hr, dBm, and ratio values depending upon type of
//				measurement.
//			The format of each line is:
//			  Channel \t Doppler(km/hr) \t Correlation \t ARQ \t DIC \t 
//			  Max Retries\t Max Repetitions \t Max Busy/Reserved \t
//			  Max Counters \t C/I ratio \t BER \t WER \t RSS(dBm) \t
//         Audit Failures \t SPACH Failures \t RDATA Failures \n.
//
//

// =================================================================


#include <stdio.h>
#include <stdlib.h>
#include	<math.h>
#include	<string.h>
#include	<time.h>
#include <conio.h>

// Define control parameters
#define	TRUE	1
#define	FALSE	0

#include "gpib_lib.c"	  // basic gpib routines
#include "ifr_lib.c"		  // ifr routines
#include "hp11713.c"	     // attenuator controller routines 
#include "hp11759b.c"	 //  hp channel simulator routines

// User entered information and key variables
FILE	*data_fileg, *data_filed, *data_files;		// data file definition
#define OUTPUT_PATH "DATA\\"
#define LOG_EXT ".log"
#define DATA_EXT ".dat"

// Define Parameters

#define CARRIER_RF_DB  -25	   /* initial carrier signal power */
#define INTERF_RF_DB	  -25    /* initial interferer signal power */
#define DEFAULT_X_ATTEN1   10     /* default x attenuator 1 (C/I = inf)*/
#define DEFAULT_Y_ATTEN1	110	 /* default y attenuator 1 (C/I = inf)*/
#define DEFAULT_X_ATTEN2   0     /* default x attenuator 1 (C/I = inf)*/
#define DEFAULT_Y_ATTEN2   0	 /* default y attenuator 1 (C/I = inf)*/
#define START_X_ATTEN1    10     /* start x attenuator 1 */
#define START_Y_ATTEN1	 100	 /* start y attenuator 1 */
#define START_X_ATTEN2    0     /* start x attenuator 1 */
#define START_Y_ATTEN2    0	 /* start y attenuator 1 */
#define C_I_DELTA 2				/* C/I delta increments */
#define MAX_CONSC_FAILS	 2    /* # consecutive audit failures to assume scanning*/
#define MIN_C_I			11   /* minimum C/I level for testing */

/* Define the GPIB address */
#define	HPFADESIM_DEV	11		/* GPIB logical device number */
#define	HP_ATT1	2				/* HP attenuator controller #1 */
#define	HP_ATT2	4				/* HP attenuator controller #2 */

/* CHANSIM Parameters */
typedef struct {
					  int	profile;
                 int doppler;
                 double	corr;
                 char terrain[MAX_MESSAGE_LENGTH];
               } CHANSIM_PARAMS;


// Define file formats
#define	STD_HEAD1	"#  Proprietray Information\n#  Property of SOUTHWESTERN BELL TECHNOLOGY RESOURCES INC\n"
#define	STD_HEAD2	"#  written by program %s Version %.2f\n", PROGRAM_NAME, VERSION



int process_command_line_arguments (int argc, char **argv, int *meas_number, int *ch_number)
/* read and do any and all command line arguments from dos */
/*  Returns TRUE if all arguments were valid */
{
	/* Required are three arguments */
	if (argc != 3)
	{
		printf ("Arguments Required: number(1-99) \n");
		return (!TRUE);
	}

	/* Transfer measurement number and channel number back */
	*meas_number = atoi (argv[1]);
	*ch_number = atoi (argv[2]);
	

	return (TRUE);
}	/* end process_command_lines */



build_base_file_name (int number, char *output)
/* Builds the base file name of mmddnnbn */
{
	char date_buffer[10];
	char	temp_buffer[10];

	/* Build mmdd */
	_strdate(date_buffer);	/* get current date */
	strncpy (temp_buffer, date_buffer, (size_t) 2);	/* get month (mm) */
	strncpy (&temp_buffer[2], &date_buffer[3], (size_t) 2);	/* get date (dd) */
	temp_buffer[4] = '\0';	/* end of string */

	strcpy (output, OUTPUT_PATH);
	strcat (output, temp_buffer);

	/* Add the measurement number */
	sprintf (temp_buffer, "%2d", number);
	if (temp_buffer[0] == ' ')
		temp_buffer[0] = '0';  /* add leading zero if needed */
	strcat (output, temp_buffer);

}  /* build_data_file_name */

int open_all_files(int number)
/* Open all output files checking to see if it exists first.  Returns
 *   TRUE if all files opened OK.
*/
{
	char	base_file_name[64];
	char	file_name[64];
	int	any_exist = !TRUE;

	build_base_file_name (number, base_file_name);

printf ("Base name %s\n", base_file_name);

/* 1. Open log file */
	strcpy (file_name, base_file_name);
	strcat (file_name, LOG_EXT);  /* summary file extension added */

	/* See if the file exists by opening as a read only */
	data_fileg = fopen (file_name, "r");

	if (data_fileg == NULL)
	{  /* File does not exist */
		/* create it, just leave it blank */
		data_fileg = fopen (file_name, "w");
		printf ("%s Opened.\n", file_name);
	}
	else
	{  /* file exists, need to pick out next Meas_ID */
		fclose (data_fileg);	/* close it */
		any_exist = TRUE;
		printf ("ERROR: File Exists: %s\n", file_name);
	}

/* 2. Open data file */
	strcpy (file_name, base_file_name);
	strcat (file_name, DATA_EXT);  /* data file extension added */

	/* See if the file exists by opening as a read only */
	data_filed = fopen (file_name, "r");

	if (data_filed == NULL)
	{  /* File does not exist */
		/* create it, just leave it blank */
		data_filed = fopen (file_name, "w");
		printf ("%s Opened.\n", file_name);
	}
	else
	{  /* file exists, need to pick out next Meas_ID */
		fclose (data_filed);	/* close it */
		any_exist = TRUE;
		printf ("ERROR: File Exists: %s\n", file_name);
	}


	return (!any_exist);
}



close_files(void)
{
	fclose (data_fileg);
  	fclose (data_filed);

}
void input_chansim_params (CHANSIM_PARAMS *chansim){
char input_msg[28];

	printf("\nInput Channel Simulator Parameters ..... \n");

	printf("Enter profile description: ");
	scanf("%s", chansim->terrain);

	printf("Enter profile number: ");
	scanf("%s", input_msg);
	chansim->profile = atoi(input_msg);
	
	printf("Enter correlation: ");
	scanf("%s", input_msg);
	chansim->corr= atof(input_msg);

	printf("Enter doppler(km/hr): ");
	scanf("%s", input_msg);
	chansim->doppler = atoi(input_msg);

}
write_header_files (void){
/* write the output header to file */
	fprintf(data_filed, "CHAN\t DOPPL\t CORR\t TW_POW\t ARQ\t DIC\t Retries\t Reps\t Busy\t Count\t C/I\t ");
   fprintf(data_filed, "Audit_CNT\t MACA_CNT\t SYNCH_FAIL\t BER\t WER\t RSS\t Audit_Fail\t Maca_Fail\n");
}

adjust_step_atten(int atten_ud, int *atten_x_dB, int *atten_y_dB, int step){
/* controls step attenuators thru HP11713A attenuator/switch driver */
	int mask = TIMO;
	int atten_adj_dB, step_y_dB=10;
	int def_atten_y = DEFAULT_Y_ATTEN1;

	/* set timeout to 3 sec */
	gpib_tmo(atten_ud, 3);

/* Reset X dB attenuator to 10 if < 0 dB, and decrease Y attenuator */
/* Y attenuator (10 dB steps) */
	if(*atten_x_dB < 0){
		*atten_y_dB = *atten_y_dB - 10;
		*atten_x_dB = *atten_x_dB + 10;
	}

	atten_set_x(atten_ud, *atten_x_dB);		// adjust x attenuation
	printf("  step x attenuation = %d dB .....\n", *atten_x_dB);
	fprintf(data_fileg, "X atten = %d dB .....\n", *atten_x_dB);

	printf("  step y attenuation = %d dB .....\n", *atten_y_dB);
	fprintf(data_fileg, "Y atten = %d dB .....\n", *atten_y_dB);


	/* If step set attenuation y to highest level and then step down by 10 dB steps*/
	if(step) {
		atten_set_y(atten_ud, def_atten_y);		// adjust y attenuation

		while((def_atten_y - step_y_dB) >= *atten_y_dB){
			/* step attenuator by 10 dB */
			atten_set_y(atten_ud, def_atten_y -step_y_dB);

			printf("\tstepping %d .....\n", step_y_dB);
			fprintf(data_fileg,"  stepping %d .....\n", step_y_dB);

			/* wait 3 seconds for channel to stabilize */
			gpib_wait(atten_ud,mask);

			/* increase attenuation by 10 dB*/
			step_y_dB = step_y_dB + 10;
		}
	}
	/* step down full attenuation */
	else	
		atten_set_y(atten_ud, *atten_y_dB);		// adjust y attenuation

}

write_output_files (int C_I, int TX_POW, ACCESS_PARAMS *acc, MACA_PARAMS *maca, CHANSIM_PARAMS *chnsm)
/* Write the output files */
{
	char access_strng[MAX_MESSAGE_LENGTH];
	char audit_strng[MAX_MESSAGE_LENGTH];

	/* access parameters */
	sprintf(access_strng, "%s\t %d\t %f\t %d\t %d\t %d\t %d\t %d\t %d \t %d\t %d\t ",  
	chnsm->terrain, chnsm->doppler, chnsm->corr, TX_POW, acc-> burst, acc->DelayInterval,
	acc->MaxRetries, acc->MaxReps, acc->MaxBusy, acc->MaxStop, C_I);

	/* audit/ maca reports */
	sprintf(audit_strng, "%d\t %d\t %d\t %f\t %f\t %f\t %f\t %f\n", maca->audit_conf_cnt, maca->rpt_cnt, 
	maca->synch_fail_cnt, maca->avg_BER, maca->avg_WER, maca->avg_RSS, maca->audit_conf_fail_rate, maca->rpt_cnt_fail_rate);

	/*concatenate strings */
	strcat(access_strng, audit_strng);

	/*print to log file */
	fprintf(data_filed, access_strng);

}
 
int parse_confirmation_msg (int ifr_ud, char response[], MACA_PARAMS *maca) {
	char *token, tokensep[]=";";
	int i, mask = TIMO;
	int registered = FALSE, exit_loop = FALSE;
	double ber, wer, rss;
	int ifr_dev = IFR_DEV;		// GPIB logical device number for IFR 1600CSA
	int def_atten_x = DEFAULT_X_ATTEN1;
	int def_atten_y = DEFAULT_Y_ATTEN1;
	int maca_rpt = FALSE;
	char token_resp[MAX_MESSAGE_LENGTH];

	/* copy response to token string */
	strcpy(token_resp, response);

	/* separate response messages */

	token = strtok(token_resp,tokensep);
	i = 0;	 			 						/* initialize token count */


   while(token != NULL & i <= 2){				/* loop thru until no tokens */
		/* initialize exit loop */
		exit_loop = FALSE;
		/* loop through messages and compare to token */
		while(!exit_loop){
			switch (i)
			{
				case 0:	/* audit confirmation message */
							if(ifr_ask_mess_rcvd(ifr_ud, token,"AUDITCON")){

								/* print audit confirmation message */
								printf("%s\\", token);
								fprintf(data_fileg,"%s\\", token);

								exit_loop = TRUE;
							}
							else {
								/* if not audit conf skip rest of msg */
								exit_loop = TRUE;
								i = 2;
							}
							break;
				case 1:	/* registration message */
							if(ifr_ask_mess_rcvd(ifr_ud, token,"REGISTRATION")){
								registered = TRUE;

								/* print registration confirmation mesage */
								printf("%s\\", token);
								fprintf(data_fileg,"%s\\", token);

								exit_loop = TRUE;
							}
							else
								exit_loop = FALSE;
							break;
				case 2:  /* maca */
							if(ifr_ask_mess_rcvd(ifr_ud, token,"MACA")){

								/* print maca confirmation message */
								printf("%s\\", token);
								fprintf(data_fileg,"%s\\", token);

								maca_rpt = TRUE;
								exit_loop = TRUE;
							}									

							else
								exit_loop = FALSE;
							break;
			default:	exit_loop = TRUE;

						printf("\\%s",response);
						fprintf(data_fileg, "\\%s", response);
						break;


				}		 /* switch */

			i++;	/* increment token counter */
		} /* while */

	 /* Get new token */
	 token = strtok(NULL, tokensep);
    } /* while */

	 i = 0;
	 printf("decode maca report ");
	 if(maca_rpt){
	   while(token != NULL & i <= 2){				/* loop thru until no tokens */

			/* Get new token */
			token = strtok(NULL, tokensep);

			switch (i)
			{
				case 0:  /* maca WER */
							/* decode WER */
							printf("decode WER");
						   wer = decode_WER(atoi(token));

							maca->avg_WER = ((maca->avg_WER * (maca->rpt_cnt - 1)) + 
							wer) / maca->rpt_cnt;

							/* print WER */
							printf("%4.2f\\", wer);
							fprintf(data_fileg,"%4.2f\\", wer);
							break;
				case 1:  /* maca BER */
							/* decode BER */
							printf("decode BER");
							ber = decode_BER(atoi(token));

							maca->avg_BER = ((maca->avg_BER * (maca->rpt_cnt - 1)) + 
							ber) / maca->rpt_cnt;

							/* print BER */
							printf("%4.2f\\", ber);
							fprintf(data_fileg,"%4.2f\\", ber);
							break;
				case 2:  /* maca Ltm RSS  */
							printf("case2:token = %s", token);
							/* decode RSS */
							if(atof(token) < 0)
								rss = 1e-13;
							else
								rss = decode_RSS(atof(token));
							printf("rss = %f", rss);
							maca->avg_RSS = ((maca->avg_RSS * (maca->rpt_cnt - 1)) + 
							rss) / maca->rpt_cnt;

							/* print RSS */
							printf("%6.2f ", 10*log10(rss));
							fprintf(data_fileg,"%6.2f", 10*log10(rss));
							break;
				default:	break;

				}		 /* switch */
			 i++;
 			} /* for loop */
		} /* if statement */
	
	return(registered);

} /* parse_confirmation_msg */

int ifr_ask2_RDCCH_msg(int ifr_ud, char message[], char response[], MACA_PARAMS *maca){
	int mask = TIMO;					// gpib timeout or request to send
	int resp_len = MAX_MESSAGE_LENGTH;	// message length
	int ifr_dev = IFR_DEV;		// GPIB logical device number for IFR 1600CSA
	int status;
	int mask_recvd;
	int error = FALSE;
	int iwait=0;
	int exit_loop = FALSE;
	char ibrsp_msg[MAX_MESSAGE_LENGTH];

	ifr_ud = gpib_open(ifr_dev);		// get the ud for the IFR 

	gpib_tmo(ifr_ud, 10);

	/* send message to IFR */
	strset(response,' ');
   gpib_write(ifr_ud, message);

	while (!exit_loop & (iwait < 3)){
  		gpib_write(ifr_ud, "disp_msgs");

  		/* record response from IFR */
  		gpib_read(ifr_ud,response,resp_len);

	   if(parse_confirmation_msg(ifr_ud, response, maca)){
				exit_loop = TRUE;
		}
		else {
			/* wait for 1 sec */
			mask_recvd = gpib_wait(ifr_ud,mask);
			printf("%d %s\n", iwait, response);
			iwait++;
		}
	}

	ifr_regaccept(ifr_ud);

	/* change mask to timeout */
	gpib_tmo(ifr_ud, 3);
	mask = TIMO;

	mask_recvd = gpib_wait(ifr_ud,mask);

	/* serial poll gpib device until rqs cleared */
  	while(mask_recvd == END_RQS_CMPL | mask_recvd == TIMO_END_RQS_CMPL){
//		printf("RQS\\");
		ibrsp(ifr_ud,ibrsp_msg); 
		mask_recvd = gpib_wait(ifr_ud,mask);
	}

	return(!exit_loop);
	
}


int ifr_audit_conf_test (int ifr_ud, int atten_ud, int *atten_x_dB, int *atten_y_dB, ACCESS_PARAMS *acc, MACA_PARAMS *maca){

	int ifr_dev = IFR_DEV;		// GPIB logical device number for IFR 1600CSA
	char response[MAX_MESSAGE_LENGTH];	// message response from IFR
	int icount = 0;							// number of iterations
	int mask = TIMO;							// timeout from IFR
	int exit_loop = FALSE;					// control loop
	int resp_len = MAX_MESSAGE_LENGTH;	// length of message
	int def_atten_x = DEFAULT_X_ATTEN1;
	int def_atten_y = DEFAULT_Y_ATTEN1;
	int consc_fail_cnt = 0;
	int synch_fail = FALSE;
	char time_buffer[9];

	/* Initialize Test Output Parameters */
	maca->passed = TRUE;
	maca->avg_BER =0.0;
	maca->avg_WER =0.0;
	maca->avg_RSS =1e-10;
	maca->audit_conf_cnt = 0;
	maca->rpt_cnt = 1;
	maca->audit_conf_fail_rate=0.0;
	maca->SPACH_conf_fail_rate=0.0;
	maca->rdata_acpt_fail_rate=0.0;
   maca->rpt_cnt_fail_rate = 0.0;
	maca->max_audit_conf_fail_rate=.1;
	maca->synch_fail_cnt = 0;

	/* Repeat audit test for NUM_ITERATIONS or until keyboard input */
	while ( (icount < NUM_ITERATIONS) & !exit_loop){

		/* wait additional 10 secs between audits */
		gpib_tmo(ifr_ud, 10);
		gpib_wait(ifr_ud,mask);

		icount++;  /*increment test counter */

		/* print time and iteration number */
		_strtime(time_buffer);
		printf("\n%s\\ %d\\", time_buffer, icount);
		fprintf(data_fileg,"\n%s\\ %d\\", time_buffer,icount);

		/* send audit confirmation msg w/ forced registration and maca enabled */
		if (ifr_ask2_RDCCH_msg (ifr_ud, "wait_for_audit_con 1", response, maca)){

			/* Increment audit count since RQS received */
//			maca->audit_conf_cnt = maca->audit_conf_cnt + 1;

			/* if RQS interrupt received parse buffer response and update message counts */
				printf("\\%s",response);
				fprintf(data_fileg, "\\%s", response);


		}
		else {
			/* print RQS not received */
			printf("Audit failure %d, RQS not received",icount - maca->rpt_cnt);
			fprintf(data_fileg,"Audit failure %d, RQS not received", icount - maca->rpt_cnt);
		}

		/* Check for keyboard input */
		exit_loop = kb_exit_loop();



	} /* while */

	if(icount > 0){
		/* compute and log failure stats */
		maca->audit_conf_fail_rate = 1 - maca->audit_conf_cnt/((double) NUM_ITERATIONS );
		maca->rpt_cnt_fail_rate = 1 - maca->rpt_cnt/((double)NUM_ITERATIONS);

		/* convert RSS to dBm */
		maca->avg_RSS = 10*log10(maca->avg_RSS);
	}

} 

rebuild_bcch(int ifr_ud, int atten_ud, ACCESS_PARAMS *acc){
	int mask = TIMO;
	int def_atten_x = DEFAULT_X_ATTEN1;
	int def_atten_y = DEFAULT_Y_ATTEN1;


	/* set C/I level to infinite by adjusting attenuation	 */
	/* set C/I level by adjusting attenuation	 */
	printf("\n\nSetting Interference attenuation ...\n");
	fprintf(data_fileg,"\nSetting Interference attenuation ...\n");
	adjust_step_atten(atten_ud, &def_atten_x, &def_atten_y,0); //interference power

	printf ("\nSetting up DCCH, please wait ...\n");

	ifr_rebuild_bcch(ifr_ud, acc);

	/* wait timeout period to setup DCCH & for mobile to camp on channel*/
	gpib_tmo(ifr_ud, 30);
	gpib_wait(ifr_ud,mask);

	
}
main(int argc, char **argv)
{
	int ifr_dev = IFR_DEV;		// GPIB logical device number for IFR 1600CSA
	int ifr_ud;					// GPIB handler for new 488 routines for IFR
	int hpfs_dev;	// GPIB logical device for HP11759B channel sim		

	int hpfs_ud;					//GPIB handler for channel simulator routines

	int	meas_number, ch_number;
	int	in_key, i;				
   int   exit_loop = FALSE;	// keyboard input loop control
	int C_I;							// C/I (dB) level
	int TX_POW_DB;					// transmit power (dBm)
	int mask = TIMO;
	int delay;

	/* Initialize Attenuator Settings */
	int atten1_x_dB =  START_X_ATTEN1;	 // 1 dB step attenuator for atten 1
	int atten1_y_dB =  START_Y_ATTEN1;  // 10 dB step attenuator for atten 1
	int atten2_x_dB =  START_X_ATTEN2;	 // 1 dB step attenuator for atten 2
	int atten2_y_dB =  START_Y_ATTEN2; 	 // 10 dB step attenuator for atten 2
	int def_atten1_x_dB =  DEFAULT_X_ATTEN1;	 // 1 dB step attenuator for atten 1
	int def_atten1_y_dB =  DEFAULT_Y_ATTEN1;  // 10 dB step attenuator for atten 1
	int def_atten2_x_dB =  DEFAULT_X_ATTEN2;	 // 1 dB step attenuator for atten 2
	int def_atten2_y_dB =  DEFAULT_Y_ATTEN2; 	 // 10 dB step attenuator for atten 2
	int att1_ud;			 	 // GPIB handler for atten 1
	int att2_ud; 			 	 // GPIB handler for atten 2

	/* Define IFR Data Structures */
	ACCESS_PARAMS acc;
	MACA_PARAMS maca;
	CHANSIM_PARAMS chansim;

	/* Initialize Access Parameters (IFR) */
	acc.MaxReps = 1;		 /* Max Repetitions */
	acc.MaxRetries = 1;	 /* Max Retries */
	acc.MaxBusy = 1;		 /* Max Busy/Reserved */
	acc.MaxStop = 1;		 /* Max Stop Counter */
	acc.DelayInterval =1; /* DIC */
	acc.burst =1;			 /* ARQ */
	
	printf ("%s Version %.2f\n", PROGRAM_NAME, VERSION);
	printf ("\nCopyright Southwestern Bell Technology Resources 1996\n\n\n\n\n");


	/* process any command line arguments */
	if (!process_command_line_arguments (argc, argv, &meas_number, &ch_number))
	{
		printf ("Command Line Error, Program aborted\n");
		exit (1);
	}

	/* Open all files */
	if (!open_all_files(meas_number))
	{
		printf ("File(s) for this number already exist\n");
		exit (1);
	}

	/* print program info to log file */
	fprintf (data_fileg,"%s Version %.2f\n", PROGRAM_NAME, VERSION);
	fprintf (data_fileg,"\nCopyright Southwestern Bell Technology Resources 1996\n\n\n\n\n");

	/* write header to data file */
	write_header_files();

	hpfs_dev = HPFADESIM_DEV;
	att1_ud = gpib_open (HP_ATT1);	/* get the ud for atten 1 */
	att2_ud = gpib_open (HP_ATT2);	/* get the ud for atten 2 */
	ifr_ud = gpib_open (ifr_dev);		/* get the ud for the IFR */
	hpfs_ud = gpib_open (hpfs_dev);  /* get the ud for the channel simulator */

	/* set high carrier level by adjusting attenuation	 */
	printf("\nSetting Carrier attenuation ...\n");
	fprintf(data_fileg,"\nSetting Carrier attenuation ...\n");
	adjust_step_atten(att2_ud, &def_atten2_x_dB, &def_atten2_y_dB,0); //carrier power

	/* rebuild bcch with current access parameters */
	rebuild_bcch(ifr_ud, att1_ud, &acc);

	/* Ask user if phone is currently powered up	*/
	printf ("\nHit 'P' to power up phone & register on DCCH %d \n or any key to continue.\n \n", ch_number);
	fprintf (data_fileg,"\nHit 'P' to power up phone & register on DCCH %d \n or any key to continue.\n \n", ch_number);


	/* Check for keyboard input */
	while (!exit_loop) {
		if(kbhit())
		{
			i = getch();
			/* if user inputs 'p' initiate registration procedure */
			if (i == 'p' | i =='P'){
				/* Power up and register phone */
				if (!registered(ifr_ud, acc.burst)) 
					printf("Phone failed to register.");
					fprintf(data_fileg,"Phone failed to register.");
			}																  					
			exit_loop = TRUE;
		}
	}

	/* set chansim parameters */
	chansim.corr = 0;
	chansim.profile = 1;
	chansim.doppler = 100;
	sprintf(chansim.terrain, "EQUAL1/4");

	while(chansim.profile <= 1) {	

//	input_chansim_params(&chansim);


	/* reset channel simulator */
	hp11759b_set_simulation (hpfs_ud);
	hp11759b_recall_user_profile (hpfs_ud, chansim.profile);

	/* Reset logical exit loop */
	exit_loop = FALSE;

	printf("\nSetting Carrier attenuation ...\n");
	fprintf(data_fileg,"\nSetting Carrier attenuation ...\n");
	adjust_step_atten(att2_ud, &atten2_x_dB, &atten2_y_dB,0); //carrier power

	/* set C/I level by adjusting attenuation	 */
	atten1_x_dB =  START_X_ATTEN1;	 // 1 dB step attenuator for atten 1
	atten1_y_dB =  START_Y_ATTEN1;  // 10 dB step attenuator for atten 1

	printf("\n\nSetting Interference attenuation ...\n");
	fprintf(data_fileg,"\nSetting Interference attenuation ...\n");
	adjust_step_atten(att1_ud, &atten1_x_dB, &atten1_y_dB,1); //interference power

	C_I = CARRIER_RF_DB - atten2_x_dB - atten2_y_dB - (INTERF_RF_DB - atten1_x_dB - atten1_y_dB);
	TX_POW_DB = CARRIER_RF_DB - atten2_x_dB - atten2_y_dB;

	/* wait timeout period for new channel to stabilize*/
	gpib_tmo(ifr_ud, 30);
	gpib_wait(ifr_ud,mask);

	/* print current parameters to screen */
 	printf("\nCHAN %s\\ DOP %d\\ TX %d\\ C/I %d\\ Max Retries %d\\ Max Reps %d\\ Max Busy %d\\ Max Stop %d\n\n",
	chansim.terrain, chansim.doppler, TX_POW_DB, C_I, acc.MaxRetries, acc.MaxReps, acc.MaxBusy, acc.MaxStop);
	fprintf(data_fileg,"\nCHAN %s\\ DOP %d\\ TX %d\\ C/I %d\\ Max Retries %d\\ Max Reps %d\\ Max Busy %d\\ Max Stop %d\n\n",
	chansim.terrain, chansim.doppler, TX_POW_DB, C_I, acc.MaxRetries, acc.MaxReps, acc.MaxBusy, acc.MaxStop);
  
/* Initialize Max Busy/Reserved Test Parameters */

	/* Find Min C/I Required for maximum value of Max Retries, Max Reps, & Max Busy */
	while(!exit_loop){
		/* audit order w/ forced registration & maca enabled	 */
   	printf( " Sending Audit Order, please wait ...\n");
   	fprintf(data_fileg, " Sending Audit Order, please wait ...\n");
		ifr_audit_conf_test(ifr_ud, att1_ud, &atten1_x_dB, &atten1_y_dB, &acc, &maca);	 
 

		/* write results to output data file */
		write_output_files(C_I, TX_POW_DB, &acc, &maca, &chansim);

		/* Check for keyboard input */
		exit_loop = kb_exit_loop();

	   /* check audit confirmation failure rate for test */
//		if(C_I <= MIN_C_I & acc.MaxBusy == 1){
//			/* reset C/I level to start value */
//			printf("\n\nSetting Interference attenuation ...\n");
//			fprintf(data_fileg,"\nSetting Interference attenuation ...\n");
//			atten1_x_dB =  START_X_ATTEN1;	 // 1 dB step attenuator for atten 1
//			atten1_y_dB =  START_Y_ATTEN1;  // 10 dB step attenuator for atten 1
//			C_I = CARRIER_RF_DB - atten2_x_dB - atten2_y_dB - (INTERF_RF_DB - atten1_x_dB - atten1_y_dB);

			/* Set Max Busy to 10 */
//			acc.MaxBusy = 10;

			/* rebuild bcch with current access parameters */
//			rebuild_bcch(ifr_ud, att1_ud, &acc);

			/* adjust C/I level */
//			adjust_step_atten(att1_ud, &atten1_x_dB, &atten1_y_dB,0);
//		}
		if (C_I > MIN_C_I){
			/* decrement C/I by attenuating interference signal */
			printf("\n\nSetting Interference attenuation ...\n");
			fprintf(data_fileg,"\nSetting Interference attenuation ...\n");
			atten1_x_dB = atten1_x_dB - C_I_DELTA;	/* X step attenuator (1 dB steps) */
			adjust_step_atten(att1_ud, &atten1_x_dB, &atten1_y_dB,0);
			C_I = CARRIER_RF_DB - atten2_x_dB - atten2_y_dB - (INTERF_RF_DB - atten1_x_dB - atten1_y_dB);

		}
		else {	/* exit if C/I < Min C/I level */
			exit_loop = TRUE;
		}
			
 		/* print current parameters to screen */
		printf("\nCHAN %s\\ DOP %d\\ TX %d\\ C/I %d\\ Max Retries %d\\ Max Reps %d\\ Max Busy %d\\ Max Stop %d\n\n",
		chansim.terrain, chansim.doppler, TX_POW_DB, C_I, acc.MaxRetries, acc.MaxReps, acc.MaxBusy, acc.MaxStop);
		fprintf(data_fileg,"\nCHAN %s\\ DOP %d\\ TX %d\\ C/I %d\\ Max Retries %d\\ Max Reps %d\\ Max Busy %d\\ Max Stop %d\n\n",
		chansim.terrain, chansim.doppler, TX_POW_DB, C_I, acc.MaxRetries, acc.MaxReps, acc.MaxBusy, acc.MaxStop);
 
		} /* while C/I > Min */
		chansim.profile++;

		}/* while chansim profile */

	/* Put channel simulator back into local mode */
   hp11759b_set_main_menu (hpfs_ud);
 	gpib_local (hpfs_ud);


	close_files();
  
}		// end main


