#define	PROGRAM_NAME	"mmds.exe"
#define	VERSION	1.10
#define	DEBUG	0

// mmds.c
//
//  Data Capture program for MMDS Measurements.
//
//  Usage:
//	  mmds [/cf:<file_name>] [/lo:<MHz>] [sa:n] [mt:seconds] ...
//				[/ag:n] [/at:cc]
// 
//   Where:
//			/cf: is the config file name, default is .\config.dat (path can be specified)
//			/lo: is override the local osc (downconversion) frequency (MHz)
//					if this arguement is present, it will ignore any config entry
//			/sa: GPIB address of spectrum analyzer (default in HP_SA).
//			/wt: Wait time (in seconds) for measurement of each channel.
//					Default is MEASURE_TIME (10 seconds).
//			/ag: Antenna gain in dB, default is 24 dB (DEFAULT_ANT_GAIN)
//			/at: Antenna type default is "P" (Parabolic) - 2 characters
//
//  Output Files:
//		NOTES:
//			(N1) All output files stored in .\data\
//			(N2) Output files are named mmdd.ext, where mmdd is month/date
//			(N3) Files are appended too if they already exist.
//				
//	   (1)  Summary file (this is mostly used output)
//				tab delimited
//				FORMAT (Per Line)
//					ID:n		 (1) Measurement ID (one per location)
//					MN:n		 (2) Measurement number (one per channel)
//					CH:ccc	 (3) Channel Code
//					FR:ffff.ffff (4) Frequency (MHz), base of channel
//					RL:xxx.x	 (5) Received signal level in dBm (95%tile)
//					HT:xx		(6) Height above ground (feet)
//					X:xxxx.xxxxx  (7) Longitude (decimal degrees) or 0
//					Y:xxxx.xxxxx  (8) Latitude (dec. deg.) or 0
//					AG:f.f		(9) Antenna Gain (floating)
//					AT:cc			(10) Antenna Type
//					d:mm/dd/yy    (11) Data of measurment
//					t:xx:xx:xx    (12) time of measurement
//
//		(2)  Detail File (this is support of summary file and contains a
//				distribution of the received signal).  Note that signal levels
//				do not have a character prefix (i.e. "ID:5").
//			   tab delimited
//				FORMAT (Per Line)
//					ID:n		 (1) Measurement ID (one per location)
//					MN:n		 (2) Measurement number (one per channel)
//					CH:ccc	 (3) Channel Code
//					FR:ffff.ffff (4) Frequency (MHz), base of channel
//					MT:n		(5) Measurement Time (seconds) of peak hold / chan.
//					xxx.x		(6) Maximum signal level
//					xxx.x		(7) 99 percentile
//					xxx.x		(8) 95 percentile
//					xxx.x		(9) 90 percentile
//					xxx.x		(10) 75 percentile
//					xxx.x		(11) 50 percentile
//					xxx.x		(12) 10 percentile
//					xxx.x		(13) 1 percentile
//					xxx.x		(14) Minimum signal level
//

//
//  Used software from SBMS Measurement System for microwave
//		activity. (/home/csp/sbms/meassys) as the shell.
//
//

/* include system stuff */
#include	<stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <stdio.h>
#include <graph.h>
#include	<time.h>
#include <search.h>
#include	<dos.h>	/* for free disk space routine */

/* include gpib handlers and devices */
#include	"gpib_lib.c"  /* basic gpib support */
#include	"hsa_prog.c"  /* hp spectrum analyzer routines */

/* necessary definitions */
#define	TRUE	1
#define	MHz	1.0E6
#define	DEFAULT_LO_FREQ	2278.0   /* Cal Amp down-converter frequency */
#define	VIDEO_OFFSET	1.25	/* Video is this above the channel base (Mhz) */

//#include	"tans_def.h"	/* declare variables for tans gps routines */
//#include	"tans_pro.c"	/* tans procedures */
//#include	"var.c"	/* development variables */
//#include	"read_con.c"	/* cameron's routines for reading config file */
//#include "ms_file.c"   /* defines output file formats and id numbers */

/* Legal stuff */
#define	SBTRI	"Southwestern Bell Technology Resources Inc. (Copyright 1996)\n"
#define	SBTRI_PROP	"PROPRIETARY--Not for release outside SBTRI\n"
#define	SBTRI_PROGRAM	"Program: %s  Version: %.2f\n\n"

/* File stuff */
#define	CONFIG_PATH	".\\"	/* where configuration files are stored */
#define	DEFAULT_CONFIG_FILE_NAME	"config.dat"
#define	DATA_PATH "data\\"	/* where data files will be stored */
#define	DATA_EXT	".fld"	/* File extension for (summary) data file */
#define	DETAIL_EXT ".det"  /* File extension for detail data file */

/* Antenna gain and type default */
#define	DEFAULT_ANTENNA_GAIN 24.0  /* dB */
#define	DEFAULT_ANTENNA_TYPE "P"  /* Parabolic of gain in above line */

/* Command Line Entries (Globals) */
char	Entered_Config_File_Name[64];  /* manually entered file name w/path */
int	Config_Override = !TRUE;  /* set true if entered */
int	LO_Override = !TRUE;  /* set true if entered on command line */
float	Antenna_Gain = (float) DEFAULT_ANTENNA_GAIN; /* 1.10 - antenna gain */
char Antenna_Type[8] = { DEFAULT_ANTENNA_TYPE };

/* Configuration File Globals */
#define NUMBER_CONFIG_CODES 3
char Config_Codes[][NUMBER_CONFIG_CODES] =  /* Order is important */
	{"LO", "CH", "AC"};

/* Instruments definitions */
#define	HP_SA	7	/* HP spectrum analyzer address (default) */
int	Hp_Sa = HP_SA;  /* Global with default address */

#define	REF_LEVEL	-10.00

double LO_Freq = DEFAULT_LO_FREQ;  /* LO_Freq, converter gain frequency to use */

#define	CAL_FACTOR	6.4  /* dBm cal factor from lab measurements */

/* These are the s/a settings for default (menu) mode */
#define	DEFAULT_FREQ_START	2500		/* Start of MMDS band */
#define	DEFAULT_FREQ_END	2696		/* Start of MMDS band */
#define 	DEFAULT_REF	-10.0	/* ref level  dBm */
#define	DEFAULT_RBW	100000.0 /* 100 kHz for resolution bw */
#define	DEFAULT_VBW	30000.0 /* 30 kHz for video bw */

/* These are the s/a settings for alignment mode */
#define	ALIGN_SPAN	0.0		/* Span */
#define 	ALIGN_REF	-10.0	/* ref level at - 0 dBm */
#define	ALIGN_SWEEP	0.020	/* sweep time mSec. */
#define	ALIGN_RBW	100000.0 /* 100 kHz for both resolution and video bw */

/* These are s/a settings for Measurement cycle */
#define	MEASURE_SPAN	0.0		/* Span */
#define 	MEASURE_REF	-20.0	/* ref level at - 20 dBm */
#define	MEASURE_SWEEP	0.020	/* sweep time mSec. */
#define	MEASURE_RBW	100000.0 /* 100 kHz for resolution bw */
#define	MEASURE_VBW	100000.0 /* 100 kHz for video bw */

#define	MEASURE_TIME 10	/* seconds - measure each channel for this long, default */
int	Measure_Wait_Time = MEASURE_TIME;  /* command line can override */


/* Table of frequencies and parameters to measure */
#define MAX_CHANNELS 33   /* Maximum table size due to number of MMDS channels */
struct MEAS_TABLE
{
	char	Chan_Code[4];
	double	Freq;
};

int	Meas_Table_Size = 0;  /* count number of entries in table */
struct MEAS_TABLE Meas_Table[MAX_CHANNELS];  /* Table of channels to measure */
struct MEAS_TABLE Align_Table = {"", 0.0};   /* Alignment Channel */

/* MMDS Lookup Table of Frequencies, Note that several of these entries
 *  May be test entries.  By adjusting the LO (downcovert frequencies), all
 *  sorts of non-MMDS entries can be accomodated.  Note that this table
 *  floats, and the actual size of the table is calculated at startup. */
int	Size_Lookup_Table = 0;  /* this is init'ed in initialize_stuff */
struct MEAS_TABLE Lookup_Table [] =
	{	{"A-1", 2500.0},
		{"A-2", 2512.0},
		{"A-3", 2524.0},
		{"A-4", 2536.0},
		{"B-1", 2506.0},
		{"B-2", 2518.0},
		{"B-3", 2530.0},
		{"B-4", 2542.0},
		{"C-1", 2548.0},
		{"C-2", 2560.0},
		{"C-3", 2572.0},
		{"C-4", 2584.0},
		{"D-1", 2554.0},
		{"D-2", 2566.0},
		{"D-3", 2578.0},
		{"D-4", 2590.0},
		{"E-1", 2596.0},
		{"E-2", 2608.0},
		{"E-3", 2620.0},
		{"E-4", 2632.0},
		{"F-1", 2602.0},
		{"F-2", 2614.0},
		{"F-3", 2626.0},
		{"F-4", 2638.0},
		{"G-1", 2644.0},
		{"G-2", 2656.0},
		{"G-3", 2668.0},
		{"G-4", 2680.0},
		{"H-1", 2650.0},
		{"H-2", 2662.0},
		{"H-3", 2674.0},
		{"C16", 132.0},
		{"", 0.0}  /* End line required */
};

/* here is the percentile structure to hold the received signal 
 *  levels.
*/
typedef struct RSL
{
	float Max;    /* Maximum value (peak) */
	float rsl99;  /* 99 percentile */
	float rsl95;  /* 95 percentile */
   float rsl90;  /* 90 percentile */
	float rsl75;  /* 75 percentile */
	float rsl50;  /* 50 percentile */
	float rsl10;  /* 10 percentile */
	float rsl01;  /* 1 percentile */
	float Min;    /* lowest value */
};

/* Measurement Data Structure - holds the measurement data before
 *  writing out to the output files.
*/
typedef struct MEASURED_DATA
{
	int Number;  /* sequential count */
	struct MEAS_TABLE code_freq;  /* holds channel code and base frequency */
	struct RSL rsl;      /* received signal distribution */
//	double	lon, lat;  /* position information */
	char	ch_meas_time[10];  /* measurement time */
	char	ch_meas_date[10];  /* measurement date */
};

struct MEASURED_DATA Measured_Data[MAX_CHANNELS];


/* Counters and other similar stuff */
int Next_Meas_ID;  /* The measurement ID variable, one per meas site */
int	Height = 0;  /* measurement height */
char	GPS_Lon_Entry[32] = {""};  /* holds last entry */
double GPS_Lon = 0.0;  /* converted to dd.dddd format */
char	GPS_Lat_Entry[32] = {""};
double GPS_Lat = 0.0;


/* misc file info (revise in future) */
//char	full_data_file_name[128] ;  /* full output data file (path + name) */
//char	data_file_name[14] = {""};	/* data file name (no path or extension) */
//int	data_file_number_input= -1;	/* output file number (from command line) */
//char	data_path[64] = DATA_PATH;	/* default path to use */
char	full_config_file_name[128] ;  /* full configuration file name (path + file) */
char	config_file_name[14] = DEFAULT_CONFIG_FILE_NAME;  /* configuration file name */
char	config_path[64] = CONFIG_PATH;
FILE	*data_file;  /* data file handler (summary file) */
FILE	*detail_file;  /* data file handler (detail file) */

/* Command Line Passed Arguments Variables and definitions */
int	NUMBER_DOS_SPECIFIERS = 6;  /* Number of valid DOS command Line Specifiers */
char	DOS_SPECIFIERS[][6] =  /* insert each DOS command line specifier here */
		{	"/cf", 	/* config file name (no path) */
			"/lo",	/* Local oscillator frequency override */
			"/sa",		/* Spectrum Analyzer GPIB Address */
			"/mt",		/* Measure time (per channel) - seconds. */
			"/ag",		/* Antenna Gain */
			"/at"		/* Antenna Type */
		};  /* end definition of specifiers--6 Char max(each), check number above */
char	SEPARATOR[]=":";

/* graphics information */
//#define	TRACE_BUFFER_LENGTH	2048	/* size of display buffer */

/*  SCREEN Positioning information and constants */

/* Prototypes (In order of occurence in source file) */
void initialize_stuff_here ();
process_command_line_arguments (int argc, char **argv);
int do_configuration_file(void);
int read_config(char *fname);
int lookup_channel (char *code);
int add_channel (char *code);
initialize_instruments (int hsa_ud);
reset_instruments (int hsa_ud);
void check_existing_data_file ();
int do_main_menu (int dev);
clear_screen();
display_parameters();
display_menu(int start_row);
display_align_screen();
default_set (int dev);
align_set (int dev);
measure_set (int dev, double base_freq);
int do_measurements(int dev);
display_measuring_screen ();
int wait_it_out ();
sort_trace (int trace[], int length);
int mycompare ( int *arg1, int *arg2);
display_measurement_summary (int trace[], int length, double freq);
int index_from_percentile (double percentile, int length);
build_meas_structure (int traceB[], int length, int i);
float cal_factor (float rsl_in, double freq);
int ask_height();
ask_X();
ask_Y();
int gps_decode(char text[], double *value);
int count_gps_seperators(char text[]);
int ask_all_is_OK();
write_summary_file ();
write_detail_file ();



main(int argc, char **argv)
{
	int hsa_ud;	/* device for spec analy and controllers */
	int	exit_loop = !TRUE;	/* exit measurement loop */

/* Print greetings here */
	printf (SBTRI);
	printf (SBTRI_PROP);
	printf (SBTRI_PROGRAM, PROGRAM_NAME, VERSION);

/* process any command line arguments */
	if (!process_command_line_arguments (argc, argv))
	{
		printf ("Command Line Error, Program aborted\n");
		exit (1);
	}

/* Do any special initializations here */
	initialize_stuff_here ();

/* set configuration file name file.  If a command line was entered
 *  specifying a specifid config, use that (Config_Override TRUE), else
 *  use the default value */
	if (Config_Override)
	{  /* comand line override */
		strcpy (full_config_file_name, Entered_Config_File_Name);
	}
	else
	{  /* use default */
		strcpy (full_config_file_name, config_path);
		strcat (full_config_file_name, config_file_name);
	}

/* Process the configuration file */
	if (!do_configuration_file()) {
		printf ("\n\nError in configuration file -- Program aborted \n");
		exit (1);
	}  /* end do configuration */	

/* Start the serial connection to instruments */
	gpib_start ();

// Open up the connection to the spectrum analyzer and attenuation controllers
	hsa_ud = gpib_open (Hp_Sa);	/* get the handler (access number) */
	if (hsa_ud < 0) {
		printf ("Error in opening connection to instrument(s)\n");
		printf ("Spectrum analyzer code: %d\n", hsa_ud);
		printf ("Check cables and power to instruments\n\nProgram Aborted...\n");
		exit (1);
	}

/* Initialize instrument settings */
	printf ("\n\nInitializing Instruments\n");
	initialize_instruments (hsa_ud);

/* Go check for any existing output file, and determine the next
 *  available measurement id. */
	check_existing_data_file();

	while (!exit_loop)
	{
		exit_loop = do_main_menu(hsa_ud);

	}	/* while */

/* Shutdown the gpib interface and disable serial link */
	reset_instruments (hsa_ud);
	gpib_local(Hp_Sa);		/* set this device to local */
	gpib_stop();

	exit (0);	/* exit w/o error */
}  /* main */

void initialize_stuff_here ()
/* Do any special initialization things here */
{
	int i;

	/* We need to count the number of entries in the lookup table */
	i = 0;

/* list channel list with strlen */
//for (i=0; i < 35; i++)
//{
//printf ("%d %s %d\n", i, Lookup_Table[i].Chan_Code, strlen(Lookup_Table[i].Chan_Code));
//getch();
//}

//exit(1);


	while (strlen(Lookup_Table[i].Chan_Code) > 0)
		i++;
	Size_Lookup_Table = i;  /* Set Size */
	
	printf ("%d Channels in Lookup Table\n", i);

}  /* initialize_stuff_here */

int process_command_line_arguments (int argc, char **argv)
/* read and do any and all command line arguments from dos */
/*  Returns TRUE if all arguments were valid */
{
	int	i, arg, specifier_number = -1;
	int	result = TRUE;  /* default if no error */
	char	*token;	/* for searching for specifier */
	char	buffer[64];

	for (arg=1; arg<argc; arg++) {
		printf ("Arg.%d in: <%s>\n", arg, argv[arg]);
		strcpy (buffer, argv[arg]);	/* copy to buffer for later analysis */
		strlwr (buffer);	/* make sure in lower case */
		token = strtok (buffer, SEPARATOR);	/* see if token (":") is in string */
/* now find if any of the tokens match a valid DOS_SEPARATOR command */
		for (i=0; i < NUMBER_DOS_SPECIFIERS; i++) {
			if (strcmp(token, &DOS_SPECIFIERS[i][0]) == 0) 
			{ /* the token matches */
				specifier_number = i;
				break;
			}	/* end for match found */
		}	/* end for NUMBER_DOS_SPECIFIERS */
/* now perform specified action */
		switch (specifier_number) {
			case 0:	/* config file name (no path) */
				strcpy (Entered_Config_File_Name, &buffer[strlen(token)+1]);
				Config_Override = TRUE;
				break;
			case 1:	/* Local Oscillator frequency (MHz) */
				LO_Freq = atof (&buffer[strlen(token)+1]);
				LO_Override = TRUE;
				break;
			case 2:	/* GPIB address of spectrum analyzer */
				Hp_Sa = atoi (&buffer[strlen(token)+1]);
				break;
			case 3:	/* Measurement wait time (seconds) */
				Measure_Wait_Time = atoi (&buffer[strlen(token)+1]);
				break;
			case 4:	/* Antenna Gain (dB) */
				Antenna_Gain = atof (&buffer[strlen(token)+1]);
				break;
			case 5:	/* Antenna Type (2 characters) */
				strcpy (Antenna_Type, &buffer[strlen(token)+1]);
				break;

#ifdef JUNK
			case 2:	/* data file nnumber 06.12 */
				data_file_number_input = atoi (&buffer[strlen(token)+1]);
				break;
			case 3:	/* data file path (no file name) */
				strcpy (data_path, &buffer[strlen(token)+1]);
				break;
			case 4: 	/* measurement index */
				measurement_index = atoi (&buffer[strlen(token)+1]);
				break;
#endif
			default:
				printf ("Invalid Command Line (on next line):\n [%s]\n", argv[arg]);
				result = !TRUE;  /* set error on exit */
				break;
		} /* end switch */
	}
	return (result);
}	/* end process_command_lines */

int do_configuration_file(void)
/*  Initialize, load, and verify the configuration file contents for
     the set of measurements.  File name is contained in config_file_name.
*/
{
	int	log;  /* just a place keeper */

	if (!read_config (full_config_file_name))
	{  /* error in file reading of configuration file */
		return (!TRUE);
	}

	/* Verify that we have at least one channel */
	if (Meas_Table_Size <= 0)
	{  /* No channels were specified to measure */
		printf ("No Channels were specified to measure\n");
		return (!TRUE);
	}

	/* If no alignment channel was specified, then use the first
	 *  channel specified in the table */
	if (strlen (Align_Table.Chan_Code) == 0)
	{  /* Assume no entry made */
		Align_Table = Meas_Table[0];  /* Use first entry */
		printf ("No Alignment Channel specified, using first CH: %s %.5lf\n",
			Align_Table.Chan_Code, Align_Table.Freq);
	}
	return (TRUE);

}  /* end do_configuration_file */

int read_config(char *fname)
//  Procedure to read configuration file (passed in "fname").
//  Returns TRUE if config file opened and read OK.
//  Does not do explicit validity checks of variables, just loads structures.
{
	char line[80], line2[80], *pos1;
	int result = TRUE, errors = 0, ltype, i;
	int line_number = 0;
	FILE *fp;

	printf ("\nConfig file <%s>", fname);
	if((fp=fopen(fname, "r")) == NULL)  {    /*  open config file  */
		printf("\n** Error opening Config File <%s> **\n",fname);    /*  return if can't  */
		result=!TRUE;
		return(result);
	}
	printf (" Opened..Reading \n", fname);
	while(fgets(line, 80, fp) != NULL)  {    /*  get one line at a time  */
		line_number++;
		/* Since a cr/lf is appended to 'line', copy into line2 and get rid of it */
		strcpy (line2, line);
		line2[strlen(line2)-1] = '\0';  /* shorten by one character */
		printf ("%2d: %s ", line_number, line2);
		if ((strlen(line) > 1) && (line[0] != '*'))
		{  /* Process this line */
			pos1=strtok(line, "\t");    /*  store first field  */
			ltype = lookup_config_type (line);
			pos1=strtok(NULL, "\t");    /*  point to argument  */
			switch (ltype)
			{
				case 0:  /* LO - Local Osc. */
					LO_Freq = atof (pos1);
					printf ("  -- OK\n");
					break;
				case 1:	/* CH - Measurement Channel */
					if (!add_channel (pos1))
					{  /* channel not found */
						printf ("\a\n\t----> Line %d: Channel Code NOT FOUND\n", line_number);
						errors++;
						result = !TRUE;
					}
					else
						printf ("  -- OK\n");
					break;
				case 2:	/* AC - Alignment channel */
					i = lookup_channel (pos1);
					if (i < 0)
					{
						printf ("\a\n\tLine %d: Alignment Channel Code NOT FOUND\n", line_number);
						errors++;
						result = !TRUE;
					}
					else
					{  /* channel codes was found */
						Align_Table = Lookup_Table[i];
						printf ("  -- OK\n");
					}

					break;
				default:  /* Error on line */
					printf ("Error on line %d\a\n", line_number);
					errors++;
					result = !TRUE;
					break;
			}  /* switch */
		}  /* if */
		else
			if (strlen(line2) > 0)
				printf (" -- OK\n");  /* do not do blank lines */
			else
				printf ("\n");
	}  /* while get one line at a time */
	if (errors > 0)
		printf ("***** %d Errors encounted in configuration file.\a\n", errors);

	fclose (fp);
	return (result);
}

int lookup_config_type (char *code)
/* Lookup the config type code (first 2 characters) from the list and
 *  return the lookup order.  If not found, returns -1 */
{
	int i;

	for (i=0; i < NUMBER_CONFIG_CODES; i++)
	{
		if (strcmp(code, Config_Codes[i]) == 0)
			return (i);
	}

	return (-1);  /* not found */
}  /* lookup_config_type */


int lookup_channel (char *code)
/* Lookup the specified channel code in the lookup table of channel codes.
 * Returns -1 if not found, else returns the index (position) where the
 *  codes was found.  This is a case sensitive search */
{
	int i;

//printf ("lookup_channel <%s>\n", code);

	for (i=0; i < Size_Lookup_Table; i++)
	{
//printf ("%d %s %s %d\n", i, code, Lookup_Table[i].Chan_Code, strcmp (code, Lookup_Table[i].Chan_Code));
//getch();
//		if (strcmp (code, Lookup_Table[i].Chan_Code) == 0)
		/* only compare up to the size of the table entry, because code had extra stuff */
		if (strncmp (code, Lookup_Table[i].Chan_Code, strlen(Lookup_Table[i].Chan_Code)) == 0)
			return (i);
	}

	return (-1);  /* not found */
}  /* lookup_channel */

int add_channel (char *code)
/* Adds a channel to the list of measured channels (Meas_Table).
 *  Ignores duplicate entries.  Inserts in freq as well.
 * Returns TRUE of item found, !TRUE if not.
*/
{
	int i;

	/* If entry already exists, just warn them and return (indicate success) */
	for (i=0; i < Meas_Table_Size; i++)
		if (strcmp (code, Meas_Table[i].Chan_Code) == 0)
		{  /* Found a match */
			printf ("Duplicate Entry for %s Found, additional entry ignored\n", code);
			return (TRUE);
		}

	/* We have no duplicate, lookup table entry and insert */
	i = lookup_channel (code);
	if (i >= 0)
	{  /* found entry */
		Meas_Table[Meas_Table_Size++] = Lookup_Table[i];
		return (TRUE);
	}

	return (!TRUE);  /* not found */
}

initialize_instruments (int hsa_ud)
/* Initialize the instruments to known initial states */
{
	char	string[32];  /* for spectrum analyzer string */
/* Spectrum analyzer initialization */
	hsa_preset (hsa_ud);		/* set to preset value (IP) */
//	hsa_title (hsa_ud, string);	// write title
	hsa_trace_format (hsa_ud, M_FORMAT);	// set M transfer (integer units)

	default_set (hsa_ud);	/* set default through this procedure */
//	hsa_sweep_mode (hsa_ud, SWEEP_SINGLE);		// set sweep mode


}  /* -------  end initialize_instruments --------------------*/

reset_instruments (int hsa_ud)
/* Release and reset any instruements now that we are done */
{
	hsa_sweep_mode (hsa_ud, SWEEP_CONTINUOUS);		// set sweep mode
	gpib_local (hsa_ud);
}	/* ------------  end reset instruments --------------------*/

void check_existing_data_file ()
/*  This will see if a data file exists for this date.  If so, this
 *  procedure will set the global (Next_Meas_ID) with the next available measurement
 *  id number.  If the file does not exists, it will create a blank file
 *  and close it so we can use the append whenever we write a new file.
 *  The file is closed upon exit. */
{
	char	file_name[32];
	char	input_line[256];
	int	no, highest = 0;
	FILE *fp;

	/* Build data file name with path and extension */
	build_data_file_name (file_name);  /* get the file name w/ path */
	strcat (file_name, DATA_EXT);  /* summary file extension added */

	/* See if the file exists by opening as a read only */
	fp = fopen (file_name, "r");

	if (fp == NULL)
	{  /* File does not exist */
		Next_Meas_ID = 1;  /* first measurement ID */
		printf ("New data file: %s", file_name);
		/* create it, just leave it blank */
		fp = fopen (file_name, "w");
		fclose (fp);	/* close it */
	}
	else
	{  /* file exists, need to pick out next Meas_ID */
		rewind (fp);
	  	while(fgets(input_line, 256, fp) != NULL)  
		{    /*  get one line at a time  */
			if (strlen(input_line) > 1 )  /* skip any blank lines */
			{  
				sscanf (input_line, "ID:%d", &no);
				if (no > highest)
					highest = no;  /* get maximum */
			}
		}  /* while */
		
		Next_Meas_ID = highest + 1;  /* save highest value found */
		printf ("Data file exists, Next Meas No is %d from %s", Next_Meas_ID, file_name);
	}
}  /* check_existing_data_file */

build_data_file_name (char *output)
/* Builds the file name, including the data path, for the (summary)
 *  data file name without extension.  The format is mmdd.
*/

{
	char date_buffer[10];
	char	temp_buffer[10];

	_strdate(date_buffer);	/* get current date */
	strncpy (temp_buffer, date_buffer, (size_t) 2);	/* get month (mm) */
	strncpy (&temp_buffer[2], &date_buffer[3], (size_t) 2);	/* get date (dd) */
	temp_buffer[4] = '\0';	/* end of string */

	strcpy (output, DATA_PATH);
	strcat (output, temp_buffer);

}  /* build_data_file_name */

int do_main_menu (int dev)
/*  Perform main menu, which includes:
 *		a) start measure	(M,m)
 *		b) align (A,a)
 *		c) quit (Q,q,ESC)
*/
{
	char	ch;
	int exit_menu = !TRUE;

	while (!exit_menu)
	{

		clear_screen();
		display_parameters();
		display_menu(10);

		ch = getch();	/* await a character */
		switch (ch)
		{
			case 'M':	/* Measure */
			case 'm':	/* Measure */
				clear_screen();
//				display_measuring_screen();
				do_measurements(dev);
				default_set(dev);	/* reset to normal */
				break;

			case 'A':	/* Align */
			case 'a':	/* Align */
				clear_screen();
				display_align_screen();
				align_set (dev);		/* set the analyzer for this mode */
				getch();		/* any key to abort */
				default_set(dev);	/* reset to normal */
				break;

			case 'Q':	/* Quit program */
			case 'q':	/* Quit program */
			case 0x1b:	/* Quit program */
				clear_screen();
				exit_menu = TRUE;
				break;

			default:		/* Other key, ignore and ask again */
				break;

		}  /* switch (ch) */

	}  /* while */


}	/* do_main_menu */

display_measuring_screen ()
{
	printf ("ID/#  Chan  Freq  RSSI\n");
}

clear_screen()
/* Clear the screen */
/*  There has to be a better way to do this!!! */
{
	short row, col=1;
	char blankrow[] =
{"                                                                               "};

	for (row = 1; row < 26; row++)
	{
		_settextposition (row, col);
		_outtext (blankrow);
	}
	_settextposition (1,1);


}	/* clear_screen */

display_parameters()
/* Display current operating parameters on lines 2-6 */
{
	char buf[80];
	char data_file_name[64];
	short	row,col;
	int	i;
	
	_settextposition (2,1);
	sprintf (buf, "Configuration File: %s", full_config_file_name);
	_outtext (buf);

	/* Line 3: Data file and Next ID */
	_settextposition (3,1);
	build_data_file_name (data_file_name);
	sprintf (buf, "Data File: %s       Next Meas ID: %d",  
		data_file_name, Next_Meas_ID);
	_outtext (buf);

	/* Line 4-6, CHannel List */
	_settextposition (4,1);
	_outtext ("Channel Measurement List:");

	row = 5;
	col = 3;
	for (i=0; i < Size_Lookup_Table; i++)
	{
		_settextposition (row, col);
		_outtext (Meas_Table[i].Chan_Code);
		col = col + 5;
		if (col > 75)
		{
			col = 3;
			row++;
		}
	} /* for */

}  /* display_parameters */

display_menu(int start_row)
/* Display the main menu on the row specified */
{
	int i;

	_settextposition (start_row,10);
	_outtext ("MENU");

	_settextposition (start_row+2, 8);
	_outtext ("M       Measure");

	_settextposition (start_row+4, 8);
	_outtext ("A       Align Mode to Position Antenna");

	_settextposition (start_row+6, 8);
	_outtext ("Q(ESC)  Quit and Exit Program");

	/* Prompt */
	_settextposition (start_row+9, 6);
	_outtext ("Select Key: ");



}	/* display_menu */

display_align_screen()
/* Display a screen saying we are in alignment mode.  Assumes screen
 *  is already clear. */
{

	char	buf[80];

	sprintf (buf, "ALIGNMENT MODE:  Channel %s (Video %.5lf)",
		 Align_Table.Chan_Code, Align_Table.Freq + VIDEO_OFFSET);
	_settextposition (2,25);
	_outtext (buf);

	_settextposition (8,25);
	_outtext ("PRESS ANY KEY TO RETURN TO MENU");

}  /* display_align_screen */


default_set (int dev)
/*  Set the analyzer to default mode, which displays the entrie MMDS band with
 *		Freq:  DEFAULT_FREQ_START to DEFAULT_FREQ_END
 *		RefLvl:	(DEFAULT_REF)
 *		Sweep:  Auto
 *		RBW: (DEFAULT_RBW)
 *		VBW: (DEFAULT_VBW)
 *		Trace A:  Normal
 *		Type:     Continuous
*/
{
		hsa_blank_B (dev);	/* be sure that trace B is disabled */
		hsa_sf (dev, (DEFAULT_FREQ_START - LO_Freq) * MHz);		// set start frequency
		hsa_ef (dev, (DEFAULT_FREQ_END - LO_Freq) * MHz);		// set end frequency
		hsa_ref_level (dev, DEFAULT_REF);	// set reference level
		hsa_sweep_auto (dev);
		hsa_atten_auto (dev);		//set attenuation to auto
		hsa_rb (dev, DEFAULT_RBW);		// set resolution bandwidth
		hsa_vb (dev, DEFAULT_VBW);		// video bandwidth = resolution bw

		hsa_clear_writeA (dev);	// clear A trace, resets from max_hold
		hsa_sweep_mode (dev, SWEEP_CONTINUOUS);		// set sweep mode
		hsa_hold (dev);  /* give full screen display */

}  /* default_set */

align_set (int dev)
/*  Set the analyzer to Align mode, which consists of the following:
 *		Freq:  .Freq + VIDEO_OFFSET
 *		Span:  0  (ALIGN_SPAN)
 *		RefLvl:	(ALIGN_REF)
 *		Sweep:  (ALIGN_SWEEP)
 *		RBW/VBW: (ALIGN_RBW)
 *		Trace A:  Normal
 *		Trace B:  Peak Hold
 *		Type:     Continuous (default)
*/
{
		hsa_cf (dev, (Align_Table.Freq + VIDEO_OFFSET - LO_Freq) * MHz);		// set center frequency
		hsa_span (dev, ALIGN_SPAN);		// set span
		hsa_ref_level (dev, ALIGN_REF);	// set reference level
		hsa_atten_auto (dev);		//set attenuation to auto
		hsa_sweep_time (dev, ALIGN_SWEEP);		// set sweep time
		hsa_rb (dev, ALIGN_RBW);		// set resolution bandwidth
		hsa_vb (dev, ALIGN_RBW);		// video bandwidth = resolution bw

		hsa_clear_writeA (dev);	// clear A trace, resets from max_hold
		hsa_clear_writeB (dev);	// clear B trace, resets from max_hold
		hsa_max_holdB (dev);		// set B trace to max hold mode
		hsa_hold (dev);  /* give full screen display */

}  /* align_set */

measure_set (int dev, double base_freq)
/*  Set the analyzer to measurement settings
 *   Do a peak hold on A and a clear trace on B
 *		Freq:  As specified
 *		RefLvl:	MEAS_REF)
 *		Sweep:  (MEAS_SWEEP)
 *		RBW: (MEAS_RBW)
 *		VBW: (MEAS_VBW)
 *		Trace A:  Peak Hold (Do a reset first to clear any values);
 *		Trace B:  Clear write;
 *		Type:     Continuous
*/
{
		hsa_cf (dev, (base_freq + VIDEO_OFFSET - LO_Freq) * MHz);		// set center frequency
		hsa_span (dev, MEASURE_SPAN);		// set span
		hsa_ref_level (dev, MEASURE_REF);	// set reference level
		hsa_atten_auto (dev);		//set attenuation to auto
		hsa_sweep_time (dev, MEASURE_SWEEP);		// set sweep time
		hsa_rb (dev, MEASURE_RBW);		// set resolution bandwidth
		hsa_vb (dev, MEASURE_VBW);		// video bandwidth = resolution bw

		hsa_clear_writeA (dev);	// clear A trace, resets from max_hold
		hsa_max_holdA(dev);	/* set to peak hold for multiple sweeps */
		hsa_clear_writeB (dev);	// clear B trace, resets from max_hold
		hsa_sweep_mode (dev, SWEEP_CONTINUOUS);		// set sweep mode
		hsa_hold (dev);  /* give full screen display */

}  /* default_set */

int do_measurements(int dev)
/* Perform the measurement cycle, allowing aborts, and ask at end
 *  if want to save.  Returns TRUE if saving results to file, else
 *  consider the run aborted.
*/
{
	int abort = !TRUE, retry = TRUE;
	int	i=0;
	char	trace_buffer[HSA_BUF_LENGTH];		// read trace text here
	int	length_trace_buffer;
	int	traceB[602];	/* holds returned trace */
	double	rlevel;	/* reference level for determining attenuation. */

	length_trace_buffer = hsa_get_trace_size();	/* revise later */
	while ((i < Meas_Table_Size) && !abort)
	{
		printf ("\n(%d / %d) %5s %.4lf ", Next_Meas_ID, i+1, Meas_Table[i].Chan_Code,
			Meas_Table[i].Freq);
		measure_set(dev, Meas_Table[i].Freq);  /* sutup analyzer */
		rlevel =	hsa_ask_ref_level (dev);
//printf ("rlevel returned is: %g\n", rlevel);
//getch();
		abort = !wait_it_out();	/* do peak hold for the specified time */
		if (!abort)
		{  /* not aborted, capture the trace */
			/* Get the trace and convert it to integer format */
			hsa_get_traceA (dev, trace_buffer);	/* do 5th sweep */
			hsa_M_FORMAT_to_int (trace_buffer, rlevel, traceB);
			/* Now sort so we can get percentile values */
			sort_trace (traceB, length_trace_buffer);
//display_trace (traceB, length_trace_buffer);
			build_meas_structure (traceB, length_trace_buffer, i);  /* build the structure of values */
			display_measurement_summary (traceB, length_trace_buffer, Meas_Table[i].Freq);
		}
		i++;
	} /* while */
	if (!abort)
	/* We have captured all traces, ask the height, X, and Y positions */
		while (retry)
		{
			if (ask_height())
			{  /* abort not asked for */
				ask_X ();
				ask_Y ();
				i = ask_all_is_OK();
//printf ("ask_all_is_OK = %d\n", i);
//getch();
				switch (i)
				{
					case 0:  /* OK */
						retry = !TRUE;
						break;
					case 1:  /* retry */
						break;
					case 2:  /* abort */
						retry = !TRUE;
						abort = TRUE;
						break;
				} /* switch */
//printf ("after switch, abort/retry = %d %d\n", abort, retry);
			}
			else
			{  /* abort asked for */
//printf ("abort due to height \n");
				abort = TRUE;
				retry = !TRUE;
				break;
			}
		}  /* while */

//printf ("bottom of while loop/do_..., abort, retry = %d %d\n", abort, retry);
//getch();

		/* Write the summary and detail file if we did not abort */
		if (!abort)
		{
			write_summary_file ();
			write_detail_file ();
			printf ("Summary files Written, Press any key...");
			getch();
			Next_Meas_ID++;

		}

	return (!abort);

}  /* do measurements */

int wait_it_out ()
/*  Wait for the global time (Measure_Wait_Time) or until an escape is pressed.
 *   Ignores all other key presses. 
 *  Returns TRUE if it waited it out (NO escape)
*/
{
	time_t now, target;
	char	ch;
	int	abort = !TRUE;

	time (&now);
	target = now + (time_t) Measure_Wait_Time;

	while ((now < target) && !abort)
	{
		if (kbhit())
		{
			ch = getch();
			if (ch == 0x1b)
				abort = TRUE;  /* ESC key is abort, ignore all others */
		} /* key pressed */
		time (&now);
	}  /* while */
	return (!abort);  /* returns TRUE if not aborted on ESC key */
}  /* wait_it_out */

sort_trace (int trace[], int length)
/* sort the trace buffer from strongest to lowest.  Note these are
 *  integer values with two implied decimal points.  The array is returned
 *  in sorted order.
*/
{
	qsort ((void *) trace, (size_t) length, (size_t) sizeof (int), mycompare);
}  /* sort_trace */


int mycompare ( int *arg1, int *arg2)
{
	if ( *arg1 > *arg2)
		return (-1);
	if ( *arg1 < *arg2)
		return (1);
	return (0);
}

display_measurement_summary (int trace[], int length, double freq)
/* display the measurement summary:
		Raw then with calibration factor
		followed by a \n
*/
{
	double perc95 = 95.0;  /* percentile to display */
	int index;
	float rl, rl_cal;

	index = index_from_percentile (perc95, length);

	rl = (float) trace[index] / 100.0;

	rl_cal = rl + cal_factor (rl, freq);

	printf ("RL-Raw: %6.1f   RL-Cal: %6.1f  dBm", rl, rl_cal);
//	printf ("RL: %6.1f dBm", (float) (trace[index_from_percentile(perc95, length)])) / 100.0;

}  /* display_measurement_summary */


int index_from_percentile (double percentile, int length)
/* Returns the index to find the specified percentile in the
 *  array of size (length).  For maximum value, specify 100.0 as the percentile.
 *  For minimum value, specify the percentile = 0.0.
 * NOTE: percentile is in percent, i.e 50% would be passed as 50.0 (NOT 0.5)
 * NOTE2: This is not totally precise as it takes the nearest integer index
 *   and does not average between two points if the center point falls
 *   in between the two.  Since our sample is large, this should not make a
 *   big difference..
 * NOTE: index is returned in range of {0,...,length-1} to correspond with
 *   index in a sorted array.
*/
{
	int	result;

	if (percentile >= 100.0)
		return (0);				  /* maximum value */
	if (percentile <= 0.0)
		return (length - 1);
	/* use y = mx + b where m = slope = - (length/100) and b = length */
	/* take nearest integer by added one-half */
	result = (int) (((-length/100.0) * percentile) + (double) length + 0.5);
	return (result);
}  /* index_from_percentile */

build_meas_structure (int trace[], int length, int i)
/* build the structure for this measuremet into the specified 
 *  index of global structure "Measured_Data", including:
		ID, Number, channel and frequency, and various percentiles of
 *    the signal level distribuition.
*/
{
	/* insert the measurement ID and number (which is "i") */
	Measured_Data[i].Number = i;

	/* Insert the channel code and base frequency */
	Measured_Data[i].code_freq = Meas_Table[i];

	/* From the trace, lookup the various percentiles - This is before
    *  adjustmenet for calibration factors. */
	Measured_Data[i].rsl.Max = 
		(float) trace[index_from_percentile (100.0, length)] / 100.0;    /* Maximum value (peak) */
	Measured_Data[i].rsl.rsl99 = 
		(float) trace[index_from_percentile (99.0, length)] / 100.0;  /* 99 percentile */
	Measured_Data[i].rsl.rsl95 = 
		(float) trace[index_from_percentile (95.0, length)] / 100.0;  /* 95 percentile */
	Measured_Data[i].rsl.rsl90 = 
		(float) trace[index_from_percentile (90.0, length)] / 100.0;  /* 90 percentile */
	Measured_Data[i].rsl.rsl75 = 
		(float) trace[index_from_percentile (75.0, length)] / 100.0;  /* 75 percentile */
	Measured_Data[i].rsl.rsl50 = 
		(float) trace[index_from_percentile (50.0, length)] / 100.0;  /* 50 percentile */
	Measured_Data[i].rsl.rsl10 = 
		(float) trace[index_from_percentile (10.0, length)] / 100.0;  /* 10 percentile */
	Measured_Data[i].rsl.rsl01 = 
		(float) trace[index_from_percentile (1.0, length)] / 100.0;  /* 1 percentile */
	Measured_Data[i].rsl.Min = 
		(float) trace[index_from_percentile (0.0, length)] / 100.0;    /* lowest value */

	/* Now adjust the signal levels for the calibration, which at this point,
	 *  is assumed to be frequency specific. */

	Measured_Data[i].rsl.Max += 
		cal_factor (Measured_Data[i].rsl.Max, Measured_Data[i].code_freq.Freq);
	Measured_Data[i].rsl.rsl99 += 
		cal_factor (Measured_Data[i].rsl.rsl99, Measured_Data[i].code_freq.Freq);
	Measured_Data[i].rsl.rsl95 += 
		cal_factor (Measured_Data[i].rsl.rsl95, Measured_Data[i].code_freq.Freq);
	Measured_Data[i].rsl.rsl90 += 
		cal_factor (Measured_Data[i].rsl.rsl90, Measured_Data[i].code_freq.Freq);
	Measured_Data[i].rsl.rsl75 += 
		cal_factor (Measured_Data[i].rsl.rsl75, Measured_Data[i].code_freq.Freq);
	Measured_Data[i].rsl.rsl50 += 
		cal_factor (Measured_Data[i].rsl.rsl50, Measured_Data[i].code_freq.Freq);
	Measured_Data[i].rsl.rsl10 += 
		cal_factor (Measured_Data[i].rsl.rsl10, Measured_Data[i].code_freq.Freq);
	Measured_Data[i].rsl.rsl01 += 
		cal_factor (Measured_Data[i].rsl.rsl01, Measured_Data[i].code_freq.Freq);
	Measured_Data[i].rsl.Min += 
		cal_factor (Measured_Data[i].rsl.Min, Measured_Data[i].code_freq.Freq);


	/* now get the measurement time and date */
	_strtime (Measured_Data[i].ch_meas_time);  /* these may have new line added */
	_strdate (Measured_Data[i].ch_meas_date);

}	/* build_meas_structure */

float cal_factor 
	(
	 float rsl_in,  /* signal level in */
	 double freq  /* frequency base of frequency */
	)
/*  Determine the calibration factor for the specified frequency (which
 *   the frequency effect is currently ignored.  Based on measurements of
 *   video source (in lab) to what was indicated.
*/

{
	return ((float) CAL_FACTOR);

}	/* cal_factor */

int ask_height()
/*  Asks for a new height and sets Height accordingly.  Returns TRUE if
 *   a valid entry (>0) is entered.  Defaults to last entry. */
{
	char h_text[8];
	int i;

	printf ("\nAntenna Height [%d feet] (0 to abort): ", Height);
	gets (h_text);
	if (strlen (h_text) > 0)
	{  /* entry was entered */
		i = atoi (h_text);
		if (i <= 0)
		{ /* abort requested */
			return (!TRUE);
		}
	} 
	else
	{  /* no entry made, use last entry */
		i = Height;  /* "i" must have the current entry */
	}

	/* check to see if the value indicates an abort */
	if (i <= 0)
	{ /* abort requested */
			return (!TRUE);
	}
	Height = i;  /* save the newly entered value */

}  /* ask_height */

ask_X ()
/* Asks for longitude (X value) in several different formats, uses last
 *  entry as default */
{
	char in_text[24], temp_text[24];
	int	retry = TRUE;
	double	value;

	while (retry)
	{
		printf ("\nX value (Longitude) [%s ]: ", GPS_Lon_Entry);

		gets (in_text);
		if (strlen (in_text) > 0)
		{  /* entry was entered */
		}
		else
			strcpy (in_text, GPS_Lon_Entry);  /* process the default */

		/* save a copy of the in_text */
		strcpy (temp_text, in_text);

		/* Now decode the input and return the result and success flag */
		if (gps_decode(in_text, &value))
		{  /* was able to decode it */
			retry = !TRUE;
			strcpy (GPS_Lon_Entry, temp_text);
		}
		else
		{  /* decode error */
			printf ("\a");
		}
	}  /* while */

	GPS_Lon = - fabs(value);  /* save the new value, force negative (east) */

//printf ("Get_X, GPS_Lon = %lf\n", GPS_Lon);
//getch();
}

ask_Y ()
/* Asks for latitude (Y value) in several different formats, uses last
 *  entry as default */
{
	char in_text[24], temp_text[24];
	int	retry = TRUE;
	double	value;

	while (retry)
	{
		printf ("\nY value (Latitude) [%s ]: ", GPS_Lat_Entry);

		gets (in_text);
		if (strlen (in_text) > 0)
		{  /* entry was entered */
		}
		else
			strcpy (in_text, GPS_Lat_Entry);  /* process the default */

		/* save a copy of the in_text */
		strcpy (temp_text, in_text);

		/* Now decode the input and return the result and success flag */
		if (gps_decode(in_text, &value))
		{  /* was able to decode it */
			retry = !TRUE;
			strcpy (GPS_Lat_Entry, temp_text);  /* save entry as next default */
		}
		else
		{  /* decode error */
			printf ("\a");
		}
	}  /* while */

	GPS_Lat = value;  /* save the new value */

}

int gps_decode(char text[], double *value)
/* Processes the input string, which is in one of several formats:
 *	   dd.dddd  (decimal degrees)
 *    dd mmm.mm  (degrees and minutes)
 *    dd mm ss.s  (degrees, minutes, seconds)
 *    dd-mm-ss.s  (same)
 *
 *  Returns TRUE if in a valid format, and value is returned in dec. degrees.
 *  Else !TRUE if format not understood and value is untouched.
 *
 * NOTES: Uses blanks (or dashes) to determine format.  Do not use extra or
 *  trailing blanks.
*/
{
	double deg = 0.0, min = 0.0, sec = 0.0;
	int success = !TRUE;
	char	*ptr;

#define	 GPS_SEP	" -"  /* the valid seperators */

	switch (count_gps_seperators(text))
	{
		case 0:  /* impied dd.dddd */
//printf ("case 0:");
			deg = atof (text);
			min = 0.0;
			sec = 0.0;
//printf ("d/m/s = %lf %lf %lf\n", deg, min, sec);
			success = TRUE;
			break;
		case 1:   /* dd mmm.mm */
			ptr = strtok (text, GPS_SEP);
//printf ("case 1:, ptr(1) = %s\n",ptr);
			deg = atof (ptr);
			ptr = strtok (NULL, GPS_SEP);
//printf ("case 1:, ptr(2) = %s\n", ptr);
			min = atof (ptr);
//printf ("d/m/s = %lf %lf %lf\n", deg, min, sec);
			success = TRUE;
			break;
		case 2:	/* dd[ -]mm[ -]ss.ss */
			ptr = strtok (text, GPS_SEP);
//printf ("case 2:, ptr(1) = %s\n",ptr);
			deg = atof (ptr);
			ptr = strtok (NULL, GPS_SEP);
//printf ("case 2:, ptr(2) = %s\n",ptr);
			min = atof (ptr);
			ptr = strtok (NULL, GPS_SEP);
//printf ("case 2:, ptr(3) = %s\n",ptr);
			sec = atof (ptr);
//printf ("d/m/s = %lf %lf %lf\n", deg, min, sec);
			success = TRUE;
			break;			
	}	/* switch */
	if (success)
		*value = deg + (min/60.0) + (sec/3600.0);

	return (success);

}  /* gps_decode */

int count_gps_seperators(char text[])
/* Count the number of seperators (" " or "-") in the text string and
 *  check for invalid characters */
{
	int result = 0;
	int	bad_format = !TRUE;  /* default to success */
	int i;

	for (i=0; i < strlen(text); i++)
	{
		switch (text[i])
		{
			case '0':
			case '1':
			case '2':
			case '3':
			case '4':
			case '5':
			case '6':
			case '7':
			case '8':
			case '9':
			case '.':
				break;
			case ' ':
			case '-':
				result++;
				break;
			default:
				bad_format = TRUE;
		}  /* switch */
	}

//printf ("count_gps_sep, bad_format/result = %d %d\n", bad_format, result);
	if (bad_format)
		return (-1);  /* error occured */
	else
		return (result);
}


int ask_all_is_OK()
/* Asks if the above entries a ok, returns an integer value of:
 *  0: entry is ok, proceed  (RETURN)
 *  1: reenter (R r)
 *  2: Abort (ESC, Q q)
 * All other characters are ignored
*/
{
	int	ch_in;
//	char	ch_in;

	/* Echo entries here */
	printf ("***  VERIFY ENTERED VALUES ***\n");
	printf ("Height %d feet\n", Height);
	printf ("Lon(X): actual: %.5lf  Entered: %s\n",
		GPS_Lon,  /* converted to dd.dddd format */
		GPS_Lon_Entry);
	printf ("Lat(Y): actual: %.5lf  Entered: %s\n",
		GPS_Lat,
		GPS_Lat_Entry);
		
	printf ("Save (RETURN or S), Reenter values (R), or Abort (ESC) :");

	while (TRUE)
	{
		ch_in = getch ();

		switch (ch_in)
		{

//			case 0x0d:
			case 13:  /* This is CR */
			case 'S':
			case 's':
				printf ("\n");
				return (0);

			case 'R':
			case 'r':
				printf ("\n");
				return (1);

			case 0x1b:  /* escape */
			case 'q':
			case 'Q':
				printf ("\n");
				return (2);
		}  /* switch */

	}  /* force a valid key to be pressed */
	return (-1);  /* should never get here... */

}  /* ask_all_is_OK */

write_summary_file ()
/* Write the entire measured structure to the (main) summary output file
 *  in the format documented at the top of this source code.
*/
{
	char	file_name[32];
	FILE *fp;
	int	i;

	/* Build data file name with path and extension */
	build_data_file_name (file_name);  /* get the file name w/ path */
	strcat (file_name, DATA_EXT);  /* summary file extension added */

	/* Open the file for append, adding onto any existing entries */
	fp = fopen (file_name, "a+");
	if (fp == NULL)
	{
		printf ("Fatal error opening file %s\n");
		exit (4);
	}

	/* Loop through each entry in the measurement table and write out
	 *  one line for each */
	for (i=0; i < Meas_Table_Size; i++)
	{
		fprintf
		(
			fp,
			"ID:%d\tMN:%d\tCH:%s%\tFR:%.5lf\tRL:%.1f\tHT:%d\tX:%.5lf\tY:%.5lf\tAG:%.2f\tAT:%s\td:%s\tt:%s\n", 
			Next_Meas_ID,   /* Measurement ID for this location */
			i+1, /* Measurement number at this location (1, 2, ...) */
			Measured_Data[i].code_freq.Chan_Code,
			Measured_Data[i].code_freq.Freq,
			Measured_Data[i].rsl.rsl95,  /* used the 95 percentile value */
			Height,
			GPS_Lon, GPS_Lat,
			Antenna_Gain, Antenna_Type,
			Measured_Data[i].ch_meas_date,
			Measured_Data[i].ch_meas_time
		);
	}  /* for */

//	Next_Meas_ID++;  /* This doesn't belong here, the next procedure will
//   need this variable.

	fclose (fp);
}  /* write_summary_file */


write_detail_file ()
/* Write the entire measured structure to the (main) summary output file
 *  in the format documented at the top of this source code.
*/
{
	char	file_name[32];
	FILE *fp;
	int	i;

	/* Build data file name with path and extension */
	build_data_file_name (file_name);  /* get the file name w/ path */
	strcat (file_name, DETAIL_EXT);  /* summary file extension added */

	/* Open the file for append, adding onto any existing entries */
	fp = fopen (file_name, "a+");
	if (fp == NULL)
	{
		printf ("Fatal error opening file %s\n");
		exit (4);
	}

	/* Loop through each entry in the measurement table and write out
	 *  one line for each */
	for (i=0; i < Meas_Table_Size; i++)
	{
		fprintf
		(
			fp,
			"ID:%d\tMN:%d\tCH:%s%\tFR:%.5lf\tMT:%d\t%.1f\t%.1f\t%.1f\t%.1f\t%.1f\t%.1f\t%.1f\t%.1f\t%.1f\n",
			Next_Meas_ID,   /* Measurement ID for this location */
			i+1, /* Measurement number at this location (1, 2, ...) */
			Measured_Data[i].code_freq.Chan_Code,
			Measured_Data[i].code_freq.Freq,
			Measure_Wait_Time,
			Measured_Data[i].rsl.Max,
			Measured_Data[i].rsl.rsl99,
			Measured_Data[i].rsl.rsl95,
			Measured_Data[i].rsl.rsl90,
			Measured_Data[i].rsl.rsl75,
			Measured_Data[i].rsl.rsl50,
			Measured_Data[i].rsl.rsl10,
			Measured_Data[i].rsl.rsl01,
			Measured_Data[i].rsl.Min
		);
	}  /* for */

	fclose (fp);
}  /* write_summary_file */


display_trace (int traceB[], int length_trace_buffer)
/* DEBUG AID, see if sorted in correct order */
{
	int i;

	printf ("\n", traceB[0]);

	for (i=1; i < length_trace_buffer; i++)
	{
		printf ("%6d ", traceB[i]);
		if ((i % 10) == 0)
		{
			printf ("\n(%3d)", i+1);
			getch();
		}
	}
}

/******************************************************************************/
/******************************************************************************/
/******************** E N D   C O D E *****************************************/
/******************************************************************************/
/******************************************************************************/
#ifdef SKIP

main(int argc, char **argv)
{
	int hsa_ud, att1_ud, att2_ud;	/* device for spec analy and controllers */
	int	exit_loop = !TRUE;	/* exit measurement loop */
	int	i, diversity;
	time_t	ending_time;	/* type 3, time to end measurements */
	time_t	time_last_atten, time_last_position, now;	/* start time of last attenuate and scan (type 4) */

/* Print greetings here */
	printf (SBTRI);
	printf (SBTRI_PROP);
	printf (SBTRI_PROGRAM, PROGRAM_NAME, VERSION);

/* process any command line arguments */
	process_command_line_arguments (argc, argv);

	gpib_start ();

// Open up the connection to the spectrum analyzer and attenuation controllers
	hsa_ud = gpib_open (HP_SA);	/* get the handler (access number) */
//	att1_ud = gpib_open (HP_ATT1);	/* get the handler (access number) */
//	att2_ud = gpib_open (HP_ATT2);	/* get the handler (access number) */
	att1_ud = 31;
	att2_ud = 32;
	if (hsa_ud < 0 | att1_ud < 0 | att2_ud < 0) {
		printf ("Error in opening connection to instruments\n");
		printf ("A negative number indicates error in the following:\n");
		printf ("Spectrum analyzer code: %d\n", hsa_ud);
		printf ("Attenuator Controller #1 (left) code: %d\n", att1_ud);
		printf ("Attenuator Controller #2 (right) code: %d\n\n", att2_ud);
		printf ("Check cables and power to instruments\n\nProgram Aborted...\n");
		exit (1);
	}

/* set configuration file name file */
	strcpy (full_config_file_name, config_path);
	strcat (full_config_file_name, config_file_name);

/* Do the configuration file stuff */
	if (!do_configuration_file()) {
		printf ("\n\nError in configuration file -- Program aborted \n");
		exit (3);
	}  /* end do configuration */	

/* Ask and open output file */
	if (!ask_data_file()) {
		printf ("Program aborted\n");
		exit(1);
	}		

/* Initialize instrument settings */
	printf ("\n\nInitializing Instruments\n");
	initialize_instruments (hsa_ud, att1_ud, att2_ud);

/* If gps is needed, initialize unit here 07.06 */
	if (ms_gps > 0) {
		_clearscreen (_GCLEARSCREEN);	/* clear the screen */
		display_message (1,"Initializing the GPS receiver.");
		tans_init();
		time (&gps_last_warning_time);	/* initialize warning timer */
		if (ms_gps == 2) {	/* wait for lock in warning mode */
			display_message (7,"Waiting for GPS to lock onto satellites");
			display_message (8,"ESC to exit, RET to continue");
			exit_loop = !TRUE;	/* wait for a reply or until */
			empty_keyboard_buffer();
			while (!exit_loop) {
/* display status and only proceed if doing fixes (and all info present) */
/* the next two lines were replaced with the following one line */
//				display_gps_status();	/* display current */
//				if ((gps_info.last_status == 0) & (gps_info.last_lon != 0.0)) exit_loop = TRUE;	/* wait for last/lon to display */
				if (display_gps_status() == 0) exit_loop = TRUE;	/* 07.08 wait for last/lon to display */

				if (kbhit()) {
					i = getch();
					if (i == 0x1b) {
						tans_restore();
						reset_instruments (hsa_ud, att1_ud, att2_ud);
						_clearscreen (_GCLEARSCREEN);	/* clear the screen */
						exit(0);	/* abort program */
					}
					else if (i == '\r')
						exit_loop = TRUE;
				}	/* endif */
			}	/* end while waiting for lock */
		_clearscreen (_GCLEARSCREEN);	/* clear the screen */
		}	/* end if warning set */
	} /* if gps is enabled */

/* If a temporary file is needed (to store results of each band), open here */
	if (ms_graphics) {
		if ((temp_file = tmpfile()) == NULL) {
			printf ("Temporary file open failed, no graphics ability\n");
			ms_graphics = !TRUE;	/* force no graphics */
		}
		else
		{
			for (i=0; i< ms_number_scans; i++) 	/* check all bands to see if the trace buffer */
			{	/* will overflow.  If so, then disable ms_graphics */
				if (ms_bands[ms_scans[i].band].number_segments * (hsa_points_per_trace-1) > TRACE_BUFFER_LENGTH)
					{  /* buffer overflow */
						ms_graphics = !TRUE;
						printf ("Buffer Overflow condition for scan %d\n", i+1);
					}  /* endif band overflow */
			}	/* loop thru scan table */
			if (ms_graphics)  /* still ok? */
				printf ("Temporary file opened ok\n");
			else {
				printf ("Graphics disabled\n");
				printf ("RET to continue.\n");  getch();
			}
		}	/* end if.else */
	}  /* end if ms_graphics */
				
/* ask Operator Name */
//	printf ("\n\nOperator Name: ");
//	gets (operator_name);

	_clearscreen (_GCLEARSCREEN);	/* clear the screen */
	printf ("Setting up data files....Please wait");

	write_initial_records();	/* write the initial stuff at beginning of data file */
	write_config_file_echo();	/* echo the configuration file */
	write_system_parameters();  /* write the system parameters to output file */
	write_band_parameters();  /* write the frequency band parameters */
	write_antenna_parameters();  /* write the antenna parameters */
	write_scan_parameters();  /* write the antenna parameters */

/* write the preliminary file headings and stuff */
	printf ("\nWriting Frequency Record.");
	write_frequency_record();  /* write out the frequency records */

	_clearscreen (_GCLEARSCREEN);	/* clear the screen */

/* now do the type of measurement specifed in ms_type */
	exit_loop = !TRUE;
	while (!exit_loop) {
		switch (ms_type) {
			case 1:	/* van in movable locations */
				diversity = 0;	/* loop counter for diversity */
				exit_loop = ask_position_record1(); /* ask and write position record */
				if (!exit_loop) {
					while ((diversity < (int) ms_type_arg1) & !exit_loop) /* 08.04 */
					{	/* do each diversity position */
						write_diversity_record(MS_POSITION_VAN1_ID, diversity);	/* write a diversity record */
						get_attenuator_settings (hsa_ud, att1_ud, att2_ud);	/* determine attenuator settings */
						do_position_record(hsa_ud, att1_ud, att2_ud);  /* do an entire position */
						if (ms_graphics) {
							display_position_graph();
						}
						if (++diversity == (int) ms_type_arg1) /* 08.04 */
						{ /* at end of loop */
							printf ("\a");
							display_sys_status ("COMPLETE");
//							display_message (24, "This location/position complete");
//							display_message (32, "Press RET to Exit");
							display_message (video_info.numtextrows, "Press RET to Exit");
							getch();
							if (ms_graphics) reset_graphics();
						}
						else
						{ /* still have more location here to measure */
							display_sys_status ("WAITING");
							display_message (24, "Ready for next Diversity Measurement.");
							display_message (25, "Move 3\" forward, RET to start measure");
							printf ("\a");
							i = getch();
							while ((i != '\r') & (i != 0x1b)) {
								printf ("\a");
								i = getch();
							}
							clr_line(7);
							clr_line(8);
							if (i == 0x1b) { /* escape */
								if (ms_graphics) reset_graphics();
								 exit_loop = TRUE;	/* escape */
							}
						}  /* end else */
						if (!exit_loop) exit_loop = check_free_disk_space();	/* chk only if not exiting 06.07 */
						exit_loop = TRUE;
//						diversity++;
					}	/* end diversity check */
					number_positions_measured++;
				}  /* end if exit_loop */
				break;
			case 2:	/* cart based measurements */
				printf ("Case %d Not Yet Implemented...Press RET to exit\n");
				exit_loop = TRUE;
				break;
			case 3:	/* fixed location measurements == same logic as type 4 08.02 */
			case 4:	/* moving mobile measurements */
				exit_loop = ask_position_record4(); /* wait for cmd to start measure */
				if (!exit_loop) {
/* set the last attenuation time and the last position time such that they trigger
	on the first pass through the loop */
					time (&time_last_atten);  /* time of auto-attenuate function */
					time_last_atten -= ms_type_arg2 + 10L;	/* this should force first attenuate  08.04 */
					time (&time_last_position);  /* time of auto-attenuate function */
					time_last_position -= ms_type_arg1 + 10L;	/* 08.04 this should force first measurement */
					time (&ending_time);	/* ending time (type 3 only) */
					if (ms_type_arg4 == 0L)	/* 08.04 if 0, then run for a month */
						ending_time += 30L * 24L *3600L;	/* d0 days * 24 hrs * 3600 sec/hr = 1 month */
					else
						ending_time += ms_type_arg4;	/* 08.04 duration in arg4 */

					display_sys_status ("WAITING");
					display_message (7, "hold (h/CR), ESC= abort");
					while (!exit_loop) {
						display_gps_status();
						time (&now);
						if (difftime(now, time_last_atten) > (double) ms_type_arg2) {  /* time to autoattenuate */
							time (&time_last_atten);  /* time of auto-attenuate function */
							get_attenuator_settings (hsa_ud, att1_ud, att2_ud);	/* determine attenuator settings */
							display_sys_status ("WAITING");
							display_message (25, "hold (h/CR), ESC= abort");
						} /* end if time to autoattenuate again */
						time (&now);
						if (difftime(now, time_last_position) > (double) ms_type_arg1) {  /* time to autoattenuate */
							time (&time_last_position);  /* time of auto-attenuate function */
							do_position_record(hsa_ud, att1_ud, att2_ud);  /* do an entire position */
							number_positions_measured++;
							display_sys_status ("WAITING");
							display_message (25, "hold (h/CR), ESC= abort");
							if (ms_graphics) {
								display_position_graph();
							}  /* end if grapics display */
						} /* end do a position */
					/* see if kbhit and act on it */
						if (kbhit()) {	/* some key was pressed */
							switch ((int) getch()) {	/* get the character */
								case 'h':	/* hold */
									empty_keyboard_buffer();	/* dump any other keystrokes */
									display_sys_status ("HOLD");
									display_message (25,"Holding (h/CR=cont or ESC=abort)");
									i = 0;
									while (i != 'h' && i != '\r' && i != 0x1b) { /* only exit with RET, ESC, or 'h' */
										while (!kbhit()) display_gps_status(); /* constantly update GPS */
										i = getch();
									}
									if (i == 0x1b) exit_loop = TRUE;
									display_message (7, "hold (h/CR), ESC= abort");
									break;
								case 0x1b:	/* escape */
									exit_loop = TRUE;
									break;
								default:
									empty_keyboard_buffer();	/* dump any other keystrokes */
									printf ("\a");
									break;
							}	/* end switch */
							clr_line (7);	/* clear this line */
						}	/* end kbhit */
						if (ms_type == 3) {	/* check to see if time expired (type 3) */
							time (&now);
							if (now > ending_time) {
								display_message (7,"Running time completed");
								exit_loop = TRUE;
							}	/* end check of time */
						}	/* end type 3 (/
					if (!exit_loop) exit_loop = check_free_disk_space();	/* chk only if not exiting 06.07 */
					} /* end while */
					reset_graphics();
				}  /* end if exit_loop */
				break;
			default:	/* not implemented */
				printf ("Case %d Not Supported...Press RET to exit\n");
				exit_loop = TRUE;
				break;
		}	/* end switch ms_type */
	}	/* end while not exit */

/* done with collection, do last record */
	write_end_record();

/* close the data file
	fclose (data_file);

/* Release instrument settings */
	reset_instruments (hsa_ud, att1_ud, att2_ud);

/* stop the serial handler */
	gpib_stop();

/* reset the GPS receiver (if enabled ) */
	if (ms_gps > 0) tans_restore();

}		/* end main() */


process_command_line_arguments (int argc, char **argv)
/* read and do any and all command line arguments from dos */
{
	int	i, arg, specifier_number = -1;
	char	*token;	/* for searching for specifier */
	char	buffer[64];

	for (arg=1; arg<argc; arg++) {
		printf ("Arg.%d in: <%s>\n", arg, argv[arg]);
		strcpy (buffer, argv[arg]);	/* copy to buffer for later analysis */
		strlwr (buffer);	/* make sure in lower case */
		token = strtok (buffer, SEPARATOR);	/* see if token (":") is in string */
/* now find if any of the tokens match a valid DOS_SEPARATOR command */
		for (i=0; i < NUMBER_DOS_SPECIFIERS; i++) {
			if (strcmp(token, &DOS_SPECIFIERS[i][0]) == 0) 
			{ /* the token matches */
				specifier_number = i;
				break;
			}	/* end for match found */
		}	/* end for NUMBER_DOS_SPECIFIERS */
/* now perform specified action */
		switch (specifier_number) {
			case 0:	/* config file name (no path) */
				strcpy (config_file_name, &buffer[strlen(token)+1]);
				break;
			case 1:	/* config file path (no file name) */
				strcpy (config_path, &buffer[strlen(token)+1]);
				break;
			case 2:	/* data file nnumber 06.12 */
				data_file_number_input = atoi (&buffer[strlen(token)+1]);
				break;
			case 3:	/* data file path (no file name) */
				strcpy (data_path, &buffer[strlen(token)+1]);
				break;
			case 4: 	/* measurement index */
				measurement_index = atoi (&buffer[strlen(token)+1]);
				break;
			default:
				printf ("Invalid Command Line (on next line):\n [%s]\n", argv[arg]);
				break;
		} /* end switch */
	}

}	/* end process_command_lines */

initialize_instruments (int hsa_ud, int att1_ud, int att2_ud)
/* Initialize the instruments to known initial states */
{
	char	string[32];  /* for spectrum analyzer string */
/* Spectrum analyzer initialization */
	hsa_preset (hsa_ud);		/* set to preset value (IP) */
//	hsa_set_time_date(hsa_ud);	/* set time and date on analyzer to match PC (local routine) */
	strcpy (string, "SBTRI(C) ");  /* set up title to include version */
	strcat (string, PROGRAM_NAME);
	strcat (string, " Vers. ");												  
	sprintf (&string[strlen(string)], "%.2f\n", VERSION);
//	hsa_title (hsa_ud, string);	// write title
	hsa_trace_format (hsa_ud, M_FORMAT);	// set M transfer (integer units)
	hsa_sf(hsa_ud, 100E6);
	hsa_ef(hsa_ud, 200E6);  /* start at some arbitrary place */
	hsa_rb (hsa_ud,1E6);
	hsa_vb (hsa_ud,30E3);
	hsa_sweep_mode (hsa_ud, SWEEP_SINGLE);		// set sweep mode

/* attenuator controller (hard coded) */
//	gpib_write (att1_ud, "B0123456789");
//	gpib_write (att2_ud, "B0123456789");

}  /* -------  end initialize_instruments --------------------*/

reset_instruments (int hsa_ud, int att1_ud, int att2_ud)
/* Release and reset any instruements now that we are done */
{
	hsa_sweep_mode (hsa_ud, SWEEP_CONTINUOUS);		// set sweep mode
	gpib_local (hsa_ud);
//	gpib_local (att1_ud);
//	gpib_local (att2_ud);
}	/* ------------  end reset instruments --------------------*/

int ask_data_file(void)
/* Revision 06.12 revised that file name is of form mmddxxxx.fld where
mmdd is the month and day, xxxx is the file number entered by the
operator, and fld is the file extension (fixed).  4/10/92 
  SBMS - revised to re-ask if file exists */
{
	int   reply, min, n;
	char	input[32], date_buffer[10],temp_str[10];
	int   reask = TRUE;

/* set up leading part of the file name, the mmdd */
	_strdate(date_buffer);	/* get current date */
	strncpy (data_file_name, date_buffer, (size_t) 2);	/* get month (mm) */
	strncpy (&data_file_name[2], &date_buffer[3], (size_t) 2);	/* get date (dd) */
	data_file_name[4] = '\0';	/* end of string */

while (reask)
{
	if (data_file_number_input > 0) /* number from command line */
	{	/* command line number was specified as input, set it up */
		n = data_file_number_input;	/* get passed number */
		data_file_number_input = 0;  /* insure if file exists, will ask */
	}	/* end if command line file name specified */
	else
	{	/* ask for file name, not entered on command line */
		printf ("\n\nEnter Output File Number:  ");
		gets (input);
		if (strlen(input) == 0 ) return(!TRUE);	
		n = atoi (input);	/* decode the number */
	}	/* end else */
	if (n < 1 || n > 9999)	{ /* error in range */
		printf ("\aEntered number (%d) is out of range--1 to 9999 allowed\n", n);
		return (!TRUE);
	}
	sprintf (temp_str, "%.4d", n);
	strcat (data_file_name, temp_str);	/* now we have mmddxxxx */

/* Concatinate the path with the entered file name to obtain full file name */
	strcpy (full_data_file_name, data_path);  /* get full file name */
	strcat (full_data_file_name, data_file_name);
	strcat (full_data_file_name, ".fld");	/* extension */

/* open first as read only, let them abort if wanted */
	data_file = fopen(full_data_file_name, "r");
  	fclose (data_file);
	if ( data_file != NULL)
	{
		printf ("\aOutput File <%s> exists, Number must be unique. ", full_data_file_name);
		_strdate(date_buffer);	/* RESET THES BUFFERS, ELSE PROBLEMS get current date */
		strncpy (data_file_name, date_buffer, (size_t) 2);	/* get month (mm) */
		strncpy (&data_file_name[2], &date_buffer[3], (size_t) 2);	/* get date (dd) */
		data_file_name[4] = '\0';	/* end of string */
	}
	else
		reask = !TRUE;  /* we're ok to proceed */
}  /* while reask */
	data_file = fopen(full_data_file_name, "w");
	printf ("Output File <%s> opened\n", full_data_file_name);
	return (TRUE);

}	/* ask_output_file */

/* =======================  get_attenuator_settings ====================*/
get_attenuator_settings(int hsa_ud, int att1_ud, int att2_ud)
/*  Go through the scan list and measure each antenna to determine what
	attenuator setting to use.  uses the .measured flag to be sure that
	antenna leg is only measured once.  When completed, the attenuation 
	settings are stored in .attenuation of the ms_antennas structure.  The
	attenuation is determined as the difference between the maximum 
	spectrum analyzer variable ms_saturation_dB and the maximum value of
	the returned array.  New with version 0.4  3/30/92. */
/*  Revision 06.04  do peak hold using hsa_sweep for "n" traces */
{
	int	i, max, min;
	char	trace_buffer[HSA_BUF_LENGTH];		// read trace text here
	int	traceB[MAX_SA_POINTS];	/* holds returned trace */
	double	set_ref_level, rlevel;	/* reference level for determining attenuation. */

	display_sys_status ("AUTO-ATTEN");	/* display current status */

	set_ref_level = REF_LEVEL + 30.0;	/* use this reference level for scope */
/* reset the attenuator to 0 dB for maximum reading */
//	set_attenuator (att1_ud, 0);	/* set to 0 dB attenuation */

/* set basic settings on spectrum analyzer */
	hsa_atten (hsa_ud, 10);	/* 10 dB attenuation */
// printf ("setting reference level to %lf\n", set_ref_level); getch();
 	hsa_ref_level (hsa_ud, set_ref_level);		//  set a reference level
	rlevel = hsa_ask_ref_level (hsa_ud);		/* ask level actually set */
// printf ("rlevel = %lf\n", rlevel); getch();

/* first reset all the antenna measured flag to not measured */
	for (i=0; i<MS_MAX_ANTENNAS; i++)
		ms_antennas[i].measured = !TRUE;

/* go through the scan list, for each scan, check if the antenna has
	already been measured, if not, then measure the antenna range using
	the spectrum analyzer values set in ms_antennas array */
	for (i=0; i < ms_number_scans; i++ )
	{
		if (!ms_antennas[ms_scans[i].antenna].measured)
		{	/* this antenna needs to be measured and attenuation determined */
			ms_antennas[ms_scans[i].antenna].measured = TRUE;
			select_antenna(ms_scans[i].antenna, att1_ud, att2_ud); /* set antenna */
			hsa_ef(hsa_ud, ms_antennas[ms_scans[i].antenna].freq_start+
					ms_antennas[ms_scans[i].antenna].freq_width);	/* set end frequency */
			hsa_sf(hsa_ud, ms_antennas[ms_scans[i].antenna].freq_start);	/* set start frequency */
			hsa_rb (hsa_ud, ms_antennas[ms_scans[i].antenna].rbw);	/* set res bw */
			hsa_vb (hsa_ud, ms_antennas[ms_scans[i].antenna].vbw);	/* set video bw */
			hsa_hold (hsa_ud);  /* clear off setting */
			hsa_max_holdA(hsa_ud);	/* set to peak hold for multiple sweeps */
//			gpib_write (hsa_ud, "MXMH TRA;");	/* manual set peak hold (above no work) */
			hsa_sweep (hsa_ud);	/* do 2 sweeps */
//			hsa_sweep (hsa_ud);	/* sweep 2/5 sweeps */
//			hsa_sweep (hsa_ud);	/* sweep 3/5 sweeps */
//			hsa_sweep (hsa_ud);	/* sweep 4/5 sweeps */
//			hsa_get_traceA (hsa_ud, trace_buffer);
//			hsa_get_traceA (hsa_ud, trace_buffer);
//			hsa_get_traceA (hsa_ud, trace_buffer);
//			hsa_get_traceA (hsa_ud, trace_buffer);
			hsa_get_traceA (hsa_ud, trace_buffer);	/* do 5th sweep */
//printf ("trace_buffer: %s\n", trace_buffer);
			hsa_M_FORMAT_to_int (trace_buffer, rlevel, traceB);

			hsa_clear_writeA(hsa_ud);	/* reset max hold */
//			gpib_write (hsa_ud, "CLRW TRA;");	/* clear peak hold */
			max = get_max(traceB, hsa_points_per_trace) / 100;	/* max value in dB */
			min = get_min(traceB, hsa_points_per_trace) / 100;	/* min value in dB */
 			ms_antennas[ms_scans[i].antenna].attenuation = ((max - ms_saturation_dB + 9) /10) * 10; /* get next higher *10 value */
			if (ms_antennas[ms_scans[i].antenna].attenuation < 0) ms_antennas[ms_scans[i].antenna].attenuation = 0;
	/*  NOTE - OVERRIDE ATTENTUATION AND ALWAYS SET TO ZERO - THE SBMS
	 *   SYSTEM DID NOT HAVE ATTENUATiON CAPABILITY */
			ms_antennas[ms_scans[i].antenna].attenuation = 0;

			write_autoattenuate_record(ms_scans[i].antenna, max, min, 
				ms_antennas[ms_scans[i].antenna].attenuation);	/* write data record for attenuator */
//printf ("max-ms_satuuration_dB + 9 = %d\n", max - ms_saturation_dB + 9);
//printf ("(max-ms_satuuration_dB + 9)/10 = %d\n", (max - ms_saturation_dB + 9)/10);
//printf ("(max-ms_satuuration_dB + 9)/10 * 10 = %d\n", ((max - ms_saturation_dB + 9)/10) * 10);
//printf ("get_atten scan max, sat pt, atten found: %d %d %d %d\n", i, max, ms_saturation_dB, ms_antennas[ms_scans[i].antenna].attenuation);
// getch();
		}	/* attenuation determined */
	}	/* end for all scans */
}	/* end get_attenuator_settings */

int	get_max (int array[], int length)
/* find maximum value of this array */
{
	int	i, max;

	max = array[0];
	for (i=1; i < length; i++)
		if (array[i] > max) max = array[i];
	return (max);
}	/* end get_max */

int	get_min (int array[], int length)
/* find minimum value of this array */
{
	int	i, m;

	m = array[0];
	for (i=1; i < length; i++)
		if (array[i] < m) m = array[i];
	return (m);
}	/* end get_min */


/* =================== ask_position_record1 ========================= */
int ask_position_record1(void)
/* ask and write the position record(s) for type 1 (Van with movable locations )
	to the output file.  Allows up to 8 lines of text and a position key (int).
	Returns TRUE if exit is selected.
	07.07  revised to display number of diversity positions. removed write of data file
   SBMS - Skips key entry
*/
{
	int	 line, i, key;
	char	buffer[8][80], line_key[10];
	char	descr[30];


	_clearscreen (_GCLEARSCREEN);	/* clear the screen */
	printf ("Position Number: %d", number_positions_measured);	
	if ((int) ms_type_arg1 > 1) {	/* 08.04 we must do diversity */
		sprintf (descr, "NOTE: %d Measurements must be taken 3\" apart!", (int) ms_type_arg1); /* 08.04 */
		_settextposition (2,1);
		rev_video();
		_outtext (descr);
		std_video();
	}	/* end diversity condition */

	_settextposition (4,1);
	printf ("Enter a Description of site (up to 8 lines):\n");
	for (line = 0; line< 8; line++) { /* enter up to 8 lines */
		gets (&buffer[line][0]);	/* get the line */
//		printf ("length = %d\n", strlen(&buffer[line][0]));
		if (strlen(&buffer[line][0]) == 0) { /* empty line, we're done here */
			break; /* line has true number of lines entered */
		}	/* end if */
	} /* end of description lines */

#if defined (OLD_DEBUG)
for (i = 0; i< line; i++) { /* enter up to 8 lines */
	printf ("line %d:<%s>\n", i, &buffer[i][0]);
} /* end print section */
#endif

#ifdef NO_SKIP
/* now ask for operator entered key */
	printf ("\nEnter location key number(if required): ");
	gets (line_key);	/* get the line */
	if (strlen(line_key) > 0)
		key = atoi (line_key);  /* get key number */
	else
#endif
		key = 0; /* no entry */

/* ask if measure or return */
	i = 0;
	while (i != 0x1b && i != '\r') { /* only exit with RET or ESC */
		printf ("\nReady to Measure, press RET to measure or ESC to exit: ");
		i = getch();
	}

	if (i == '\r') {  /* go ahead and measure */	
/* no write this information to the ontput file */
		fprintf (data_file, MS_REC_TYPE, MS_POSITION_INDEX_ID);
		fprintf (data_file, MS_POSITION_INDEX, measurement_index);  /* index */
	
//		fprintf (data_file, MS_REC_TYPE, MS_POSITION_VAN1_ID); /* type 1 record */
//		fprintf (data_file, MS_POSITION_VAN1);	/* no records at this time
//			0.0, 0.0, 0.0);	/* dummy lat, lon, and elevation */

		fprintf (data_file, MS_REC_TYPE, MS_POSITION_OPERATOR_KEY_ID); /* Operatore entered key */
		fprintf (data_file, MS_POSITION_OPERATOR_KEY, key);

		for (i=0; i<line; i++) {  /* only write the valid comment lines */
			fprintf (data_file, MS_REC_TYPE, MS_POSITION_OPERATOR_TEXT_ID); /* Operatore entered text */
			fprintf (data_file, MS_POSITION_OPERATOR_TEXT, &buffer[i][0]);
		}  /* end write text */	
		_clearscreen (_GCLEARSCREEN);	/* clear the screen */
		return (!TRUE);	/* go ahead and measure this location */
	}	/* end for */
	else
		return (TRUE); /* stop process (escape hit ) */
}	/* end ask_position_record1 */

/* =================== ask_position_record4 ========================= */
int ask_position_record4(void)
/* ask and write record(s) for type 4 (moving measurements )
	to the output file.  Only text output of up to 8 lines is currently selected.
	Returns TRUE if exit is selected. (abort)
*/
{
	int	 line, i;
	char	buffer[8][80];


	_clearscreen (_GCLEARSCREEN);	/* clear the screen */

	printf ("Enter a Description of run (up to 8 lines):\n");
	for (line = 0; line< 8; line++) { /* enter up to 8 lines */
		gets (&buffer[line][0]);	/* get the line */
//		printf ("length = %d\n", strlen(&buffer[line][0]));
		if (strlen(&buffer[line][0]) == 0) { /* empty line, we're done here */
			break; /* line has true number of lines entered */
		}	/* end if */
	} /* end of description lines */

/* ask if measure or return */
	i = 0;
	while (i != 0x1b && i != '\r') { /* only exit with RET or ESC */
		printf ("\nReady to Start, press RET to measure or ESC to exit: ");
		i = getch();
	}

	if (i == '\r') {  /* go ahead and measure */	
/* no write this information to the ontput file */
//		fprintf (data_file, MS_REC_TYPE, MS_POSITION_INDEX_ID);
//		fprintf (data_file, MS_POSITION_INDEX, measurement_index);  /* index */
//	
//		fprintf (data_file, MS_REC_TYPE, MS_POSITION_VAN1_ID); /* type 1 record */
//		fprintf (data_file, MS_POSITION_VAN1);	/* no records at this time
//			0.0, 0.0, 0.0);	/* dummy lat, lon, and elevation */

//		fprintf (data_file, MS_REC_TYPE, MS_POSITION_OPERATOR_KEY_ID); /* Operatore entered key */
//		fprintf (data_file, MS_POSITION_OPERATOR_KEY, key);

		for (i=0; i<line; i++) {  /* only write the valid comment lines */
			fprintf (data_file, MS_REC_TYPE, MS_POSITION_OPERATOR_TEXT_ID); /* Operatore entered text */
			fprintf (data_file, MS_POSITION_OPERATOR_TEXT, &buffer[i][0]);
		}  /* end write text */	
		_clearscreen (_GCLEARSCREEN);	/* clear the screen */
		return (!TRUE);	/* go ahead and measure this location */
	}	/* end for */
	else
		return (TRUE); /* stop process (escape hit ) */
}	/* end ask_position_record1 */

do_position_record(int hsa_ud, int att1_ud, int att2_ud)
/*  Process the data for this single location */
/*  removed increment of number_positions_measured 07.07 */
{
	int	pass;
#if defined(OLD_DEBUG)
printf ("do position, ms_passes = %d\n", ms_passes);
#endif
	if (ms_graphics) rewind (temp_file);  /* if graphics, start of temp_file */

/* if gps is enable, write the location record here */
	if (ms_gps > 0) {
			write_gps_record();
	}

	display_sys_status ("MEASURING");	/* display current status */

	for (pass=0; pass < ms_passes; pass++) {
	 	hsa_ref_level (hsa_ud, REF_LEVEL);		// default reference level
		do_scan_record(hsa_ud, att1_ud, att2_ud, pass);
	}  /* end for pass */
//	number_positions_measured++;
}	/* end do_position_record */


do_scan_record(int hsa_ud, int att1_ud, int att2_ud, int pass)
/*  Process a single scan (one set of all scans in structure) */
{
	int	scan;
#if defined(OLD_DEBUG)
printf ("do_scan_record, scans = %d\n", ms_number_scans);
#endif
	for (scan=0; scan < ms_number_scans; scan++) {
// set antenna here
		select_antenna (ms_scans[scan].antenna, att1_ud, att2_ud);
		do_band(hsa_ud, att1_ud, pass, scan);
	}  /* end for scan */
}  /* end do_scan_record */

// this is here as a global because wouldn't work correctly if in below rroutine
// still couldn't get it to work, screws up analyzer???
//char	sa_string[101];  /* spec analyzer title */

do_band(int hsa_ud, int att1_ud, int pass, int scan)
/*  Process a single band (for a fixed antenna) */
/*   scan is the index to used in ms_scans[*] */
{
//	char	sa_string[101];  /* spec analyzer title */
	double	seg_start, seg_end;
	int	seg, end_point, temp;
	int	atten;	/* attenuation */

#if defined(OLD_DEBUG)
printf("do_band pass, scan, ms_scans[scan].band, ms_bands[ms_scans[scan].band].number_segments:\n %d %d %d %d \n",
pass, scan, ms_scans[scan].band, ms_bands[ms_scans[scan].band].number_segments );
#endif

/* set the proper attenuation setting */
	atten = ms_antennas[ms_scans[scan].antenna].attenuation;	/* attenuation */
	set_attenuator (att1_ud, atten);
	seg_start = ms_bands[ms_scans[scan].band].freq_start;  /* start s/a frequency */

/* set the spectrum analyzer for this band */
	hsa_rb (hsa_ud, ms_bands[ms_scans[scan].band].rbw);		// set resolution bandwidth
	hsa_vb (hsa_ud, ms_bands[ms_scans[scan].band].vbw);		// set video bandwidth
	hsa_atten (hsa_ud, ms_bands[ms_scans[scan].band].atten);

/* tell the analyzer what's going on */
// the following code screwed up the analyzer's initial trace???
//   sure don't know why--ignore for now.. csp 3/19/92
//	strcpy (sa_string, "SBTRI(C) ");
//	sprintf (&sa_string[strlen(sa_string)], "ID %d ", rec_id);
//	sprintf (&sa_string[strlen(sa_string)], "Pass %d/%d ", pass+1, ms_passes+1);
//	sprintf (&sa_string[strlen(sa_string)], "Ant (%d) %s ", 
//#if defined(DEBUG)
//		ms_scans[scan].antenna+1, ms_antennas[ms_scans[scan].antenna].name);
//printf ("%s\n", sa_string); getch();
//#endif
//	hsa_title(hsa_ud, sa_string);

/* Write band information record */
	write_band_record (pass, scan, atten);

/* write start of DATA_RECORD */
	fprintf (data_file, MS_REC_TYPE, 
		MS_DATA_RECORD_ID_BASE + ms_scans[scan].index); /* record number */
	fprintf (data_file, MS_DATA_RECORD_1, 
		ms_bands[ms_scans[scan].band].number_segments * (hsa_points_per_trace-1) + 1,  /* total number of points */
		measurement_index);

/* if graphics is enabled, write out record number and size to temporary file */
	if (ms_graphics) {  /* graphics is enabled */
		temp = MS_DATA_RECORD_ID_BASE + ms_scans[scan].index; /* record number */
		fwrite((void *)&temp, sizeof(int), 1, temp_file);  /* write record number */
		temp = ms_bands[ms_scans[scan].band].number_segments * (hsa_points_per_trace-1) ; /* number of points (last point skipped ) */		
		fwrite((void *)&temp, sizeof(int), 1, temp_file);  /* write record number */
	}  /* end if graphics is enabled */

	for (seg=0; seg<ms_bands[ms_scans[scan].band].number_segments; seg++)
	{	/* do for each segment the following */
		seg_end = seg_start + ms_bands[ms_scans[scan].band].segment_delta_f;  /* end of segment */
		end_point = do_segment (hsa_ud, pass, scan, seg, seg_start, seg_end);
		seg_start = seg_end;
	}  /* end for segment */
/* now write last point of record to file with END of line character */
	fprintf (data_file, MS_DATA_RECORD_END_POINT, end_point);

	number_data_record_written++;
	measurement_index++;
}  /* end do_band */

int do_segment(int hsa_ud, int pass, int scan, int seg, double fstart, double fstop)
/*  Process a single segment by setting and reading spectrum analyzer */
/* uses ascii format, could be improved in future by binary??? */
/* returns the value of the last scan so the very last point can be written
    to the file */
{

	char	trace_buffer[HSA_BUF_LENGTH];		// read trace text here
	int	traceB[MAX_SA_POINTS], i;
	int	gain;

#if defined(OLD_DEBUG)
printf ("do_segment %d %d %d %lf %lf\n", pass, scan, seg, fstart, fstop);
#endif
	_settextposition(1,1);
	display_sys_status ("MEASURING");	/* display current status */
	display_meas_status (pass, scan, seg, fstart, fstop); /* display measurement status */

	hsa_ef(hsa_ud, fstop);	/* set end frequency */
	hsa_sf(hsa_ud, fstart);	/* set start frequency */
	hsa_hold (hsa_ud);  /* clear off setting */
/* 07.05  If type 4, the do the specified number of passes in peak hold */
	switch (ms_type) {	/* do according to type */
		case 1:	/* type 1 */
		case 2:	/* type 2 */
			hsa_get_traceA (hsa_ud, trace_buffer);
			hsa_M_FORMAT_to_int (trace_buffer, REF_LEVEL, traceB);
			break;
		case 3:	/* type 3 */
		case 4:	/* type 4 */
			if ((int) ms_type_arg3 > 0) {	/* 08.04 do peak_hold for specified number of times */
				hsa_max_holdA(hsa_ud);	/* set to peak hold for multiple sweeps */
				for (i=0; i < (int) ms_type_arg3-1; i++) /* 08.04 */
					hsa_sweep (hsa_ud);	/* do sweeps */
				hsa_get_traceA (hsa_ud, trace_buffer);	/* do final sweep */
				hsa_M_FORMAT_to_int (trace_buffer, REF_LEVEL, traceB);
				hsa_clear_writeA(hsa_ud);	/* reset max hold */
			}	/* end number of traces */
			else	{ /* no peak hold required */
				hsa_get_traceA (hsa_ud, trace_buffer);
				hsa_M_FORMAT_to_int (trace_buffer, REF_LEVEL, traceB);
			}
			break;
	}		/* end switch on type */


/* Remove net gain of amplifier (=max_gain - attenuation) */
	gain = (int) ((ms_antennas[ms_scans[scan].antenna].gain
		- (double) ms_antennas[ms_scans[scan].antenna].attenuation) * 100.0);
/* write the data to the file in integer format */
	for (i=0; i < hsa_points_per_trace - 1; i++) {
		fprintf (data_file, MS_DATA_RECORD_POINT, traceB[i] - gain);
		traceB[i] = traceB[i] - gain;  /* subtract off gain for temp file write below */
	} /* end for */

/*     if graphics is enabled, write out to temporary file */
	if (ms_graphics) fwrite((void *)traceB, sizeof(int), hsa_points_per_trace-1, temp_file);

	return (traceB[hsa_points_per_trace] - gain);  /* return the last point of trace */

}  /* end do_segment */

select_antenna(int ant_no, int att1_ud, int att2_ud)
/* select antenna using attenuator controllers */
/* only 1-4 are currently supported */
{
	struct COMMANDS {
		char cmd[16];
	};
	struct COMMANDS code1 [] = {"A1B23490", "B0134A29", "B12490A3", "A04B123"};
	struct COMMANDS code2 [] = {"B90", "B90", "A9B0", "A9B0"};
	
//	printf ("select antenna code = %d %s %s\n", ant_no, code1[ant_no].cmd, code2[ant_no].cmd);
//	gpib_write (att1_ud, code1[ant_no].cmd);
//	gpib_write (att2_ud, code2[ant_no].cmd);

} /* end select_antenna */

int do_configuration_file(void)
/*  Initialize, load, and verify the configuration file contents for
     the set of measurements.  File name is contained in config_file_name.
*/
{
	int	log;  /* just a place keeper */

	init_structures();	/* initialize the band, antenna, and scan structures */
	if (!read_config (full_config_file_name, log))
	{  /* error in file reading of configuration file */
		return (!TRUE);
	}

	if (!verify_config ())
	{  /* error in information file */
		return (!TRUE);
	}

	display_system_info();
	display_bands();
//	getch();
	display_antennas();
	display_scans();
//	getch();

}  /* end do_configuration_file */

write_initial_records(void)
/* write the initial data records to the output file */
{
	char	date_buffer[10], time_buffer[10];  /* holds date and time strings */

/* SBTR Ownership and legal stuff */
	fprintf (data_file, MS_REC_TYPE, MS_COMMENT_PROGRAM_ID); /* line 1 */
	fprintf (data_file, MS_COMMENT_HEADER1);
	fprintf (data_file, MS_REC_TYPE, MS_COMMENT_PROGRAM_ID); /* line 2 */
	fprintf (data_file, MS_COMMENT_HEADER2);
	fprintf (data_file, MS_REC_TYPE, MS_COMMENT_PROGRAM_ID); /* line 3 */
	fprintf (data_file, MS_COMMENT_HEADER3);
	fprintf (data_file, MS_REC_TYPE, MS_COMMENT_PROGRAM_ID); /* line 4 */
	fprintf (data_file, MS_COMMENT_HEADER4);
	fprintf (data_file, MS_REC_TYPE, MS_COMMENT_PROGRAM_ID); /* line 5 */
	fprintf (data_file, MS_COMMENT_HEADER5);

/* Program name and version */
	fprintf (data_file, MS_REC_TYPE, MS_COMMENT_PROGRAM_ID); /* prog name/version */
	fprintf (data_file, MS_COMMENT_HEADER6, PROGRAM_NAME, VERSION);

/* Data and time */
	_strdate(date_buffer);
	_strtime(time_buffer);
	fprintf (data_file, MS_REC_TYPE, MS_COMMENT_PROGRAM_ID); /* data and time */
	fprintf (data_file, MS_COMMENT_HEADER7, date_buffer, time_buffer);

/* Configuration file name */
	fprintf (data_file, MS_REC_TYPE, MS_COMMENT_PROGRAM_ID); /* config file name */
	fprintf (data_file, MS_COMMENT_HEADER8, full_config_file_name);

/* Output data file name */
	fprintf (data_file, MS_REC_TYPE, MS_COMMENT_PROGRAM_ID); /* config file name */
	fprintf (data_file, MS_COMMENT_HEADER9, data_file_name);

} /* end write_initial_records */

write_config_file_echo(void)
/*  echo the configuration file to the output file */
{
	char line[132];
	FILE *fp;
 
	if((fp=fopen(full_config_file_name, "r")) == NULL)  {    /*  open config file  */
		printf("\nwrite_config_echo:Error opening <%s>\n", full_config_file_name);    /*  return if can't  */
//		var=!TRUE;
		return(!TRUE);
	}
	rewind (fp);

/* header file of config file contents */
	fprintf (data_file, MS_REC_TYPE, MS_COMMENT_ECHO_HEADER_ID);
	fprintf (data_file, MS_COMMENT_ECHO_HEADER1, full_config_file_name);

	while(fgets(line, 80, fp) != NULL)  {    /*  get one line at a time  */
		fprintf (data_file, MS_REC_TYPE, MS_COMMENT_ECHO_ID);
		fprintf (data_file, MS_COMMENT_ECHO, line);
	}    /*  end while  */

/* end of config file header */
	fprintf (data_file, MS_REC_TYPE, MS_COMMENT_ECHO_HEADER_ID);
	fprintf (data_file, MS_COMMENT_ECHO_HEADER2, full_config_file_name);

	fclose (fp);
	return (TRUE);
}

write_system_parameters(void)
/* Write the system parameters to the output file */
{
/* (10) header record */
	fprintf (data_file, MS_REC_TYPE, MS_SYSTEM_PARAM_HEADER_ID);
	fprintf (data_file, MS_SYSTEM_PARAM_HEADER);

/* (11) Type and description of type 0.07.01 */
	fprintf (data_file, MS_REC_TYPE, MS_SYSTEM_PARAM_TYPE_ID);
	switch (ms_type) {
		case 1:
			fprintf (data_file, MS_SYSTEM_PARAM_TYPE, ms_type, 
				ms_type_arg1,
				ms_type_arg2,
				ms_type_arg3,
				ms_type_arg4,
				ms_type_arg5,
				"Outdoor in Van");
			break;
		case 2:
			fprintf (data_file, MS_SYSTEM_PARAM_TYPE, ms_type, 
				ms_type_arg1,
				ms_type_arg2,
				ms_type_arg3,
				ms_type_arg4,
				ms_type_arg5,
				"Indoor/Cart");
			break;
		case 3:
			fprintf (data_file, MS_SYSTEM_PARAM_TYPE, ms_type, 
				ms_type_arg1,
				ms_type_arg2,
				ms_type_arg3,
				ms_type_arg4,
				ms_type_arg5,
				"Fixed Location");
			break;
		case 4:
			fprintf (data_file, MS_SYSTEM_PARAM_TYPE, ms_type, 
				ms_type_arg1,
				ms_type_arg2,
				ms_type_arg3,
				ms_type_arg4,
				ms_type_arg5,
				"Moving Measurement");
			break;
		default:
			fprintf (data_file, MS_SYSTEM_PARAM_TYPE, ms_type, 
				ms_type_arg1,
				ms_type_arg2,
				ms_type_arg3,
				ms_type_arg4,
				ms_type_arg5,
				"UNKNOWN");
			break;
	}  /* end switch */

/* (12) Operator Nmae */
	fprintf (data_file, MS_REC_TYPE, MS_SYSTEM_PARAM_OPERATOR_ID);
	fprintf (data_file, MS_SYSTEM_PARAM_OPERATOR, operator_name);

/* (13) Number of passes */
	fprintf (data_file, MS_REC_TYPE, MS_SYSTEM_PARAM_NUM_PASSES_ID);
	fprintf (data_file, MS_SYSTEM_PARAM_NUM_PASSES, ms_passes);

/* (14) Spectrum analyzer saturation (attenuation) level (dB) */
	fprintf (data_file, MS_REC_TYPE, MS_SYSTEM_PARAM_SAT_ATTEN_ID);
	fprintf (data_file, MS_SYSTEM_PARAM_SAT_ATTEN, ms_saturation_dB);

/* (15) GPS setting for run 0.07.02 */
	fprintf (data_file, MS_REC_TYPE, MS_SYSTEM_PARAM_GPS_ID);
	fprintf (data_file, MS_SYSTEM_PARAM_GPS, ms_gps);


}	/* end write_system_parameters */

write_band_parameters()
/* Write the frequency band parameters to the file */
{
	int	i;

/* (30) Header */
	fprintf (data_file, MS_REC_TYPE, MS_BAND_PARAM_HEADER_ID);
	fprintf (data_file, MS_BAND_PARAM_HEADER1);
	fprintf (data_file, MS_REC_TYPE, MS_BAND_PARAM_HEADER_ID);
	fprintf (data_file, MS_BAND_PARAM_HEADER2);

/* (31) Number of Bands */
	fprintf (data_file, MS_REC_TYPE, MS_BAND_PARAM_NUMBER_ID);
	fprintf (data_file, MS_BAND_PARAM_NUMBER, ms_number_bands);

/* (32)	list all valid bands */
	for(i=0; i<MS_MAX_BANDS; i++)  {
		if(ms_bands[i].defined == TRUE)  {
			fprintf (data_file, MS_REC_TYPE, MS_BAND_PARAM_DEFINITION_ID);
			fprintf (data_file, MS_BAND_PARAM_DEFINITION,
				i+1,
				ms_bands[i].freq_start/1E6,
				ms_bands[i].freq_width/1E6,
				ms_bands[i].rbw/1E3,
				ms_bands[i].vbw/1E3,
				ms_bands[i].atten,
				ms_bands[i].number_segments,
				ms_bands[i].segment_delta_f/1.E6);
		} /* endif valid */
	} /* end for */
}	/* end write_band_parameters */

write_antenna_parameters()
/* Write the antennas band parameters to the file */
{
	int	i;

/* (30) Header */
	fprintf (data_file, MS_REC_TYPE, MS_ANTENNA_PARAM_HEADER_ID);
	fprintf (data_file, MS_ANTENNA_PARAM_HEADER);

/* Number of Bands */
	fprintf (data_file, MS_REC_TYPE, MS_ANTENNA_PARAM_NUMBER_ID);
	fprintf (data_file, MS_ANTENNA_PARAM_NUMBER, ms_number_antennas);

/* do all valid antennas */
	for(i=0; i<MS_MAX_ANTENNAS; i++) {
		if(ms_antennas[i].defined == TRUE) {
			fprintf (data_file, MS_REC_TYPE, MS_ANTENNA_PARAM_DEFINITION_ID);
			fprintf (data_file, MS_ANTENNA_PARAM_DEFINITION,
				i+1,
				ms_antennas[i].gain, 
				ms_antennas[i].amp_id, 
				ms_antennas[i].name);
		} /* endif valid */
	} /* end for */

/* do passband definitions of antennas */
	for(i=0; i<MS_MAX_ANTENNAS; i++) {
		if(ms_antennas[i].defined == TRUE) {
			fprintf (data_file, MS_REC_TYPE, MS_ANTENNA_PASSBAND_DEF_ID);
			fprintf (data_file, MS_ANTENNA_PASSBAND_DEFINITION,
				i+1,
				ms_antennas[i].freq_start/1.E6, 
				ms_antennas[i].freq_width/1.E6, 
				ms_antennas[i].rbw/1.E3, 
				ms_antennas[i].vbw/1.E3); 
		} /* endif valid */
	} /* end for */

}	/* end write_antenna_parameters */

write_scan_parameters()
/* Write the scan Table to the file */
{
	int	i;

/* (50) Header */
	fprintf (data_file, MS_REC_TYPE, MS_SCAN_PARAM_HEADER_ID);
	fprintf (data_file, MS_SCAN_PARAM_HEADER1);
	fprintf (data_file, MS_REC_TYPE, MS_SCAN_PARAM_HEADER_ID);
	fprintf (data_file, MS_SCAN_PARAM_HEADER2);

/* Number of scans */
	fprintf (data_file, MS_REC_TYPE, MS_SCAN_PARAM_NUMBER_ID);
	fprintf (data_file, MS_SCAN_PARAM_NUMBER, ms_number_scans);

/* do all valid antennas */
	for(i=0; i<ms_number_scans; i++) {
		fprintf (data_file, MS_REC_TYPE, MS_SCAN_PARAM_DEFINITION_ID);
		fprintf (data_file, MS_SCAN_PARAM_DEFINITION,
			i+1, ms_scans[i].band+1, 
			ms_scans[i].antenna+1,
			ms_scans[i].index,
			ms_scans[i].index+MS_DATA_RECORD_ID_BASE);
	} /* end for */
}	/* end write_scan_parameters */

write_diversity_record (int rec_no, int no)
/* Write the diversity record (passed as rec_no) and 3 values which are
		position_index			int
		diversity #			int	(0...n)
		total diversity positions	int	
*/
{
//printf ("write_div, rec_no, no = %d %d\n", rec_no, no); getch();
		fprintf (data_file, MS_REC_TYPE, rec_no); /* type record */
		fprintf (data_file, MS_POSITION_VAN1, number_positions_measured,
			no, (int) ms_type_arg1); /* 08.04 */
}	/* end write_diversity_record */

write_autoattenuate_record(int ant, int max, int min, int atten_value)
/* write out autoattenuate record for antenna, max, min of sweep, and
	the attenuator value determined */
{
	fprintf (data_file, MS_REC_TYPE, MS_AUTOATTEN_ID);
	fprintf (data_file, MS_AUTOATTEN, ant+1, max, min, atten_value);
}	/* end write_autoattenuate_record */

write_frequency_record(void)
/* Write the frequency records for each scan */
{
	int	i, j;
	double	f, df;

	for (i=0; i < ms_number_scans; i++)
	{  /* do each scan */
		fprintf (data_file, MS_REC_TYPE, 
			MS_FREQUENCY_RECORD_ID_BASE + ms_scans[i].index); /* record number */
  /* NOTE:  REVISED FORMAT FOR SBMS - CONENTS ARE:
	*    Start Frequency (Hz)
   *	  End Frequency (Hz)
	*    Delta Frequency (Hz)
	*  
	*  Some of this may seem cryptic since stolen from houston code and hacked.
  */
//		fprintf (data_file, MS_DATA_FREQUENCY_1, 
//			ms_bands[ms_scans[i].band].number_segments * (hsa_points_per_trace-1) + 1,  /* total number of points */
//		  	0L); /* number of points and a long 0 */
		f = ms_bands[ms_scans[i].band].freq_start;  /* start s/a frequency */
		df = ms_bands[ms_scans[i].band].rbw;

		fprintf (data_file, MS_DATA_FREQUENCY_POINT,
			f );  /* start point */
		fprintf (data_file, MS_DATA_FREQUENCY_POINT,  /* end point */
			f + df * (double) ms_bands[ms_scans[i].band].number_segments * (hsa_points_per_trace-1) + 1 );
		fprintf (data_file, MS_DATA_FREQUENCY_POINT,
			df );

//		for (j=0; j <= ms_bands[ms_scans[i].band].number_segments * (hsa_points_per_trace-1) + 1; j++)
//		{
//			fprintf (data_file, MS_DATA_FREQUENCY_POINT,
//				f + df * (double) j );
//		}
		fprintf (data_file, "\n");  /* add end of line */
	}
}	/* end write_frequency_record */

write_band_record (int pass, int scan, int atten)
/* Write the band record (which preceeds each data record)
	07.03 added the peak_hold which depends upon the type of measurement */
{
	time_t	now;

	time(&now);		/* get current time */

	fprintf (data_file, MS_REC_TYPE, MS_BAND_RECORD_ID_BASE+ms_scans[scan].index);
	switch (ms_type) {
		case 1:	/* van--fixed locations, no_peak_hold in this type */
			fprintf (data_file, MS_BAND_RECORD, atten, 0, now, ms_scans[scan].band+1,
				ms_scans[scan].antenna+1, measurement_index, pass, scan);
			break;
		case 2:	/* cart--stationary locations no_peak_hold in this type */
			fprintf (data_file, MS_BAND_RECORD, atten, 0, now, ms_scans[scan].band+1,
				ms_scans[scan].antenna+1, measurement_index, pass, scan);
			break;
		case 3:	/* stationary long term no_peak_hold in this type(??) */
			fprintf (data_file, MS_BAND_RECORD, atten, 0, now, ms_scans[scan].band+1,
				ms_scans[scan].antenna+1, measurement_index, pass, scan);
			break;
		case 4:	/* moving mobile measurements */
			fprintf (data_file, MS_BAND_RECORD, atten, (int) ms_type_arg3, now, ms_scans[scan].band+1, /* 08.04 */
				ms_scans[scan].antenna+1, measurement_index, pass, scan);
			break;
	}	/* end switch on type */
}	/* end write_band_record */

write_gps_record()
/* write a gps record (get the record here into global variables gps_info) */
/*  Note, also does update of the sceen (which updates the structure) */
{
//	tans_get_gps_status (&gps_info);	/* get the gps information */
	int	ret_status;

	ret_status = display_gps_status();

	fprintf (data_file, MS_REC_TYPE, MS_POSITION_GPS_ID);
	fprintf (data_file, MS_POSITION_GPS, 
		number_positions_measured,
		gps_info.stale,
//			gps_info.c_status,	/* replace c_status with the returned status which */
		ret_status,		/* may be -1 if not all packets are in to accompany a fix 07.08 */
		gps_info.c_pdop,
		gps_info.last_lat,
		gps_info.last_lon,
		gps_info.last_alt,
		gps_info.last_pdop,
		gps_info.last_sats[0],
		gps_info.last_sats[1],
		gps_info.last_sats[2],
		gps_info.last_sats[3]);
}	/* end write_gps_record */



write_end_record()
/* Write the ending records of the data file */
{
	time_t	end;
	char	date_buffer[10], time_buffer[10];  /* holds date and time strings */

/* End Header (time/date in text since next record is lest user friendly) */
	_strdate(date_buffer);
	_strtime(time_buffer);
	fprintf (data_file, MS_REC_TYPE, MS_COMMENT_END_HEADER_ID);
	fprintf (data_file, MS_COMMENT_END_HEADER, date_buffer, time_buffer);

/* final record contains 
		index, date/time(time_t), number_data_records, number_positions */
	time (&end);	/* get current time in time_t format */
	fprintf (data_file, MS_REC_TYPE, MS_COMMENT_END_RECORD_ID);
	fprintf (data_file, MS_COMMENT_END_RECORD, measurement_index,
		end, number_data_record_written, number_positions_measured);
}	/* end write_end_record */


display_position_graph()
/* Display the traces stored in the temporary file which match the windows
	specified in the configuration file.  */
/*	Rev 06.10, fixed bug where each graph erased on each subsequent trace */
{
	int	trace[TRACE_BUFFER_LENGTH];  /* read trace here */
	int	record_type, record_length;
	int	index;
	int	window_drawn[MS_MAX_SCANS];	/* keeps track if window already drawn */

/* clear out window already drawn, preset to !TRUE (not yet drawn) 06.10 */
	for (index = 0; index < MS_MAX_SCANS; index++)
		window_drawn[index] = !TRUE;

	rewind (temp_file);  /* rewind so at start */

	fread ((void *)&record_type, sizeof(int), 1, temp_file); /* read initial */
	while (!feof(temp_file)) {
		fread ((void *)&record_length, sizeof(int), 1, temp_file);
//printf ("display_ type, length = %d %d\n", record_type, record_length);
		fread ((void *)trace, sizeof(int), record_length, temp_file);

/* now determine if this record type is one to be graphed */
		if ((index = window_in_scan_list(record_type)) >= 0) { /* this record should be graphed */
//			printf ("This record (%d) found at index %d\n", record_type, index);
			set_graphics();	/* set graphics mode (if necessary) */
			draw_graph (trace, record_length, window_drawn[index], ms_scans[index].xl,
				ms_scans[index].xr, ms_scans[index].yt, ms_scans[index].yb,
				ms_scans[index].display_grid, 
				ms_bands[ms_scans[index].band].freq_start, 
				ms_bands[ms_scans[index].band].freq_width, 
				ms_scans[index].grid_spacing);
			window_drawn[index] = TRUE;
		} /* end if */
		fread ((void *)&record_type, sizeof(int), 1, temp_file); /* read next */
	}	/* end while */

}	/* end display_position_graph */

int window_in_scan_list (int record_index)
/* determine if this record type is one to be displayed, 
		returns the index where it was found (if found) else
		returns -1 */
{
	int	i;

	for (i=0; i<ms_number_scans; i++) 
	{  /* look through each scan record for index which matches record_index */
		if (ms_scans[i].index + MS_DATA_RECORD_ID_BASE == record_index)
			if (ms_scans[i].xl >= 0.0) { /* window defined if not (-) */
				return (i);
		}	/* end if record matches */
	}	/* end for loop */
	return (-1);
}  /* end window_in_scan_list */


set_graphics(void)
/* set the graphics screen (only if necessary) */
{
	if (!graphics_mode) { /* set into graphics mode */
/*		_getvideoconfig (&video_info);
printf ("getvideoconfig contains: %d %d\n%d %d %d\n", video_info.numxpixels,
video_info.numypixels, video_info.numtextcols, video_info.numtextrows,
video_info.numcolors); getch();
*/

		graphics_mode = TRUE;

	   if (_registerfonts("*.fon") < 0)
		{
			printf ("\bUnable to register fonts\n");
			getch();
		}
//	   if (_setfont("t'Tms Rmn' h10 w8 b") < 0)
	   if (_setfont("t'Tms Rmn' h15 w12 b") < 0)  /* larger size */
		{
			ms_graphics = !TRUE;  /* display display */
			graphics_mode = !TRUE;
			printf ("\bUnable to set font\n");
			getch();
		}

		if (_setvideomode(_VRES16COLOR) != 0)  /* try VGA */
			return(0);
		if (_setvideomode(_HRES16COLOR) != 0)  /* try EGA */
			return(0);
//		if (_setvideomode(_MRES4COLOR) != 0)  /* try CGA */
//			return(0);
/* no graphics available */
		printf ("No graphics ability available -- No display will be used\n");
		ms_graphics = !TRUE;  /* display display */
	} /* end set graphics mode */
} /* end "set_graphics " */

draw_graph (int array[], int length, int no_erase, 
	double xl, double xr, double yt, double yb,
	int enable_grid, double freq_start, double freq_width, double grid_spacing)
/* draw a window and display graph (no border yet) */
/*  if no_erase is TRUE, do not erase graph in place */
/* Added grid spacing and axis stuff 1.00 */
{
	int	pix_xl, pix_xr, pix_yt, pix_yb; /* pixels for window */
	int	i;		
	short	c;	/* for color */
	char buffer[16];
	double slope = (double) length / freq_width;  /* for scaling x axis */
	double freq, line_index;
	double scale_top = DISP_REF_LEVEL*100.0;  /* screen units, top of plot */
	double scale_bottom = (DISP_REF_LEVEL-DISP_RANGE)*100.0;

	_getvideoconfig (&video_info);  /* get current video window */
/* printf ("getvideoconfig contains: %d %d\n%d %d %d\n", video_info.numxpixels,
video_info.numypixels, video_info.numtextcols, video_info.numtextrows,
video_info.numcolors); getch(); */

#if defined (OLD_DEBUG)
printf ("draw_graph %lf %lf %lf %lf\n", xl, xr, yt, yb);
printf ("numberxpixels = %d\n", video_info.numxpixels);
#endif

	pix_xl = (int) (xl * (double) video_info.numxpixels - 1.0);
	pix_xr = (int) (xr * (double) video_info.numxpixels - 1.0);
	pix_yt = (int) (yt * (double) video_info.numypixels - 1.0);
	pix_yb = (int) (yb * (double) video_info.numypixels - 1.0);

//printf ("_setviewport (%d %d %d %d\n", pix_xl, pix_yb, pix_xr, pix_yt);
	_setviewport (pix_xl, pix_yb, pix_xr, pix_yt);

//printf ("_setwindow %lf %lf %lf %lf\n", 0.0, DISP_REF_LEVEL*100.0, (double) length, REF_LEVEL*100.0-8000.);
//	_setwindow (TRUE, 0.0, DISP_REF_LEVEL*100.0, (double) length, DISP_REF_LEVEL*100.0-11000.); /* 06.09 */
	_setwindow (TRUE, 0.0, scale_top, (double) length, scale_bottom);

/* clear out any existing windows */
	if (!no_erase) {	/* clear graph if no_erase is !TRUE 06.10 */
		c = _setcolor(0);	/* dark backgroud */
		_rectangle_w (_GFILLINTERIOR, 0.0, scale_top, (double) length,
	   	scale_bottom);
		_setcolor(c);
		_rectangle_w (_GBORDER, 0.0, scale_top, (double) length, 
	   	scale_bottom);
	}

	/* Draw the frequency scale */
	if (enable_grid)
	{
		_setlinestyle (LINE_DOTS_1);  /* set line style for small dots */
		for (freq = grid_spacing; freq < freq_width; freq += grid_spacing)
		{
			line_index = freq * slope;  /* where to put in screen units */
			_moveto_w (line_index, scale_top);
			_lineto_w (line_index, scale_bottom);

			/* Add freq */
			sprintf (buffer, "%.0lf", (freq_start + freq) / 1E6); /* to MHz */
			_moveto_w (line_index, scale_top); /* midpt */
			_outgtext (buffer);

		}  /* while freq */
		_setlinestyle (SOLID_LINE);  /* restore solid line */
	}  /* if */

	/* draw this trace */
	_moveto_w (0.0, (double) array[0]);
	for (i=1; i<length;i++) {
		_lineto_w ((double) i, (double) array[i]);
	}
}	/* end draw_graph */

reset_graphics(void)
/*  reset back to default mode */
{
	_setvideomode (_TEXTC80);	/* reset to a known state */
	graphics_mode = !TRUE;
}	/* end "reset graphics" */

set_attenuator (int unit, int dB)
/* set attenuator on unit, entry in dB (assume step of 10) */
/*  Note that lower number (5 in this case) is lsb, higher number (8) is
		msb (backwards from what I thought)  */
{

	struct COMMANDS {
		char cmd[16];
	};
	struct COMMANDS codes[] = { /* define the 16 steps of attenuation */
		"B5678", "B678A5", "B578A6", "B78A56", /* 0, 10, 20, 30 dB steps */
		"B568A7", "B68A75", "B58A76", "B8A567", /* 40, 50, 60, 70 dB */
		"B567A8", "B67A58", "B57A86", "B7A865",
		"A78B56", "A578B6", "A876B5", "A5678"} ;
	int	code;

	code = dB / 10;
//	gpib_write (unit, codes[code].cmd);
}	/* end set_attenuator */

#ifdef XXXX
sa_set_time_date(int dev)
//   Procedure to set time and date on spectrum analyzer.
//   Commands to set time and date are stored in strings
//   time_cmd and date_cmd, respectively.
{
	char time_cmd[25]="SETTIME ";
	char date_cmd[25]="SETDATE ";
	char time_buf[9], date_buf[9];
	char *time_hr, *time_min, *time_sec;
	char *date_yr, *date_mo, *date_day;

//   Get time, put in HHMMSS form, and store in string
 	_strtime(time_buf);
	time_hr=strtok(time_buf,":");
	time_min=strtok(NULL,":");
	time_sec=strtok(NULL,"\0");
	strcat(time_hr, time_min);
	strcat(time_hr, time_sec);
	strcat(time_cmd, time_hr);
//	printf ("time_cmd = %s\n", time_cmd); getch();
	gpib_write (dev, time_cmd);	/* write out time */

//   Get date, put in YYMODA form, and store in string
	_strdate(date_buf);
	date_mo=strtok(date_buf,"/");
	date_day=strtok(NULL,"/");
	date_yr=strtok(NULL,"\0");
	strcat(date_yr, date_mo);
	strcat(date_yr, date_day);
	strcat(date_cmd, date_yr);
//	printf ("date_cmd = %s\n", date_cmd); getch();
	gpib_write (dev, date_cmd);	/* write out date */
}	/* end sa_set_time_date */
#endif


display_sys_status (char message[])
/* display current status (and message) on lines 1 and 2 */
/* added kb free to display */
{
	char	descr[50];

/* Do header line */
	_settextposition(SYS_LINE1_ROW, 1);
	rev_video();	/* set reverse video */
	_outtext (SYS_TEXT_LINE);

/* Now do information */
	_settextposition(SYS_LINE2_ROW, 1);
	std_video();	/* set standard video */
	sprintf (descr, "%7ld %4d  %8s(%5ld) %10s",
		measurement_index,
		number_positions_measured,
		data_file_name,
		disk_free_kb (),	/* free disk space */
		message);
	_outtext (descr);
}	/* end display_sys_status */

display_meas_status (int pass, int scan, int seg, double fstart, double fstop)
/* display measurement status on lines 3-4 including current pass, scan, segment */
{
	char	descr[50];

/* Do header line */
	_settextposition(MEAS_LINE1_ROW, 1);
	rev_video();
	_outtext (MEAS_TEXT_LINE);

	std_video();
	_settextposition(MEAS_LINE2_ROW, 1);

	sprintf (descr, "%2d/%2d %2d/%2d %1d %2d  %5.3lf to %5.3lf (%d/%d)",
		pass+1, ms_passes, 
		scan+1, ms_number_scans,
		ms_scans[scan].antenna+1, 
		ms_antennas[ms_scans[scan].antenna].attenuation,
		fstart/1E9, fstop/1E9,
		seg+1, ms_bands[ms_scans[scan].band].number_segments);
	_outtext (descr);

}	/* end display_measurement_status */

int display_gps_status(void)
/* take the current GPS reading and load the strcutre (through gans_get_gps_status
	and display the result on the GPS status lines (standardized display) 
	based on ms_gps, use the warning or standard line.

	Returns current gps status byte (0 == locked o.k.) */
{
	char	descr[50];
	time_t	now;	/* for current time */
	int	i, j;	/* used for delay loop */
	int	ret_status;	/* returned status, if =-1, then not all information correct */

/* Do header line */
	_settextposition(GPS_LINE1_ROW, 1);
	rev_video();
	if (ms_gps == 2) /* warning line */
		_outtext (GPS_TEXT_LINE_WARN);
	else
		_outtext (GPS_TEXT_LINE);

/* Now do information */
	std_video();
	_settextposition(GPS_LINE2_ROW, 1);
	switch (ms_gps) {
		case 1:	/* gps on */
		case 2:	/* gps set to warning */
			ret_status = tans_get_gps_status (&gps_info);	/* get the gps information 07.08 */

			if (ms_gps == 2) {	/* do warning checks */
				time(&now);	/* get current time */
				if (ret_status == 0)		/* 07.08 */
//				if (gps_info.c_status == 0)	
				{ /* doing fixes */
					gps_last_warning_time = now;
				}
				else
				{	/* gps not locked, beep if time exceeded */
					if (fabs(difftime(now, gps_last_warning_time)) > (double) GPS_WARNING_TIME) 
					{  /* time to warn them, do a nasty display thing too... */
						/* do a funky message here */
						for (i = 0; i < 20; i++ ){
							strnset (descr, '.', 40);
							strcpy (&descr[i], "GPS NOT LOCKED.");
							display_message (6, descr);
							for (j=0; j < 15000; j++)	;	/* delay loop */
						}	/* end of funky message */
						printf ("\a");	/* sound bell here */
						for (i = 20; i > 0; --i ){ /* now do backwards */
							strnset (descr, '.', 40);
							strcpy (&descr[i], "GPS NOT LOCKED.");
							display_message (6, descr);
							for (j=0; j < 15000; j++)	;	/* delay loop */
						}	/* end of funky message */

						_settextposition(GPS_LINE2_ROW, 1);
						gps_last_warning_time = now;
					}
				}
			}	/* end gps warning check */
			switch (ret_status) {  /* 07.08 */
//			switch (gps_info.c_status) {
				case -1:	/* still waiting for all packets info to accompany a fix 07.08 */
					sprintf (descr, "Wait   %2d.%2d.%2d.%2d %3.1f %7.4lf %9.4lf",
						gps_info.c_sats[0],
						gps_info.c_sats[1],
						gps_info.c_sats[2],
						gps_info.c_sats[3],
						gps_info.c_pdop,
						gps_info.last_lat,
						gps_info.last_lon);
						_outtext (descr);
					break;
				case 0:	/* doing fixes */
					sprintf (descr, "Locked %2d.%2d.%2d.%2d %3.1f %7.4lf %9.4lf",
						gps_info.c_sats[0],
						gps_info.c_sats[1],
						gps_info.c_sats[2],
						gps_info.c_sats[3],
						gps_info.c_pdop,
						gps_info.last_lat,
						gps_info.last_lon);
						_outtext (descr);
					break;
				case 1:	/* no GPS time yet */
					sprintf (descr, "NoTime %2d.%2d.%2d.%2d %3.1f %7.4lf %9.4lf",
						gps_info.c_sats[0],
						gps_info.c_sats[1],
						gps_info.c_sats[2],
						gps_info.c_sats[3],
						gps_info.c_pdop,
						gps_info.last_lat,
						gps_info.last_lon);
						_outtext (descr);
					break;
				case 8:	/* no sats */
					sprintf (descr, "NO Sat %2d.%2d.%2d.%2d %3.1f %7.4lf %9.4lf",
						gps_info.c_sats[0],
						gps_info.c_sats[1],
						gps_info.c_sats[2],
						gps_info.c_sats[3],
						gps_info.c_pdop,
						gps_info.last_lat,
						gps_info.last_lon);
						_outtext (descr);
					break;
				case 9:	/* 1 satellite */
					sprintf (descr, "1 Sat  %2d.%2d.%2d.%2d %3.1f %7.4lf %9.4lf",
						gps_info.c_sats[0],
						gps_info.c_sats[1],
						gps_info.c_sats[2],
						gps_info.c_sats[3],
						gps_info.c_pdop,
						gps_info.last_lat,
						gps_info.last_lon);
						_outtext (descr);
					break;
				case 0xa:	/* 2 satellites */
					sprintf (descr, "2 Sat  %2d.%2d.%2d.%2d %3.1f %7.4lf %9.4lf",
						gps_info.c_sats[0],
						gps_info.c_sats[1],
						gps_info.c_sats[2],
						gps_info.c_sats[3],
						gps_info.c_pdop,
						gps_info.last_lat,
						gps_info.last_lon);
						_outtext (descr);
					break;
				case 0xb:	/* 3 satellites */
					sprintf (descr, "3 Sat  %2d.%2d.%2d.%2d %3.1f %7.4lf %9.4lf",
						gps_info.c_sats[0],
						gps_info.c_sats[1],
						gps_info.c_sats[2],
						gps_info.c_sats[3],
						gps_info.c_pdop,
						gps_info.last_lat,
						gps_info.last_lon);
						_outtext (descr);
					break;
				default:
					sprintf (descr, "Unknwn %2d.%2d.%2d.%2d %3.1f %7.4lf %9.4lf",
						gps_info.c_sats[0],
						gps_info.c_sats[1],
						gps_info.c_sats[2],
						gps_info.c_sats[3],
						gps_info.c_pdop,
						gps_info.last_lat,
						gps_info.last_lon);
						_outtext (descr);
					break;
			}	/* end switch on status */
			break;
		case 0:  /* no gps in use */
			strcpy (descr, "Off");
			_outtext (descr);
			break;
	} /* end switch on ms_gps */
	if (ms_gps == 0) /* no gps */
		return (0);	/* return all o.k. in this case */
	else
		return (ret_status);	/* return current status (including -1) 07.08 */
}	/* end display_gps_status */
	
display_message (int line, char message[])
/* display a 'message' on specified line (in reverse video) */
{

/* Do header line */
	_settextposition(line, 1);
	_settextcolor(2);	/* fix it here */

/* now write the header */	
	_outtext (message);
	std_video();

}	/* end display_message */

rev_video(void)
/*  Set the colors if text and backround (as appropriate) based upon the
	graphics set for REVERSE VIDEO */
{
	if (graphics_mode)
	{	/* graphics is on, set low level */
		_settextcolor (1);
	}
	else
	{	/* no graphics, set appropriate color */
		_setbkcolor (3L);
		_settextcolor (0);
	}
}	/* end reverse video */

std_video(void)
/*  Set the colors if text and backround (as appropriate) based upon the
	graphics set for STANDARD VIDEO */
{
	if (graphics_mode)
	{	/* graphics is on, set low level */
		_settextcolor (3);
	}
	else
	{	/* no graphics, set appropriate color */
		_setbkcolor (0L);
		_settextcolor (3);
	}
}	/* end reverse video */

clr_line (int line)
/* Clear the specified line (40cols) in std_video */
{
	int	i;

	_settextposition (line,1);
	std_video();
	for (i=0; i< 40; i++) 
		_outtext (" ");
}	/* end clr_line */

int check_free_disk_space (void)
/* gets the free disk space (from disk_free_kb procedure which checks
	disk C:) and returns true if the available space is less than the level
	specified by MIN_FREE_DISK_SIZE. Also display message and waits for operator
	to acknowledge this error. version 0.06.07 */
{
	int	ret = !TRUE;
	if (disk_free_kb() <= (long) MIN_FREE_DISK_SIZE)
	{	/* low disk space condition is here */
		ret = TRUE;
		printf ("\a");	/* alert */
		display_message (7,"The data disk is nearly full");
		display_message (8,"Program will stop.  PRESS RETURN");
		printf ("\a");	/* alert */
		getch();
	}	/* end low disk space */
	return (ret);
}	/* end check_free_disk_space */

unsigned long disk_free_kb(void)
/* returns kbytes free (rounded down) as long integer value */
/*  Only checks the C drive (disk 3) */
/* Source Microcsoft C Bible  pg 608 */

{
//unsigned long total_space, free_space, bytes_per_cluster;
	unsigned long free_space, bytes_per_cluster;
	struct	diskfree_t	dfinfo;

	if (_dos_getdiskfree(3, &dfinfo) != 0) /* can't do this disk */
	{
		printf ("Error in function\n");
		return (0L);
	}
	bytes_per_cluster = dfinfo.sectors_per_cluster * dfinfo.bytes_per_sector;
	free_space = dfinfo.avail_clusters * bytes_per_cluster;
	free_space = free_space / 1000L;
	return (free_space);
}	/* end disk_free_kb */

		
empty_keyboard_buffer(void)
/*  Be sure the keyboard buffer is empty by dumping any outstanding keystrokes */
{
	while (kbhit())
		getch();
}	/* end empty_keyboard_buffer */


/*  read_con.c		Houston Phase 1A data collection software  */

/*  Cameron Coursey	started 3/92  */

/* REVISION HISTORY -------------------------------------------
Version 0.4  3/30/92	added in autoattenuate functions
Version 0.05  4/4/92  added gps ability in ms_gps 
Version 0.06  4/9/92  Bug fixes
	0.06.1	Calculated number of segments incorrectly, revised calc. csp 
	0.06.2	Make sure (in verify) that all scan id's are unique
Version 0.07  4/11/92  New features
	0.07.01  type 1 and 2 have one argument, type 4 now has 3 arguements
	0.07.02	gps now has three possible arguments
Version 0.08  4/12/92 csp
	0.08.01	accept arg1 and arg2 with "s", "m", "h" appended to value and
				adjust to minutes, hours as necessary.
	0.08.02  arg 4 is duration for type 3 measurements
	0.08.03	made all ms_type_arg as long integers
	0.08.04	allow 'd' for days in arguments for time duration, etc.
 -------------------------------------------------------------*/

init_structures()
{
	int i;
	for(i=0; i<MS_MAX_BANDS; i++)
		ms_bands[i].defined=!TRUE;    /*  no bands defined before reading config file  */
	for(i=0; i<MS_MAX_ANTENNAS; i++)
		ms_antennas[i].defined=!TRUE;    /*  no antennas defined before reading config file  */
		strcpy(ms_antennas[i].name," ");
	for(i=0; i<MS_MAX_SCANS; i++)    /*  no graphics display window defined before reading config file  */
		ms_scans[i].xl = ms_scans[i].xr = ms_scans[i].yt = ms_scans[i].yb = -1.;
}

read_config(fname,log)
//  Procedure to read configuration file (passed in "fname").
//  Returns TRUE if config file opened and read OK.
//  Does not do explicit validity checks of variables, just loads structures.
//  Cameron Coursey, 3/19/92
//	Revised type 3 and type 4 to take s,m,h arguments
char fname[];
{
	double atof();
	char line[80], *pos1, *grph, *gps, buf2[16], char_in;
	int var=TRUE, errors=0;
	int num_bnd, temp_type, scan;
	FILE *fopen(), *fp;
	printf ("\nTrying to open config file <%s>\n", fname);
	if((fp=fopen(fname, "r")) == NULL)  {    /*  open config file  */
		printf("\n** Error opening <%s> **\n",fname);    /*  return if can't  */
		var=!TRUE;
		return(var);
	}
	printf ("\nReading config file <%s>\n", fname);
	while(fgets(line, 80, fp) != NULL)  {    /*  get one line at a time  */
		pos1=strtok(line, "\t");    /*  store first field  */
		if(strnicmp(pos1, "type", 4) == 0)  { /* DEFINE TYPE */
			temp_type=atoi(strtok(NULL, "\t"));
			if((temp_type > 0) && (temp_type <= 4)) { /* valid type */
				ms_type=temp_type;    /*  define type of measurement  */
/* Read in arguments for various types 0.07.01 */
				switch (ms_type) {
					case 1:	/* van, one argument required */
						ms_type_arg1=atol(strtok(NULL, "\t"));    /*  argument 1  */
						if (ms_type_arg1 < 1L) ms_type_arg1 = 1L;	/* ensure one diversity pass */
						break;
					case 2:	/* cart, one argument required */
						ms_type_arg1=atol(strtok(NULL, "\t"));    /*  argument 1  */
						if (ms_type_arg1 < 1L) ms_type_arg1 = 1L;	/* ensure one pass */
						break;
					case 3:	/* fixed location same as moving measurements */
					case 4:	/* mobile, three arguements reqd */
/* arg 1 = time arguemnt--allow "s,m.h" for seconds, minutes, hours 08.01 */
						strcpy (buf2, strtok(NULL, "\t"));	/* get string */
						ms_type_arg1 = atol (buf2);	/* default is seconds */
						if (strcspn(buf2, "d") != strlen(buf2)) /* hours is specified */
							ms_type_arg1 *= 24L * 3600L;	/* convert days to seconds */
						if (strcspn(buf2, "h") != strlen(buf2)) /* hours is specified */
							ms_type_arg1 *= 3600L;	/* convert minutes to seconds */
						if (strcspn(buf2, "m") != strlen(buf2)) /* minutes is specified */
							ms_type_arg1 *= 60L;	/* convert minutes to seconds */
//						ms_type_arg1=atoi(strtok(NULL, "\t"));    /*  argument 1  */

/* arg 2 = time arguemnt--allow "s,m.h" for seconds, minutes, hours 08.01 */
						strcpy (buf2, strtok(NULL, "\t"));	/* get string */
						ms_type_arg2 = atol (buf2);	/* default is seconds */
						if (strcspn(buf2, "d") != strlen(buf2)) /* hours is specified */
							ms_type_arg2 *= 24L * 3600L;	/* convert days to seconds */
						if (strcspn(buf2, "h") != strlen(buf2)) /* hours is specified */
							ms_type_arg2 *= 3600L;	/* convert minutes to seconds */
						if (strcspn(buf2, "m") != strlen(buf2)) /* minutes is specified */
							ms_type_arg2 *= 60L;	/* convert minutes to seconds */
//						ms_type_arg2=atoi(strtok(NULL, "\t"));    /*  argument 2  */

						ms_type_arg3=atol(strtok(NULL, "\t"));    /*  argument 3  */
						if (ms_type_arg3 == 1L) ms_type_arg3 = 0L;	/* one trace = no peak hold */
						if (ms_type == 3) { /* type 3 has duration as arg #4 */
							strcpy (buf2, strtok(NULL, "\t"));	/* get string */
							ms_type_arg4 = atol (buf2);	/* default is seconds */
							if (strcspn(buf2, "d") != strlen(buf2)) /* hours is specified */
								ms_type_arg4 *= 24L * 3600L;	/* convert days to seconds */
							if (strcspn(buf2, "h") != strlen(buf2)) /* hours is specified */
								ms_type_arg4 *= 3600L;	/* convert minutes to seconds */
							if (strcspn(buf2, "m") != strlen(buf2)) /* minutes is specified */
								ms_type_arg4 *= 60L;	/* convert minutes to seconds */
						}
						break;	/* case 3:4 */
				}	/* end switch_type */				
			}	/* endif */
			else  {
				printf("\n** Illegal measurement type **\n");
				errors++;
			}
		}
		else if(strnicmp(pos1, "sat", 3) == 0)  { /* SPEC ANALYZER SATURATION POINT */
			temp_type=atoi(strtok(NULL, "\t"));
			if((temp_type <= 0) && (temp_type >= -110))
				ms_saturation_dB=temp_type;    /*  define saturation point */
			else  {
				printf("\n** Illegal saturation Point: %d dB **\n", temp_type);
				errors++;
			}
		}
		else if(strnicmp(pos1, "db",2) == 0)  {  /* DEFINE BAND */
			num_bnd=atoi(strtok(NULL, "\t"));
			if((num_bnd > 0) && (num_bnd <= MS_MAX_BANDS))  {
				ms_bands[num_bnd-1].freq_start=atof(strtok(NULL, "\t"));    /*  define start of band  */
				ms_bands[num_bnd-1].freq_width=atof(strtok(NULL, "\t"));    /*  define total span of band  */
				ms_bands[num_bnd-1].rbw=atof(strtok(NULL, "\t"));    /*  define resolution bandwidth  */
				ms_bands[num_bnd-1].vbw=atof(strtok(NULL, "\t"));    /*  define video bandwidth  */
				ms_bands[num_bnd-1].atten=atof(strtok(NULL, "\t"));    /*  define attenuation for spectrum analyzer  */
				ms_bands[num_bnd-1].defined=TRUE;
				ms_number_bands++;	/* number of bands defined */
			}
			else  {
				printf("\n** Illegal band number **\n");
				errors++;
			}
		}
		else if(strnicmp(pos1, "da",2) == 0)  {	/* DEFINE ANTENNA */
			num_bnd=atoi(strtok(NULL, "\t"));
			if((num_bnd > 0) && (num_bnd <= MS_MAX_ANTENNAS))  {
				ms_antennas[num_bnd-1].gain=atof(strtok(NULL, "\t"));    /*  define gain of antenna  */
				ms_antennas[num_bnd-1].amp_id=atoi(strtok(NULL, "\t"));    /*  define amplifier id  */
				ms_antennas[num_bnd-1].freq_start=atof(strtok(NULL, "\t"));    /*  passband freq start  */
				ms_antennas[num_bnd-1].freq_width=atof(strtok(NULL, "\t"));    /*  passband freq width  */
				ms_antennas[num_bnd-1].rbw=atof(strtok(NULL, "\t"));    /*  passband res bw width  */
				ms_antennas[num_bnd-1].vbw=atof(strtok(NULL, "\t"));    /*  passband video bw width  */
				ms_antennas[num_bnd-1].defined=TRUE;
				ms_number_antennas++;
			}
			else  {
				printf("\n** Illegal band number for antenna **\n");
				errors++;
			}
		}
		else if(strnicmp(pos1, "na",2) == 0)  {
			num_bnd=atoi(strtok(NULL, "\t"));
			strcpy(ms_antennas[num_bnd-1].name,strtok(NULL, "\n"));    /*  define antenna name  */
		}
		else if(strnicmp(pos1, "ds",2) == 0)  {  /* put in MS_MAX_SCANS  */
			ms_number_scans++;
			if(ms_number_scans <= MS_MAX_SCANS) {
				ms_scans[ms_number_scans-1].band=atoi(strtok(NULL, "\t"))-1;    /*  define band number for scan  */
				ms_scans[ms_number_scans-1].antenna=atoi(strtok(NULL, "\t"))-1;    /*  define antenna number  */
				ms_scans[ms_number_scans-1].index=atoi(strtok(NULL, "\t"));    /*  define index for writing records  */
			}
			else  {
				printf("\n** Illegal number of scans (too many) **\n");
				errors++;
			}
		}
		else if(strnicmp(pos1, "np",2) == 0)
			ms_passes=atoi(strtok(NULL, "\t"));    /*  define number of passes per scan  */
		else if(strnicmp(pos1, "graph",4) == 0)  {
			grph=strtok(NULL, "\t");
			if(strnicmp(grph, "on",2) == 0)    /*  define graphics display on  */
				ms_graphics=TRUE;
			else if(strnicmp(grph, "off",3) != 0)  {
				printf("\n** Illegal graph command  **\n");
				errors++;
			}
		}
/* check for gps (0.07.02) */
		else if(strnicmp(pos1, "gps",3) == 0)  {
			gps=strtok(NULL, "\t");
			if(strnicmp(gps, "on",2) == 0)    /*  define gps on  */
				ms_gps=TRUE; /* = 1 */
			else if (strnicmp(gps, "warn", 4) == 0)
				ms_gps = 2;	/* warning option on 0.07.02 */
			else if(strnicmp(gps, "off",3) != 0)  {
				printf("\n** Illegal graph command  **\n");
				errors++;
			}
		}

		else if(strnicmp(pos1, "win",3) == 0)  {    /*  define graphics window size  */
			scan=atoi(strtok(NULL, "\t"))-1;
			if((scan >= 0) && (scan < MS_MAX_SCANS))  {
				ms_scans[scan].xl=atof(strtok(NULL, "\t"));    /*  define left margin  */
				if((ms_scans[scan].xl < 0.) || (ms_scans[scan].xl > 1.))  {
					printf("\n** Illegal left margin of graphics window  **\n");
					errors++;
				}
				ms_scans[scan].xr=atof(strtok(NULL, "\t"));    /*  define right margin  */
				if((ms_scans[scan].xr <= ms_scans[scan].xl) || (ms_scans[scan].xr > 1.))  {
					printf("\n** Illegal right margin of graphics window **\n");
					errors++;
				}
				ms_scans[scan].yt=atof(strtok(NULL, "\t"));    /*  define top margin  */
				if((ms_scans[scan].yt < 0) || (ms_scans[scan].yt > 1.))  {
					printf("\n** Illegal top margin of graphics window **\n");
					errors++;
				}
				ms_scans[scan].yb=atof(strtok(NULL, "\t"));    /*  define bottom margin  */
				if((ms_scans[scan].yb < 0) || (ms_scans[scan].yb > 1.))  {
					printf("\n** Illegal bottom margin of graphics window **\n");
					errors++;
				}
				if(ms_scans[scan].yt <= ms_scans[scan].yb)  {
					printf("\n** Illegal top margin of graphics window **\n");
					errors++;
				}
				/* Now check for grids enabled (ver 1.00) */
				char_in = *(strtok(NULL, "\t"));    /*  get first characater  */
			   switch (char_in)
				{
					case 'T':
					case 't':
					case 'Y':
					case 'y':
						ms_scans[scan].display_grid = 1;  /* TRUE */
						break;
					default:
						ms_scans[scan].display_grid = 0;  /* !TRUE */
						break;
				}  /* switch */
				/* If display grid is true, then read the grid spacing */
				if (ms_scans[scan].display_grid)
				{
					ms_scans[scan].grid_spacing=atof(strtok(NULL, "\t\n"));    /* spacing  */
				}  /* if display_grid */
			}
			else  {
				printf("\n** Illegal scan number for graphics window  **\n");
				errors++;
			}
		}
//                else if(strnicmp(pos1, "gps",3) == 0)
//                        ms_gps=atoi(strtok(NULL, "\t"));    /*  define whether to use GPS positioning (future)  */
	}    /*  end while  */
	printf("\nFinished reading config file <%s>\n",fname);
	printf("\n** %d error(s) encountered in read **\n", errors);
	if(errors > 0)
		var=!TRUE;
	return(var);
}

verify_config()
{
	int i, j, band_chk, ant_chk;
	int verf=TRUE, errors=0;
	printf("\nVerifying configuration\n");
/* set up the delta_f for each band and number of segments */
	for(i=0; i<MS_MAX_BANDS; i++)  {
		if(ms_bands[i].defined == TRUE)  {
			ms_bands[i].segment_delta_f=(double)(hsa_points_per_trace-1)*ms_bands[i].rbw;
//			ms_bands[i].number_segments=(int)(ms_bands[i].freq_width/ms_bands[i].segment_delta_f);
			ms_bands[i].number_segments=(int)  /* revision 0.06.1 */
			((ms_bands[i].freq_width-ms_bands[i].segment_delta_f)/ms_bands[i].segment_delta_f) + 1; 
		}    /*  end if  */
	}    /*  end for  */
/* do the scan table */
	for(i=0; i<ms_number_scans; i++)  {
		band_chk=ms_scans[i].band;
		if((band_chk >= 0) && (band_chk < MS_MAX_BANDS))  {
			if(ms_bands[band_chk].defined != TRUE)  {
				printf("\n** Band %d not defined for scan **\n",band_chk+1);
				++errors;
			}
		}
		else  {
			printf("\n** Band %d out of range for scan **\n");
			++errors;
		}
		ant_chk=ms_scans[i].antenna;
		if((ant_chk >= 0) && (ant_chk < MS_MAX_ANTENNAS))  {
			if(ms_antennas[ant_chk].defined != TRUE)  {
				printf("\n** Antenna %d not defined for scan **\n",ant_chk+1);
				++errors;
			}
		}
		else  {
			printf("\n** Antenna %d out of range for scan **\n");
			++errors;
		}
		if((ms_scans[i].index <= 0) || (ms_scans[i].index > 99))  {
			printf("\n** Scan index (%d) out of range **\n", ms_scans[i].index);
			++errors;
		}
/* verify that this index is unique for all scans (rev 0.6.2) */
/* NOTE: Only need to check those records beyond the current record, the previus
	ones will already have been checked (otherwise, get twice as many errors as there
	actually are. */
		for (j=i+1; j < ms_number_scans; j++) { //			if (j != i) {	/* do not compare agains myself */
				if (ms_scans[i].index == ms_scans[j].index) 
				{	/* a match indicates duplicate index's, this is an error */
					errors++;
					printf ("Duplicate index found in scan table\n");
					printf ("  in scan line %d and %d--duplicate index is %d\n",
						i+1, j+1, ms_scans[i].index);
				}	/* end if duplicate */
//			}	/* end do not compare against myself */
		}	/* end for */
	}
	printf("\nVerification completed with %d errors\n",errors);
	if(errors > 0)
		verf=!TRUE;
	return(verf);
}

display_system_info()
{
	char    string[24];
	printf ("\nSYSTEM INFORMATION =======================\n");
	switch (ms_type) { /* set string descriptor of type */
		case 1:
			strcpy (string, "Van (std)");
			break;
		case 2:
			strcpy (string, "Cart");
			break;
		case 3:
			strcpy (string, "Fixed Location");
			break;
		case 4:
			strcpy (string, "Moving Measurements");
			break;
		default:
			strcpy (string, "UNKNOWN");
			break;
	}  /* end case */
	printf ("Type : %d <%s>\n", ms_type, string);
	printf ("Number_passes: %d\n", ms_passes);
}    /*  end display_system_info  */

display_bands()
{
	int i, band, atten;
	double sfreq, span, res, vid;
	printf ("\n\nFREQUENCY BAND INFORMATION ===========================\n");
	printf("\nBand \t Start Freq \t  Span   \t Res BW \t Video BW \t Atten");
	printf(  "\n     \t    (MHz)   \t  (kHz)  \t (kHz)  \t   (kHz)  \t (dB) ");
	printf(  "\n---- \t ---------- \t  -----  \t ------ \t -------- \t -----");
	for(i=0; i<MS_MAX_BANDS; i++)  {
		if(ms_bands[i].defined == TRUE)  {
			band=i+1;
			sfreq=ms_bands[i].freq_start/1.e6;
			span=ms_bands[i].freq_width/1.e6;
			res=ms_bands[i].rbw/1.e3;
			vid=ms_bands[i].vbw/1.e3;
			atten=ms_bands[i].atten;
			printf("\n %d \t %-.3f \t %-.3f \t %-.3f \t  %-.3f \t   %d", band, sfreq, span, res, vid, atten);
		}
	}
	for(i=0; i<MS_MAX_BANDS; i++)  {
		if(ms_bands[i].defined == TRUE)  {
			printf("\nnumber_segments(%d) = %d \t segment_delta_f(%d) = %-.3f",
			i, ms_bands[i].number_segments, i, ms_bands[i].segment_delta_f);
		}
	}
}    /*  end display_bands  */

display_antennas()
{
	int i;
	printf ("\n\nANTENNA INFORMATION ============================\n");
	printf("\nAntenna \t Gain (dB) \t  Amp  \t Name");
	printf("\n------- \t --------- \t  ---  \t ----");
	for(i=0; i<MS_MAX_ANTENNAS; i++)
		if(ms_antennas[i].defined == TRUE)
			printf("\n %d \t\t %-.3f \t   %d \t %s", i+1, ms_antennas[i].gain, ms_antennas[i].amp_id, ms_antennas[i].name);
}    /*  end display_antennas  */

display_scans()
{
	int i;
	printf ("\n\nSCAN INFORMATION ============================\n");
	printf ("\nScan\tBand\tAntenna\tIndex\n");
	printf ("----\t----\t-------\t-----\n");
	for (i=0; i < ms_number_scans; i++)
		printf("\n%d \t %d \t   %d \t  %d", i+1, ms_scans[i].band+1, ms_scans[i].antenna+1,
		  ms_scans[i].index);
	if(ms_graphics != TRUE)
		printf("\n\nGraphics display off");
	else  {
		printf("\n\nGraphics display on\n");
		printf("Scan \t Left(%%) \t Right(%%) \t Top(%%) \t Bottom(%%)\n");
		printf("---- \t ------- \t -------- \t ------ \t ---------");
		for(i=0; i < ms_number_scans; i++)
			if(ms_scans[i].xl != -1.)
				printf("\n%d \t %-.3f \t\t %-.3f \t\t %-.3f \t\t %-.3f", i+1,
				ms_scans[i].xl, ms_scans[i].xr, ms_scans[i].yt, ms_scans[i].yb);
	}

	printf("\n\n");
}    /* end display_scans */

#define	PROGRAM_NAME	"cs_sort"
#define	VERSION 0.01	/* current version number of program */

/* -----------------------------------------------------------
	 cs_sort.c

	Invoked with:

		cs_sort in_file.ext [help level]
		 where in_file.ext is the input file
		 help level is the level of messages displayed
			blank or 0:  minimal (normal)
			1: a little more
			9: most (debug mode)

		THE OUTPUT FILE is in_file.srt (for sorted data)

		Input/Output format is 1 line per signal strength reading and write in
		 the line format of:
			(long) time (hhmmss, ex 8:29:55 AM would be 82955)
			(int)	channel
			(char) direction (F or R)
			(int) signal strength (average value, the peak avg if reduced from
				more than 1 line)
		 All fields are separated by tabs.


		New: 8/6/92 csp

------------------------------------------------------------------- */

#include <stdio.h>
#include	<search.h>
#define	TRUE	1

#define	IN_BUF_SIZE	132	/* Max buffer size of input file */
#define	MAX_SORT_SIZE	8000	/* Maximum number lines we can sort */

int	debug = 1;	/* debug level (passed in on command line) */

/* Sorting structure will use qsort on the following structure*/
	struct  S {
		int	key;	/* key to sort on */
		fpos_t	fpos;	/* file position (attribute */
	};

	struct S dummy_S;	/* need this definition for sizeof (dummy_S) */
	struct S s[MAX_SORT_SIZE];	/* array to hold what we will sort */

/* Compare function for qsort below */
int mycompare ( int *arg1, int *arg2)
{
	if ( *arg1 < *arg2)
		return (-1);
	if ( *arg1 > *arg2)
		return (1);
	return (0);
}

main (int argc, char *argv[])
{
	char	input_file_name[128];
	char	output_file_name[128];
	char	in_buf[IN_BUF_SIZE];
	FILE	*in_file, *out_file;
	int	dot_index;
	int	lines_input = 0;	/* counts number of input lines read */
	int	chan, avg;	/* readings from a record */
	char	direction;	/* holds direction flag (F or R) */
	long	record_time;	/* holds time in hhmmss (base 10) */
	fpos_t	curpos;	/* current position of file */
	int	i;
	int	index;

	/* Print the legal stuff */
	if (debug > 0) 
	{
		printf ("RESTRICTED USE for SOUTHWESTERN BELL MOBILE SYSTEMS ONLY\n");
		printf ("%s Version %5.2f -- Sort Utility\n", PROGRAM_NAME, VERSION);
	}

	/* Check if proper parameters passed */
	if (argc < 2)
	{	/* not enough paramters given */
		display_proper_usage();
		exit(1);
	}
	strcpy (input_file_name, argv[1]);	/* get full file name */
	if (argc >= 3)
	{	/* check if a debug is specified */
		debug = atoi (argv[2]);	/* get the debug number */
	}
	if (debug > 8) 
	{
		printf ("argc = %d\n", argc);
		printf ("debug level = %d, file name = <%s>\n", debug, input_file_name);
	}

	/* open input file */
	if ((in_file = fopen(input_file_name, "r")) == NULL)
	{	/* open failed */
		printf ("The input file, %s, does not exist\n", input_file_name);
		display_proper_usage();
		exit (1);
	}

	/* Open output file */
	/* NOTE:  Uses the same file name but with extension of .srt */
	strcpy (output_file_name, input_file_name);
	dot_index = strcspn (output_file_name, ".");	/* look for start of extension */
	if (dot_index == strlen(output_file_name))
	{	/* no extension specified on file name, add it here */
		strcat (output_file_name, ".SRT");
	}
	else
	{	/* extension specified, revise with .SRT */
		output_file_name[dot_index] = (char) NULL;	/* terminate string, overwrite .ext */
		strcat (output_file_name, ".SRT");
	}

	if (debug > 8) printf ("Output file name is <%s>\n", output_file_name);

	if ((out_file = fopen(output_file_name, "w")) == NULL)
	{	/* open failed */
		printf ("The output file, %s, could not be created.\n", output_file_name);
		display_proper_usage();
		exit (1);
	}

	/* Go through each line and build a memory based list to sort */
	rewind (in_file);

	if (debug > 1) printf ("Reading Input File...Please Wait");

	while (!feof (in_file))
	{	/* read each input line */
		fgetpos (in_file, &curpos);
		fscanf (in_file, "%ld %d %c %d\n", &record_time, &chan, &direction, &avg);
		if (lines_input >= MAX_SORT_SIZE)
		{	/* too many lines to sort */
			printf ("\aToo Many Lines to Sort\n");
			printf ("Max Line Length is %d lines\n", MAX_SORT_SIZE);
			exit (1);
		}
		
		s[lines_input].key = chan;	/* sort on channel number */
		s[lines_input].fpos = curpos;	/* save attribute of current file position */
		
		lines_input++;	/* next line */
	}	/* go to next line of file */

	/* now sort the list of keys with attribute tags */
	if (debug > 1) printf ("\nSorting File...Please Wait");
	qsort ((void *) s, (size_t) lines_input, (size_t) sizeof (dummy_S), mycompare);

//	printf ("\nAfter Sort\n");
//	for (i=0; i < lines_input; i++)
//		printf ("%d %d %ld\n", i, s[i].key, s[i].fpos);

	/* Now write out the file (to out_file) in sorted order */
	if (debug > 1) printf ("\nWriting to Output File...Please Wait");
	for (index = 0; index < lines_input; index++)
	{	/* read each input line */
		fsetpos (in_file, &s[index].fpos);	/* set to the desired record */
		fscanf (in_file, "%ld %d %c %d\n", &record_time, &chan, &direction, &avg);
		fprintf (out_file, "%ld\t%d\t%c\t%d\n", record_time, chan, direction, avg);
	}	/* go to next line of file */

#if defined (DUMBO)
		if (strlen (in_buf) > 0)
		{	/* valid input line */
			if (strstr (in_buf, "Avg:") != NULL)
			{	/* this is a valid line, it contains the string "Avg:" */
				hour = atoi (&in_buf[9]);	/* hour of record */
				min = atoi (&in_buf[12]);	/* minute of record */
				sec = atoi (&in_buf[15]);	/* hour of record */
				record_time = (long) hour * 10000L + (long) min * 100 + (long) sec;
				chan = atoi (&in_buf[24]);	/* second number */
				direction = in_buf[28];	/* F or R for forward or reverse channel */
				avg = atoi (&in_buf[52]);	/* get avg signal strength */

				fprintf (out_file, "%6ld\t%4d\t%c\t%5d\n", record_time, chan, direction, avg);
//				fputs (in_buf, out_file);	/* temporary */
				lines_output++;
			}
		}
	}
#endif

	/* Close the files */
	fclose (in_file);
	fclose (out_file);

	/* Display counts */
	printf ("%d Lines Sorted\n", lines_input);

	/* If ready, go onto the next routine, cs_sort */

}

display_proper_usage(void)
/*  Display the proper usage for this program. */
{
	printf ("PROPER USAGE: cs_sort in_file.ext [help level]\n");
	printf ("  where in_file.ext is the input file\n");
	printf ("  help level is the level of messages displayed (0=minimal, 9 = max)\n");
	printf ("  THE OUTPUT FILE (sorted data) in_file.srt\n");
}

wait(void)
/* A debug routine which tells waiting for a cr */
{
	printf ("waiting");
	getch();
}

#if defined (DUMBO)
/* Test of qsort to see if can sort on a structure */

#include	<stdio.h>

int mycompare (int *, int *);	/* define my sort */
main(void)
{

	struct  S {
		unsigned	key;
		long	index;
	};

	struct S dummy_S;
	struct S s[5]; /* = { {1,10L}, {4, 40L}, {3, 30L}, (10, 100L}, {8, 80L} }; */

	int	len_s = 5;	/* elements to sort */
	int	i;

printf ("Size of s is %d\n", sizeof (dummy_S));
	s[0].key = 1;
	s[0].index = 10L;
	s[1].key = 4;
	s[1].index = 40L;
	s[2].key = 3;
	s[2].index = 30L;
	s[3].key = 10;
	s[3].index = 100L;
	s[4].key = 8;
	s[4].index = 80L;



	printf ("Before sort\n");
	for (i=0; i < len_s; i++)
		printf ("%d %d %ld\n", i, s[i].key, s[i].index);

//	qsort ((void *) s, (size_t) len_s, (size_t) sizeof (s), mycompare);
	qsort ((void *) s, (size_t) len_s, (size_t) sizeof (dummy_S), mycompare);

	printf ("\nAfter Sort\n");
	for (i=0; i < len_s; i++)
		printf ("%d %d %ld\n", i, s[i].key, s[i].index);
}

int mycompare ( int *arg1, int *arg2)
{
	if ( *arg1 < *arg2)
		return (-1);
	if ( *arg1 > *arg2)
		return (1);
	return (0);
}

#endif



#endif
